pro calc_deriv_vb, ELarr, elfix, mfit, time, theta_d, rho_d, err_t, err_r, theta_f, rho_f, chimat, colmat, flag_wa = flag_wa

; Set up matrices for error determination of orbital elements
; (derivatives)
;
; Add option of fixing values of orbital elements
;
; INPUT:
;	ELarr: array of orbital elements [P,T,e,a,i,Omega,omega]
;       elfix: 7-element array: 0 if value to be held fixed
;                               1 if value to be free
;       mfit: number of parameters for which to solve (number of 1's in elfix)
;	time: time of observation
;	theta_d: array containing position angle of observations (in rad) 
;	rho_d: array containing separation of observation
;	err_t: error in theta
;	err_r: error in rho
;       flag_wa: set this flag if omega = omega_A (otherwise omega = omega_B)
;                This is necessary for simultaneous VB+SB fits.  SB orbits
;                define omega for primary, whereas VB orbits define omega 
;                for secondary. [, /flag_wa]
;
; OUTPUT:
;       theta_f: fit-value of PA
;       rho_f: fit- value of separation 
;	chimat: 7x7 covariance matrix
;	colmat: column matrix containing differences between data and fit
;
; Calls the following routines:
;     calc_Ei.pro
;
; Equations:  
; Initial estimates (P0,T0,e0,A0,B0,F0,G0) and a set of observations 
; (ti,rhoi,thetai) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Radius vector:
;	r = a(1-e^2)/(1 + ecos(nu))
; Fitted positions (rho_fit,theta_fit) are determined from the orbital elements
; 	tan(theta - Omega) = tan(nu + omega)*cosi
;	rho = r*cos(nu + omega)*sec(theta - Omega)
;
; Procedure:
; Minimize chi squared between data point positions and fitted positions:
;	chi^2 = sum[(rhodata - rhofit)^2/sigmarho^2 
;		    + (thetadata - thetafit)^2/sigmatheta^2]
; Take partial derivatives of chi^2 with respect to the orbital elements,
; (P,T,e,a,i,Omega,omega) and set to zero. Solve for the value of the orbital 
; elements that minimize chi^2.
;
; Since the partial derivatives are non-linear, cannot solve for the orbital
; elements that minimize chi^2 analytically.
;
; In turn, use a Newton-Raphson technique to converge upon the solution.
; Replace (rhoi,thetai) in the chi^2 equation with the Taylor series 
; approximation,
; 	x = x| + dx/dP|(P-P0) + dx/dT|(T-T0) + dx/de|(e-e0) 
;	       + dx/da|(a-a0) + dx/di|(i-i0) + dx/dW|(W-W0) + dx/dw|(w-w0)
; where x=(rho,theta) and the | represents the value evaluated at for the
; initial estimates for the orbital elements. Now, rhofit and thetafit are 
; linear in (P,T,e,a,i,W,w), so the partial derivatives can easily be taken 
; and solved by setting up a matrix and using Cramer's method.
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;	major:	 semi-major axis (in mas) (a)
;	inc:	 inclination (i)
;	Omega:	 position angle of node (W)
;	omega:	 angle between node and periastron (w)
;	ti:	 time of observation
;	rho_d:	 separation - data
;	theta_d: postion angle - data
;	rho_f:	 sep - fit (determined from orbital elements for time of obs.)
;	theta_f: PA - fit (determined from orbital elements for time of obs.)
;
; 06 October 2003: correct dependency of errors on epoch of periastron passage
; 11 November 2003: modified to include the fitted values for theta and rho
;    as output variables (convenient for chi2 calculations)
; 13 January 2005: fix 360 degree roll overs in (PAdata - PAfit)
; 25 March 2016:
;    Allow for sep,PA to be fit independently so that micrometer 
;    measurements where only sep or only PA are given can be included
;    in the fit.  Assume errors in sep,PA to be 0.0 when no measurement
;    exists.  Previous version saved as calc_deriv_vb_version1.pro


; Define orbital element parameters

period = ELarr(0)
Tperi = ELarr(1)
ecc = ELarr(2)
major = ELarr(3)
inc = ELarr(4)
W_cap = ELarr(5)
W_low = ELarr(6)

; If flag_wa is set then define omega as omega_A
; omega_B = omega_A + 180.0
; keyword_set returns True (1) if psf_name is defined, False (0) if undefined
if (keyword_set(flag_wa)) then w_low = w_low + !dpi

EL = dblarr(mfit)
i=0
for k=0, mfit-1 do begin
	if (elfix(k) ne 0) then EL(k) = ELarr(i)
	i = i+1
endfor

num = n_elements(time)

; Determine the eccentric anomalies Ei: 
; (Ei: array to the eccentric anomaly)

calc_Ei, time, period, Tperi, ecc, Ei

mu = 2*!dpi/period

; Determine true anomalies:

nu = 2*atan(sqrt((1+ecc)/(1-ecc))*tan(Ei/2.0))

; Determine radius vectors:

rad = major*(1 - ecc*cos(Ei))

; in order to check the value (correct quadrant) of nu:

rad1 = major*(1-ecc^2)/(1 + ecc*cos(nu))

; Determine (rho_f,theta_f) position from orbital elements for times 
; of observation

theta_f = W_cap + atan(tan(nu + w_low)*cos(inc))
rho_f = rad*cos(nu + w_low)/cos(theta_f - W_cap)

for i=0, num-1 do begin

	; convert rad and rad1 from double precision to float
	; this will avoid round-off errors when making comparisons

	frad = float(rad)
	frad1 = float(rad1)

	;check that nu is in appropriate quadrant
	if (frad(i) ne frad1(i)) then nu(i) = nu(i) + !dpi
	if (nu(i) ge 2*!dpi) then nu(i) = nu(i) - 2*!dpi

	; keep nu between 0 and 2Pi
	if (nu(i) lt 0.0) then nu(i) = nu(i) + 2*!dpi

	;check that theta_f is in correct quadrant
	if (rho_f(i) lt 0) then begin

		rho_f(i) = abs(rho_f(i))
		theta_f(i) = theta_f(i) + !dpi
	endif

	; theta between 0 and 360 deg
	if (theta_f(i) ge 2*!dpi) then theta_f(i) = theta_f(i) - 2*!dpi
	if (theta_f(i) lt 0) then theta_f(i) = theta_f(i) + 2*!dpi

endfor

; Calculate derivatives evaluated at initial conditions 
;	drho/del = (dx/dP,dx/dT,dx/de,dx/da,dx/di,dx/dW,dx/dw)
;	dtheta/del = (dy/dP,dy/dT,dy/de,dy/da,dy/di,dy/dW,dy/dw)
; where (x,y) -> (rho,theta) and
; del = (P,T,e,a,i,Omega,omega)

;initialize derivative arrays 

drho_del = dblarr(mfit,num)	;7: number of orbital elements
dtheta_del = dblarr(mfit,num)	;num: number of data points

chimat = dblarr(mfit,mfit) 	;matrix for minimizing chi^2
colmat = dblarr(mfit)		;column matrix for minimizing chi^2
delta_el = dblarr(mfit)		;solution to matrix equation

timediff=dblarr(num)

for i=0, num-1 do begin

	; reduce time difference to same epoch
	; (fraction of period covered at time t)

	fracP = (time(i) - Tperi)/period - fix((time(i) - Tperi)/period)

;	; (remove effects of positive and negative time -> arbitrary zero-point
;	;  for Tperi...... fracP goes from 0.0 to 1.0)
;
;	if (fracP lt 0) then fracP = 1.0d + fracP	

	timediff(i) = fracP

endfor


;dEi_dP = -2*!dpi*timediff/period/(1d - ecc*cos(Ei))
dEi_dP = -2*!dpi*(time - Tperi)/period^2/(1d - ecc*cos(Ei))
dEi_dT = -2*!dpi/period/(1d - ecc*cos(Ei))
dEi_de = sin(Ei)/(1d - ecc*cos(Ei))

dnu_dP = sqrt(1d - ecc^2)/(1d - ecc*cos(Ei))*dEi_dP
dnu_dT = sqrt(1d - ecc^2)/(1d - ecc*cos(Ei))*dEi_dT
dnu_de = ((1d - ecc^2)*dEi_de + sin(Ei))/sqrt(1d - ecc^2)/(1d - ecc*cos(Ei))

;drad_dP = major*ecc*sin(Ei)*dEi_dP
;drad_dT = major*ecc*sin(Ei)*dEi_dT
;drad_de = major*(ecc*sin(Ei)*dEi_de - cos(Ei))
;drad_da = 1d - ecc*cos(Ei)

dtheta_dP = cos(inc)*(cos(theta_f - W_cap)/cos(nu + w_low))^2*dnu_dP
dtheta_dT = cos(inc)*(cos(theta_f - W_cap)/cos(nu + w_low))^2*dnu_dT
dtheta_de = cos(inc)*(cos(theta_f - W_cap)/cos(nu + w_low))^2*dnu_de
dtheta_da = dblarr(num)
dtheta_di = -tan(inc)*cos(theta_f - W_cap)*sin(theta_f - W_cap)
dtheta_dWc = dblarr(num) + 1d
dtheta_dwl = cos(inc)*(cos(theta_f - W_cap)/cos(nu + w_low))^2

drho_dP = rho_f*(ecc*sin(Ei)/(1d - ecc*cos(Ei))*dEi_dP $
		- tan(theta_f - W_cap)/cos(inc)*dnu_dP $
		+ tan(theta_f - W_cap)*dtheta_dP)
drho_dT = rho_f*(ecc*sin(Ei)/(1d - ecc*cos(Ei))*dEi_dT $
		- tan(theta_f - W_cap)/cos(inc)*dnu_dT $
		+ tan(theta_f - W_cap)*dtheta_dT)
drho_de = rho_f*((-cos(Ei) + ecc*sin(Ei)*dEi_de)/(1d - ecc*cos(Ei)) $
		- tan(theta_f - W_cap)/cos(inc)*dnu_de $
		+ tan(theta_f - W_cap)*dtheta_de)
drho_da = rho_f/major
drho_di = rho_f*tan(theta_f - W_cap)*dtheta_di
drho_dWc = dblarr(num)
drho_dwl = rho_f*tan(theta_f - W_cap)*(-1d/cos(inc) + dtheta_dwl)

;drho_del = (dx_dP,dx_dT,dx_de,dx_da,dx_di,dx_dW,dx_dw)
;dtheta_del = (dy_dP,dy_dT,dy_de,dy_da,dy_di,dy_dW,dy_dw)

k=0
if (elfix(0) ne 0) then begin
	drho_del(k,*) = drho_dP
	k=k+1
endif
if (elfix(1) ne 0) then begin
	drho_del(k,*) = drho_dT
	k=k+1
endif
if (elfix(2) ne 0) then begin
	drho_del(k,*) = drho_de
	k=k+1
endif
if (elfix(3) ne 0) then begin
	drho_del(k,*) = drho_da
	k=k+1
endif
if (elfix(4) ne 0) then begin
	drho_del(k,*) = drho_di
	k=k+1
endif
if (elfix(5) ne 0) then begin
	drho_del(k,*) = drho_dWc
	k=k+1
endif
if (elfix(6) ne 0) then begin
	drho_del(k,*) = drho_dwl
	k=k+1
endif

k=0
if (elfix(0) ne 0) then begin
	dtheta_del(k,*) = dtheta_dP
	k=k+1
endif
if (elfix(1) ne 0) then begin
	dtheta_del(k,*) = dtheta_dT
	k=k+1
endif
if (elfix(2) ne 0) then begin
	dtheta_del(k,*) = dtheta_de
 	k=k+1
endif
if (elfix(3) ne 0) then begin
	dtheta_del(k,*) = dtheta_da
	k=k+1
endif
if (elfix(4) ne 0) then begin
	dtheta_del(k,*) = dtheta_di
	k=k+1
endif
if (elfix(5) ne 0) then begin
	dtheta_del(k,*) = dtheta_dWc
	k=k+1
endif
if (elfix(6) ne 0) then begin
	dtheta_del(k,*) = dtheta_dwl
	k=k+1
endif

; Set up matrix for minimizing chi squared
; Set up column matrix too
; (col) = (chimat)(delta_el)

diff_rho = dblarr(num)	  ; array for [rho(data) - rho(fit)]
diff_theta = dblarr(num)  ; array for [theta(data) - theta(fit)]

for k=0, num-1 do begin

	; weight derivative for each data point by corresponding 
	; measurement error

	diff_rho(k) = (rho_d(k)-rho_f(k))
	diff_theta(k) = (theta_d(k)-theta_f(k))

	; the ## operator performs typical matrix multiplication
	; the * multiplies individual elements (no summing)

	; account for any wrap-arounds from 360 deg to 0 deg

	if (diff_theta(k) ge !dpi) then $
		diff_theta(k) = diff_theta(k) - 2*!dpi
	if (diff_theta(k) le -!dpi) then $
		diff_theta(k) = 2*!dpi + diff_theta(k)	

        ; NOTE: break summation up into two parts so that we can
        ; account for separation-only or PA-only measurements
        ; If measurement does not exist then err_r = 0 or err_t = 0.

	for i=0, mfit-1 do begin

		for j=0, mfit-1 do begin

		    ;chimat(i,j) = chimat(i,j) $
		    ;	+ drho_del(i,k)*drho_del(j,k)/err_r(k)^2 $
		    ;	+ dtheta_del(i,k)*dtheta_del(j,k)/err_t(k)^2

                    if (err_r(k) ne 0.0) then $
                       chimat(i,j) = chimat(i,j) $
                                     + drho_del(i,k)*drho_del(j,k)/err_r(k)^2 

                    if (err_t(k) ne 0.0) then $
                       chimat(i,j) = chimat(i,j) $
                                     + dtheta_del(i,k)*dtheta_del(j,k)/err_t(k)^2

		endfor

		;colmat(i) = colmat(i) + diff_rho(k)*drho_del(i,k)/err_r(k)^2 $
		;	    + diff_theta(k)*dtheta_del(i,k)/err_t(k)^2

                if (err_r(k) ne 0.0) then $
		colmat(i) = colmat(i) + diff_rho(k)*drho_del(i,k)/err_r(k)^2 

                if (err_t(k) ne 0.0) then $
		colmat(i) = colmat(i) + diff_theta(k)*dtheta_del(i,k)/err_t(k)^2

	endfor		
endfor

end

