pro calc_orb, data_file, num

; Compute the orbital elements of a binary system, using the least squares
; technique described by Hartkopf, McAlister, Franz 1989, AJ, 98, 1014
;
; Procedure: Given (P,T,e) and a set of observations (ti,xi,yi), the eccentric
; anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)
; Four Thiele-Innes elements determined form least squares solution of 
; equations:
;	xi = AXi + FYi
;	yi = BXi + GYi
; Calculate geometric elements a("),i,Omega,omega from (A,F,B,G)
; 
; Paramers:
;	P:	period
;	T: 	time of periastron passage
;	e:	eccentricity
;	a:	semi-major axis (in mas)
;	i:	inclination
;	Omega:	position angle of node
;	omega:	angle between node and periastron
;	ti:	time of observation
;	xi:	RA
;	yi:	DEC
;
; Note: xi and yi are not the same convention for RA and DEC chosen to make 
; the orbital plots. Specifically, in the fitting the apparent ellipse 
; (ORBFITE.PRO):
;	x = RA = -rho*sin(theta),   y = DEC = rho*cos(theta)
;	(i.e. North is in the direction of the positive y axis, theta is
;	measured counterclockwise from positive y-axis)
; Whereas in this program, the coordinate system set up by Couteau,
;	x = rho*cos(theta),   y = rho*sin(theta)
;	(i.e. North id in the direction of the positive x-axis and the 
;	y-axis points east)
; Both coordinate systems have the same separation and position angle 
; orietations, they are just rotated by 90 deg with respect to each other.
;
; Input: 
;	data_file: file containing data points in the format:
;		      time  theta  rho
;		      where theta is the position angle and rho the separation
;		      (enter file name using quotes)
;	num: number of data points contained in data_file
;
; Prompted inputs:
;	period and error
;	time of periastron passage and error
;	eccentricity and error
;
; Began: May 2002

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file

time = dblarr(num)
theta = dblarr(num)
rho = dblarr(num)
xarr = dblarr(num)
yarr = dblarr(num)

openr,lun,data_file,/get_lun

for i=0, num-1 do begin

	readf, lun, temp1, temp2, temp3

	time(i) = temp1
	theta(i) = temp2
	rho(i) = temp3	

endfor

close,lun

; convert data points to x and y coordinates

xarr = rho * cos(theta*!dpi/180)	; x coordinate
yarr = rho * sin(theta*!dpi/180)	; y coordinate

; Obtain values for P,T,e and their respective errors

period = 0d
dperiod = 0d
Tperi = 0d
dTperi = 0d
ecc = 0d
decc = 0d

print,"Enter period:"
read,period
;print,"Enter error in period:"
;read,dperiod
print,"Enter time of periastron passage:"
read, Tperi
;print,"Enter error in time of periastron passage:"
;read,dTperi
print,"Enter eccentricity:"
read,ecc
;print,"Enter error in eccentricity:"
;read, decc

; Determine the eccentric anomalies Ei: 
;	mu(ti - T) = Ei - esin(Ei)    ... Kepler's Equation
;	where mu = 360/P = 2*pi/P
; Solve this transcendental equation through an iterative procedure.
; Use SOLVE_TRANS.PRO
; Formerly, this was done graphically by determining the intersection between 
; the line Ei - mu(ti - T) and the curve esin(Ei)

Ei = dblarr(num)	; array to hold Ei's (the eccentric anomaly)

mu = 2*!dpi/period

for i=0, num-1 do begin


	Mi = mu*(time(i) - Tperi) 

	; reduce to same epoch

	Mi = 2*!dpi*((time(i) - Tperi)/period - fix((time(i) - Tperi)/period))

	solve_trans,ecc,Mi,Eit

	Ei(i) = Eit

	print, "E - esinE = M"
	print, "Iterative E:",Ei(i)
	print, "M", Mi 
	print,"E - esinE:", Ei(i) - ecc*sin(Ei(i)) 

	waitforin=0
	print,"Enter any number to continue"
	read,waitforin

endfor

for i=0, num-1 do print,"time:", time(i)," Ei:",Ei(i)," x",xarr(i)," y",yarr(i)

; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)

; Xi and Yi are both dblarr(num)

Xi = cos(Ei) - ecc
Yi = sqrt(1 - ecc^2)*sin(Ei)

print,"Xi:",Xi
print,"Yi:",Yi

; Four Thiele-Innes elements determined form least squares solution of 
; equations:
;	xi = AXi + FYi
;	yi = BXi + GYi
; Calculate geometric elements a("),i,Omega,omega from (A,F,B,G)

; Determine initial estimates for A,F,B,G using straight-forward solution
; of two simultaneous equations, using the first and last data points.

N = num-1	; index of last element in data array

A_cap = (xarr(0)*Yi(N) - xarr(N)*Yi(0))/(Xi(0)*Yi(N) - Xi(N)*Yi(0))

F_cap = (xarr(N)*Xi(0) - xarr(0)*Xi(N))/(Xi(0)*Yi(N) - Xi(N)*Yi(0))

B_cap = (yarr(0)*Yi(N) - yarr(N)*Yi(0))/(Xi(0)*Yi(N) - Xi(N)*Yi(0))

G_cap = (yarr(N)*Xi(0) - yarr(0)*Xi(N))/(Xi(0)*Yi(N) - Xi(N)*Yi(0))

print,"(A,F,B,G) from solving simultaneous equations:"
print, A_cap,F_cap,B_cap,G_cap

linfit, Xi,Yi,xarr,A_cap,F_cap
linfit, Xi,Yi,yarr,B_cap,G_cap

print,"(A,F,B,G) from least squares fit:"
print, A_cap,F_cap,B_cap,G_cap

print, "A B F G:", A_cap, B_cap, F_cap, G_cap

BmF = (B_cap - F_cap)
BpF = (B_cap + F_cap)
AmG = (A_cap - G_cap)
ApG = (A_cap + G_cap)

; Determine sign of (Omega + omega) and (Omega - omega)
print, "Check sign of (Omega + omega) and (Omega - omega)"

sinOpo = BmF		; sin(Omega + omega) is propotional to (B-F)
cosOpo = ApG		; cos(Omega + omega) is propotional to (A+G)

sinOmo = BpF		; sin(Omega - omega) is propotional to (B+F)
cosOmo = AmG		; cos(Omega - omega) is propotional to (A-G)

Opo = atan((BmF)/(ApG))
Omo = atan((BpF)/(AmG))

print,"Check sign of Omega +/- omega"
print,"O+o:", Opo
print,"O-o:", Omo

if ((sinOpo/abs(sinOpo)) ne (sin(Opo)/abs(sin(Opo)))) then Opo = !dpi+Opo
if ((sinOmo/abs(sinOmo)) ne (sin(Omo)/abs(sin(Omo)))) then Omo = !dpi+Omo

print,"Corrected O+o:", Opo
print,"Corrected O-o:", Omo

Omega_cap = (Opo + Omo)/2d

print,"Check if 0 < Omega < pi"
print,"Omega:", Omega_cap 

if (Omega_cap gt !dpi) then omega_cap = omega_cap - !dpi
if (Omega_cap lt 0d) then omega_cap = omega_cap + !dpi

print,"Corrected Omega:", Omega_cap 

omega_low = Opo - Omega_cap

inc = 2* atan(sqrt((BpF*sin(Opo))/(BmF*sin(Omo))))

true_a = sqrt((A_cap*G_cap - B_cap*F_cap)/cos(inc))

print, "Omega omega i a:", (180/!dpi)*omega_cap, (180/!dpi)*omega_low, (180/!dpi)*inc, true_a

end


pro linfit, x, y, z, A, B

; Least squares linear fit to the equation: z = Ax + By
; The user supplies the x,y,z arrays and initial estimates for A,B
; The program uses the method of gradient descent to iteratively to alter
; the intial A,B to minimize the function:

num_data = n_elements(x)	;number of data points
ind_data = num_data -1

num_iterate = 100		;number of iterations
ind_iterate = num_iterate -1

; Loop through number of iterations to determine appropriate correction
; to the parameter vector (A,B)

for i=0, ind_iterate do begin

	; Initialize e^2 and del(e^2) each time through the loop.
	; e^2 and del(e^2) are calculated from a sum over all the data points 
	; for a given estimate of (A,B).  Therefore, it is necessary
	; to reset these values back to zero when parameter vector (A,B) 
	; is adjusted and e^2 and del(e^2) are re-calculated by summing over 
	; the number of data points again.

	e2 = 0d
	del_e2 = [0d,0d,0d,0d,0d]

	; loop through the number of data points to calculate e^2 and del(e^2)

	for j=0, ind_data do begin

		;Equation for linear fit:
		;
		; f(x,y,z,w) = Ax + By - z
		;
		; Calculate value of this function for each 
		; individual data point [which should be equal to 
		; (or as close as possible to) zero]

		func = A*x(j) + B*y(j) - z(j)

		; Calculate e^2 and del(e^2) (sum over data points):
		; e^2 = sum{f^2/[(df/dx)^2 + (df/dy)^2 + (df/dz)^2]}  
		; ... minimize dist. between points and curve under 
		; constraint f = 0
		; del(e^2) = sum{(de^2/dA, de^2/dB)}

		df_dx = A  
		df_dy = B
		df_dz = -1

		den = (df_dx)^2 + (df_dy)^2 + (df_dz)^2

		e2 = e2 + func^2/den
		
		de2_dA = 2*x(j)*func/den - 2*A*func^2/den^2
		de2_dB = 2*Y(j)*func/den - 2*B*func^2/den^2

		del_e2 = del_e2 + [de2_dA, de2_dB]
	
	endfor

	; Use gradient descent to determine corrections to (A,B)
	; (beta = 1, alpha^2 = 1 + i)

	grad = 1d/(del_e2^2 + 1 + i)

	delta_w = -0.01*e2*grad*del_e2	; initial guess

	w=[A,B]

	alter=step(x,y,z,w,delta_w,e2)

	A = A + alter*delta_w(0)
	B = B + alter*delta_w(1)

	w=[A,B]
	e2new = epsilon2(x,y,z,w)

	;Check to see if e^2 after alteration of step size is really less than 
	;the previous e^2. If not, then exit loop because we've reached a 
	;plateau.

	if (e2new ge e2) then i=num_iterate

;	print,"A,B:"
;	print,w
;	print,"e2:",e2new
;	wait,0.1

;	if (i eq 0) then print, "A, B, e^2, del(e^2), delta_w: "
;	print, w
;	print, e2
;	print, del_e2
;	print, delta_w

endfor

print, "e^2:", e2new

end

;
;
;

function step, x, y, z, w, delta_w, e2old

wnew = w + delta_w

e2new = epsilon2(x,y,z,wnew)

c=1d
stepsize=1d

if (e2new lt e2old) then begin	; search larger step sizes that still provide
				; e2new < e2old

	inc = 3
	for m=0, 4 do begin
	
;		print,"e2 OK.  Search larger step sizes"
;		print,"Iteration #",m

		c=c*inc
		wtemp=w+c*delta_w
		e2temp = epsilon2(x,y,z,wtemp)
		if (e2temp lt e2new) then begin
			e2new=e2temp
			stepsize=c
		endif else m=5	; end loop if larger step size does not produce
				; a smaller e2
	endfor	
endif else begin		; i.e. if e2new >= e2old
				; search smaller step sizes

	dec = 0.01	
	count = 0
	while (e2new ge e2old) do begin

;		print,"e2 not OK.  Search smaller step sizes"
;		print,"e2new: ", e2new, "  e2old: ", e2old

		c=c*dec
		wnew = w + c*delta_w
		e2new = epsilon2(x,y,z,wnew)

		if (count gt 50) then begin
			;Artificially exit loop if better step size not found
			;after 50 iterations.  This could be the result of 
			;either reaching a plateau or having to constantly
			;readjust parameters to bracket valley
				
			e2new = e2old - 0.001*e2old
			print, "loop exited automatically"
			qtemp=0
			print,"Type any number to continue"
			read, qtemp
		endif

		count = count+1

	endwhile
	stepsize=c
endelse

return, stepsize

end

;
;
;

function epsilon2, x, y, z, w

num = n_elements(x)

A=w(0)
B=w(1)

;initilize e^2 variable

e2 = 0d

; loop through the number of data points to calculate e^2

for j=0, num-1 do begin

	;Equation for linear fit:
	;
	; f(x,y,z,w) = Ax + By - z
	;
	; Calculate value of this function for each 
	; individual data point [which should be equal to 
	; (or as close as possible to) zero]

	func = A*x(j) + B*y(j) - z(j)

	; Calculate e^2 and del(e^2) (sum over data points):
	; e^2 = sum{f^2/[(df/dx)^2 + (df/dy)^2 + (df/dz)^2]}  
	; ... minimize dist. between points and curve under 
	; constraint f = 0
	; del(e^2) = sum{(de^2/dA, de^2/dB)}

	df_dx = A  
	df_dy = B
	df_dz = -1

	den = (df_dx)^2 + (df_dy)^2 + (df_dz)^2

	e2 = e2 + func^2/den
			
endfor

return, e2

end

;
;
;

pro solve_trans, e, M, EE

; Solve transcendental equation of the form E - esinE = M.
; Use iterative procedure to determine E.
; Initial approximation: E_0 = M + esinM - e^2/2 sin(2M)
; Improve solution by iterating the following to formulae:
;	M_0 = E_0 - esin(E_0)
;	E_1 = E_0 + (M - M_0)/(1 - ecos(E_0))
;	(derivative of Kepler's equation)
;
; Method adapted from Heintz 1978 (p.34-35)
; Results compared with point-&-click graphical method.  Iterative approach 
; leads to exact solution that satisfies E - esinE = M.  Therefore, 
; point-&-click method is subsequently removed from orbit fitting.
;
; INPUT:
;	e: eccentricity
;	M: mean anomaly   M= 2*Pi/P
;
; OUTPUT:
;	EE: eccentric anomaly
;
; Created: 9 May 2002

; Initial approximation:

EE = M + e*sin(M) + e^2/2*sin(2*M)

;print, "Initial approximation for E:", EE

EEi = 0d	; parameter to hold initial value once enter while loop

count = 0

while (abs(EE - EEi) gt 0.000001) do begin

	EEi = EE

	Mi = EEi - e*sin(EEi)

	EE = EEi + (M - Mi)/(1 - e*cos(EEi))

	count=count+1

endwhile

;print, "Final iterated E:", EE

;print,"Number of iterations:",count

end
