pro calc_vbfit, ELarr, time, theta_f, rho_f, flag_wa = flag_wa

; Compute model orbital positions for Visual Binary
; Input parameters:
;    ELarr: array of orbital parameters
;           [period,Tperi,ecc,major,inc,W_cap,w_low]
;    time:  array containing times for which to compute position
; Output parameters:
;    theta_f: array for model PA
;    rho_f: array for model separation
;    flag_wa: set this flag if omega = omega_A (otherwise omega = omega_B)
;             This is necessary for simultaneous VB+SB fits.  SB orbits
;             define omega for primary, whereas VB orbits define omega 
;             for secondary. [, /flag_wa]

; Define orbital element parameters

period = ELarr(0)
Tperi = ELarr(1)
ecc = ELarr(2)
major = ELarr(3)
inc = ELarr(4)
W_cap = ELarr(5)
W_low = ELarr(6)

; If flag_wa is set then define omega as omega_A
; omega_B = omega_A + 180.0
; keyword_set returns True (1) if psf_name is defined, False (0) if undefined
if (keyword_set(flag_wa)) then w_low = w_low + !dpi

num = n_elements(time)

; Determine the eccentric anomalies Ei: 
; (Ei: array to the eccentric anomaly)

calc_Ei, time, period, Tperi, ecc, Ei

mu = 2*!dpi/period

; Determine true anomalies:

nu = 2*atan(sqrt((1+ecc)/(1-ecc))*tan(Ei/2.0))

; Determine radius vectors:

rad = major*(1 - ecc*cos(Ei))

; in order to check the value (correct quadrant) of nu:

rad1 = major*(1-ecc^2)/(1 + ecc*cos(nu))

; Determine (rho_f,theta_f) position from orbital elements for times 
; of observation

theta_f = W_cap + atan(tan(nu + w_low)*cos(inc))
rho_f = rad*cos(nu + w_low)/cos(theta_f - W_cap)

for i=0, num-1 do begin

	; convert rad and rad1 from double precision to float
	; this will avoid round-off errors when making comparisons

	frad = float(rad)
	frad1 = float(rad1)

	;check that nu is in appropriate quadrant
	if (frad(i) ne frad1(i)) then nu(i) = nu(i) + !dpi
	if (nu(i) ge 2*!dpi) then nu(i) = nu(i) - 2*!dpi

	; keep nu between 0 and 2Pi
	if (nu(i) lt 0.0) then nu(i) = nu(i) + 2*!dpi

	;check that theta_f is in correct quadrant
	if (rho_f(i) lt 0) then begin

		rho_f(i) = abs(rho_f(i))
		theta_f(i) = theta_f(i) + !dpi
	endif

	; theta between 0 and 360 deg
	if (theta_f(i) ge 2*!dpi) then theta_f(i) = theta_f(i) - 2*!dpi
	if (theta_f(i) lt 0) then theta_f(i) = theta_f(i) + 2*!dpi

endfor

end
