pro compute_position

; Given a set of orbital elements {P,T,e,a,i,Omega,omega},
; compute position at a given time.
; 
; Equations:  
; Orbital elements (P,T,e,a,i,Omega,omega) and time are known.
; The four Thiele-Innes elements (A,B,F,G) are computed from the
; geometric elements (a,i,Omega,omega):
;       A = major*(cos(w_low)*cos(W_cap) - sin(w_low)*sin(W_cap)*cos(inc))
;       B = major*(cos(w_low)*sin(W_cap) + sin(w_low)*cos(W_cap)*cos(inc))
;       F = major*(-sin(w_low)*cos(W_cap) - cos(w_low)*sin(W_cap)*cos(inc))
;       G = major*(-sin(w_low)*sin(W_cap) + cos(w_low)*cos(W_cap)*cos(inc))
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)
; The separation in RA and DEC are given by (see notes on conventions below):
;	xi = AXi + FYi
;	yi = BXi + GYi
;
; Essentially, the equations one needs to solve are:
;	E - esinE = 2Pi/P * (t - T)
;	x = A*(cos(Ei) - e) + F*(sqrt(1 - e^2)*sin(Ei))
;	y = B*(cos(Ei) - e) + G*(sqrt(1 - e^2)*sin(Ei))
;
; Parameters:
;	Period:	period (P)
;	Tperi: 	time of periastron passage (T)
;	ecc:	eccentricity (e)
;	major:	semi-major axis in angular measure (a)
;	inc:	inclination (i) (in degrees)
;	Omega:	position angle of node (in degrees)
;	omega:	angle between node and periastron (in degrees)
;	time:	time of observation
;	xi:	sep in x (determined from orbital elements for time of obs.)
;	yi:	sep in y (determined from orbital elements for time of obs.)
;	A,B,F,G: Thiele-Innes elements
;
; Note: xi and yi are not the same convention for RA and DEC chosen to make 
; the orbital plots. Specifically, in the fitting the apparent ellipse 
; (ORBFITE.PRO):
;	x = RA = -rho*sin(theta),   y = DEC = rho*cos(theta)
;	(i.e. North is in the direction of the positive y axis, theta is
;	measured counterclockwise from positive y-axis)
; Whereas in this program, the coordinate system set up by Couteau,
;	x = rho*cos(theta),   y = rho*sin(theta)
;	(i.e. North is in the direction of the positive x-axis and the 
;	y-axis points east)
; Both coordinate systems have the same separation and position angle 
; orientations, they are just rotated by 90 deg with respect to each other.
;	
; Prompted inputs:
;	All of the orbital elements: P,T,e
;       Time that you wish to compute separation and 
;
; OUTPUT: 
;	Plot of orbit:
;          - square marks periastron passage
;          - position of secondary during requested time marked by diamond 
;          - position of primary marked with an asterisk

!P.charsize=1.5
!P.charthick=4.0
!P.thick=4.0
!X.thick=4.0
!Y.thick=4.0
frac=1.4	;scale factor for adjusting size of plotting symbols


; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1

period = 0d
Tperi = 0d
ecc = 0d
major = 0d
inc = 0d
W_cap = 0d
w_low = 0d

; Obtain initial values for P,T,e,a,i,W,w

print,"Enter P T e a i Omega(W) omega(w)"
read, period, Tperi, ecc, major, inc, W_cap, w_low

;convert angles from degrees to radians

inc = inc*!dpi/180.0
W_cap = W_cap*!dpi/180.0
w_low = w_low*!dpi/180.0

time = 0d

print,'Enter time you wish to compute position for:'
read,time

; Determine the eccentric anomalies Ei: 
;     mu(ti - T) = Ei - esin(Ei)  ... Kepler's Equation
;     where mu = 360/P = 2*pi/P
; Solve this transcendental equation through an 
; iterative procedure.
; Use SOLVE_TRANS.PRO

; array to hold Ei's (the eccentric anomaly)

Ei = 0d

mu = 2*!dpi/period

Mi = mu*(time - Tperi) 

;print,'Initial M:',Mi

; reduce to same epoch

Mi = 2*!dpi*((time - Tperi)/period $	
	- fix((time - Tperi)/period))

;print,'Final M',Mi

solve_trans,ecc,Mi,Eit

Ei = Eit

;print,'E',Ei

;print,'E - esinE =', Ei - ecc*sin(Ei)

; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)

Xi = cos(Ei) - ecc
Yi = sqrt(1 - ecc^2)*sin(Ei)

;print,'(X,Y) = ',Xi,Yi

; Compute the four Thiele-Innes elements

A_cap = major*(cos(w_low)*cos(W_cap) - sin(w_low)*sin(W_cap)*cos(inc))
B_cap = major*(cos(w_low)*sin(W_cap) + sin(w_low)*cos(W_cap)*cos(inc))
F_cap = major*(-sin(w_low)*cos(W_cap) - cos(w_low)*sin(W_cap)*cos(inc))
G_cap = major*(-sin(w_low)*sin(W_cap) + cos(w_low)*cos(W_cap)*cos(inc))

C_cap = major*sin(w_low)*sin(inc)
H_cap = major*cos(w_low)*sin(inc)

;print,'(A,B,F,G) = ', A_cap,B_cap,F_cap,G_cap

; Compute coordinates of secondary

;	xi = AXi + FYi
;	yi = BXi + GYi

xorb = A_cap*Xi + F_cap*Yi
yorb = B_cap*Xi + G_cap*Yi

zorb = C_cap*Xi + H_cap*Yi

; convert (xorb,yorb) to (PAfit,sepfit)
; x = rho*cos(theta), y=rho*sin(theta)

sep = sqrt(xorb^2 + yorb^2)

; Get correct sign on PA by entering yorb,xorb separately into atan:
PA = atan(yorb,xorb)

;print,'PA',PA*180.0/!dpi

if (PA lt 0.0) then PA = 2*!dpi + PA

print,'Position of companion for t= ',time
;print,'(x,y) = ',xorb,yorb
print,'(sepRA,sepDec) = ',yorb,xorb
print,'z-position = ',zorb
print,'(sep,PA) = ',sep,PA*180.0/!dpi

; Plot orbit with 1000 points

num = 1001.0

tstep = period/(num - 1.0)

; x points N (DEC)
; y points E (increasing RA)
; therefore plot as (-y,x)

xarr=dblarr(num)
yarr=dblarr(num)

tarr = Tperi

for i=0, num-1 do begin

;	print,'t=',time

	mu = 2*!dpi/period

	Mi = mu*(tarr - Tperi) 

	; reduce to same epoch

	Mi = 2*!dpi*((tarr - Tperi)/period $	
		- fix((tarr - Tperi)/period))

	solve_trans,ecc,Mi,Ei

	; Normalized rectangular coordinates Xi & Yi:
	;	Xi = cos(Ei) - e
	;	Yi = sqrt(1 - e^2)*sin(Ei)

	Xi = cos(Ei) - ecc
	Yi = sqrt(1 - ecc^2)*sin(Ei)

	; Compute the four Thiele-Innes elements

	A_cap = major*(cos(w_low)*cos(W_cap) - sin(w_low)*sin(W_cap)*cos(inc))
	B_cap = major*(cos(w_low)*sin(W_cap) + sin(w_low)*cos(W_cap)*cos(inc))
	F_cap = major*(-sin(w_low)*cos(W_cap) - cos(w_low)*sin(W_cap)*cos(inc))
	G_cap = major*(-sin(w_low)*sin(W_cap) + cos(w_low)*cos(W_cap)*cos(inc))

	; Compute coordinates of secondary

	;	xi = AXi + FYi
	;	yi = BXi + GYi

	xarr(i) = A_cap*Xi + F_cap*Yi
	yarr(i) = B_cap*Xi + G_cap*Yi

	tarr = tarr + tstep

;	print,'(x,y)=',xarr(i),yarr(i)
endfor

sepmax = round(1.2*max(abs([yarr,xarr])))

plot,yarr,xarr,xtitle='!4D!3RA',ytitle='!4D!3DEC',ISOTROPIC=1, $
     xrange=[sepmax,-sepmax],yrange=[-sepmax,sepmax], xstyle=1,ystyle=1

; mark periastron passage with a square

oplot,[yarr(0)],[xarr(0)],psym=6

; mark primary with an asterisk

oplot,[0],[0],psym=2

; mark secondary at computed time with a diamond

oplot,[yorb],[xorb],psym=4

print, 'Position of perisastron (sepRA,sepDEC) = ',yarr(0),xarr(0)

; Define user symbol to be closed circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)

;Define the symbol to be a unit circle with 16 points
usersym, frac*cos(Z), frac*sin(Z), /fill

set_plot, 'ps'
device, /Color, Bits_per_pixel=8, filename='temp.ps',xsize=15,ysize=15

;plot,-yarr,xarr,xtitle='E-W sep',ytitle='N-S sep',ISOTROPIC=1
plot,yarr,xarr,xtitle='!4D!3RA',ytitle='!4D!3DEC',ISOTROPIC=1, $
     xrange=[sepmax,-sepmax],yrange=[-sepmax,sepmax], xstyle=1,ystyle=1

; mark periastron passage with a square

oplot,[yarr(0)],[xarr(0)],psym=6

; mark primary with an asterisk

oplot,[0],[0],psym=2

; mark secondary at computed time with a diamond

oplot,[yorb],[xorb],psym=4

; Plot dotted lines through origin

oplot,[0,0],[-sepmax,sepmax],linestyle=1
oplot,[sepmax,-sepmax],[0,0],linestyle=1

device, /close
set_plot, 'x'

print,'t          sep    PA'
print,format='(f9.4,f7.1,f7.1)',time,sep,PA*180.0/!dpi
;print,format='(f9.4,f8.3,f7.1)',time,sep,PA*180.0/!dpi
;print,format='(f9.4,f9.4,f7.1)',time,sep,PA*180.0/!dpi

end

;
;
;

pro solve_trans, e, M, EE

; Solve transcendental equation of the form E - esinE = M.
; Use iterative procedure to determine E.
; Initial approximation: E_0 = M + esinM - e^2/2 sin(2M)
; Improve solution by iterating the following to formulae:
;	M_0 = E_0 - esin(E_0)
;	E_1 = E_0 + (M - M_0)/(1 - ecos(E_0))
;	(derivative of Kepler's equation)
;
; Method adapted from Heintz 1978 (p.34-35)
; Results compared with point-&-click graphical method.  Iterative approach 
; leads to exact solution that satisfies E - esinE = M.  Therefore, 
; point-&-click method is subsequently removed from orbit fitting.
;
; INPUT:
;	e: eccentricity
;	M: mean anomaly   M= 2*Pi/P
;
; OUTPUT:
;	EE: eccentric anomaly
;
; Created: 9 May 2002

; Initial approximation:

EE = M + e*sin(M) + e^2/2*sin(2*M)

;print, "Initial approximation for E:", EE

EEi = 0d	; parameter to hold initial value once enter while loop

count = 0

while (abs(EE - EEi) gt 0.000001) do begin

	EEi = EE

	Mi = EEi - e*sin(EEi)

	EE = EEi + (M - Mi)/(1 - e*cos(EEi))

	count=count+1

endwhile

;print, "Final iterated E:", EE

;print,"Number of iterations:",count

end

