pro compute_rv_sb2

; Given a set of orbital elements {P,T,e,K1,omega,Vsys}
; compute SB1 radial velocity and phase at a given time
; 
; Calls the following routines:
;     calc_Ei.pro
;     solve_trans.pro
;     calc_sb2fit.pro - for plotting model orbits
;
; Equations:  
; Initial estimates (P0,T0,e0,K1_0,omega0,Vsys0) and  
; a set of observations (ti,V1i) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Projected semi-major axis (a1sini):
;       a1sini = (P/2pi)*K1*(1 - e^2)^1/2
; The fitted velocities (V1_i) are determined from the orbital elements:
;       V1 = K1*[e*cos(omega) + cos(nu + omega)] 
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;       K1:      velocity amplitude of primary (in km/s) (K1)
;	omega:	 angle between node and periastron (w)
;	ti:	 time of observation
;	V1_d:	 velocity of primary - measured data
;	V1_f:	 V1 - fit (determined from orbital elements for time of obs.)
;       Vsys:    systemic velocity
;
;
; Prompted inputs:
;	all of the orbital elements: P,T,e
;       time to compute orbital phase and radial velocity
;


!P.charsize=1.5
!P.charthick=4.0
!P.thick=5.0
!X.thick=5.0
!Y.thick=5.0
frac=1.4	;scale factor for adjusting size of plotting symbols


; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1

; Obtain values for P,T,e,a1sini,omega,Vsys

period = 0d
Tperi = 0d
ecc = 0d
K1 = 0d
omega = 0d
Vsys = 0d

print,"Enter P,T,e,K1(km/s),K2(km/s)omega,Vsys:"
read,period,Tperi,ecc,K1,K2,omega,Vsys

;convert angles from degrees to radians

omega = omega*!dpi/180.0

time = 0d

print,'Enter time you wish to compute position for:'
read,time

EL = [period,Tperi,ecc,K1,K2,omega,Vsys]

calc_sb2fit, EL, time, V1fit, V2fit


; compute phase from time of periastron passage
phase = (time - Tperi)/period

; reduce phase to -1.0 to 1.0
phase = phase - fix(phase)

; keep phase between 0.0 to 1.0
if (phase lt 0.0) then phase = phase + 1.0

print,'Time:',time
print,'V1:',v1fit
print,'V2:',v2fit
print,'Phase:',phase


end

;
;
;

pro solve_trans, e, M, EE

; Solve transcendental equation of the form E - esinE = M.
; Use iterative procedure to determine E.
; Initial approximation: E_0 = M + esinM - e^2/2 sin(2M)
; Improve solution by iterating the following to formulae:
;	M_0 = E_0 - esin(E_0)
;	E_1 = E_0 + (M - M_0)/(1 - ecos(E_0))
;	(derivative of Kepler's equation)
;
; Method adapted from Heintz 1978 (p.34-35)
; Results compared with point-&-click graphical method.  Iterative approach 
; leads to exact solution that satisfies E - esinE = M.  Therefore, 
; point-&-click method is subsequently removed from orbit fitting.
;
; INPUT:
;	e: eccentricity
;	M: mean anomaly   M= 2*Pi/P
;
; OUTPUT:
;	EE: eccentric anomaly
;
; Created: 9 May 2002

; Initial approximation:

EE = M + e*sin(M) + e^2/2*sin(2*M)

;print, "Initial approximation for E:", EE

EEi = 0d	; parameter to hold initial value once enter while loop

count = 0

while (abs(EE - EEi) gt 0.000001) do begin

	EEi = EE

	Mi = EEi - e*sin(EEi)

	EE = EEi + (M - Mi)/(1 - e*cos(EEi))

	count=count+1

endwhile

;print, "Final iterated E:", EE

;print,"Number of iterations:",count

end

