pro compute_rv_sb2_ephemeris

; Given a set of orbital elements {P,T,e,K1,omega,Vsys}
; compute SB1 radial velocity and phase at a given time
; 
; Calls the following routines:
;     calc_Ei.pro
;     solve_trans.pro
;     calc_sb2fit.pro - for plotting model orbits
;
; Equations:  
; Initial estimates (P0,T0,e0,K1_0,omega0,Vsys0) and  
; a set of observations (ti,V1i) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Projected semi-major axis (a1sini):
;       a1sini = (P/2pi)*K1*(1 - e^2)^1/2
; The fitted velocities (V1_i) are determined from the orbital elements:
;       V1 = K1*[e*cos(omega) + cos(nu + omega)] 
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;       K1:      velocity amplitude of primary (in km/s) (K1)
;	omega:	 angle between node and periastron (w)
;	ti:	 time of observation
;	V1_d:	 velocity of primary - measured data
;	V1_f:	 V1 - fit (determined from orbital elements for time of obs.)
;       Vsys:    systemic velocity
;
;
; Prompted inputs:
;	all of the orbital elements: P,T,e
;       time to compute orbital phase and radial velocity
;


;!P.charsize=1.5
;!P.charthick=4.0
;!P.thick=5.0
;!X.thick=5.0
;!Y.thick=5.0
;frac=1.4	;scale factor for adjusting size of plotting symbols

!P.font = 0
!P.charsize=1.0
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.0        ; scale factor for adjusting size of plotting symbols
;frac = 0.6
symthick=2.0    ; thickness for symbol lines
lthick = 2


; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1

; Obtain values for P,T,e,a1sini,omega,Vsys

period = 0d
Tperi = 0d
ecc = 0d
K1 = 0d
omega = 0d
Vsys = 0d

print,"Enter P,T,e,K1(km/s),K2(km/s)omega,Vsys:"
read,period,Tperi,ecc,K1,K2,omega,Vsys

;convert angles from degrees to radians

omega = omega*!dpi/180.0

time = 0d
tstep = 0d
trange = 0d

print,'Enter starting time for RV ephemeris:'
read,time

print,'Enter step size and length of time for which to compute ephemeris'
read, tstep,trange

tnum = trange/tstep + 1
tarr = findgen(tnum)*tstep + time

EL = [period,Tperi,ecc,K1,K2,omega,Vsys]

calc_sb2fit, EL, tarr, V1fit, V2fit


; compute phase from time of periastron passage
phase = (tarr - Tperi)/period

; reduce phase to -1.0 to 1.0
phase = phase - fix(phase)

; keep phase between 0.0 to 1.0
for i=0, tnum-1 do begin
   if (phase(i) lt 0.0) then phase(i) = phase(i) + 1.0
endfor

print,'       Time         V1        V2     phase'

for i=0,tnum-1 do print,format='(f15.3,f10.3,f10.3,f7.3)',tarr(i),v1fit(i),v2fit(i),phase(i)

vmin=min([v1fit,v2fit])
vmax=max([v1fit,v2fit])

; Extend plotting ranges by 10% 
; (be careful - this will be different depending whether RVs or
; positve or negative)
if (vmax ge 0) then vmax = vmax*1.1 else vmax = vmax*1.1
if (vmin lt 0) then vmin = vmin*1.1 else vmin = vmin*0.9

plot,phase,v1fit,psym=4,yrange=[vmin,vmax],xrange =[-0.1,1.1], $
  xtitle = 'Phase', ytitle = 'Radial Velocity (km/s)', xstyle=1,ystyle=1,/NoData

oplot,[-0.1,1.1],[Vsys,Vsys],Linestyle=1

oplot,phase,v1fit,psym=6
oplot,phase,v2fit,psym=4

print,'Squares are primary.  Diamonds are secondary'

; Define user symbol to be rectangle
xrect = frac*[-1.0,-1.0,1.0,1.0,-1.0]
yrect = frac*[-1.0,1.0,1.0,-1.0,-1.0]
usersym,xrect,yrect

; Define user symbol to be plus sign
xplus = frac*[-1.0,1.0,0.0,0.0,0.0]
yplus = frac*[0.0,0.0,0.0,1.0,-1.0]
usersym,xplus,yplus

; Define user symbol to be open circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)
;Define the symbol to be a unit circle with 16 points
usersym, frac*cos(Z), frac*sin(Z)

; Extend phase coverage:
ans_extend = 'y'
if (ans_extend eq 'y') then begin
   phase = [phase-1,phase,phase+1]
   V1fit = [V1fit,V1fit,V1fit]
   V2fit = [V2fit,V2fit,V2fit]
endif

set_plot, 'ps'
device, /Color, Bits_per_pixel=8, filename='temp.eps',xsize=12,ysize=10,/encap

plot,phase,v1fit,psym=4,yrange=[vmin,vmax],xrange =[-0.1,1.1], $
  xtitle = '!3Phase', ytitle = 'Radial Velocity (km/s)', xstyle=1,ystyle=1,/NoData

oplot,[-0.1,1.1],[Vsys,Vsys],Linestyle=1

oplot,phase,V1fit,LineStyle=0.0,color=3,thick=lthick
oplot,phase,V2fit,Linestyle=0.0,color=1,thick=lthick

usersym, frac*cos(Z), frac*sin(Z),thick=symthick,/fill
oplot,phase,V1fit,psym=8,color=3
usersym,xrect,yrect,thick=symthick,/fill
oplot,phase,V2fit,psym=8,color=1


device, /close
set_plot, 'x'


end

;
;
;

pro solve_trans, e, M, EE

; Solve transcendental equation of the form E - esinE = M.
; Use iterative procedure to determine E.
; Initial approximation: E_0 = M + esinM - e^2/2 sin(2M)
; Improve solution by iterating the following to formulae:
;	M_0 = E_0 - esin(E_0)
;	E_1 = E_0 + (M - M_0)/(1 - ecos(E_0))
;	(derivative of Kepler's equation)
;
; Method adapted from Heintz 1978 (p.34-35)
; Results compared with point-&-click graphical method.  Iterative approach 
; leads to exact solution that satisfies E - esinE = M.  Therefore, 
; point-&-click method is subsequently removed from orbit fitting.
;
; INPUT:
;	e: eccentricity
;	M: mean anomaly   M= 2*Pi/P
;
; OUTPUT:
;	EE: eccentric anomaly
;
; Created: 9 May 2002

; Initial approximation:

EE = M + e*sin(M) + e^2/2*sin(2*M)

;print, "Initial approximation for E:", EE

EEi = 0d	; parameter to hold initial value once enter while loop

count = 0

while (abs(EE - EEi) gt 0.000001) do begin

	EEi = EE

	Mi = EEi - e*sin(EEi)

	EE = EEi + (M - Mi)/(1 - e*cos(EEi))

	count=count+1

endwhile

;print, "Final iterated E:", EE

;print,"Number of iterations:",count

end

