pro element_range, data_file, num, pte_file, fnum

; Computes statistics for the orbital elements
;
; Modification of PLOT_PAE.PRO (simplified - no histograms)
; Modification of LEAST_PTE.PRO
;
; INPUT:
; 	data_file: name of file containing the observations
;		format: time PA ePA sep esep
;	num: number of lines in data_file
;	pte_file: name of file containing P,T,e,chi2 for possible solutions
;		format: P T e chi2
;	fnum: number of lines in pte_file
;
; PROMPTED INPUTS:
;	system parallax (in mas)
;	upper bound for mass restriction
;
; PROCEDURE:
; 	performs least squares fit
; 	computes a,i,Omega,omega,M for each (P,T,e) set
;	computes min, max, mean, median, and stdev for each orbital element
;
; OUTPUT: 
;	statistics for orbital elements (P,T,a,e,i,Omega,omega,mass)
;	min, max, mean, median, and stdev for each element
;	(on-screen)
;
; 7 October 2003: created

; Read in pte file names

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time = dblarr(num)
theta = dblarr(num)
rho = dblarr(num)
dtheta = dblarr(num)
drho = dblarr(num)

openr,3,data_file

for i=0, num-1 do begin

	readf, 3, temp1, temp2, temp3, temp4, temp5

	time(i) = temp1
	theta(i) = temp2
	dtheta(i) = temp3
	rho(i) = temp4	
	drho(i) = temp5
	
endfor

close,3

openr,1,pte_file

; convert theta and dtheta to radians

theta = theta*!dpi/180
dtheta = dtheta*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(theta)^2*drho^2 + rho^2*sin(theta)^2*dtheta^2)
dy = sqrt(sin(theta)^2*drho^2 + rho^2*cos(theta)^2*dtheta^2)

period = 0d
Tperi = 0d
ecc = 0d
xchi = 0d

massi = dblarr(fnum)
a_arr = dblarr(fnum)
e_arr = dblarr(fnum)
i_arr = dblarr(fnum)
P_arr = dblarr(fnum)
T_arr = dblarr(fnum)
W_cap_arr = dblarr(fnum)
w_low_arr = dblarr(fnum)
	
parallax = 0.0d

print, 'Enter the parallax of the system (in same units as separation):'
print,'(distance of 140, parallax = 7.14 mas)'
read, parallax

ansyd=' '
print,'Is period in years or days (y/d)?'
read,ansyd

if (ansyd eq 'y') then yd = 1.0d
if (ansyd eq 'd') then yd = 1.0d/365.25

print, 'Computing masses...'

openw,2,"result_all"
printf,2, "P T e a i Omega omega mass chi2:"

for j=0, fnum-1 do begin

	; Obtain initial values for P,T,e

;	print,"Enter P T e (period, time of periastron passage, eccentricity):"
	readf, 1, period, Tperi, ecc, xchi

	; degrees of freedom
	dof = 2*num - 7

	; Determine fitted points (xfit,yfit) by computing the Thiele Innes
	; elements A,F,B,G

	calc_AFBG, time, xarr, dx, yarr, dy, period, Tperi, ecc, $
		   A_cap, F_cap, B_cap, G_cap, xfit, yfit

	; Determine chi squared

	chi2 = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

	; reduced chi squared:
;	chi2 = chi2/dof

	; determine a,i,Omega,omega (using AIWW.PRO)

	aiWw, A_cap,B_cap,F_cap,G_cap,major,inc,W_cap,w_low

;	print, "a i Omega omega:", major, (180/!dpi)*inc, $
;	(180/!dpi)*W_cap, (180/!dpi)*w_low

	EL = [period,Tperi,ecc,major,inc,W_cap,w_low]

	; calculate mass of system using distance of 140 pc
	; parallax = 7.14 mas
	
	mass = major^3/parallax^3/(yd*period)^2

	massi(j) = mass
	a_arr(j) = major
	e_arr(j) = ecc
	i_arr(j) = 180*inc/!dpi
	P_arr(j) = period
	T_arr(j) = Tperi
	W_cap_arr(j) = 180*W_cap/!dpi
	w_low_arr(j) = 180*w_low/!dpi

	printf,2,EL(0),El(1),EL(2),EL(3),180*EL(4)/!dpi,180*EL(5)/!dpi, $
		180*EL(6)/!dpi, mass, chi2


endfor

close,1
close,2

; restrict range of mass to be within Mass_min

Mass_min = 0d

print, "Enter upper bound for mass restriction"
read,Mass_min

massnum = where(massi le Mass_min, mcount)

mrestrict = dblarr(mcount)
Prestrict = dblarr(mcount)
arestrict = dblarr(mcount)
erestrict = dblarr(mcount)
irestrict = dblarr(mcount)
Trestrict = dblarr(mcount)
W_caprestrict = dblarr(mcount)
w_lowrestrict = dblarr(mcount)

n=0

for l=0, fnum-1 do begin

	if (massi(l) le Mass_min) then begin

		mrestrict(n) = massi(l)
		Prestrict(n) = P_arr(l)
		arestrict(n) = a_arr(l)
		erestrict(n) = e_arr(l)
		irestrict(n) = i_arr(l)
		Trestrict(n) = T_arr(l)
		W_caprestrict(n) = W_cap_arr(l)
		w_lowrestrict(n) = w_low_arr(l)
		n=n+1
	endif
endfor

;temppind = where(P_arr le 100, pcount)
;print,'Number of solutions where P < 100: ', pcount
;temppind = where(Prestrict le 100, pcount)
;print,'Number of solutions in restricted range where P < 100: ', pcount
;temppind = where(P_arr gt 100 and P_arr le 200, pcount)
;print,'Number of solutions where 100 < P < 200: ', pcount
;temppind = where(Prestrict gt 100 and Prestrict le 200, pcount)
;print,'Number of solutions in restricted range where 100 < P < 200: ', pcount

print,"Error estimates: min max mean median stddev"
print,format='(%"M:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(massi),max(massi),mean(massi),median(massi),stddev(massi)
print,format='(%"P:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(P_arr),max(P_arr),mean(P_arr),median(P_arr),stddev(P_arr)
print,format='(%"a:",f10.6,f10.6,f10.6,f10.6,f10.6)', $
	min(a_arr),max(a_arr),mean(a_arr),median(a_arr),stddev(a_arr)
print,format='(%"e:",f10.6,f10.6,f10.6,f10.6,f10.6)', $
	min(e_arr),max(e_arr),mean(e_arr),median(e_arr),stddev(e_arr)
print,format='(%"i:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(i_arr),max(i_arr),mean(i_arr),median(i_arr),stddev(i_arr)
print,format='(%"T:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(T_arr),max(T_arr),mean(T_arr),median(T_arr),stddev(T_arr)
print,format='(%"W:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(W_cap_arr),max(W_cap_arr),mean(W_cap_arr), $
	median(W_cap_arr),stddev(W_cap_arr)
print,format='(%"w:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(w_low_arr),max(w_low_arr),mean(w_low_arr), $
	median(w_low_arr),stddev(w_low_arr)

; convert Mass_min to a string and remove leading white space

Mass_value = string(Mass_min)

Mass_value = strtrim(Mass_min,1)

print,"Restrict mass to be below ", Mass_value, " Msun"
print,format='(%"M:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(mrestrict),max(mrestrict), mean(mrestrict), $
	median(mrestrict),stddev(mrestrict)
print,format='(%"P:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(Prestrict),max(Prestrict), mean(Prestrict), $
	median(Prestrict),stddev(Prestrict)
print,format='(%"a:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(arestrict),max(arestrict), mean(arestrict), $
	median(arestrict),stddev(arestrict)
print,format='(%"e:",f10.6,f10.6,f10.6,f10.6,f10.6)', $
	min(erestrict),max(erestrict), mean(erestrict), $
	median(erestrict),stddev(erestrict)
print,format='(%"i:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(irestrict),max(irestrict), mean(irestrict), $
	median(irestrict),stddev(irestrict)
print,format='(%"T:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(Trestrict),max(Trestrict),mean(Trestrict), $
	median(Trestrict),stddev(Trestrict)
print,format='(%"W:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(W_caprestrict),max(W_caprestrict),mean(W_caprestrict), $
	median(W_caprestrict),stddev(W_caprestrict)
print,format='(%"w:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(w_lowrestrict),max(w_lowrestrict),mean(w_lowrestrict), $
	median(w_lowrestrict),stddev(w_lowrestrict)

end

