pro ephemeris, data_file, num

; Plot the PA and sep over the next 50 years at one-year intervals.
; Start with 2003.0
;
; Enter P,T,e for each solution
; Program does a weighted least squares fit to determine Theile-Innes
; elements A,B,F,G (and hence a,i,Omega,omega)
; 
; Equations:  
; Orbital elements (P,T,e) and a set of observations (ti,xi,yi) are known
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)
; The four Thiele-Innes elements are solved through linear least squares fit:
;	xi = AXi + FYi
;	yi = BXi + GYi
; Calculate geometric elements a("),i,Omega,omega from (A,F,B,G)
;
; Essentially, the equations one needs to solve are:
;	E - esinE = 2Pi/P * (t - T)
;	x = A*(cos(Ei) - e) + F*(sqrt(1 - e^2)*sin(Ei))
;	y = B*(cos(Ei) - e) + G*(sqrt(1 - e^2)*sin(Ei))
;
; Paramers:
;	Period:	period (P)
;	Tperi: 	time of periastron passage (T)
;	ecc:	eccentricity (e)
;	major:	semi-major axis (in mas) (a)
;	inc:	inclination (i)
;	Omega:	position angle of node
;	omega:	angle between node and periastron
;	ti:	time of observation
;	xd:	RA - data
;	yd:	DEC - data
;	xf:	RA - fit (determined from orbital elements for time of obs.)
;	yf:	DEC - fit (determined from orbital elements for time of obs.)
;	A,B,F,G: Thiele-Innes elements
;
; Note: xi and yi are not the same convention for RA and DEC chosen to make 
; the orbital plots. Specifically, in the fitting the apparent ellipse 
; (ORBFITE.PRO):
;	x = RA = -rho*sin(theta),   y = DEC = rho*cos(theta)
;	(i.e. North is in the direction of the positive y axis, theta is
;	measured counterclockwise from positive y-axis)
; Whereas in this program, the coordinate system set up by Couteau,
;	x = rho*cos(theta),   y = rho*sin(theta)
;	(i.e. North id in the direction of the positive x-axis and the 
;	y-axis points east)
; Both coordinate systems have the same separation and position angle 
; orietations, they are just rotated by 90 deg with respect to each other.
;
; INPUT:
;	data_file: file containing data points in the format:
;		      time  theta  error(theta)  rho  error(rho)
;		      where theta is the position angle and rho the separation
;		      and their respective errors
;		      (enter file name using quotes)
;	num: number of data points contained in data_file
;	
; Prompted inputs:
;	all of the orbital elements: P,T,e
;
; OUTPUT: 
;	plot of data points and orbital solutions
;
; Modifications to be added:
; 	- error bars on data points
;	- line connecting measurement to claculated point


; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time = dblarr(num)
theta = dblarr(num)
rho = dblarr(num)
dtheta = dblarr(num)
drho = dblarr(num)

openr,lun,data_file,/get_lun

for i=0, num-1 do begin

	readf, lun, temp1, temp2, temp3, temp4, temp5

	time(i) = temp1
	theta(i) = temp2
	dtheta(i) = temp3
	rho(i) = temp4	
	drho(i) = temp5
	
endfor

close,lun

; convert theta and dtheta to radians

theta = theta*!dpi/180
dtheta = dtheta*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(theta)^2*drho^2 + rho^2*sin(theta)^2*dtheta^2)
dy = sqrt(sin(theta)^2*drho^2 + rho^2*cos(theta)^2*dtheta^2)

print, "Do you want to save plot as a postscript file?  (Type yes or no):"
pans = ""
read,pans
; Set plotting device to ps if answer is yes
if (pans eq "yes") then begin
	mydevice=!d.name
	set_plot, 'ps'
	device, /Color, Bits_per_pixel=8, filename='temp.ps' 
endif

answer = "yes"		; answer for plotting another orbital solution
count = 0
	
period = 0d
Tperi = 0d
ecc = 0d

; Define user symbol to be closed circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)

;Define the symbol to be a unit circle with 16 points
usersym, 0.5*cos(Z), 0.5*sin(Z), /fill

while (answer eq "yes") do begin

	; Obtain initial values for P,T,e

	print,"Enter P T e (period, time of periastron passage, eccentricity):"
	read, period, Tperi, ecc

	; degrees of freedom
	dof = 2*num - 7

	; Determine fitted points (xfit,yfit) by computing the Thiele Innes
	; elements A,F,B,G

	calc_AFBG, time, xarr, dx, yarr, dy, period, Tperi, ecc, $
		   A_cap, F_cap, B_cap, G_cap, xfit, yfit

	; Determine chi squared

	chi2 = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

	; reduced chi squared:
	chi2red = chi2/dof

	; determine a,i,Omega,omega (using AIWW.PRO)

	aiWw, A_cap,B_cap,F_cap,G_cap,major,inc,W_cap,w_low

	print, "a i Omega omega:", major, (180/!dpi)*inc, $
	(180/!dpi)*W_cap, (180/!dpi)*w_low

	EL = [period,Tperi,ecc,major,inc,W_cap,w_low]

	print, "chi2", chi2
	print, "A B F G:", A_cap,B_cap,F_cap,G_cap
	print, "P T e a i Omega omega"
	print,EL(0),El(1),EL(2),EL(3),180*EL(4)/!dpi,180*EL(5)/!dpi, $
		180*EL(6)/!dpi

	; determine the O-C residuals (Observed - Calculated) for each data pt

	xdiff = xarr - xfit
	ydiff = yarr - yfit

;	print, "O-C residuals (Observed - Calculated):"
;	print, "time xdata xfit e(x) xdif"
;	for i=0, num-1 do print, time(i), xarr(i), xfit(i), dx(i), xdiff(i)

;	print, "time ydata yfit e(y) ydiff"
;	for i=0, num-1 do print, time(i), yarr(i), yfit(i), dy(i), ydiff(i)


	; compute ephemeris over the next 10 years, starting with 2003.0 
	; at one-year intervals

	pnum = 50

	pfac = 1.0

	; determine x,y locations at yearly intervals along orbit

	xorb = dblarr(pnum+1)
	yorb = dblarr(pnum+1)
	torb = dblarr(pnum+1)
	sepfit = dblarr(pnum+1)
	PAfit = dblarr(pnum+1)

	torb(0) = 2003.0

	print, "Period (years):", period
	print, "Date PA sep:"

	for i=0, pnum do begin

		; Determine the eccentric anomalies Ei:
		; (reduce to same epoch)

		Morb = 2*!dpi*((torb(i) - Tperi)/period $ 
			- fix((torb(i) - Tperi)/period))

		solve_trans,ecc,Morb,Eorb

		; Normalized rectangular coordinates Xi & Yi:
		;	Xi = cos(Ei) - e
		;	Yi = sqrt(1 - e^2)*sin(Ei)
	
		Xiorb = cos(Eorb) - ecc
		Yiorb = sqrt(1 - ecc^2)*sin(Eorb)
		
		xorb(i) = A_cap*Xiorb + F_cap*Yiorb	;fitted points
		yorb(i) = B_cap*Xiorb + G_cap*Yiorb
	
		if (i lt pnum) then torb(i+1) = torb(i) + 1*pfac

		; convert (xorb,yorb) to (PAfit,sepfit)
		; x = rho*sin(theta), y=rho*sin(theta)

		sepfit(i) = sqrt(xorb(i)^2 + yorb(i)^2)

		; Get correct sign on PA by entering y,x separately into atan:
		PAfit(i) = atan(yorb(i),xorb(i))

		if (PAfit(i) lt 0.0) then PAfit(i) = 2*!dpi + PAfit(i)

;		print, torb(i), PAfit(i)*180d/!dpi, 6.0, sepfit(i), 5.4
		print, torb(i), PAfit(i)*180d/!dpi, 1.3, sepfit(i), 2.0

	endfor

	;plot data and fitted orbit

	if (count eq 0) then begin

		; axes will be rotated by +90 deg (clockwise direction) inorder
		; to align N-S sep with +y and E-W sep with +x
		; transformation (new coordinates are x',y')
		; 	x' = -y
		;	y' = +x

		maxx = max(-yorb)
		minx = min(-yorb)
		maxy = max(xorb)
		miny = min(xorb)

		ex = 0d
		print, "x: max min ", maxx,minx
		print, "y: max min ", maxy,miny
		print, "By what value would you like to extend x and y axes?"
		read, ex

		plot_title=""
		print, "Enter plot title:"
		read, plot_title

		plot, -yarr, xarr, psym=8, XSTYLE=1, YSTYLE=1, ISOTROPIC=1, $
			xrange=[minx-ex,maxx+ex], yrange=[miny-ex,maxy+ex], $
			xtitle="E-W sep (mas)", ytitle="N-S sep (mas)", $
			title = plot_title

		plot_err,-yarr,abs(dy),xarr,abs(dx),num

		count = count + 1
	endif

	oplot, -yorb, xorb, psym=3

	print, "Would you like to plot another orbit (yes/no)?"
	read, answer

endwhile

; draw axes through origin

plotS, [minx-ex,maxx+ex],[0,0], LineStyle=1
plotS, [0,0],[miny-ex,maxy+ex], LineStyle=1

; plot primary as an asterisk

oplot, [0],[0],psym=2

; set plotting device back to terminal if previously saving to ps file
if (pans eq "yes") then begin 
	device, /close
	set_plot, mydevice
endif

free_lun,lun

end


pro plot_err, x, dx, y, dy, N

; plot error bars associated with each data point
; Input
;	x: array of x values
;	dx: array of error in x values
;	y: array of y values
;	dy: erray of error in y values
;	N: number of data points


for i=0, N-1 do begin

	oplot, [x(i)-dx(i), x(i)+dx(i)], [y(i), y(i)]
	oplot, [x(i), x(i)], [y(i)-dy(i), y(i)+dy(i)]

endfor

end

