pro ephemeris_par

; Plot the PA and sep over a specified number of years into the future.
;
; Enter P,T,e,a,i,Omega,omega for each solution
; Enter starting year and number of year into the future from which to
; compute the ephemeris.
; 
; Equations:  
; Orbital elements (P,T,e) and a set of observations (ti,xi,yi) are known
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)
; The four Thiele-Innes elements are solved through linear least squares fit:
;	xi = AXi + FYi
;	yi = BXi + GYi
; Calculate geometric elements a("),i,Omega,omega from (A,F,B,G)
;
; Essentially, the equations one needs to solve are:
;	E - esinE = 2Pi/P * (t - T)
;	x = A*(cos(Ei) - e) + F*(sqrt(1 - e^2)*sin(Ei))
;	y = B*(cos(Ei) - e) + G*(sqrt(1 - e^2)*sin(Ei))
;
; Paramers:
;	Period:	period (P)
;	Tperi: 	time of periastron passage (T)
;	ecc:	eccentricity (e)
;	major:	semi-major axis (in mas) (a)
;	inc:	inclination (i)
;	Omega:	position angle of node
;	omega:	angle between node and periastron
;	ti:	time of observation
;	xd:	RA - data
;	yd:	DEC - data
;	xf:	RA - fit (determined from orbital elements for time of obs.)
;	yf:	DEC - fit (determined from orbital elements for time of obs.)
;	A,B,F,G: Thiele-Innes elements
;
; Note: xi and yi are not the same convention for RA and DEC chosen to make 
; the orbital plots. Specifically, in the fitting the apparent ellipse 
; (ORBFITE.PRO):
;	x = RA = -rho*sin(theta),   y = DEC = rho*cos(theta)
;	(i.e. North is in the direction of the positive y axis, theta is
;	measured counterclockwise from positive y-axis)
; Whereas in this program, the coordinate system set up by Couteau,
;	x = rho*cos(theta),   y = rho*sin(theta)
;	(i.e. North id in the direction of the positive x-axis and the 
;	y-axis points east)
; Both coordinate systems have the same separation and position angle 
; orietations, they are just rotated by 90 deg with respect to each other.
;
; INPUT:
;       P,T,e,a,i,Omega,omega
;       tstart and trange
;	
; Prompted inputs:
;	all of the orbital elements: P,T,e
;
; OUTPUT: 
;	plot of data points and orbital solutions
;
; Modifications to be added:
; 	- error bars on data points
;	- line connecting measurement to claculated point
;
; 2006 July 19: Modify EPHEMERIS.PRO to work with entire set of 
;     orbital elements (P,T,e,a,i,Omega,omega).  This avoids needing
;     to input data file for the least-squares fitting.

!P.charsize=1.0
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.0	;scale factor for adjusting size of plotting symbols


; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1

period = 0d
Tperi = 0d
ecc = 0d
major = 0d
inc = 0d
W_cap = 0d
w_low = 0d

; Obtain initial values for P,T,e,a,i,W,w

print,"Enter P T e a i W w (period, time of periastron passage, eccentricity):"
read, period, Tperi, ecc, major, inc, W_cap, w_low

;convert angles from degrees to radians

inc = inc*!dpi/180.0
W_cap = W_cap*!dpi/180.0
w_low = w_low*!dpi/180.0

EL = [period, Tperi, ecc, major, inc, W_cap, w_low]

tstart = 0d
trange = 0d

print,'Enter time you wish to start ephemeris at:'
read,tstart
print,'Enter number of years into future you want to compute ephemeris:'
read,trange

; Compute ephemeris at one year intervals

tstep = 1.0
;tstep = 5.0
tnum = trange/tstep + 1.0
time = findgen(tnum)*tstep + tstart

calc_vbfit, EL, time, PA, sep

xarr = sep*cos(PA)	; x coordinate
yarr = sep*sin(PA)	; y coordinate

print, "Period (years):", period
print, "Date PA sep:"

for i=0,tnum-1 do print,time(i),PA(i)*180d/!dpi,1.0,sep(i),2.0

plot_title=""
print, "Enter plot title:"
read, plot_title

; Define user symbol to be closed circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)

;Define the symbol to be a unit circle with 16 points
usersym, frac*cos(Z), frac*sin(Z), /fill

print, "Do you want to save plot as a postscript file?  (Type yes or no):"
pans = ""
read,pans
; Set plotting device to ps if answer is yes
if (pans eq "yes") then begin
	set_plot, 'ps'
	device, /Color, Bits_per_pixel=8, filename='temp.ps' 
endif

		maxx = max(yarr)
		minx = min(yarr)
		maxy = max(xarr)
		miny = min(xarr)

		ex = 0d
		print, "x: max min ", maxx,minx
		print, "y: max min ", maxy,miny
		print, "By what value would you like to extend x and y axes?"
		read, ex

plot, yarr, xarr, psym=8, XSTYLE=1, YSTYLE=1, ISOTROPIC=1, $
	xrange=[maxx+ex,minx-ex], yrange=[miny-ex,maxy+ex], $
	xtitle="E-W sep (mas)", ytitle="N-S sep (mas)", $
	title = plot_title

zl = (maxx - maxy)
xyouts,yarr+0.2*zl, xarr+0.1*zl,fix(time-tstart),alignment=0.5,charsize=0.8

; set plotting device back to terminal if previously saving to ps file
if (pans eq "yes") then begin 
	device, /close
	set_plot, 'x'
endif

end
