pro fit_orbit_vb_astrom, vb_file, astrom_file

; Compute the orbital elements through a Newton-Raphson technique.
; Fits simultaneously for VB + astrometric photocenter (a1) orbit
;
; Calls the following routines:
;     calc_deriv_vb.pro
;     calc_deriv_sb1.pro
;     calc_Ei.pro
;     solve_trans.pro
;     calc_vbfit.pro - for plotting model orbits!
;     calc_sb1fit.pro - for plotting model orbits!
;
; Equations:
; + Visual Binary +  
; Initial estimates (P0,T0,e0,a0,i0,Omega0,omega0) and a set of observations 
; (ti,rhoi,thetai) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Radius vector:
;	r = a(1-e^2)/(1 + ecos(nu))
; The fitted (rho_i,theta_i) positions are determined from the orbital elements
; 	tan(theta - Omega) = tan(nu + omega)*cosi
;	rho = r*cos(nu + omega)*sec(theta - Omega)
; + Spectroscopic Binary +  
; Initial estimates (P0,T0,e0,K1|0,omega0,Vsys0) and  
; a set of observations (ti,V1i) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Projected semi-major axis (a1sini):
;       a1sini = (P/2pi)*K1*(1 - e^2)^1/2
; The fitted velocities (V1_i) are determined from the orbital elements:
;       V1 = K1*[e*cos(omega) + cos(nu + omega)] + Vsys
;
; Procedure:
; Minimize chi squared between data point positions and fitted positions:
;	chi^2 = sum[(rhodata - rhoi)^2/sigmarho^2 
;		    + (thetadata - thetai)^2/sigmatheta^2]
;             + sum[(V1data - V1fit)^2/sigmaV1^2]
; Take partial derivatives of chi^2 with respect to the orbital elements,
; (P,T,e,a(mas),i,Omega,omega,K1(km/s),Vsys) and set to zero. Solve for the 
; value of the orbital elements that minimize chi^2.
;
; Since the partial derivatives are non-linear, cannot solve for the orbital
; elements that minimize chi^2 analytically.
;
; In turn, use a Newton-Raphson technique to converge upon the solution.
; Replace (rhoi,thetai) in the chi^2 equation with the Taylor series 
; approximation,
; 	x = x| + dx/dP|(P-P0) + dx/dT|(T-T0) + dx/de|(e-e0) 
;	       + dx/da|(a-Aa) + dx/di|(i-i0) + dx/dW|(W-W0) + dx/dw|(w-w0)
;	       + dx/d(K1)|(K1 - K1|0) + dx/d(Vsys)|(Vsys - Vsys0)
; where x=(rho,theta,V1) and the | represents the value evaluated at for the
; initial estimates for the orbital elements. Now, rhofit, thetafit, and V1 are
; linear in (P,T,e,a,i,W,w,K1,K2,Vsys), so the partial derivatives can easily
; be taken and solved by setting up a matrix and using Cramer's method.
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;	major:	 semi-major axis (in mas) (a)
;	inc:	 inclination (i)
;	Omega:	 position angle of node (W)
;	omega:	 angle between node and periastron (w)
;       K1:      velocity amplitude of primary (in km/s) (K1)
;       Vsys:    systemic velocity
;	t_vb:	 time of observation
;	rho_d:	 separation - data
;	theta_d: postion angle - data
;	rho_f:	 sep - fit (determined from orbital elements for time of obs.)
;	theta_f: PA - fit (determined from orbital elements for time of obs.)
;	t_sb1:	 time of observation
;	V1_d:	 velocity of primary - measured data
;	V1_f:	 V1 - fit (determined from orbital elements for time of obs.)
;
; INPUT:
;	vb_file: file containing data points in the format:
;		      time  theta  etheta  rho  erho
;		      where theta is the position angle, rho is the separation
;                     and etheta, erho are the measurement uncertainties
;		      (enter file name using quotes)
;	vb_num: number of data points contained in data_file
;	
;	sb1_file: file containing data points in the format:
;		      time  V1  errV1
;		      where V1 is the radial velocity of the primary
;                     and errV1 is the measurement error
;		      (enter file name using quotes)
;	sb1_num: number of data points contained in data_file
;
; Prompted inputs:
;	initial estimates for all the orbital elements:
;             P, T, e, a, i, Omega, omega, K1, Vsys
;       vector identifying which orbital elements to solve for
;             enter 0 to hold element fixed, 1 to vary)
;             ex.   1 1 1 1 1 1 1 1 1 -> solves for all
;                   0 0 0 0 1 0 0 0 0 -> solves only for i
;
; OUTPUT:
;	best fit values of orbital elements: P,T,e,a,i,Omega,omega,K1,Vsys
;       and corresponding errors
;
; Notes on units:
;    P, T, and time should be in same units (days or years)
;    K1 and V1 should be in same units (km/s)
;
; Began 14 May 2002
; Modified 11 Nov 2003: 
;       This is the current version of NEWT_RAPH.PRO
;	(v1.1 and v1.2 are older versions)
; 	- restructured program into separate subroutines
; 	- add option of holding elements fixed during iterations
; 13 January 2005:
;       Fix bugs
;       - if solving for only one element, don't use cramer.pro to solve eqn's
;       - fix 360 degree roll overs in (PAdata - PAfit)
;       Save version v4 as backup
; 11 July 2005:
;       - Include Marquadt Method of adjusting diagonal elements of
;         the alpha matrix to ensure convergence.
;       - Fixed non-convergence problems 
;           + do not remove P wrap around effects from dx/dP
;             leave (t-T)/P in full form; net effect is that adjustments
;             are smaller
;       Save v5 as backup 
;
; 31 May 2006: Save unmodified program as version 2
; 31 May 2006: Fit SB1 parameters in terms of K1
;              (as opposed to a1sini)
;
; 22 September 2006:
;       - Fixed the way subsets of parameters are handled
;       - Wasn't properly constructing matrices if not all parameters vary
;
; 30 January 2020:
;       - Add flag_wa to calc_deriv_vb_ell.pro and calc_vbfit.pro.  This allows
;         the option of defining omega = omega_A in both the SB2 and VB orbits.

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

vb_num = 10000
astrom_num = 10000

time_vb = dblarr(vb_num)
theta = dblarr(vb_num)
rho = dblarr(vb_num)
dtheta = dblarr(vb_num)
drho = dblarr(vb_num)

time_astrom = dblarr(astrom_num)
theta_astrom = dblarr(astrom_num)
rho_astrom = dblarr(astrom_num)
dtheta_astrom = dblarr(astrom_num)
drho_astrom = dblarr(astrom_num)

; Read VB measurements

openr,1,vb_file

i = 0
while not EOF(1) do begin

	readf, 1, temp1, temp2, temp3, temp4, temp5

	time_vb(i) = temp1
	theta(i) = temp2
	dtheta(i) = temp3
	rho(i) = temp4	
	drho(i) = temp5

        i = i+1
endwhile
close,1
vb_num = i
time_vb = time_vb(0:vb_num-1)
theta = theta(0:vb_num-1)
dtheta = dtheta(0:vb_num-1)
rho = rho(0:vb_num-1)
drho = drho (0:vb_num-1)

; convert theta and dtheta to radians

theta = theta*!dpi/180
dtheta = dtheta*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(theta)^2*drho^2 + rho^2*sin(theta)^2*dtheta^2)
dy = sqrt(sin(theta)^2*drho^2 + rho^2*cos(theta)^2*dtheta^2)

; Read in astrometric photocenter measurements for primary wobble

openr,1,astrom_file

i = 0
while not EOF(1) do begin

	readf, 1, temp1, temp2, temp3, temp4, temp5

	time_astrom(i) = temp1
	theta_astrom(i) = temp2
	dtheta_astrom(i) = temp3
	rho_astrom(i) = temp4	
	drho_astrom(i) = temp5
	
        i = i+1
endwhile
close,1
astrom_num = i
time_astrom   = time_astrom(0:astrom_num-1)
theta_astrom  = theta_astrom (0:astrom_num-1)
dtheta_astrom = dtheta_astrom(0:astrom_num-1)
rho_astrom    = rho_astrom(0:astrom_num-1)   	
drho_astrom  =  drho_astrom(0:astrom_num-1)  

; convert theta and dtheta to radians

theta_astrom = theta_astrom*!dpi/180
dtheta_astrom = dtheta_astrom*!dpi/180

; convert data points to x and y coordinates

xarr_astrom = rho_astrom * cos(theta_astrom)	; x coordinate
yarr_astrom = rho_astrom * sin(theta_astrom)	; y coordinate

; propagate errors in sep and PA to x and y:

dx_astrom = sqrt(cos(theta_astrom)^2*drho_astrom^2 + rho_astrom^2*sin(theta_astrom)^2*dtheta_astrom^2)
dy_astrom = sqrt(sin(theta_astrom)^2*drho_astrom^2 + rho_astrom^2*cos(theta_astrom)^2*dtheta_astrom^2)


; Obtain values for P,T,e,a(mas),i,Omega,omega,a1sini(Gm),Vsys

period = 0d
Tperi = 0d
ecc = 0d
major = 0d
inc = 0d
W_cap = 0d
w_low = 0d
a1 = 0d

print,"Enter P,T,e,a(mas),i,Omega,omega,a1:"
read,period,Tperi,ecc,major,inc,W_cap,w_low,a1

print, "Vary each orbital element?"
print, "For each element, enter 0 to hold element fixed, 1 to vary:"
print, "[P,T,e,a,i,Omega,omega,a1]"
read, f0,f1,f2,f3,f4,f5,f6,f7

;; Order of orbital element arrays
;; do not set here yet (need to get angles into radians)
;; EL_vb = [period,Tperi,ecc,major,inc,W_cap,w_low_B]
;;             0    1     2    3    4    5     6
;; EL_vb = [period,Tperi,ecc,a1,inc,W_cap,w_low_A]
;;             0    1     2   7  4    5     6

elfix=[f0,f1,f2,f3,f4,f5,f6,f7]
nEl = n_elements(elfix)

elfix_vb=[f0,f1,f2,f3,f4,f5,f6]
nEl_vb = n_elements(elfix_vb)

elfix_astrom=[f0,f1,f2,f7,f4,f5,f6]
nEl_astrom = n_elements(elfix_vb)

;convert i, Omega, omega to radians
inc = inc*!dpi/180
W_cap = W_cap*!dpi/180
w_low = w_low*!dpi/180

k=0
for i=0, nEl-1 do if (elfix(i) eq 0) then k=k+1
mfit = nEl - k 		; number of elements to improve

k=0
for i=0, nEl_vb-1 do if (elfix_vb(i) eq 0) then k=k+1
mfit_vb = nEl_vb - k 		; number of elements to improve

k=0
for i=0, nEl_astrom-1 do if (elfix_astrom(i) eq 0) then k=k+1
mfit_astrom = nEl_astrom - k 		; number of elements to improve

EL = [period,Tperi,ecc,major,inc,W_cap,w_low,a1]
EL_vb = [period,Tperi,ecc,major,inc,W_cap,w_low]
EL_astrom = [period,Tperi,ecc,a1,inc,W_cap,w_low]

ELadj = EL
elLabel = ['P','T','e','a(mas)','i','Omega','omega_A','a1']

; determine which indices of full set are being varied

vb_subset = dblarr(mfit_vb)
astrom_subset = dblarr(mfit_astrom)

; determine subarray of indices that are VB, SB1, and SB2 parameters

vb_par = dblarr(nEl_vb)
astrom_par = dblarr(nEl_astrom)

; determine indices of elements for which to vary
el_subset = where(elfix ne 0)
Elvar = El(el_subset)

k=0
for i=0, nEl_vb-1 do begin
    ind = where(EL_vb(i) eq EL)
    vb_par(i) = ind
    if (elfix_vb(i) ne 0) then begin
        ind = where(EL_vb(i) eq Elvar)
        vb_subset(k) = ind
        k=k+1
    endif
endfor

k=0
for i=0, nEl_astrom-1 do begin
    ind = where(EL_astrom(i) eq EL)
    astrom_par(i) = ind
    if (elfix_astrom(i) ne 0) then begin
        ind = where(EL_astrom(i) eq Elvar)
        astrom_subset(k) = ind
        k=k+1
    endif
endfor


; **** Plot model orbits ****

ans_cont = ' '

; plot model and data of initial guess for visual orbit

tnum=1000.0
tstep = period/tnum
tmin = Tperi

tarr = findgen(tnum)*tstep + tmin

calc_vbfit, EL_vb, tarr, theta_mod, rho_mod, /flag_wa

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

print,'Visual orbit'
plot,-yarr,xarr,psym=6
oplot,-ymod,xmod

print,'Hit return to continue'
read,ans_cont

; plot model and data of initial guess for astrom orbit

tnum=1000.0
tstep = period/tnum
tmin = Tperi

tarr = findgen(tnum)*tstep + tmin

calc_vbfit, EL_astrom, tarr, theta_mod, rho_mod

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

print,'Astrometric orbit'
plot,-yarr_astrom,xarr_astrom,psym=6
oplot,-ymod,xmod

print,'Hit return to continue'
read,ans_cont

count = 0

delta_chi = 1.0		; set initially to begin loop

lambda = 0.001   ; Marquardt method to ensure convergence

!P.MULTI = [0,2,0]

; ***** Begin while loop ***** 

while(delta_chi gt 0.001) do begin

    ; Determine errors in orbital elements
    ; set up matrices for error determination
    ; Invert matrix through Gauss Jordan elimination (Numerical Recipes in C)

    ; set up covariance matrix and column matrix
    ; alpha = dblarr(7,7)
    ; beta = dblarr(7)

    calc_deriv_vb, EL_vb, elfix_vb, mfit_vb, time_vb, theta, rho, dtheta, drho, theta_f, rho_f, alpha_vb, beta_vb, /flag_wa

    calc_deriv_vb, EL_astrom, elfix_astrom, mfit_astrom, time_astrom, theta_astrom, rho_astrom, dtheta_astrom, drho_astrom, theta_astrom_f, rho_astrom_f, alpha_astrom, beta_astrom

    ; Determine chi squared

    diff_theta = theta - theta_f
    diff_theta_astrom = theta_astrom - theta_astrom_f

    ; account for any wrap-arounds from 360 deg to 0 deg

    for k=0, vb_num-1 do begin
        if (diff_theta(k) ge !dpi) then $
          diff_theta(k) = diff_theta(k) - 2*!dpi
        if (diff_theta(k) le -!dpi) then $
          diff_theta(k) = 2*!dpi + diff_theta(k)	
    endfor

    ; account for any wrap-arounds from 360 deg to 0 deg

    for k=0, astrom_num-1 do begin
        if (diff_theta_astrom(k) ge !dpi) then $
          diff_theta_astrom(k) = diff_theta_astrom(k) - 2*!dpi
        if (diff_theta_astrom(k) le -!dpi) then $
          diff_theta_astrom(k) = 2*!dpi + diff_theta_astrom(k)	
    endfor

    chi2old = total((diff_theta)^2/dtheta^2 + (rho - rho_f)^2/drho^2) $
            + total((diff_theta_astrom)^2/dtheta_astrom^2 $
                  + (rho_astrom - rho_astrom_f)^2/drho_astrom^2)

    ; combine SB1 and VB matrices...

    ; initialize full alpha,beta arrays
    alpha = dblarr(mfit,mfit)
    beta = dblarr(mfit)

    beta(vb_subset) = beta(vb_subset) + beta_vb
    beta(astrom_subset) = beta(astrom_subset) + beta_astrom

    for i=0, mfit_vb-1 do begin
        for j=0, mfit_vb-1 do begin
            alpha(vb_subset(i),vb_subset(j)) = $
              alpha(vb_subset(i),vb_subset(j)) + alpha_vb(i,j)
        endfor
    endfor

    for i=0, mfit_astrom-1 do begin
        for j=0, mfit_astrom-1 do begin
            alpha(astrom_subset(i),astrom_subset(j)) = $
              alpha(astrom_subset(i),astrom_subset(j)) + alpha_astrom(i,j)
        endfor
    endfor

    invmat = invert(alpha, stat, /double)

    print,"stat (0 successful):",stat

    ;determine errors:

    ELerr = dblarr(mfit)

    for i=0, mfit-1 do ELerr(i) = sqrt(invmat(i,i))

    ; adjust alpha matrix by Marquardt parameter lambda

    for i=0, mfit-1 do alpha(i,i) = alpha(i,i)*(1.0 + lambda)

    if (mfit eq 1) then begin
        delta_el = dblarr(1)
        delta_el(0) = beta/alpha
    endif else delta_el = cramer(alpha,beta)   ; adjustments

    ; adjust orbital parameters

    ELadj = EL
    ELadj(el_subset) = ELadj(el_subset) + delta_el 

    ELadj_vb  = ELadj(vb_par) 
    ELadj_astrom = ELadj(astrom_par) 

    k=0
    print,"Orbital Element     Adjustment"
    for i=0, 3 do begin
        if (elfix(i) eq 0) then begin
            print, elLabel(i), ': ', El(i)
        endif else begin
            print, elLabel(i), ': ', El(i), delta_el(k)
            k=k+1
        endelse
    endfor
    for i=4, 6 do begin
         if (elfix(i) eq 0) then begin
             print, elLabel(i), ': ', El(i)*180/!dpi
         endif else begin
             print, elLabel(i), ': ', El(i)*180/!dpi, delta_el(k)*180/!dpi
             k = k+1
         endelse
    endfor
    for i=7, 7 do begin
        if (elfix(i) eq 0) then begin
            print, elLabel(i), ': ', El(i)
        endif else begin
            print, elLabel(i), ': ', El(i), delta_el(k)
            k=k+1
        endelse
    endfor

    print,"Chi2 of current solution:",chi2old

    ; convert data points to x and y coordinates

    xfit = rho_f * cos(theta_f)	; x coordinate
    yfit = rho_f * sin(theta_f)	; y coordinate

    ; plot visual orbit

    plot,-yarr,xarr,psym=6
    oplot,-yfit,xfit,psym=7
    for l=0, vb_num-1 do oplot,[-yarr(l),-yfit(l)],[xarr(l),xfit(l)]

    ; convert data points to x and y coordinates

    xfit_astrom = rho_astrom_f * cos(theta_astrom_f)	; x coordinate
    yfit_astrom = rho_astrom_f * sin(theta_astrom_f)	; y coordinate

    ; plot visual orbit

    plot,-yarr_astrom,xarr_astrom,psym=6
    oplot,-yfit_astrom,xfit_astrom,psym=7
    for l=0, vb_num-1 do oplot,[-yarr_astrom(l),-yfit_astrom(l)],[xarr_astrom(l),xfit_astrom(l)]

    wait,0.1

    if (count eq 0) then begin
        print,'Hit return to continue'
        read,ans_cont
    endif

    ; calculate new chi2 for adjusted parameters

    calc_deriv_vb, ELadj_vb, elfix_vb, mfit_vb, time_vb, theta, rho, dtheta, drho, theta_f, rho_f, alpha_vb, beta_vb, /flag_wa

    calc_deriv_vb, ELadj_astrom, elfix_astrom, mfit_astrom, time_astrom, theta_astrom, rho_astrom, dtheta_astrom, drho_astrom, theta_astrom_f, rho_astrom_f, alpha_astrom, beta_astrom

    ; Determine chi squared

    diff_theta = theta - theta_f
    diff_theta_astrom = theta_astrom - theta_astrom_f

    ; account for any wrap-arounds from 360 deg to 0 deg

    for k=0, vb_num-1 do begin
        if (diff_theta(k) ge !dpi) then $
          diff_theta(k) = diff_theta(k) - 2*!dpi
        if (diff_theta(k) le -!dpi) then $
          diff_theta(k) = 2*!dpi + diff_theta(k)	
    endfor

    for k=0, astrom_num-1 do begin
        if (diff_theta_astrom(k) ge !dpi) then $
          diff_theta_astrom(k) = diff_theta_astrom(k) - 2*!dpi
        if (diff_theta_astrom(k) le -!dpi) then $
          diff_theta_astrom(k) = 2*!dpi + diff_theta_astrom(k)	
    endfor

    chi2new = total((diff_theta)^2/dtheta^2 + (rho - rho_f)^2/drho^2) $
            + total((diff_theta_astrom)^2/dtheta_astrom^2 $
                  + (rho_astrom - rho_astrom_f)^2/drho_astrom^2)


    print,"chi2 of next modification:",chi2new

    if (chi2new le chi2old) then begin
        EL = ELadj
        EL_vb = ELadj_vb
        EL_astrom = ELadj_astrom
	chi2 = chi2new
        delta_chi = chi2old - chi2new
        lambda = lambda/10.0
        print, 'lambda', lambda
    endif else begin
	print, "next adjustment does not improve chi2"
        print, "increasing lambda by 10.0"
        lambda = lambda*10.0
        print, 'lambda', lambda
        delta_chi = 1.0
    endelse

    count = count+1

    ; do not exit unless lambda is less than 0.001

;    if (lambda gt 0.001) and (delta_chi lt 0.001) then delta_chi = 1.0
;    print,'lambda',lambda
;    print,'delta_chi',delta_chi

endwhile

lambda = 0.0

print, "Final values:"

period = EL(0)
Tperi = EL(1)
ecc = EL(2)
major =  EL(3)
inc = EL(4)
W_cap = EL(5)
w_low = EL(6)
a1 = EL(7)

; Determine final error matrix:

calc_deriv_vb, EL_vb, elfix_vb, mfit_vb, time_vb, theta, rho, dtheta, drho, theta_f, rho_f, alpha_vb, beta_vb, /flag_wa

calc_deriv_vb, EL_astrom, elfix_astrom, mfit_astrom, time_astrom, theta_astrom, rho_astrom, dtheta_astrom, drho_astrom, theta_astrom_f, rho_astrom_f, alpha_astrom, beta_astrom

; combine SB1 and VB matrices...

; initialize full alpha,beta arrays
alpha = dblarr(mfit,mfit)
beta = dblarr(mfit)

beta(vb_subset) = beta(vb_subset) + beta_vb
beta(astrom_subset) = beta(astrom_subset) + beta_astrom

for i=0, mfit_vb-1 do begin
    for j=0, mfit_vb-1 do begin
        alpha(vb_subset(i),vb_subset(j)) = $
          alpha(vb_subset(i),vb_subset(j)) + alpha_vb(i,j)
    endfor
endfor

for i=0, mfit_astrom-1 do begin
    for j=0, mfit_astrom-1 do begin
        alpha(astrom_subset(i),astrom_subset(j)) = $
          alpha(astrom_subset(i),astrom_subset(j)) + alpha_astrom(i,j)
    endfor
endfor

;determine errors:

invmat = invert(alpha, stat, /double)

diff_theta = theta - theta_f
diff_theta_astrom = theta_astrom - theta_astrom_f

; account for any wrap-arounds from 360 deg to 0 deg

for k=0, vb_num-1 do begin
if (diff_theta(k) ge !dpi) then $
	diff_theta(k) = diff_theta(k) - 2*!dpi
if (diff_theta(k) le -!dpi) then $
	diff_theta(k) = 2*!dpi + diff_theta(k)	
endfor

for k=0, astrom_num-1 do begin
if (diff_theta_astrom(k) ge !dpi) then $
	diff_theta_astrom(k) = diff_theta_astrom(k) - 2*!dpi
if (diff_theta_astrom(k) le -!dpi) then $
	diff_theta_astrom(k) = 2*!dpi + diff_theta_astrom(k)	
endfor

chi2 = total((diff_theta)^2/dtheta^2 + (rho - rho_f)^2/drho^2) $
     + total((diff_theta_astrom)^2/dtheta_astrom^2 $
           + (rho_astrom - rho_astrom_f)^2/drho_astrom^2)

chi2_vb = total((diff_theta)^2/dtheta^2 + (rho - rho_f)^2/drho^2)

chi2_astrom = total((diff_theta_astrom)^2/dtheta_astrom^2 $
                  + (rho_astrom - rho_astrom_f)^2/drho_astrom^2)

;; convert data points to x and y coordinates
;xfit = rho_f * cos(theta_f)	; x coordinate
;yfit = rho_f * sin(theta_f)	; y coordinate
;chi2xy = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

; degrees of freedom
dof = 2d*vb_num + 2d*astrom_num - mfit
dof_vb = 2d*vb_num - mfit_vb
dof_astrom = 2d*astrom_num - mfit_astrom

; reduced chi squared:
chi2red = chi2/dof

chi2red_vb = chi2_vb/dof_vb
chi2red_astrom = chi2_astrom/dof_astrom

ELerr = dblarr(mfit)
for i=0,mfit-1 do ELerr(i) = sqrt(chi2red)*sqrt(invmat(i,i))

k=0
print,"Orbital Element     Error"
for i=0, 3 do begin
    if (elfix(i) eq 0) then begin
        print, elLabel(i), ': ', El(i)
    endif else begin
        print, elLabel(i), ': ', El(i), ELerr(k)
        k=k+1
    endelse
endfor
for i=4, 6 do begin
     if (elfix(i) eq 0) then begin
         print, elLabel(i), ': ', El(i)*180/!dpi
     endif else begin
         print, elLabel(i), ': ', El(i)*180/!dpi, ELerr(k)*180/!dpi
         k = k+1
     endelse
endfor
for i=7,7 do begin
    if (elfix(i) eq 0) then begin
        print, elLabel(i), ': ', El(i)
    endif else begin
        print, elLabel(i), ': ', El(i), ELerr(k)
        k=k+1
    endelse
endfor
print, "Final chi2:", chi2
print, "Reduced chi2:", chi2red
print, 'Degrees of freedom: ', dof
print, 'Number of VB data points:     ',vb_num 
print, 'Number of ASTROM data points: ', astrom_num
print, 'VB only chi2/redchi2/dof:     ',chi2_vb,chi2red_vb,dof_vb
print, 'ASTROM only chi2/redchi2/dof: ',chi2_astrom,chi2red_astrom,dof_astrom
print,'Number of iterations:',count
;print,'Final chi2:',chi2

print,EL

!P.MULTI = [0,0,0]

; plot best-fit orbit and data for visual orbit

tnum=1000.0
tstep = period/tnum
tmin = Tperi

tarr = findgen(tnum)*tstep + tmin

calc_vbfit, EL_vb, tarr, theta_mod, rho_mod, /flag_wa

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

; plot data and best-fit positions for visual orbit
print,'Visual orbit'
plot,-yarr,xarr,psym=6
oplot,-yfit,xfit,psym=7
for l=0, vb_num-1 do oplot,[-yarr(l),-yfit(l)],[xarr(l),xfit(l)]

; plot best-fit orbit for visual orbit
oplot,-ymod,xmod

set_plot,'ps'
device, /Color, Bits_per_pixel=8, filename='temp_vb.ps'
; plot data and best-fit positions for visual orbit
plot,-yarr,xarr,psym=6
oplot,-yfit,xfit,psym=7
for l=0, vb_num-1 do oplot,[-yarr(l),-yfit(l)],[xarr(l),xfit(l)]
; plot best-fit orbit for visual orbit
oplot,-ymod,xmod
device,/close
set_plot,'x'



print,'Hit return to continue'
read,ans_cont

; plot best-fit orbit and data for astrometric a1 orbit

tnum=1000.0
tstep = period/tnum
tmin = Tperi

tarr = findgen(tnum)*tstep + tmin

calc_vbfit, EL_astrom, tarr, theta_mod, rho_mod

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

; plot data and best-fit positions for visual orbit
print,'Astrometric orbit'
plot,-yarr_astrom,xarr_astrom,psym=6
oplot,-yfit_astrom,xfit_astrom,psym=7
for l=0, astrom_num-1 do oplot,[-yarr_astrom(l),-yfit_astrom(l)],[xarr_astrom(l),xfit_astrom(l)]

; plot best-fit orbit for visual orbit
oplot,-ymod,xmod

set_plot,'ps'
device, /Color, Bits_per_pixel=8, filename='temp_astrom.ps'
; plot data and best-fit positions for visual orbit
plot,-yarr_astrom,xarr_astrom,psym=6
oplot,-yfit_astrom,xfit_astrom,psym=7
for l=0, vb_num-1 do oplot,[-yarr_astrom(l),-yfit_astrom(l)],[xarr_astrom(l),xfit_astrom(l)]
; plot best-fit orbit for visual orbit
oplot,-ymod,xmod
device,/close
set_plot,'x'

end
