pro fit_orbit_vbpm, vb_file, vb_num, pm_file

; Compute the orbital elements through a Newton-Raphson technique.
; Fits simultaneously for VB + delta(proper motion)
; Includes comparison of FK5 and HIP proper motions
;
; Calls the following routines:
;     calc_deriv_vb.pro
;     calc_deriv_pm.pro
;     calc_Ei.pro
;     solve_trans.pro
;     auto_ephemeris.pro - for computing predicted diff in pm
;     calc_vbfit.pro - for plotting model orbits!
;
; Equations:
; + Visual Binary +  
; Initial estimates (P0,T0,e0,a0,i0,Omega0,omega0) and a set of observations 
; (ti,rhoi,thetai) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Radius vector:
;	r = a(1-e^2)/(1 + ecos(nu))
; The fitted (rho_i,theta_i) positions are determined from the orbital elements
; 	tan(theta - Omega) = tan(nu + omega)*cosi
;	rho = r*cos(nu + omega)*sec(theta - Omega)
; + Proper Motion data +


;
; Procedure:
; Minimize chi squared between data point positions and fitted positions:
;	chi^2 = sum[(rhodata - rhoi)^2/sigmarho^2 
;		    + (thetadata - thetai)^2/sigmatheta^2]
;             + sum[(V1data - V1fit)^2/sigmaV1^2]
; Take partial derivatives of chi^2 with respect to the orbital elements,
; (P,T,e,a(mas),i,Omega,omega,K1(km/s),Vsys) and set to zero. Solve for the 
; value of the orbital elements that minimize chi^2.
;
; Since the partial derivatives are non-linear, cannot solve for the orbital
; elements that minimize chi^2 analytically.
;
; In turn, use a Newton-Raphson technique to converge upon the solution.
; Replace (rhoi,thetai) in the chi^2 equation with the Taylor series 
; approximation,
; 	x = x| + dx/dP|(P-P0) + dx/dT|(T-T0) + dx/de|(e-e0) 
;	       + dx/da|(a-Aa) + dx/di|(i-i0) + dx/dW|(W-W0) + dx/dw|(w-w0)
;	       + dx/d(K1)|(K1 - K1|0) + dx/d(Vsys)|(Vsys - Vsys0)
; where x=(rho,theta) and the | represents the value evaluated at for the
; initial estimates for the orbital elements. Now, rhoi and thetai are linear 
; in (P,T,e,a,i,W,w,K1,K2,Vsys), so the partial derivatives can easily be
; taken and solved by setting up a matrix and using Cramer's method.
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;	major:	 semi-major axis (in mas) (a)
;	inc:	 inclination (i)
;	Omega:	 position angle of node (W)
;	omega:	 angle between node and periastron (w)
;       K1:      velocity amplitude of K1 in km/s  (K1)
;       a1sini:  projected semi-major axis of primary from SB1 in physcial
;                units (assumed constant - used to calculate aphsini)
;       aphsini  projected semi-major axis of the photocenter
;                fully calculated value - do not adjust (unless fitting SB1)
;	t_vb:	 time of observation
;	rho_d:	 separation - data
;	theta_d: postion angle - data
;	rho_f:	 sep - fit (determined from orbital elements for time of obs.)
;	theta_f: PA - fit (determined from orbital elements for time of obs.)
;       t_hip:    time of Hipparcos observation
;       trange:  time span of Hipparcos observation
;       tstep:   step size used to integrate over Hipparcos time frame
;	dpm_ra_d:  difference between HIP (instantaneous) and  
;                  FK5 proper motion (cms) [ra*cos(dec) in mas/yr] 
;	dpm_dec_d: difference between HIP (instantaneous) and  
;                  FK5 proper motion (cms) [mas/yr]
;       dpm_ra_f:  fit-value of delta_pm based on orbital parameters and date
;                  of Hipparcos observation
;       dpm_dec_f: fit-value of delta_pm based on orbital parameters and date
;                  of Hipparcos observation 
;
; INPUT:
;	vb_file: file containing data points in the format:
;		      time  theta  etheta  rho  erho
;		      where theta is the position angle, rho is the separation
;                     and etheta, erho are the measurement uncertainties
;		      (enter file name using quotes)
;	vb_num: number of data points contained in data_file
;	
;       pm_file:  file containing details of proper motion data
;                     t_hip  trange_ave  tstep_ave
;                     pm_ra_hip   pm_dec_hip
;                     pm_ra_fk5   pm_dec_fk5
;                 where the pm in ra is pm_ra*cos(dec)
;                 and the FK5 pm hvae been converted to the HIP frame
;
; Prompted inputs:
;	initial estimates for all the orbital elements:
;             P, T, e, a, i, Omega, omega
;       vector identifying which orbital elements to solve for
;             enter 0 to hold element fixed, 1 to vary)
;             ex.   1 1 1 1 1 1 1  -> solves for all
;                   0 0 0 0 1 0 0 -> solves only for i
;       a1(AU)sini, parallax, and delta_mag - used to calculate aphsini
;
; OUTPUT:
;	best fit values of orbital elements: P,T,e,a,i,Omega,omega,K1,Vsys
;       and corresponding errors
;
; Notes on units:
;    P, T, and time should be in same units (days or years)
;    parallax should be entered as mas
;    K1 should be entered as km/s
;    a should be entered in same units as separation measurements ("/mas)
;
; Began 14 May 2002
; Modified 11 Nov 2003: 
;       This is the current version of NEWT_RAPH.PRO
;	(v1.1 and v1.2 are older versions)
; 	- restructured program into separate subroutines
; 	- add option of holding elements fixed during iterations
; 13 January 2005:
;       Fix bugs
;       - if solving for only one element, don't use cramer.pro to solve eqn's
;       - fix 360 degree roll overs in (PAdata - PAfit)
;       Save version v4 as backup
; 11 July 2005:
;       - Include Marquadt Method of adjusting diagonal elements of
;         the alpha matrix to ensure convergence.
;       - Fixed non-convergence problems 
;           + do not remove P wrap around effects from dx/dP
;             leave (t-T)/P in full form; net effect is that adjustments
;             are smaller
;       Save v5 as backup 
;
; 21 September 2006: Begin fit_orbit_vbpm.pro
;
; 22 September 2006:
;       - Fixed the way subsets of parameters are handled
;       - Wasn't properly constructing matrices if not all parameters vary

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time_vb = dblarr(vb_num)
theta = dblarr(vb_num)
rho = dblarr(vb_num)
dtheta = dblarr(vb_num)
drho = dblarr(vb_num)

time_hip = 0.0d
trange_ave = 0.0d
tstep_ave  = 0.0d
pm_ra_hip  = 0.0d
pm_dec_hip = 0.0d
pm_ra_fk5  = 0.0d
pm_dec_fk5 = 0.0d
err_ra_hip  = 0.0d
err_dec_hip = 0.0d
err_ra_fk5  = 0.0d
err_dec_fk5 = 0.0d


; Read VB measurements

openr,lun,vb_file,/get_lun

for i=0, vb_num-1 do begin

	readf, lun, temp1, temp2, temp3, temp4, temp5

	time_vb(i) = temp1
	theta(i) = temp2
	dtheta(i) = temp3
	rho(i) = temp4	
	drho(i) = temp5
	
endfor

close,lun

; convert theta and dtheta to radians

theta = theta*!dpi/180
dtheta = dtheta*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(theta)^2*drho^2 + rho^2*sin(theta)^2*dtheta^2)
dy = sqrt(sin(theta)^2*drho^2 + rho^2*cos(theta)^2*dtheta^2)

; Read proper motion data
pm_num = 1.0    ; (one measurement of pm_ra and pm_dec at T_hip)

openr,lun,pm_file
readf, lun, time_hip, trange_ave, tstep_ave
readf, lun, pm_ra_hip, err_ra_hip, pm_dec_hip, err_dec_hip
readf, lun, pm_ra_fk5, err_ra_fk5, pm_dec_fk5, err_dec_fk5
close,lun

dpm_ra_d  = pm_ra_hip - pm_ra_fk5
dpm_dec_d = pm_dec_hip - pm_dec_fk5

err_ra  = sqrt(err_ra_hip^2 + err_ra_fk5^2)
err_dec = sqrt(err_dec_hip^2 + err_dec_fk5^2)

; Obtain values for P,T,e,a(mas),i,Omega,omega,K1

period = 0d
Tperi = 0d
ecc = 0d
major = 0d
inc = 0d
W_cap = 0d
w_low = 0d
K1 = 0.0d

print,'Enter P,T,e,a("),i,Omega,omega,K1(km/s):'
read,period,Tperi,ecc,major,inc,W_cap,w_low,K1

;convert i, Omega, omega to radians
inc = inc*!dpi/180
W_cap = W_cap*!dpi/180
w_low = w_low*!dpi/180

print, "Vary each orbital element?"
print, "For each element, enter 0 to hold element fixed, 1 to vary:"
print, "[P,T,e,a,i,Omega,omega,K1]"
read, f0,f1,f2,f3,f4,f5,f6,f7

; Obtain values for parallax and dmag

par = 0.0d
dmag = 0.0d

print,'Enter parallax (mas) and dmag:'
read,par,dmag

ansyd=' '
print,'Is period in years or days (y/d)?'
read,ansyd
if (ansyd eq 'y') then y_d = 1.0d
if (ansyd eq 'd') then y_d = 365.25d

ansam=' '
print,'Are separation measurements in arcsec or mas (a/m)?'
read,ansam
if (ansam eq 'a') then a_m = 1000.0d
if (ansam eq 'm') then a_m = 1.0d

; compute masses
; K in km/s, P in yrs, M in Msun, a in "
a1sini = 0.033573182d * K1*(period/y_d)*sqrt(1.0 - ecc^2)
Mtot = (major*a_m)^3/par^3/(period/y_d)^2
M2 = 0.033573182d * (major*a_m)^2/par^2*K1/(period/y_d)/sin(inc)*sqrt(1.0 - ecc^2)
M1 = (major*a_m)^3/par^3/(period/y_d)^2 - M2

print,'a1sini (AU):',a1sini
print,'M1 (Msun)',M1
print,'M2 (Msun)',M2
print,'Mtot (Msun)',Mtot

; compute mass and light ratios
mratio = M2/(M1 + M2)
Lratio = 1.0d/(1.0 + 10^(0.4*dmag))

print,'mratio:',mratio
print,'Lratio:',Lratio
print,'1 - L/M',1.0 - Lratio/mratio

; Compute aphsini
aphsini = a1sini*par*(1.0 - Lratio/mratio)
aph = aphsini/sin(inc)

print,'aphsini (mas):', aphsini
print,'aph (mas):', aph


; Time intervals used to compute predicted difference in proper motions
; averaged across Hipparcos epoch'
tnum_ave = trange_ave/tstep_ave + 1.0
time_ave = findgen(tnum_ave)*tstep_ave + time_hip - trange_ave/2.0 

; compute predicted difference in proper motion
; (note: Wph = W - 180.0)
astromEL = [period,Tperi,ecc,aph,inc,W_cap-!dpi,w_low]
auto_ephemeris,astromEL,time_ave,RAsep,DECsep,/noprint

RAsep_ave = total(RAsep)*tstep_ave/trange_ave
DECsep_ave = total(DECsep)*tstep_ave/trange_ave

dpm_ra_f = 12.0*total(RAsep*(time_ave - time_hip))*tstep_ave/trange_ave^3
dpm_dec_f = 12.0*total(DECsep*(time_ave - time_hip))*tstep_ave/trange_ave^3

print,'Measured dpm_ra:',dpm_ra_d,err_ra
print,'Measured dpm_dec:',dpm_dec_d,err_dec

print,'Time_HIP  RAsep_ave  DECsep_ave:'
print,time_hip,RAsep_ave,DECsep_ave
print, 'Predicted dpm_ra:',dpm_ra_f
print, 'Predicted dpm_dec:',dpm_dec_f

;; Order of orbital element arrays
;; do not set here yet (need to get angles into radians)
;; EL_vb = [period,Tperi,ecc,major,inc,W_cap,w_low]
;;             0    1     2    3    4    5     6
;; EL_pm = [period,Tperi,ecc,major,inc,W_cap,w_low,K1]
;;             0    1     2    3    4    5     6    7

elfix=[f0,f1,f2,f3,f4,f5,f6,f7]
nEl = n_elements(elfix)

elfix_vb=[f0,f1,f2,f3,f4,f5,f6]
nEl_vb = n_elements(elfix_vb)

elfix_pm=[f0,f1,f2,f3,f4,f5,f6,f7]
nEl_pm = n_elements(elfix_pm)

k=0
for i=0, nEl-1 do if (elfix(i) eq 0) then k=k+1
mfit = nEl - k 		; number of elements to improve

k=0
for i=0, nEl_vb-1 do if (elfix_vb(i) eq 0) then k=k+1
mfit_vb = nEl_vb - k 		; number of elements to improve

k=0
for i=0, nEl_pm-1 do if (elfix_pm(i) eq 0) then k=k+1
mfit_pm = nEl_pm - k 		; number of elements to improve

EL    = [period,Tperi,ecc,major,inc,W_cap,w_low,K1]
EL_vb = [period,Tperi,ecc,major,inc,W_cap,w_low]
EL_pm = [period,Tperi,ecc,major,inc,W_cap,w_low,K1]

pm_conv = [par,dmag,y_d,a_m]

ELadj = EL
elLabel = ['P','T','e','a(")','i','Omega','omega','K1(km/s)']

; determine which indices of full set are being varied

vb_subset = dblarr(mfit_vb)
pm_subset = dblarr(mfit_pm)

; determine subarray of indices that are VB, SB1, and SB2 parameters

vb_par = dblarr(nEl_vb)
pm_par = dblarr(nEl_pm)

; determine indices of elements for which to vary
el_subset = where(elfix ne 0)
Elvar = El(el_subset)

k=0
for i=0, nEl_vb-1 do begin
    ind = where(EL_vb(i) eq EL)
    vb_par(i) = ind
    if (elfix_vb(i) ne 0) then begin
        ind = where(EL_vb(i) eq Elvar)
        vb_subset(k) = ind
        k=k+1
    endif
endfor

k=0
for i=0, nEl_pm-1 do begin
    ind = where(EL_pm(i) eq EL)
    pm_par(i) = ind
    if (elfix_pm(i) ne 0) then begin
         ind = where(EL_pm(i) eq Elvar)
         pm_subset(k) = ind
        k=k+1
    endif
endfor

; **** Plot model orbits ****

ans_cont = ' '

; plot model and data of initial guess for visual orbit

tnum=1000.0
tstep = period/tnum
tmin = Tperi

tarr = findgen(tnum)*tstep + tmin

calc_vbfit, EL_vb, tarr, theta_mod, rho_mod

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

print,'Visual orbit'
plot,-yarr,xarr,psym=6
oplot,-ymod,xmod

print,'Hit return to continue'
read,ans_cont

count = 0

delta_chi = 1.0		; set initially to begin loop

lambda = 0.001   ; Marquardt method to ensure convergence

!P.MULTI = [0,2,0]

; ***** Begin while loop ***** 

while(delta_chi gt 0.001) do begin

    ; Determine errors in orbital elements
    ; set up matrices for error determination
    ; Invert matrix through Gauss Jordan elimination (Numerical Recipes in C)

    ; set up covariance matrix and column matrix
    ; alpha = dblarr(7,7)
    ; beta = dblarr(7)

    calc_deriv_vb, EL_vb, elfix_vb, mfit_vb, time_vb, theta, rho, dtheta, drho, theta_f, rho_f, alpha_vb, beta_vb

    calc_deriv_pm, EL_pm, elfix_pm, mfit_pm, pm_conv,time_hip, trange_ave, tstep_ave, dpm_ra_d, dpm_dec_d, err_ra, err_dec, dpm_ra_f, dpm_dec_f, alpha_pm, beta_pm

    ; Determine chi squared

    diff_theta = theta - theta_f

    ; account for any wrap-arounds from 360 deg to 0 deg

    for k=0, vb_num-1 do begin
        if (diff_theta(k) ge !dpi) then $
          diff_theta(k) = diff_theta(k) - 2*!dpi
        if (diff_theta(k) le -!dpi) then $
          diff_theta(k) = 2*!dpi + diff_theta(k)	
    endfor

    chi2old = total((diff_theta)^2/dtheta^2 + (rho - rho_f)^2/drho^2) $
            + total((dpm_ra_d - dpm_ra_f)^2/err_ra^2 $
                   + (dpm_dec_d - dpm_dec_f)^2/err_dec^2)
 
;    print, 'Predicted dpm_ra:',dpm_ra_f
;    print, 'Predicted dpm_dec:',dpm_dec_f
;    print, 'chi2:',chi2old
;    print,'Hit return to continue'
;    read,ans_cont

    ; combine PM and VB matrices...

    ; initialize full alpha,beta arrays
    alpha = dblarr(mfit,mfit)
    beta = dblarr(mfit)

    beta(vb_subset) = beta(vb_subset) + beta_vb
    beta(pm_subset) = beta(pm_subset) + beta_pm

    for i=0, mfit_vb-1 do begin
        for j=0, mfit_vb-1 do begin
            alpha(vb_subset(i),vb_subset(j)) = $
              alpha(vb_subset(i),vb_subset(j)) + alpha_vb(i,j)
        endfor
    endfor

    for i=0, mfit_pm-1 do begin
        for j=0, mfit_pm-1 do begin
            alpha(pm_subset(i),pm_subset(j)) = $
              alpha(pm_subset(i),pm_subset(j)) + alpha_pm(i,j)
        endfor
    endfor

    invmat = invert(alpha, stat, /double)

    print,"stat (0 successful):",stat

    ;determine errors:

    ELerr = dblarr(mfit)

    for i=0, mfit-1 do ELerr(i) = sqrt(invmat(i,i))

    ; adjust alpha matrix by Marquardt parameter lambda

    for i=0, mfit-1 do alpha(i,i) = alpha(i,i)*(1.0 + lambda)

    if (mfit eq 1) then begin
        delta_el = dblarr(1)
        delta_el(0) = beta/alpha
    endif else delta_el = cramer(alpha,beta)   ; adjustments

    ; adjust orbital parameters

    ELadj = EL
    ELadj(el_subset) = ELadj(el_subset) + delta_el 

    ELadj_vb = ELadj(vb_par) 
    ELadj_pm = ELadj(pm_par) 

    k=0
    print,"Orbital Element     Adjustment"
    for i=0, 3 do begin
        if (elfix(i) eq 0) then begin
            print, elLabel(i), ': ', El(i)
        endif else begin
            print, elLabel(i), ': ', El(i), delta_el(k)
            k=k+1
        endelse
    endfor
    for i=4, 6 do begin
         if (elfix(i) eq 0) then begin
             print, elLabel(i), ': ', El(i)*180/!dpi
         endif else begin
             print, elLabel(i), ': ', El(i)*180/!dpi, delta_el(k)*180/!dpi
             k = k+1
         endelse
    endfor
    for i=7, 7 do begin
        if (elfix(i) eq 0) then begin
            print, elLabel(i), ': ', El(i)
        endif else begin
            print, elLabel(i), ': ', El(i), delta_el(k)
            k=k+1
        endelse
    endfor

    print,"Chi2 of current solution:",chi2old

    ; convert data points to x and y coordinates

    xfit = rho_f * cos(theta_f)	; x coordinate
    yfit = rho_f * sin(theta_f)	; y coordinate

    ; plot visual orbit

    plot,-yarr,xarr,psym=6
    oplot,-yfit,xfit,psym=7
    for l=0, vb_num-1 do oplot,[-yarr(l),-yfit(l)],[xarr(l),xfit(l)]

    wait,0.1

    if (count eq 0) then begin
        print,'Hit return to continue'
        read,ans_cont
    endif

    ; calculate new chi2 for adjusted parameters

    calc_deriv_vb, ELadj_vb, elfix_vb, mfit_vb, time_vb, theta, rho, dtheta, drho, theta_f, rho_f, alpha_vb, beta_vb

    calc_deriv_pm, ELadj_pm, elfix_pm, mfit_pm, pm_conv,time_hip, trange_ave, tstep_ave, dpm_ra_d, dpm_dec_d, err_ra, err_dec, dpm_ra_f, dpm_dec_f, alpha_pm, beta_pm

    ; Determine chi squared

    diff_theta = theta - theta_f

    ; account for any wrap-arounds from 360 deg to 0 deg

    for k=0, vb_num-1 do begin
        if (diff_theta(k) ge !dpi) then $
          diff_theta(k) = diff_theta(k) - 2*!dpi
        if (diff_theta(k) le -!dpi) then $
          diff_theta(k) = 2*!dpi + diff_theta(k)	
    endfor

    chi2new = total((diff_theta)^2/dtheta^2 + (rho - rho_f)^2/drho^2) $
            + total((dpm_ra_d - dpm_ra_f)^2/err_ra^2 $
                   + (dpm_dec_d - dpm_dec_f)^2/err_dec^2)

    print,"chi2 of next modification:",chi2new

;    print, 'Predicted dpm_ra:',dpm_ra_f
;    print, 'Predicted dpm_dec:',dpm_dec_f
;    print, 'chi2:',chi2new
;    print,'Hit return to continue'
;    read,ans_cont

    if (chi2new le chi2old) then begin
        EL = ELadj
        EL_vb = ELadj_vb
        EL_pm = ELadj_pm
	chi2 = chi2new
        delta_chi = chi2old - chi2new
        lambda = lambda/10.0
        print, 'lambda', lambda
    endif else begin
	print, "next adjustment does not improve chi2"
        print, "increasing lambda by 10.0"
        lambda = lambda*10.0
        print, 'lambda', lambda
        delta_chi = 1.0
    endelse

    count = count+1

    ; do not exit unless lambda is less than 0.001

;    if (lambda gt 0.001) and (delta_chi lt 0.001) then delta_chi = 1.0
;    print,'lambda',lambda
;    print,'delta_chi',delta_chi

endwhile

lambda = 0.0

print, "Final values:"

period = EL(0)
Tperi = EL(1)
ecc = EL(2)
major =  EL(3)
inc = EL(4)
W_cap = EL(5)
w_low = EL(6)
K1  = EL(7)

; Determine final error matrix:

calc_deriv_vb, EL_vb, elfix_vb, mfit_vb, time_vb, theta, rho, dtheta, drho, theta_f, rho_f, alpha_vb, beta_vb

calc_deriv_pm, EL_pm, elfix_pm, mfit_pm, pm_conv,time_hip, trange_ave, tstep_ave, dpm_ra_d, dpm_dec_d, err_ra, err_dec, dpm_ra_f, dpm_dec_f, alpha_pm, beta_pm

; combine SB1 and VB matrices...

; initialize full alpha,beta arrays
alpha = dblarr(mfit,mfit)
beta = dblarr(mfit)

beta(vb_subset) = beta(vb_subset) + beta_vb
beta(pm_subset) = beta(pm_subset) + beta_pm

for i=0, mfit_vb-1 do begin
    for j=0, mfit_vb-1 do begin
        alpha(vb_subset(i),vb_subset(j)) = $
          alpha(vb_subset(i),vb_subset(j)) + alpha_vb(i,j)
    endfor
endfor

for i=0, mfit_pm-1 do begin
    for j=0, mfit_pm-1 do begin
        alpha(pm_subset(i),pm_subset(j)) = $
          alpha(pm_subset(i),pm_subset(j)) + alpha_pm(i,j)
    endfor
endfor

;determine errors:

invmat = invert(alpha, stat, /double)

diff_theta = theta - theta_f

; account for any wrap-arounds from 360 deg to 0 deg

for k=0, vb_num-1 do begin
if (diff_theta(k) ge !dpi) then $
	diff_theta(k) = diff_theta(k) - 2*!dpi
if (diff_theta(k) le -!dpi) then $
	diff_theta(k) = 2*!dpi + diff_theta(k)	
endfor

chi2 = total((diff_theta)^2/dtheta^2 + (rho - rho_f)^2/drho^2) $
            + total((dpm_ra_d - dpm_ra_f)^2/err_ra^2 $
                   + (dpm_dec_d - dpm_dec_f)^2/err_dec^2)

;    print, 'Predicted dpm_ra:',dpm_ra_f
;    print, 'Predicted dpm_dec:',dpm_dec_f
;    print, 'chi2:',chi2
;    print,'Hit return to continue'
;    read,ans_cont

;; convert data points to x and y coordinates
;xfit = rho_f * cos(theta_f)	; x coordinate
;yfit = rho_f * sin(theta_f)	; y coordinate
;chi2xy = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

; degrees of freedom
dof = 2d*vb_num + 2d*pm_num - mfit
if (dof le 0.0) then begin
    dof = 1.0d
    print,'NOTE: dof < 0; setting to 1'
endif

; reduced chi squared:
chi2red = chi2/dof

ELerr = dblarr(mfit)
for i=0,mfit-1 do ELerr(i) = sqrt(chi2red)*sqrt(invmat(i,i))

k=0
print,"Orbital Element     Error"
for i=0, 3 do begin
    if (elfix(i) eq 0) then begin
        print, elLabel(i), ': ', El(i)
    endif else begin
        print, elLabel(i), ': ', El(i), ELerr(k)
        k=k+1
    endelse
endfor
for i=4, 6 do begin
     if (elfix(i) eq 0) then begin
         print, elLabel(i), ': ', El(i)*180/!dpi
     endif else begin
         print, elLabel(i), ': ', El(i)*180/!dpi, ELerr(k)*180/!dpi
         k = k+1
     endelse
endfor
for i=7,7 do begin
    if (elfix(i) eq 0) then begin
        print, elLabel(i), ': ', El(i)
    endif else begin
        print, elLabel(i), ': ', El(i), ELerr(k)
        k=k+1
    endelse
endfor
print, "Final chi2:", chi2
print, "Reduced chi2:", chi2red
print,'Number of iterations:',count
;print,'Final chi2:',chi2
; compute masses
; K in km/s, P in yrs, M in Msun, a in "
a1sini = 0.033573182d * K1*(period/y_d)*sqrt(1.0 - ecc^2)
Mtot = (major*a_m)^3/par^3/(period/y_d)^2
M2 = 0.033573182d * (major*a_m)^2/par^2*K1/(period/y_d)/sin(inc)*sqrt(1.0 - ecc^2)
M1 = (major*a_m)^3/par^3/(period/y_d)^2 - M2

;print,period,major,ecc,inc*180.0/!dpi,par

; compute mass and light ratios
mratio = M2/(M1 + M2)
Lratio = 1.0d/(1.0 + 10^(0.4*dmag))
; Compute aphsini
aphsini = a1sini*par*(1.0 - Lratio/mratio)
aph = aphsini/sin(inc)
; compute predicted difference in proper motion
; (note: Wph = W - 180.0)
astromEL = [period,Tperi,ecc,aph,inc,W_cap-!dpi,w_low]
auto_ephemeris,astromEL,time_ave,RAsep,DECsep,/noprint
dpm_ra_f = 12.0*total(RAsep*(time_ave - time_hip))*tstep_ave/trange_ave^3
dpm_dec_f = 12.0*total(DECsep*(time_ave - time_hip))*tstep_ave/trange_ave^3
print,'aphsini (mas):',aphsini
print,'aph (mas):',aph
print,'M1 (Msun)',M1
print,'M2 (Msun)',M2
print,'Mtot (Msun)',Mtot
print, 'Final Predicted dpm_ra:',dpm_ra_f
print, 'Final Predicted dpm_dec:',dpm_dec_f
;print,'time_hip trange_ave tstep_ave',time_hip, trange_ave, tstep_ave
;print,'help,time_ave'
;help,time_ave

;stop,'Type .cont to continue'

print,EL*[1.0,1.0,1.0,1.0,180/!dpi,180/!dpi,180/!dpi]

!P.MULTI = [0,0,0]

; plot best-fit orbit and data for visual orbit

tnum=1000.0
tstep = period/tnum
tmin = Tperi

tarr = findgen(tnum)*tstep + tmin

calc_vbfit, EL_vb, tarr, theta_mod, rho_mod

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

; plot data and best-fit positions for visual orbit
print,'Visual orbit'
plot,-yarr,xarr,psym=6
oplot,-yfit,xfit,psym=7
for l=0, vb_num-1 do oplot,[-yarr(l),-yfit(l)],[xarr(l),xfit(l)]

; plot best-fit orbit for visual orbit
oplot,-ymod,xmod

set_plot,'ps'
device, /Color, Bits_per_pixel=8, filename='temp_vb.ps'
; plot data and best-fit positions for visual orbit
plot,-yarr,xarr,psym=6
oplot,-yfit,xfit,psym=7
for l=0, vb_num-1 do oplot,[-yarr(l),-yfit(l)],[xarr(l),xfit(l)]
; plot best-fit orbit for visual orbit
oplot,-ymod,xmod
device,/close
set_plot,'x'

;print,'Hit return to continue'
;read,ans_cont


free_lun,lun

end
