pro fit_orbit_vbsb1_ell_bootstrap_replicate, vb_file, vb_num, sb1_file, sb1_num, mcrit=mcrit, noboot=noboot
  
; Compute the orbital elements through a Newton-Raphson technique.
; Fits simultaneously for VB + SB1 orbits
;
; This version can handle elliptical uncertainties for visual orbits.  
; Minimize chi2 based on projections along the direction of error 
; ellipse (x',y').
;
; Add bootstrap sampling to compute errors
;   - Randomly sample observations (with repeatition) and apply
;     Gaussian uncertainties.
;   - Refit orbit and iterate for Nboot samples.
;   - Compute uncertainties based on bootstrap distributions
;
; For each iteration, compute the orbital elements through a
; Newton-Raphson technique.
;
; For Polaris observations, add a "replicate" step to even out
; weighting between small number of VB measurements and large
; number of SB measurements. The replication method follow this approach
; 1. Use bootstrap technique to resample measurements and apply Gaussian
;    uncertainties.
; 2. Replicate each measurement in the resampled array Nrepl times to
;    generate a swarm of Nrepl x vbnum observables for each measurement.
; 3. Fit an orbit to the bottstrapped + replicated arrays. Apply to VB
;    measurements only (sepRA, sepDEC). Fit RV measurements directly
;    as is.
;
; Options:
;   /noboot - do not bootstrap observations
;           - replicate measurements only - multiple times
;
; Calls the following routines:
;     calc_deriv_vb_ell.pro
;     calc_deriv_sb1.pro
;     calc_Ei.pro
;     solve_trans.pro
;     calc_sb1fit.pro - for plotting model orbits
;     calc_vbfit.pro - for plotting model orbits
;
; Equations:
; + Visual Binary +  
; Initial estimates (P0,T0,e0,a0,i0,Omega0,omega0) and a set of observations 
; (ti,rhoi,thetai) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Radius vector:
;	r = a(1-e^2)/(1 + ecos(nu))
; Fitted positions (rho_fit,theta_fit) are determined from the orbital elements
; 	tan(theta - Omega) = tan(nu + omega)*cosi
;	rho = r*cos(nu + omega)*sec(theta - Omega)
; Project the separation into x',y' components in the direction of the
; error ellipse:
;       xp = rho*cos(theta - theta_err)
;       yp = rho*sin(theta - theta_err)
; + Spectroscopic Binary +  
; Initial estimates (P0,T0,e0,omega0,K1|0,K2|0,Vsys0) and  
; a set of observations (ti,V1i,V2i) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Projected semi-major axes (a1sini, a2sini):
;       a1sini = (P/2pi)*K1*(1 - e^2)^1/2
;       a2sini = (P/2pi)*K2*(1 - e^2)^1/2
; The fitted velocities (V1_i, V2_i) are determined from the orbital elements:
;       V1 =  K1*[e*cos(omega) + cos(nu + omega)] + Vsys
;       V2 = -K2*[e*cos(omega) + cos(nu + omega)] + Vsys
;
; Procedure:
; Minimize chi squared between data point positions and fitted positions:
;	chi^2 = sum[(xp_data - xp_fit)^2/sigma_major^2 
;		    + (yp_data - yp_fit)^2/sigma_minor^2]
;             + sum[(V1data - V1fit)^2/sigmaV1^2]
;		    + (V2data - V2fit)^2/sigmaV2^2]
; Take partial derivatives of chi^2 with respect to the orbital elements,
; (P,T,e,a(mas),i,Omega,omega,K1,K2,Vsys) and set to zero. 
; Solve for the  value of the orbital elements that minimize chi^2.
;
; Since the partial derivatives are non-linear, cannot solve for the orbital
; elements that minimize chi^2 analytically.
;
; In turn, use a Newton-Raphson technique to converge upon the solution.
; Replace (xpfit,ypfit,V1fit,V2fit) in the chi^2 equation with the
; Taylor series approximation,
; 	x = x| + dx/dP|(P-P0) + dx/dT|(T-T0) + dx/de|(e-e0) 
;	       + dx/da|(a-Aa) + dx/di|(i-i0) + dx/dW|(W-W0) + dx/dw|(w-w0)
;	       + dx/d(K1)|(K1 - K1|0) + dx/d(K2)|(K2 - K2|0)
;              + dx/d(Vsys)|(Vsys - Vsys0)
; where x=(xp,yp,V1,V2) and the | represents the value evaluated at for the
; initial estimates for the orbital elements. Now, xp, yp, V1, and V2 
; are linear in (P,T,e,a,i,W,w,K1,K2,Vsys), so the partial derivatives can 
; easily be taken and solved by setting up a matrix and using Cramer's method.
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;	major:	 semi-major axis (in mas) (a)
;	inc:	 inclination (i)
;	Omega:	 position angle of node (W)
;	omega:	 angle between node and periastron (w)
;       K1:      velocity amplitude of primary (in km/s) (K1)
;       K2:      velocity amplitude of secondary (in km/s) (K2)
;       Vsys:    systemic velocity
;	t_vb:	 time of observation
;	rho_d:	 separation - data
;	theta_d: postion angle - data
;	rho_f:	 sep - fit (determined from orbital elements for time of obs.)
;	theta_f: PA - fit (determined from orbital elements for time of obs.)
;	t_sb1:	 time of observation
;	V1_d:	 velocity of primary - measured data
;	V1_f:	 V1 - fit (determined from orbital elements for time of obs.)
;
; INPUT:
;	vb_file: file containing data points in the format:
;		      time  PA  sep  dmajor  dminor  theta_err
;		    where theta is the position angle and rho is the separation.
;                   The error ellipse is defined by the size of the major and 
;                   minor axes, and the position angle of the major axis in the
;                   plane of the sky (measured East of North).
;		    (enter file name using quotes)
;	vb_num: number of data points contained in data_file
;	
;	sb1_file: file containing data points in the format:
;		      time  V1  errV1  V2  errV2
;		      where V1 is the radial velocity of the primary
;		            V2 is the radial velocity of the secondary
;                     and errV1 and errV2 are the measurement errors
;		      (enter file name using quotes)
;	sb1_num: number of data points contained in data_file
;
; Prompted inputs:
;	initial estimates for all the orbital elements:
;             P, T, e, a, i, Omega, omega, K1, K2, Vsys
;       vector identifying which orbital elements to solve for
;             enter 0 to hold element fixed, 1 to vary)
;             ex.   1 1 1 1 1 1 1 1 1 1 -> solves for all
;                   0 0 0 0 1 0 0 0 0 0 -> solves only for i
;
; OUTPUT:
;	best fit values of orbital elements: P,T,e,a,i,Omega,omega,K1,K2,Vsys
;       and corresponding errors
;
; Notes on units:
;    P, T, and time should be in same units (days or years)
;    K1, K2, V1, V2 should be in same units (km/s)
;
;
; Began 14 May 2002
; Modified 11 Nov 2003: 
;       This is the current version of NEWT_RAPH.PRO
;	(v1.1 and v1.2 are older versions)
; 	- restructured program into separate subroutines
; 	- add option of holding elements fixed during iterations
; 13 January 2005:
;       Fix bugs
;       - if solving for only one element, don't use cramer.pro to solve eqn's
;       - fix 360 degree roll overs in (PAdata - PAfit)
;       Save version v4 as backup
; 11 July 2005:
;       - Include Marquadt Method of adjusting diagonal elements of
;         the alpha matrix to ensure convergence.
;       - Fixed non-convergence problems 
;           + do not remove P wrap around effects from dx/dP
;             leave (t-T)/P in full form; net effect is that adjustments
;             are smaller
;       Save v5 as backup 
;
; 31 May 2006: Save unmodified program as version 2
; 31 May 2006: Fit SB1 parameters in terms of K1 and K2
;              (as opposed to a1sini and a2sini)
;
; 22 September 2006:
;       - Fixed the way subsets of parameters are handled
;       - Wasn't properly constructing matrices if not all parameters vary
;
; 06 November 2015:
;       - Add flag_wa to calc_deriv_vb_ell.pro and calc_vbfit.pro.  This allows
;         the option of defining omega = omega_A in both the SB1 and VB orbits.
;
; 19 January 2023
;       - Add bootstrap sampling to compute errors
;       - Randomly sample observations (with repeatition) and apply
;         Gaussian uncertainties.
;       - Refit orbit and iterate for Nboot samples.
;       - Compute uncertainties based on bootstrap distributions


close,1
close,2
  
!P.font=0
!P.multi=0
!P.charsize=1.2
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.2	;scale factor for adjusting size of plotting symbols

; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1

; keyword_set returns True (1) if variable is defined, False (0) if undefined

ans_mcrit = keyword_set(mcrit)

ans_noboot = keyword_set(noboot)

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d
temp6 = 0d

time_vb = dblarr(vb_num)
theta = dblarr(vb_num)
rho = dblarr(vb_num)
dmajor = dblarr(vb_num)
dminor = dblarr(vb_num)
theta_err = dblarr(vb_num)

time_sb1 = dblarr(sb1_num)
V1_sb1 = dblarr(sb1_num)
dV1_sb1 = dblarr(sb1_num)

; Read VB measurements

openr,1,vb_file

for i=0, vb_num-1 do begin

	readf, 1, temp1, temp2, temp3, temp4, temp5, temp6

	time_vb(i) = temp1
	theta(i) = temp2
	rho(i) = temp3	
	dmajor(i) = temp4
	dminor(i)= temp5
	theta_err(i) = temp6

endfor

close,1

; convert theta and theta_err to radians

theta = theta*!dpi/180
theta_err = theta_err*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; project separation into components along the direction of the error ellipse

xparr = rho*cos(theta - theta_err)  ; x' (major)
yparr = rho*sin(theta - theta_err)  ; y' (minor) 

; Read SB1 measurements

openr,1,sb1_file

for i=0, sb1_num-1 do begin

	readf, 1, temp1, temp2, temp3

	time_sb1(i) = temp1
	V1_sb1(i) = temp2
	dV1_sb1(i) = temp3
	
endfor

close,1

; Obtain values for P,T,e,a(mas),i,Omega,omega,K1(km/s),K2(km/s),Vsys

period = 0d
Tperi = 0d
ecc = 0d
major = 0d
inc = 0d
W_cap = 0d
w_low = 0d
K1 = 0d
Vsys = 0d

print,"Enter P,T,e,a(mas),i,Omega,omega,K1(km/s),Vsys:"
read,period,Tperi,ecc,major,inc,W_cap,w_low,K1,Vsys

print, "Vary each orbital element?"
print, "For each element, enter 0 to hold element fixed, 1 to vary:"
print, "[P,T,e,a,i,Omega,omega,K1,Vsys]"
read, f0,f1,f2,f3,f4,f5,f6,f7,f8

;; Order of orbital element arrays
;; do not set here yet (need to get angles into radians)
;; EL_vb = [period,Tperi,ecc,major,inc,W_cap,w_low]
;;             0    1     2    3    4    5     6
;; EL_sb1 = [period,Tperi,ecc,K1,K2,w_low,Vsys]
;;              0    1     2   7  8   6     9
;; EL_sb1 = [period,Tperi,ecc,K1,w_low,Vsys]
;;              0    1     2   7   6     9/8

elfix=[f0,f1,f2,f3,f4,f5,f6,f7,f8]
nEl = n_elements(elfix)

elfix_vb=[f0,f1,f2,f3,f4,f5,f6]
nEl_vb = n_elements(elfix_vb)

elfix_sb1=[f0,f1,f2,f7,f6,f8]
nEl_sb1 = n_elements(elfix_sb1)

;convert i, Omega, omega to radians
inc = inc*!dpi/180.0
W_cap = W_cap*!dpi/180.0
w_low = w_low*!dpi/180.0

k=0
for i=0, nEl-1 do if (elfix(i) eq 0) then k=k+1
mfit = nEl - k 		; number of elements to improve

k=0
for i=0, nEl_vb-1 do if (elfix_vb(i) eq 0) then k=k+1
mfit_vb = nEl_vb - k 		; number of elements to improve

k=0
for i=0, nEl_sb1-1 do if (elfix_sb1(i) eq 0) then k=k+1
mfit_sb1 = nEl_sb1 - k 		; number of elements to improve

EL = [period,Tperi,ecc,major,inc,W_cap,w_low,K1,Vsys]
EL_vb = [period,Tperi,ecc,major,inc,W_cap,w_low]
EL_sb1 = [period,Tperi,ecc,K1,w_low,Vsys]

EL_orig = EL

ELadj = EL
elLabel = ['P','T','e','a(mas)','i','Omega','omega_A','K1(km/s)','Vsys']

; determine which indices of full set are being varied

vb_subset = dblarr(mfit_vb)
sb1_subset = dblarr(mfit_sb1)

; determine subarray of indices that are VB, SB1, and SB1 parameters

vb_par = dblarr(nEl_vb)
sb1_par = dblarr(nEl_sb1)

; determine indices of elements for which to vary
el_subset = where(elfix ne 0)
Elvar = El(el_subset)

k=0
for i=0, nEl_vb-1 do begin
    ind = where(EL_vb(i) eq EL)
    vb_par(i) = ind
    if (elfix_vb(i) ne 0) then begin
        ind = where(EL_vb(i) eq Elvar)
        vb_subset(k) = ind
        k=k+1
    endif
endfor

k=0
for i=0, nEl_sb1-1 do begin
    ind = where(EL_sb1(i) eq EL)
    sb1_par(i) = ind
    if (elfix_sb1(i) ne 0) then begin
         ind = where(EL_sb1(i) eq Elvar)
         sb1_subset(k) = ind
        k=k+1
    endif
endfor


nsamp = 10000
print,'Enter number of bootstrap iterations (e.g., 10000):'
read,nsamp

n_replicate = 0.0d
print,'How many time do you want to replicate (randomly) each data point? '
read,n_replicate

; Plot random sample of multiple solutions:
nplot = 0
print,'How many orbits do you want to plot:'
read,nplot

parallax = 0.0d

print, 'Enter the parallax of the system (in same units as separation):'
print,'(distance of 140, parallax = 7.14 mas)'
read, parallax

ansyd=' '
print,'Is period in years or days (y/d)?'
read,ansyd

if (ansyd eq 'y') then yd = 1.0d
if (ansyd eq 'd') then yd = 1.0d/365.25

; calculate mass of system
mass = major^3/parallax^3/(yd*period)^2

Mfunc = 3.784229924d-5 * K1^3 * (yd*period) * (1.0- ecc^2)^(1.5)
   
M2 = 0.033573182d * major^2/ parallax^2*K1/(yd*period)/sin(inc)*sqrt(1.0 - ecc^2)

M1 = major^3/parallax^3/(yd*period)^2 - M2

Mratio = M2/M1

mass_orig = mass
Mratio_orig = Mratio
Mfunc_orig = Mfunc
M1_orig = M1
M2_orig = M2

if (ans_mcrit) then begin
   
   Mmin = 0.0d
   Mmax = 0.0d
   print,'Enter minimum and maximum total mass (in Msun) as search constraint:'
   read,Mmin,Mmax

   remove_M = 0

endif

; **** Plot model orbits ****

ans_cont = ' '

; plot model and data of initial guess for visual orbit

tnum=1000.0
tstep = period/tnum
tmin = Tperi

tarr = findgen(tnum)*tstep + tmin

calc_vbfit, EL_vb, tarr, theta_mod, rho_mod, /flag_wa

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

print,'Visual orbit'
plot,-yarr,xarr,psym=6
oplot,-ymod,xmod

print,'Hit return to continue'
read,ans_cont

; plot model and data of initial guess for SB1 data

tnum = 1000.0
tmin = min(time_sb1)
tmax = max(time_sb1)
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb1fit, EL_sb1, tarr, V1mod

vmax = max([max(V1mod)])
vmin = min([min(V1mod)])

print,'SB1 velocity curve'
plot,time_sb1,V1_sb1,psym=2,yrange=[vmin,vmax]
oplot,tarr,V1mod

vmax = max([max(V1_sb1)])
vmin = min([min(V1_sb1)])

print,'Hit return to continue'
read,ans_cont


; Intialize parameters to hold results from bootstrap iterations

P_boot = dblarr(nsamp)
T_boot = dblarr(nsamp)
e_boot = dblarr(nsamp)
a_boot = dblarr(nsamp)
i_boot = dblarr(nsamp)
Wc_boot = dblarr(nsamp)
wl_boot = dblarr(nsamp)
K1_boot = dblarr(nsamp)
Vsys_boot = dblarr(nsamp)
M1_boot = dblarr(nsamp)
M2_boot = dblarr(nsamp)
Mratio_boot = dblarr(nsamp)
Mtot_boot = dblarr(nsamp)
Mfunc_boot = dblarr(nsamp)

; Compute chi2 relative to original set of measurements
; (not bootstrapped, not replicated)
chi2_orig_boot = dblarr(nsamp)

vb_num_repl = n_replicate*vb_num

repl_time_vb = dblarr(vb_num_repl)
repl_xparr = dblarr(vb_num_repl)
repl_yparr = dblarr(vb_num_repl)
repl_dmajor = dblarr(vb_num_repl)
repl_dminor = dblarr(vb_num_repl)
repl_theta_err = dblarr(vb_num_repl)

repeat_iter = 0

!P.MULTI = [0,2,0]

for ns=0, nsamp-1 do begin

   ; Reset to original parameters:
   
   EL = [period,Tperi,ecc,major,inc,W_cap,w_low,K1,Vsys]
   EL_vb = [period,Tperi,ecc,major,inc,W_cap,w_low]
   EL_sb1 = [period,Tperi,ecc,K1,w_low,Vsys]

   ELadj = EL

   ; If we don't want to resample the data arrays (only replicate):

   if (ans_noboot) then begin

      new_time_vb = time_vb
      new_xparr = xparr
      new_yparr = yparr
      new_dmajor = dmajor
      new_dminor = dminor
      new_theta_err = theta_err

      new_time_sb1 = time_sb1
      new_V1_sb1 = V1_sb1
      new_dV1_sb1 = dV1_sb1

   endif else begin
      
      ; Re-sample measurements with replacement

      ind = fix(vb_num*randomu(seed,vb_num,/uniform))

      new_time_vb = time_vb(ind)
      resamp_xparr = xparr(ind)
      resamp_yparr = yparr(ind)
      new_dmajor = dmajor(ind)
      new_dminor = dminor(ind)
      new_theta_err = theta_err(ind)
  
      ; Compute new array of resampled visibilities based on 
      ; randomly distributed errors
   
      new_xparr = resamp_xparr + new_dmajor*randomn(seed,vb_num,/normal) 
      new_yparr = resamp_yparr + new_dminor*randomn(seed,vb_num,/normal) 

      ; Comment next few lines if you don't want to resample RV
      ind = fix(sb1_num*randomu(seed,sb1_num,/uniform))
      
      new_time_sb1 = time_sb1(ind)
      resamp_V1_sb1 = V1_sb1(ind)
      new_dV1_sb1 = dV1_sb1(ind)
      
      new_V1_sb1 = resamp_V1_sb1 + new_dV1_sb1*randomn(seed,sb1_num,/normal) 

   endelse
      
   ; Now we have to replicate these arrays

   if (n_replicate gt 0) then begin
   
      vrn = 0
      for vn=0, vb_num-1 do begin
         for rn=0, n_replicate-1 do begin
   
            repl_time_vb(vrn) = new_time_vb(vn)
            repl_xparr(vrn) = new_xparr(vn) + new_dmajor(vn)*randomn(seed,/normal)
            repl_yparr(vrn) = new_yparr(vn) + new_dminor(vn)*randomn(seed,/normal)   
            repl_dmajor(vrn) =  new_dmajor(vn) 
            repl_dminor(vrn) =  new_dminor(vn) 
            repl_theta_err(vrn) = new_theta_err(vn)

            vrn = vrn+1
         
         endfor
      endfor

      if (vrn ne vb_num_repl) then print,'WARNING: Problem with number of replications.'
   
   endif else begin

      repl_time_vb = new_time_vb
      repl_xparr = new_xparr
      repl_yparr = new_yparr   
      repl_dmajor =  new_dmajor 
      repl_dminor =  new_dminor 
      repl_theta_err = new_theta_err
      
   endelse

   count = 0

   delta_chi = 1.0		; set initially to begin loop

   lambda = 0.001               ; Marquardt method to ensure convergence

; ***** Begin while loop ***** 

while(delta_chi gt 0.001) do begin

    ; Determine errors in orbital elements
    ; set up matrices for error determination
    ; Invert matrix through Gauss Jordan elimination (Numerical Recipes in C)

    ; set up covariance matrix and column matrix
    ; alpha = dblarr(7,7)
    ; beta = dblarr(7)

    calc_deriv_vb_ell, EL_vb, elfix_vb, mfit_vb, repl_time_vb, repl_xparr, repl_yparr, repl_dmajor, repl_dminor, repl_theta_err, theta_f, rho_f, xpfit, ypfit, alpha_vb, beta_vb, /flag_wa

    ; Fit RV directly
    calc_deriv_SB1, EL_sb1, elfix_sb1, mfit_sb1, new_time_sb1, new_V1_sb1, new_dV1_sb1, V1_f_sb1, alpha_sb1, beta_sb1

    ; Determine chi squared

    chi2old = total((repl_xparr - xpfit)^2/repl_dmajor^2 + (repl_yparr - ypfit)^2/repl_dminor^2) $
            + total((new_V1_sb1 - V1_f_sb1)^2/new_dV1_sb1^2)

    ; combine SB1 and VB matrices...

    ; initialize full alpha,beta arrays
    alpha = dblarr(mfit,mfit)
    beta = dblarr(mfit)

    beta(vb_subset) = beta(vb_subset) + beta_vb
    beta(sb1_subset) = beta(sb1_subset) + beta_sb1

    for i=0, mfit_vb-1 do begin
        for j=0, mfit_vb-1 do begin
            alpha(vb_subset(i),vb_subset(j)) = $
              alpha(vb_subset(i),vb_subset(j)) + alpha_vb(i,j)
        endfor
    endfor

    for i=0, mfit_sb1-1 do begin
        for j=0, mfit_sb1-1 do begin
            alpha(sb1_subset(i),sb1_subset(j)) = $
              alpha(sb1_subset(i),sb1_subset(j)) + alpha_sb1(i,j)
        endfor
    endfor

    invmat = invert(alpha, stat, /double)

    ;print,"stat (0 successful):",stat

    ;determine errors:

    ELerr = dblarr(mfit)

    for i=0, mfit-1 do ELerr(i) = sqrt(invmat(i,i))

    ; adjust alpha matrix by Marquardt parameter lambda

    for i=0, mfit-1 do alpha(i,i) = alpha(i,i)*(1.0 + lambda)

    if (mfit eq 1) then begin
        delta_el = dblarr(1)
        delta_el(0) = beta/alpha
     endif else begin

        if (determ(alpha) ne 0) then delta_el = cramer(alpha,beta) $ ; adjustments
        else begin
           repeat_iter = 1
           delta_el = dblarr(mfit)
           ;print,'inside check'
           ;stop
        endelse
           
     endelse

    ; adjust orbital parameters

    ELadj = EL
    ELadj(el_subset) = ELadj(el_subset) + delta_el 

    ELadj_vb  = ELadj(vb_par) 
    ELadj_sb1 = ELadj(sb1_par) 

    k=0
    print,"Orbital Element     Adjustment"
    for i=0, 3 do begin
        if (elfix(i) eq 0) then begin
            print, elLabel(i), ': ', El(i)
        endif else begin
            print, elLabel(i), ': ', El(i), delta_el(k)
            k=k+1
        endelse
    endfor
    for i=4, 6 do begin
         if (elfix(i) eq 0) then begin
             print, elLabel(i), ': ', El(i)*180/!dpi
         endif else begin
             print, elLabel(i), ': ', El(i)*180/!dpi, delta_el(k)*180/!dpi
             k = k+1
         endelse
    endfor
    for i=7, 8 do begin
        if (elfix(i) eq 0) then begin
            print, elLabel(i), ': ', El(i)
        endif else begin
            print, elLabel(i), ': ', El(i), delta_el(k)
            k=k+1
        endelse
    endfor

    print,"Chi2 of current solution:",chi2old

    ; convert data points to x and y coordinates

    xfit = rho_f * cos(theta_f)	; x coordinate
    yfit = rho_f * sin(theta_f)	; y coordinate

    ;; plot visual orbit
    ;
    ;plot,-yarr,xarr,psym=6
    ;oplot,-yfit,xfit,psym=7
    ;for l=0, vb_num-1 do oplot,[-yarr(l),-yfit(l)],[xarr(l),xfit(l)]
    ;
    ;;plot SB velocity curves
    ;
    ;plot,time_sb1,V1_sb1,psym=2,yrange=[vmin,vmax],xrange=[tmin,tmax]
    ;oplot,time_sb1,V1_f_sb1,psym=7
    ;for l=0, sb1_num-1 do oplot,[time_sb1(l),time_sb1(l)],[V1_sb1(l),V1_f_sb1(l)]
    ;
    ;wait,0.1

    ; calculate new chi2 for adjusted parameters

    calc_deriv_vb_ell, ELadj_vb, elfix_vb, mfit_vb, repl_time_vb, repl_xparr, repl_yparr, repl_dmajor, repl_dminor, repl_theta_err, theta_f, rho_f, xpfit, ypfit, alpha_vb, beta_vb, /flag_wa

    calc_deriv_SB1, ELadj_sb1, elfix_sb1, mfit_sb1, new_time_sb1, new_V1_sb1, new_dV1_sb1, V1_f_sb1, alpha_sb1, beta_sb1

    ; Determine chi squared

    chi2new = total((repl_xparr - xpfit)^2/repl_dmajor^2 + (repl_yparr - ypfit)^2/repl_dminor^2) $
            + total((new_V1_sb1 - V1_f_sb1)^2/new_dV1_sb1^2)

    print,"chi2 of next modification:",chi2new

    if (chi2new le chi2old) then begin
        EL = ELadj
        EL_vb = ELadj_vb
        EL_sb1 = ELadj_sb1
	chi2 = chi2new
        delta_chi = chi2old - chi2new
        lambda = lambda/10.0
        print, 'lambda', lambda
    endif else begin
	print, "next adjustment does not improve chi2"
        print, "increasing lambda by 10.0"
        lambda = lambda*10.0
        print, 'lambda', lambda
        delta_chi = 1.0
    endelse

    count = count+1

    ; do not exit unless lambda is less than 0.001

;    if (lambda gt 0.001) and (delta_chi lt 0.001) then delta_chi = 1.0
;    print,'lambda',lambda
;    print,'delta_chi',delta_chi

    ; alpha is singular - exit and repeat iteration
    if (repeat_iter) then begin
       ;print,'inside repeat loop'
       count = count - 1
       delta_chi = 1d-6
    endif

endwhile

; Save results from current bootstrap iteration

P_boot(ns) = EL(0) 
T_boot(ns) = EL(1) 
e_boot(ns) = EL(2) 
a_boot(ns) = EL(3) 
i_boot(ns) = EL(4) 
Wc_boot(ns) = EL(5)
wl_boot(ns) = EL(6)
K1_boot(ns) = EL(7)
Vsys_boot(ns) = EL(8)

Mfunc= 3.784229924d-5 * K1_boot(ns)^3 * (yd*P_boot(ns)) * (1.0- e_boot(ns)^2)^(1.5)
   
M2 = 0.033573182d * a_boot(ns)^2/ parallax^2*K1_boot(ns)/(yd*P_boot(ns))/sin(i_boot(ns))*sqrt(1.0 - e_boot(ns)^2)

M1 = a_boot(ns)^3/parallax^3/(yd*P_boot(ns))^2 - M2

Mratio = M2/M1

mass = a_boot(ns)^3/parallax^3/(yd*P_boot(ns))^2

M1_boot(ns) = M1
M2_boot(ns) = M2
Mratio_boot(ns) = Mratio
Mtot_boot(ns) = mass
Mfunc_boot(ns) = Mfunc

lambda = 0.0

; Compute chi2 relative to original measurements (not bootstrapped,
; not replicated):

calc_vbfit, EL_vb, time_vb, theta_fit_orig, rho_fit_orig,/flag_wa

; convert data points to x and y coordinates
; project separation into components along the direction of the error ellipse

xp_fit_orig = rho_fit_orig*cos(theta - theta_err) ; x' (major)
yp_fit_orig = rho_fit_orig*sin(theta - theta_err)  ; y' (minor) 

calc_sb1fit, EL_sb1, time_sb1, V1_fit_orig


chi2_orig_boot(ns) = total((xparr - xp_fit_orig)^2/dmajor^2 + (yparr - yp_fit_orig)^2/dminor^2) $
            + total((V1_sb1 - V1_fit_orig)^2/dV1_sb1^2)


print,'Bootstrap iteration: ',ns

if (repeat_iter) then begin
   ns = ns - 1
   print,'alpha is singular - repeat iteration'
   repeat_iter = 0
   ;stop
endif else begin
   if (ans_mcrit) then begin
      ; Repeat iteration if mass is out of range
      if (mass lt Mmin) or (mass gt Mmax) then begin
         ns = ns - 1
         remove_M = remove_M + 1
      endif else begin
         if (M1 lt Mmin) or (M1 gt Mmax) then begin
            ns = ns - 1
            remove_M = remove_M + 1
         endif else if (M2 lt Mmin) or (M2 gt Mmax) then begin
            ns = ns - 1
            remove_M = remove_M + 1
         endif               
      endelse
   endif 
endelse

endfor  ; end bootstrap iterations

period = EL_orig(0)
Tperi = EL_orig(1)
ecc = EL_orig(2)
major = EL_orig(3)
inc = EL_orig(4)
W_cap = EL_orig(5)
w_low = EL_orig(6)
K1 = EL_orig(7)
Vsys = EL_orig(8)

; Save results to file
openw,1,'result_orb'
openw,2,'result_orb_mass'

printf,1,'#   P         T             e          a           i           Omega       omega         K1         Vsys'
printf,1,format='(f12.5,f12.5,f12.6,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5)',period,Tperi,ecc,major,inc*180.0/!dpi,W_cap*180.0/!dpi,w_low*180.0/!dpi,K1,Vsys

printf,2,'#   P         T             e          a           i           Omega       omega         K1         Vsys          M1          M2          Mratio      Mtot        Mfunc'
printf,2,format='(f12.5,f12.5,f12.6,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5)',period,Tperi,ecc,major,inc*180.0/!dpi,W_cap*180.0/!dpi,w_low*180.0/!dpi,K1,Vsys,M1_orig,M2_orig,Mratio_orig,mass_orig,Mfunc_orig

for i=0, nsamp-1 do begin

   printf,1,format='(f12.5,f12.5,f12.6,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5)',P_boot(i),T_boot(i),e_boot(i),a_boot(i),i_boot(i)*180.0/!dpi,Wc_boot(i)*180.0/!dpi,wl_boot(i)*180.0/!dpi,K1_boot(i),Vsys_boot(i)

   printf,2,format='(f12.5,f12.5,f12.6,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5)',P_boot(i),T_boot(i),e_boot(i),a_boot(i),i_boot(i)*180.0/!dpi,Wc_boot(i)*180.0/!dpi,wl_boot(i)*180.0/!dpi,K1_boot(i),Vsys_boot(i),M1_boot(i),M2_boot(i),Mratio_boot(i),Mtot_boot(i),Mfunc_boot(i)

endfor

close,1
close,2

ind = where(chi2_orig_boot eq min(chi2_orig_boot))

print,'Solution with minimum chi2:'
print,P_boot(ind),T_boot(ind),e_boot(ind),a_boot(ind),i_boot(ind)*180.0/!dpi,Wc_boot(ind)*180.0/!dpi,Wl_boot(ind)*180.0/!dpi,K1_boot(ind),Vsys_boot(ind)
print,'Min chi2:',min(chi2_orig_boot)

print, "Final values:"

P_best = mean(P_boot)
T_best = mean(T_boot)
e_best = mean(e_boot)
a_best = mean(a_boot)
i_best = mean(i_boot)
Wc_best = mean(Wc_boot)
wl_best = mean(wl_boot)
K1_best = mean(K1_boot)
Vsys_best = mean(Vsys_boot)

P_err  = stdev(P_boot)
T_err  = stdev(T_boot)
e_err  = stdev(e_boot)
a_err  = stdev(a_boot)
i_err  = stdev(i_boot)
Wc_err = stdev(Wc_boot)
wl_err = stdev(wl_boot)
K1_err = stdev(K1_boot)
Vsys_err = stdev(Vsys_boot)

rtd = 180.0/!dpi

print,'Par  Min  Max  Mean  Median  Stdev'

print,'P:        ',min(P_boot),max(P_boot),mean(P_boot),median(P_boot),stdev(P_boot)
print,'T:        ',min(T_boot),max(T_boot),mean(T_boot),median(T_boot),stdev(T_boot)
print,'e:        ',min(e_boot),max(e_boot),mean(e_boot),median(e_boot),stdev(e_boot)
print,'a:        ',min(a_boot),max(a_boot),mean(a_boot),median(a_boot),stdev(a_boot)
print,'i:        ',min(i_boot)*rtd,max(i_boot)*rtd,mean(i_boot)*rtd,median(i_boot)*rtd,stdev(i_boot)*rtd
print,'Omega:    ',min(Wc_boot)*rtd,max(Wc_boot)*rtd,mean(Wc_boot)*rtd,median(Wc_boot)*rtd,stdev(Wc_boot)*rtd
print,'omega_A:  ',min(wl_boot)*rtd,max(wl_boot)*rtd,mean(wl_boot)*rtd,median(wl_boot)*rtd,stdev(wl_boot)*rtd
print,'K1(km/s): ',min(K1_boot),max(K1_boot),mean(K1_boot),median(K1_boot),stdev(K1_boot)
print,'Vsys:     ',min(Vsys_boot),max(Vsys_boot),mean(Vsys_boot),median(Vsys_boot),stdev(Vsys_boot)
print,'Mfunc:   ',min(Mfunc_boot),max(Mfunc_boot),mean(Mfunc_boot),median(Mfunc_boot),stdev(Mfunc_boot)
print,'Mratio:   ',min(Mratio_boot),max(Mratio_boot),mean(Mratio_boot),median(Mratio_boot),stdev(Mratio_boot)
print,'Mtot:     ',min(Mtot_boot),max(Mtot_boot),mean(Mtot_boot),median(Mtot_boot),stdev(Mtot_boot)
print,'M1:       ',min(M1_boot),max(M1_boot),mean(M1_boot),median(M1_boot),stdev(M1_boot)
print,'M2:    ',min(M2_boot),max(M2_boot),mean(M2_boot),median(M2_boot),stdev(M2_boot)

; Compute uncertainties from 68% of values around median
; (better accomodates long, sparsely populated tails in distributions)

num34 = 0.34*nsamp
bootLabel = ['P      ','T      ','e      ','a      ','i      ','Omega  ','omega_A','K1(km/s)','Vsys   ','Mfunc  ','Mratio ','Mtot   ','M1   ','M2   ']

bs = [1.0,1.0,1.0,1.0,180.0/!dpi,180.0/!dpi,180.0/!dpi,1.0,1.0,1.0,1.0,1.0,1.0,1.0]


print,"Error range within +/-34% of median value:"

for i=0, 13 do begin

   if (i eq 0) then data_arr = P_boot
   if (i eq 1) then data_arr = T_boot
   if (i eq 2) then data_arr = e_boot
   if (i eq 3) then data_arr = a_boot
   if (i eq 4) then data_arr = i_boot
   if (i eq 5) then data_arr = Wc_boot
   if (i eq 6) then data_arr = wl_boot
   if (i eq 7) then data_arr = K1_boot
   if (i eq 8) then data_arr = Vsys_boot
   if (i eq 9) then data_arr = Mfunc_boot
   if (i eq 10) then data_arr = Mratio_boot
   if (i eq 11) then data_arr = Mtot_boot
   if (i eq 12) then data_arr = M1_boot
   if (i eq 13) then data_arr = M2_boot


   ind_sort = sort(data_arr,/L64)

   data_sort = data_arr(ind_sort)
   
   med_val = median(data_arr)
   med_ind = where(abs(data_sort - med_val) eq min(abs(data_sort - med_val)))

   sum_lower = 0.0
   sum_upper = 0.0

   ; determine lower error bar
   arr_i = med_ind(0)
   while(sum_lower le num34) do begin
      sum_lower = sum_lower + 1
      arr_i = arr_i - 1
   endwhile
   bin_lower = arr_i + 1.0
   error_lower = med_val - data_sort(bin_lower(0))

   ; determine upper error bar
   arr_i = med_ind(0)
   while(sum_upper le num34) do begin
      sum_upper = sum_upper + 1
      arr_i = arr_i + 1
   endwhile
   bin_upper = arr_i - 1.0
   error_upper = data_sort(bin_upper(0)) - med_val

   print,bootLabel(i),med_val*bs(i),' + ',error_upper*bs(i),' - ',error_lower*bs(i)

   
endfor


print,"Compute uncertainties relative to best fit rather than median:"

EL_full = [EL_orig,Mfunc_orig,Mratio_orig,mass_orig,M1_orig,M2_orig] 

for i=0, 13 do begin

   if (i eq 0) then data_arr = P_boot
   if (i eq 1) then data_arr = T_boot
   if (i eq 2) then data_arr = e_boot
   if (i eq 3) then data_arr = a_boot
   if (i eq 4) then data_arr = i_boot
   if (i eq 5) then data_arr = Wc_boot
   if (i eq 6) then data_arr = wl_boot
   if (i eq 7) then data_arr = K1_boot
   if (i eq 8) then data_arr = Vsys_boot
   if (i eq 9) then data_arr = Mfunc_boot
   if (i eq 10) then data_arr = Mratio_boot
   if (i eq 11) then data_arr = Mtot_boot
   if (i eq 12) then data_arr = M1_boot
   if (i eq 13) then data_arr = M2_boot


   ind_sort = sort(data_arr,/L64)

   data_sort = data_arr(ind_sort)
   
   med_val = median(data_arr)
   med_ind = where(abs(data_sort - med_val) eq min(abs(data_sort - med_val)))

   sum_lower = 0.0
   sum_upper = 0.0

   ; determine lower error bar
   arr_i = med_ind(0)
   while(sum_lower le num34) do begin
      sum_lower = sum_lower + 1
      arr_i = arr_i - 1
   endwhile
   bin_lower = arr_i + 1.0
   ;error_lower = med_val - data_sort(bin_lower(0))
   error_lower = EL_full(i) - data_sort(bin_lower(0))

   ; determine upper error bar
   arr_i = med_ind(0)
   while(sum_upper le num34) do begin
      sum_upper = sum_upper + 1
      arr_i = arr_i + 1
   endwhile
   bin_upper = arr_i - 1.0
   ;error_upper = data_sort(bin_upper(0)) - med_val
   error_upper = data_sort(bin_upper(0)) - EL_full(i)

   print,bootLabel(i),EL_full(i)*bs(i),' + ',error_upper*bs(i),' - ',error_lower*bs(i)

endfor


if (ans_mcrit) then begin
   print,'Number of solutions removed for Mcrit:',Remove_M
   print,'Fraction removed: ',double(Remove_M)/double(nsamp)
endif

; Set to best fit orbit:

period  = EL_orig(0)
Tperi   = EL_orig(1)
ecc     = EL_orig(2)
major   = EL_orig(3)
inc     = EL_orig(4)
W_cap   = EL_orig(5)
w_low   = EL_orig(6)
K1      = EL_orig(7)
Vsys   =  EL_orig(8)

EL = [period,Tperi,ecc,major,inc,W_cap,w_low,K1,Vsys]
EL_vb = [period,Tperi,ecc,major,inc,W_cap,w_low]
EL_sb1 = [period,Tperi,ecc,K1,w_low,Vsys]

EL_vb_orig = EL_vb
EL_sb1_orig = EL_sb1
   
!P.MULTI = [0,0,0]

; plot best-fit orbit and data for visual orbit

tnum=1000.0
tstep = period/tnum
tmin = Tperi

tarr = findgen(tnum)*tstep + tmin

calc_vbfit, EL_vb, tarr, theta_mod, rho_mod, /flag_wa

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

calc_vbfit, EL_vb, time_vb, theta_f, rho_f,/flag_wa

xfit = rho_f * cos(theta_f)	; x coordinate
yfit = rho_f * sin(theta_f)	; y coordinate

; plot data and best-fit positions for visual orbit

print,'Visual orbit'
plot,-yarr,xarr,psym=6
oplot,-yfit,xfit,psym=7
for l=0, vb_num-1 do oplot,[-yarr(l),-yfit(l)],[xarr(l),xfit(l)]

; plot best-fit orbit for visual orbit
oplot,-ymod,xmod


;print,'Hit return to continue'
;read,ans_cont

; Define user symbol to be closed circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)

; Plot measured positions as filled circles

usersym, frac*cos(Z), frac*sin(Z),/fill

sepmax = 1.2*max(abs([yarr,xarr]))

set_plot, 'ps'
device,/Color,Bits_per_pixel=8,filename='temp_vb.eps',xsize=12,ysize=12,isolatin1=1,/encap

plot,yarr,xarr,xtitle='!9D!3RA (mas)',ytitle='!9D!3DEC (mas)',ISOTROPIC=1, $
     xrange=[sepmax,-sepmax], yrange=[-sepmax,sepmax], $
     xstyle=1,ystyle=1, /nodata, position=[0.19,0.12,0.95,0.98]

; Connected measured and fit values
for i=0,vb_num-1 do oplot,[yfit(i),yarr(i)],[xfit(i),xarr(i)]

; Mark computed location with x
;oplot,[yfit],[xfit],psym=7

                            ; mark primary with an asterisk
oplot,[0],[0],psym=2
oplot,[-sepmax,sepmax],[0,0],linestyle=1
oplot,[0,0],[-sepmax,sepmax],linestyle=1

; plot best-fit orbit
oplot,ymod,xmod,color=3,thick=2

; Plot measured values
;Define the symbol to be a closed circle
usersym, frac*cos(Z), frac*sin(Z), /fill
oplot,yarr,xarr,psym=8

; Plot error ellipses
for i=0, vb_num-1 do begin

   create_ellipse,xe,ye,100,major=dmajor(i),minor=dminor(i),xcent=xarr(i),ycent=yarr(i),theta=theta_err(i)*180.0/!dpi

   oplot,ye,xe,thick=4,color=1

endfor
    
device,/close
set_plot, 'x'


; Plot random sample of multiple solutions:

set_plot, 'ps'
device,/Color,Bits_per_pixel=8,filename='temp_vb_multi.eps',xsize=12,ysize=12,isolatin1=1,/encap

plot,yarr,xarr,xtitle='!9D!3RA (mas)',ytitle='!9D!3DEC (mas)',ISOTROPIC=1, $
	 xrange=[sepmax,-sepmax], yrange=[-sepmax,sepmax], $
         xstyle=1,ystyle=1, /nodata, position=[0.19,0.12,0.95,0.98]

for i=1, nplot-1 do begin

   P_i    = P_boot(i)
   T_i    = T_boot(i)
   e_i    = e_boot(i)
   a_i    = a_boot(i)
   i_i    = i_boot(i)
   Wc_i   = Wc_boot(i)
   wl_i   = wl_boot(i)
   K1_i   = K1_boot(i)
   Vsys_i = Vsys_boot(i)

   EL_i = [P_i,T_i,e_i,a_i,i_i,Wc_i,wl_i,K1_i,Vsys_i]
   EL_vb_i = [P_i,T_i,e_i,a_i,i_i,Wc_i,wl_i]
   EL_sb1_i = [P_i,T_i,e_i,K1_i,wl_i,Vsys_i]

   calc_vbfit, EL_vb_i, tarr, theta_i, rho_i, /flag_wa

   xi = rho_i * cos(theta_i) ; x coordinate
   yi = rho_i * sin(theta_i) ; y coordinate

   ; plot best-fit orbit
   oplot,yi,xi,color=130
   
endfor

; mark primary with an asterisk
oplot,[0],[0],psym=2
oplot,[-sepmax,sepmax],[0,0],linestyle=1
oplot,[0,0],[-sepmax,sepmax],linestyle=1

; Plot best-fit orbit
oplot,ymod,xmod,color=3,thick=4

; Plot measured values
;Define the symbol to be a closed circle
usersym, frac*cos(Z), frac*sin(Z), /fill
oplot,yarr,xarr,psym=8

; Plot error ellipses
for i=0, vb_num-1 do begin

   create_ellipse,xe,ye,100,major=dmajor(i),minor=dminor(i),xcent=xarr(i),ycent=yarr(i),theta=theta_err(i)*180.0/!dpi

   oplot,ye,xe,thick=4,color=1

endfor

device,/close
set_plot, 'x'


; PLOT SB1 ORBITS

tnum = 1000.0
tmin = Tperi - period*0.25
tmax = Tperi + period + period*0.25
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb1fit, EL_sb1, tarr, V1mod

vmax = max([max(V1mod),max(V1_sb1+dV1_sb1)])
vmin = min([min(V1mod),min(V1_sb1-dV1_sb1)])
vrange = vmax - vmin
vmin = vmin - 0.05*vrange
vmax = vmax + 0.05*vrange

; compute phase from time of periastron passage
phase_sb1 = (time_sb1 - Tperi)/period
; reduce phase to -1.0 to 1.0
phase_sb1 = phase_sb1 - fix(phase_sb1)
; keep phase between 0.0 to 1.0
for i=0, sb1_num-1 do if (phase_sb1(i) lt 0.0) $
  then phase_sb1(i) = phase_sb1(i) + 1.0

phase_mod = (tarr-Tperi)/period

print,'SB1 velocity curve'
plot,phase_sb1,V1_sb1,psym=2,yrange=[vmin,vmax],xrange =[-0.1,1.1], $
  xtitle = 'Phase', ytitle = 'RV (km/s)', xstyle=1
oplot,phase_mod,V1mod


set_plot,'ps'
device, /Color, Bits_per_pixel=8, filename='temp_sb.eps',xsize=12,ysize=10,/encap

;toff = 50000.0d               ; RJD offset
;tlabel='JD - 2450000 (days)'
toff = 0.0d                     ; no offset
tlabel='Time'

; Define user symbol to be open rectangle
xrect = frac*[-1.0,-1.0,1.0,1.0,-1.0]
yrect = frac*[-1.0,1.0,1.0,-1.0,-1.0]

ploterror,phase_sb1,V1_sb1,dV1_sb1,psym=2,yrange=[vmin,vmax],xrange =[-0.1,1.1], $
  xtitle = 'Phase', ytitle = 'RV (km/s)', xstyle=1,/NoData
usersym, frac*cos(Z), frac*sin(Z),/fill
oplot,phase_mod,V1mod,color=3,thick=2
oploterror,phase_sb1,V1_sb1,dV1_sb1,psym=8,/nodata
oplot,phase_sb1,V1_sb1,psym=8

device,/close
set_plot, 'x'



set_plot,'ps'
device, /Color, Bits_per_pixel=8, filename='temp_sb_multi.eps',xsize=12,ysize=10,/encap

ploterror,phase_sb1,V1_sb1,dV1_sb1,psym=2,yrange=[vmin,vmax],xrange =[-0.1,1.1], $
  xtitle = 'Phase', ytitle = 'RV (km/s)', xstyle=1,/NoData


for i=1, nplot-1 do begin

   P_i    = P_boot(i)
   T_i    = T_boot(i)
   e_i    = e_boot(i)
   a_i    = a_boot(i)
   i_i    = i_boot(i)
   Wc_i   = Wc_boot(i)
   wl_i   = wl_boot(i)
   K1_i   = K1_boot(i)
   Vsys_i = Vsys_boot(i)

   EL_i = [P_i,T_i,e_i,a_i,i_i,Wc_i,wl_i,K1_i,Vsys_i]
   EL_vb_i = [P_i,T_i,e_i,a_i,i_i,Wc_i,wl_i]
   EL_sb1_i = [P_i,T_i,e_i,K1_i,wl_i,Vsys_i]

   calc_sb1fit, EL_sb1_i, tarr, V1i

   oplot,phase_mod,V1i,color=130

endfor

; Overplot best fit and measurements:

oplot,phase_mod,V1mod,color=3,thick=4
usersym, frac*cos(Z), frac*sin(Z),/fill
oploterror,phase_sb1,V1_sb1,dV1_sb1,psym=8,/nodata
oplot,phase_sb1,V1_sb1,psym=8

device,/close
set_plot,'x'


end

