pro fit_orbit_vbsb1_lunar_vshift_bootstrap, vb_file, vb_num, lun_file, lun_num, sb1_file1, sb1_num1, sb1_file2, sb1_num2, mcrit=mcrit

; Fit simultaneous orbit for VB + SB1 + lunar occultation
;
; Modified to include lunar occultation projection in orbit fit
;
; Add bootstrap sampling to compute errors
;   - Randomly sample observations (with repeatition) and apply
;     Gaussian uncertainties.
;   - Refit orbit and iterate for Nboot samples.
;   - Compute uncertainties based on bootstrap distributions
;
; For each iteration, compute the orbital elements through a
; Newton-Raphson technique.
;
; This program should work directly with the data points (rho,theta,t), not 
; the fitting points. 
;
; The program requires an initial guess that can be calculated from using
; the points from the fitting curves.
;
; Equations:  
; Initial estimates (P0,T0,e0,A0,B0,F0,G0) and a set of observations 
; (ti,rhoi,thetai) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Radius vector:
;	r = a(1-e^2)/(1 + ecos(nu))
; The fitted (rho_i,theta_i) positions are determined from the orbital elements
; 	tan(theta - Omega) = tan(nu + omega)*cosi
;	rho = r*cos(nu + omega)*sec(theta - Omega)
;
; Procedure:
; Minimize chi squared between data point positions and fitted positions:
;	chi^2 = sum[(rhodata - rhoi)^2/sigmarho^2 
;		    + (thetadata - thetai)^2/sigmatheta^2]
; Take partial derivatives of chi^2 with respect to the orbital elements,
; (P,T,e,a,i,Omega,omega) and set to zero. Solve for the value of the orbital 
; elements that minimize chi^2.
;
; Since the partial derivatives are non-linear, cannot solve for the orbital
; elements that minimize chi^2 analytically.
;
; In turn, use a Newton-Raphson technique to converge upon the solution.
; Replace (rhoi,thetai) in the chi^2 equation with the Taylor series 
; approximation,
; 	x = x| + dx/dP|(P-P0) + dx/dT|(T-T0) + dx/de|(e-e0) 
;	       + dx/da|(a-Aa) + dx/di|(i-i0) + dx/dW|(W-W0) + dx/dw|(w-w0)
; where x=(rho,theta) and the | represents the value evaluated at for the
; initial estimates for the orbital elements. Now, rhoi and thetai are linear 
; in (P,T,e,a,i,W,w), so the partial derivatives can easily be taken and solved
; by setting up a matrix and using Cramer's method.
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;	major:	 semi-major axis (in mas) (a)
;	inc:	 inclination (i)
;	Omega:	 position angle of node (W)
;	omega:	 angle between node and periastron (w)
;	ti:	 time of observation
;	rho_d:	 separation - data
;	theta_d: postion angle - data
;	rho_f:	 sep - fit (determined from orbital elements for time of obs.)
;	theta_f: PA - fit (determined from orbital elements for time of obs.)
;
; INPUT:
;	vb_file: file containing data points in the format:
;		      time  theta  etheta  rho  erho
;		      where theta is the position angle, rho is the separation
;                     and etheta, erho are the measurement uncertainties
;		      (enter file name using quotes)
;	num: number of data points contained in vb_file
;	
; Prompted inputs:
;	initial estimates for all the orbital elements: P,T,e,a,i,Omega,omega
;
; Calls the following routines:
;     calc_deriv_vb.pro
;     calc_deriv_lunar
;     calc_Ei.pro
;     solve_trans.pro
;     calc_vbfit.pro - for plotting model orbits!
;
; Began 14 May 2002
; Modified 11 Nov 2003: 
;       This is the current version of NEWT_RAPH.PRO
;	(v1.1 and v1.2 are older versions)
; 	- restructured program into separate subroutines
; 	- add option of holding elements fixed during iterations
; 13 January 2005:
;       Fix bugs
;       - if solving for only one element, don't use cramer.pro to solve eqn's
;       - fix 360 degree roll overs in (PAdata - PAfit)
;       Save version v4 as backup
; 11 July 2005:
;       - Include Marquadt Method of adjusting diagonal elements of
;         the alpha matrix to ensure convergence.
;       - Fixed non-convergence problems 
;           + do not remove P wrap around effects from dx/dP
;             leave (t-T)/P in full form; net effect is that adjustments
;             are smaller
;       Save v5 as backup 
; 13 July 2005:
;       - Change format of plotting routines to include a call to
;         calc_vbfit
;       Save v6 as backup
; 03 May 2006:
;       - Modified to include lunar occultation projection in orbit fit
; 04 November 2022
;       - Add bootstrap sampling to compute errors
;       - Randomly sample observations (with repeatition) and apply
;         Gaussian uncertainties.
;       - Refit orbit and iterate for Nboot samples.
;       - Compute uncertainties based on bootstrap distributions

close,1
close,2

!P.font=0
!P.multi=0
!P.charsize=1.2
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.2	;scale factor for adjusting size of plotting symbols

; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1

; keyword_set returns True (1) if variable is defined, False (0) if undefined

ans_mcrit = keyword_set(mcrit)

; Read in visual orbit data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time_vb = dblarr(vb_num)
theta = dblarr(vb_num)
rho = dblarr(vb_num)
dtheta = dblarr(vb_num)
drho = dblarr(vb_num)

openr,lun,vb_file,/get_lun

for i=0, vb_num-1 do begin

   readf, lun, temp1, temp2, temp3, temp4, temp5

   time_vb(i) = temp1
   theta(i) = temp2
   dtheta(i) = temp3
   rho(i) = temp4	
   drho(i) = temp5
   
endfor

close,lun

; convert theta and dtheta to radians

theta = theta*!dpi/180
dtheta = dtheta*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(theta)^2*drho^2 + rho^2*sin(theta)^2*dtheta^2)
dy = sqrt(sin(theta)^2*drho^2 + rho^2*cos(theta)^2*dtheta^2)

; Read in lunar occultation measurements:

time_lun = dblarr(lun_num)
PA_lun = dblarr(lun_num)
sep_lun = dblarr(lun_num)
dsep_lun = dblarr(lun_num)

openr,lun,lun_file

for i=0, lun_num-1 do begin

   
   readf, lun, temp1, temp2, temp3, temp4

   time_lun(i) = temp1
   PA_lun(i) = temp2
   sep_lun(i) = temp3
   dsep_lun(i) = temp4

endfor

close,lun

; convert PA to radians

PA_lun = PA_lun*!dpi/180

; Read SB1 measurements

time_sb1_1 = dblarr(sb1_num1)
V1_sb1_1 = dblarr(sb1_num1)
dV1_sb1_1 = dblarr(sb1_num1)

openr,lun,sb1_file1

for i=0, sb1_num1-1 do begin

	readf, lun, temp1, temp2, temp3

	time_sb1_1(i) = temp1
	V1_sb1_1(i) = temp2
	dV1_sb1_1(i) = temp3
	
endfor

close,lun

; Read second set of SB1 measurements - zero-point vel shift

time_sb1_2 = dblarr(sb1_num2)
V1_sb1_2 = dblarr(sb1_num2)
dV1_sb1_2 = dblarr(sb1_num2)

openr,lun,sb1_file2

for i=0, sb1_num2-1 do begin

	readf, lun, temp1, temp2, temp3

	time_sb1_2(i) = temp1
	V1_sb1_2(i) = temp2
	dV1_sb1_2(i) = temp3
	
endfor

close,lun


; Obtain values for P,T,e,a,i,Omega,omega 

period = 0d
Tperi = 0d
ecc = 0d
major = 0d
inc = 0d
W_cap = 0d
w_low = 0d
K1 = 0d
Vsys1 = 0d
Vsys2 = 0d

print,"Enter P,T,e,a,i,Omega,omega,K1(km/s),Vsys,Vsys2:"
read,period,Tperi,ecc,major,inc,W_cap,w_low,K1,Vsys1,Vsys2

print, "Vary each orbital element?"
print, "For each element, enter 0 to hold element fixed, 1 to vary:"
print, "[P,T,e,a,i,Omega,omega,K1,Vsys1,Vsys2]"
read, f0,f1,f2,f3,f4,f5,f6,f7,f8,f9

elfix=[f0,f1,f2,f3,f4,f5,f6,f7,f8,f9]
nEl = n_elements(elfix)

elfix_vb=[f0,f1,f2,f3,f4,f5,f6]
nEl_vb = n_elements(elfix_vb)

elfix_lun=[f0,f1,f2,f3,f4,f5,f6]
nEl_lun = n_elements(elfix_lun)

elfix_sb1_1=[f0,f1,f2,f7,f6,f8]
nEl_sb1_1 = n_elements(elfix_sb1_1)

elfix_sb1_2=[f0,f1,f2,f7,f6,f9]
nEl_sb1_2 = n_elements(elfix_sb1_2)

print,"Initial P,T,e,a,i,Omega,omega,K1,Vsys,Vsys2:"
print,period,Tperi,ecc,major,inc,W_cap,w_low,K1,Vsys1,Vsys2

;convert i, Omega, omega to radians
inc = inc*!dpi/180
W_cap = W_cap*!dpi/180
w_low = w_low*!dpi/180

k=0
for i=0, nEl-1 do if (elfix(i) eq 0) then k=k+1
mfit = nEl - k 		; number of elements to improve

k=0
for i=0, nEl_vb-1 do if (elfix_vb(i) eq 0) then k=k+1
mfit_vb = nEl_vb - k 		; number of elements to improve

k=0
for i=0, nEl_lun-1 do if (elfix_lun(i) eq 0) then k=k+1
mfit_lun = nEl_lun - k 		; number of elements to improve

k=0
for i=0, nEl_sb1_1-1 do if (elfix_sb1_1(i) eq 0) then k=k+1
mfit_sb1_1 = nEl_sb1_1 - k 		; number of elements to improve

k=0
for i=0, nEl_sb1_2-1 do if (elfix_sb1_2(i) eq 0) then k=k+1
mfit_sb1_2 = nEl_sb1_2 - k 		; number of elements to improve

; determine indices of elements for which to vary
elvar = where(elfix ne 0)

EL = [period,Tperi,ecc,major,inc,W_cap,w_low,K1,Vsys1,Vsys2]
EL_vb = [period,Tperi,ecc,major,inc,W_cap,w_low]
EL_lun = [period,Tperi,ecc,major,inc,W_cap,w_low]
EL_sb1_1 = [period,Tperi,ecc,K1,w_low,Vsys1]
EL_sb1_2 = [period,Tperi,ecc,K1,w_low,Vsys2]

EL_orig = EL

ELadj = EL
elLabel = ['P','T','e','a','i','Omega','omega_A','K1(km/s)','Vsys1','Vsys2']

; determine which indices of full set are being varied

vb_subset = dblarr(mfit_vb)
lun_subset = dblarr(mfit_lun)
sb1_subset_1 = dblarr(mfit_sb1_1)
sb1_subset_2 = dblarr(mfit_sb1_2)

; determine subarray of indices that are VB, SB1, and SB2 parameters

vb_par = dblarr(nEl_vb)
lun_par = dblarr(nEl_lun)
sb1_par_1 = dblarr(nEl_sb1_1)
sb1_par_2 = dblarr(nEl_sb1_2)

; determine indices of elements for which to vary
el_subset = where(elfix ne 0)
Elvar = El(el_subset)

k=0
for i=0, nEl_vb-1 do begin
    ind = where(EL_vb(i) eq EL)
    vb_par(i) = ind
    if (elfix_vb(i) ne 0) then begin
        ind = where(EL_vb(i) eq Elvar)
        vb_subset(k) = ind
        k=k+1
    endif
endfor

k=0
for i=0, nEl_lun-1 do begin
    ind = where(EL_lun(i) eq EL)
    lun_par(i) = ind
    if (elfix_lun(i) ne 0) then begin
        ind = where(EL_lun(i) eq Elvar)
        lun_subset(k) = ind
        k=k+1
    endif
endfor

k=0
for i=0, nEl_sb1_1-1 do begin
    ind = where(EL_sb1_1(i) eq EL)
    sb1_par_1(i) = ind
    if (elfix_sb1_1(i) ne 0) then begin
         ind = where(EL_sb1_1(i) eq Elvar)
         sb1_subset_1(k) = ind
        k=k+1
    endif
endfor

k=0
for i=0, nEl_sb1_2-1 do begin
    ind = where(EL_sb1_2(i) eq EL)
    sb1_par_2(i) = ind
    if (elfix_sb1_2(i) ne 0) then begin
         ind = where(EL_sb1_2(i) eq Elvar)
         sb1_subset_2(k) = ind
        k=k+1
    endif
endfor

nsamp = 10000
print,'Enter number of bootstrap iterations (e.g., 10000):'
read,nsamp

parallax = 0.0d

print, 'Enter the parallax of the system (in same units as separation):'
print,'(distance of 140, parallax = 7.14 mas)'
read, parallax

ansyd=' '
print,'Is period in years or days (y/d)?'
read,ansyd

if (ansyd eq 'y') then yd = 1.0d
if (ansyd eq 'd') then yd = 1.0d/365.25

; calculate mass of system
mass = major^3/parallax^3/(yd*period)^2

Mfunc = 3.784229924d-5 * K1^3 * (yd*period) * (1.0- ecc^2)^(1.5)
   
M2 = 0.033573182d * major^2/ parallax^2*K1/(yd*period)/sin(inc)*sqrt(1.0 - ecc^2)

M1 = major^3/parallax^3/(yd*period)^2 - M2

Mratio = M2/M1

mass_orig = mass
Mratio_orig = Mratio
Mfunc_orig = Mfunc
M1_orig = M1
M2_orig = M2

if (ans_mcrit) then begin
   
   Mmin = 0.0d
   Mmax = 0.0d
   print,'Enter minimum and maximum total mass (in Msun) as search constraint:'
   read,Mmin,Mmax

   remove_M = 0

endif

; **** Plot model orbits ****

ans_cont = ' '

; plot model and data of initial guess

tnum=1000.0
tstep = period/tnum
tmin = Tperi

tarr = findgen(tnum)*tstep + tmin

calc_vbfit, EL, tarr, theta_mod, rho_mod, /flag_wa

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

; plot best-fit orbit

plot,-yarr,xarr,psym=6
oplot,-ymod,xmod

print,'Hit enter to continue'
read,ans_cont

; plot model and data of initial guess for SB2 data

tnum = 1000.0
tmin = min([min(time_sb1_1),min(time_sb1_2)])
tmax = max([max(time_sb1_1),max(time_sb1_2)])
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb1fit, EL_sb1_1, tarr, V1mod_1

vmax = max([max(V1mod_1),max(V1_sb1_1),max(V1_sb1_2)])
vmin = min([min(V1mod_1),min(V1_sb1_1),min(V1_sb1_2)])

print,'SB1 velocity curve'
plot,time_sb1_1,V1_sb1_1,psym=2,yrange=[vmin,vmax]
oplot,time_sb1_2,V1_sb1_2 + (Vsys1 - Vsys2),psym=2
oplot,tarr,V1mod_1

print,'Hit return to continue'
read,ans_cont

; Intialize parameters to hold results from bootstrap iterations

P_boot = dblarr(nsamp)
T_boot = dblarr(nsamp)
e_boot = dblarr(nsamp)
a_boot = dblarr(nsamp)
i_boot = dblarr(nsamp)
Wc_boot = dblarr(nsamp)
wl_boot = dblarr(nsamp)
K1_boot = dblarr(nsamp)
V1_boot = dblarr(nsamp)
V2_boot = dblarr(nsamp)
M1_boot = dblarr(nsamp)
M2_boot = dblarr(nsamp)
Mratio_boot = dblarr(nsamp)
Mtot_boot = dblarr(nsamp)
Mfunc_boot = dblarr(nsamp)

repeat_iter = 0

for ns=0, nsamp-1 do begin

   ; Reset to original parameters:
   
   EL = [period,Tperi,ecc,major,inc,W_cap,w_low,K1,Vsys1,Vsys2]
   EL_vb = [period,Tperi,ecc,major,inc,W_cap,w_low]
   EL_lun = [period,Tperi,ecc,major,inc,W_cap,w_low]
   EL_sb1_1 = [period,Tperi,ecc,K1,w_low,Vsys1]
   EL_sb1_2 = [period,Tperi,ecc,K1,w_low,Vsys2]

   ELadj = EL

   ; Re-sample measurements with replacement

   ind = fix(vb_num*randomu(seed,vb_num,/uniform))

   new_time_vb = time_vb(ind)
   resamp_theta = theta(ind)
   resamp_rho = rho(ind)
   new_dtheta = dtheta(ind)
   new_drho = drho(ind)

   ind = fix(lun_num*randomu(seed,lun_num,/uniform))

   new_time_lun = time_vb(ind)
   new_PA_lun = PA_lun(ind)
   resamp_sep_lun = sep_lun(ind)
   new_dsep_lun = dsep_lun(ind)
   
   ind = fix(sb1_num1*randomu(seed,sb1_num1,/uniform))

   new_time_sb1_1 = time_sb1_1(ind)
   resamp_V1_sb1_1 = V1_sb1_1(ind)
   new_dV1_sb1_1 = dV1_sb1_1(ind)

   ind = fix(sb1_num2*randomu(seed,sb1_num2,/uniform))

   new_time_sb1_2 = time_sb1_2(ind)
   resamp_V1_sb1_2 = V1_sb1_2(ind)
   new_dV1_sb1_2 = dV1_sb1_2(ind)
   
   ; Compute new array of resampled visibilities based on 
   ; randomly distributed errors

   new_theta = resamp_theta + new_dtheta*randomn(seed,vb_num,/normal) 
   new_rho = resamp_rho + new_drho*randomn(seed,vb_num,/normal) 

   new_sep_lun = resamp_sep_lun + new_dsep_lun*randomn(seed,lun_num,/normal) 

   new_V1_sb1_1 = resamp_V1_sb1_1 + new_dV1_sb1_1*randomn(seed,sb1_num1,/normal) 
  
   new_V1_sb1_2 = resamp_V1_sb1_2 + new_dV1_sb1_2*randomn(seed,sb1_num2,/normal) 
   
   count = 0

   delta_chi = 1.0		; set initially to begin loop

   lambda = 0.001               ; Marquardt method to ensure convergence

; ***** Begin while loop ***** 

while(delta_chi gt 0.001) do begin

    ; Determine errors in orbital elements
    ; set up matrices for error determination
    ; Invert matrix through Gauss Jordan elimination (Numerical Recipes in C)

    ; set up covariance matrix and column matrix
    ; alpha = dblarr(7,7)
    ; beta = dblarr(7)

    calc_deriv_vb, EL_vb, elfix_vb, mfit_vb, new_time_vb, new_theta, new_rho, new_dtheta, new_drho, theta_f, rho_f, alpha_vb, beta_vb, /flag_wa

    calc_deriv_lunar, EL_lun, elfix_lun, mfit_lun, new_time_lun, new_PA_lun, new_sep_lun, new_dsep_lun, sep_f, alpha_lun, beta_lun, /flag_wa

    calc_deriv_sb1, EL_sb1_1, elfix_sb1_1, mfit_sb1_1, new_time_sb1_1, new_V1_sb1_1, new_dV1_sb1_1, V1_f_sb1_1, alpha_sb1_1, beta_sb1_1

    calc_deriv_sb1, EL_sb1_2, elfix_sb1_2, mfit_sb1_2, new_time_sb1_2, new_V1_sb1_2, new_dV1_sb1_2, V1_f_sb1_2, alpha_sb1_2, beta_sb1_2

    ; Determine chi squared

    ; convert data points to x and y coordinates

    xfit = rho_f * cos(theta_f)	; x coordinate
    yfit = rho_f * sin(theta_f)	; y coordinate
			
    ;chi2old = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

    diff_theta = new_theta - theta_f

    ; account for any wrap-arounds from 360 deg to 0 deg

    for k=0, vb_num-1 do begin
        if (diff_theta(k) ge !dpi) then $
          diff_theta(k) = diff_theta(k) - 2*!dpi
        if (diff_theta(k) le -!dpi) then $
          diff_theta(k) = 2*!dpi + diff_theta(k)	
    endfor

    chi2old = total((diff_theta)^2/new_dtheta^2 + (new_rho - rho_f)^2/new_drho^2) $
                    + total((new_sep_lun - sep_f)^2/new_dsep_lun^2) $
                    + total((new_V1_sb1_1 - V1_f_sb1_1)^2/new_dV1_sb1_1^2) $
                    + total((new_V1_sb1_2 - V1_f_sb1_2)^2/new_dV1_sb1_2^2)

    ; combine VB lunar matrices (same number of elements)...

    ; initialize full alpha,beta arrays
    alpha = dblarr(mfit,mfit)
    beta = dblarr(mfit)

    beta(vb_subset) = beta(vb_subset) + beta_vb
    beta(lun_subset) = beta(lun_subset) + beta_lun
    beta(sb1_subset_1) = beta(sb1_subset_1) + beta_sb1_1
    beta(sb1_subset_2) = beta(sb1_subset_2) + beta_sb1_2

    for i=0, mfit_vb-1 do begin
        for j=0, mfit_vb-1 do begin
            alpha(vb_subset(i),vb_subset(j)) = $
              alpha(vb_subset(i),vb_subset(j)) + alpha_vb(i,j)
        endfor
    endfor

    for i=0, mfit_lun-1 do begin
        for j=0, mfit_lun-1 do begin
            alpha(lun_subset(i),lun_subset(j)) = $
              alpha(lun_subset(i),lun_subset(j)) + alpha_lun(i,j)
        endfor
    endfor

    for i=0, mfit_sb1_1-1 do begin
        for j=0, mfit_sb1_1-1 do begin
            alpha(sb1_subset_1(i),sb1_subset_1(j)) = $
              alpha(sb1_subset_1(i),sb1_subset_1(j)) + alpha_sb1_1(i,j)
        endfor
    endfor

    for i=0, mfit_sb1_2-1 do begin
        for j=0, mfit_sb1_2-1 do begin
            alpha(sb1_subset_2(i),sb1_subset_2(j)) = $
              alpha(sb1_subset_2(i),sb1_subset_2(j)) + alpha_sb1_2(i,j)
        endfor
    endfor

    invmat = invert(alpha, stat, /double)

    print,"stat (0 successful):",stat

    ;determine errors:

    ELerr = dblarr(mfit)

    for i=0, mfit-1 do ELerr(i) = sqrt(invmat(i,i))

    ; adjust alpha matrix by Marquardt parameter lambda

    for i=0, mfit-1 do alpha(i,i) = alpha(i,i)*(1.0 + lambda)

    if (mfit eq 1) then begin
        delta_el = dblarr(1)
        delta_el(0) = beta/alpha
     endif else begin

        if (determ(alpha) ne 0) then delta_el = cramer(alpha,beta) $ ; adjustments
        else begin
           repeat_iter = 1
           delta_el = dblarr(mfit)
           ;print,'inside check'
           ;stop
        endelse
           
     endelse
                                                
           
    ; adjust orbital parameters
    
    ELadj = EL
    ELadj(el_subset) = ELadj(el_subset) + delta_el 

    ELadj_vb  = ELadj(vb_par) 
    ELadj_lun  = ELadj(lun_par) 
    ELadj_sb1_1 = ELadj(sb1_par_1) 
    ELadj_sb1_2 = ELadj(sb1_par_2) 


    k=0
    print,"Orbital Element     Adjustment"
    for i=0, 3 do begin
        if (elfix(i) eq 0) then begin
            print, elLabel(i), ': ', El(i)
        endif else begin
            print, elLabel(i), ': ', El(i), delta_el(k)
            k=k+1
        endelse
    endfor
    for i=4, 6 do begin
         if (elfix(i) eq 0) then begin
             print, elLabel(i), ': ', El(i)*180/!dpi
         endif else begin
             print, elLabel(i), ': ', El(i)*180/!dpi, delta_el(k)*180/!dpi
             k = k+1
         endelse
    endfor
    for i=7, 9 do begin
        if (elfix(i) eq 0) then begin
            print, elLabel(i), ': ', El(i)
        endif else begin
            print, elLabel(i), ': ', El(i), delta_el(k)
            k=k+1
        endelse
    endfor

    print,"Chi2 of current solution:",chi2old


    ; calculate new chi2 for adjusted parameters

    calc_deriv_vb, ELadj_vb, elfix_vb, mfit_vb, new_time_vb, new_theta, new_rho, new_dtheta, new_drho, theta_f, rho_f, alpha_vb, beta_vb, /flag_wa

    calc_deriv_lunar, ELadj_lun, elfix_lun, mfit_lun, new_time_lun, new_PA_lun, new_sep_lun, new_dsep_lun, sep_f, alpha_lun, beta_lun, /flag_wa

    calc_deriv_sb1, ELadj_sb1_1, elfix_sb1_1, mfit_sb1_1, new_time_sb1_1, new_V1_sb1_1, new_dV1_sb1_1, V1_f_sb1_1, alpha_sb1_1, beta_sb1_1

    calc_deriv_sb1, ELadj_sb1_2, elfix_sb1_2, mfit_sb1_2, new_time_sb1_2, new_V1_sb1_2, new_dV1_sb1_2, V1_f_sb1_2, alpha_sb1_2, beta_sb1_2


    ; Determine chi squared

    ; convert data points to x and y coordinates

    xfit = rho_f * cos(theta_f)	; x coordinate
    yfit = rho_f * sin(theta_f)	; y coordinate
			
    ;chi2new = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

    diff_theta = new_theta - theta_f

    ; account for any wrap-arounds from 360 deg to 0 deg

    for k=0, vb_num-1 do begin
        if (diff_theta(k) ge !dpi) then $
          diff_theta(k) = diff_theta(k) - 2*!dpi
        if (diff_theta(k) le -!dpi) then $
          diff_theta(k) = 2*!dpi + diff_theta(k)	
    endfor


    chi2new = total((diff_theta)^2/new_dtheta^2 + (new_rho - rho_f)^2/new_drho^2) $
                    + total((new_sep_lun - sep_f)^2/new_dsep_lun^2) $
                    + total((new_V1_sb1_1 - V1_f_sb1_1)^2/new_dV1_sb1_1^2) $
                    + total((new_V1_sb1_2 - V1_f_sb1_2)^2/new_dV1_sb1_2^2)


    print,"chi2 of next modification:",chi2new

    if (chi2new le chi2old) then begin
        EL = ELadj
        EL_vb = ELadj_vb
        EL_lun = ELadj_lun
        EL_sb1_1 = ELadj_sb1_1
        EL_sb1_2 = ELadj_sb1_2
	chi2 = chi2new
        delta_chi = chi2old - chi2new
        lambda = lambda/10.0
        print, 'lambda', lambda
    endif else begin
	print, "next adjustment does not improve chi2"
        print, "increasing lambda by 10.0"
        lambda = lambda*10.0
        print, 'lambda', lambda
        delta_chi = 1.0
    endelse

    count = count+1

    ; do not exit unless lambda is less than 0.001

;    if (lambda gt 0.001) and (delta_chi lt 0.001) then delta_chi = 1.0
;    print,'lambda',lambda
;    print,'delta_chi',delta_chi

    ; alpha is singular - exit and repeat iteration
    if (repeat_iter) then begin
       ;print,'inside repeat loop'
       count = count - 1
       delta_chi = 1d-6
    endif
    
endwhile

; Save results from current bootstrap iteration

P_boot(ns) = EL(0) 
T_boot(ns) = EL(1) 
e_boot(ns) = EL(2) 
a_boot(ns) = EL(3) 
i_boot(ns) = EL(4) 
Wc_boot(ns) = EL(5)
wl_boot(ns) = EL(6)
K1_boot(ns) = EL(7)
V1_boot(ns) = EL(8)
V2_boot(ns) = EL(9) 

Mfunc = 3.784229924d-5 * K1_boot(ns)^3 * (yd*P_boot(ns)) * (1.0- e_boot(ns)^2)^(1.5)
   
M2 = 0.033573182d * a_boot(ns)^2/ parallax^2*K1_boot(ns)/(yd*P_boot(ns))/sin(i_boot(ns))*sqrt(1.0 - e_boot(ns)^2)

M1 = a_boot(ns)^3/parallax^3/(yd*P_boot(ns))^2 - M2

Mratio = M2/M1

mass = a_boot(ns)^3/parallax^3/(yd*P_boot(ns))^2

M1_boot(ns) = M1
M2_boot(ns) = M2
Mratio_boot(ns) = Mratio
Mtot_boot(ns) = mass
Mfunc_boot(ns) = Mfunc

lambda = 0.0

print,'Bootstrap iteration: ',ns

if (repeat_iter) then begin
   ns = ns - 1
   print,'alpha is singular - repeat iteration'
   repeat_iter = 0
   ;stop
endif else begin
   if (ans_mcrit) then begin
      ; Repeat iteration if mass is out of range
      if (mass lt Mmin) or (mass gt Mmax) then begin
         ns = ns - 1
         remove_M = remove_M + 1
      endif else begin
         if (M1 lt Mmin) or (M1 gt Mmax) then begin
            ns = ns - 1
            remove_M = remove_M + 1
         endif else if (M2 lt Mmin) or (M2 gt Mmax) then begin
            ns = ns - 1
            remove_M = remove_M + 1
         endif               
      endelse
   endif 
endelse

endfor  ; end bootstrap iterations



period = EL_orig(0)
Tperi = EL_orig(1)
ecc = EL_orig(2)
major = EL_orig(3)
inc = EL_orig(4)
W_cap = EL_orig(5)
w_low = EL_orig(6)
K1 = EL_orig(7)
Vsys1 = EL_orig(8)
Vsys2 = EL_orig(9)

; Save results to file
openw,1,'result_orb'
openw,2,'result_orb_mass'

printf,1,'#   P         T             e          a           i           Omega       omega         K1         Vsys_vis    Vsys_ir'
printf,1,format='(f12.5,f12.5,f12.6,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5)',period,Tperi,ecc,major,inc*180.0/!dpi,W_cap*180.0/!dpi,w_low*180.0/!dpi,K1,Vsys1,Vsys2

printf,2,'#   P         T             e          a           i           Omega       omega         K1         Vsys_vis    Vsys_ir      M1          M2          Mratio      Mtot        Mfunc'
printf,2,format='(f12.5,f12.5,f12.6,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5)',period,Tperi,ecc,major,inc*180.0/!dpi,W_cap*180.0/!dpi,w_low*180.0/!dpi,K1,Vsys1,Vsys2,M1_orig,M2_orig,Mratio_orig,mass_orig,Mfunc_orig

for i=0, nsamp-1 do begin

   printf,1,format='(f12.5,f12.5,f12.6,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5)',P_boot(i),T_boot(i),e_boot(i),a_boot(i),i_boot(i)*180.0/!dpi,Wc_boot(i)*180.0/!dpi,wl_boot(i)*180.0/!dpi,K1_boot(i),V1_boot(i),V2_boot(i)

   printf,2,format='(f12.5,f12.5,f12.6,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5,f12.5)',P_boot(i),T_boot(i),e_boot(i),a_boot(i),i_boot(i)*180.0/!dpi,Wc_boot(i)*180.0/!dpi,wl_boot(i)*180.0/!dpi,K1_boot(i),V1_boot(i),V2_boot(i),M1_boot(i),M2_boot(i),Mratio_boot(i),Mtot_boot(i),Mfunc_boot(i)

endfor

close,1
close,2


print, "Final values:"

P_best = mean(P_boot)
T_best = mean(T_boot)
e_best = mean(e_boot)
a_best = mean(a_boot)
i_best = mean(i_boot)
Wc_best = mean(Wc_boot)
wl_best = mean(wl_boot)
K1_best = mean(K1_boot)
V1_best = mean(V1_boot)
V2_best = mean(V2_boot)

P_err  = stdev(P_boot)
T_err  = stdev(T_boot)
e_err  = stdev(e_boot)
a_err  = stdev(a_boot)
i_err  = stdev(i_boot)
Wc_err = stdev(Wc_boot)
wl_err = stdev(wl_boot)
K1_err = stdev(K1_boot)
V1_err = stdev(V1_boot)
V2_err = stdev(V2_boot)

rtd = 180.0/!dpi

print,'Par  Min  Max  Mean  Median  Stdev'

print,'P:        ',min(P_boot),max(P_boot),mean(P_boot),median(P_boot),stdev(P_boot)
print,'T:        ',min(T_boot),max(T_boot),mean(T_boot),median(T_boot),stdev(T_boot)
print,'e:        ',min(e_boot),max(e_boot),mean(e_boot),median(e_boot),stdev(e_boot)
print,'a:        ',min(a_boot),max(a_boot),mean(a_boot),median(a_boot),stdev(a_boot)
print,'i:        ',min(i_boot)*rtd,max(i_boot)*rtd,mean(i_boot)*rtd,median(i_boot)*rtd,stdev(i_boot)*rtd
print,'Omega:    ',min(Wc_boot)*rtd,max(Wc_boot)*rtd,mean(Wc_boot)*rtd,median(Wc_boot)*rtd,stdev(Wc_boot)*rtd
print,'omega_A:  ',min(wl_boot)*rtd,max(wl_boot)*rtd,mean(wl_boot)*rtd,median(wl_boot)*rtd,stdev(wl_boot)*rtd
print,'K1(km/s): ',min(K1_boot),max(K1_boot),mean(K1_boot),median(K1_boot),stdev(K1_boot)
print,'Vsys1:    ',min(V1_boot),max(V1_boot),mean(V1_boot),median(V1_boot),stdev(V1_boot)
print,'Vsys2:    ',min(V2_boot),max(V2_boot),mean(V2_boot),median(V2_boot),stdev(V2_boot)
print,'Mfunc:   ',min(Mfunc_boot),max(Mfunc_boot),mean(Mfunc_boot),median(Mfunc_boot),stdev(Mfunc_boot)
print,'Mratio:   ',min(Mratio_boot),max(Mratio_boot),mean(Mratio_boot),median(Mratio_boot),stdev(Mratio_boot)
print,'Mtot:     ',min(Mtot_boot),max(Mtot_boot),mean(Mtot_boot),median(Mtot_boot),stdev(Mtot_boot)
print,'M1:       ',min(M1_boot),max(M1_boot),mean(M1_boot),median(M1_boot),stdev(M1_boot)
print,'M2:    ',min(M2_boot),max(M2_boot),mean(M2_boot),median(M2_boot),stdev(M2_boot)

; Compute uncertainties from 68% of values around median
; (better accomodates long, sparsely populated tails in distributions)

num34 = 0.34*nsamp
bootLabel = ['P      ','T      ','e      ','a      ','i      ','Omega  ','omega_A','K1(km/s)','Vsys1  ','Vsys2  ','Mfunc  ','Mratio ','Mtot   ','M1   ','M2   ']

bs = [1.0,1.0,1.0,1.0,180.0/!dpi,180.0/!dpi,180.0/!dpi,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0]


print,"Error range within +/-34% of median value:"

for i=0, 14 do begin

   if (i eq 0) then data_arr = P_boot
   if (i eq 1) then data_arr = T_boot
   if (i eq 2) then data_arr = e_boot
   if (i eq 3) then data_arr = a_boot
   if (i eq 4) then data_arr = i_boot
   if (i eq 5) then data_arr = Wc_boot
   if (i eq 6) then data_arr = wl_boot
   if (i eq 7) then data_arr = K1_boot
   if (i eq 8) then data_arr = V1_boot
   if (i eq 9) then data_arr = V2_boot
   if (i eq 10) then data_arr = Mfunc_boot
   if (i eq 11) then data_arr = Mratio_boot
   if (i eq 12) then data_arr = Mtot_boot
   if (i eq 13) then data_arr = M1_boot
   if (i eq 14) then data_arr = M2_boot


   ind_sort = sort(data_arr,/L64)

   data_sort = data_arr(ind_sort)
   
   med_val = median(data_arr)
   med_ind = where(abs(data_sort - med_val) eq min(abs(data_sort - med_val)))

   sum_lower = 0.0
   sum_upper = 0.0

   ; determine lower error bar
   arr_i = med_ind(0)
   while(sum_lower le num34) do begin
      sum_lower = sum_lower + 1
      arr_i = arr_i - 1
   endwhile
   bin_lower = arr_i + 1.0
   error_lower = med_val - data_sort(bin_lower(0))

   ; determine upper error bar
   arr_i = med_ind(0)
   while(sum_upper le num34) do begin
      sum_upper = sum_upper + 1
      arr_i = arr_i + 1
   endwhile
   bin_upper = arr_i - 1.0
   error_upper = data_sort(bin_upper(0)) - med_val

   print,bootLabel(i),med_val*bs(i),' + ',error_upper*bs(i),' - ',error_lower*bs(i)

   
endfor


print,"Compute uncertainties relative to best fit rather than median:"

EL_full = [EL_orig,Mfunc_orig,Mratio_orig,mass_orig,M1_orig,M2_orig] 

for i=0, 14 do begin

   if (i eq 0) then data_arr = P_boot
   if (i eq 1) then data_arr = T_boot
   if (i eq 2) then data_arr = e_boot
   if (i eq 3) then data_arr = a_boot
   if (i eq 4) then data_arr = i_boot
   if (i eq 5) then data_arr = Wc_boot
   if (i eq 6) then data_arr = wl_boot
   if (i eq 7) then data_arr = K1_boot
   if (i eq 8) then data_arr = V1_boot
   if (i eq 9) then data_arr = V2_boot
   if (i eq 10) then data_arr = Mfunc_boot
   if (i eq 11) then data_arr = Mratio_boot
   if (i eq 12) then data_arr = Mtot_boot
   if (i eq 13) then data_arr = M1_boot
   if (i eq 14) then data_arr = M2_boot


   ind_sort = sort(data_arr,/L64)

   data_sort = data_arr(ind_sort)
   
   med_val = median(data_arr)
   med_ind = where(abs(data_sort - med_val) eq min(abs(data_sort - med_val)))

   sum_lower = 0.0
   sum_upper = 0.0

   ; determine lower error bar
   arr_i = med_ind(0)
   while(sum_lower le num34) do begin
      sum_lower = sum_lower + 1
      arr_i = arr_i - 1
   endwhile
   bin_lower = arr_i + 1.0
   ;error_lower = med_val - data_sort(bin_lower(0))
   error_lower = EL_full(i) - data_sort(bin_lower(0))

   ; determine upper error bar
   arr_i = med_ind(0)
   while(sum_upper le num34) do begin
      sum_upper = sum_upper + 1
      arr_i = arr_i + 1
   endwhile
   bin_upper = arr_i - 1.0
   ;error_upper = data_sort(bin_upper(0)) - med_val
   error_upper = data_sort(bin_upper(0)) - EL_full(i)

   print,bootLabel(i),EL_full(i)*bs(i),' + ',error_upper*bs(i),' - ',error_lower*bs(i)

endfor



if (ans_mcrit) then begin
   print,'Number of solutions removed for Mcrit:',Remove_M
   print,'Fraction removed: ',double(Remove_M)/double(nsamp)
endif

;print,'P: ',P_best,P_err
;print,'T: ',T_best,T_err
;print,'e: ',e_best,e_err
;print,'a: ',a_best,a_err
;print,'i: ',i_best*180/!dpi,i_err*180/!dpi
;print,'Omega:    ',Wc_best*180/!dpi,Wc_err*180/!dpi
;print,'omega_A:  ',wl_best*180/!dpi,wl_err*180/!dpi
;print,'K1(km/s): ',K1_best,K1_err
;print,'Vsys1:    ',V1_best,V1_err
;print,'Vsys2:    ',V2_best,V2_err

; Set to best fit orbit:

period  = EL_orig(0)
Tperi   = EL_orig(1)
ecc     = EL_orig(2)
major   = EL_orig(3)
inc     = EL_orig(4)
W_cap   = EL_orig(5)
w_low   = EL_orig(6)
K1      = EL_orig(7)
Vsys1   = EL_orig(8)
Vsys2   = EL_orig(9)

   EL = [period,Tperi,ecc,major,inc,W_cap,w_low,K1,Vsys1,Vsys2]
   EL_vb = [period,Tperi,ecc,major,inc,W_cap,w_low]
   EL_lun = [period,Tperi,ecc,major,inc,W_cap,w_low]
   EL_sb1_1 = [period,Tperi,ecc,K1,w_low,Vsys1]
   EL_sb1_2 = [period,Tperi,ecc,K1,w_low,Vsys2]

; plot best-fit orbit and data

tnum=1000.0
tstep = period/tnum
tmin = Tperi

tarr = findgen(tnum)*tstep + tmin

calc_vbfit, EL_vb, tarr, theta_mod, rho_mod,/flag_wa

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

calc_vbfit, EL_vb, time_vb, theta_f, rho_f,/flag_wa

xfit = rho_f * cos(theta_f)	; x coordinate
yfit = rho_f * sin(theta_f)	; y coordinate

; plot data and best-fit positions for visual orbit
print,'Visual orbit'
plot,-yarr,xarr,psym=6
oplot,-yfit,xfit,psym=7
for l=0, vb_num-1 do oplot,[-yarr(l),-yfit(l)],[xarr(l),xfit(l)]

; plot best-fit orbit
oplot,-ymod,xmod

print,'Hit return to continue'
read,ans_cont


; plot model and data of initial guess for SB2 data

tnum = 1000.0
tmin = min([min(time_sb1_1),min(time_sb1_2)])
tmax = max([max(time_sb1_1),max(time_sb1_2)])
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb1fit, EL_sb1_1, tarr, V1mod_1

vmax = max([max(V1mod_1),max(V1_sb1_1+dV1_sb1_1),max(V1_sb1_2+dV1_sb1_2)])
vmin = min([min(V1mod_1),min(V1_sb1_1+dV1_sb1_1),min(V1_sb1_2+dV1_sb1_2)])
vrange = vmax - vmin
vmin = vmin - 0.05*vrange
vmax = vmax + 0.05*vrange

print,'SB1 velocity curve'
plot,time_sb1_1,V1_sb1_1,psym=2,yrange=[vmin,vmax]
oplot,time_sb1_2,V1_sb1_2 + (Vsys1 - Vsys2),psym=2
oplot,tarr,V1mod_1


print,'Lunar occultation separation (meas/calc):'
print,sep_lun,sep_f



; Define user symbol to be closed circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)

; Plot measured positions as filled circles

usersym, frac*cos(Z), frac*sin(Z),/fill

sepmax = 1.2*max(abs([yarr,xarr]))

set_plot, 'ps'
device,/Color,Bits_per_pixel=8,filename='temp_vb.eps',xsize=12,ysize=12,isolatin1=1,/encap

    ploterror,yarr,xarr,dy,dx,xtitle='!9D!3RA (mas)',ytitle='!9D!3DEC (mas)',ISOTROPIC=1, $
	 xrange=[sepmax,-sepmax], yrange=[-sepmax,sepmax], $
         xstyle=1,ystyle=1, /nodata, position=[0.19,0.12,0.95,0.98]

    ; Connected measured and fit values
    for i=0,vb_num-1 do oplot,[yfit(i),yarr(i)],[xfit(i),xarr(i)]

    ; Mark computed location with x
    oplot,[yfit],[xfit],psym=7

                                ; mark primary with an asterisk
    oplot,[0],[0],psym=2
    oplot,[-sepmax,sepmax],[0,0],linestyle=1
    oplot,[0,0],[-sepmax,sepmax],linestyle=1

    ;Define the symbol to be a closed circle
    usersym, frac*cos(Z), frac*sin(Z), /fill
    oplot,yarr,xarr,psym=8

; plot best-fit orbit
oplot,ymod,xmod,color=3



; plot lunar occultation projection and projected separation of binary

calc_vbfit, EL_vb, time_lun, theta_fit_lun, rho_fit_lun, /flag_wa

xfit_lun = rho_fit_lun * cos(theta_fit_lun)	; x coordinate
yfit_lun = rho_fit_lun * sin(theta_fit_lun)	; y coordinate

ra_lun = 150.0*sin(PA_lun)
dec_lun = 150.0*cos(PA_lun)

;oplot,[ra_lun,-ra_lun],[dec_lun,-dec_lun]
;print,ra_lun,dec_lun

ra_lun_meas = sep_lun*sin(PA_lun)
dec_lun_meas = sep_lun*cos(PA_lun)

if (yfit_lun le 0) then oplot,[0,ra_lun_meas],[0,dec_lun_meas],color=1,thick=5
if (yfit_lun ge 0) then oplot,[0,-ra_lun_meas],[0,-dec_lun_meas],color=1,thick=5


; find equation of line that passes through projected separation
; (perpendicular to lunar occultation PA)
; b = y - mx

m_slope = -sin(PA_lun)/cos(PA_lun)
b_int = sep_lun*cos(PA_lun) - m_slope*sep_lun*sin(PA_lun)

oplot,[150,-150],[150*m_slope+b_int,-150*m_slope+b_int],LineStyle=2

; and equivalent line at minus projected sep
b_int = -sep_lun*cos(PA_lun) + m_slope*sep_lun*sin(PA_lun)

;oplot,[100,-100],[100*m_slope+b_int,-100*m_slope+b_int],LineStyle=2

; Filled Squares for lunar occultation prediction
; Define user symbol to be open rectangle
xrect = frac*[-1.0,-1.0,1.0,1.0,-1.0]
yrect = frac*[-1.0,1.0,1.0,-1.0,-1.0]
usersym,xrect,yrect,/fill
oplot,[yfit_lun],[xfit_lun],psym=8,color=127;,symsize=1.5
usersym,xrect,yrect
oplot,[yfit_lun],[xfit_lun],psym=8

device,/close

tnum = 1000.0
tmin = min([time_sb1_1,time_sb1_2])
tmax = max([time_sb1_1,time_sb1_2])
trange = tmax - tmin
tmin = tmin - 0.05*trange
tmax = tmax + 0.05*trange
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb1fit, EL_sb1_1, tarr, V1mod

vmax = max([max(V1mod_1),max(V1_sb1_1+dV1_sb1_1),max(V1_sb1_2+dV1_sb1_2)])
vmin = min([min(V1mod_1),min(V1_sb1_1+dV1_sb1_1),min(V1_sb1_2+dV1_sb1_2)])
vrange = vmax - vmin
vmin = vmin - 0.05*vrange
vmax = vmax + 0.05*vrange


set_plot,'ps'
device, /Color, Bits_per_pixel=8, filename='temp_sb.eps',xsize=12,ysize=10,/encap

;toff = 50000.0d               ; RJD offset
;tlabel='JD - 2450000 (days)'
toff = 0.0d                     ; no offset
tlabel='Time'


ploterror,time_sb1_1-toff,V1_sb1_1,dV1_sb1_1,psym=2,yrange=[vmin,vmax],xrange=[tmin,tmax]-toff,/nodata, $
     xtitle=tlabel,ytitle='RV (km/s)'
usersym, frac*cos(Z), frac*sin(Z),/fill
oplot,time_sb1_1-toff,V1_sb1_1,psym=8,color=3
usersym,xrect,yrect,/fill
oploterror,time_sb1_2-toff,V1_sb1_2 - (Vsys2 - Vsys1),dV1_sb1_2,psym=8,/nodata
oplot,time_sb1_2-toff,V1_sb1_2 - (Vsys2 - Vsys1),psym=8,color=1
oplot,tarr-toff,V1mod

;oplot,time_sb1_2,V1_f_sb1_2 - (Vsys2 - Vsys1),psym=2,color=0

device,/close

set_plot,'x'

;stop

free_lun,lun

end
