pro fit_orbit_vbsb2_ell, vb_file, vb_num, sb2_file, sb2_num
  
; This version can handle elliptical uncertainties for visual orbits.  
; Minimize chi2 based on projections along the direction of error 
; ellipse (x',y').
;
; Compute the orbital elements through a Newton-Raphson technique.
; Fits simultaneously for VB + SB2 orbits
;
; Calls the following routines:
;     calc_deriv_vb_ell.pro
;     calc_deriv_sb2.pro
;     calc_Ei.pro
;     solve_trans.pro
;     calc_sb2fit.pro - for plotting model orbits
;     calc_vbfit.pro - for plotting model orbits
;
; Equations:
; + Visual Binary +  
; Initial estimates (P0,T0,e0,a0,i0,Omega0,omega0) and a set of observations 
; (ti,rhoi,thetai) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Radius vector:
;	r = a(1-e^2)/(1 + ecos(nu))
; Fitted positions (rho_fit,theta_fit) are determined from the orbital elements
; 	tan(theta - Omega) = tan(nu + omega)*cosi
;	rho = r*cos(nu + omega)*sec(theta - Omega)
; Project the separation into x',y' components in the direction of the
; error ellipse:
;       xp = rho*cos(theta - theta_err)
;       yp = rho*sin(theta - theta_err)
; + Spectroscopic Binary +  
; Initial estimates (P0,T0,e0,omega0,K1|0,K2|0,Vsys0) and  
; a set of observations (ti,V1i,V2i) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Projected semi-major axes (a1sini, a2sini):
;       a1sini = (P/2pi)*K1*(1 - e^2)^1/2
;       a2sini = (P/2pi)*K2*(1 - e^2)^1/2
; The fitted velocities (V1_i, V2_i) are determined from the orbital elements:
;       V1 =  K1*[e*cos(omega) + cos(nu + omega)] + Vsys
;       V2 = -K2*[e*cos(omega) + cos(nu + omega)] + Vsys
;
; Procedure:
; Minimize chi squared between data point positions and fitted positions:
;	chi^2 = sum[(xp_data - xp_fit)^2/sigma_major^2 
;		    + (yp_data - yp_fit)^2/sigma_minor^2]
;             + sum[(V1data - V1fit)^2/sigmaV1^2]
;		    + (V2data - V2fit)^2/sigmaV2^2]
; Take partial derivatives of chi^2 with respect to the orbital elements,
; (P,T,e,a(mas),i,Omega,omega,K1,K2,Vsys) and set to zero. 
; Solve for the  value of the orbital elements that minimize chi^2.
;
; Since the partial derivatives are non-linear, cannot solve for the orbital
; elements that minimize chi^2 analytically.
;
; In turn, use a Newton-Raphson technique to converge upon the solution.
; Replace (xpfit,ypfit,V1fit,V2fit) in the chi^2 equation with the
; Taylor series approximation,
; 	x = x| + dx/dP|(P-P0) + dx/dT|(T-T0) + dx/de|(e-e0) 
;	       + dx/da|(a-Aa) + dx/di|(i-i0) + dx/dW|(W-W0) + dx/dw|(w-w0)
;	       + dx/d(K1)|(K1 - K1|0) + dx/d(K2)|(K2 - K2|0)
;              + dx/d(Vsys)|(Vsys - Vsys0)
; where x=(xp,yp,V1,V2) and the | represents the value evaluated at for the
; initial estimates for the orbital elements. Now, xp, yp, V1, and V2 
; are linear in (P,T,e,a,i,W,w,K1,K2,Vsys), so the partial derivatives can 
; easily be taken and solved by setting up a matrix and using Cramer's method.
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;	major:	 semi-major axis (in mas) (a)
;	inc:	 inclination (i)
;	Omega:	 position angle of node (W)
;	omega:	 angle between node and periastron (w)
;       K1:      velocity amplitude of primary (in km/s) (K1)
;       K2:      velocity amplitude of secondary (in km/s) (K2)
;       Vsys:    systemic velocity
;	t_vb:	 time of observation
;	rho_d:	 separation - data
;	theta_d: postion angle - data
;	rho_f:	 sep - fit (determined from orbital elements for time of obs.)
;	theta_f: PA - fit (determined from orbital elements for time of obs.)
;	t_sb1:	 time of observation
;	V1_d:	 velocity of primary - measured data
;	V1_f:	 V1 - fit (determined from orbital elements for time of obs.)
;
; INPUT:
;	vb_file: file containing data points in the format:
;		      time  PA  sep  dmajor  dminor  theta_err
;		    where theta is the position angle and rho is the separation.
;                   The error ellipse is defined by the size of the major and 
;                   minor axes, and the position angle of the major axis in the
;                   plane of the sky (measured East of North).
;		    (enter file name using quotes)
;	vb_num: number of data points contained in data_file
;	
;	sb2_file: file containing data points in the format:
;		      time  V1  errV1  V2  errV2
;		      where V1 is the radial velocity of the primary
;		            V2 is the radial velocity of the secondary
;                     and errV1 and errV2 are the measurement errors
;		      (enter file name using quotes)
;	sb2_num: number of data points contained in data_file
;
; Prompted inputs:
;	initial estimates for all the orbital elements:
;             P, T, e, a, i, Omega, omega, K1, K2, Vsys
;       vector identifying which orbital elements to solve for
;             enter 0 to hold element fixed, 1 to vary)
;             ex.   1 1 1 1 1 1 1 1 1 1 -> solves for all
;                   0 0 0 0 1 0 0 0 0 0 -> solves only for i
;
; OUTPUT:
;	best fit values of orbital elements: P,T,e,a,i,Omega,omega,K1,K2,Vsys
;       and corresponding errors
;
; Notes on units:
;    P, T, and time should be in same units (days or years)
;    K1, K2, V1, V2 should be in same units (km/s)
;
;
; Began 14 May 2002
; Modified 11 Nov 2003: 
;       This is the current version of NEWT_RAPH.PRO
;	(v1.1 and v1.2 are older versions)
; 	- restructured program into separate subroutines
; 	- add option of holding elements fixed during iterations
; 13 January 2005:
;       Fix bugs
;       - if solving for only one element, don't use cramer.pro to solve eqn's
;       - fix 360 degree roll overs in (PAdata - PAfit)
;       Save version v4 as backup
; 11 July 2005:
;       - Include Marquadt Method of adjusting diagonal elements of
;         the alpha matrix to ensure convergence.
;       - Fixed non-convergence problems 
;           + do not remove P wrap around effects from dx/dP
;             leave (t-T)/P in full form; net effect is that adjustments
;             are smaller
;       Save v5 as backup 
;
; 31 May 2006: Save unmodified program as version 2
; 31 May 2006: Fit SB2 parameters in terms of K1 and K2
;              (as opposed to a1sini and a2sini)
;
; 22 September 2006:
;       - Fixed the way subsets of parameters are handled
;       - Wasn't properly constructing matrices if not all parameters vary
;
; 06 November 2015:
;       - Add flag_wa to calc_deriv_vb_ell.pro and calc_vbfit.pro.  This allows
;         the option of defining omega = omega_A in both the SB2 and VB orbits.

  
!P.font=0
!P.multi=0
!P.charsize=1.2
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.2	;scale factor for adjusting size of plotting symbols

; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d
temp6 = 0d

time_vb = dblarr(vb_num)
theta = dblarr(vb_num)
rho = dblarr(vb_num)
dmajor = dblarr(vb_num)
dminor = dblarr(vb_num)
theta_err = dblarr(vb_num)

time_sb2 = dblarr(sb2_num)
V1_sb2 = dblarr(sb2_num)
V2_sb2 = dblarr(sb2_num)
dV1_sb2 = dblarr(sb2_num)
dV2_sb2 = dblarr(sb2_num)

; Read VB measurements

openr,lun,vb_file,/get_lun

for i=0, vb_num-1 do begin

	readf, lun, temp1, temp2, temp3, temp4, temp5, temp6

	time_vb(i) = temp1
	theta(i) = temp2
	rho(i) = temp3	
	dmajor(i) = temp4
	dminor(i)= temp5
	theta_err(i) = temp6

endfor

close,lun

; convert theta and theta_err to radians

theta = theta*!dpi/180
theta_err = theta_err*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; project separation into components along the direction of the error ellipse

xparr = rho*cos(theta - theta_err)  ; x' (major)
yparr = rho*sin(theta - theta_err)  ; y' (minor) 

; Read SB2 measurements

openr,lun,sb2_file,/get_lun

for i=0, sb2_num-1 do begin

	readf, lun, temp1, temp2, temp3, temp4, temp5

	time_sb2(i) = temp1
	V1_sb2(i) = temp2
	dV1_sb2(i) = temp3
	V2_sb2(i) = temp4
	dV2_sb2(i) = temp5
	
endfor

close,lun

; Obtain values for P,T,e,a(mas),i,Omega,omega,K1(km/s),K2(km/s),Vsys

period = 0d
Tperi = 0d
ecc = 0d
major = 0d
inc = 0d
W_cap = 0d
w_low = 0d
K1 = 0d
K2 = 0d
Vsys = 0d

print,"Enter P,T,e,a(mas),i,Omega,omega,K1(km/s),K2(km/s),Vsys:"
read,period,Tperi,ecc,major,inc,W_cap,w_low,K1,K2,Vsys

print, "Vary each orbital element?"
print, "For each element, enter 0 to hold element fixed, 1 to vary:"
print, "[P,T,e,a,i,Omega,omega,K1,K2,Vsys]"
read, f0,f1,f2,f3,f4,f5,f6,f7,f8,f9

;; Order of orbital element arrays
;; do not set here yet (need to get angles into radians)
;; EL_vb = [period,Tperi,ecc,major,inc,W_cap,w_low]
;;             0    1     2    3    4    5     6
;; EL_sb2 = [period,Tperi,ecc,K1,K2,w_low,Vsys]
;;              0    1     2   7  8   6     9
;; EL_sb1 = [period,Tperi,ecc,K1,w_low,Vsys]
;;              0    1     2   7   6     9

elfix=[f0,f1,f2,f3,f4,f5,f6,f7,f8,f9]
nEl = n_elements(elfix)

elfix_vb=[f0,f1,f2,f3,f4,f5,f6]
nEl_vb = n_elements(elfix_vb)

elfix_sb2=[f0,f1,f2,f7,f8,f6,f9]
nEl_sb2 = n_elements(elfix_sb2)

;convert i, Omega, omega to radians
inc = inc*!dpi/180.0
W_cap = W_cap*!dpi/180.0
w_low = w_low*!dpi/180.0

k=0
for i=0, nEl-1 do if (elfix(i) eq 0) then k=k+1
mfit = nEl - k 		; number of elements to improve

k=0
for i=0, nEl_vb-1 do if (elfix_vb(i) eq 0) then k=k+1
mfit_vb = nEl_vb - k 		; number of elements to improve

k=0
for i=0, nEl_sb2-1 do if (elfix_sb2(i) eq 0) then k=k+1
mfit_sb2 = nEl_sb2 - k 		; number of elements to improve

EL = [period,Tperi,ecc,major,inc,W_cap,w_low,K1,K2,Vsys]
EL_vb = [period,Tperi,ecc,major,inc,W_cap,w_low]
EL_sb2 = [period,Tperi,ecc,K1,K2,w_low,Vsys]

ELadj = EL
elLabel = ['P','T','e','a(mas)','i','Omega','omega_A','K1(km/s)','K2(km/s)','Vsys']

; determine which indices of full set are being varied

vb_subset = dblarr(mfit_vb)
sb2_subset = dblarr(mfit_sb2)

; determine subarray of indices that are VB, SB1, and SB2 parameters

vb_par = dblarr(nEl_vb)
sb2_par = dblarr(nEl_sb2)

; determine indices of elements for which to vary
el_subset = where(elfix ne 0)
Elvar = El(el_subset)

k=0
for i=0, nEl_vb-1 do begin
    ind = where(EL_vb(i) eq EL)
    vb_par(i) = ind
    if (elfix_vb(i) ne 0) then begin
        ind = where(EL_vb(i) eq Elvar)
        vb_subset(k) = ind
        k=k+1
    endif
endfor

k=0
for i=0, nEl_sb2-1 do begin
    ind = where(EL_sb2(i) eq EL)
    sb2_par(i) = ind
    if (elfix_sb2(i) ne 0) then begin
         ind = where(EL_sb2(i) eq Elvar)
         sb2_subset(k) = ind
        k=k+1
    endif
endfor


; **** Plot model orbits ****

ans_cont = ' '

; plot model and data of initial guess for visual orbit

tnum=1000.0
tstep = period/tnum
tmin = Tperi

tarr = findgen(tnum)*tstep + tmin

calc_vbfit, EL_vb, tarr, theta_mod, rho_mod, /flag_wa

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

print,'Visual orbit'
plot,-yarr,xarr,psym=6
oplot,-ymod,xmod

print,'Hit return to continue'
read,ans_cont

; plot model and data of initial guess for SB2 data

tnum = 1000.0
tmin = min(time_sb2)
tmax = max(time_sb2)
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb2fit, EL_sb2, tarr, V1mod, V2mod

vmax = max([max(V1mod),max(V2mod)])
vmin = min([min(V1mod),min(V2mod)])

print,'SB2 velocity curve'
plot,time_sb2,V1_sb2,psym=2,yrange=[vmin,vmax]
oplot,time_sb2,V2_sb2,psym=6
oplot,tarr,V1mod
oplot,tarr,V2mod

vmax = max([max(V1_sb2),max(V2_sb2)])
vmin = min([min(V1_sb2),min(V2_sb2)])

print,'Hit return to continue'
read,ans_cont

count = 0

delta_chi = 1.0		; set initially to begin loop

lambda = 0.001   ; Marquardt method to ensure convergence

!P.MULTI = [0,2,0]

; ***** Begin while loop ***** 

while(delta_chi gt 0.001) do begin

    ; Determine errors in orbital elements
    ; set up matrices for error determination
    ; Invert matrix through Gauss Jordan elimination (Numerical Recipes in C)

    ; set up covariance matrix and column matrix
    ; alpha = dblarr(7,7)
    ; beta = dblarr(7)

    calc_deriv_vb_ell, EL_vb, elfix_vb, mfit_vb, time_vb, xparr, yparr, dmajor, dminor, theta_err, theta_f, rho_f, xpfit, ypfit, alpha_vb, beta_vb, /flag_wa

    calc_deriv_SB2, EL_sb2, elfix_sb2, mfit_sb2, time_sb2, V1_sb2, V2_sb2, dV1_sb2, dV2_sb2, V1_f_sb2, V2_f_sb2, alpha_sb2, beta_sb2

    ; Determine chi squared

    chi2old = total((xparr - xpfit)^2/dmajor^2 + (yparr - ypfit)^2/dminor^2) $
            + total((V1_sb2 - V1_f_sb2)^2/dV1_sb2^2 $
                  + (V2_sb2 - V2_f_sb2)^2/dV2_sb2^2)

    ; combine SB1 and VB matrices...

    ; initialize full alpha,beta arrays
    alpha = dblarr(mfit,mfit)
    beta = dblarr(mfit)

    beta(vb_subset) = beta(vb_subset) + beta_vb
    beta(sb2_subset) = beta(sb2_subset) + beta_sb2

    for i=0, mfit_vb-1 do begin
        for j=0, mfit_vb-1 do begin
            alpha(vb_subset(i),vb_subset(j)) = $
              alpha(vb_subset(i),vb_subset(j)) + alpha_vb(i,j)
        endfor
    endfor

    for i=0, mfit_sb2-1 do begin
        for j=0, mfit_sb2-1 do begin
            alpha(sb2_subset(i),sb2_subset(j)) = $
              alpha(sb2_subset(i),sb2_subset(j)) + alpha_sb2(i,j)
        endfor
    endfor

    invmat = invert(alpha, stat, /double)

    print,"stat (0 successful):",stat

    ;determine errors:

    ELerr = dblarr(mfit)

    for i=0, mfit-1 do ELerr(i) = sqrt(invmat(i,i))

    ; adjust alpha matrix by Marquardt parameter lambda

    for i=0, mfit-1 do alpha(i,i) = alpha(i,i)*(1.0 + lambda)

    if (mfit eq 1) then begin
        delta_el = dblarr(1)
        delta_el(0) = beta/alpha
    endif else delta_el = cramer(alpha,beta)   ; adjustments

    ; adjust orbital parameters

    ELadj = EL
    ELadj(el_subset) = ELadj(el_subset) + delta_el 

    ELadj_vb  = ELadj(vb_par) 
    ELadj_sb2 = ELadj(sb2_par) 

    k=0
    print,"Orbital Element     Adjustment"
    for i=0, 3 do begin
        if (elfix(i) eq 0) then begin
            print, elLabel(i), ': ', El(i)
        endif else begin
            print, elLabel(i), ': ', El(i), delta_el(k)
            k=k+1
        endelse
    endfor
    for i=4, 6 do begin
         if (elfix(i) eq 0) then begin
             print, elLabel(i), ': ', El(i)*180/!dpi
         endif else begin
             print, elLabel(i), ': ', El(i)*180/!dpi, delta_el(k)*180/!dpi
             k = k+1
         endelse
    endfor
    for i=7, 9 do begin
        if (elfix(i) eq 0) then begin
            print, elLabel(i), ': ', El(i)
        endif else begin
            print, elLabel(i), ': ', El(i), delta_el(k)
            k=k+1
        endelse
    endfor

    print,"Chi2 of current solution:",chi2old

    ; convert data points to x and y coordinates

    xfit = rho_f * cos(theta_f)	; x coordinate
    yfit = rho_f * sin(theta_f)	; y coordinate

    ; plot visual orbit

    plot,-yarr,xarr,psym=6
    oplot,-yfit,xfit,psym=7
    for l=0, vb_num-1 do oplot,[-yarr(l),-yfit(l)],[xarr(l),xfit(l)]

    ;plot SB velocity curves

    plot,time_sb2,V1_sb2,psym=2,yrange=[vmin,vmax],xrange=[tmin,tmax]
    oplot,time_sb2,V1_f_sb2,psym=7
    for l=0, sb2_num-1 do oplot,[time_sb2(l),time_sb2(l)],[V1_sb2(l),V1_f_sb2(l)]
    oplot,time_sb2,V2_sb2,psym=6
    oplot,time_sb2,V2_f_sb2,psym=4
    for l=0, sb2_num-1 do oplot,[time_sb2(l),time_sb2(l)],[V2_sb2(l),V2_f_sb2(l)]

    wait,0.1

    if (count eq 0) then begin
        print,'Hit return to continue'
        read,ans_cont
    endif

    ; calculate new chi2 for adjusted parameters

    calc_deriv_vb_ell, ELadj_vb, elfix_vb, mfit_vb, time_vb, xparr, yparr, dmajor, dminor, theta_err, theta_f, rho_f, xpfit, ypfit, alpha_vb, beta_vb, /flag_wa

    calc_deriv_SB2, ELadj_sb2, elfix_sb2, mfit_sb2, time_sb2, V1_sb2, V2_sb2, dV1_sb2, dV2_sb2, V1_f_sb2, V2_f_sb2, alpha_sb2, beta_sb2

    ; Determine chi squared

    chi2new = total((xparr - xpfit)^2/dmajor^2 + (yparr - ypfit)^2/dminor^2) $
            + total((V1_sb2 - V1_f_sb2)^2/dV1_sb2^2 $
                  + (V2_sb2 - V2_f_sb2)^2/dV2_sb2^2)

    print,"chi2 of next modification:",chi2new

    if (chi2new le chi2old) then begin
        EL = ELadj
        EL_vb = ELadj_vb
        EL_sb2 = ELadj_sb2
	chi2 = chi2new
        delta_chi = chi2old - chi2new
        lambda = lambda/10.0
        print, 'lambda', lambda
    endif else begin
	print, "next adjustment does not improve chi2"
        print, "increasing lambda by 10.0"
        lambda = lambda*10.0
        print, 'lambda', lambda
        delta_chi = 1.0
    endelse

    count = count+1

    ; do not exit unless lambda is less than 0.001

;    if (lambda gt 0.001) and (delta_chi lt 0.001) then delta_chi = 1.0
;    print,'lambda',lambda
;    print,'delta_chi',delta_chi

endwhile

lambda = 0.0

print, "Final values:"

period = EL(0)
Tperi = EL(1)
ecc = EL(2)
major =  EL(3)
inc = EL(4)
W_cap = EL(5)
w_low = EL(6)
K1 = EL(7)
K2 = EL(8)
Vsys = EL(9)

;; Determine the velocity amplitudes
;K1 = (2.0*!dpi/(period*sec))*(a1sini*km/sqrt(1 - ecc^2))
;K2 = (2.0*!dpi/(period*sec))*(a2sini*km/sqrt(1 - ecc^2))
;
;
;; Compute parallax
;parallax = major*sin(inc)/((a1sini + a2sini)*meters/1.0495d11)

; Determine final error matrix:

calc_deriv_vb_ell, EL_vb, elfix_vb, mfit_vb, time_vb, xparr, yparr, dmajor, dminor, theta_err, theta_f, rho_f, xpfit, ypfit, alpha_vb, beta_vb, /flag_wa

calc_deriv_SB2, EL_sb2, elfix_sb2, mfit_sb2, time_sb2, V1_sb2, V2_sb2, dV1_sb2, dV2_sb2, V1_f_sb2, V2_f_sb2, alpha_sb2, beta_sb2

; combine SB2 and VB matrices...

; initialize full alpha,beta arrays
alpha = dblarr(mfit,mfit)
beta = dblarr(mfit)

beta(vb_subset) = beta(vb_subset) + beta_vb
beta(sb2_subset) = beta(sb2_subset) + beta_sb2

for i=0, mfit_vb-1 do begin
    for j=0, mfit_vb-1 do begin
        alpha(vb_subset(i),vb_subset(j)) = $
          alpha(vb_subset(i),vb_subset(j)) + alpha_vb(i,j)
    endfor
endfor

for i=0, mfit_sb2-1 do begin
    for j=0, mfit_sb2-1 do begin
        alpha(sb2_subset(i),sb2_subset(j)) = $
          alpha(sb2_subset(i),sb2_subset(j)) + alpha_sb2(i,j)
    endfor
endfor

;determine errors:

invmat = invert(alpha, stat, /double)

chi2 = total((xparr-xpfit)^2/dmajor^2 + (yparr-ypfit)^2/dminor^2) $
            + total((V1_sb2 - V1_f_sb2)^2/dV1_sb2^2 $
                  + (V2_sb2 - V2_f_sb2)^2/dV2_sb2^2) 

;; convert data points to x and y coordinates
;xfit = rho_f * cos(theta_f)	; x coordinate
;yfit = rho_f * sin(theta_f)	; y coordinate
;chi2xy = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

; degrees of freedom
dof = 2.0*vb_num + 2.0*sb2_num - mfit

; reduced chi squared:
chi2red = chi2/dof

ELerr = dblarr(mfit)
for i=0,mfit-1 do ELerr(i) = sqrt(chi2red)*sqrt(invmat(i,i))

k=0
print,"Orbital Element     Error"
for i=0, 3 do begin
    if (elfix(i) eq 0) then begin
        print, elLabel(i), ': ', El(i)
    endif else begin
        print, elLabel(i), ': ', El(i), ELerr(k)
        k=k+1
    endelse
endfor
for i=4, 6 do begin
     if (elfix(i) eq 0) then begin
         print, elLabel(i), ': ', El(i)*180/!dpi
     endif else begin
         print, elLabel(i), ': ', El(i)*180/!dpi, ELerr(k)*180/!dpi
         k = k+1
     endelse
endfor
for i=7,9 do begin
    if (elfix(i) eq 0) then begin
        print, elLabel(i), ': ', El(i)
    endif else begin
        print, elLabel(i), ': ', El(i), ELerr(k)
        k=k+1
    endelse
endfor
;print,'K1:',K1
;print,'K2:',K2
;print,'Parallax',parallax
print, "Final chi2:", chi2
print, "Reduced chi2:", chi2red
print,'Number of iterations:',count
;print,'Final chi2:',chi2

!P.MULTI = [0,0,0]

; Define user symbol to be rectangle
xrect = frac*[-1.0,-1.0,1.0,1.0,-1.0]
yrect = frac*[-1.0,1.0,1.0,-1.0,-1.0]
usersym,xrect,yrect

; Define user symbol to be open circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)
;Define the symbol to be a unit circle with 16 points
usersym, frac*cos(Z), frac*sin(Z)

; plot best-fit orbit and data for visual orbit

tnum=1000.0
tstep = period/tnum
tmin = Tperi

tarr = findgen(tnum)*tstep + tmin

calc_vbfit, EL_vb, tarr, theta_mod, rho_mod, /flag_wa

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

calc_vbfit, EL_vb, time_vb, theta_f, rho_f,/flag_wa

xfit = rho_f * cos(theta_f)	; x coordinate
yfit = rho_f * sin(theta_f)	; y coordinate

; plot data and best-fit positions for visual orbit
print,'Visual orbit'
plot,-yarr,xarr,psym=6
oplot,-yfit,xfit,psym=7
for l=0, vb_num-1 do oplot,[-yarr(l),-yfit(l)],[xarr(l),xfit(l)]

; plot best-fit orbit for visual orbit
oplot,-ymod,xmod

print,'Hit return to continue'
read,ans_cont


sepmax = 1.2*max(abs([yarr,xarr]))

set_plot, 'ps'
device,/Color,Bits_per_pixel=8,filename='temp_vb.eps',xsize=12,ysize=12,isolatin1=1,/encap

plot,yarr,xarr,xtitle='!9D!3RA (mas)',ytitle='!9D!3DEC (mas)',ISOTROPIC=1, $
     xrange=[sepmax,-sepmax], yrange=[-sepmax,sepmax], $
     xstyle=1,ystyle=1, /nodata, position=[0.19,0.12,0.95,0.98]

; Connected measured and fit values
for i=0,vb_num-1 do oplot,[yfit(i),yarr(i)],[xfit(i),xarr(i)]

; Mark computed location with x
;oplot,[yfit],[xfit],psym=7

                            ; mark primary with an asterisk
oplot,[0],[0],psym=2
oplot,[-sepmax,sepmax],[0,0],linestyle=1
oplot,[0,0],[-sepmax,sepmax],linestyle=1

; plot best-fit orbit
oplot,ymod,xmod,color=3

; Plot measured values
;Define the symbol to be a closed circle
usersym, frac*cos(Z), frac*sin(Z), /fill
oplot,yarr,xarr,psym=8

; Plot error ellipses
for i=0, vb_num-1 do begin

   create_ellipse,xe,ye,100,major=dmajor(i),minor=dminor(i),xcent=xarr(i),ycent=yarr(i),theta=theta_err(i)*180.0/!dpi

   oplot,ye,xe,thick=4,color=1

endfor

device,/close
set_plot, 'x'

; plot best-fit orbit and data for SB1 data

tnum = 1000.0
tmin = min(time_sb2)
tmax = max(time_sb2)
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb2fit, EL_sb2, tarr, V1mod, V2mod

vmax = max([max(V1mod),max(V2mod)])
vmin = min([min(V1mod),min(V2mod)])

; plot data and best-fit positions
print,'SB2 velocity curve'
plot,time_sb2,V1_sb2,psym=2,yrange=[vmin,vmax]
oplot,time_sb2,V1_f_sb2,psym=7
for l=0, sb2_num-1 do oplot,[time_sb2(l),time_sb2(l)],[V1_sb2(l),V1_f_sb2(l)]
oplot,time_sb2,V2_sb2,psym=6
oplot,time_sb2,V2_f_sb2,psym=4
for l=0, sb2_num-1 do oplot,[time_sb2(l),time_sb2(l)],[V2_sb2(l),V2_f_sb2(l)]

; plot best-fit orbit
oplot,tarr,V1mod
oplot,tarr,V2mod

tnum = 1000.0
tmin = Tperi - period*0.25
tmax = Tperi + period + period*0.25
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb2fit, EL_sb2, tarr, V1mod, V2mod

; Determine plotting ranges based on RV curves and measurements
vmax = max([max(V1mod),max(V2mod),V1_sb2+dV1_sb2,V1_sb2-dV1_sb2,V2_sb2+dV2_sb2,V2_sb2-dV2_sb2])
vmin = min([min(V1mod),min(V2mod),V1_sb2+dV1_sb2,V1_sb2-dV1_sb2,V2_sb2+dV2_sb2,V2_sb2-dV2_sb2])

vrange = vmax - vmin
vmin = vmin - 0.05*vrange
vmax = vmax + 0.05*vrange

; compute phase from time of periastron passage
phase_sb2 = (time_sb2 - Tperi)/period
; reduce phase to -1.0 to 1.0
phase_sb2 = phase_sb2 - fix(phase_sb2)
; keep phase between 0.0 to 1.0
for i=0, sb2_num-1 do if (phase_sb2(i) lt 0.0) $
  then phase_sb2(i) = phase_sb2(i) + 1.0

phase_mod = (tarr-Tperi)/period

set_plot,'ps'
device, /Color, Bits_per_pixel=8, filename='temp_sb.eps',xsize=12,ysize=10,/encap

ploterror,phase_sb2,V1_sb2,dV1_sb2,psym=4,yrange=[vmin,vmax],xrange =[-0.1,1.1], $
  xtitle = '!3Phase', ytitle = 'Radial Velocity (km/s)', xstyle=1,ystyle=1,/NoData

oplot,[-0.1,1.1],[Vsys,Vsys],Linestyle=1

oplot,phase_mod,V1mod,LineStyle=0.0,color=3,thick=4
oplot,phase_mod,V2mod,Linestyle=0.0,color=1,thick=4

usersym, frac*cos(Z), frac*sin(Z),thick=2,/fill
oploterror,phase_sb2,V1_sb2,dV1_sb2,psym=8,color=3
usersym,xrect,yrect,thick=2,/fill
oploterror,phase_sb2,V2_sb2,dV2_sb2,psym=8,color=1


device,/close
set_plot, 'x'

free_lun,lun

end
