pro fit_orbit_vbsb_ell_multifile, vb_file, sb_pri_file, sb_sec_file
  
; This version can handle elliptical uncertainties for visual orbits.  
; Minimize chi2 based on projections along the direction of error 
; ellipse (x',y').
;
; Compute the orbital elements through a Newton-Raphson technique.
; Fits simultaneously for VB + SB2 orbits
; Allows for different number of RV measurements for primary/secondary.
;
; Calls the following routines:
;     calc_deriv_vb_ell.pro
;     calc_deriv_sb2.pro
;     calc_Ei.pro
;     solve_trans.pro
;     calc_sb2fit.pro - for plotting model orbits
;     calc_vbfit.pro - for plotting model orbits
;
; Equations:
; + Visual Binary +  
; Initial estimates (P0,T0,e0,a0,i0,Omega0,omega0) and a set of observations 
; (ti,rhoi,thetai) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Radius vector:
;	r = a(1-e^2)/(1 + ecos(nu))
; Fitted positions (rho_fit,theta_fit) are determined from the orbital elements
; 	tan(theta - Omega) = tan(nu + omega)*cosi
;	rho = r*cos(nu + omega)*sec(theta - Omega)
; Project the separation into x',y' components in the direction of the
; error ellipse:
;       xp = rho*cos(theta - theta_err)
;       yp = rho*sin(theta - theta_err)
; + Spectroscopic Binary +  
; Initial estimates (P0,T0,e0,omega0,K1|0,K2|0,Vsys0) and  
; a set of observations (ti,V1i,V2i) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Projected semi-major axes (a1sini, a2sini):
;       a1sini = (P/2pi)*K1*(1 - e^2)^1/2
;       a2sini = (P/2pi)*K2*(1 - e^2)^1/2
; The fitted velocities (V1_i, V2_i) are determined from the orbital elements:
;       V1 =  K1*[e*cos(omega) + cos(nu + omega)] + Vsys
;       V2 = -K2*[e*cos(omega) + cos(nu + omega)] + Vsys
;
; Procedure:
; Minimize chi squared between data point positions and fitted positions:
;	chi^2 = sum[(xp_data - xp_fit)^2/sigma_major^2 
;		    + (yp_data - yp_fit)^2/sigma_minor^2]
;             + sum[(V1data - V1fit)^2/sigmaV1^2]
;		    + (V2data - V2fit)^2/sigmaV2^2]
; Take partial derivatives of chi^2 with respect to the orbital elements,
; (P,T,e,a(mas),i,Omega,omega,K1,K2,Vsys) and set to zero. 
; Solve for the  value of the orbital elements that minimize chi^2.
;
; Since the partial derivatives are non-linear, cannot solve for the orbital
; elements that minimize chi^2 analytically.
;
; In turn, use a Newton-Raphson technique to converge upon the solution.
; Replace (xpfit,ypfit,V1fit,V2fit) in the chi^2 equation with the
; Taylor series approximation,
; 	x = x| + dx/dP|(P-P0) + dx/dT|(T-T0) + dx/de|(e-e0) 
;	       + dx/da|(a-Aa) + dx/di|(i-i0) + dx/dW|(W-W0) + dx/dw|(w-w0)
;	       + dx/d(K1)|(K1 - K1|0) + dx/d(K2)|(K2 - K2|0)
;              + dx/d(Vsys)|(Vsys - Vsys0)
; where x=(xp,yp,V1,V2) and the | represents the value evaluated at for the
; initial estimates for the orbital elements. Now, xp, yp, V1, and V2 
; are linear in (P,T,e,a,i,W,w,K1,K2,Vsys), so the partial derivatives can 
; easily be taken and solved by setting up a matrix and using Cramer's method.
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;	major:	 semi-major axis (in mas) (a)
;	inc:	 inclination (i)
;	Omega:	 position angle of node (W)
;	omega:	 angle between node and periastron (w)
;       K1:      velocity amplitude of primary (in km/s) (K1)
;       K2:      velocity amplitude of secondary (in km/s) (K2)
;       Vsys:    systemic velocity
;	t_vb:	 time of observation
;	rho_d:	 separation - data
;	theta_d: postion angle - data
;	rho_f:	 sep - fit (determined from orbital elements for time of obs.)
;	theta_f: PA - fit (determined from orbital elements for time of obs.)
;	t_sb1:	 time of observation
;	V1_d:	 velocity of primary - measured data
;	V1_f:	 V1 - fit (determined from orbital elements for time of obs.)
;
; INPUT:
;	vb_file: file containing data points in the format:
;		      time  PA  sep  dmajor  dminor  theta_err
;		    where theta is the position angle and rho is the separation.
;                   The error ellipse is defined by the size of the major and 
;                   minor axes, and the position angle of the major axis in the
;                   plane of the sky (measured East of North).
;		    (enter file name using quotes)
;	vb_num: number of data points contained in data_file
;	
;	sb2_file: file containing data points in the format:
;		      time  V1  errV1  V2  errV2
;		      where V1 is the radial velocity of the primary
;		            V2 is the radial velocity of the secondary
;                     and errV1 and errV2 are the measurement errors
;		      (enter file name using quotes)
;	sb2_num: number of data points contained in data_file
;
; Prompted inputs:
;	initial estimates for all the orbital elements:
;             P, T, e, a, i, Omega, omega, K1, K2, Vsys
;       vector identifying which orbital elements to solve for
;             enter 0 to hold element fixed, 1 to vary)
;             ex.   1 1 1 1 1 1 1 1 1 1 -> solves for all
;                   0 0 0 0 1 0 0 0 0 0 -> solves only for i
;
; OUTPUT:
;	best fit values of orbital elements: P,T,e,a,i,Omega,omega,K1,K2,Vsys
;       and corresponding errors
;
; Notes on units:
;    P, T, and time should be in same units (days or years)
;    K1, K2, V1, V2 should be in same units (km/s)
;
;
; Began 14 May 2002
; Modified 11 Nov 2003: 
;       This is the current version of NEWT_RAPH.PRO
;	(v1.1 and v1.2 are older versions)
; 	- restructured program into separate subroutines
; 	- add option of holding elements fixed during iterations
; 13 January 2005:
;       Fix bugs
;       - if solving for only one element, don't use cramer.pro to solve eqn's
;       - fix 360 degree roll overs in (PAdata - PAfit)
;       Save version v4 as backup
; 11 July 2005:
;       - Include Marquadt Method of adjusting diagonal elements of
;         the alpha matrix to ensure convergence.
;       - Fixed non-convergence problems 
;           + do not remove P wrap around effects from dx/dP
;             leave (t-T)/P in full form; net effect is that adjustments
;             are smaller
;       Save v5 as backup 
;
; 31 May 2006: Save unmodified program as version 2
; 31 May 2006: Fit SB2 parameters in terms of K1 and K2
;              (as opposed to a1sini and a2sini)
;
; 22 September 2006:
;       - Fixed the way subsets of parameters are handled
;       - Wasn't properly constructing matrices if not all parameters vary
;
; 06 November 2015:
;       - Add flag_wa to calc_deriv_vb_ell.pro and calc_vbfit.pro.  This allows
;         the option of defining omega = omega_A in both the SB2 and VB orbits.

  
close,1
close,2

!P.font=0
!P.multi=0
!P.charsize=1.2
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.2	;scale factor for adjusting size of plotting symbols

; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d
temp6 = 0d

vb_num = 10000
sb_pri_num = 10000
sb_sec_num = 10000

time_vb = dblarr(vb_num)
theta = dblarr(vb_num)
rho = dblarr(vb_num)
dmajor = dblarr(vb_num)
dminor = dblarr(vb_num)
theta_err = dblarr(vb_num)

time_sb_pri = dblarr(sb_pri_num)
V1_sb_pri = dblarr(sb_pri_num)
dV1_sb_pri = dblarr(sb_pri_num)

time_sb_sec = dblarr(sb_sec_num)
V2_sb_sec = dblarr(sb_sec_num)
dV2_sb_sec = dblarr(sb_sec_num)

; Read VB measurements

openr,1,vb_file
i = 0
while not EOF(1) do begin

	readf, 1, temp1, temp2, temp3, temp4, temp5, temp6

	time_vb(i) = temp1
	theta(i) = temp2
	rho(i) = temp3	
	dmajor(i) = temp4
	dminor(i)= temp5
	theta_err(i) = temp6

        i = i + 1
        
endwhile
close,1
vb_num = i
time_vb = time_vb(0:vb_num-1)
theta = theta(0:vb_num-1)
rho = rho(0:vb_num-1)
dmajor = dmajor(0:vb_num-1)
dminor = dminor(0:vb_num-1)
theta_err = theta_err(0:vb_num-1)

; convert theta and theta_err to radians

theta = theta*!dpi/180
theta_err = theta_err*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; project separation into components along the direction of the error ellipse

xparr = rho*cos(theta - theta_err)  ; x' (major)
yparr = rho*sin(theta - theta_err)  ; y' (minor) 

; Read SB measurements for primary

openr,1,sb_pri_file
i = 0
while not EOF(1) do begin

   readf,1, temp1, temp2, temp3
   
   time_sb_pri(i) = temp1
   V1_sb_pri(i) = temp2
   dV1_sb_pri(i) = temp3

   i = i+1
        
endwhile
close,1
sb_pri_num = i
time_sb_pri = time_sb_pri(0:sb_pri_num-1)
V1_sb_pri = V1_sb_pri(0:sb_pri_num-1)
dV1_sb_pri = dV1_sb_pri(0:sb_pri_num-1)

; Read SB measurements for secondary

openr,1,sb_sec_file
i = 0
while not EOF(1) do begin

	readf,1, temp1, temp2, temp3

	time_sb_sec(i) = temp1
	V2_sb_sec(i) = temp2
	dV2_sb_sec(i) = temp3

        i = i + 1
        
endwhile
close,1
sb_sec_num = i
time_sb_sec = time_sb_sec(0:sb_sec_num-1)
V2_sb_sec = V2_sb_sec(0:sb_sec_num-1)
dV2_sb_sec = dV2_sb_sec(0:sb_sec_num-1)


; Obtain values for P,T,e,a(mas),i,Omega,omega,K1(km/s),K2(km/s),Vsys

period = 0d
Tperi = 0d
ecc = 0d
major = 0d
inc = 0d
W_cap = 0d
w_low = 0d
K1 = 0d
K2 = 0d
Vsys1 = 0d
Vsys2 = 0d

print,"Enter P,T,e,a(mas),i,Omega,omega,K1(km/s),K2(km/s),Vsys1,Vsys2:"
read,period,Tperi,ecc,major,inc,W_cap,w_low,K1,K2,Vsys1,Vsys2

print, "Vary each orbital element?"
print, "For each element, enter 0 to hold element fixed, 1 to vary:"
print, "[P,T,e,a,i,Omega,omega,K1,K2,Vsys1,Vsys2]"
read, f0,f1,f2,f3,f4,f5,f6,f7,f8,f9,f10

;; Order of orbital element arrays
;; do not set here yet (need to get angles into radians)
;; EL_vb = [period,Tperi,ecc,major,inc,W_cap,w_low]
;;             0    1     2    3    4    5     6
;; EL_sb1 = [period,Tperi,ecc,K1,w_low,Vsys]
;;              0    1     2  7  6     9
;; EL_sb2 = [period,Tperi,ecc,K1,w_low,Vsys]
;;              0    1     2  8  6     10

elfix=[f0,f1,f2,f3,f4,f5,f6,f7,f8,f9,f10]
nEl = n_elements(elfix)

elfix_vb=[f0,f1,f2,f3,f4,f5,f6]
nEl_vb = n_elements(elfix_vb)

elfix_sb_pri=[f0,f1,f2,f7,f6,f9]
nEl_sb_pri = n_elements(elfix_sb_pri)

elfix_sb_sec=[f0,f1,f2,f8,f6,f10]
nEl_sb_sec = n_elements(elfix_sb_sec)

;convert i, Omega, omega to radians
inc = inc*!dpi/180.0
W_cap = W_cap*!dpi/180.0
w_low = w_low*!dpi/180.0


k=0
for i=0, nEl-1 do if (elfix(i) eq 0) then k=k+1
mfit = nEl - k 		; number of elements to improve

k=0
for i=0, nEl_vb-1 do if (elfix_vb(i) eq 0) then k=k+1
mfit_vb = nEl_vb - k 		; number of elements to improve

k=0
for i=0, nEl_sb_pri-1 do if (elfix_sb_pri(i) eq 0) then k=k+1
mfit_sb_pri = nEl_sb_pri - k 		; number of elements to improve

k=0
for i=0, nEl_sb_sec-1 do if (elfix_sb_sec(i) eq 0) then k=k+1
mfit_sb_sec = nEl_sb_sec - k 		; number of elements to improve

EL = [period,Tperi,ecc,major,inc,W_cap,w_low,K1,K2,Vsys1,Vsys2]
EL_vb = [period,Tperi,ecc,major,inc,W_cap,w_low]
EL_sb_pri = [period,Tperi,ecc,K1,w_low,Vsys1]
EL_sb_sec = [period,Tperi,ecc,K2,w_low,Vsys2]

ELadj = EL
elLabel = ['P','T','e','a(mas)','i','Omega','omega_A','K1(km/s)','K2(km/s)','Vsys1','Vsys2']

; determine which indices of full set are being varied

vb_subset = dblarr(mfit_vb)
sb_pri_subset = dblarr(mfit_sb_pri)
sb_sec_subset = dblarr(mfit_sb_sec)

; determine subarray of indices that are VB, SB1, and SB2 parameters

vb_par = dblarr(nEl_vb)
sb_pri_par = dblarr(nEl_sb_pri)
sb_sec_par = dblarr(nEl_sb_sec)

; determine indices of elements for which to vary
el_subset = where(elfix ne 0)
Elvar = El(el_subset)

k=0
for i=0, nEl_vb-1 do begin
    ind = where(EL_vb(i) eq EL)
    vb_par(i) = ind
    if (elfix_vb(i) ne 0) then begin
        ind = where(EL_vb(i) eq Elvar)
        vb_subset(k) = ind
        k=k+1
    endif
endfor

k=0
for i=0, nEl_sb_pri-1 do begin
    ind = where(EL_sb_pri(i) eq EL)
    sb_pri_par(i) = ind
    if (elfix_sb_pri(i) ne 0) then begin
         ind = where(EL_sb_pri(i) eq Elvar)
         sb_pri_subset(k) = ind
        k=k+1
    endif
endfor

k=0
for i=0, nEl_sb_sec-1 do begin
    ind = where(EL_sb_sec(i) eq EL)
    sb_sec_par(i) = ind
    if (elfix_sb_sec(i) ne 0) then begin
         ind = where(EL_sb_sec(i) eq Elvar)
         sb_sec_subset(k) = ind
        k=k+1
    endif
endfor

; **** Plot model orbits ****

ans_cont = ' '

; plot model and data of initial guess for visual orbit

tnum=1000.0
tstep = period/tnum
tmin = Tperi

tarr = findgen(tnum)*tstep + tmin

calc_vbfit, EL_vb, tarr, theta_mod, rho_mod, /flag_wa

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

print,'Visual orbit'
plot,-yarr,xarr,psym=6
oplot,-ymod,xmod

print,'Hit return to continue'
read,ans_cont

; plot model and data of initial guess for SB2 data


tnum = 1000.0
tmin = min(time_sb_pri)
tmax = max(time_sb_pri)
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb1fit, EL_sb_pri, tarr, V1mod

vmax = max(V1mod)
vmin = min(V1mod)

print,'SB1 velocity curve'
plot,time_sb_pri,V1_sb_pri,psym=2,yrange=[vmin,vmax]
oplot,tarr,V1mod

vmax = max(V1_sb_pri)
vmin = min(V1_sb_pri)


tnum = 1000.0
tmin = min(time_sb_sec)
tmax = max(time_sb_sec)
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb1fit, EL_sb_sec, tarr, V2mod

vmax = max(V2mod)
vmin = min(V2mod)

print,'SB1 velocity curve'
plot,time_sb_sec,V2_sb_sec,psym=2,yrange=[vmin,vmax]
oplot,tarr,V1mod

vmax = max(V2_sb_sec)
vmin = min(V2_sb_sec)

print,'Hit return to continue'
read,ans_cont

count = 0

delta_chi = 1.0		; set initially to begin loop

lambda = 0.001   ; Marquardt method to ensure convergence

!P.MULTI = [0,2,0]

; ***** Begin while loop ***** 

while(delta_chi gt 0.001) do begin

    ; Determine errors in orbital elements
    ; set up matrices for error determination
    ; Invert matrix through Gauss Jordan elimination (Numerical Recipes in C)

    ; set up covariance matrix and column matrix
    ; alpha = dblarr(7,7)
    ; beta = dblarr(7)

    calc_deriv_vb_ell, EL_vb, elfix_vb, mfit_vb, time_vb, xparr, yparr, dmajor, dminor, theta_err, theta_f, rho_f, xpfit, ypfit, alpha_vb, beta_vb, /flag_wa

    calc_deriv_sb1, EL_sb_pri, elfix_sb_pri, mfit_sb_pri, time_sb_pri, V1_sb_pri, dV1_sb_pri, V1_f_sb_pri, alpha_sb_pri, beta_sb_pri

    calc_deriv_sb1, EL_sb_sec, elfix_sb_sec, mfit_sb_sec, time_sb_sec, V2_sb_sec, dV2_sb_sec, V2_f_sb_sec, alpha_sb_sec, beta_sb_sec, /flag_wa

    ; Determine chi squared

    chi2old = total((xparr - xpfit)^2/dmajor^2 + (yparr - ypfit)^2/dminor^2) $
                    + total((V1_sb_pri - V1_f_sb_pri)^2/dV1_sb_pri^2) $
                    + total((V2_sb_sec - V2_f_sb_sec)^2/dV2_sb_sec^2)

    ; combine SB1 and VB matrices...

    ; initialize full alpha,beta arrays
    alpha = dblarr(mfit,mfit)
    beta = dblarr(mfit)

    beta(vb_subset) = beta(vb_subset) + beta_vb
    beta(sb_pri_subset) = beta(sb_pri_subset) + beta_sb_pri
    beta(sb_sec_subset) = beta(sb_sec_subset) + beta_sb_sec

    for i=0, mfit_vb-1 do begin
        for j=0, mfit_vb-1 do begin
            alpha(vb_subset(i),vb_subset(j)) = $
              alpha(vb_subset(i),vb_subset(j)) + alpha_vb(i,j)
        endfor
    endfor

    for i=0, mfit_sb_pri-1 do begin
        for j=0, mfit_sb_pri-1 do begin
            alpha(sb_pri_subset(i),sb_pri_subset(j)) = $
              alpha(sb_pri_subset(i),sb_pri_subset(j)) + alpha_sb_pri(i,j)
        endfor
    endfor

    for i=0, mfit_sb_sec-1 do begin
        for j=0, mfit_sb_sec-1 do begin
            alpha(sb_sec_subset(i),sb_sec_subset(j)) = $
              alpha(sb_sec_subset(i),sb_sec_subset(j)) + alpha_sb_sec(i,j)
        endfor
    endfor

    invmat = invert(alpha, stat, /double)

    print,"stat (0 successful):",stat

    ;determine errors:

    ELerr = dblarr(mfit)

    for i=0, mfit-1 do ELerr(i) = sqrt(invmat(i,i))

    ; adjust alpha matrix by Marquardt parameter lambda

    for i=0, mfit-1 do alpha(i,i) = alpha(i,i)*(1.0 + lambda)

    if (mfit eq 1) then begin
        delta_el = dblarr(1)
        delta_el(0) = beta/alpha
    endif else delta_el = cramer(alpha,beta)   ; adjustments

    
    ; adjust orbital parameters

    ELadj = EL
    ELadj(el_subset) = ELadj(el_subset) + delta_el 

    ELadj_vb  = ELadj(vb_par) 
    ELadj_sb_pri = ELadj(sb_pri_par) 
    ELadj_sb_sec = ELadj(sb_sec_par) 
 
    k=0
    print,"Orbital Element     Adjustment"
    for i=0, 3 do begin
        if (elfix(i) eq 0) then begin
            print, elLabel(i), ': ', El(i)
        endif else begin
            print, elLabel(i), ': ', El(i), delta_el(k)
            k=k+1
        endelse
    endfor
    for i=4, 6 do begin
         if (elfix(i) eq 0) then begin
             print, elLabel(i), ': ', El(i)*180/!dpi
         endif else begin
             print, elLabel(i), ': ', El(i)*180/!dpi, delta_el(k)*180/!dpi
             k = k+1
         endelse
    endfor
    for i=7, 10 do begin
        if (elfix(i) eq 0) then begin
            print, elLabel(i), ': ', El(i)
        endif else begin
            print, elLabel(i), ': ', El(i), delta_el(k)
            k=k+1
        endelse
    endfor

    print,"Chi2 of current solution:",chi2old

    ; convert data points to x and y coordinates

    xfit = rho_f * cos(theta_f)	; x coordinate
    yfit = rho_f * sin(theta_f)	; y coordinate

    ; plot visual orbit

    plot,-yarr,xarr,psym=6
    oplot,-yfit,xfit,psym=7
    for l=0, vb_num-1 do oplot,[-yarr(l),-yfit(l)],[xarr(l),xfit(l)]

    ;plot SB velocity curves

    plot,time_sb_pri,V1_sb_pri,psym=2,yrange=[vmin,vmax],xrange=[tmin,tmax]
    oplot,time_sb_pri,V1_f_sb_pri,psym=7
    for l=0, sb_pri_num-1 do oplot,[time_sb_pri(l),time_sb_pri(l)],[V1_sb_pri(l),V1_f_sb_pri(l)]

    plot,time_sb_sec,V2_sb_sec,psym=2,yrange=[vmin,vmax],xrange=[tmin,tmax]
    oplot,time_sb_sec,V2_f_sb_sec,psym=7
    for l=0, sb_sec_num-1 do oplot,[time_sb_sec(l),time_sb_sec(l)],[V2_sb_sec(l),V2_f_sb_sec(l)]

    wait,0.1

    if (count eq 0) then begin
        print,'Hit return to continue'
        read,ans_cont
    endif

    ; calculate new chi2 for adjusted parameters

    calc_deriv_vb_ell, ELadj_vb, elfix_vb, mfit_vb, time_vb, xparr, yparr, dmajor, dminor, theta_err, theta_f, rho_f, xpfit, ypfit, alpha_vb, beta_vb, /flag_wa

    calc_deriv_SB1, ELadj_sb_pri, elfix_sb_pri, mfit_sb_pri, time_sb_pri, V1_sb_pri, dV1_sb_pri, V1_f_sb_pri, alpha_sb_pri, beta_sb_pri

    calc_deriv_SB1, ELadj_sb_sec, elfix_sb_sec, mfit_sb_sec, time_sb_sec, V2_sb_sec, dV2_sb_sec, V2_f_sb_sec, alpha_sb_sec, beta_sb_sec, /flag_wa

    ; Determine chi squared

    chi2new = total((xparr - xpfit)^2/dmajor^2 + (yparr - ypfit)^2/dminor^2) $
                    + total((V1_sb_pri - V1_f_sb_pri)^2/dV1_sb_pri^2) $
                    + total((V2_sb_sec - V2_f_sb_sec)^2/dV2_sb_sec^2)

    print,"chi2 of next modification:",chi2new

    if (chi2new le chi2old) then begin
        EL = ELadj
        EL_vb = ELadj_vb
        EL_sb_pri = ELadj_sb_pri
        EL_sb_sec = ELadj_sb_sec
	chi2 = chi2new
        delta_chi = chi2old - chi2new
        lambda = lambda/10.0
        print, 'lambda', lambda
    endif else begin
	print, "next adjustment does not improve chi2"
        print, "increasing lambda by 10.0"
        lambda = lambda*10.0
        print, 'lambda', lambda
        delta_chi = 1.0
    endelse

    count = count+1

    ; do not exit unless lambda is less than 0.001

;    if (lambda gt 0.001) and (delta_chi lt 0.001) then delta_chi = 1.0
;    print,'lambda',lambda
;    print,'delta_chi',delta_chi

endwhile

lambda = 0.0

print, "Final values:"

period = EL(0)
Tperi = EL(1)
ecc = EL(2)
major =  EL(3)
inc = EL(4)
W_cap = EL(5)
w_low = EL(6)
K1 = EL(7)
K2 = EL(8)
Vsys1 = EL(9)
Vsys2 = EL(10)

;; Determine the velocity amplitudes
;K1 = (2.0*!dpi/(period*sec))*(a1sini*km/sqrt(1 - ecc^2))
;K2 = (2.0*!dpi/(period*sec))*(a2sini*km/sqrt(1 - ecc^2))
;
;
;; Compute parallax
;parallax = major*sin(inc)/((a1sini + a2sini)*meters/1.0495d11)

; Determine final error matrix:

calc_deriv_vb_ell, EL_vb, elfix_vb, mfit_vb, time_vb, xparr, yparr, dmajor, dminor, theta_err, theta_f, rho_f, xpfit, ypfit, alpha_vb, beta_vb, /flag_wa

calc_deriv_sb1, EL_sb_pri, elfix_sb_pri, mfit_sb_pri, time_sb_pri, V1_sb_pri, dV1_sb_pri, V1_f_sb_pri, alpha_sb_pri, beta_sb_pri

calc_deriv_sb1, EL_sb_sec, elfix_sb_sec, mfit_sb_sec, time_sb_sec, V2_sb_sec, dV2_sb_sec, V2_f_sb_sec, alpha_sb_sec, beta_sb_sec, /flag_wa

; combine SB2 and VB matrices...

; initialize full alpha,beta arrays
alpha = dblarr(mfit,mfit)
beta = dblarr(mfit)

beta(vb_subset) = beta(vb_subset) + beta_vb
beta(sb_pri_subset) = beta(sb_pri_subset) + beta_sb_pri
beta(sb_sec_subset) = beta(sb_sec_subset) + beta_sb_sec

for i=0, mfit_vb-1 do begin
    for j=0, mfit_vb-1 do begin
        alpha(vb_subset(i),vb_subset(j)) = $
          alpha(vb_subset(i),vb_subset(j)) + alpha_vb(i,j)
    endfor
endfor

for i=0, mfit_sb_pri-1 do begin
    for j=0, mfit_sb_pri-1 do begin
        alpha(sb_pri_subset(i),sb_pri_subset(j)) = $
          alpha(sb_pri_subset(i),sb_pri_subset(j)) + alpha_sb_pri(i,j)
    endfor
endfor

for i=0, mfit_sb_sec-1 do begin
    for j=0, mfit_sb_sec-1 do begin
        alpha(sb_sec_subset(i),sb_sec_subset(j)) = $
          alpha(sb_sec_subset(i),sb_sec_subset(j)) + alpha_sb_sec(i,j)
    endfor
endfor

;determine errors:

invmat = invert(alpha, stat, /double)

chi2 = total((xparr-xpfit)^2/dmajor^2 + (yparr-ypfit)^2/dminor^2) $
             + total((V1_sb_pri - V1_f_sb_pri)^2/dV1_sb_pri^2) $
             + total((V2_sb_sec - V2_f_sb_sec)^2/dV2_sb_sec^2)

chi2_vb = total((xparr-xpfit)^2/dmajor^2 + (yparr-ypfit)^2/dminor^2)

chi2_sb_pri = total((V1_sb_pri - V1_f_sb_pri)^2/dV1_sb_pri^2)
chi2_sb_sec = total((V2_sb_sec - V2_f_sb_sec)^2/dV2_sb_sec^2)

;; convert data points to x and y coordinates
;xfit = rho_f * cos(theta_f)	; x coordinate
;yfit = rho_f * sin(theta_f)	; y coordinate
;chi2xy = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

; degrees of freedom
dof = 2d*vb_num + sb_pri_num + sb_sec_num - mfit
dof_vb = 2d*vb_num - mfit_vb
dof_sb_pri = sb_pri_num - mfit_sb_pri
dof_sb_sec = sb_sec_num - mfit_sb_sec

; reduced chi squared:
chi2red = chi2/dof

chi2red_vb = chi2_vb/dof_vb
chi2red_sb_pri = chi2_sb_pri/dof_sb_pri
chi2red_sb_sec = chi2_sb_sec/dof_sb_sec

ELerr = dblarr(mfit)
for i=0,mfit-1 do ELerr(i) = sqrt(chi2red)*sqrt(invmat(i,i))

k=0
print,"Orbital Element     Error"
for i=0, 3 do begin
    if (elfix(i) eq 0) then begin
        print, elLabel(i), ': ', El(i)
    endif else begin
        print, elLabel(i), ': ', El(i), ELerr(k)
        k=k+1
    endelse
endfor
for i=4, 6 do begin
     if (elfix(i) eq 0) then begin
         print, elLabel(i), ': ', El(i)*180/!dpi
     endif else begin
         print, elLabel(i), ': ', El(i)*180/!dpi, ELerr(k)*180/!dpi
         k = k+1
     endelse
endfor
for i=7,10 do begin
    if (elfix(i) eq 0) then begin
        print, elLabel(i), ': ', El(i)
    endif else begin
        print, elLabel(i), ': ', El(i), ELerr(k)
        k=k+1
    endelse
endfor
;print,'K1:',K1
;print,'K2:',K2
;print,'Parallax',parallax
print, "Final chi2:", chi2
print, "Reduced chi2:", chi2red
print, 'Degrees of freedom: ', dof
print, 'Number of VB data points: ',vb_num 
print, 'Number of SB primary data points: ', sb_pri_num
print, 'Number of SB secondary data points: ', sb_sec_num
print, 'VB only chi2/redchi2/dof: ',chi2_vb,chi2red_vb,dof_vb
print, 'SB pri only chi2/redchi2/dof: ',chi2_sb_pri,chi2red_sb_pri,dof_sb_pri
print, 'SB sec only chi2/redchi2/dof: ',chi2_sb_sec,chi2red_sb_sec,dof_sb_sec
print,'Number of iterations:',count
;print,'Final chi2:',chi2

bs = [1.0,1.0,1.0,1.0,180.0/!dpi,180.0/!dpi,180.0/!dpi,1.0,1.0,1.0,1.0]

print,EL*bs

stop

!P.MULTI = [0,0,0]

; Define user symbol to be rectangle
xrect = frac*[-1.0,-1.0,1.0,1.0,-1.0]
yrect = frac*[-1.0,1.0,1.0,-1.0,-1.0]
usersym,xrect,yrect

; Define user symbol to be open circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)
;Define the symbol to be a unit circle with 16 points
usersym, frac*cos(Z), frac*sin(Z)

; plot best-fit orbit and data for visual orbit

tnum=1000.0
tstep = period/tnum
tmin = Tperi

tarr = findgen(tnum)*tstep + tmin

calc_vbfit, EL_vb, tarr, theta_mod, rho_mod, /flag_wa

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

calc_vbfit, EL_vb, time_vb, theta_f, rho_f,/flag_wa

xfit = rho_f * cos(theta_f)	; x coordinate
yfit = rho_f * sin(theta_f)	; y coordinate

; plot data and best-fit positions for visual orbit
print,'Visual orbit'
plot,-yarr,xarr,psym=6
oplot,-yfit,xfit,psym=7
for l=0, vb_num-1 do oplot,[-yarr(l),-yfit(l)],[xarr(l),xfit(l)]

; plot best-fit orbit for visual orbit
oplot,-ymod,xmod

print,'Hit return to continue'
read,ans_cont


;sepmax = 1.2*max(abs([yarr,xarr]))
sepmax = 1.05*max(abs([yarr,xarr,ymod,xmod]))

set_plot, 'ps'
device,/Color,Bits_per_pixel=8,filename='temp_vb.eps',xsize=12,ysize=12,isolatin1=1,/encap

plot,yarr,xarr,xtitle='!9D!3RA (mas)',ytitle='!9D!3DEC (mas)',ISOTROPIC=1, $
     xrange=[sepmax,-sepmax], yrange=[-sepmax,sepmax], $
     xstyle=1,ystyle=1, /nodata, position=[0.19,0.12,0.95,0.98]

; Connected measured and fit values
for i=0,vb_num-1 do oplot,[yfit(i),yarr(i)],[xfit(i),xarr(i)]

; Mark computed location with x
;oplot,[yfit],[xfit],psym=7

                            ; mark primary with an asterisk
oplot,[0],[0],psym=2
oplot,[-sepmax,sepmax],[0,0],linestyle=1
oplot,[0,0],[-sepmax,sepmax],linestyle=1

; plot best-fit orbit
oplot,ymod,xmod,color=3

; Plot measured values
;Define the symbol to be a closed circle
usersym, frac*cos(Z), frac*sin(Z), /fill
oplot,yarr,xarr,psym=8

; Plot error ellipses
for i=0, vb_num-1 do begin

   create_ellipse,xe,ye,100,major=dmajor(i),minor=dminor(i),xcent=xarr(i),ycent=yarr(i),theta=theta_err(i)*180.0/!dpi

   oplot,ye,xe,thick=4,color=1

endfor

device,/close
set_plot, 'x'


; plot best-fit orbit and data for SB data for primary

tnum = 1000.0
tmin = min(time_sb_pri)
tmax = max(time_sb_pri)
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb1fit, EL_sb_pri, tarr, V1mod

vmax = max(V1mod)
vmin = min(V1mod)

; plot data and best-fit positions
print,'SB1 velocity curve'
plot,time_sb_pri,V1_sb_pri,psym=2,yrange=[vmin,vmax]
oplot,time_sb_pri,V1_f_sb_pri,psym=7
for l=0, sb_pri_num-1 do oplot,[time_sb_pri(l),time_sb_pri(l)],[V1_sb_pri(l),V1_f_sb_pri(l)]

; plot best-fit orbit
oplot,tarr,V1mod


tnum = 1000.0
tmin = Tperi - period*0.25
tmax = Tperi + period + period*0.25
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb1fit, EL_sb_pri, tarr, V1mod

vmax = max([max(V1mod),max(V1_sb_pri+dV1_sb_pri)])
vmin = min([min(V1mod),min(V1_sb_pri-dV1_sb_pri)])
vrange = vmax - vmin
vmin = vmin - 0.05*vrange
vmax = vmax + 0.05*vrange

; compute phase from time of periastron passage
phase_sb_pri = (time_sb_pri - Tperi)/period
; reduce phase to -1.0 to 1.0
phase_sb_pri = phase_sb_pri - fix(phase_sb_pri)
; keep phase between 0.0 to 1.0
for i=0, sb_pri_num-1 do if (phase_sb_pri(i) lt 0.0) $
  then phase_sb_pri(i) = phase_sb_pri(i) + 1.0

phase_mod = (tarr-Tperi)/period

set_plot,'ps'
device, /Color, Bits_per_pixel=8, filename='temp_sb_pri.eps',xsize=12,ysize=10,/encap

;toff = 50000.0d               ; RJD offset
;tlabel='JD - 2450000 (days)'
toff = 0.0d                     ; no offset
tlabel='Time'

; Define user symbol to be rectangle
xrect = frac*[-1.0,-1.0,1.0,1.0,-1.0]
yrect = frac*[-1.0,1.0,1.0,-1.0,-1.0]

ploterror,phase_sb_pri,V1_sb_pri,dV1_sb_pri,psym=2,yrange=[vmin,vmax],xrange =[-0.1,1.1], $
  xtitle = 'Phase', ytitle = 'RV (km/s)', xstyle=1,/NoData
oplot,phase_mod,V1mod
usersym, frac*cos(Z), frac*sin(Z),/fill
oplot,phase_sb_pri,V1_sb_pri,psym=8,color=3

device,/close
set_plot, 'x'


; plot best-fit orbit and data for SB1 data secondary

tnum = 1000.0
tmin = min(time_sb_sec)
tmax = max(time_sb_sec)
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb1fit, EL_sb_sec, tarr, V2mod, /flag_wa

vmax = max(V1mod)
vmin = min(V1mod)

; plot data and best-fit positions
print,'SB1 velocity curve'
plot,time_sb_sec,V2_sb_sec,psym=2,yrange=[vmin,vmax]
oplot,time_sb_sec,V2_f_sb_sec,psym=7
for l=0, sb_sec_num-1 do oplot,[time_sb_sec(l),time_sb_sec(l)],[V2_sb_sec(l),V2_f_sb_sec(l)]

; plot best-fit orbit
oplot,tarr,V2mod


tnum = 1000.0
tmin = Tperi - period*0.25
tmax = Tperi + period + period*0.25
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb1fit, EL_sb_sec, tarr, V2mod, /flag_wa

vmax = max([max(V2mod),max(V2_sb_sec+dV2_sb_sec)])
vmin = min([min(V2mod),min(V2_sb_sec-dV2_sb_sec)])
vrange = vmax - vmin
vmin = vmin - 0.05*vrange
vmax = vmax + 0.05*vrange

; compute phase from time of periastron passage
phase_sb_sec = (time_sb_sec - Tperi)/period
; reduce phase to -1.0 to 1.0
phase_sb_sec = phase_sb_sec - fix(phase_sb_sec)
; keep phase between 0.0 to 1.0
for i=0, sb_sec_num-1 do if (phase_sb_sec(i) lt 0.0) $
  then phase_sb_sec(i) = phase_sb_sec(i) + 1.0

phase_mod = (tarr-Tperi)/period

set_plot,'ps'
device, /Color, Bits_per_pixel=8, filename='temp_sb_sec.eps',xsize=12,ysize=10,/encap

;toff = 50000.0d               ; RJD offset
;tlabel='JD - 2450000 (days)'
toff = 0.0d                     ; no offset
tlabel='Time'

; Define user symbol to be rectangle
xrect = frac*[-1.0,-1.0,1.0,1.0,-1.0]
yrect = frac*[-1.0,1.0,1.0,-1.0,-1.0]

ploterror,phase_sb_sec,V2_sb_sec,dV2_sb_sec,psym=2,yrange=[vmin,vmax],xrange =[-0.1,1.1], $
  xtitle = 'Phase', ytitle = 'RV (km/s)', xstyle=1,/NoData
oplot,phase_mod,V2mod
usersym, frac*cos(Z), frac*sin(Z),/fill
oplot,phase_sb_sec,V2_sb_sec,psym=8,color=3

device,/close
set_plot, 'x'

end
