pro fit_velocity, sb2_file, sb2_num, sb1_name, sb1_num

; Compute the spectroscopic orbital elements through a Newton-Raphson 
; technique.  Fits SB1 and SB2 simultaneously!
;
; The program requires an initial guess that can be calculated from using
; the points from the fitting curves.
;
; Calls the following routines:
;     calc_deriv_SB1.pro
;     calc_deriv_SB2.pro
;     calc_Ei.pro
;     solve_trans.pro
;
; Equations:  
; Initial estimates (P0,T0,e0,a1sini|0,a2sini|0,omega0,Vsys0) and  
; a set of observations (ti,V1i,V2i) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Velocity amplitudes (K1, K2):
;       K = (2pi/P)(asini/(1 - e^2)^1/2)
; The fitted velocities (V1_i, V2_i) are determined from the orbital elements:
;       V1 = K1*[e*cos(omega) + cos(nu + omega)] + Vsys
;       V2 = K2*[e*cos(omega) + cos(nu + omega)] + Vsys
;
; Procedure:
; Minimize chi squared between measured velocities and fitted velocities:
;	chi^2 = sum[(V1data - V1fit)^2/sigmaV1^2 
;		    + (V2data - V2fit)^2/sigmaV2^2]
; where the sums run over all data points
; Take partial derivatives of chi^2 with respect to the orbital elements,
; (P,T,e,a1sini,a2sini,omega,Vsys) and set to zero. Solve for the value of
; the orbital elements that minimize chi^2.
;
; Since the equations for V1 and V2 are non-linear, cannot solve for 
; the orbital elements that minimize chi^2 analytically.
;
; In turn, use a Newton-Raphson technique to converge upon the solution.
; Replace (V1fit,V2fit) in the chi^2 equation with the Taylor series 
; approximation,
; 	x = x| + dx/dP|(P-P0) + dx/dT|(T-T0) + dx/de|(e-e0) + dx/dw|(w-w0)
;	       + dx/d(a1sini)|(a1sini - a1sini|0) 
;              + dx/d(a2sini)|(a2sini - a2sini|0)
;              + dx/d(Vsys)|(Vsys - Vsys0)
; where x=(V1,V2) and the | represents the value evaluated for the
; initial estimates of the orbital elements. Now, V1fit and V2fit 
; are linear in (P,T,e,a1sini,a2sini,w,Vsys), so the partial derivatives
; can easily be taken and solved by setting up a matrix and using 
; Cramer's method.
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;	a1sini:	 projected semi-major axis of primary orbit (in km) (a1sini)
;	a2sini:	 projected semi-major axis of secondary orbit (in km) (a2sini)
;	omega:	 angle between node and periastron (w)
;	ti:	 time of observation
;	V1_d:	 velocity of primary - measured data
;	V2_d:    velocity of secondary - measured data
;	V1_f:	 V1 - fit (determined from orbital elements for time of obs.)
;	V2_f:    V2 - fit (determined from orbital elements for time of obs.)
;       Vsys:    systemic velocity
;
; INPUT:
;	data_file: file containing data points in the format:
;		      time  V1  errV1  V2  errV2
;		      where V1 is the radial velocity of the primary,
;                     V2 is the radial velocity of the secondary,
;                     and errV1 and errV2 are the measurement errors
;		      (enter file name using quotes)
;	num: number of data points contained in data_file
;
; Prompted inputs:
;	initial estimates for all the orbital elements: 
;             P, T, e, a1sini, a2sini, omega, Vsys
;       vector identifying which orbital elements to solve for
;             enter 0 to hold element fixed, 1 to vary)
;             ex.   1 1 1 1 1 1 1 -> solves for all
;                   0 0 0 1 0 0 0 -> solves only for a1 of primary
; OUTPUT:
;	best fit values of orbital elements: P,T,e,a1sini,a2sini,omega,Vsys
;       and corresponding errors
;
; Notes on units:
;    Velocities should be in km/s
;    P, T, and time should be in same units (days or years)
;    asini can be in any units (conversion to meters is prompted for)
;        -> program converts to km when computed fit-velocities    
;
; Began 23 June 2005
;
; Version 1: Orignal copy
; Version 2: 
;    - Include Marquadt Method of adjusting diagonal elements of
;      the alpha matrix to ensure convergence. 5 July 2005
;    - Fixed non-convergence problems   7 July 2005
;        + do not remove P wrap around effects from dV/dP
;          leave (t-T)/P in full form; net effect is that adjustments
;          smaller
;        + fixed primary/secondary indices in setting dV1/dVsys and dV2/dVsys
;    - Designed for SB2's only!!!
; Version 3: 7 July 2005
;    - Add SB1 + SB2 joint solutions

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time = dblarr(num)
V1 = dblarr(num)
V2 = dblarr(num)
dV1 = dblarr(num)
dV2 = dblarr(num)

openr,lun,data_file,/get_lun

for i=0, num-1 do begin

	readf, lun, temp1, temp2, temp3, temp4, temp5

	time(i) = temp1
	V1(i) = temp2
	dV1(i) = temp3
	V2(i) = temp4	
	dV2(i) = temp5
	
endfor

close,lun

; Obtain values for P,T,e,a1sini,a2sini,omega,Vsys

period = 0d
Tperi = 0d
ecc = 0d
a1sini = 0d
a2sini = 0d
omega = 0d
Vsys = 0d

print,"Enter P,T,e,a1sini,a2sini,omega,Vsys:"
read,period,Tperi,ecc,a1sini,a2sini,omega,Vsys

print, "Vary each orbital element?"
print, "For each element, enter 0 to hold element fixed, 1 to vary:"
print, "[P,T,e,a1sini,a2sini,omega,Vsys]"
read, f0,f1,f2,f3,f4,f5,f6

elfix=[f0,f1,f2,f3,f4,f5,f6]
nEl = n_elements(elfix)

print,"Initial P,T,e,a1sini,a2sini,omega,Vsys:"
print,period,Tperi,ecc,a1sini,a2sini,omega,Vsys

;convert omega to radians
omega = omega*!dpi/180

ansyd=' '
print,'Is period in years or days (y/d)?'
read,ansyd

if (ansyd eq 'y') then y_d = 365.25
if (ansyd eq 'd') then y_d = 1.0

; conversion factor for y/d to seconds
; (need to convert period to sections when computing velocities)
sec = 1.0*y_d*24.0*60.0*60.0

meters = 0.0
print,'How many meters are in semi-major axis units?'
print,'(1 Gm = 1d9 m)'
read,meters

km = meters/1d3

; plot model and data of initial guess

tnum = 1000.0
V1mod = dblarr(tnum)
V2mod = dblarr(tnum)
tarr = dblarr(tnum)
tmin = min(time)
tmax = max(time)
tstep = (tmax-tmin)/tnum
ti = tmin

; Determine the velocity amplitudes
K1 = (2.0*!dpi/(period*sec))*(a1sini*km/sqrt(1 - ecc^2))
K2 = (2.0*!dpi/(period*sec))*(a2sini*km/sqrt(1 - ecc^2))
print,'K1',K1
print,'K2',K2

for i=0, tnum-1 do begin

    tarr(i) = ti

    ; Determine eccentric anomalies

    calc_Ei, ti, period, Tperi, ecc, Ei

    ; Determine true anomalies:

    nu = 2*atan(sqrt((1+ecc)/(1-ecc))*tan(Ei/2.0))

    ; Determin (V1_f, V2_f) velocities from orbital elements and times of 
    ; observations

    V1mod(i) = K1*(ecc*cos(omega) + cos(nu + omega)) + Vsys

    V2mod(i) = -K2*(ecc*cos(omega) + cos(nu + omega)) + Vsys

    ti = ti + tstep

endfor

vmax = max(V2mod)
vmin = min(V2mod)

plot,time,V1,psym=2,yrange=[vmin,vmax]
oplot,time,V2,psym=6
oplot,tarr,V1mod
oplot,tarr,V2mod

stop,'Type .cont to continue'

vmax = max([max(V1),max(V2)])
vmin = min([min(V1),min(V2)])

k=0
for i=0, nEl-1 do begin

	if (elfix(i) eq 0) then k=k+1
endfor

mfit = nEl - k 		; number of elements to improve

; determine indices of elements for which to vary

elvar = where(elfix ne 0)
    
EL = [period,Tperi,ecc,a1sini,a2sini,omega,Vsys]

ELadj = EL

elLabel = ['P','T','e','a1sini','a2sini','omega','Vsys']

count = 0

delta_chi = 1.0		; set initially to begin loop

lambda = 0.001   ; Marquardt method to ensure convergence

while (delta_chi gt 0.001) do begin

    ; Determine errors in orbital elements
    ; set up matrices for error determination
    ; Invert matrix through Gauss Jordan elimination (Numerical Recipes in C)

    ; set up covariance matrix and column matrix
    ; alpha = dblarr(7,7)
    ; beta = dblarr(7)

    calc_deriv_SB, EL, elfix, mfit, time, sec, km, V1, V2, dV1, dV2, V1_f, V2_f, alpha, beta

    ; Determine chi squared

    chi2old = total((V1 - V1_f)^2/dV1^2 + (V2 - V2_f)^2/dV2^2)

    invmat = invert(alpha, stat, /double)

    print,"stat (0 successful):",stat

    ;determine errors:

    ELerr = dblarr(mfit)

    for i=0, mfit-1 do ELerr(i) = sqrt(invmat(i,i))

    ; adjust alpha matrix by Marquardt parameter lambda

    for i=0, mfit-1 do alpha(i,i) = alpha(i,i)*(1.0 + lambda)

    if (mfit eq 1) then begin
        delta_el = dblarr(1)
        delta_el(0) = beta/alpha
    endif else delta_el = cramer(alpha,beta)   ; adjustments

;    print,'alpha',alpha
;    print,'beta',beta

    ; adjust orbital parameters
    
    for i=0, mfit-1 do ELadj(elvar(i)) = EL(elvar(i)) + delta_el(i)

    k=0
    print,"Orbital Element     Adjustment"
    for i=0, nEL - 1 do begin
        if (elfix(i) eq 0) then begin
            if (i eq 5) then print, elLabel(i), ': ', El(i)*180/!dpi $
            else print, elLabel(i), ': ', El(i)
        endif else begin
            if (i eq 5) then $
              print, elLabel(i), ': ', El(i)*180/!dpi, delta_el(k)*180/!dpi $
            else print, elLabel(i), ': ', El(i), delta_el(k)
            k=k+1
        endelse
    endfor

    print,"Chi2 of current solution:",chi2old

    plot,time,V1,psym=2,yrange=[vmin,vmax]
    oplot,time,V1_f,psym=7
    for l=0, num-1 do oplot,[time(l),time(l)],[V1(l),V1_f(l)]
    oplot,time,V2,psym=6
    oplot,time,V2_f,psym=4
    for l=0, num-1 do oplot,[time(l),time(l)],[V2(l),V2_f(l)]

    stop,'Type .cont to continue'

    calc_deriv_SB, ELadj, elfix, mfit, time, sec, km, V1, V2, dV1, dV2, V1_f, V2_f, alpha, beta

    ; Determine chi squared

    chi2new = total((V1 - V1_f)^2/dV1^2 + (V2 - V2_f)^2/dV2^2)

    print,"chi2 of next modification:",chi2new

    if (chi2new le chi2old) then begin
        EL = ELadj
	chi2 = chi2new
        delta_chi = chi2old - chi2new
        lambda = lambda/10.0
        print, 'lambda', lambda
    endif else begin
	print, "next adjustment does not improve chi2"
        print, "increasing lambda by 10.0"
        lambda = lambda*10.0
        print, 'lambda', lambda
        delta_chi = 1.0
    endelse

    count = count+1

    ; do not exit unless lambda is less than 0.001

    if (lambda gt 0.001) and (delta_chi lt 0.001) then delta_chi = 1.0
;    print,'lambda',lambda
;    print,'delta_chi',delta_chi

    ;stop,'Type .cont to continue'

endwhile

lambda = 0.0

print, "Final values:"

calc_deriv_SB, EL, elfix, mfit, time, sec, km, V1, V2, dV1, dV2, V1_f, V2_f, alpha, beta

invmat = invert(alpha, stat, /double)

;determine errors:

chi2 = total((V1 - V1_f)^2/dV1^2 + (V2 - V2_f)^2/dV2^2)

; degrees of freedom
dof = 2d*num - mfit

; reduced chi squared:
chi2red = chi2/dof

ELerr = dblarr(mfit)
for i=0,mfit-1 do ELerr(i) = sqrt(chi2red)*sqrt(invmat(i,i))

k=0
print,"Orbital Element     Error     "
for i=0, nEL - 1 do begin
    if (elfix(i) eq 0) then begin
        if (i eq 5) then print, elLabel(i), ': ', El(i)*180/!dpi $
        else print, elLabel(i), ': ', El(i)
    endif else begin
        if (i eq 5) then $
          print, elLabel(i), ': ', El(i)*180/!dpi, Elerr(k)*180/!dpi $
        else print, elLabel(i), ': ', El(i), Elerr(k)
        k=k+1
    endelse
endfor
print, "Final chi2:", chi2
print, "Reduced chi2:", chi2red
print,'Number of iterations:',count

free_lun, lun

end

;
;
;

pro calc_deriv_SB, ELarr, elfix, mfit, time, sec, km, V1_d, V2_d, err_V1, err_V2, V1_f, V2_f, chimat, colmat

; Notes for future reference:
; Equivalent to the function of calc_err for the VB case
; For doing a joint fit - treat a1, a2, i separately 
;                       - rename to calc_deriv_SBj  
; -endnotes-

; Set up matrices for error determination of orbital elements
; (derivatives)
;
; Add option of fixing values of orbital elements
;
; INPUT:
;	ELarr: array of orbital elements [P,T,e,a1sini,a2sini,omega]
;       elfix: 7-element array: 0 if value to be held fixed
;                               1 if value to be free
;       mfit: number of parameters for which to solve (number of 1's in elfix)
;	Ei: eccentric anomalies
;	time: time of observation
;	V1_d: array containing velocity measurements for primary (in km/s) 
;	V2_d: array containing velocity measurements for secondary (in km/s)
;	err_V1: error in V1_d
;	err_V2: error in V2_d
;
; OUTPUT:
;       V1_f: fit-value of primary velocity
;       V2_f: fit- value of secondary velocity
;	chimat: 6x6 covariance matrix
;	colmat: column matrix containing differences between data and fit
;
; Calls the following routines:
;     calc_Ei.pro
;
; Equations:  
; Initial estimates (P0,T0,e0,a1sini|0,a2sini|0,omega0) and  
; a set of observations (ti,V1i,V2i) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Velocity amplitudes (K1, K2):
;       K = (2pi/P)(asini/(1 - e^2)^1/2)
; The fitted velocities (V1_i, V2_i) are determined from the orbital elements:
;       V1 = K1*[e*cos(omega) + cos(nu + omega)] 
;       V2 = K2*[e*cos(omega) + cos(nu + omega)] 
;
; Procedure:
; Minimize chi squared between measured velocities and fitted velocities:
;	chi^2 = sum[(V1data - V1fit)^2/sigmaV1^2 
;		    + (V2data - V2fit)^2/sigmaV2^2]
; where the sums run over all data points
; Take partial derivatives of chi^2 with respect to the orbital elements,
; (P,T,e,a1sini,a2sini,omega) and set to zero. Solve for the value of
; the orbital elements that minimize chi^2.
;
; Since the equations for V1 and V2 are non-linear, cannot solve for 
; the orbital elements that minimize chi^2 analytically.
;
; In turn, use a Newton-Raphson technique to converge upon the solution.
; Replace (V1fit,V2fit) in the chi^2 equation with the Taylor series 
; approximation,
; 	x = x| + dx/dP|(P-P0) + dx/dT|(T-T0) + dx/de|(e-e0) + dx/dw|(w-w0)
;	       + dx/d(a1sini)|(a1sini - a1sini|0) 
;              + dx/d(a2sini)|(a2sini - a2sini|0)
; where x=(V1,V2) and the | represents the value evaluated for the
; initial estimates of the orbital elements. Now, V1fit and V2fit 
; are linear in (P,T,e,a1sini,a2sini,w), so the partial derivatives
; can easily be taken and solved by setting up a matrix and using 
; Cramer's method.
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;	a1sini:	 projected semi-major axis of primary orbit (in km) (a1sini)
;	a2sini:	 projected semi-major axis of secondary orbit (in km) (a2sini)
;	inc:	 inclination (i)
;	omega:	 angle between node and periastron (w)
;	ti:	 time of observation
;	V1_d:	 velocity of primary - measured data
;	V2_d:    velocity of secondary - measured data
;	V1_f:	 V1 - fit (determined from orbital elements for time of obs.)
;	V2_f:    V2 - fit (determined from orbital elements for time of obs.)
;       Vsys:    systemic velocity
;
; 23 June 2005: Begin writing program

; Define orbital element parameters

period = ELarr(0)
Tperi = ELarr(1)
ecc = ELarr(2)
a1sini = ELarr(3)
a2sini = ELarr(4)
omega = ELarr(5)
Vsys = ELarr(6)

EL = dblarr(mfit)
i=0
for k=0, mfit-1 do begin
	if (elfix(k) ne 0) then EL(k) = ELarr(i)
	i = i+1
endfor

num = n_elements(time)

; Determine the eccentric anomalies Ei: 
; (Ei: array to the eccentric anomaly)

calc_Ei, time, period, Tperi, ecc, Ei

; Determine true anomalies:

nu = 2*atan(sqrt((1+ecc)/(1-ecc))*tan(Ei/2.0))

;print,'Ei',Ei
;print,'nu',nu

; check sign of true anomaly

cosnu = cos(nu)
sinnu = sin(nu)

actualcosnu = (cos(Ei) - ecc)/(1 - ecc*cos(Ei))
actualsinnu = sqrt(1 -ecc^2)*sin(Ei)/(1-ecc*cos(Ei))

;print,'sinnnu actualsinnu cosnu actualcosnu'

for i=0, num-1 do begin
;    print,nu(i)*180.0/!dpi
;    print, sinnu(i), actualsinnu(i), cosnu(i), actualcosnu(i)
    if (cosnu(i)/abs(cosnu(i)) ne actualcosnu(i)/abs(actualcosnu(i))) and $
       (sinnu(i)/abs(sinnu(i)) ne actualsinnu(i)/abs(actualsinnu(i))) then $
       nu(i) = nu(i) + !dpi
;    print,sin(nu(i)), actualsinnu(i), cos(nu(i)),actualcosnu(i)
;    Strange!  It seems that nu is ALWAYS in the proper quadrant ???
endfor

; Determine the velocity amplitudes

K1 = (2.0*!dpi/(period*sec))*(a1sini*km/sqrt(1 - ecc^2))

K2 = (2.0*!dpi/(period*sec))*(a2sini*km/sqrt(1 - ecc^2))

;print,'K1',K1
;print,'K2',K2

; Determin (V1_f, V2_f) velocities from orbital elements and times of 
; observations

V1_f = K1*(ecc*cos(omega) + cos(nu + omega))

V2_f = -K2*(ecc*cos(omega) + cos(nu + omega))

;print,'V1_f',V1_f
;print,'V2_f',V2_f

; Calculate derivatives evaluated at initial conditions 
;	dx/dEL = (dx/dP,dx/dT,dx/de,dx/d(a1sini),dx/d(a2sini),dx/dw)
;	dy/dEL = (dy/dP,dy/dT,dy/de,dy/d(a1sini),dy/d(a2sini),dy/dw)
; where (x,y) -> (V1,V2) and
; dEL = (P,T,e,a1sini,a2sini,omega)

;initialize derivative arrays 

dV1_del = dblarr(mfit,num)	;6: number of orbital elements
dV2_del = dblarr(mfit,num)	;num: number of data points

chimat = dblarr(mfit,mfit) 	;matrix for minimizing chi^2
colmat = dblarr(mfit)		;column matrix for minimizing chi^2
delta_el = dblarr(mfit)		;solution to matrix equation

timediff=dblarr(num)

for i=0, num-1 do begin

	; reduce time difference to same epoch
	; (fraction of period covered at time t)

	fracP = (time(i) - Tperi)/period - fix((time(i) - Tperi)/period)

	;(remove effects of positive and negative time -> arbitrary zero-point
	; for Tperi...... fracP goes from 0.0 to 1.0)

;	if (fracP lt 0) then fracP = 1.0d + fracP	

	timediff(i) = fracP

endfor

;print,'dEi_dP = -2*!dpi*(time - Tperi)/period^2/(1.0 - ecc*cos(Ei))'
;print,-2*!dpi*(time - Tperi)/period^2/(1.0 - ecc*cos(Ei))
;print,'dEi_dP = -2*!dpi*timediff/period/(1.0 - ecc*cos(Ei))'
;print,-2*!dpi*timediff/period/(1.0 - ecc*cos(Ei))
;print,'dEi_dP = -(Ei - ecc*sin(Ei))/(1.0 - ecc*cos(Ei))/period'
;print,-(Ei - ecc*sin(Ei))/(1.0 - ecc*cos(Ei))/period
;print, ' '


;gamma = 2*!dpi/period*sin(nu + omega)*sqrt(1.0 - ecc^2)/(1.0 - ecc*cos(Ei))^2
;
;print,'gamma',gamma
;print,'gamma*K1',gamma*K1
;print,'gamma*K2',gamma*K2

;dEi_dP = -2*!dpi*timediff/period/(1.0 - ecc*cos(Ei))
dEi_dP = -2*!dpi*(time - Tperi)/period^2/(1.0 - ecc*cos(Ei))
;dEi_dP = -(Ei - ecc*sin(Ei))/(1.0 - ecc*cos(Ei))/period
dEi_dT = -2*!dpi/period/(1.0 - ecc*cos(Ei))
dEi_de = sin(Ei)/(1.0 - ecc*cos(Ei))

dnu_dP = sqrt(1.0 - ecc^2)/(1.0 - ecc*cos(Ei))*dEi_dP
dnu_dT = sqrt(1.0 - ecc^2)/(1.0 - ecc*cos(Ei))*dEi_dT
dnu_de =((1.0 - ecc^2)*dEi_de + sin(Ei))/sqrt(1.0 - ecc^2)/(1.0 - ecc*cos(Ei))

dV1_dP = -V1_f/period - K1*sin(nu + omega)*dnu_dP
dV1_dT = -K1*sin(nu + omega)*dnu_dT
dV1_de =  V1_f*ecc/(1.0 - ecc^2) + K1*(cos(omega) - sin(nu + omega)*dnu_de)
dV1_dw = -K1*(ecc*sin(omega) + sin(nu + omega))
dV1_da1sini = V1_f/a1sini
dV1_da2sini = dblarr(num)
dV1_dVsys = dblarr(num) + 1.0

dV2_dP = -V2_f/period + K2*sin(nu + omega)*dnu_dP
dV2_dT =  K2*sin(nu + omega)*dnu_dT
dV2_de =  V2_f*ecc/(1.0 - ecc^2) - K2*(cos(omega) - sin(nu + omega)*dnu_de)
dV2_dw =  K2*(ecc*sin(omega) + sin(nu + omega))
dV2_da2sini = V2_f/a2sini
dV2_da1sini = dblarr(num)
dV2_dVsys = dblarr(num) + 1.0

;dV1_dEL = (dx_dP,dx_dT,dx_de,dx_d(a1sini),dx_d(a2sini),dx_dw,dx_dVsys)
;dV2_dEL = (dy_dP,dy_dT,dy_de,dy_d(a1sini),dy_d(a2sini),dy_dw,dy_dVsys)

k=0
if (elfix(0) ne 0) then begin
	dV1_del(k,*) = dV1_dP
	k=k+1
endif
if (elfix(1) ne 0) then begin
	dV1_del(k,*) = dV1_dT
	k=k+1
endif
if (elfix(2) ne 0) then begin
	dV1_del(k,*) = dV1_de
	k=k+1
endif
if (elfix(3) ne 0) then begin
	dV1_del(k,*) = dV1_da1sini
	k=k+1
endif
if (elfix(4) ne 0) then begin
	dV1_del(k,*) = dV1_da2sini
	k=k+1
endif
if (elfix(5) ne 0) then begin
	dV1_del(k,*) = dV1_dw
	k=k+1
endif
if (elfix(6) ne 0) then begin
	dV1_del(k,*) = dV1_dVsys
	k=k+1
endif

k=0
if (elfix(0) ne 0) then begin
	dV2_del(k,*) = dV2_dP
	k=k+1
endif
if (elfix(1) ne 0) then begin
	dV2_del(k,*) = dV2_dT
	k=k+1
endif
if (elfix(2) ne 0) then begin
	dV2_del(k,*) = dV2_de
 	k=k+1
endif
if (elfix(3) ne 0) then begin
	dV2_del(k,*) = dV2_da1sini
	k=k+1
endif
if (elfix(4) ne 0) then begin
	dV2_del(k,*) = dV2_da2sini
	k=k+1
endif
if (elfix(5) ne 0) then begin
	dV2_del(k,*) = dV2_dw
	k=k+1
endif
if (elfix(6) ne 0) then begin
	dV2_del(k,*) = dV2_dVsys
	k=k+1
endif

; Set up matrix for minimizing chi squared
; Set up column matrix too
; (col) = (chimat)(delta_el)

diff_V1 = dblarr(num)	  ; array for [V1(data) - V1(fit)]
diff_V2 = dblarr(num)     ; array for [V2(data) - V2(fit)]

; correct velocities for Vsys
V1_f = V1_f + Vsys
V2_f = V2_f + Vsys

;print,'Vsys',Vsys
;print,'Vsys included:'
;print,'V1_f',V1_f
;print,'V2_f',V2_f
;print,'-V1_f/P',-V1_f/period
;print,'-V2_f/P',-V2_f/period
;
;print,'V1_d',V1_d
;print,'V2_d',V2_d
;
;print,'err_V1',err_V1
;print,'err_V2',err_V2

for k=0, num-1 do begin

	; weight derivative for each data point by corresponding 
	; measurement error

	diff_V1(k) = (V1_d(k)-V1_f(k))
	diff_V2(k) = (V2_d(k)-V2_f(k))

	; the ## operator performs typical matrix multiplication
	; the * multiplies individual elements (no summing)

	for i=0, mfit-1 do begin

		for j=0, mfit-1 do begin

			chimat(i,j) = chimat(i,j) $
				+ dV1_del(i,k)*dV1_del(j,k)/err_V1(k)^2 $
				+ dV2_del(i,k)*dV2_del(j,k)/err_V2(k)^2
		endfor

		colmat(i) = colmat(i) + diff_V1(k)*dV1_del(i,k)/err_V1(k)^2 $
			    + diff_V2(k)*dV2_del(i,k)/err_V2(k)^2
	endfor		
endfor

;print,'diff_V1',diff_V1
;print,'diff_V2',diff_V2
;
;print,'dV1_del',dV1_del
;print,'dV2_del',dV2_del
;
;print,'colmat',colmat
;print,'chimat',chimat

;stop,'Type .cont to continue'

end


;
;
;

pro calc_Ei, time, period, Tperi, ecc, Ei

; Determine the eccentric anomalies Ei: 
;     mu(ti - T) = Ei - esin(Ei)  ... Kepler's Equation
;     where mu = 360/P = 2*pi/P
; Solve this transcendental equation through an 
; iterative procedure.
; Use SOLVE_TRANS.PRO

; array to hold Ei's (the eccentric anomaly)

num = n_elements(time)

Ei = dblarr(num) 

mu = 2*!dpi/period

for i=0, num-1 do begin

	; Mi: mean anomoly - increases uniformly with time
	; zero at time Tperi, 2Pi each orbit

	Mi = mu*(time(i) - Tperi) 

	; reduce to same epoch

	Mi = 2*!dpi*((time(i) - Tperi)/period $	
		- fix((time(i) - Tperi)/period))

	; keep Mi between 0 and 2Pi

	if (Mi lt 0.0) then Mi = Mi + 2*!dpi
	if (Mi gt 2*!dpi) then Mi = Mi - 2*!dpi

	solve_trans,ecc,Mi,Eit

	; keep Ei between 0 and 2Pi

	if (Eit lt 0.0) then Eit = Eit + 2*!dpi
	if (Eit gt 2*!dpi) then Eit = Eit - 2*!dpi

	Ei(i) = Eit

endfor

end

;
;
;

pro solve_trans, e, M, EE

; Solve transcendental equation of the form E - esinE = M.
; Use iterative procedure to determine E.
; Initial approximation: E_0 = M + esinM - e^2/2 sin(2M)
; Improve solution by iterating the following to formulae:
;	M_0 = E_0 - esin(E_0)
;	E_1 = E_0 + (M - M_0)/(1 - ecos(E_0))
;	(derivative of Kepler's equation)
;
; Method adapted from Heintz 1978 (p.34-35)
; Results compared with point-&-click graphical method.  Iterative approach 
; leads to exact solution that satisfies E - esinE = M.  Therefore, 
; point-&-click method is subsequently removed from orbit fitting.
;
; INPUT:
;	e: eccentricity
;	M: mean anomaly   M= 2*Pi/P
;
; OUTPUT:
;	EE: eccentric anomaly
;
; Created: 9 May 2002

; Initial approximation:

EE = M + e*sin(M) + e^2/2*sin(2*M)

;print, "Initial approximation for E:", EE

EEi = 0d	; parameter to hold initial value once enter while loop

count = 0

while (abs(EE - EEi) gt 0.000001) do begin

	EEi = EE

	Mi = EEi - e*sin(EEi)

	EE = EEi + (M - Mi)/(1 - e*cos(EEi))

	count=count+1

endwhile

end
