pro fit_velocity_sb2_multifile, sb_pri_file, sb_sec_file

; Compute the spectroscopic orbital elements through a Newton-Raphson 
; technique.  Works for SB2!!!
;
; The program requires an initial guess that can be calculated from using
; the points from the fitting curves.
;
; Calls the following routines:
;     calc_deriv_SB2.pro
;     calc_Ei.pro
;     solve_trans.pro
;     calc_sb2fit.pro - for plotting model orbits
;
; Equations:  
; Initial estimates (P0,T0,e0,K1_0,K2_0,omega0,Vsys0) and  
; a set of observations (ti,V1i,V2i) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Projected semi-major axes (a1sini, a2sini):
;       a1sini = (P/2pi)*K1*(1 - e^2)^1/2
;       a2sini = (P/2pi)*K2*(1 - e^2)^1/2
; The fitted velocities (V1_i, V2_i) are determined from the orbital elements:
;       V1 =  K1*[e*cos(omega) + cos(nu + omega)] + Vsys 
;       V2 = -K2*[e*cos(omega) + cos(nu + omega)] + Vsys
;
; Procedure:
; Minimize chi squared between measured velocities and fitted velocities:
;	chi^2 = sum[(V1data - V1fit)^2/sigmaV1^2 
;		    + (V2data - V2fit)^2/sigmaV2^2]
; where the sums run over all data points
; Take partial derivatives of chi^2 with respect to the orbital elements,
; (P,T,e,K1,K2,omega,Vsys) and set to zero. Solve for the value of
; the orbital elements that minimize chi^2.
;
; Since the equations for V1 and V2 are non-linear, cannot solve for 
; the orbital elements that minimize chi^2 analytically.
;
; In turn, use a Newton-Raphson technique to converge upon the solution.
; Replace (V1fit,V2fit) in the chi^2 equation with the Taylor series 
; approximation,
; 	x = x| + dx/dP|(P-P0) + dx/dT|(T-T0) + dx/de|(e-e0) + dx/dw|(w-w0)
;	       + dx/d(K1)|(K1 - K1|0) + dx/d(K2)|(K2 - K2|0)
;              + dx/d(Vsys)|(Vsys - Vsys0)
; where x=(V1,V2) and the | represents the value evaluated for the
; initial estimates of the orbital elements. Now, V1fit and V2fit 
; are linear in (P,T,e,K1,K2,w,Vsys), so the partial derivatives
; can easily be taken and solved by setting up a matrix and using 
; Cramer's method.
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;       K1:      velocity amplitude of primary (in km/s) (K1)
;       K2:      velocity amplitude of secondary (in km/s) (K2)
;	omega:	 angle between node and periastron (w)
;	ti:	 time of observation
;	V1_d:	 velocity of primary - measured data
;	V2_d:    velocity of secondary - measured data
;	V1_f:	 V1 - fit (determined from orbital elements for time of obs.)
;	V2_f:    V2 - fit (determined from orbital elements for time of obs.)
;       Vsys:    systemic velocity
;
; INPUT:
;	data_file: file containing data points in the format:
;		      time  V1  errV1  V2  errV2
;		      where V1 is the radial velocity of the primary,
;                     V2 is the radial velocity of the secondary,
;                     and errV1 and errV2 are the measurement errors
;		      (enter file name using quotes)
;	num: number of data points contained in data_file
;
; Prompted inputs:
;	initial estimates for all the orbital elements: 
;             P, T, e, K1, K2, omega, Vsys
;       vector identifying which orbital elements to solve for
;             enter 0 to hold element fixed, 1 to vary)
;             ex.   1 1 1 1 1 1 1 -> solves for all
;                   0 0 0 1 0 0 0 -> solves only for a1 of primary
; Notes: make sure P,T are given in same units
;        make sure K1,K2,V1,V2 are given in same units

; OUTPUT:
;	best fit values of orbital elements: P,T,e,a1sini,a2sini,omega,Vsys
;       and corresponding errors
;
; Notes on units:
;    P, T, and time should be in same units (days or years)
;    K1, K2, V1, and V2 should be in same units (km/s)
;
; Began 23 June 2005
;
; Version 1: Orignal copy
; Version 2: 
;    - Include Marquadt Method of adjusting diagonal elements of
;      the alpha matrix to ensure convergence. 5 July 2005
;    - Fixed non-convergence problems   7 July 2005
;        + do not remove P wrap around effects from dV/dP
;          leave (t-T)/P in full form; net effect is that adjustments
;          smaller
;        + fixed primary/secondary indices in setting dV1/dVsys and dV2/dVsys
;    - Designed for SB2's only!!!
;
; 31 May 2006: Save unmodified program as version 2
; 31 May 2006: Fit SB2 parameters in terms of K1 and K2
;              (as opposed to a1sini and a2sini)


close,1
close,2

!P.font=0
!P.multi=0
!P.charsize=1.2
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.2	;scale factor for adjusting size of plotting symbols

; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

sb_pri_num = 10000
sb_sec_num = 10000

time_sb_pri = dblarr(sb_pri_num)
V1_sb_pri = dblarr(sb_pri_num)
dV1_sb_pri = dblarr(sb_pri_num)

time_sb_sec = dblarr(sb_sec_num)
V2_sb_sec = dblarr(sb_sec_num)
dV2_sb_sec = dblarr(sb_sec_num)


; Read SB measurements for primary

openr,1,sb_pri_file
i = 0
while not EOF(1) do begin

   readf,1, temp1, temp2, temp3
   
   time_sb_pri(i) = temp1
   V1_sb_pri(i) = temp2
   dV1_sb_pri(i) = temp3

   i = i+1
        
endwhile
close,1
sb_pri_num = i
time_sb_pri = time_sb_pri(0:sb_pri_num-1)
V1_sb_pri = V1_sb_pri(0:sb_pri_num-1)
dV1_sb_pri = dV1_sb_pri(0:sb_pri_num-1)

; Read SB measurements for secondary

openr,1,sb_sec_file
i = 0
while not EOF(1) do begin

	readf,1, temp1, temp2, temp3

	time_sb_sec(i) = temp1
	V2_sb_sec(i) = temp2
	dV2_sb_sec(i) = temp3

        i = i + 1
        
endwhile
close,1
sb_sec_num = i
time_sb_sec = time_sb_sec(0:sb_sec_num-1)
V2_sb_sec = V2_sb_sec(0:sb_sec_num-1)
dV2_sb_sec = dV2_sb_sec(0:sb_sec_num-1)


; Obtain values for P,T,e,K1,K2,omega,Vsys

period = 0d
Tperi = 0d
ecc = 0d
K1 = 0d
K2 = 0d
omega = 0d
Vsys1 = 0d
Vsys2 = 0d

print,"Enter P,T,e,K1,K2,omega,Vsys1,Vsys2:"
read,period,Tperi,ecc,K1,K2,omega,Vsys1,Vsys2

print, "Vary each orbital element?"
print, "For each element, enter 0 to hold element fixed, 1 to vary:"
print, "[P,T,e,K1,K2,omega,Vsys1,Vsys2]"
read, f0,f1,f2,f3,f4,f5,f6,f7

;; Order of orbital element arrays
;; do not set here yet (need to get angles into radians)
;; EL_sb1 = [period,Tperi,ecc,K1,omega,Vsys]
;;              0    1     2  3  5     6
;; EL_sb2 = [period,Tperi,ecc,K2,omega,Vsys]
;;              0    1     2  4  5     7

elfix=[f0,f1,f2,f3,f4,f5,f6,f7]
nEl = n_elements(elfix)

elfix_sb_pri=[f0,f1,f2,f3,f5,f6]
nEl_sb_pri = n_elements(elfix_sb_pri)

elfix_sb_sec=[f0,f1,f2,f4,f5,f7]
nEl_sb_sec = n_elements(elfix_sb_sec)

print,"Initial P,T,e,K1,K2,omega,Vsys1,Vsys2:"
print,period,Tperi,ecc,K1,K2,omega,Vsys1,Vsys2


;convert omega to radians
omega = omega*!dpi/180


k=0
for i=0, nEl-1 do if (elfix(i) eq 0) then k=k+1
mfit = nEl - k 		; number of elements to improve

k=0
for i=0, nEl_sb_pri-1 do if (elfix_sb_pri(i) eq 0) then k=k+1
mfit_sb_pri = nEl_sb_pri - k 		; number of elements to improve

k=0
for i=0, nEl_sb_sec-1 do if (elfix_sb_sec(i) eq 0) then k=k+1
mfit_sb_sec = nEl_sb_sec - k 		; number of elements to improve


EL = [period,Tperi,ecc,K1,K2,omega,Vsys1,Vsys2]
EL_sb_pri = [period,Tperi,ecc,K1,omega,Vsys1]
EL_sb_sec = [period,Tperi,ecc,K2,omega,Vsys2]

ELadj = EL

elLabel = ['P','T','e','K1','K2','omega','Vsys1','Vsys2']

; determine which indices of full set are being varied

sb_pri_subset = dblarr(mfit_sb_pri)
sb_sec_subset = dblarr(mfit_sb_sec)

; determine subarray of indices that are VB, SB1, and SB2 parameters

sb_pri_par = dblarr(nEl_sb_pri)
sb_sec_par = dblarr(nEl_sb_sec)

; determine indices of elements for which to vary
el_subset = where(elfix ne 0)
Elvar = El(el_subset)

k=0
for i=0, nEl_sb_pri-1 do begin
    ind = where(EL_sb_pri(i) eq EL)
    sb_pri_par(i) = ind
    if (elfix_sb_pri(i) ne 0) then begin
         ind = where(EL_sb_pri(i) eq Elvar)
         sb_pri_subset(k) = ind
        k=k+1
    endif
endfor

k=0
for i=0, nEl_sb_sec-1 do begin
    ind = where(EL_sb_sec(i) eq EL)
    sb_sec_par(i) = ind
    if (elfix_sb_sec(i) ne 0) then begin
         ind = where(EL_sb_sec(i) eq Elvar)
         sb_sec_subset(k) = ind
        k=k+1
    endif
endfor


; plot model and data of initial guess for SB2 data


tnum = 1000.0
tmin = min(time_sb_pri)
tmax = max(time_sb_pri)
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb1fit, EL_sb_pri, tarr, V1mod

vmax = max(V1mod)
vmin = min(V1mod)

print,'SB1 velocity curve'
plot,time_sb_pri,V1_sb_pri,psym=2,yrange=[vmin,vmax]
oplot,tarr,V1mod

vmax = max(V1_sb_pri)
vmin = min(V1_sb_pri)


tnum = 1000.0
tmin = min(time_sb_sec)
tmax = max(time_sb_sec)
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb1fit, EL_sb_sec, tarr, V2mod

vmax = max(V2mod)
vmin = min(V2mod)

print,'SB1 velocity curve'
plot,time_sb_sec,V2_sb_sec,psym=2,yrange=[vmin,vmax]
oplot,tarr,V1mod

vmax = max(V2_sb_sec)
vmin = min(V2_sb_sec)

ans_cont = ''
print,'Hit enter to continue'
read,ans_cont

vmax = max([max(V1_sb_pri),max(V2_sb_sec)])
vmin = min([min(V1_sb_pri),min(V2_sb_sec)])

count = 0

delta_chi = 1.0		; set initially to begin loop

lambda = 0.001   ; Marquardt method to ensure convergence

while (delta_chi gt 0.001) do begin

    ; Determine errors in orbital elements
    ; set up matrices for error determination
    ; Invert matrix through Gauss Jordan elimination (Numerical Recipes in C)

    ; set up covariance matrix and column matrix
    ; alpha = dblarr(7,7)
    ; beta = dblarr(7)

    calc_deriv_sb1, EL_sb_pri, elfix_sb_pri, mfit_sb_pri, time_sb_pri, V1_sb_pri, dV1_sb_pri, V1_f_sb_pri, alpha_sb_pri, beta_sb_pri

    calc_deriv_sb1, EL_sb_sec, elfix_sb_sec, mfit_sb_sec, time_sb_sec, V2_sb_sec, dV2_sb_sec, V2_f_sb_sec, alpha_sb_sec, beta_sb_sec, /flag_wa

    ; Determine chi squared

    chi2old = total((V1_sb_pri - V1_f_sb_pri)^2/dV1_sb_pri^2) $
            + total((V2_sb_sec - V2_f_sb_sec)^2/dV2_sb_sec^2)

    ; combine SB1 and VB matrices...


    ; initialize full alpha,beta arrays
    alpha = dblarr(mfit,mfit)
    beta = dblarr(mfit)

    beta(sb_pri_subset) = beta(sb_pri_subset) + beta_sb_pri
    beta(sb_sec_subset) = beta(sb_sec_subset) + beta_sb_sec

    for i=0, mfit_sb_pri-1 do begin
        for j=0, mfit_sb_pri-1 do begin
            alpha(sb_pri_subset(i),sb_pri_subset(j)) = $
              alpha(sb_pri_subset(i),sb_pri_subset(j)) + alpha_sb_pri(i,j)
        endfor
    endfor

    for i=0, mfit_sb_sec-1 do begin
        for j=0, mfit_sb_sec-1 do begin
            alpha(sb_sec_subset(i),sb_sec_subset(j)) = $
              alpha(sb_sec_subset(i),sb_sec_subset(j)) + alpha_sb_sec(i,j)
        endfor
    endfor

    invmat = invert(alpha, stat, /double)

    print,"stat (0 successful):",stat

    ;determine errors:

    ELerr = dblarr(mfit)

    for i=0, mfit-1 do ELerr(i) = sqrt(invmat(i,i))

    ; adjust alpha matrix by Marquardt parameter lambda

    for i=0, mfit-1 do alpha(i,i) = alpha(i,i)*(1.0 + lambda)

    if (mfit eq 1) then begin
        delta_el = dblarr(1)
        delta_el(0) = beta/alpha
    endif else delta_el = cramer(alpha,beta)   ; adjustments

;    print,'alpha',alpha
;    print,'beta',beta

    ; adjust orbital parameters
    
    ELadj = EL
    ELadj(el_subset) = ELadj(el_subset) + delta_el 

    ELadj_sb_pri = ELadj(sb_pri_par) 
    ELadj_sb_sec = ELadj(sb_sec_par) 
 
    k=0
    print,"Orbital Element     Adjustment"
    for i=0, 4 do begin
        if (elfix(i) eq 0) then begin
            print, elLabel(i), ': ', El(i)
        endif else begin
            print, elLabel(i), ': ', El(i), delta_el(k)
            k=k+1
        endelse
    endfor
    for i=5, 5 do begin
         if (elfix(i) eq 0) then begin
             print, elLabel(i), ': ', El(i)*180/!dpi
         endif else begin
             print, elLabel(i), ': ', El(i)*180/!dpi, delta_el(k)*180/!dpi
             k = k+1
         endelse
    endfor
    for i=6, 7 do begin
        if (elfix(i) eq 0) then begin
            print, elLabel(i), ': ', El(i)
        endif else begin
            print, elLabel(i), ': ', El(i), delta_el(k)
            k=k+1
        endelse
    endfor

    print,"Chi2 of current solution:",chi2old

    ;plot SB velocity curves

    plot,time_sb_pri,V1_sb_pri,psym=2,yrange=[vmin,vmax],xrange=[tmin,tmax]
    oplot,time_sb_pri,V1_f_sb_pri,psym=7
    for l=0, sb_pri_num-1 do oplot,[time_sb_pri(l),time_sb_pri(l)],[V1_sb_pri(l),V1_f_sb_pri(l)]

    plot,time_sb_sec,V2_sb_sec,psym=2,yrange=[vmin,vmax],xrange=[tmin,tmax]
    oplot,time_sb_sec,V2_f_sb_sec,psym=7
    for l=0, sb_sec_num-1 do oplot,[time_sb_sec(l),time_sb_sec(l)],[V2_sb_sec(l),V2_f_sb_sec(l)]

    wait,0.1

    if (count eq 0) then begin
       print,'Hit enter to continue'
       read,ans_cont
    endif

    ; calculate new chi2 for adjusted parameters

    calc_deriv_SB1, ELadj_sb_pri, elfix_sb_pri, mfit_sb_pri, time_sb_pri, V1_sb_pri, dV1_sb_pri, V1_f_sb_pri, alpha_sb_pri, beta_sb_pri

    calc_deriv_SB1, ELadj_sb_sec, elfix_sb_sec, mfit_sb_sec, time_sb_sec, V2_sb_sec, dV2_sb_sec, V2_f_sb_sec, alpha_sb_sec, beta_sb_sec, /flag_wa

    ; Determine chi squared

    chi2new = total((V1_sb_pri - V1_f_sb_pri)^2/dV1_sb_pri^2) $
            + total((V2_sb_sec - V2_f_sb_sec)^2/dV2_sb_sec^2)

    print,"chi2 of next modification:",chi2new

    if (chi2new le chi2old) then begin
        EL = ELadj
        EL_sb_pri = ELadj_sb_pri
        EL_sb_sec = ELadj_sb_sec
	chi2 = chi2new
        delta_chi = chi2old - chi2new
        lambda = lambda/10.0
        print, 'lambda', lambda
    endif else begin
	print, "next adjustment does not improve chi2"
        print, "increasing lambda by 10.0"
        lambda = lambda*10.0
        print, 'lambda', lambda
        delta_chi = 1.0
    endelse

    count = count+1

    ; do not exit unless lambda is less than 0.001

    if (lambda gt 0.001) and (delta_chi lt 0.001) then delta_chi = 1.0
;    print,'lambda',lambda
;    print,'delta_chi',delta_chi

    ;stop,'Type .cont to continue'

endwhile

lambda = 0.0

print, "Final values:"

period = EL(0)
Tperi = EL(1)
ecc = EL(2)
K1 = EL(3)
K2 = EL(4)
omega = EL(5)
Vsys1 = EL(6)
Vsys2 = EL(7)

;; Determine the velocity amplitudes
;K1 = (2.0*!dpi/(period*sec))*(a1sini*km/sqrt(1 - ecc^2))
;K2 = (2.0*!dpi/(period*sec))*(a2sini*km/sqrt(1 - ecc^2))

; Determine final error matrix:

calc_deriv_sb1, EL_sb_pri, elfix_sb_pri, mfit_sb_pri, time_sb_pri, V1_sb_pri, dV1_sb_pri, V1_f_sb_pri, alpha_sb_pri, beta_sb_pri

calc_deriv_sb1, EL_sb_sec, elfix_sb_sec, mfit_sb_sec, time_sb_sec, V2_sb_sec, dV2_sb_sec, V2_f_sb_sec, alpha_sb_sec, beta_sb_sec, /flag_wa


; combine SB2 and VB matrices...

; initialize full alpha,beta arrays
alpha = dblarr(mfit,mfit)
beta = dblarr(mfit)

beta(sb_pri_subset) = beta(sb_pri_subset) + beta_sb_pri
beta(sb_sec_subset) = beta(sb_sec_subset) + beta_sb_sec

for i=0, mfit_sb_pri-1 do begin
    for j=0, mfit_sb_pri-1 do begin
        alpha(sb_pri_subset(i),sb_pri_subset(j)) = $
          alpha(sb_pri_subset(i),sb_pri_subset(j)) + alpha_sb_pri(i,j)
    endfor
endfor

for i=0, mfit_sb_sec-1 do begin
    for j=0, mfit_sb_sec-1 do begin
        alpha(sb_sec_subset(i),sb_sec_subset(j)) = $
          alpha(sb_sec_subset(i),sb_sec_subset(j)) + alpha_sb_sec(i,j)
    endfor
endfor

;determine errors:

invmat = invert(alpha, stat, /double)

chi2 = total((V1_sb_pri - V1_f_sb_pri)^2/dV1_sb_pri^2) $
     + total((V2_sb_sec - V2_f_sb_sec)^2/dV2_sb_sec^2)

chi2_sb_pri = total((V1_sb_pri - V1_f_sb_pri)^2/dV1_sb_pri^2)
chi2_sb_sec = total((V2_sb_sec - V2_f_sb_sec)^2/dV2_sb_sec^2)


; degrees of freedom
dof = sb_pri_num + sb_sec_num - mfit
dof_sb_pri = sb_pri_num - mfit_sb_pri
dof_sb_sec = sb_sec_num - mfit_sb_sec

; reduced chi squared:
chi2red = chi2/dof

chi2red_sb_pri = chi2_sb_pri/dof_sb_pri
chi2red_sb_sec = chi2_sb_sec/dof_sb_sec

ELerr = dblarr(mfit)
for i=0,mfit-1 do ELerr(i) = sqrt(chi2red)*sqrt(invmat(i,i))


k=0
print,"Orbital Element     Error     "
for i=0, nEL - 1 do begin
    if (elfix(i) eq 0) then begin
        if (i eq 5) then print, elLabel(i), ': ', El(i)*180/!dpi $
        else print, elLabel(i), ': ', El(i)
    endif else begin
        if (i eq 5) then $
          print, elLabel(i), ': ', El(i)*180/!dpi, Elerr(k)*180/!dpi $
        else print, elLabel(i), ': ', El(i), Elerr(k)
        k=k+1
    endelse
endfor
;print,'K1:',K1
;print,'K2:',K2
print, "Final chi2:", chi2
print, "Reduced chi2:", chi2red
print, 'Degrees of freedom: ', dof
print, 'Number of SB primary data points: ', sb_pri_num
print, 'Number of SB secondary data points: ', sb_sec_num
print, 'SB pri only chi2/redchi2/dof: ',chi2_sb_pri,chi2red_sb_pri,dof_sb_pri
print, 'SB sec only chi2/redchi2/dof: ',chi2_sb_sec,chi2red_sb_sec,dof_sb_sec
print,'Number of iterations:',count

bs = [1.0,1.0,1.0,1.0,1.0,180.0/!dpi,1.0,1.0]

print,EL*bs

!P.MULTI = [0,0,0]

; Define user symbol to be rectangle
xrect = frac*[-1.0,-1.0,1.0,1.0,-1.0]
yrect = frac*[-1.0,1.0,1.0,-1.0,-1.0]
usersym,xrect,yrect

; Define user symbol to be open circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)
;Define the symbol to be a unit circle with 16 points
usersym, frac*cos(Z), frac*sin(Z)


; plot best-fit orbit and data for SB data for primary

tnum = 1000.0
tmin = min(time_sb_pri)
tmax = max(time_sb_pri)
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb1fit, EL_sb_pri, tarr, V1mod

vmax = max(V1mod)
vmin = min(V1mod)

; plot data and best-fit positions
print,'SB1 velocity curve'
plot,time_sb_pri,V1_sb_pri,psym=2,yrange=[vmin,vmax]
oplot,time_sb_pri,V1_f_sb_pri,psym=7
for l=0, sb_pri_num-1 do oplot,[time_sb_pri(l),time_sb_pri(l)],[V1_sb_pri(l),V1_f_sb_pri(l)]

; plot best-fit orbit
oplot,tarr,V1mod


tnum = 1000.0
tmin = Tperi - period*0.25
tmax = Tperi + period + period*0.25
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb1fit, EL_sb_pri, tarr, V1mod

vmax = max([max(V1mod),max(V1_sb_pri+dV1_sb_pri)])
vmin = min([min(V1mod),min(V1_sb_pri-dV1_sb_pri)])
;vmax = max([max(V1mod),max(V1_sb_pri)]);+dV1_sb_pri)])
;vmin = min([min(V1mod),min(V1_sb_pri)]);-dV1_sb_pri)])
vrange = vmax - vmin
vmin = vmin - 0.05*vrange
vmax = vmax + 0.05*vrange

; compute phase from time of periastron passage
phase_sb_pri = (time_sb_pri - Tperi)/period
; reduce phase to -1.0 to 1.0
phase_sb_pri = phase_sb_pri - fix(phase_sb_pri)
; keep phase between 0.0 to 1.0
for i=0, sb_pri_num-1 do if (phase_sb_pri(i) lt 0.0) $
  then phase_sb_pri(i) = phase_sb_pri(i) + 1.0

phase_mod = (tarr-Tperi)/period

set_plot,'ps'
device, /Color, Bits_per_pixel=8, filename='temp_sb_pri.eps',xsize=12,ysize=10,/encap

;toff = 50000.0d               ; RJD offset
;tlabel='JD - 2450000 (days)'
toff = 0.0d                     ; no offset
tlabel='Time'

; Define user symbol to be rectangle
xrect = frac*[-1.0,-1.0,1.0,1.0,-1.0]
yrect = frac*[-1.0,1.0,1.0,-1.0,-1.0]

ploterror,phase_sb_pri,V1_sb_pri,dV1_sb_pri,psym=2,yrange=[vmin,vmax],xrange =[-0.1,1.1], $
  xtitle = 'Phase', ytitle = 'RV (km/s)', xstyle=1,/NoData
oplot,phase_mod,V1mod
usersym, frac*cos(Z), frac*sin(Z),/fill
oplot,phase_sb_pri,V1_sb_pri,psym=8,color=3

device,/close
set_plot, 'x'


; plot best-fit orbit and data for SB1 data secondary

tnum = 1000.0
tmin = min(time_sb_sec)
tmax = max(time_sb_sec)
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb1fit, EL_sb_sec, tarr, V2mod, /flag_wa

vmax = max(V1mod)
vmin = min(V1mod)

; plot data and best-fit positions
print,'SB1 velocity curve'
plot,time_sb_sec,V2_sb_sec,psym=2,yrange=[vmin,vmax]
oplot,time_sb_sec,V2_f_sb_sec,psym=7
for l=0, sb_sec_num-1 do oplot,[time_sb_sec(l),time_sb_sec(l)],[V2_sb_sec(l),V2_f_sb_sec(l)]

; plot best-fit orbit
oplot,tarr,V2mod


tnum = 1000.0
tmin = Tperi - period*0.25
tmax = Tperi + period + period*0.25
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb1fit, EL_sb_sec, tarr, V2mod, /flag_wa

vmax = max([max(V2mod),max(V2_sb_sec+dV2_sb_sec)])
vmin = min([min(V2mod),min(V2_sb_sec-dV2_sb_sec)])
vrange = vmax - vmin
vmin = vmin - 0.05*vrange
vmax = vmax + 0.05*vrange

; compute phase from time of periastron passage
phase_sb_sec = (time_sb_sec - Tperi)/period
; reduce phase to -1.0 to 1.0
phase_sb_sec = phase_sb_sec - fix(phase_sb_sec)
; keep phase between 0.0 to 1.0
for i=0, sb_sec_num-1 do if (phase_sb_sec(i) lt 0.0) $
  then phase_sb_sec(i) = phase_sb_sec(i) + 1.0

phase_mod = (tarr-Tperi)/period

set_plot,'ps'
device, /Color, Bits_per_pixel=8, filename='temp_sb_sec.eps',xsize=12,ysize=10,/encap

;toff = 50000.0d               ; RJD offset
;tlabel='JD - 2450000 (days)'
toff = 0.0d                     ; no offset
tlabel='Time'

; Define user symbol to be rectangle
xrect = frac*[-1.0,-1.0,1.0,1.0,-1.0]
yrect = frac*[-1.0,1.0,1.0,-1.0,-1.0]

ploterror,phase_sb_sec,V2_sb_sec,dV2_sb_sec,psym=2,yrange=[vmin,vmax],xrange =[-0.1,1.1], $
  xtitle = 'Phase', ytitle = 'RV (km/s)', xstyle=1,/NoData
oplot,phase_mod,V2mod
usersym, frac*cos(Z), frac*sin(Z),/fill
oplot,phase_sb_sec,V2_sb_sec,psym=8,color=3

device,/close
set_plot, 'x'


end
