pro fit_velocity_sb2sb1, sb2_file, sb2_num, sb1_file, sb1_num

; Compute the spectroscopic orbital elements through a Newton-Raphson 
; technique.  Fits SB1 and SB2 simultaneously!
;
; The program requires an initial guess that can be calculated from using
; the points from the fitting curves.
;
; Calls the following routines:
;     calc_deriv_SB1.pro
;     calc_deriv_SB2.pro
;     calc_Ei.pro
;     solve_trans.pro
;     calc_sb2fit.pro - for plotting model orbits
;
; Equations:  
; Initial estimates (P0,T0,e0,K1_0,K2_0,omega0,Vsys0) and  
; a set of observations (ti,V1i,V2i) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Projected semi-major axes (a1sini, a2sini):
;       a1sini = (P/2pi)*K1*(1 - e^2)^1/2
;       a2sini = (P/2pi)*K2*(1 - e^2)^1/2
; The fitted velocities (V1_i, V2_i) are determined from the orbital elements:
;       V1 = K1*[e*cos(omega) + cos(nu + omega)] + Vsys
;       V2 = K2*[e*cos(omega) + cos(nu + omega)] + Vsys
;
; Procedure:
; Minimize chi squared between measured velocities and fitted velocities:
;	chi^2 = sum[(V1data - V1fit)^2/sigmaV1^2 
;		    + (V2data - V2fit)^2/sigmaV2^2]
; where the sums run over all data points
; Take partial derivatives of chi^2 with respect to the orbital elements,
; (P,T,e,K1,K2,omega,Vsys) and set to zero. Solve for the value of
; the orbital elements that minimize chi^2.
;
; Since the equations for V1 and V2 are non-linear, cannot solve for 
; the orbital elements that minimize chi^2 analytically.
;
; In turn, use a Newton-Raphson technique to converge upon the solution.
; Replace (V1fit,V2fit) in the chi^2 equation with the Taylor series 
; approximation,
; 	x = x| + dx/dP|(P-P0) + dx/dT|(T-T0) + dx/de|(e-e0) + dx/dw|(w-w0)
;	       + dx/d(K1)|(K1 - K1|0) + dx/d(K2)|(K2 - K2|0)
;              + dx/d(Vsys)|(Vsys - Vsys0)
; where x=(V1,V2) and the | represents the value evaluated for the
; initial estimates of the orbital elements. Now, V1fit and V2fit 
; are linear in (P,T,e,K1,K2,w,Vsys), so the partial derivatives
; can easily be taken and solved by setting up a matrix and using 
; Cramer's method.
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;       K1:      velocity amplitude of primary (in km/s) (K1)
;       K2:      velocity amplitude of secondary (in km/s) (K2)
;	omega:	 angle between node and periastron (w)
;	ti:	 time of observation
;	V1_d:	 velocity of primary - measured data
;	V2_d:    velocity of secondary - measured data
;	V1_f:	 V1 - fit (determined from orbital elements for time of obs.)
;	V2_f:    V2 - fit (determined from orbital elements for time of obs.)
;       Vsys:    systemic velocity
;
; INPUT:
;	sb2_file: file containing data points in the format:
;		      time  V1  errV1  V2  errV2
;		      where V1 is the radial velocity of the primary
;		            V2 is the radial velocity of the secondary
;                     and errV1 and errV2 are the measurement errors
;		      (enter file name using quotes)
;	sb2_num: number of data points contained in data_file
;	
;	sb1_file: file containing data points in the format:
;		      time  V1  errV1
;		      where V1 is the radial velocity of the primary
;                     and errV1 is the measurement error
;		      (enter file name using quotes)
;	sb1_num: number of data points contained in data_file
;
; Prompted inputs:
;	initial estimates for all the orbital elements: 
;             P, T, e, K1, K2, omega, Vsys
;       vector identifying which orbital elements to solve for
;             enter 0 to hold element fixed, 1 to vary)
;             ex.   1 1 1 1 1 1 1 -> solves for all
;                   0 0 0 1 0 0 0 -> solves only for K1 of primary
;
; OUTPUT:
;	best fit values of orbital elements: P,T,e,K1,K2,omega,Vsys
;       and corresponding errors
;
; Notes on units:
;    P, T, and time should be in same units (days or years)
;    K1, K2, V1, V2 should be in same units (km/s)
;
; Began 23 June 2005
;
; Version 1: Orignal copy
; Version 2: 
;    - Include Marquadt Method of adjusting diagonal elements of
;      the alpha matrix to ensure convergence. 5 July 2005
;    - Fixed non-convergence problems   7 July 2005
;        + do not remove P wrap around effects from dV/dP
;          leave (t-T)/P in full form; net effect is that adjustments
;          smaller
;        + fixed primary/secondary indices in setting dV1/dVsys and dV2/dVsys
;    - Designed for SB2's only!!!
; Version 3: 7 July 2005
;    - Add SB1 + SB2 joint solutions
;
; 31 May 2006: Save unmodified program as version 2
; 31 May 2006: Fit SB2 parameters in terms of K1 and K2
;              (as opposed to a1sini and a2sini)
;
; 22 September 2006:
;       - Fixed the way subsets of parameters are handled
;       - Wasn't properly constructing matrices if not all parameters vary

; Read in data points from data files

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time_sb2 = dblarr(sb2_num)
V1_sb2 = dblarr(sb2_num)
V2_sb2 = dblarr(sb2_num)
dV1_sb2 = dblarr(sb2_num)
dV2_sb2 = dblarr(sb2_num)

time_sb1 = dblarr(sb1_num)
V1_sb1 = dblarr(sb1_num)
dV1_sb1 = dblarr(sb1_num)

; Read SB2 measurements

openr,lun,sb2_file,/get_lun

for i=0, sb2_num-1 do begin

	readf, lun, temp1, temp2, temp3, temp4, temp5

	time_sb2(i) = temp1
	V1_sb2(i) = temp2
	dV1_sb2(i) = temp3
	V2_sb2(i) = temp4	
	dV2_sb2(i) = temp5
	
endfor

close,lun

; Read SB1 measurements

openr,lun,sb1_file

for i=0, sb1_num-1 do begin

	readf, lun, temp1, temp2, temp3

	time_sb1(i) = temp1
	V1_sb1(i) = temp2
	dV1_sb1(i) = temp3
	
endfor

close,lun

totnum = sb2_num + sb1_num
time = dblarr(totnum)

for i=0, sb2_num-1 do time(i) = time_sb2(i)
for i=sb2_num, totnum-1 do time(i) = time_sb1(i-sb2_num)

; Obtain values for P,T,e,a1sini,a2sini,omega,Vsys

period = 0d
Tperi = 0d
ecc = 0d
K1 = 0d
K2 = 0d
omega = 0d
Vsys = 0d

print,"Enter P,T,e,K1,K2,omega,Vsys:"
read,period,Tperi,ecc,K1,K2,omega,Vsys

print, "Vary each orbital element?"
print, "For each element, enter 0 to hold element fixed, 1 to vary:"
print, "[P,T,e,K1,K2,omega,Vsys]"
read, f0,f1,f2,f3,f4,f5,f6

; full array 
; (for this case, full array IS the SB2 array) 
elfix=[f0,f1,f2,f3,f4,f5,f6]
nEl = n_elements(elfix)

elfix_sb2=[f0,f1,f2,f3,f4,f5,f6]
nEl_sb2 = n_elements(elfix_sb2)

elfix_sb1=[f0,f1,f2,f3,f5,f6]
nEl_sb1 = n_elements(elfix_sb1)

print,"Initial P,T,e,K1,K2,omega,Vsys:"
print,period,Tperi,ecc,K1,K2,omega,Vsys

;convert omega to radians
omega = omega*!dpi/180

k=0
for i=0, nEl-1 do if (elfix(i) eq 0) then k=k+1
mfit = nEl - k 		; number of elements to improve

k=0
for i=0, nEl_sb2-1 do if (elfix_sb2(i) eq 0) then k=k+1
mfit_sb2 = nEl_sb2 - k 		; number of elements to improve

k=0
for i=0, nEl_sb1-1 do if (elfix_sb1(i) eq 0) then k=k+1
mfit_sb1 = nEl_sb1 - k 		; number of elements to improve

EL = [period,Tperi,ecc,K1,K2,omega,Vsys]
EL_sb2 = [period,Tperi,ecc,K1,K2,omega,Vsys]
EL_sb1 = [period,Tperi,ecc,K1,omega,Vsys]

ELadj = EL
ELadj_sb2 = EL_sb2
ELadj_sb1 = EL_sb1

elLabel = ['P','T','e','K1','K2','omega','Vsys']

; determine which indices of full set are being varied

sb2_subset = dblarr(mfit_sb2)
sb1_subset = dblarr(mfit_sb1)

; determine subarray of indices that are VB, SB1, and SB2 parameters

sb2_par = dblarr(nEl_sb2)
sb1_par = dblarr(nEl_sb1)

; determine indices of elements for which to vary
el_subset = where(elfix ne 0)
Elvar = El(el_subset)

k=0
for i=0, nEl_sb2-1 do begin
    ind = where(EL_sb2(i) eq EL)
    sb2_par(i) = ind
    if (elfix_sb2(i) ne 0) then begin
         ind = where(EL_sb2(i) eq Elvar)
         sb2_subset(k) = ind
        k=k+1
    endif
endfor

k=0
for i=0, nEl_sb1-1 do begin
    ind = where(EL_sb1(i) eq EL)
    sb1_par(i) = ind
    if (elfix_sb1(i) ne 0) then begin
         ind = where(EL_sb1(i) eq Elvar)
         sb1_subset(k) = ind
        k=k+1
    endif
endfor


; plot model and data of initial guess

tnum = 1000.0
tmin = min(time)
tmax = max(time)
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb2fit, EL_sb2, tarr, V1mod, V2mod

vmax = max([max(V1mod),max(V2mod)])
vmin = min([min(V1mod),min(V2mod)])

plot,time_sb2,V1_sb2,psym=2,yrange=[vmin,vmax],xrange =[tmin,tmax]
oplot,time_sb2,V2_sb2,psym=6
oplot,time_sb1,V1_sb1,psym=7
oplot,tarr,V1mod
oplot,tarr,V2mod

;stop,'Type .cont to continue'
ans = ''
print,'Hit enter to continue'
read,ans

vmax = max([max(V1_sb2),max(V2_sb2),max(V1_sb1)])
vmin = min([min(V1_sb2),min(V2_sb2),min(V1_sb1)])

count = 0

delta_chi = 1.0		; set initially to begin loop

lambda = 0.001   ; Marquardt method to ensure convergence

while (delta_chi gt 0.001) do begin

    ; Determine errors in orbital elements
    ; set up matrices for error determination
    ; Invert matrix through Gauss Jordan elimination (Numerical Recipes in C)

    ; set up covariance matrix and column matrix
    ; alpha = dblarr(7,7)
    ; beta = dblarr(7)

    calc_deriv_SB2, EL_sb2, elfix_sb2, mfit_sb2, time_sb2, V1_sb2, V2_sb2, dV1_sb2, dV2_sb2, V1_f_sb2, V2_f_sb2, alpha_sb2, beta_sb2

    calc_deriv_SB1, EL_sb1, elfix_sb1, mfit_sb1, time_sb1, V1_sb1, dV1_sb1, V1_f_sb1, alpha_sb1, beta_sb1

    ; Determine chi squared

    chi2old = total((V1_sb2 - V1_f_sb2)^2/dV1_sb2^2 $
                    + (V2_sb2 - V2_f_sb2)^2/dV2_sb2^2) $
                    + total((V1_sb1 - V1_f_sb1)^2/dV1_sb1^2)

    ; combine SB1 and SB2 matrices...

    ; initialize full alpha,beta arrays
    alpha = dblarr(mfit,mfit)
    beta = dblarr(mfit)

    beta(sb2_subset) = beta(sb2_subset) + beta_sb2
    beta(sb1_subset) = beta(sb1_subset) + beta_sb1

    for i=0, mfit_sb2-1 do begin
        for j=0, mfit_sb2-1 do begin
            alpha(sb2_subset(i),sb2_subset(j)) = $
              alpha(sb2_subset(i),sb2_subset(j)) + alpha_sb2(i,j)
        endfor
    endfor

    for i=0, mfit_sb1-1 do begin
        for j=0, mfit_sb1-1 do begin
            alpha(sb1_subset(i),sb1_subset(j)) = $
              alpha(sb1_subset(i),sb1_subset(j)) + alpha_sb1(i,j)
        endfor
    endfor

    invmat = invert(alpha, stat, /double)

    print,"stat (0 successful):",stat

    ;determine errors:

    ELerr = dblarr(mfit)

    for i=0, mfit-1 do ELerr(i) = sqrt(invmat(i,i))

    ; adjust alpha matrix by Marquardt parameter lambda

    for i=0, mfit-1 do alpha(i,i) = alpha(i,i)*(1.0 + lambda)

    if (mfit eq 1) then begin
        delta_el = dblarr(1)
        delta_el(0) = beta/alpha
    endif else delta_el = cramer(alpha,beta)   ; adjustments

;    print,'alpha',alpha
;    print,'beta',beta

    ; adjust orbital parameters

    ELadj = EL
    ELadj(el_subset) = ELadj(el_subset) + delta_el 

    ELadj_sb2 = ELadj(sb2_par) 
    ELadj_sb1 = ELadj(sb1_par) 

    k=0
    print,"Orbital Element     Adjustment"
    for i=0, nEL - 1 do begin
        if (elfix(i) eq 0) then begin
            if (i eq 5) then print, elLabel(i), ': ', El(i)*180/!dpi $
            else print, elLabel(i), ': ', El(i)
        endif else begin
            if (i eq 5) then $
              print, elLabel(i), ': ', El(i)*180/!dpi, delta_el(k)*180/!dpi $
            else print, elLabel(i), ': ', El(i), delta_el(k)
            k=k+1
        endelse
    endfor

    print,"Chi2 of current solution:",chi2old

    plot,time_sb2,V1_sb2,psym=2,yrange=[vmin,vmax],xrange=[tmin,tmax]
    oplot,time_sb2,V1_f_sb2,psym=7
    for l=0, sb2_num-1 do oplot,[time_sb2(l),time_sb2(l)],[V1_sb2(l),V1_f_sb2(l)]
    oplot,time_sb2,V2_sb2,psym=6
    oplot,time_sb2,V2_f_sb2,psym=4
    for l=0, sb2_num-1 do oplot,[time_sb2(l),time_sb2(l)],[V2_sb2(l),V2_f_sb2(l)]
    oplot,time_sb1,V1_sb1,psym=2
    oplot,time_sb1,V1_f_sb1,psym=7
    for l=0, sb1_num-1 do oplot,[time_sb1(l),time_sb1(l)],[V1_sb1(l),V1_f_sb1(l)]

    ;if (count eq 0) then stop,'Type .cont to continue'

    calc_deriv_SB2, ELadj_sb2, elfix_sb2, mfit_sb2, time_sb2, V1_sb2, V2_sb2, dV1_sb2, dV2_sb2, V1_f_sb2, V2_f_sb2, alpha_sb2, beta_sb2

    calc_deriv_SB1, ELadj_sb1, elfix_sb1, mfit_sb1, time_sb1, V1_sb1, dV1_sb1, V1_f_sb1, alpha_sb1, beta_sb1

    ; Determine chi squared

    chi2new = total((V1_sb2 - V1_f_sb2)^2/dV1_sb2^2 $
                    + (V2_sb2 - V2_f_sb2)^2/dV2_sb2^2) $
                    + total((V1_sb1 - V1_f_sb1)^2/dV1_sb1^2)

    print,"chi2 of next modification:",chi2new

    if (chi2new le chi2old) then begin
        EL_sb2 = ELadj_sb2
        EL_sb1 = ELadj_sb1
        EL = EL_sb2
	chi2 = chi2new
        delta_chi = chi2old - chi2new
        lambda = lambda/10.0
        print, 'lambda', lambda
    endif else begin
	print, "next adjustment does not improve chi2"
        print, "increasing lambda by 10.0"
        lambda = lambda*10.0
        print, 'lambda', lambda
        delta_chi = 1.0
    endelse

    count = count+1

    ; do not exit unless lambda is less than 0.001

;    if (lambda gt 0.001) and (delta_chi lt 0.001) then delta_chi = 1.0
;    print,'lambda',lambda
;    print,'delta_chi',delta_chi

    ;stop,'Type .cont to continue'

endwhile

lambda = 0.0

print, "Final values:"

period = EL(0)
Tperi = EL(1)
ecc = EL(2)
K1 = EL(3)
K2 = EL(4)
omega = EL(5)
Vsys = EL(6)

;; Determine the velocity amplitudes
;K1 = (2.0*!dpi/(period*sec))*(a1sini*km/sqrt(1 - ecc^2))
;K2 = (2.0*!dpi/(period*sec))*(a2sini*km/sqrt(1 - ecc^2))

; Determine final error matrix:

calc_deriv_SB2, EL_sb2, elfix_sb2, mfit_sb2, time_sb2, V1_sb2, V2_sb2, dV1_sb2, dV2_sb2, V1_f_sb2, V2_f_sb2, alpha_sb2, beta_sb2

calc_deriv_SB1, EL_sb1, elfix_sb1, mfit_sb1, time_sb1, V1_sb1, dV1_sb1, V1_f_sb1, alpha_sb1, beta_sb1

; combine SB1 and SB2 matrices...

; initialize full alpha,beta arrays
alpha = dblarr(mfit,mfit)
beta = dblarr(mfit)

beta(sb2_subset) = beta(sb2_subset) + beta_sb2
beta(sb1_subset) = beta(sb1_subset) + beta_sb1

for i=0, mfit_sb2-1 do begin
    for j=0, mfit_sb2-1 do begin
        alpha(sb2_subset(i),sb2_subset(j)) = $
          alpha(sb2_subset(i),sb2_subset(j)) + alpha_sb2(i,j)
    endfor
endfor

for i=0, mfit_sb1-1 do begin
    for j=0, mfit_sb1-1 do begin
        alpha(sb1_subset(i),sb1_subset(j)) = $
          alpha(sb1_subset(i),sb1_subset(j)) + alpha_sb1(i,j)
    endfor
endfor

invmat = invert(alpha, stat, /double)

;determine errors:

chi2 = total((V1_sb2 - V1_f_sb2)^2/dV1_sb2^2 $
             + (V2_sb2 - V2_f_sb2)^2/dV2_sb2^2) $
             + total((V1_sb1 - V1_f_sb1)^2/dV1_sb1^2)

; degrees of freedom
dof = 2.0*sb2_num + 1.0*sb1_num - mfit

; reduced chi squared:
chi2red = chi2/dof

ELerr = dblarr(mfit)
for i=0,mfit-1 do ELerr(i) = sqrt(chi2red)*sqrt(invmat(i,i))

k=0
print,"Orbital Element     Error     "
for i=0, nEL - 1 do begin
    if (elfix(i) eq 0) then begin
        if (i eq 5) then print, elLabel(i), ': ', El(i)*180/!dpi $
        else print, elLabel(i), ': ', El(i)
    endif else begin
        if (i eq 5) then $
          print, elLabel(i), ': ', El(i)*180/!dpi, Elerr(k)*180/!dpi $
        else print, elLabel(i), ': ', El(i), Elerr(k)
        k=k+1
    endelse
endfor
;print,'K1:',K1
;print,'K2:',K2
print, "Final chi2:", chi2
print, "Reduced chi2:", chi2red
print,'Number of iterations:',count
print,EL

free_lun, lun

; plot best-fit orbit and data

tnum = 1000.0
tmin = min(time)
tmax = max(time)
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb2fit, EL_sb2, tarr, V1mod, V2mod

vmax = max([max(V1mod),max(V2mod)])
vmin = min([min(V1mod),min(V2mod)])

; plot data
plot,time_sb2,V1_sb2,psym=2,yrange=[vmin,vmax],xrange =[tmin,tmax]
oplot,time_sb2,V2_sb2,psym=6
oplot,time_sb1,V1_sb1,psym=7

; since model is plotted as solid line, it will suffice just to
; connect data point directly to solid lne (without plotting fit positions)
for l=0, sb2_num-1 do oplot,[time_sb2(l),time_sb2(l)],[V1_sb2(l),V1_f_sb2(l)]
for l=0, sb2_num-1 do oplot,[time_sb2(l),time_sb2(l)],[V2_sb2(l),V2_f_sb2(l)]
for l=0, sb1_num-1 do oplot,[time_sb1(l),time_sb1(l)],[V1_sb1(l),V1_f_sb1(l)]

; plot best-fit orbit
oplot,tarr,V1mod
oplot,tarr,V2mod

end

