pro hartkopf, data_file, num

; Compute the orbital elements of a binary system, using the least squares
; technique described by Hartkopf, McAlister, Franz 1989, AJ, 98, 1014
;
; Procedure: Given (P,T,e) and a set of observations (ti,xi,yi), the eccentric
; anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)
; Four Thiele-Innes elements determined form least squares solution of 
; equations:
;	xi = AXi + FYi
;	yi = BXi + GYi
; Calculate geometric elements a("),i,Omega,omega from (A,F,B,G)
; 
; Paramers:
;	P:	period
;	T: 	time of periastron passage
;	e:	eccentricity
;	a:	semi-major axis (in mas)
;	i:	inclination
;	Omega:	position angle of node
;	omega:	angle between node and periastron
;	ti:	time of observation
;	xi:	RA
;	yi:	DEC
;
; Note: xi and yi are not the same convention for RA and DEC chosen to make 
; the orbital plots. Specifically, in the fitting the apparent ellipse 
; (ORBFITE.PRO):
;	x = RA = -rho*sin(theta),   y = DEC = rho*cos(theta)
;	(i.e. North is in the direction of the positive y axis, theta is
;	measured counterclockwise from positive y-axis)
; Whereas in this program, the coordinate system set up by Couteau,
;	x = rho*cos(theta),   y = rho*sin(theta)
;	(i.e. North id in the direction of the positive x-axis and the 
;	y-axis points east)
; Both coordinate systems have the same separation and position angle 
; orietations, they are just rotated by 90 deg with respect to each other.
;
; Input: 
;	data_file: file containing data points in the format:
;		      time  theta rho 
;		      where theta is the position angle and rho the separation
;		      (enter file name using quotes)
;	num: number of data points contained in data_file
;
; Prompted inputs:
;	period
;	time of periastron passage
;	eccentricity
;
; Began: May 2002

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time = dblarr(num)
theta = dblarr(num)
rho = dblarr(num)

openr,lun,data_file,/get_lun

for i=0, num-1 do begin

	readf, lun, temp1, temp2, temp3

	time(i) = temp1
	theta(i) = temp2
	rho(i) = temp3	
	
endfor

close,lun

; convert theta to radians

theta = theta*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; Obtain initial values for P,T,e

P0 = 0d
T0 = 0d
e0 = 0d

print,"Enter P T e (period, time of periastron passage, eccentricity):"
read, P0, T0, e0

; read in step sizes and searching ranges for P, T, e

Pstep = 0d
minP = 0d
maxP = 0d
Tstep = 0d
minT = 0d
maxT = 0d
estep = 0d
mine = 0d
maxe = 0d

print," "
print,"Determine searching ranges and step sizes for P,T,e."
print,"step is the incremental size for which to vary given parameter"
print,"min is the lower value for the searching range, ie: Pi = P-minP"
print,"max is the higher value for the searching range, ie: Pf = P-maxP"
print," "
print, "Enter Pstep size, minP, maxP for search in P:"
read, Pstep, minP, maxP
print, "Enter Tstep size, minT, maxT for search in T:"
read, Tstep, minT, maxT
print, "Enter estep size, mine, maxe for search in e:"
read, estep, mine, maxe

; begin modifying P,T,e

firsttime=0

for period = (P0 - minP) , (P0 + maxP), Pstep do begin

	for Tperi = (T0 - minT) , (T0 + maxT), Tstep do begin

		for ecc = (e0 - mine) , (e0 + maxe), estep do begin

			; Determine the eccentric anomalies Ei: 
			;     mu(ti - T) = Ei - esin(Ei)  ... Kepler's Equation
			;     where mu = 360/P = 2*pi/P
			; Solve this transcendental equation through an 
			; iterative procedure.
			; Use SOLVE_TRANS.PRO

			; array to hold Ei's (the eccentric anomaly)

			Ei = dblarr(num) 

			mu = 2*!dpi/period

			for i=0, num-1 do begin


				Mi = mu*(time(i) - Tperi) 

				; reduce to same epoch

				Mi = 2*!dpi*((time(i) - Tperi)/period $	
					- fix((time(i) - Tperi)/period))

				solve_trans,ecc,Mi,Eit

				Ei(i) = Eit

			endfor


			; Normalized rectangular coordinates Xi & Yi:
			;	Xi = cos(Ei) - e
			;	Yi = sqrt(1 - e^2)*sin(Ei)

			; Xi and Yi are both dblarr(num)

			Xi = cos(Ei) - ecc
			Yi = sqrt(1 - ecc^2)*sin(Ei)

			; print,"Xi:",Xi
			; print,"Yi:",Yi

			; Four Thiele-Innes elements determined from least 
			; squares solution of equations:
			;	xi = AXi + FYi
			;	yi = BXi + GYi
			; Calculate geometric elements a("),i,Omega,omega 
			; from (A,F,B,G)

			linfith, Xi,Yi,xarr,A_cap,F_cap
			linfith, Xi,Yi,yarr,B_cap,G_cap

;			print,"(A,F,B,G) from least squares fit:"
;			print, A_cap,F_cap,B_cap,G_cap
;
;			print, "A B F G:", A_cap, B_cap, F_cap, G_cap

;			; determine a,i,Omega,omega (using AIWW.PRO)
;
;			aiWw, A_cap,B_cap,F_cap,G_cap,major,inc,W_cap,w_low
;
;			print, "a i Omega omega:", major, (180/!dpi)*inc, $
;				(180/!dpi)*W_cap, (180/!dpi)*w_low

			; Determine chi squared

			xfit = A_cap*Xi + F_cap*Yi	;fitted points
			yfit = B_cap*Xi + G_cap*Yi

			chi2 = total((xarr-xfit)^2 + (yarr-yfit)^2)

;			print, "chi2", chi2

			if (firsttime eq 0) then begin
				chimin = chi2
				bestpar = [period,Tperi,ecc,chi2]
				firsttime=1
			endif 

			if (chi2 le chimin) then begin
				chimin = chi2
				bestpar = [period,Tperi,ecc,chi2]
				print,"Best fit P T e chi^2"
				print, bestpar

				;plot data and fitted points

				plot, xarr, yarr, psym=4, XSTYLE=1, YSTYLE=1, $
			     		ISOTROPIC=1
;			     		ISOTROPIC=1, xrange=[-50,+150], $
;					yrange=[-150,+50]
				oplot, xfit, yfit, psym=5 

				wait,0.2

			endif else begin
	
				if (chi2 le 0.001) then begin
				print,"P T e chi^2"
				print,period,Tperi,ecc,chi2
		
				;plot data and fitted points

;				plot, xarr, yarr, psym=4, XSTYLE=1, YSTYLE=1, $
;			     		ISOTROPIC=1, xrange=[-50,+150], $
;					yrange=[-150,+50]
;				oplot, xfit, yfit, psym=5 

;				wait,0.2
				endif
			endelse
		endfor
	endfor
endfor

print,"Best fit P T e chi^2"
print, bestpar

;plot data and fitted points

;print, "Do you want to save plot as a postscript file?  (Type yes or no):"
;answer = ""
;read,answer
;; Set plotting device to ps if answer is yes
;if (answer eq "yes") then begin
;	mydevice=!d.name
;	set_plot, 'ps'
;	device, /Color, Bits_per_pixel=8, filename='temp.ps' 
;endif
;
;plot, xarr, yarr, psym=4, XSTYLE=1, YSTYLE=1, ISOTROPIC=1, $
;	xrange=[-50,+150], yrange=[-150,+50]
;oplot, xfit, yfit, psym=5 
;
;; set plotting device back to terminal if previously saving to ps file
;if (answer eq "yes") then begin 
;	device, /close
;	set_plot, mydevice
;endif
 
free_lun,lun

end


pro linfith, X, Y, z, c, d

; Least squares linear fit to the equation: z = cX + dY
; where z = (x,y), c = (A,B), and d = (F,G)
; The user supplies the x,y,z arrays
; Program returns values for c,d
; The least squares fit is performed in the same manner as Harthopf's code
; Minimize chi squared with respect to A,B,F,G (set partial derivatives to 0):
; 	chi^2 = sum[(x - AX - FY)^2 + (y - BX - GY)^2]
; Because x is only a function of (A,F), and y is only a funtion of (B,G),
; minimizing chi squared can proceed independently for each of the x,y 
; variables.

num_data = n_elements(z)	;number of data points
ind_data = num_data -1


c = (total(X*Y)*total(z*Y) - total(z*X)*total(Y*Y))/ $
	(total(X*Y)^2 - total(X*X)*total(Y*Y))

d = (total(X*Y)*total(z*X) - total(z*Y)*total(X*X))/ $
	(total(X*Y)^2 - total(X*X)*total(Y*Y))

;print, "e^2:", e2new

end

;
;
;

pro solve_trans, e, M, EE

; Solve transcendental equation of the form E - esinE = M.
; Use iterative procedure to determine E.
; Initial approximation: E_0 = M + esinM - e^2/2 sin(2M)
; Improve solution by iterating the following to formulae:
;	M_0 = E_0 - esin(E_0)
;	E_1 = E_0 + (M - M_0)/(1 - ecos(E_0))
;	(derivative of Kepler's equation)
;
; Method adapted from Heintz 1978 (p.34-35)
; Results compared with point-&-click graphical method.  Iterative approach 
; leads to exact solution that satisfies E - esinE = M.  Therefore, 
; point-&-click method is subsequently removed from orbit fitting.
;
; INPUT:
;	e: eccentricity
;	M: mean anomaly   M= 2*Pi/P
;
; OUTPUT:
;	EE: eccentric anomaly
;
; Created: 9 May 2002

; Initial approximation:

EE = M + e*sin(M) + e^2/2*sin(2*M)

;print, "Initial approximation for E:", EE

EEi = 0d	; parameter to hold initial value once enter while loop

count = 0

while (abs(EE - EEi) gt 0.000001) do begin

	EEi = EE

	Mi = EEi - e*sin(EEi)

	EE = EEi + (M - Mi)/(1 - e*cos(EEi))

	count=count+1

endwhile

;print, "Final iterated E:", EE

;print,"Number of iterations:",count

end

;
;
;

pro aiWw, A_cap, B_cap, F_cap, G_cap, true_a, inc, Omega_cap, omega_low

; Determine a,i,Omega,omega from Thiele-Innes elements A,B,F,G
; INPUT: A,B,F,G
; OUTPUT: a,i,Omega,omega

BmF = (B_cap - F_cap)
BpF = (B_cap + F_cap)
AmG = (A_cap - G_cap)
ApG = (A_cap + G_cap)

; Determine sign of (Omega + omega) and (Omega - omega)
print, "Check sign of (Omega + omega) and (Omega - omega)"

sinOpo = BmF		; sin(Omega + omega) is propotional to (B-F)
cosOpo = ApG		; cos(Omega + omega) is propotional to (A+G)

sinOmo = BpF		; sin(Omega - omega) is propotional to (B+F)
cosOmo = AmG		; cos(Omega - omega) is propotional to (A-G)

Opo = atan((BmF)/(ApG))
Omo = atan((BpF)/(AmG))

print,"Check sign of Omega +/- omega"
print,"O+o:", Opo
print,"O-o:", Omo

if ((sinOpo/abs(sinOpo)) ne (sin(Opo)/abs(sin(Opo)))) then Opo = !dpi+Opo
if ((sinOmo/abs(sinOmo)) ne (sin(Omo)/abs(sin(Omo)))) then Omo = !dpi+Omo

print,"Corrected O+o:", Opo
print,"Corrected O-o:", Omo

Omega_cap = (Opo + Omo)/2d

print,"Check if 0 < Omega < pi"
print,"Omega:", Omega_cap 

if (Omega_cap gt !dpi) then omega_cap = omega_cap - !dpi
if (Omega_cap lt 0d) then omega_cap = omega_cap + !dpi

print,"Corrected Omega:", Omega_cap 

omega_low = Opo - Omega_cap

inc = 2* atan(sqrt((BpF*sin(Opo))/(BmF*sin(Omo))))

true_a = sqrt((A_cap*G_cap - B_cap*F_cap)/cos(inc))

end
