pro hartkopf_mc_constraint, data_file, num

; Do a Monte Carlo search through P,T,e parameter space
; -> remove sampling biases  (modification of test_chi.pro)
;
; This is a copy of HARTKOPF_E.PRO that uses a variation of 1 in chi2, rather
; than in the reduced chi2, to compute errors.
;
; 6 April 2006: scale chi2 so that reduced chi = 1
;
; Compute the orbital elements of a binary system, using the least squares
; technique described by Hartkopf, lMcAlister, Franz 1989, AJ, 98, 1014
;
; Procedure: Given (P,T,e) and a set of observations (ti,xi,yi), the eccentric
; anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)
; Four Thiele-Innes elements determined from least squares solution of 
; equations:
;	xi = AXi + FYi
;	yi = BXi + GYi
; Calculate geometric elements a("),i,Omega,omega from (A,F,B,G)
; 
; Parameters:
;	P:	period
;	T: 	time of periastron passage
;	e:	eccentricity
;	a:	semi-major axis (in mas)
;	i:	inclination
;	Omega:	position angle of node
;	omega:	angle between node and periastron
;	ti:	time of observation
;	xi:	RA
;	yi:	DEC
;
; Note: xi and yi are not the same convention for RA and DEC chosen to make 
; the orbital plots. Specifically, in the fitting the apparent ellipse 
; (ORBFITE.PRO):
;	x = RA = -rho*sin(theta),   y = DEC = rho*cos(theta)
;	(i.e. North is in the direction of the positive y axis, theta is
;	measured counterclockwise from positive y-axis)
; Whereas in this program, the coordinate system set up by Couteau,
;	x = rho*cos(theta),   y = rho*sin(theta)
;	(i.e. North id in the direction of the positive x-axis and the 
;	y-axis points east)
; Both coordinate systems have the same separation and position angle 
; orietations, they are just rotated by 90 deg with respect to each other.
;
; IINPUT: 
;	data_file: file containing data points in the format:
;		      time  theta  dtheta rho drho 
;		      where theta is the position angle and rho the separation
;		      dtheta and drho are the errors
;		      (enter file name using quotes)
;	num: number of data points contained in data_file
;
; Prompted inputs:
;	period (P)
;	time of periastron passage (T)
;	eccentricity (e)
;       epoch: search for T within P/2 of epoch
;
; OUTPUT: 
;       result_chi:  file containing all solutions within Delta(chi2)=1
;                    format: P T e chi2
;
;       xyresiduals: file containing (Observed-Calculated) residuals
;                    format:
;                    date     dx       xO-C   xOC/dx dy      yO-C   yOC/dy
;                    where dx and dy are the measurement uncertainties
;                          xO-C and yO-C are the (data - fit) values
;                          xOC/dx and yOC/dy are the ratio of the two
;
; Calls the following routines:
;     calc_AFBG.pro
;     calc_Ei.pro
;     solve_trans.pro
;     linfith.pro
;     aiWw.pro
;     calc_deriv_vb.pro
;
; Began: May 2002
; 25 November 2003: Monte Carlo search technique
; 31 August 2011: Move subroutines at end to separate files

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time = dblarr(num)
theta = dblarr(num)
rho = dblarr(num)
dtheta = dblarr(num)
drho = dblarr(num)

openr,lun,data_file,/get_lun

for i=0, num-1 do begin

	readf, lun, temp1, temp2, temp3, temp4, temp5

	time(i) = temp1
	theta(i) = temp2
	dtheta(i) = temp3
	rho(i) = temp4	
	drho(i) = temp5
	
endfor

close,lun

; convert theta and dtheta to radians

theta = theta*!dpi/180
dtheta = dtheta*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(theta)^2*drho^2 + rho^2*sin(theta)^2*dtheta^2)
dy = sqrt(sin(theta)^2*drho^2 + rho^2*cos(theta)^2*dtheta^2)

; Obtain initial values for P,T,e

P0 = 0d
T0 = 0d
e0 = 0d

print,"Enter P T e (period, time of periastron passage, eccentricity):"
read, P0, T0, e0

; only search for orbital solutions where the time of periaston
; passage is within a cycle of the period from the current
; epoch (ex. 2000)

epoch=0d

print,' '
print,'This program will only search for solutions of the time of'
print,'periastron passage within +/- P/2 of the chosen epoch.'

print,'Enter epoch of periastron passage:'
read, epoch

; read in searching ranges for P, T, e

minP = 0d
maxP = 0d
minT = 0d
maxT = 0d
mine = 0d
maxe = 0d

print," "
print,"Determine searching ranges for P,T,e."
print,'Search range goes from min to max:'
print," "
print, "Enter minP, maxP for search in P:"
read, minP, maxP
print, "Enter minT, maxT for search in T:"
read, minT, maxT
print, "Enter mine, maxe for search in e:"
read, mine, maxe

rangeP = maxP - minP
rangeT = maxT - minT
rangee = maxe - mine

sep_max = 0.0d
i_min = 0.0d
i_max = 0.0d

print,'Enter constraint on maximum separation:'
read,sep_max

print,'Enter constraints on inclination (min max):'
read,i_min,i_max
i_min = i_min*!dpi/180.0
i_max = i_max*!dpi/180.0

; open file "result_chi" to save chi squared values
openw,lun,"result_chi"

; degrees of freedom
dof = 2d*num - 7d

; Create initial model:
			
	period = P0
	Tperi = T0
	ecc =e0

	; Determine fitted points (xfit,yfit) by computing the Thiele Innes
	; elements A,F,B,G

	calc_AFBG, time, xarr, dx, yarr, dy, period, Tperi, ecc, $
		   A_cap, F_cap, B_cap, G_cap, xfit, yfit

	; Determine chi squared

	chi2 = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

	; reduced chi squared:
	chi2red = chi2/dof

        print,'Unscaled chi2:',chi2
        ans_scale = 'y'
        print,'Do you want to scale chi2 so that chi2red = 1.0 (y/n)?'
        read,ans_scale
        if (ans_scale eq 'y') then scale = 1.0/chi2red else scale = 1.0

        chi2 = chi2*scale
        print,'Scaled chi2:',chi2
        print,'Scale factor for chi2:',scale

	chimin = chi2
	chiredmin = chi2/dof
	bestpar = [period,Tperi,ecc,chi2]
	bestABFG = [A_cap,B_cap,F_cap,G_cap]
	xbest = xfit
	ybest = yfit


; begin modifying P,T,e

total_num = 0
chidiff = 0.0

print, 'Enter the variation of chi2 you wish to examine:'
read, chidiff
print, 'Enter number of solutions you wish to find within this variation of chi2:'
read,total_num

seed=1001L
print,'Enter seed for random generator (ex. 1001L):'
read,seed

count = 0

while (count lt total_num) do begin

	period = minP + rangeP*randomu(seed,/double,/uniform)
	ecc = mine + rangee*randomu(seed,/double,/uniform)

	; only search for orbital solutions where the time of periaston
	; passage is within a cycle of the period from the current
	; epoch (ex. 2000)

        if (rangeT gt period) then begin
           minTP = epoch - 0.5*period
           Tperi = minTP + period*randomu(seed,/double,/uniform)
        endif else $
           Tperi = minT + rangeT*randomu(seed,/double,/uniform)

			; Calculate Thiele-Innes elements and fitted points
			; (A,F,B,G;xfit,yfit)	

			calc_AFBG, time, xarr, dx, yarr, dy, $
				   period, Tperi, ecc, $
		   		   A_cap, F_cap, B_cap, G_cap, xfit, yfit
	
			; Determine chi squared

;			chi2 = total((xarr-xfit)^2 + (yarr-yfit)^2)
			chi2 = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

                        chi2 = chi2*scale

			; reduced chi squared:
			chi2red = chi2/dof

;			print, "chi2", chi2

			if (chi2 le chimin) then begin

                           ; determine a,i,Omega,omega (using AIWW.PRO)
                           aiWw, A_cap,B_cap,F_cap,G_cap,major,inc,W_cap,w_low
                           
                           ; Compute max sep at apastron:
                           ap_sep = major*(1+ecc)

                           if (ap_sep le sep_max) then begin
                              if (inc ge i_min) and (inc le i_max) then begin

				chimin = chi2
				chiredmin = chi2red
				bestpar = [period,Tperi,ecc,chi2]
				print,"Best fit P T e chi^2"
				print, bestpar
				printf,lun, bestpar
				bestABFG = [A_cap,B_cap,F_cap,G_cap]
				xbest = xfit
				ybest = yfit

				;plot data and fitted points

				plot, xarr, yarr, psym=4, XSTYLE=1, YSTYLE=1, $
			     		ISOTROPIC=1
;			     		ISOTROPIC=1, xrange=[-50,+150], $
;					yrange=[-150,+50]
				oplot, xfit, yfit, psym=5 

				wait,0.2
                                
                             endif
                           endif

			endif else begin
	
				if (chi2 le (chimin + chidiff)) then begin

                                ; determine a,i,Omega,omega (using AIWW.PRO)
                                aiWw, A_cap,B_cap,F_cap,G_cap,major,inc,W_cap,w_low
                                ; Compute max sep at apastron:
                                ap_sep = major*(1+ecc)

                                if (ap_sep le sep_max) then begin
                                if (inc ge i_min) and (inc le i_max) then begin

                                   print,"P T e chi^2"
                                   print,period,Tperi,ecc,chi2
                                   printf,lun,period,Tperi,ecc,chi2
                                   count = count+1
                                   print,count
		
				   ;plot data and fitted points

;			  	plot, xarr, yarr, psym=4, XSTYLE=1, YSTYLE=1, $
;			     		ISOTROPIC=1, xrange=[-50,+150], $
;					yrange=[-150,+50]
;				oplot, xfit, yfit, psym=5 

;				wait,0.2
                            
                                endif
                                endif

				endif
			endelse
endwhile

close,lun

print,"Best fit P T e chi^2"
print, bestpar

print,"reduced chi2:", chiredmin

print, "A B F G:", bestABFG

period=bestpar(0)
Tperi = bestpar(1)
ecc = bestpar(2)
A_cap = bestABFG(0)
B_cap = bestABFG(1)
F_cap = bestABFG(2)
G_cap = bestABFG(3)

xfit = xbest
yfit = ybest

; determine a,i,Omega,omega (using AIWW.PRO)

aiWw, A_cap,B_cap,F_cap,G_cap,major,inc,W_cap,w_low

print, "a i Omega omega:", major, (180/!dpi)*inc, $
(180/!dpi)*W_cap, (180/!dpi)*w_low

EL = [period,Tperi,ecc,major,inc,W_cap,w_low]

; Determine errors in orbital elements
; set up matrices for error determination
; Invert matrix through Gauss Jordan elimination (Numerical Recipes in C)

; set up covariance matrix and column matrix
alpha = dblarr(7,7)
beta = dblarr(7)
;For each element, enter 0 to hold element fixed, 1 to vary
elfix = [1.0,1.0,1.0,1.0,1.0,1.0,1.0]
indfix = where(elfix eq 1.0, mfit)

;calc_err, EL, time, theta, rho, dtheta, drho, alpha, beta
calc_deriv_vb, EL, elfix, mfit, time, theta, rho, dtheta, drho, theta_f, rho_f, alpha, beta

;print,"covariance matrix and right-hand side vector:"
;print,"alpha"
;print,alpha
;print,"beta"
;print,beta	

invmat = invert(alpha, stat, /double)

;print,"stat (0 successful):",stat

;print,"invmat"
;print,invmat

;determine errors:

ELerr = dblarr(7)

for i=0,6 do ELerr(i) = sqrt(chiredmin)*sqrt(invmat(i,i))

print, "P T e a i Omega omega"
print,EL(0),El(1),EL(2),EL(3),180*EL(4)/!dpi,180*EL(5)/!dpi,180*EL(6)/!dpi
print, "Errors:"
print,Elerr(0),Elerr(1),ELerr(2),ELerr(3),180*ELerr(4)/!dpi, $
	180*ELerr(5)/!dpi,180*ELerr(6)/!dpi

;; perform gauss-jordan elimination to invert alpha and find solution vector B
;
;gaussj, alpha, beta
;
;print,"after gauss-jordan elimination:"
;print,"alpha"
;print,alpha
;print,"beta"
;print,beta	

; print O-C residuals (Observed - Calculated)

xOC = xarr - xfit
yOC = yarr - yfit

openw,lun,'xyresiduals'
printf,lun,"O-C residuals (data - fit)"
printf,lun,"date     dx       xO-C   xOC/dx dy      yO-C   yOC/dy"
;printf,lun,"date   dx   xdata - xfit   xOC/dx   dy   ydata - yfit   yOC/dy"
for i=0, num-1 do printf,lun,format='(f8.3,f7.3,f8.3,f7.3,f7.3,f8.3,f7.3)', $
	time(i), dx(i), xOC(i), xOC(i)/dx(i), dy(i), yOC(i), yOC(i)/dy(i)
close,lun

;for i=0, num-1 do print, "x",xarr(i)," dx",dx(i)," y",yarr(i)," dy",dy(i)

;plot data and fitted points

;print, "Do you want to save plot as a postscript file?  (Type yes or no):"
;answer = ""
;read,answer
;; Set plotting device to ps if answer is yes
;if (answer eq "yes") then begin
;	mydevice=!d.name
;	set_plot, 'ps'
;	device, /Color, Bits_per_pixel=8, filename='temp.ps' 
;endif
;
;plot, xarr, yarr, psym=4, XSTYLE=1, YSTYLE=1, ISOTROPIC=1, $
;	xrange=[-50,+150], yrange=[-150,+50]
;oplot, xfit, yfit, psym=5 
;
;; set plotting device back to terminal if previously saving to ps file
;if (answer eq "yes") then begin 
;	device, /close
;	set_plot, mydevice
;endif
 
free_lun,lun

end

