pro newt_raph, data_file, num

; Compute the orbital elements through a Newton-Raphson technique.
;
; This program should work directly with the data points (rho,theta,t), not 
; the fitting points. 
;
; The program requires an initial guess that can be calculated from using
; the points from the fitting curves.
;
; INPUT:
;       datafile: time  PA  dPA  sep  dsep
;       num: number of observations in datafile
;
; Equations:  
; Initial estimates (P0,T0,e0,A0,B0,F0,G0) and a set of observations 
; (ti,rhoi,thetai) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Radius vector:
;	r = a(1-e^2)/(1 + ecos(nu))
; Fitted positions (rho_fit,theta_fit) are determined from the orbital elements
; 	tan(theta - Omega) = tan(nu + omega)*cosi
;	rho = r*cos(nu + omega)*sec(theta - Omega)
;
; Procedure:
; Minimize chi squared between data point positions and fitted positions:
;	chi^2 = sum[(rhodata - rhofit)^2/sigmarho^2 
;		    + (thetadata - thetafit)^2/sigmatheta^2]
; Take partial derivatives of chi^2 with respect to the orbital elements,
; (P,T,e,a,i,Omega,omega) and set to zero. Solve for the value of the orbital 
; elements that minimize chi^2.
;
; Since the partial derivatives are non-linear, cannot solve for the orbital
; elements that minimize chi^2 analytically.
;
; In turn, use a Newton-Raphson technique to converge upon the solution.
; Replace (rhoi,thetai) in the chi^2 equation with the Taylor series 
; approximation,
; 	x = x| + dx/dP|(P-P0) + dx/dT|(T-T0) + dx/de|(e-e0) 
;	       + dx/da|(a-Aa) + dx/di|(i-i0) + dx/dW|(W-W0) + dx/dw|(w-w0)
; where x=(rho,theta) and the | represents the value evaluated at for the
; initial estimates for the orbital elements. Now, rhofit and thetafit are 
; linear in (P,T,e,a,i,W,w), so the partial derivatives can easily be
; taken and solved by setting up a matrix and using Cramer's method.
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;	major:	 semi-major axis (in mas) (a)
;	inc:	 inclination (i)
;	Omega:	 position angle of node (W)
;	omega:	 angle between node and periastron (w)
;	ti:	 time of observation
;	rho_d:	 separation - data
;	theta_d: postion angle - data
;	rho_f:	 sep - fit (determined from orbital elements for time of obs.)
;	theta_f: PA - fit (determined from orbital elements for time of obs.)
;
; INPUT:
;	data_file: file containing data points in the format:
;		      time  theta  etheta  rho  erho
;		      where theta is the position angle, rho is the separation
;                     and etheta, erho are the measurement uncertainties
;		      (enter file name using quotes)
;	num: number of data points contained in data_file
;	
; Prompted inputs:
;	initial estimates for all the orbital elements: P,T,e,a,i,Omega,omega
;
; Calls the following routines:
;     calc_deriv_vb.pro
;     calc_Ei.pro
;     solve_trans.pro
;     calc_vbfit.pro - for plotting model orbits!
;
; Began 14 May 2002
; Modified 11 Nov 2003: 
;       This is the current version of NEWT_RAPH.PRO
;	(v1.1 and v1.2 are older versions)
; 	- restructured program into separate subroutines
; 	- add option of holding elements fixed during iterations
; 13 January 2005:
;       Fix bugs
;       - if solving for only one element, don't use cramer.pro to solve eqn's
;       - fix 360 degree roll overs in (PAdata - PAfit)
;       Save version v4 as backup
; 11 July 2005:
;       - Include Marquadt Method of adjusting diagonal elements of
;         the alpha matrix to ensure convergence.
;       - Fixed non-convergence problems 
;           + do not remove P wrap around effects from dx/dP
;             leave (t-T)/P in full form; net effect is that adjustments
;             are smaller
;       Save v5 as backup 
; 13 July 2005:
;       - Change format of plotting routines to include a call to
;         calc_vbfit
;       Save v6 as backup

!P.font=0
!P.multi=0
!P.charsize=1.2
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.2	;scale factor for adjusting size of plotting symbols

; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time = dblarr(num)
theta = dblarr(num)
rho = dblarr(num)
dtheta = dblarr(num)
drho = dblarr(num)

openr,lun,data_file,/get_lun

for i=0, num-1 do begin

	readf, lun, temp1, temp2, temp3, temp4, temp5

	time(i) = temp1
	theta(i) = temp2
	dtheta(i) = temp3
	rho(i) = temp4	
	drho(i) = temp5
	
endfor

close,lun

; convert theta and dtheta to radians

theta = theta*!dpi/180
dtheta = dtheta*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(theta)^2*drho^2 + rho^2*sin(theta)^2*dtheta^2)
dy = sqrt(sin(theta)^2*drho^2 + rho^2*cos(theta)^2*dtheta^2)

; Obtain values for P,T,e,a,i,Omega,omega 

period = 0d
Tperi = 0d
ecc = 0d
major = 0d
inc = 0d
W_cap = 0d
w_low = 0d

print,"Enter P,T,e,a,i,Omega,omega:"
read,period,Tperi,ecc,major,inc,W_cap,w_low

print, "Vary each orbital element?"
print, "For each element, enter 0 to hold element fixed, 1 to vary:"
print, "[P,T,e,a,i,Omega,omega]"
read, f0,f1,f2,f3,f4,f5,f6

elfix=[f0,f1,f2,f3,f4,f5,f6]
nEl = n_elements(elfix)

print,"Initial P,T,e,a,i,Omega,omega:"
print,period,Tperi,ecc,major,inc,W_cap,w_low

;convert i, Omega, omega to radians
inc = inc*!dpi/180
W_cap = W_cap*!dpi/180
w_low = w_low*!dpi/180

k=0
for i=0, 6 do if (elfix(i) eq 0) then k=k+1

mfit = nEl - k 		; number of elements to improve

; determine indices of elements for which to vary
elvar = where(elfix ne 0)

EL = [period,Tperi,ecc,major,inc,W_cap,w_low]
ELadj = EL
elLabel = ['P','T','e','a','i','Omega','omega']

; plot model and data of initial guess

tnum=1000.0
tstep = period/tnum
tmin = Tperi

tarr = findgen(tnum)*tstep + tmin

calc_vbfit, EL, tarr, theta_mod, rho_mod

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

; plot best-fit orbit

plot,-yarr,xarr,psym=6
oplot,-ymod,xmod

;stop,'Type .cont to continue'
ans_cont = ' ' 
print,'Hit enter to continue'
read,ans_cont

count = 0

delta_chi = 1.0		; set initially to begin loop

lambda = 0.001   ; Marquardt method to ensure convergence

; ***** Begin while loop ***** 

while(delta_chi gt 0.001) do begin

    ; Determine errors in orbital elements
    ; set up matrices for error determination
    ; Invert matrix through Gauss Jordan elimination (Numerical Recipes in C)

    ; set up covariance matrix and column matrix
    ; alpha = dblarr(7,7)
    ; beta = dblarr(7)

    calc_deriv_vb, EL, elfix, mfit, time, theta, rho, dtheta, drho, theta_f, rho_f, alpha, beta

    ; Determine chi squared

    ; convert data points to x and y coordinates

    xfit = rho_f * cos(theta_f)	; x coordinate
    yfit = rho_f * sin(theta_f)	; y coordinate
			
    ;chi2old = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

    diff_theta = theta - theta_f

    ; account for any wrap-arounds from 360 deg to 0 deg

    for k=0, num-1 do begin
        if (diff_theta(k) ge !dpi) then $
          diff_theta(k) = diff_theta(k) - 2*!dpi
        if (diff_theta(k) le -!dpi) then $
          diff_theta(k) = 2*!dpi + diff_theta(k)	
    endfor

    chi2old = total((diff_theta)^2/dtheta^2 + (rho - rho_f)^2/drho^2)

    invmat = invert(alpha, stat, /double)

    print,"stat (0 successful):",stat

    ;determine errors:

    ELerr = dblarr(mfit)

    for i=0, mfit-1 do ELerr(i) = sqrt(invmat(i,i))

    ; adjust alpha matrix by Marquardt parameter lambda

    for i=0, mfit-1 do alpha(i,i) = alpha(i,i)*(1.0 + lambda)

    if (mfit eq 1) then begin
        delta_el = dblarr(1)
        delta_el(0) = beta/alpha
    endif else delta_el = cramer(alpha,beta)   ; adjustments

    ; adjust orbital parameters
    
    for i=0, mfit-1 do ELadj(elvar(i)) = EL(elvar(i)) + delta_el(i)

    k=0
    print,"Orbital Element     Adjustment"
    for i=0, 3 do begin
        if (elfix(i) eq 0) then begin
            print, elLabel(i), ': ', El(i)
        endif else begin
            print, elLabel(i), ': ', El(i), delta_el(k)
            k=k+1
        endelse
    endfor
    for i=4, nEl-1 do begin
         if (elfix(i) eq 0) then begin
             print, elLabel(i), ': ', El(i)*180/!dpi
         endif else begin
             print, elLabel(i), ': ', El(i)*180/!dpi, delta_el(k)*180/!dpi
             k = k+1
         endelse
    endfor

    print,"Chi2 of current solution:",chi2old

    plot,-yarr,xarr,psym=6
    oplot,-yfit,xfit,psym=7
    for l=0, num-1 do oplot,[-yarr(l),-yfit(l)],[xarr(l),xfit(l)]

    ;if (count eq 0) then stop,'Type .cont to continue'

    calc_deriv_vb, ELadj, elfix, mfit, time, theta, rho, dtheta, drho, theta_f, rho_f, alpha, beta

    ; Determine chi squared

    ; convert data points to x and y coordinates

    xfit = rho_f * cos(theta_f)	; x coordinate
    yfit = rho_f * sin(theta_f)	; y coordinate
			
    ;chi2new = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

    diff_theta = theta - theta_f

    ; account for any wrap-arounds from 360 deg to 0 deg

    for k=0, num-1 do begin
        if (diff_theta(k) ge !dpi) then $
          diff_theta(k) = diff_theta(k) - 2*!dpi
        if (diff_theta(k) le -!dpi) then $
          diff_theta(k) = 2*!dpi + diff_theta(k)	
    endfor

    chi2new = total((diff_theta)^2/dtheta^2 + (rho - rho_f)^2/drho^2)

    print,"chi2 of next modification:",chi2new

    if (chi2new le chi2old) then begin
        EL = ELadj
	chi2 = chi2new
        delta_chi = chi2old - chi2new
        lambda = lambda/10.0
        print, 'lambda', lambda
    endif else begin
	print, "next adjustment does not improve chi2"
        print, "increasing lambda by 10.0"
        lambda = lambda*10.0
        print, 'lambda', lambda
        delta_chi = 1.0 ; prevent exit when lambda is increased
    endelse

    count = count+1

    ; do not exit unless lambda is less than 0.001

    ; NOTE: This next step might not be necessary!
    if (lambda gt 0.001) and (delta_chi lt 0.001) then delta_chi = 1.0
;    print,'lambda',lambda
;    print,'delta_chi',delta_chi

endwhile

lambda = 0.0

print, "Final values:"

period = EL(0)
Tperi = EL(1)
ecc = EL(2)
major =  EL(3)
inc = EL(4)
W_cap = EL(5)
w_low = EL(6)

calc_deriv_vb, EL, elfix, mfit, time, theta, rho, dtheta, drho, theta_f, rho_f, alpha, beta

invmat = invert(alpha, stat, /double)

;determine errors:

diff_theta = theta - theta_f

; account for any wrap-arounds from 360 deg to 0 deg

for k=0, num-1 do begin
if (diff_theta(k) ge !dpi) then $
	diff_theta(k) = diff_theta(k) - 2*!dpi
if (diff_theta(k) le -!dpi) then $
	diff_theta(k) = 2*!dpi + diff_theta(k)	
endfor

chi2rt = total((diff_theta)^2/dtheta^2 + (rho - rho_f)^2/drho^2)

; convert data points to x and y coordinates
xfit = rho_f * cos(theta_f)	; x coordinate
yfit = rho_f * sin(theta_f)	; y coordinate
chi2xy = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

; degrees of freedom
dof = 2d*num - mfit

; reduced chi squared:
chi2red = chi2rt/dof

ELerr = dblarr(mfit)
for i=0,mfit-1 do ELerr(i) = sqrt(chi2red)*sqrt(invmat(i,i))

k=0
print,"Orbital Element     Error"
for i=0, 3 do begin
    if (elfix(i) eq 0) then begin
        print, elLabel(i), ': ', El(i)
    endif else begin
        print, elLabel(i), ': ', El(i), ELerr(k)
        k=k+1
    endelse
endfor
for i=4, nEl-1 do begin
     if (elfix(i) eq 0) then begin
         print, elLabel(i), ': ', El(i)*180/!dpi
     endif else begin
         print, elLabel(i), ': ', El(i)*180/!dpi, ELerr(k)*180/!dpi
         k = k+1
     endelse
endfor
print, "chi2(PA,sep):", chi2rt
print, "chi2red:", chi2red
print, "chi2(x,y):", chi2xy

print,'Number of iterations:',count
;print,'Final chi2:',chi2

print,EL(0),EL(1),EL(2),EL(3),EL(4)*180/!dpi,EL(5)*180/!dpi,EL(6)*180/!dpi

; plot best-fit orbit and data

tnum=1000.0
tstep = period/tnum
tmin = Tperi

tarr = findgen(tnum)*tstep + tmin

calc_vbfit, EL, tarr, theta_mod, rho_mod

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

calc_vbfit, EL, time, theta_f, rho_f

xfit = rho_f * cos(theta_f)	; x coordinate
yfit = rho_f * sin(theta_f)	; y coordinate

; plot data and best-fit positions
plot,-yarr,xarr,psym=6
oplot,-yfit,xfit,psym=7
for l=0, num-1 do oplot,[-yarr(l),-yfit(l)],[xarr(l),xfit(l)]

; plot best-fit orbit
oplot,-ymod,xmod

; Define user symbol to be closed circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)

; Plot measured positions as filled circles

usersym, frac*cos(Z), frac*sin(Z),/fill

;sepmax = 1.2*max(abs([yarr,xarr]))
sepmax = 1.05*max(abs([yarr,xarr,ymod,xmod]))

set_plot, 'ps'
device,/Color,Bits_per_pixel=8,filename='temp_vb.eps',xsize=12,ysize=12,isolatin1=1,/encap

ploterror,yarr,xarr,dy,dx,xtitle='!9D!3RA (mas)',ytitle='!9D!3DEC (mas)',ISOTROPIC=1, $
     xrange=[sepmax,-sepmax], yrange=[-sepmax,sepmax], $
     xstyle=1,ystyle=1, /nodata, position=[0.19,0.12,0.95,0.98]

; Connected measured and fit values
for i=0,num-1 do oplot,[yfit(i),yarr(i)],[xfit(i),xarr(i)]

; Mark computed location with x
;oplot,[yfit],[xfit],psym=7

                            ; mark primary with an asterisk
oplot,[0],[0],psym=2
oplot,[-sepmax,sepmax],[0,0],linestyle=1
oplot,[0,0],[-sepmax,sepmax],linestyle=1

; plot best-fit orbit
oplot,ymod,xmod,color=3

; Plot measured positions
;Define the symbol to be a closed circle
usersym, frac*cos(Z), frac*sin(Z), /fill
oplot,yarr,xarr,psym=8

device,/close
set_plot, 'x'

xOC = xarr - xfit
yOC = yarr - yfit

openw,lun,'xyresiduals'
printf,lun,"O-C residuals (data - fit)"
printf,lun,"date        dx     xO-C    xOC/dx  dy     yO-C    yOC/dy"
;printf,lun,"date   dx   xdata - xfit   xOC/dx   dy   ydata - yfit   yOC/dy"
for i=0, num-1 do printf,lun,format='(f10.4,f7.3,f8.3,f8.3,f7.3,f8.3,f8.3)', $
	time(i), dx(i), xOC(i), xOC(i)/dx(i), dy(i), yOC(i), yOC(i)/dy(i)
close,lun

free_lun,lun

end
