pro plot_PaeM, data_file, num, pte_file, fnum

; Make colored plots for thesis
;
; Visualize parameter space: 
;	e vs log(P) 
;	log(a) vs log(P) 
;	M vs e 
;	M vs log(P) 
;
; Color scheme: red   = 1-sigma (Deltachi2 = 1)
;		blue  = 2-sigma (Deltachi2 = 4)
;		green = 3-sigma (Deltachi2 = 9)
;
; Computes statistics for orbital elements
;
; Modification of LEAST_PTE.PRO
;
; INPUT:
; 	data_file: name of file containing the observations
;		format: time PA ePA sep esep
;	num: number of lines in data_file
;	pte_file: name of file containing P,T,e,chi2 for possible solutions
;		format: P T e chi2
;	fnum: number of lines in pte_file
;
; PROMPTED INPUTS:
;	system parallax (in mas)
;
; PROCEDURE:
; 	performs least squares fit
; 	computes a,i,Omega,omega,M for each P,T,e set
;	computes min, max, mean, median, and stdev for each orbital element
;
; OUTPUT: 
;	temp.ps: 4-panel colored plots of cross-cuts through M,P,a,e
;            parameter spaces

close,1
close,2
close,3

!P.charsize=1.0
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0


;!P.charsize=1.0
;!P.charthick=3.0
;!P.thick=4.0
;!X.thick=4.0
;!Y.thick=4.0

; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1

; Read in pte file names

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time = dblarr(num)
theta = dblarr(num)
rho = dblarr(num)
dtheta = dblarr(num)
drho = dblarr(num)

openr,3,data_file

for i=0, num-1 do begin

	readf, 3, temp1, temp2, temp3, temp4, temp5

	time(i) = temp1
	theta(i) = temp2
	dtheta(i) = temp3
	rho(i) = temp4;*1000.0	
	drho(i) = temp5;*1000.0
	
endfor

close,3

openr,1,pte_file

; convert theta and dtheta to radians

theta = theta*!dpi/180
dtheta = dtheta*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(theta)^2*drho^2 + rho^2*sin(theta)^2*dtheta^2)
dy = sqrt(sin(theta)^2*drho^2 + rho^2*cos(theta)^2*dtheta^2)

period = 0d
Tperi = 0d
ecc = 0d
xchi = 0d

massi = dblarr(fnum)
a_arr = dblarr(fnum)
e_arr = dblarr(fnum)
i_arr = dblarr(fnum)
P_arr = dblarr(fnum)
T_arr = dblarr(fnum)
W_cap_arr = dblarr(fnum)
w_low_arr = dblarr(fnum)
chi_arr = dblarr(fnum)

parallax = 0.0d

print, 'Enter the parallax of the system (in same units as separation):'
print,'(distance of 140, parallax = 7.14 mas)'
read, parallax

ansyd=' '
print,'Is period in years or days (y/d)?'
read,ansyd

if (ansyd eq 'y') then yd = 1.0d
if (ansyd eq 'd') then yd = 1.0d/365.25

print, 'Computing masses...'

for j=0, fnum-1 do begin

	; Obtain initial values for P,T,e

;	print,"Enter P T e (period, time of periastron passage, eccentricity):"
	readf, 1, period, Tperi, ecc, xchi

	; degrees of freedom
	dof = 2*num - 7

	; Determine fitted points (xfit,yfit) by computing the Thiele Innes
	; elements A,F,B,G

	calc_AFBG, time, xarr, dx, yarr, dy, period, Tperi, ecc, $
		   A_cap, F_cap, B_cap, G_cap, xfit, yfit

	; Determine chi squared

	chi2 = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

	; reduced chi squared:
	chi2red = chi2/dof

	; determine a,i,Omega,omega (using AIWW.PRO)

	aiWw, A_cap,B_cap,F_cap,G_cap,major,inc,W_cap,w_low

;	print, "a i Omega omega:", major, (180/!dpi)*inc, $
;	(180/!dpi)*W_cap, (180/!dpi)*w_low

	EL = [period,Tperi,ecc,major,inc,W_cap,w_low]

	; calculate mass of system using distance of 140 pc
	; parallax = 7.14 mas
	
	mass = major^3/parallax^3/(yd*period)^2

	massi(j) = mass
	a_arr(j) = major
	e_arr(j) = ecc
	i_arr(j) = 180*inc/!dpi
	P_arr(j) = period
	T_arr(j) = Tperi
	W_cap_arr(j) = 180*W_cap/!dpi
	w_low_arr(j) = 180*w_low/!dpi
;	chi_arr(j) = chi2
	chi_arr(j) = xchi

endfor

close,1

print,"Error estimates: min max mean median stddev"
print,format='(%"M:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(massi),max(massi),mean(massi),median(massi),stddev(massi)
print,format='(%"P:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(P_arr),max(P_arr),mean(P_arr),median(P_arr),stddev(P_arr)
print,format='(%"a:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(a_arr),max(a_arr),mean(a_arr),median(a_arr),stddev(a_arr)
print,format='(%"e:",f10.6,f10.6,f10.6,f10.6,f10.6)', $
	min(e_arr),max(e_arr),mean(e_arr),median(e_arr),stddev(e_arr)
print,format='(%"i:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(i_arr),max(i_arr),mean(i_arr),median(i_arr),stddev(i_arr)
print,format='(%"T:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(T_arr),max(T_arr),mean(T_arr),median(T_arr),stddev(T_arr)
print,format='(%"W:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(W_cap_arr),max(W_cap_arr),mean(W_cap_arr), $
	median(W_cap_arr),stddev(W_cap_arr)
print,format='(%"w:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(w_low_arr),max(w_low_arr),mean(w_low_arr), $
	median(w_low_arr),stddev(w_low_arr)

Mmin = 0.0
Mmax = 0.0

;print,'Enter min max of M range for plot:'
;read,Mmin,Mmax

Pmin = min(P_arr)
Pmax = max(P_arr)
;print,'Enter max P for plotting range'
;read,Pmax
;print,'Enter min P for plotting range'
;read,Pmin

Pmin = alog10(Pmin)
Pmax = alog10(Pmax)

logP = alog10(P_arr)
loga = alog10(a_arr)
logM = alog10(massi)

; Separate solutions into 1-, 2-, and 3- sigma intervals

chimin = min(chi_arr)

print,'Minimum chi2:',chimin

sig1 = where(chi_arr le (chimin+1.0), count1)
sig2 = where(chi_arr gt (chimin+1.0) and chi_arr le (chimin+4.0), count2)
sig3 = where(chi_arr gt (chimin+4.0) and chi_arr le (chimin+9.0), count3)

; Define user symbol to be closed circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)

fracsmall = 0.25

; Plot orbits as small filled circles
usersym, fracsmall*cos(Z), fracsmall*sin(Z), /fill

;!P.MULTI = [0,2,2]
!P.charsize=1.5
!P.MULTI = [0,2,3]
;!P.charsize=1.8
;!P.MULTI = [0,3,1]
;Save plots to file temp.ps


set_plot, 'ps'
;device, filename='temp.ps',xsize=18,ysize=13
;device, /Color, filename='temp.ps',xsize=15,ysize=13.5
;device, /Color, filename='temp.ps',xsize=15,ysize=13.5
;device, /Color, filename='temp.ps',xsize=14,ysize=20
device, /Color, filename='temp.ps',xsize=14,ysize=20
;device, /ENCAPSULATED, /Color, filename='temp.eps',xsize=14,ysize=20
;device, /ENCAPSULATED, /Color, filename='temp.eps',xsize=15,ysize=13.5
;device, /ENCAPSULATED, /Color, filename='temp.eps',xsize=18,ysize=6  ; 3x1

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;plot, logP, e_arr, psym=3, xtitle="log(P) (yrs)", ytitle="e", /nodata, $
;  xstyle=1, ystyle=1, xrange=[1.0,2.7],yrange=[0,1]
;oplot, logP(sig3), e_arr(sig3), psym=8, color=2
;oplot, logP(sig2), e_arr(sig2), psym=8, color=3
;oplot, logP(sig1), e_arr(sig1), psym=8, color=1
;
;plot, logP, loga, psym=3, xtitle="log(P) (yrs)",ytitle="log(a) (mas)", /nodata, $
;  xstyle=1, ystyle=1, xrange=[1.0,2.7],yrange=[1.3,3.0]
;;plot, logP, loga, psym=3, xtitle="log(P) (yrs)",ytitle="log(a) (arcsec)", /nodata, $
;;  xstyle=1, ystyle=1, xrange=[1.0,2.7],yrange=[-0.6,0.5]
;;  xstyle=1, ystyle=1, xrange=[1.0,2.7],yrange=[0.0,2.0]
;oplot, logP(sig3), loga(sig3), psym=8, color=2
;oplot, logP(sig2), loga(sig2), psym=8, color=3
;oplot, logP(sig1), loga(sig1), psym=8, color=1
;
;plot, e_arr, logM, psym=3, xtitle="e",ytitle="log(M) (M!9!dn!3!n)", /nodata, $
;  xstyle=1, ystyle=1,xrange=[0,1],yrange=[-0.5,1.5]
;oplot, e_arr(sig3), logM(sig3), psym=8, color=2
;oplot, e_arr(sig2), logM(sig2), psym=8, color=3
;oplot, e_arr(sig1), logM(sig1), psym=8, color=1
;
;plot, logP, logM, psym=3, xtitle="log(P) (yrs)",ytitle="log(M) (M!9!dn!3!n)", /nodata, $
;  xstyle=1, ystyle=1, xrange=[1.0,2.7],yrange=[-0.5,1.5]
;oplot, logP(sig3), logM(sig3), psym=8, color=2
;oplot, logP(sig2), logM(sig2), psym=8, color=3
;oplot, logP(sig1), logM(sig1), psym=8, color=1
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;plot,loga,e_arr,psym=3,xtitle="log(a) (mas)",ytitle="e"
;plot,loga,logM,psym=3,xtitle="log(a) (mas)",ytitle="log(M) (Msun)"

;plot, e_arr, massi, psym=3, xtitle="e",ytitle="M (M!9!dn!3!n)", /nodata, $
;  xstyle=1, ystyle=1,xrange=[0,1],yrange=[0.0,20.0]
;oplot, e_arr(sig3), massi(sig3), psym=8, color=2
;oplot, e_arr(sig2), massi(sig2), psym=8, color=3
;oplot, e_arr(sig1), massi(sig1), psym=8, color=1
;;
;
;plot, P_arr, massi, psym=3, xtitle="P (yrs)",ytitle="M (M!9!dn!3!n)", /nodata, $
;;  xstyle=1, ystyle=1, xrange=[1.0,2.7],yrange=[-0.5,1.5]
;  xstyle=1, ystyle=1, xrange=[0.0,100],yrange=[0.0,20.0]
;oplot, P_arr(sig3), massi(sig3), psym=8, color=2
;oplot, P_arr(sig2), massi(sig2), psym=8, color=3
;oplot, P_arr(sig1), massi(sig1), psym=8, color=1


Mmax = max(massi)
print,'Enter maximum mass for chi2 plots:'
read,Mmax

Mrange = Mmax - min(massi)
Mmin = min(massi)   ;round(10.0*(min(massi) - 0.05*Mmax))/10.0

;amax = max(a_arr)
;indm = where(massi le Mmax)
;amax = max(a_arr(indm))

Prange = max(P_arr) - min(P_arr)
Pmin = min(P_arr)- 0.05*Prange   ;round(min(P_arr) - 0.05*Prange)
Pmax = max(P_arr)+ 0.05*Prange   ;round(max(P_arr) + 0.05*Prange)

erange = max(e_arr) - min(e_arr)
emin = min(e_arr) - 0.05*erange   ;round(100.0*(min(e_arr) - 0.05*erange))/100.0
emax = max(e_arr) + 0.05*erange   ;round(100.0*(max(e_arr) + 0.05*erange))/100.0

arange = max(a_arr) - min(a_arr)
amin = min(a_arr) - 0.05*arange   ;round(min(a_arr) - 0.05*arange)
amax = max(a_arr) + 0.05*arange   ;round(max(a_arr) + 0.05*arange)

irange = max(i_arr) - min(i_arr)
imin = min(i_arr) - 0.05*irange   ;round(min(i_arr) - 0.05*irange)
imax = max(i_arr) + 0.05*irange   ;round(max(i_arr) + 0.05*irange)

Wcrange = max(W_cap_arr) - min(W_cap_arr)
Wcmin = min(W_cap_arr) - 0.05*Wcrange   ;round(min(i_arr) - 0.05*irange)
Wcmax = max(W_cap_arr) + 0.05*Wcrange   ;round(max(i_arr) + 0.05*irange)


print,'M and a are cut-off in chi2 plots'

; For T Tau 3-panel plot
;emin = 0.1
;emax= 0.8
;Pmin=20
;Pmax=55
;Mmin = 2.4
;Mmax=5.0
;amin=80
;amax = 140

plot, P_arr, e_arr, psym=3, xtitle="P (yrs)", ytitle="e", /nodata, $
  xstyle=1, ystyle=1, xrange=[Pmin,Pmax],yrange=[emin,emax];, $
;  position=[0.07,0.17,0.323,0.98]
if (count3 gt 0) then oplot, P_arr(sig3), e_arr(sig3), psym=8, color=2
if (count2 gt 0) then oplot, P_arr(sig2), e_arr(sig2), psym=8, color=3
if (count1 gt 0) then oplot, P_arr(sig1), e_arr(sig1), psym=8, color=1

plot, P_arr, a_arr, psym=3, xtitle="P (yrs)",ytitle="a (mas)", /nodata, $
  xstyle=1, ystyle=1, xrange=[Pmin,Pmax],yrange=[amin,amax];, $
;  position=[0.404,0.17,0.657,0.98]
if (count3 gt 0) then oplot, P_arr(sig3), a_arr(sig3), psym=8, color=2
if (count2 gt 0) then oplot, P_arr(sig2), a_arr(sig2), psym=8, color=3
if (count1 gt 0) then oplot, P_arr(sig1), a_arr(sig1), psym=8, color=1

plot, P_arr, massi, psym=3, xtitle="P (yrs)",ytitle="M (M!9!dn!3!n)", /nodata, $
  xstyle=1, ystyle=1, xrange=[Pmin,Pmax],yrange=[Mmin,Mmax];, $
;  position=[0.737,0.17,0.99,0.98]
if (count3 gt 0) then oplot, P_arr(sig3), massi(sig3), psym=8, color=2
if (count2 gt 0) then oplot, P_arr(sig2), massi(sig2), psym=8, color=3
if (count1 gt 0) then oplot, P_arr(sig1), massi(sig1), psym=8, color=1

plot, i_arr, massi, psym=3, xtitle="i (deg)",ytitle="M (M!9!dn!3!n)", /nodata, $
  xstyle=1, ystyle=1,xrange=[imin,imax],yrange=[Mmin,Mmax]
if (count3 gt 0) then oplot, i_arr(sig3), massi(sig3), psym=8, color=2
if (count2 gt 0) then oplot, i_arr(sig2), massi(sig2), psym=8, color=3
if (count1 gt 0) then oplot, i_arr(sig1), massi(sig1), psym=8, color=1

plot, e_arr, massi, psym=3, xtitle="e",ytitle="M (M!9!dn!3!n)", /nodata, $
  xstyle=1, ystyle=1, xrange=[emin,emax],yrange=[Mmin,Mmax]
if (count3 gt 0) then oplot, e_arr(sig3), massi(sig3), psym=8, color=2
if (count2 gt 0) then oplot, e_arr(sig2), massi(sig2), psym=8, color=3
if (count1 gt 0) then oplot, e_arr(sig1), massi(sig1), psym=8, color=1

plot, a_arr, i_arr, psym=3, xtitle="a (mas)",ytitle="i (deg)", /nodata, $
  xstyle=1, ystyle=1, yrange=[imin,imax],xrange=[amin,amax]
if (count3 gt 0) then oplot, a_arr(sig3), i_arr(sig3), psym=8, color=2
if (count2 gt 0) then oplot, a_arr(sig2), i_arr(sig2), psym=8, color=3
if (count1 gt 0) then oplot, a_arr(sig1), i_arr(sig1), psym=8, color=1

device, /close
set_plot,'x'

!P.charsize=1.5
!P.MULTI = [0,3,1]

!P.font=0
!P.thick=2
!P.charthick=2.0
!P.charsize=1.6

set_plot, 'ps'
device, /Color, filename='temp_orb.eps',xsize=30,ysize=10,/encap


plot, P_arr, e_arr, psym=3, xtitle="P (yrs)", ytitle="e", /nodata, $
  xstyle=1, ystyle=1, xrange=[Pmin,Pmax],yrange=[emin,emax], $
  position=[0.05,0.12,0.33,0.98]
if (count3 gt 0) then oplot, P_arr(sig3), e_arr(sig3), psym=8, color=2
if (count2 gt 0) then oplot, P_arr(sig2), e_arr(sig2), psym=8, color=3
if (count1 gt 0) then oplot, P_arr(sig1), e_arr(sig1), psym=8, color=1

plot, P_arr, a_arr, psym=3, xtitle="P (yrs)",ytitle="a (mas)", /nodata, $
  xstyle=1, ystyle=1, xrange=[Pmin,Pmax],yrange=[amin,amax], $
  position=[0.38,0.12,0.66,0.98]
if (count3 gt 0) then oplot, P_arr(sig3), a_arr(sig3), psym=8, color=2
if (count2 gt 0) then oplot, P_arr(sig2), a_arr(sig2), psym=8, color=3
if (count1 gt 0) then oplot, P_arr(sig1), a_arr(sig1), psym=8, color=1

plot, a_arr, i_arr, psym=3, xtitle="a (mas)",ytitle="i (deg)", /nodata, $
  xstyle=1, ystyle=1, yrange=[imin,imax],xrange=[amin,amax], $
  position=[0.71,0.12,0.99,0.98]
if (count3 gt 0) then oplot, a_arr(sig3), i_arr(sig3), psym=8, color=2
if (count2 gt 0) then oplot, a_arr(sig2), i_arr(sig2), psym=8, color=3
if (count1 gt 0) then oplot, a_arr(sig1), i_arr(sig1), psym=8, color=1

;plot, W_cap_arr, i_arr, psym=3, xtitle="Omega (deg)",ytitle="i (deg)", /nodata, $
;  xstyle=1, ystyle=1, yrange=[imin,imax],xrange=[Wcmin,Wcmax]
;if (count3 gt 0) then oplot, W_cap_arr(sig3), i_arr(sig3), psym=8, color=2
;if (count2 gt 0) then oplot, W_cap_arr(sig2), i_arr(sig2), psym=8, color=3
;if (count1 gt 0) then oplot, W_cap_arr(sig1), i_arr(sig1), psym=8, color=1

device, /close
set_plot,'x'


!P.charsize=1.0
!P.MULTI = [0,0,0]

set_plot, 'ps'
;device, filename='temp.ps',xsize=18,ysize=13
device, /Color, filename='tempip.eps',xsize=15,ysize=15,/encap

plot, P_arr, i_arr, psym=3, xtitle="P (yr)",ytitle="i (deg)", /nodata, $
  xstyle=1, ystyle=1, yrange=[imin,imax],xrange=[Pmin,Pmax];, $
  ;position=[0.71,0.12,0.99,0.98]
if (count3 gt 0) then oplot, P_arr(sig3), i_arr(sig3), psym=8, color=2
if (count2 gt 0) then oplot, P_arr(sig2), i_arr(sig2), psym=8, color=3
if (count1 gt 0) then oplot, P_arr(sig1), i_arr(sig1), psym=8, color=1

device, /close
set_plot,'x'


!P.charsize=1.0
!P.MULTI = [0,0,0]

set_plot, 'ps'
;device, filename='temp.ps',xsize=18,ysize=13
device, /Color, filename='tempai.ps',xsize=15,ysize=15

plot, loga, i_arr, psym=3, xtitle="log(a) (mas)", ytitle="i (deg)", /nodata 
if (count3 gt 0) then oplot, loga(sig3), i_arr(sig3), psym=8, color=2
if (count2 gt 0) then oplot, loga(sig2), i_arr(sig2), psym=8, color=3
if (count1 gt 0) then oplot, loga(sig1), i_arr(sig1), psym=8, color=1

device, /close
set_plot,'x'

!P.font=0
!P.charsize=1.0

set_plot, 'ps'
device, /Color, filename='tempiM.eps',xsize=15,ysize=15,/encap

plot, i_arr, massi, psym=3, xtitle="i (deg)",ytitle="Total Mass (Msun)", /nodata, $
  xstyle=1, ystyle=1,xrange=[imin,imax],yrange=[Mmin,Mmax]
if (count3 gt 0) then oplot, i_arr(sig3), massi(sig3), psym=8, color=2
if (count2 gt 0) then oplot, i_arr(sig2), massi(sig2), psym=8, color=3
if (count1 gt 0) then oplot, i_arr(sig1), massi(sig1), psym=8, color=1

;plot, i_arr, logM, psym=3, xtitle="i (deg)", ytitle="log(M) (Msun)", /nodata 
;if (count3 gt 0) then oplot, i_arr(sig3), logM(sig3), psym=8, color=2
;if (count2 gt 0) then oplot, i_arr(sig2), logM(sig2), psym=8, color=3
;if (count1 gt 0) then oplot, i_arr(sig1), logM(sig1), psym=8, color=1

device, /close
set_plot,'x'


set_plot, 'ps'
device, /Color, filename='tempeM.eps',xsize=15,ysize=15,/encap

plot, e_arr, massi, psym=3, xtitle="e",ytitle="Total Mass (Msun)", /nodata, $
  xstyle=1, ystyle=1, xrange=[emin,emax],yrange=[Mmin,Mmax]
if (count3 gt 0) then oplot, e_arr(sig3), massi(sig3), psym=8, color=2
if (count2 gt 0) then oplot, e_arr(sig2), massi(sig2), psym=8, color=3
if (count1 gt 0) then oplot, e_arr(sig1), massi(sig1), psym=8, color=1


device, /close
set_plot,'x'

!P.font=0
!P.thick=2
!P.charthick=2.0
!P.charsize=1.2

set_plot, 'ps'
device, /Color, filename='temp_aP.eps',xsize=15,ysize=15,/encap

;amin=1.5
;amax=5.0
;Pmin=0.94
;Pmax=1.09

fracmed=0.4
usersym, fracmed*cos(Z), fracmed*sin(Z), /fill

plot, P_arr, a_arr, psym=3, xtitle="P (yrs)",ytitle="a (mas)", /nodata, $
  xstyle=1, ystyle=1, xrange=[Pmin,Pmax],yrange=[amin,amax], $
  position=[0.12,0.1,0.98,0.98]
if (count3 gt 0) then oplot, P_arr(sig3), a_arr(sig3), psym=8, color=2
if (count2 gt 0) then oplot, P_arr(sig2), a_arr(sig2), psym=8, color=3
if (count1 gt 0) then oplot, P_arr(sig1), a_arr(sig1), psym=8, color=1

device, /close
set_plot,'x'


; Make histograms

!P.font=0
!P.multi=0
!P.charsize=1.5
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.0	;scale factor for adjusting size of plotting symbols


elLabel = ['P','T','e','a','i','Omega','omega']

nbin = 20

histo_P = histogram(P_arr, locations=P_bin, nbins=nbin)
histo_T = histogram(T_arr, locations=T_bin, nbins=nbin)
histo_e = histogram(e_arr, locations=e_bin, nbins=nbin)
histo_a = histogram(a_arr, locations=a_bin, nbins=nbin)
histo_i = histogram(i_arr, locations=i_bin, nbins=nbin)
histo_Wcap = histogram(W_cap_arr, locations=Wcap_bin, nbins=nbin)
histo_wlow = histogram(w_low_arr, locations=wlow_bin, nbins=nbin)

set_plot, 'ps'
device, /Color, Bits_per_pixel=8, /encap, filename='temp_histo.eps',xsize=30,ysize=15
;device, /Color, Bits_per_pixel=8, filename='temp_histo.ps',xsize=15,ysize=15

!P.multi=[0,4,2]

plot,P_bin,histo_P,psym=10,xtitle=ellabel(0),ytitle='Number of Solutions'
plot,T_bin,histo_T,psym=10,xtitle=ellabel(1),ytitle='Number of Solutions'
plot,e_bin,histo_e,psym=10,xtitle=ellabel(2),ytitle='Number of Solutions'
plot,a_bin,histo_a,psym=10,xtitle=ellabel(3),ytitle='Number of Solutions'
plot,i_bin,histo_i,psym=10,xtitle=ellabel(4),ytitle='Number of Solutions'
plot,Wcap_bin,histo_Wcap,psym=10,xtitle=ellabel(5),ytitle='Number of Solutions'
plot,wlow_bin,histo_wlow,psym=10,xtitle=ellabel(6),ytitle='Number of Solutions'

device,/close
set_plot, 'x'



!P.font=1
!P.MULTI = [0,0,0]
!P.charsize=1.0

print,'1-sigma Error estimates: min max mean median stddev'
;print,format='(%"P:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
print,format='(%"P:",f10.2,f10.2,f10.2,f10.2,f10.2)', $
	min(P_arr(sig1)),max(P_arr(sig1)),mean(P_arr(sig1)),median(P_arr(sig1)),stddev(P_arr(sig1))
print,format='(%"T:",f10.2,f10.2,f10.2,f10.2,f10.2)', $
	min(T_arr(sig1)),max(T_arr(sig1)),mean(T_arr(sig1)),median(T_arr(sig1)),stddev(T_arr(sig1))
print,format='(%"e:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(e_arr(sig1)),max(e_arr(sig1)),mean(e_arr(sig1)),median(e_arr(sig1)),stddev(e_arr(sig1))
print,format='(%"a:",f10.2,f10.2,f10.2,f10.2,f10.2)', $
	min(a_arr(sig1)),max(a_arr(sig1)),mean(a_arr(sig1)),median(a_arr(sig1)),stddev(a_arr(sig1))
print,format='(%"i:",f10.2,f10.2,f10.2,f10.2,f10.2)', $
	min(i_arr(sig1)),max(i_arr(sig1)),mean(i_arr(sig1)),median(i_arr(sig1)),stddev(i_arr(sig1))
print,format='(%"W:",f10.2,f10.2,f10.2,f10.2,f10.2)', $
	min(W_cap_arr(sig1)),max(W_cap_arr(sig1)),mean(W_cap_arr(sig1)), $
	median(W_cap_arr(sig1)),stddev(W_cap_arr(sig1))
print,format='(%"w:",f10.2,f10.2,f10.2,f10.2,f10.2)', $
	min(w_low_arr(sig1)),max(w_low_arr(sig1)),mean(w_low_arr(sig1)), $
	median(w_low_arr(sig1)),stddev(w_low_arr(sig1))
print,format='(%"M:",f10.2,f10.2,f10.2,f10.2,f10.2)', $
	min(massi(sig1)),max(massi(sig1)),mean(massi(sig1)),median(massi(sig1)),stddev(massi(sig1))

sigbest = where(chi_arr le min(chi_arr))
sigbest = sigbest(0)

print,'1-sigma confidence intervals:'
print,'P:',P_arr(sigbest),' + ',max(P_arr(sig1)) - P_arr(sigbest),' - ',P_arr(sigbest) - min(P_arr(sig1))
print,'T:',T_arr(sigbest),' + ',max(T_arr(sig1)) - T_arr(sigbest),' - ',T_arr(sigbest) - min(T_arr(sig1))
print,'e:',e_arr(sigbest),' + ',max(e_arr(sig1)) - e_arr(sigbest),' - ',e_arr(sigbest) - min(e_arr(sig1))
print,'a:',a_arr(sigbest),' + ',max(a_arr(sig1)) - a_arr(sigbest),' - ',a_arr(sigbest) - min(a_arr(sig1))
print,'i:',i_arr(sigbest),' + ',max(i_arr(sig1)) - i_arr(sigbest),' - ',i_arr(sigbest) - min(i_arr(sig1))
print,'Omega:',W_cap_arr(sigbest),' + ',max(W_cap_arr(sig1)) - W_cap_arr(sigbest),' - ',W_cap_arr(sigbest) - min(W_cap_arr(sig1))
print,'omega:',w_low_arr(sigbest),' + ',max(w_low_arr(sig1)) - w_low_arr(sigbest),' - ',w_low_arr(sigbest) - min(w_low_arr(sig1))
print,'Mtot:',massi(sigbest),' + ',max(massi(sig1)) - massi(sigbest),' - ',massi(sigbest) - min(massi(sig1))


end

