pro plot_Pae_weight, data_file, num, pte_file, fnum

; Weight each value in histogram by the chi2 probability.  Output is then
; parameter vs. weighted number of solutions

; Computes statistics for orbital elements
; Make weighted histogram of mass
;
; Modification of LEAST_PTE.PRO
;
; INPUT:
; 	data_file: name of file containing the observations
;		format: time PA ePA sep esep
;	num: number of lines in data_file
;	pte_file: name of file containing P,T,e,chi2 for possible solutions
;		format: P T e chi2
;	fnum: number of lines in pte_file
;
; PROMPTED INPUTS:
;	system parallax (in mas)
;	step sizes and ranges for histograms
;
; PROCEDURE:
; 	performs least squares fit
; 	computes a,i,Omega,omega,M for each P,T,e set
;	computes min, max, mean, median, and stdev for each orbital element
;	computes frequency distribution and histograms for M
;	computes mass errors based on 34% of sample above and below median
;
; OUTPUT: 
;	tempM.ps: weighted mass distribution histogram
;       tempchi.ps: chi2 vs. mass surface
;       tempprob.ps: chi2 probability vs. chi2
;	result_errors: file of statistics for orbital elements
;		min, max, mean, median, and stdev for each element
;		error estimates for mass (+/-34% of sample within median)
;		mass distribution used to calculate errors
;
; 7 October 2003: modified to calculate mass errors based on mass distribution
; 	within the program
; 13 April 2006:  read chi2 in from data file rather than 
;     re-computing in data file.  Reason being: changed
;     hartkopf_mc.pro to scale chi2 (don't need to do that
;     here - would require inputing best-fit solution)
;
; 10 December 2007: Add subroutine compute_histo_uncty.pro that computes
;   median and uncertainties by sorting the arrays, rather than using
;   step sizes to bin the arrays.  Old version saved as plot_Pae_weight_v2.pro


!P.charsize=1.5
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.2

;!P.charsize=1.1
;!P.charthick=4.0
!P.charsize=1.5
!P.charthick=3.0
!P.thick=4.0
!X.thick=4.0
!Y.thick=4.0
frac=1.2

; Read in pte file names

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time = dblarr(num)
theta = dblarr(num)
rho = dblarr(num)
dtheta = dblarr(num)
drho = dblarr(num)

openr,3,data_file

for i=0, num-1 do begin

	readf, 3, temp1, temp2, temp3, temp4, temp5

	time(i) = temp1
	theta(i) = temp2
	dtheta(i) = temp3
	rho(i) = temp4	
	drho(i) = temp5
	
endfor

close,3

openr,1,pte_file
openw,2,"result_all"
printf,2, "P T e a i Omega omega mass chi2:"

; convert theta and dtheta to radians

theta = theta*!dpi/180
dtheta = dtheta*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(theta)^2*drho^2 + rho^2*sin(theta)^2*dtheta^2)
dy = sqrt(sin(theta)^2*drho^2 + rho^2*cos(theta)^2*dtheta^2)

period = 0d
Tperi = 0d
ecc = 0d
xchi = 0d

massi = dblarr(fnum)
a_arr = dblarr(fnum)
e_arr = dblarr(fnum)
i_arr = dblarr(fnum)
P_arr = dblarr(fnum)
T_arr = dblarr(fnum)
W_cap_arr = dblarr(fnum)
w_low_arr = dblarr(fnum)
chi_arr = dblarr(fnum)

freepar = 1.0		;number of free parameters (up to 7)

; degrees of freedom
dof = 2.0*num - freepar
	
parallax = 0.0d

print, 'Enter the parallax of the system (in same units as separation):'
print,'(distance of 140, parallax = 7.14 mas)'
read, parallax

ansyd=' '
print,'Is period in years or days (y/d)?'
read,ansyd

if (ansyd eq 'y') then yd = 1.0d
if (ansyd eq 'd') then yd = 1.0d/365.25

print, 'Computing masses...'

for j=0, fnum-1 do begin

	; Obtain initial values for P,T,e

;	print,"Enter P T e (period, time of periastron passage, eccentricity):"
	readf, 1, period, Tperi, ecc, xchi

	; Determine fitted points (xfit,yfit) by computing the Thiele Innes
	; elements A,F,B,G

	calc_AFBG, time, xarr, dx, yarr, dy, period, Tperi, ecc, $
		   A_cap, F_cap, B_cap, G_cap, xfit, yfit

	; Determine chi squared

	chi2 = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

	; reduced chi squared:
	chi2red = chi2/dof

	; determine a,i,Omega,omega (using AIWW.PRO)

	aiWw, A_cap,B_cap,F_cap,G_cap,major,inc,W_cap,w_low

;	print, "a i Omega omega:", major, (180/!dpi)*inc, $
;	(180/!dpi)*W_cap, (180/!dpi)*w_low

	EL = [period,Tperi,ecc,major,inc,W_cap,w_low]

	; calculate mass of system using distance of 140 pc
	; parallax = 7.14 mas
	
	mass = major^3/parallax^3/(yd*period)^2

	massi(j) = mass
	a_arr(j) = major
	e_arr(j) = ecc
	i_arr(j) = 180*inc/!dpi
	P_arr(j) = period
	T_arr(j) = Tperi
	W_cap_arr(j) = 180*W_cap/!dpi
	w_low_arr(j) = 180*w_low/!dpi
	chi_arr(j) = xchi      ; read chi2 in from data file
;	chi_arr(j) = chi2

	printf,2,EL(0),El(1),EL(2),EL(3),180*EL(4)/!dpi,180*EL(5)/!dpi, $
		180*EL(6)/!dpi, mass, chi2

endfor

close,1
close,2
	
; restrict range of mass to be within Mass_min

Mass_min = 0d

print, "Enter upper bound for mass restriction"
read,Mass_min

massnum = where(massi le Mass_min, mcount)

mrestrict = dblarr(mcount)
Prestrict = dblarr(mcount)
arestrict = dblarr(mcount)
erestrict = dblarr(mcount)
irestrict = dblarr(mcount)
Trestrict = dblarr(mcount)
W_caprestrict = dblarr(mcount)
w_lowrestrict = dblarr(mcount)
chirestrict = dblarr(mcount)

n=0

for l=0, fnum-1 do begin

	if (massi(l) le Mass_min) then begin

		mrestrict(n) = massi(l)
		Prestrict(n) = P_arr(l)
		arestrict(n) = a_arr(l)
		erestrict(n) = e_arr(l)
		irestrict(n) = i_arr(l)
		Trestrict(n) = T_arr(l)
		W_caprestrict(n) = W_cap_arr(l)
		w_lowrestrict(n) = w_low_arr(l)
		chirestrict(n) = chi_arr(l)
		n=n+1
	endif
endfor

chiprob = dblarr(mcount)

; Chi2 probability distribution
; compute the probability of exceeding chi2

for i=0, mcount-1 do begin
chiprob(i) = 1.0 - chisqr_pdf(chirestrict(i),dof)
endfor

;Save mass histogram to file tempprob.ps
set_plot, 'ps'
device, filename='tempprob.ps',xsize=18,ysize=13

plot,chirestrict,chiprob,psym=3, xtitle='chi2', ytitle='prob(chi2,nu)', $
	title = 'nu = '+string(dof)

device, /close
set_plot,'x'

print,"Error estimates: min max mean median stddev"
print,format='(%"M:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(massi),max(massi),mean(massi),median(massi),stddev(massi)
print,format='(%"P:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(P_arr),max(P_arr),mean(P_arr),median(P_arr),stddev(P_arr)
print,format='(%"a:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(a_arr),max(a_arr),mean(a_arr),median(a_arr),stddev(a_arr)
print,format='(%"e:",f10.6,f10.6,f10.6,f10.6,f10.6)', $
	min(e_arr),max(e_arr),mean(e_arr),median(e_arr),stddev(e_arr)
print,format='(%"i:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(i_arr),max(i_arr),mean(i_arr),median(i_arr),stddev(i_arr)
print,format='(%"T:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(T_arr),max(T_arr),mean(T_arr),median(T_arr),stddev(T_arr)
print,format='(%"W:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(W_cap_arr),max(W_cap_arr),mean(W_cap_arr), $
	median(W_cap_arr),stddev(W_cap_arr)
print,format='(%"w:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(w_low_arr),max(w_low_arr),mean(w_low_arr), $
	median(w_low_arr),stddev(w_low_arr)

; convert Mass_min to a string and remove leading white space

Mass_value = string(Mass_min)

Mass_value = strtrim(Mass_min,1)

print,"Restrict mass to be below ", Mass_value, " Msun"
print,format='(%"M:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(mrestrict),max(mrestrict), mean(mrestrict), $
	median(mrestrict),stddev(mrestrict)
print,format='(%"P:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(Prestrict),max(Prestrict), mean(Prestrict), $
	median(Prestrict),stddev(Prestrict)
print,format='(%"a:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(arestrict),max(arestrict), mean(arestrict), $
	median(arestrict),stddev(arestrict)
print,format='(%"e:",f10.6,f10.6,f10.6,f10.6,f10.6)', $
	min(erestrict),max(erestrict), mean(erestrict), $
	median(erestrict),stddev(erestrict)
print,format='(%"i:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(irestrict),max(irestrict), mean(irestrict), $
	median(irestrict),stddev(irestrict)
print,format='(%"T:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(Trestrict),max(Trestrict),mean(Trestrict), $
	median(Trestrict),stddev(Trestrict)
print,format='(%"W:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(W_caprestrict),max(W_caprestrict),mean(W_caprestrict), $
	median(W_caprestrict),stddev(W_caprestrict)
print,format='(%"w:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(w_lowrestrict),max(w_lowrestrict),mean(w_lowrestrict), $
	median(w_lowrestrict),stddev(w_lowrestrict)


medM = median(mrestrict)
;medP = median(Prestrict)
;meda = median(arestrict)
;mede = median(erestrict)
;meani = mean(irestrict)

mrange = max(mrestrict) - min(mrestrict)
;Prange = max(Prestrict) - min(Prestrict)
;arange = max(arestrict) - min(arestrict)
;erange = max(erestrict) - min(erestrict)
;irange = max(irestrict) - min(irestrict)

; minimum of ranges (needed for creating histogram)

mmin = min(mrestrict) ;+ mrange/2d
;Pmin = min(Prestrict) ;+ Prange/2d
;amin = min(arestrict) ;+ arange/2d
;emin = min(erestrict) ;+ erange/2d
;imin = min(irestrict) ;+ irange/2d

mstep = 0d
;Pstep = 0d
;astep = 0d
;estep = 0d
;istep = 0d

print, "Enter step size for mass histogram:"
read, mstep
;print, "Enter step size for period histogram:"
;read, Pstep
;print, "Enter step size for a histogram:"
;read, astep
;print, "Enter step size for e histogram:"
;read, estep
;print, "Enter step size for i histogram:"
;read, istep

print, "Enter desired Mass minimum:", mmin
read,mmin
;print, "Enter desired P minimum:", Pmin
;read,Pmin
;print, "Enter desired a minimum:", amin
;read,amin
;print, "Enter desired e minimum:", emin
;read,emin
;print, "Enter desired i minimum:", imin
;read,imin

mnum = round(mrange/double(mstep)) + 2 ; number of mass steps searched
;Pnum = round(Prange/double(Pstep)) + 2
;anum = round(arange/double(astep)) + 2
;enum = round(erange/double(estep)) + 2
;inum = round(irange/double(istep)) + 2

print, 'Creating histograms...'

; Create histogram arrays 

mbins = dblarr(mnum)	; array of all possible masses searched
h_marr = dblarr(mnum)	; array to contain the number of values at each step
;Pbins = dblarr(Pnum)	
;h_Parr = intarr(Pnum)
;abins = dblarr(anum)	
;h_aarr = intarr(anum)
;ebins = dblarr(enum)	
;h_earr = intarr(enum)
;ibins = dblarr(inum)	
;h_iarr = intarr(inum)

create_histo_w, mmin, mrange, mstep, mrestrict, chiprob, mbins, h_marr

;create_histo, mmin, mrange, mstep, mrestrict, mbins, h_marr
;create_histo, Pmin, Prange, Pstep, Prestrict, Pbins, h_Parr
;create_histo, amin, arange, astep, arestrict, abins, h_aarr
;create_histo, emin, erange, estep, erestrict, ebins, h_earr
;create_histo, imin, irange, istep, irestrict, ibins, h_iarr

;print, 'Mass bin   Number in each bin'
;for i=0, mnum-1 do print, mbins(i), h_marr(i)

!P.MULTI = [0,3,2]
;;Save histograms to file temp.ps
;mydevice = !d.name
;set_plot, 'ps'
;device, filename='temp.ps',xsize=18,ysize=13
;
;plot,mbins,h_marr,psym=10,xtitle='Mass (Msun)'
;plot,Pbins,h_Parr,psym=10,xtitle='Period (years)'
;plot,abins,h_aarr,psym=10,xtitle='Semi-Major Axis (mas)'
;plot,ebins,h_earr,psym=10,xtitle='Eccentricity'
;plot,ibins,h_iarr,psym=10,xtitle='Inclination (deg)'
;
;device, /close
;set_plot,mydevice

!P.MULTI = [0,0,0]

;ptitle=' '
;print,'Enter plot title:'
;read,ptitle
;xloc = 0.0
;yloc = 0.0
;print,'Enter title location (x,y):'
;read,xloc,yloc

;Save mass histogram to file tempM.ps
mydevice = !d.name
set_plot, 'ps'
device, filename='tempM.ps',xsize=18,ysize=13
;device, filename='tempM.ps',xsize=12,ysize=8

plot,mbins,h_marr,psym=10,xtitle='Mass (M!9!dn!3!n)', $
 	ytitle="Weighted Number of Solutions", xrange=[0,Mass_min]

;xyouts,xloc,yloc,ptitle

device, /close
set_plot,mydevice



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 34% Uncertainty Intervals ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; no longer use step sizes in computing median and uncertainties
; just sort the arrays

; determine error: range of values (away from the median) that contains
; 2/3 of the sample (68%)
; i.e. 34% of total values on either side of median

; it seems that if we set median mass equal to best fitting, we
; reproduce the 1-sigma uncty's and don't take advantage of the 
; narrowly defined peak of the distribution

compute_histo_uncty, mrestrict, chiprob, medM, dM_low, dM_high

print, 'Number of solutions in restricted range:', mcount

;; Determine 95% probability upper mass limit
;
;;num95 = 0.05*mcount	; number of solutions included in 34% of total
;num95 = 0.05*sum_weight	; number of solutions included in 34% of total
;
;sum_95prob = 0.0
;
;arr_i = n_elements(h_marr)
;
;while(sum_95prob le num95) do begin
;
;	arr_i = arr_i - 1
;
;	sum_95prob = sum_95prob + h_marr(arr_i)
;
;endwhile
;
;m95prob = mbins(arr_i)

print,"Error range within +/-34% of median mass:"
print,"Median:", medM
print,'Errors:  +', dM_high 
print,'         -', dM_low
;print,"Total number of solutions:", sum_weight
;print,'34% of total number:', num34
;print,'Total number in upper error estimate:', sum_lower
;print,'Total number in lower error estimate:', sum_upper
;print, ' '

;print,'There is a 95% probability that the mass is below:',m95prob


; Compute uncertainties for inclination:


compute_histo_uncty, irestrict, chiprob, imed, di_med_low, di_med_high
compute_histo_uncty, irestrict, chiprob, imean, di_mean_low, di_mean_high,/mean_dist

print,"Error range within +/-34% of mean and median inclination:"
print,"Median:", imed,' + ',di_med_high,' - ',di_med_low
print,"Mean:", imean,' + ',di_mean_high,' - ',di_mean_low


;logP = alog10(P_arr)
;loga = alog10(a_arr)
;logM = alog10(massi)
;rlogP = alog10(Prestrict)
;rloga = alog10(arestrict)

;!P.MULTI = [0,3,2]
;Save histograms to file templog.ps

;mydevice = !d.name
;set_plot, 'ps'
;
;device, filename='templog.ps',xsize=18,ysize=13
;plot,logP,e_arr,psym=3,xtitle="log(P) (yrs)",ytitle="e" 
;plot,loga,e_arr,psym=3,xtitle="log(a) (mas)",ytitle="e"
;plot,logP,loga,psym=3,xtitle="log(P) (yrs)",ytitle="log(a) (mas)" 
;plot,e_arr,logM,psym=3,xtitle="e",ytitle="log(M) (Msun)"
;plot,logP,logM,psym=3,xtitle="log(P) (yrs)",ytitle="log(M) (Msun)" 
;plot,loga,logM,psym=3,xtitle="log(a) (mas)",ytitle="log(M) (Msun)"
;plot,e_arr,massi,psym=3,xtitle="e",ytitle="M (Msun)"
;plot,logP,massi,psym=3,xtitle="log(P) (yrs)",ytitle="M (Msun)" 
;plot,loga,massi,psym=3,xtitle="log(a) (mas)",ytitle="M (Msun)"

;plot,rlogP,erestrict,psym=3,xtitle="log(P) (yrs)",ytitle="e" 
;plot,rloga,erestrict,psym=3,xtitle="log(a) (mas)",ytitle="e"
;plot,rlogP,rloga,psym=3,xtitle="log(P) (yrs)",ytitle="log(a) (mas)"
;plot,e_arr,mrestrict,psym=3,xtitle="e",ytitle="M (Msun)"
;plot,rlogP,mrestrict,psym=3,xtitle="log(P) (yrs)",ytitle="M (Msun)"
;plot,rloga,mrestrict,psym=3,xtitle="log(a) (mas)",ytitle="M (Msun)"
;device, /close
;
;set_plot,mydevice
;
;!P.MULTI = [0,0,0]


openw,1,"result_errors"

printf,1,'Results from plot_Pae_weight.pro'
printf,1,'Data file name:', data_file
printf,1,'Number of lines in data_file:', num
printf,1,'PTe file name:', pte_file
printf,1,'Number of lines in pte_file:', fnum

printf,1,' '

printf,1,"Error estimates: min max mean median stddev"
printf,1,format='(%"M:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(massi),max(massi),mean(massi),median(massi),stddev(massi)
printf,1,format='(%"P:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(P_arr),max(P_arr),mean(P_arr),median(P_arr),stddev(P_arr)
printf,1,format='(%"a:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(a_arr),max(a_arr),mean(a_arr),median(a_arr),stddev(a_arr)
printf,1,format='(%"e:",f10.6,f10.6,f10.6,f10.6,f10.6)', $
	min(e_arr),max(e_arr),mean(e_arr),median(e_arr),stddev(e_arr)
printf,1,format='(%"i:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(i_arr),max(i_arr),mean(i_arr),median(i_arr),stddev(i_arr)
printf,1,format='(%"T:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(T_arr),max(T_arr),mean(T_arr),median(T_arr),stddev(T_arr)
printf,1,format='(%"W:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(W_cap_arr),max(W_cap_arr),mean(W_cap_arr), $
	median(W_cap_arr),stddev(W_cap_arr)
printf,1,format='(%"w:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(w_low_arr),max(w_low_arr),mean(w_low_arr), $
	median(w_low_arr),stddev(w_low_arr)

; convert Mass_min to a string and remove leading white space

Mass_value = string(Mass_min)

Mass_value = strtrim(Mass_min,1)

printf,1,"Restrict mass to be below ", Mass_value, " Msun"
printf,1,format='(%"M:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(mrestrict),max(mrestrict), mean(mrestrict), $
	median(mrestrict),stddev(mrestrict)
printf,1,format='(%"P:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(Prestrict),max(Prestrict), mean(Prestrict), $
	median(Prestrict),stddev(Prestrict)
printf,1,format='(%"a:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(arestrict),max(arestrict), mean(arestrict), $
	median(arestrict),stddev(arestrict)
printf,1,format='(%"e:",f10.6,f10.6,f10.6,f10.6,f10.6)', $
	min(erestrict),max(erestrict), mean(erestrict), $
	median(erestrict),stddev(erestrict)
printf,1,format='(%"i:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(irestrict),max(irestrict), mean(irestrict), $
	median(irestrict),stddev(irestrict)
printf,1,format='(%"T:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(Trestrict),max(Trestrict),mean(Trestrict), $
	median(Trestrict),stddev(Trestrict)
printf,1,format='(%"W:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(W_caprestrict),max(W_caprestrict),mean(W_caprestrict), $
	median(W_caprestrict),stddev(W_caprestrict)
printf,1,format='(%"w:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(w_lowrestrict),max(w_lowrestrict),mean(w_lowrestrict), $
	median(w_lowrestrict),stddev(w_lowrestrict)

printf,1," "

printf,1, 'Number of solutions in restricted range:', mcount
;printf,1, 'Sum of weighted number of solutions:', sum_weight
printf,1," "
printf,1,"Error range within +/-34% of median mass:"
printf,1,"Median:", medM
printf,1,'Errors:  +', dM_high  ;merror_upper 
printf,1,'         -', dM_low   ;merror_lower
;printf,1,"Total number of solutions:", sum_weight
;printf,1,'34% of total number:', num34
;printf,1,'Total number in upper error estimate:', sum_lower
;printf,1,'Total number in lower error estimate:', sum_upper
;printf,1, ' '

;printf,1, "Mass Histogram (bin value, number in bin)"
;for l=0, mnum-1 do begin
;
;	printf,1, mbins(l), h_marr(l)
;endfor

close,1

;Make plot of chi2 vs. mass (tempchi.ps)
; View chi2 surface

print,'Min chi2:',min(chi_arr)
print,'Max chi2:',max(chi_arr)

chimin = 0.0
chimax = 0.0
print,'Enter min max of chi range for plot:'
read,chimin,chimax

mydevice = !d.name
set_plot, 'ps'

device, filename='tempchi.ps'
;plot,massi,chi_arr,psym=3,xtitle='Mass (Msun)',ytitle='chi^2',yrange=[chimin,chimax]
plot,massi,chi_arr,psym=3,xtitle='Mass (M!9!dn!3!n)',ytitle='!7v!3!u2!n', $
  xrange=[0,Mass_min], yrange=[chimin,chimax],xstyle=1,ystyle=1
device, /close

set_plot,mydevice

print,'---------------------------------------------'
minmass = min(massi)
ind = where(massi eq minmass)
print,'Minimum Mass:',minmass
print,'P T e a i W w mass chi2'
print,P_arr(ind),T_arr(ind),e_arr(ind),a_arr(ind),i_arr(ind),W_cap_arr(ind),w_low_arr(ind),massi(ind),chi_arr(ind)
;ind = where(massi le minmass+0.1)
;chi_select = chi_arr(ind)
;indbest = where(chi_arr eq min(chi_select))
;print,'Minimum Mass with best chi2:',massi(indbest)
;print,'P T e a i W w mass chi2'
;print,P_arr(indbest),T_arr(indbest),e_arr(indbest),a_arr(indbest),i_arr(indbest),W_cap_arr(indbest),w_low_arr(indbest),massi(indbest),chi_arr(indbest)


end

;
;
;

pro create_histo, xmin, xrange, xstep, data, xbins, h_xarr

; INPUT
; xmid: best fit solution for separation/flux fraction
; xrange: search range   (xmid-xrange:xmid+xrange)
; xstep: step size through search range
; data: array of solutions found in grid that are within 1 chi^2 of best fit
; OUTPUT
; xbins: array of all searched separations/flux fractions
; h_xarr: array to contain the number of values at each step

xnum = round(xrange/double(xstep)) + 2 ; number of separation steps searched

;xbins = dblarr(xnum)	; array of all possible separations searched
;h_xarr = intarr(xnum)	; array to contain the number of values at each step

x = xmin

for i=0, xnum-1 do begin

	xbins(i) = x
	x = x + xstep

endfor

datanum = n_elements(data)

for i=long(0), datanum-1 do begin

	for j=long(0), xnum-1 do begin

	if (round(data(i)/xstep)*xstep eq round(xbins(j)/xstep)*xstep) $ 
			then h_xarr(j) = h_xarr(j) + 1

	endfor
endfor

end

;
;
;

pro create_histo_w, xmin, xrange, xstep, data, prob, xbins, h_xarr

; INPUT
; xmid: best fit solution for separation/flux fraction
; xrange: search range   (xmid-xrange:xmid+xrange)
; xstep: step size through search range
; data: array of solutions found in grid that are within 1 chi^2 of best fit
; OUTPUT
; xbins: array of all searched separations/flux fractions
; h_xarr: array to contain the number of values at each step

xnum = round(xrange/double(xstep)) + 2 ; number of separation steps searched

;xbins = dblarr(xnum)	; array of all possible separations searched
;h_xarr = intarr(xnum)	; array to contain the number of values at each step

x = xmin

for i=0, xnum-1 do begin

	xbins(i) = x
	x = x + xstep

endfor

datanum = n_elements(data)

for i=long(0), datanum-1 do begin

	for j=long(0), xnum-1 do begin

	if (round(data(i)/xstep)*xstep eq round(xbins(j)/xstep)*xstep) $ 
			then h_xarr(j) = h_xarr(j) + 1.0*prob(i)

	endfor
endfor

end

;
;
;

pro compute_histo_uncty, mass_array, chiprob, medM, merror_lower, merror_upper, mean_dist=mean_dist, npercent=npercent

; Set keyword /mean_dist if you wish to use mean rather than median.
; Set keyword npercent = 0.9973 (e.g., 3 sigma) if you wish to change 
; the confidence interval.

; no longer use step sizes in computing median and uncertainties
; just sort the arrays

;print,"Error range within +/-34% of median mass:"
;print,"Median:", medM
;print,'Errors:  +', merror_upper 
;print,'         -', merror_lower

; determine error: range of values (away from the median) that contains
; 2/3 of the sample (68%)
; i.e. 34% of total values on either side of median
; print, "Enter finer step size for mass histogram:"


ind_sort = sort(mass_array,/L64)

mass_sort = mass_array(ind_sort)
chi_sort = chiprob(ind_sort)

; check if best fit mass is already given... 
; ... if not, compute median mass
key_medM = keyword_set(medM)
key_mean = keyword_set(mean_dist)
key_percent = keyword_set(npercent)

sum_weight = total(chi_sort)

med_num = 0.5*sum_weight
med_sum = 0
med_ind = 0
while(med_sum le 0.5*sum_weight) do begin
	med_sum = med_sum + chi_sort(med_ind)
	med_ind = med_ind + 1.0
endwhile
med_ind = med_ind - 1.0
;med_sum = med_sum - chi_sort(med_ind)
;bin_diff = 0.5*sum_weight - med_sum
;bin_frac = bin_diff/chi_sort(med_ind)
;med_ind = med_ind - 1.0
if (key_medM ne 1) then medM = mass_sort(med_ind) $ ;+ bin_frac*mstep
else med_ind = where(abs(mass_sort - medM) eq min(abs(mass_sort - medM)))

if (key_mean) then begin
   medM = mean(mass_array)
   med_ind = where(abs(mass_sort - medM) eq min(abs(mass_sort - medM)))
endif

if (key_percent ne 1) then npercent = 0.34

num34 = npercent*sum_weight	; number of solutions included in 34% of total

;print,npercent,num34

print,'Computing errors...'

sum_lower = 0.0
sum_upper = 0.0

; determine lower error bar
arr_i = med_ind
while(sum_lower le num34) do begin
	sum_lower = sum_lower + chi_sort(arr_i)
	arr_i = arr_i - 1
endwhile
mbin_lower = arr_i + 1.0
merror_lower = medM - mass_sort(mbin_lower(0))

; determine upper error bar
arr_i = med_ind
while(sum_upper le num34) do begin
	sum_upper = sum_upper + chi_sort(arr_i)
	arr_i = arr_i + 1
endwhile
mbin_upper = arr_i - 1.0
merror_upper = mass_sort(mbin_upper(0)) - medM

end

