pro plot_data, data_file, num

; Plot components in a binary system

; Paramers:
;	Period:	period (P)
;	Tperi: 	time of periastron passage (T)
;	ecc:	eccentricity (e)
;	major:	semi-major axis (in mas) (a)
;	inc:	inclination (i)
;	Omega:	position angle of node
;	omega:	angle between node and periastron
;	ti:	time of observation
;	xd:	RA - data
;	yd:	DEC - data
;	xf:	RA - fit (determined from orbital elements for time of obs.)
;	yf:	DEC - fit (determined from orbital elements for time of obs.)
;	A,B,F,G: Thiele-Innes elements
;
; Note: xi and yi are not the same convention for RA and DEC chosen to make 
; the orbital plots. Specifically, in the fitting the apparent ellipse 
; (ORBFITE.PRO):
;	x = RA = -rho*sin(theta),   y = DEC = rho*cos(theta)
;	(i.e. North is in the direction of the positive y axis, theta is
;	measured counterclockwise from positive y-axis)
; Whereas in this program, the coordinate system set up by Couteau,
;	x = rho*cos(theta),   y = rho*sin(theta)
;	(i.e. North id in the direction of the positive x-axis and the 
;	y-axis points east)
; Both coordinate systems have the same separation and position angle 
; orietations, they are just rotated by 90 deg with respect to each other.
;
; INPUT:
;	data_file: file containing data points in the format:
;		      time  theta  error(theta)  rho  error(rho)
;		      where theta is the position angle and rho the separation
;		      and their respective errors
;		      (enter file name using quotes)
;	num: number of data points contained in data_file
;
; OUTPUT: 
;	plot of data points 
;
; Modifications to be added:
; 	- error bars on data points
;	- line connecting measurement to claculated point


!P.font=0
!P.multi=0
!P.charsize=1.2
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.5        ;scale factor for adjusting size of plotting symbols

ans_nirc2 = 'n'
;time_nirc2 = [2014.5118]     ; VSSG 14
;time_nirc2 = [2015.2599d]     ; ROXR1 12
;time_nirc2 = [2014.5090d,2015.5275d]     ; ROXs 5
;time_nirc2 = [2015.2597d]     ; ROXs 16 AaAb
;time_nirc2 = [2014.5119d,2015.2597d]     ; ROXs 16 AaAb
;time_nirc2 = [2015.5273d]     ; ROXs 43 AaAb
;time_nirc2 = [2015.5273d]     ; ROXs 43 AB
;time_nirc2 = [2004.5349d,2014.5119d,2015.2600d,2015.5277d]     ; ROXs47A
;time_nirc2 = [2015.2596d]     ; WSB 18 
;time_nirc2 = [2015.2597d]     ; WSB 21
;time_nirc2 = [2015.2598d]     ; WSB 37
;time_nirc2 = [2014.5119d,2015.5276d]     ; WSB 41 - SR 24N
;time_nirc2 = [2014.5091d,2014.5119d,2015.2598d]     ; WSB 59
;time_nirc2 = [2014.5092d,2015.5275d]     ; HBC263_SR12
;time_nirc2 = [2015.5274d]     ; HBC643_SR20
;time_nirc2 = [2019.0525d]     ; CZ Tau
;time_nirc2 = [2019.0519d]     ; IW Tau
;time_nirc2 = [2019.0523d]     ; XZ Tau
;time_nirc2 = [2019.0525d]     ; HBC 422
;time_nirc2 = [2019.0525d]     ; HBC 423
;time_nirc2 = [2019.0519d]     ; V955 Tau

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time = dblarr(num)
theta = dblarr(num)
rho = dblarr(num)
dtheta = dblarr(num)
drho = dblarr(num)

openr,lun,data_file,/get_lun

for i=0, num-1 do begin

	readf, lun, temp1, temp2, temp3, temp4, temp5

	time(i) = temp1
	theta(i) = temp2
	dtheta(i) = temp3
	rho(i) = temp4;*1000.0	
	drho(i) = temp5;*1000.0
	
endfor

close,lun

; convert theta and dtheta to radians

theta = theta*!dpi/180
dtheta = dtheta*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(theta)^2*drho^2 + rho^2*sin(theta)^2*dtheta^2)
dy = sqrt(sin(theta)^2*drho^2 + rho^2*cos(theta)^2*dtheta^2)

if (ans_nirc2 eq 'y') then begin
; Figure out NIRC2 positions
num_nirc2 = n_elements(time_nirc2)
ind_nirc2 = dblarr(num_nirc2)
for i=0, num_nirc2-1 do begin
   tdiff = abs(time - time_nirc2(i))
   ind = where(tdiff eq min(tdiff))
   ind_nirc2(i) = ind(0)
endfor
endif


;print, "Do you want to save plot as a postscript file?  (Type yes or no):"
pans = "yes"
;read,pans
; Set plotting device to ps if answer is yes
if (pans eq "yes") then begin
	mydevice=!d.name
	set_plot, 'ps'
	device, /Color, Bits_per_pixel=8, filename='temp.eps', xsize=12,ysize=12,isolatin1=1,/encap
endif

ans_fit = 'y'
print,'Do you want to fit linear motion in RA/DEC (y/n)?'
read,ans_fit

if (ans_fit eq 'y') then begin

l_regress, time, yarr, dy, a_y, b_y
l_regress, time, xarr, dx, a_x, b_x

yfit = a_y(0) + b_y(0)*time
xfit = a_x(0) + b_x(0)*time

; Compute chi2:
chi2 = total(((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2))

ychi2 = total(((yarr-yfit)^2/dy^2))
ny = n_elements(yarr)
yredchi2 = ychi2/(ny - 2.0)
xchi2 = total(((xarr-xfit)^2/dx^2))
nx = n_elements(xarr)
xredchi2 = xchi2/(nx - 2.0)

redchi2 = chi2/(nx + ny - 4.0)

print,'Linear motion in RA:'
print,'a: ',a_y * [1.0,sqrt(yredchi2)]
print,'b: ',b_y * [1.0,sqrt(yredchi2)]
print,'Linear motion in DEC:'
print,'a: ',a_x * [1.0,sqrt(xredchi2)]
print,'b: ',b_x * [1.0,sqrt(xredchi2)]

print,'chi2: ',chi2


; The following comands produce the same results and uncertainties as l_regress
; (and also show that uncty's need to be scale by reduced chi2)
;fit_y = poly_fit(time, yarr, 1, measure_errors=dy, covar=ycovar, sigma=ysigma)
;fit_x = poly_fit(time, xarr, 1, measure_errors=dx, covar=xcovar, sigma=xsigma)


; extrapolate before and after:
trange = max(time) - min(time)
t_ext = findgen(101)/100.0 *5*trange + min(time) - 2*trange
y_ext = a_y(0) + b_y(0)*t_ext
x_ext = a_x(0) + b_x(0)*t_ext

;tpred = 2013.97d
;ypred = a_y(0) + b_y(0)*tpred
;xpred = a_x(0) + b_x(0)*tpred
;print,'Sep  PA: ',sqrt(xpred^2 + ypred^2), atan(ypred,xpred)*180.0/!dpi

endif
	
; Define user symbol to be closed circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)

;Define the symbol to be a unit circle with 16 points
usersym, frac*cos(Z), frac*sin(Z), /fill

;plot data

; axes will be rotated by +90 deg (clockwise direction) inorder
; to align N-S sep with +y and E-W sep with +x
; transformation (new coordinates are x',y')
; 	x' = -y
;	y' = +x

sepmax = round(1.2*max(abs([yarr,xarr])))
;sepmax = 150.0
;sepmax = 30.0

plot, yarr, xarr, psym=8, /nodata,  $
	XSTYLE=1, YSTYLE=1, ISOTROPIC=1, $
     	xrange=[sepmax,-sepmax], yrange=[-sepmax,sepmax], $
;	xrange=[3,-3], yrange=[-3,3], $
;	xrange=[70,-70], yrange=[-70,70], $
;	xrange=[100,-100], yrange=[-100,100], $
;	xrange=[150,-150], yrange=[-150,150], $
;	xrange=[0,-70], yrange=[0,70], $
;       xrange=[450,-50],yrange=[-250,250], $ ; CZ Tau
;       xrange=[50,-300],yrange=[-250,100], $ ; HBC 422
;       xrange=[300,-100],yrange=[-200,200], $ ; HBC 423
;       xrange=[200,-250],yrange=[-400,50], $ ; IW Tau
;       xrange=[300,-50],yrange=[-300,50], $ ; XZ Tau
;       xrange=[50,-400],yrange=[-400,50], $ ; V955 Tau
;	xrange=[350, -50], yrange=[-200,200], $  ; HBC263_SR12
;	xrange=[60, -60], yrange=[-60,60], $  ; HBC643_SR20
;	xrange=[200, -200], yrange=[-350,50], $  ; GY92 371
;	xrange=[60, -10], yrange=[-10,60], $  ; MML 53
;	xrange=[200,-50], yrange=[-50,200], $ ; ROXR1 12
;	xrange=[50,-220], yrange=[-50,220], $ ; ROXs 5
;	xrange=[110,-50], yrange=[-50,110], $ ; ROXs 16 AaAb
;	xrange=[700,-100], yrange=[-400,400], $ ; ROXs 16 AB
;	xrange=[250,-250], yrange=[-400,100], $ ; ROXs 43 AaAb
;	xrange=[4.0,-4.0], yrange=[-1.0,7.0], $ ; ROXs 43 AB
;	xrange=[100,-1000.0], yrange=[-550.0,550.0], $ ; ROXs 47A wide
;	xrange=[1300,-100], yrange=[-300,1000], $ ; WSB 18 wide
;	xrange=[60, -60], yrange=[-10,110], $  ; WSB 18 close
;	xrange=[250, -50], yrange=[-100,200], $  ; WSB 21
;	xrange=[200, -200], yrange=[-200,200], $  ; WSB 37
;	xrange=[3.0, -3.0], yrange=[-1.0,6.0], $  ; WSB 41 - SR 24 NS
;	xrange=[150,-150], yrange=[-150,150], $ ; WSB 59
;	xrange=[50, -400], yrange=[-200,250], $  ; V853 Oph
;	xrange=[30, -30], yrange=[-30,30], $  ; V853 Oph - close
;	xrange=[50, -270], yrange=[-160,160], $  ; VSSG 17
;	xrange=[0,-200], yrange=[0,200], $
;	xrange=[2500,0], yrange=[0,2500], $
;	xrange=[0.2,-2.0], yrange=[-2.0,0.2], $  ; TWA 3
	;xtitle="E-W sep (mas)", ytitle="N-S sep (mas)"
	xtitle='!9D!3RA (mas)',ytitle='!9D!3DEC (mas)', $
;	xtitle='!4D!3RA (mas)',ytitle='!4D!3DEC (mas)'
;	xtitle="!4D!3RA ('')",ytitle="!4D!3DEC ('')"
;	xtitle="!9D!3RA ('')",ytitle="!9D!3DEC ('')"
        position=[0.19,0.12,0.95,0.98] ;position=[0.19,0.12,0.96,0.98], $ ;, position=[0.18,0.12,0.95,0.98], $ ;, position=[0.18,0.11,0.95,0.98],

plot_err,yarr,abs(dy),xarr,abs(dx),num

if (ans_nirc2 eq 'y') then begin
   ; Plot all data as grey circles
   usersym, frac*cos(Z), frac*sin(Z), /fill
   oplot, yarr, xarr, psym=8,color=150
   usersym, frac*cos(Z), frac*sin(Z), thick=2 ; black open circle outline
   oplot, yarr, xarr, psym=8

   ; Plot nirc2 data as black circles
   usersym, frac*cos(Z), frac*sin(Z), /fill
   oplot, yarr(ind_nirc2), xarr(ind_nirc2), psym=8
endif else begin
   ;usersym, frac*cos(Z), frac*sin(Z), /fill
   ;oplot, yarr, xarr,psym=8
   usersym, frac*cos(Z), frac*sin(Z), /fill
   oplot, yarr, xarr, psym=8,color=150
   usersym, frac*cos(Z), frac*sin(Z), thick=2 ; black open circle outline
   oplot, yarr, xarr, psym=8
endelse

;if (ans_fit eq 'y') then oplot,yfit,xfit
;if (ans_fit eq 'y') then oplot,y_ext,x_ext
if (ans_fit eq 'y') then begin
   oplot,y_ext,x_ext
   ; connect data points to fit
   for i=0,num-1 do oplot,[yfit(i),yarr(i)],[xfit(i),xarr(i)]
endif

; draw axes through origin

;plotS, [-100,100],[0,0], LineStyle=1
;plotS, [0,0],[-100,100], LineStyle=1

;plotS, [-70,70],[0,0], LineStyle=1
;plotS, [0,0],[-70,70], LineStyle=1

;plotS, [-70,0],[0,0], LineStyle=1
;plotS, [0,0],[0,70], LineStyle=1

; plot primary as an asterisk

oplot, [0],[0],psym=2
;sepmax = 300
oplot,[-sepmax,sepmax],[0,0],linestyle=1
oplot,[0,0],[-sepmax,sepmax],linestyle=1
;oplot,[-200,200],[0,0],linestyle=1
;oplot,[0,0],[-200,200],linestyle=1

; Plot time labels
;xyouts,yarr,xarr+8,string(format='(f6.1)',time),alignment=0.5,charsize=0.9
;xyouts,yarr-0.20*sepmax,xarr-0.025*sepmax,string(format='(f6.1)',time),alignment=0.5,charsize=1.0

ptitle = ' '
print,'Enter plot title:'
read,ptitle
xyouts,sepmax - 0.1*sepmax,-sepmax + 0.1*sepmax,ptitle,charsize=1.5
;xyouts,sepmax - 0.1*sepmax,sepmax - 0.2*sepmax,ptitle,charsize=1.5
;xyouts,90,85,ptitle
;xyouts,70,65,ptitle,charsize=1.5
;xyouts,60,55,ptitle,charsize=1.5
;xyouts,2425,75,ptitle,charsize=1.5
;xyouts,-1.4,-1.8,ptitle,charsize=1.5
;xyouts,430,-230,ptitle,charsize=1.5  ; CZ Tau
;xyouts,-20,-230,ptitle,charsize=1.5  ; HBC 422
;xyouts,280,160,ptitle,charsize=1.5  ; HBC 423
;xyouts,180,15,ptitle,charsize=1.5  ; IW Tau
;xyouts,280,15,ptitle,charsize=1.5  ; XZ Tau
;xyouts,-250,-40,ptitle,charsize=1.5  ; V955 Tau
;xyouts,330,-180,ptitle,charsize=1.5  ; HBC263_SR12
;xyouts,50,48,ptitle,charsize=1.5  ; HBC643_SR20
;xyouts,180,15,ptitle,charsize=1.5,alignment=0.0  ; GY92 371
;xyouts,56,-7,ptitle,charsize=1.5  ; MML 53
;xyouts,180,-30,ptitle,charsize=1.5  ; ROXR1 12
;xyouts,-130,-30,ptitle,charsize=1.5  ; ROXs 5
;xyouts,100,-40,ptitle,charsize=1.5  ; ROXs 16 AaAb
;xyouts,650,300,ptitle,charsize=1.5  ; ROXs 16 AB
;xyouts,-25,45,ptitle,charsize=1.3  ; ROXs 43 AaAb
;xyouts,-0.5,0.5,ptitle,charsize=1.5  ; ROXs 47 A wide
;xyouts,-50,420,ptitle,charsize=1.5  ; ROXs 43 AB
;xyouts,1200,850,ptitle,charsize=1.5  ; WSB 18 wide
;xyouts,-20,5,ptitle,charsize=1.5  ; WSB 18 close
;xyouts,230,-80,ptitle,charsize=1.5  ; WSB 21
;xyouts,180,-180,ptitle,charsize=1.5  ; WSB 37
;xyouts,2.7,-0.6,ptitle,charsize=1.5  ; WSB 41 - SR 24 NS
;xyouts,130,120,'WSB 59',charsize=1.5,alignment=0.0
;xyouts,-250,120,ptitle,charsize=1.5,alignment=1.0  ; VSSG 17
;xyouts,-28,25,ptitle,charsize=1.5,alignment=1.0  ; V853 Oph

;xyouts,100,5,'!9r !3< 20 mas in 2015',charsize=1.5  ; ROXs 16
;xyouts,50,38,'!9r !3< 16 mas in 2015',charsize=1.5  ; HBC643_SR20

; Plot time labels
;xl = -0.05*sepmax
;yl =  0.08*sepmax
xl =  0.0*sepmax
yl =  0.08*sepmax
xyouts,yarr+xl,xarr+yl,string(format='(f6.1)',time),alignment=0.5;,charsize=1.2
;xyouts,yarr(0)-20,xarr(0)+25,string(format='(f6.1)',time(0)),alignment=0.5;,charsize=1.2  ; CZ Tau
;xyouts,yarr(num-2)-20,xarr(num-2)-35,string(format='(f6.1)',time(num-2)),alignment=0.5;,charsize=1.2  ; CZ Tau
;xyouts,yarr(num-1)+10,xarr(num-1)-35,string(format='(f6.1)',time(num-1)),alignment=0.5;,charsize=1.2  ; CZ Tau
;xyouts,yarr(0)+5,xarr(0)-30,string(format='(f6.1)',time(0)),alignment=0.5;,charsize=1.2  ; HBC 422
;xyouts,yarr(7)+15,xarr(7)-7,string(format='(f6.1)',time(7)),alignment=1.0;,charsize=1.2  ; HBC 422
;xyouts,yarr(num-1)+15,xarr(num-1)-7,string(format='(f6.1)',time(num-1)),alignment=1.0;,charsize=1.2  ; HBC 422
;xyouts,yarr(0)-0,xarr(0)+15,string(format='(f6.1)',time(0)),alignment=0.5;,charsize=1.2  ; HBC 423
;xyouts,yarr(num-1)-0,xarr(num-1)-30,string(format='(f6.1)',time(num-1)),alignment=0.5;,charsize=1.2  ; HBC 423
;xyouts,yarr(10)-15,xarr(10)-9,string(format='(f6.1)',time(10)),alignment=0.0;,charsize=1.2  ; HBC 423
;xyouts,yarr(num-1)-15,xarr(num-1)-9,string(format='(f6.1)',time(num-1)),alignment=0.0;,charsize=1.2  ; HBC 423
;xyouts,yarr(0)+20,xarr(0)-10,string(format='(f6.1)',time(0)),alignment=1.0;,charsize=1.2  ; IW Tau
;xyouts,yarr(num-2)-20,xarr(num-2)-10,string(format='(f6.1)',time(num-2)),alignment=0.0;,charsize=1.2  ; IW Tau
;xyouts,yarr(num-1)-15,xarr(num-1)-10,string(format='(f6.1)',time(num-1)),alignment=0.0;,charsize=1.2  ; IW Tau
;xyouts,yarr(0)-15,xarr(0)-5,string(format='(f6.1)',time(0)),alignment=0.0;,charsize=1.2  ; XZ Tau
;;xyouts,yarr(0)-0,xarr(0)-25,string(format='(f6.1)',time(0)),alignment=0.5;,charsize=1.2  ; XZ Tau
;;xyouts,yarr(num-2)-0,xarr(num-2)+15,string(format='(f6.1)',time(num-2)),alignment=0.5;,charsize=1.2  ; XZ Tau
;xyouts,yarr(num-1)+10,xarr(num-1)-5,string(format='(f6.1)',time(num-1)),alignment=1.0;,charsize=1.2  ; XZ Tau
;xyouts,yarr,xarr+20,string(format='(f6.1)',time),alignment=0.5;,charsize=1.2 ; GY92 371
;xyouts,yarr(0)+20,xarr(0)-2,string(format='(f6.1)',time(0)),alignment=0.5;,charsize=1.2 ; HBC 243 SR 20
;xyouts,yarr(1)+0,xarr(1)+6,string(format='(f6.1)',time(1)),alignment=0.5;,charsize=1.2 ; HBC 243 SR 20
;xyouts,yarr(2)+0,xarr(2)+6,string(format='(f6.1)',time(2)),alignment=0.5;,charsize=1.2 ; HBC 243 SR 20
;xyouts,yarr(3)+0,xarr(3)+5,string(format='(f6.1)',time(3)),alignment=0.5;,charsize=1.2 ; HBC 243 SR 20
;xyouts,yarr(4)+0,xarr(4)-10,string(format='(f6.1)',time(4)),alignment=0.5;,charsize=1.2 ; HBC 243 SR 20
;xyouts,yarr+5,xarr-1,string(format='(f6.1)',time),alignment=1.0;,charsize=1.2  ; MML 53
;xyouts,yarr(0),xarr(0)+60,string(format='(f6.1)',time(0)),alignment=0.5;,charsize=1.2  ; HBC263_SR12
;xyouts,yarr(num-1),xarr(num-1)+20,string(format='(f6.1)',time(num-1)),alignment=0.5;,charsize=1.2  ; HBC263_SR12
;xyouts,yarr(0),xarr(0)+25,string(format='(f6.1)',time(0)),alignment=0.5;,charsize=1.2  ; ROXR1 12
;xyouts,yarr(num-1)+5,xarr(num-1)+10,string(format='(f6.1)',time(num-1)),alignment=0.5;,charsize=1.2  ; ROXR1 12
;xyouts,yarr(0),xarr(0)+10,string(format='(f6.1)',time(0)),alignment=0.5;,charsize=1.2  ; ROXs 5
;xyouts,yarr(num-1)-2,xarr(num-1)-20,string(format='(f6.1)',time(num-1)),alignment=0.5;,charsize=1.2  ; ROXs 5
;xyouts,yarr(0)+5,xarr(0)+5,string(format='(f6.1)',time(0)),alignment=1.0;,charsize=1.2  ; ROXs 16 AaAb
;xyouts,yarr(num-1)-8,xarr(num-1)-3.5,string(format='(f6.1)',time(num-1)),alignment=0.0;,charsize=1.2  ; ROXs 16 AaAb
;xyouts,yarr(0),xarr(0)+50,string(format='(f6.1)',time(0)),alignment=0.5;,charsize=1.2  ; ROXs 16 AB
;xyouts,yarr(num-1),xarr(num-1)-60,string(format='(f6.1)',time(num-1)),alignment=0.5;,charsize=1.2  ; ROXs 16 AB
;xyouts,yarr(0)+20,xarr(0)-8,string(format='(f6.1)',time(0)),alignment=1.0;,charsize=1.2  ;ROXs 43 AaAb
;xyouts,yarr(num-1)+20,xarr(num-1)-8,string(format='(f6.1)',time(num-1)),alignment=1.0;,charsize=1.2  ;ROXs 43 AaAb
;xyouts,yarr(0)+0.5,xarr(0)-0.2,string(format='(f6.1)',time(0)),alignment=1.0;,charsize=1.2  ;ROXs 43 AB
;xyouts,yarr(1)+0.5,xarr(1)-0.2,string(format='(f6.1)',time(1)),alignment=1.0;,charsize=1.2  ;ROXs 43 AB
;xyouts,yarr(num-1)+0.5,xarr(num-1)-0.2,string(format='(f6.1)',time(num-1)),alignment=1.0;,charsize=1.2  ;ROXs 43 AB
;xyouts,yarr,xarr+5,string(format='(f6.1)',time),alignment=0.5;,charsize=1.2  ; WSB 18 close
;xyouts,yarr,xarr+18,string(format='(f6.1)',time),alignment=0.5;,charsize=1.2 ; WSB 21
;xyouts,yarr,xarr+12,string(format='(f6.1)',time),alignment=0.5;,charsize=1.2 ; WSB 37
;xyouts,yarr(0)-0,xarr(0)-25,string(format='(f6.1)',time(0)),alignment=0.5 ; WSB 59
;xyouts,yarr(3)-0,xarr(3)-25,string(format='(f6.1)',time(3)),alignment=0.5 ; WSB 59
;xyouts,yarr(0),xarr(0)+15,string(format='(f6.1)',time(0)),alignment=0.5;,charsize=1.2  ; VSSG 17
;xyouts,yarr(1),xarr(1)-25,string(format='(f6.1)',time(1)),alignment=0.5;,charsize=1.2  ; VSSG 17
;xyouts,yarr(1)+20,xarr(1)-8,string(format='(f6.1)',time(1)),alignment=1.0;,charsize=1.2  ; V955 Tau
;xyouts,yarr(9)-20,xarr(9)-8,string(format='(f6.1)',time(9)),alignment=0.0;,charsize=1.2  ; V955 Tau
;xyouts,yarr(10),xarr(10)+15,string(format='(f6.1)',time(10)),alignment=0.5;,charsize=1.2  ; V955 Tau

; Plot alternate PA solution for speckle?
ans_alt = 'n'
if (ans_alt eq 'y') then begin
   plot_err,[-yarr(0)],[abs(dy(0))],[-xarr(0)],[abs(dx(0))],1
   usersym, frac*cos(Z), frac*sin(Z), /fill
   oplot, [-yarr(0)], [-xarr(0)], psym=8,color=150
   usersym, frac*cos(Z), frac*sin(Z), thick=2 ; black open circle outline
   oplot, [-yarr(0)], [-xarr(0)], psym=8
   ;xyouts,-yarr(0)+xl,-xarr(0)+yl,string(format='(f6.1)',time(0)),alignment=0.5 ;,charsize=1.2
   ;xyouts,-yarr(0),-xarr(0)-30,string(format='(f6.1)',time(0)),alignment=0.5 ;,charsize=1.2 ; WSB 37
   xyouts,-yarr(0)-0,-xarr(0)+15,string(format='(f6.1)',time(0)),alignment=0.5 ; WSB 59
endif

; set plotting device back to terminal if previously saving to ps file
if (pans eq "yes") then begin 
	device, /close
	set_plot, mydevice
endif

set_plot,'ps'

!P.multi=[0,0,2]
; Define user symbol to be rectangle
xrect = frac*[-1.0,-1.0,1.0,1.0,-1.0]
yrect = frac*[-1.0,1.0,1.0,-1.0,-1.0]
usersym,xrect,yrect,/fill


device, /Color, Bits_per_pixel=8, filename='temp_xy.ps', xsize=12,ysize=12,isolatin1=1

yr = max(yarr) - min(yarr)
xr = max(xarr) - min(xarr)

tr = max(time) - min(time)

plot,time,yarr,xtitle='Year',ytitle='!9D!3RA (mas)',psym=8, $
;plot,time,yarr,xtitle='Year',ytitle='!4D!3RA (mas)',psym=8, $
     yrange=[min(yarr) - 0.05*yr, max(yarr) + 0.05*yr];, $
     ;xrange=[2007.8,2012],xstyle=1
     ;xrange=[min(time) - 0.05*tr, max(time) + 0.05*tr]
for i=0, num-1 do oplot,[time(i),time(i)],[yarr(i) + dy(i),yarr(i) - dy(i)]
;if (ans_fit eq 'y') then oplot,time,yfit
if (ans_fit eq 'y') then oplot,t_ext,y_ext

plot,time,xarr,xtitle='Year',ytitle='!9D!3DEC (mas)',psym=8, $
;plot,time,xarr,xtitle='Year',ytitle='!4D!3DEC (mas)',psym=8, $
     yrange=[min(xarr) - 0.05*xr, max(xarr) + 0.05*xr];, $
     ;xrange=[2007.8,2012],xstyle=1
     ;xrange=[min(time) - 0.05*tr, max(time) + 0.05*tr]
for i=0, num-1 do oplot,[time(i),time(i)],[xarr(i) + dx(i),xarr(i) - dx(i)]
;if (ans_fit eq 'y') then oplot,time,xfit
if (ans_fit eq 'y') then oplot,t_ext,x_ext


device,/close

set_plot,'x'

free_lun,lun

!P.Charsize=1.0
!P.multi=0

;stop,'Type .cont to continue'

end

;
;
;

pro linfith, X, Y, z, sigz, c, d

; Least squares linear fit to the equation: z = cX + dY
; where z = (x,y), c = (A,B), and d = (F,G)
; sigz = errors in z=(x,y) ... (for a weighted least squares fit)
; The user supplies the X,Y,z arrays and errors in z
; Program returns values for c,d
; The least squares fit is performed in the same manner as Harthopf's code
; Minimize chi squared with respect to A,B,F,G (set partial derivatives to 0):
; 	chi^2 = sum[(x - AX - FY)^2/sigx^2 + (y - BX - GY)^2/sigy^2]
; Because x is only a function of (A,F), and y is only a funtion of (B,G),
; minimizing chi squared can proceed independently for each of the x,y 
; variables.

num_data = n_elements(z)	;number of data points
ind_data = num_data -1


c = (total(X*Y/sigz^2)*total(z*Y/sigz^2) - $
	total(z*X/sigz^2)*total(Y*Y/sigz^2))/ $
	(total(X*Y/sigz^2)^2 - total(X*X/sigz^2)*total(Y*Y/sigz^2))

d = (total(X*Y/sigz^2)*total(z*X/sigz^2) - $
	total(z*Y/sigz^2)*total(X*X/sigz^2))/ $
	(total(X*Y/sigz^2)^2 - total(X*X/sigz^2)*total(Y*Y/sigz^2))

;print, "e^2:", e2new

end

;
;
;

pro plot_err, x, dx, y, dy, N

; plot error bars associated with each data point
; Input
;	x: array of x values
;	dx: array of error in x values
;	y: array of y values
;	dy: erray of error in y values
;	N: number of data points


for i=0, N-1 do begin

	oplot, [x(i)-dx(i), x(i)+dx(i)], [y(i), y(i)]
	oplot, [x(i), x(i)], [y(i)-dy(i), y(i)+dy(i)]

endfor

end
