pro plot_lunar_elias12, data_file, num

; Given a set of orbital elements {P,T,e,a,i,Omega,omega}
; compute position at a given set of times
;
; Compare with measured values of (sep, PA)
;
; Read times of observations and measurements in from data file
;
; INPUT:
;	data_file: file containing data points in the format:
;		      time  theta  error(theta)  rho  error(rho)
;		      where theta is the position angle and rho the separation
;		      and their respective errors
;		      (enter file name using quotes)
;	num: number of data points contained in data_file
;
; Equations:  
; Orbital elements (P,T,e) and a set of observations (ti,xi,yi) are known
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)
; The four Thiele-Innes elements are solved through linear least squares fit:
;	xi = AXi + FYi
;	yi = BXi + GYi
; Calculate geometric elements a("),i,Omega,omega from (A,F,B,G)
;
; Essentially, the equations one needs to solve are:
;	E - esinE = 2Pi/P * (t - T)
;	x = A*(cos(Ei) - e) + F*(sqrt(1 - e^2)*sin(Ei))
;	y = B*(cos(Ei) - e) + G*(sqrt(1 - e^2)*sin(Ei))
;
; Paramers:
;	Period:	period (P)
;	Tperi: 	time of periastron passage (T)
;	ecc:	eccentricity (e)
;	major:	semi-major axis (in mas) (a)
;	inc:	inclination (i)
;	Omega:	position angle of node
;	omega:	angle between node and periastron
;	ti:	time of observation
;	xd:	RA - data
;	yd:	DEC - data
;	xf:	RA - fit (determined from orbital elements for time of obs.)
;	yf:	DEC - fit (determined from orbital elements for time of obs.)
;	A,B,F,G: Thiele-Innes elements
;
; Note: xi and yi are not the same convention for RA and DEC chosen to make 
; the orbital plots. Specifically, in the fitting the apparent ellipse 
; (ORBFITE.PRO):
;	x = RA = -rho*sin(theta),   y = DEC = rho*cos(theta)
;	(i.e. North is in the direction of the positive y axis, theta is
;	measured counterclockwise from positive y-axis)
; Whereas in this program, the coordinate system set up by Couteau,
;	x = rho*cos(theta),   y = rho*sin(theta)
;	(i.e. North id in the direction of the positive x-axis and the 
;	y-axis points east)
; Both coordinate systems have the same separation and position angle 
; orietations, they are just rotated by 90 deg with respect to each other.
;
; INPUT:
;	data_file: file containing data points in the format:
;		      time  theta  error(theta)  rho  error(rho)
;		      where theta is the position angle and rho the separation
;		      and their respective errors
;		      (enter file name using quotes)
;	num: number of data points contained in data_file
;	
; Prompted inputs:
;	all of the orbital elements: P,T,e
;
; OUTPUT: 
;	plot of data points and orbital solutions
;
; Modifications to be added:
; 	- error bars on data points
;	- line connecting measurement to claculated point


!P.multi=0
!P.charsize=1.2
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.2	;scale factor for adjusting size of plotting symbols

; Set color table
;  red=1, green=2, blue=3

;tvlct, [255,0,0], [0,255,0], [0,0,255], 1

; Read in measurements from data file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time = dblarr(num)
sepmeas = dblarr(num)
PAmeas = dblarr(num)
dsep = dblarr(num)
dPA = dblarr(num)

openr,lun,data_file,/get_lun

for i=0, num-1 do begin

	readf, lun, temp1, temp2, temp3, temp4, temp5

	time(i) = temp1
	PAmeas(i) = temp2
	dPA(i) = temp3
	sepmeas(i) = temp4	
	dsep(i) = temp5
	
endfor

close,lun

; convert theta and dtheta to radians

PAmeas = PAmeas

PAmeas = PAmeas*!dpi/180
dPA = dPA*!dpi/180

; convert data points to x and y coordinates

xmeas = sepmeas * cos(PAmeas)	; x coordinate
ymeas = sepmeas * sin(PAmeas)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(PAmeas)^2*dsep^2 + sepmeas^2*sin(PAmeas)^2*dPA^2)
dy = sqrt(sin(PAmeas)^2*dsep^2 + sepmeas^2*cos(PAmeas)^2*dPA^2)

period = 0d
Tperi = 0d
ecc = 0d
major = 0d
inc = 0d
W_cap = 0d
w_low = 0d

answer = 'y'		; answer for plotting another orbital solution
count = 0

while (answer eq 'y') do begin

    ; Obtain initial values for P,T,e,a,i,W,w

    print,"Enter P T e a i W w (period, time of periastron passage, eccentricity):"
    read, period, Tperi, ecc, major, inc, W_cap, w_low

    ;convert angles from degrees to radians

    inc = inc*!dpi/180.0
    W_cap = W_cap*!dpi/180.0
    w_low = w_low*!dpi/180.0

    orbEL = [period, Tperi, ecc, major, inc, W_cap, w_low]

    ; plot orbit with 1000 points

    pnum = 1000.0

    tstep = period/pnum

    ; x points N (DEC)
    ; y points E (increasing RA)
    ; therefore plot as (-y,x)

    xarr=dblarr(pnum+1)
    yarr=dblarr(pnum+1)

    ti = Tperi

    for i=0, pnum do begin

        ; compute orbital position at given time:

        orbpos, ti, orbEL, xpos, ypos, tsep, tPA

	xarr(i) = xpos
	yarr(i) = ypos

	ti = ti + tstep

    endfor

    xmax = max(xarr)
    xmin = min(xarr)

    ymax = max(yarr)
    ymin = min(yarr)

    if (count eq 0) then begin    
        set_plot, 'ps'
        device,/Color,Bits_per_pixel=8,filename='temp.ps',xsize=15,ysize=15 

        plot,yarr,xarr, $
          xtitle='E-W sep (mas)', ytitle='N-S sep (mas)',ISOTROPIC=1, $
          xrange=[55,-50], yrange=[-65,40],xstyle=1,ystyle=1, /nodata
;          xrange=[60,-60], yrange=[-60,60],xstyle=1,ystyle=1, /nodata
;          xrange=[60,-60], yrange=[-50,70],xstyle=1,ystyle=1, /nodata

        ; mark primary with an asterisk
        oplot,[0],[0],psym=2

        ; plot title
        xyouts, 50, 30, 'Elias 12 Na-Nb',charsize=1.5,charthick=3


    endif

    if (count eq 0) then nline=0 else nline=2

    oplot,yarr,xarr,linestyle=nline

    ; compare measured and computed positions

    ; compute predicted position on date of observation

    xfit = dblarr(num)
    yfit = dblarr(num)

    for i=0, num-1 do begin

        ti = time(i)

        orbpos, ti, orbEL, xi, yi, sep, PA

        xfit(i) = xi
        yfit(i) = yi

       ;print,'Position of companion for t= ',time
       ;print,'(x,y) = ',xorb,yorb
       ;print,'(sepRA,sepDec) = ',yorb,xorb
       ;print,'(sep,PA) = ',sep,PA*180.0/!dpi

    endfor

    ; print O-C residuals (Observed - Calculated)

    xOC = xmeas - xfit
    yOC = ymeas - yfit

    openw,lun,'xyresiduals'
    printf,lun,"O-C residuals (data - fit)"
    printf,lun,"date     dx       xO-C   xOC/dx dy      yO-C   yOC/dy"
    ;printf,lun,"date   dx   xdata - xfit   xOC/dx   dy   ydata - yfit   yOC/dy"
    for i=0, num-1 do printf,lun,format='(f8.3,f7.3,f8.3,f8.3,f7.3,f8.3,f8.3)', $
	time(i), dx(i), xOC(i), xOC(i)/dx(i), dy(i), yOC(i), yOC(i)/dy(i)
    close,lun

    ;2004.863  0.001  -0.051-96.010  0.000   0.063233.463

    ; mark secondary at computed time with a cross

    ans='y'
    print, 'Do you want to plot computed position of binary (y/n)?'
    read, ans

    if (ans eq 'y') then oplot,[yfit],[xfit],psym=7

    ; connect computed and measured positions with a line

    print, 'Do you want to connect computed and measured positions with a line (y/n)?'
    read, ans

    if (ans eq 'y') then begin
        for i=0, num-1 do $
            oplot,[yfit(i),ymeas(i)],[xfit(i),xmeas(i)]
    endif

    count = count + 1

    print,'Do you want to plot another orbit (y/n)?'
    read,answer

endwhile

; Define user symbol to be closed circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)

; Plot measured positions as filled circles

usersym, frac*cos(Z), frac*sin(Z),/fill

oplot,ymeas(1:num-1),xmeas(1:num-1),psym=8

; plot error bars
plot_err,ymeas(1:num-1),abs(dy(1:num-1)),xmeas(1:num-1),abs(dx(1:num-1)),num-1

; Filled Squares for lunar occultation prediction
; Define user symbol to be open rectangle
xrect = frac*[-1.0,-1.0,1.0,1.0,-1.0]
yrect = frac*[-1.0,1.0,1.0,-1.0,-1.0]
usersym,xrect,yrect,/fill
oplot,[ymeas(0)],[xmeas(0)],psym=8,color=127;,symsize=1.5
usersym, frac*cos(Z), frac*sin(Z),/fill

; plot lunar occultation projection and projected separation of binary

PA_lunar = 95.0*!dpi/180
sep_lunar = 23.0

xlun = 100.0*sin(PA_lunar)
ylun = 100.0*cos(PA_lunar)

oplot,[xlun,-xlun],[ylun,-ylun]
print,xlun,ylun

; find equation of line that passes through projected separation
; (perpendicular to lunar occultation PA)
; b = y - mx

m_slope = -sin(PA_lunar)/cos(PA_lunar)
b_int = sep_lunar*cos(PA_lunar) - m_slope*sep_lunar*sin(PA_lunar)

oplot,[100,-100],[100*m_slope+b_int,-100*m_slope+b_int],LineStyle=2

; and equivalent line at minus projected sep
b_int = -sep_lunar*cos(PA_lunar) + m_slope*sep_lunar*sin(PA_lunar)

;oplot,[100,-100],[100*m_slope+b_int,-100*m_slope+b_int],LineStyle=2

print, 'Do you want to plot date labels (y/n)?'
read, ans

if (ans eq 'y') then begin

    offset = [5.0,5.0]

    deltax = offset(0)*sin(PAmeas)
    deltay = offset(1)*cos(PAmeas)

    ; plot date labels

;    xyouts,ymeas+deltax,xmeas+deltay,string(format='(f7.2)',time),alignment=0.5,charsize=0.6

tlab = round(time*100.0)/100.0

!P.charsize=1.0
xyouts,ymeas(0)+12,xmeas(0)+2.5,string(format='(f7.2)',tlab(0))
xyouts,ymeas(1)-6,xmeas(1)-0.8,string(format='(f7.2)',tlab(1))
xyouts,ymeas(2)-6,xmeas(2)-0.8,string(format='(f7.2)',tlab(2))
xyouts,ymeas(3)-6,xmeas(3)-0.8,string(format='(f7.2)',tlab(3))
xyouts,ymeas(4)-6,xmeas(4)-0.8,string(format='(f7.2)',tlab(4))
xyouts,ymeas(5)-6,xmeas(5)-0.8,string(format='(f7.2)',tlab(5))
xyouts,ymeas(6)-6,xmeas(6)-0.8,string(format='(f7.2)',tlab(6))
xyouts,ymeas(7)-4,xmeas(7)-4,string(format='(f7.2)',tlab(7))
xyouts,ymeas(8)+7,xmeas(8)-7.0,string(format='(f7.2)',tlab(8))
xyouts,ymeas(9)-4,xmeas(9)-0.5,string(format='(f7.2)',tlab(9))
xyouts,ymeas(10)+8,xmeas(10)-6.0,string(format='(f7.2)',tlab(10))
xyouts,ymeas(11)+16,xmeas(11)-3.0,string(format='(f7.2)',tlab(11))
xyouts,ymeas(12)+16,xmeas(12)-2.6,string(format='(f7.2)',tlab(12))
xyouts,ymeas(13)+16,xmeas(13)+0.0,string(format='(f7.2)',tlab(13))
xyouts,ymeas(14)+16,xmeas(14)-0.8,string(format='(f7.2)',tlab(14))
xyouts,ymeas(15)+16,xmeas(15)-0.8,string(format='(f7.2)',tlab(15))
endif else begin

tlab = round(time*100.0)/100.0
; P=11.85 yr label
xyouts,ymeas(0)+0,xmeas(0)+2.0,string(format='(f7.2)',tlab(0))
; P=8.65 yr label
;xyouts,ymeas(0)-5,xmeas(0)-1.0,string(format='(f7.2)',tlab(0))

endelse

usersym,xrect,yrect,/fill
oplot,[ymeas(0)],[xmeas(0)],psym=8,color=127
usersym,xrect,yrect
oplot,[ymeas(0)],[xmeas(0)],psym=8

oplot,[0,0],[-100,100],linestyle=1
oplot,[100,-100],[0,0],linestyle=1

!P.Charsize=1.0

device, /close
set_plot, 'x'

free_lun,lun

end

;
;
;

pro solve_trans, e, M, EE

; Solve transcendental equation of the form E - esinE = M.
; Use iterative procedure to determine E.
; Initial approximation: E_0 = M + esinM - e^2/2 sin(2M)
; Improve solution by iterating the following to formulae:
;	M_0 = E_0 - esin(E_0)
;	E_1 = E_0 + (M - M_0)/(1 - ecos(E_0))
;	(derivative of Kepler's equation)
;
; Method adapted from Heintz 1978 (p.34-35)
; Results compared with point-&-click graphical method.  Iterative approach 
; leads to exact solution that satisfies E - esinE = M.  Therefore, 
; point-&-click method is subsequently removed from orbit fitting.
;
; INPUT:
;	e: eccentricity
;	M: mean anomaly   M= 2*Pi/P
;
; OUTPUT:
;	EE: eccentric anomaly
;
; Created: 9 May 2002

; Initial approximation:

EE = M + e*sin(M) + e^2/2*sin(2*M)

;print, "Initial approximation for E:", EE

EEi = 0d	; parameter to hold initial value once enter while loop

count = 0

while (abs(EE - EEi) gt 0.000001) do begin

	EEi = EE

	Mi = EEi - e*sin(EEi)

	EE = EEi + (M - Mi)/(1 - e*cos(EEi))

	count=count+1

endwhile

;print, "Final iterated E:", EE

;print,"Number of iterations:",count

end

;
;
;

pro orbpos, time, orbEL, x, y, sep, PA

; Given set of orbital parameters, compute position at a given time
;
; INPUT:
; time
; orbEL = [period, Tperi, ecc, major, inc, W_cap, w_low]
;
; OUTPUT:
; (x,y)
; (sep, PA)

period = orbEL(0)
Tperi = orbEL(1)
ecc = orbEL(2)
major = orbEL(3)
inc = orbEL(4)
W_cap = orbEL(5)
w_low = orbEL(6)

; Determine the eccentric anomalies Ei: 
;     mu(ti - T) = Ei - esin(Ei)  ... Kepler's Equation
;     where mu = 360/P = 2*pi/P
; Solve this transcendental equation through an 
; iterative procedure.
; Use SOLVE_TRANS.PRO

; array to hold Ei's (the eccentric anomaly)

Ei = 0d

mu = 2*!dpi/period

Mi = mu*(time - Tperi) 

; reduce to same epoch

Mi = 2*!dpi*((time - Tperi)/period $	
	- fix((time - Tperi)/period))

solve_trans,ecc,Mi,Eit

Ei = Eit

; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)

Xi = cos(Ei) - ecc
Yi = sqrt(1 - ecc^2)*sin(Ei)

; Compute the four Thiele-Innes elements

A_cap = major*(cos(w_low)*cos(W_cap) - sin(w_low)*sin(W_cap)*cos(inc))
B_cap = major*(cos(w_low)*sin(W_cap) + sin(w_low)*cos(W_cap)*cos(inc))
F_cap = major*(-sin(w_low)*cos(W_cap) - cos(w_low)*sin(W_cap)*cos(inc))
G_cap = major*(-sin(w_low)*sin(W_cap) + cos(w_low)*cos(W_cap)*cos(inc))

; Compute coordinates of secondary

;	xi = AXi + FYi
;	yi = BXi + GYi

x = A_cap*Xi + F_cap*Yi
y = B_cap*Xi + G_cap*Yi

; convert (x,y) to (PAfit,sepfit)
; x = rho*sin(theta), y=rho*sin(theta)

sep = sqrt(x^2 + y^2)

; Get correct sign on PA by entering y,x separately into atan:
PA = atan(y,x)

if (PA lt 0.0) then PA = 2*!dpi + PA

end

;
;
;

pro plot_err, x, dx, y, dy, N

; plot error bars associated with each data point
; Input
;	x: array of x values
;	dx: array of error in x values
;	y: array of y values
;	dy: erray of error in y values
;	N: number of data points


for i=0, N-1 do begin

	oplot, [x(i)-dx(i), x(i)+dx(i)], [y(i), y(i)]
	oplot, [x(i), x(i)], [y(i)-dy(i), y(i)+dy(i)]

endfor

end
