pro plot_orbit, data_file, num

; Given the 7 orbital elements (P,T,e,A,B,F,G) or (P,T,e,a,i,Omega,omega)
; plot orbit and compare with measurements
;
; Options include entering P,T,e and then doing a least squares fit for
; a,i,Omega,omega or entering (P,T,e,a,i,Omega,omega)
; [(P,T,e,A,B,F,G) entering to be added at a later date]
;
; Read times of observations and measurements in from data file
;	
; Equations:  
; Orbital elements (P,T,e,A,B,F,G) and a set of observations (ti,xi,yi) 
; are known
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)
; The fitted (xi,yi) positions are determined from the four Thiele-Innes 
; elements and the normalized rectangular coordinates:
;	xi = AXi + FYi
;	yi = BXi + GYi
; Calculate geometric elements a("),i,Omega,omega from (A,F,B,G)
;
; Essentially, the equations one needs to solve are:
;	E - esinE = 2Pi/P * (t - T)
;	x = A*(cos(Ei) - e) + F*(sqrt(1 - e^2)*sin(Ei))
;	y = B*(cos(Ei) - e) + G*(sqrt(1 - e^2)*sin(Ei))
;
; Paramers:
;	Period:	period (P)
;	Tperi: 	time of periastron passage (T)
;	ecc:	eccentricity (e)
;	major:	semi-major axis (in mas) (a)
;	inc:	inclination (i)
;	Omega:	position angle of node
;	omega:	angle between node and periastron
;	ti:	time of observation
;	xd:	RA - data
;	yd:	DEC - data
;	xf:	RA - fit (determined from orbital elements for time of obs.)
;	yf:	DEC - fit (determined from orbital elements for time of obs.)
;	A,B,F,G: Thiele-Innes elements
;
; Note: xi and yi are not the same convention for RA and DEC chosen to make 
; the orbital plots. Specifically, in the fitting the apparent ellipse 
; (ORBFITE.PRO):
;	x = RA = -rho*sin(theta),   y = DEC = rho*cos(theta)
;	(i.e. North is in the direction of the positive y axis, theta is
;	measured counterclockwise from positive y-axis)
; Whereas in this program, the coordinate system set up by Couteau,
;	x = rho*cos(theta),   y = rho*sin(theta)
;	(i.e. North id in the direction of the positive x-axis and the 
;	y-axis points east)
; Both coordinate systems have the same separation and position angle 
; orietations, they are just rotated by 90 deg with respect to each other.
;
; INPUT:
;	data_file: file containing data points in the format:
;		      time  theta  error(theta)  rho  error(rho)
;		      where theta is the position angle and rho the separation
;		      and their respective errors
;		      (enter file name using quotes)
;	num: number of data points contained in data_file
;	
; Prompted inputs:
;	all of the orbital elements: P,T,e,A,B,F,G
;
; Began 22 May 2002
; Modified on 9 July 2002 to include least squares fit
;
; 1 Jan 2008: Modify to make consistent with othe plot_orbit programs
;  - data file includes errors in sep and PA
;  - plot orbit as solid line

!P.font=0
!P.multi=0
!P.charsize=1.2
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
;frac=0.7	;scale factor for adjusting size of plotting symbols
;frac=1.2	;scale factor for adjusting size of plotting symbols
frac=1.5	;scale factor for adjusting size of plotting symbols

ans_nirc2 = 'n'
;time_nirc2 = [2015.5274d]     ; HBC643_SR20
;time_nirc2 = [2014.5089d,2015.5276d]     ; ROX1 - SR2 (BY)
;time_nirc2 = [2014.5073d,2015.5260d]     ; ROX1 - SR2 (JY)
;time_nirc2 = [2014.5090d,2015.5275d]     ; ROXs 5 (BY)
;time_nirc2 = [2014.5074d,2015.5258d]     ; ROXs 5 (JY)
;time_nirc2 = [2014.5090d,2015.5275d]     ; ROXs42C (BY)
;time_nirc2 = [2014.5074d,2015.5259d]     ; ROXs42C (JY)
;time_nirc2 = [2004.5349d,2014.5119d,2015.2600d,2015.5277d]     ; ROXs47A (BY)
;time_nirc2 = [2004.5336d,2014.5103d,2015.2584d,2015.5261d]     ; ROXs47A (JY)
;time_nirc2 = [2014.5119d,2015.5276d]     ; WSB41_SR24N (BY)
;time_nirc2 = [2014.5103d,2015.5260d]     ; WSB41_SR24N (JY)
;time_nirc2 = [2015.2580d]     ; WSB 18 (JY)
;time_nirc2 = [2014.5091d,2014.5119d,2015.2598d]     ; WSB 59
;time_nirc2 = [2014.5091d,2014.5118d,2015.2600d]     ; VSSG 14 (BY)
;time_nirc2 = [2014.5075d,2014.5102d,2015.2583d]     ; VSSG 14 (JY)
;time_nirc2 = [2003.2869d,2004.5349d,2014.5119d,2015.2600d,2015.5276d]     ; V853 Oph (BY)
;time_nirc2 = [2003.2855d,2004.5335d,2014.5104d,2015.2584d,2015.5260d]     ; V853 Oph (JY)
;time_nirc2 = [56924.901d,57279.936d]; atlas

; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1

; Read in measurements from data file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time = dblarr(num)
sepmeas = dblarr(num)
PAmeas = dblarr(num)
dsep = dblarr(num)
dPA = dblarr(num)

openr,lun,data_file,/get_lun

for i=0, num-1 do begin

	readf, lun, temp1, temp2, temp3, temp4, temp5

	time(i) = temp1
	PAmeas(i) = temp2
	dPA(i) = temp3
	sepmeas(i) = temp4;*1000.0;/1000.0	
	dsep(i) = temp5;*1000.0;/1000.0
	
endfor

close,lun

; convert theta and dtheta to radians

PAmeas = PAmeas*!dpi/180
dPA = dPA*!dpi/180

; convert data points to x and y coordinates

xmeas = sepmeas * cos(PAmeas)	; x coordinate
ymeas = sepmeas * sin(PAmeas)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(PAmeas)^2*dsep^2 + sepmeas^2*sin(PAmeas)^2*dPA^2)
dy = sqrt(sin(PAmeas)^2*dsep^2 + sepmeas^2*cos(PAmeas)^2*dPA^2)

if (ans_nirc2 eq 'y') then begin
; Figure out NIRC2 positions
num_nirc2 = n_elements(time_nirc2)
ind_nirc2 = dblarr(num_nirc2)
for i=0, num_nirc2-1 do begin
   tdiff = abs(time - time_nirc2(i))
   ind = where(tdiff eq min(tdiff))
   ind_nirc2(i) = ind(0)
endfor
endif


ans_fit = 'y'
print,'Do you want to fit linear motion in RA/DEC (y/n)?'
read,ans_fit

if (ans_fit eq 'y') then begin

l_regress, time, ymeas, dy, a_y, b_y
l_regress, time, xmeas, dx, a_x, b_x

yfit_lin = a_y(0) + b_y(0)*time
xfit_lin = a_x(0) + b_x(0)*time

print,'Linear motion in RA:'
print,'a: ',a_y
print,'b: ',b_y
print,'Linear motion in DEC:'
print,'a: ',a_x
print,'b: ',b_x

; extrapolate before and after:
trange = max(time) - min(time)
t_ext = findgen(101)/100.0 *21*trange + min(time) - 10*trange
y_ext = a_y(0) + b_y(0)*t_ext
x_ext = a_x(0) + b_x(0)*t_ext

; Compute chi2:
chi2 = total(((xmeas-xfit_lin)^2/dx^2 + (ymeas-yfit_lin)^2/dy^2))

print,'chi2: ',chi2

endif




; Do you want to plot another orbit?
count = 0
answer = 'y'
while (answer eq 'y') do begin

; Prompt user for orbital elements to be entered

selection = ""

print,'If you would like to enter P,T,e and then have the program do a ', $
'least squares fit for a,i,Omega,omega, enter "pte" at prompt.'
print,'If you would like to enter all the orbital elements, ', $
'(P,T,e,a,i,Omega,omega) then enter "all" at the prompt'
read,selection

; Obtain values for P,T,e

period = 0d
Tperi = 0d
ecc = 0d
major = 0d
inc = 0d
W_cap = 0d
w_low = 0d


if (selection eq "pte") then begin

   print,"Enter P,T,e:"
   read,period,Tperi,ecc


   calc_AFBG, time, xmeas, dx, ymeas, dy, period, Tperi, ecc, $
              A_cap, F_cap, B_cap, G_cap, xfit, yfit

  
   ; determine a,i,Omega,omega (using AIWW.PRO)

   aiWw, A_cap,B_cap,F_cap,G_cap,major,inc,W_cap,w_low

   print,"P,T,e,a,i,Omega,omega:"
   print,period,Tperi,ecc,major,(180/!dpi)*inc, (180/!dpi)*W_cap, $
   	(180/!dpi)*w_low

   orbEL = [period, Tperi, ecc, major, inc, W_cap, w_low]

endif else begin 

	if (selection eq "all") then begin

           ; Obtain initial values for P,T,e,a,i,W,w

           print,"Enter P T e a i W w (period, time of periastron passage, eccentricity):"
           read, period, Tperi, ecc, major, inc, W_cap, w_low

           ;convert angles from degrees to radians

           inc = inc*!dpi/180.0
           W_cap = W_cap*!dpi/180.0
           w_low = w_low*!dpi/180.0

           orbEL = [period, Tperi, ecc, major, inc, W_cap, w_low]

	endif else STOP,"incorrect input selection"
endelse 

; OK now plot data and orbital fit


; plot orbit with 1000 points

pnum = 1000.0

tstep = period/pnum

; x points N (DEC)
; y points E (increasing RA)
; therefore plot as (-y,x)

xarr=dblarr(pnum+1)
yarr=dblarr(pnum+1)
tarr=dblarr(pnum+1)

ti = Tperi

for i=0, pnum do begin

    ; compute orbital position at given time:

    orbpos, ti, orbEL, xpos, ypos, tsep, tPA

    xarr(i) = xpos
    yarr(i) = ypos
    tarr(i) = ti

    ti = ti + tstep

endfor

xmax = max([xarr,xmeas])
xmin = min([xarr,xmeas])

ymax = max([yarr,ymeas])
ymin = min([yarr,ymeas])

;sepmax = max([xmax,ymax])
;dl = 0.10*sepmax
dl = 10.0

sepmax = 1.2*max(abs([yarr,xarr]))
;sepmax = 1.05*max(abs([yarr,xarr,ymod,xmod]))
;sepmax = round(1.2*max(abs([yarr,xarr])))
;sepmax = 400.0
;sepmax = 0.5
;sepmax = 0.3
;sepmax=5.0

print,sepmax

if (count eq 0) then begin

   ; Plot once to get fonts correct
   set_plot, 'ps'
   device,/Color,Bits_per_pixel=8,filename='temp.eps',xsize=12,ysize=12,isolatin1=1,/encap
   plot,yarr,xarr,xtitle='!9D!3RA (mas)',ytitle='!9D!3DEC (mas)',ISOTROPIC=1
   device,/close

   set_plot, 'ps'
;   device,/Color,Bits_per_pixel=8,filename='temp.ps',xsize=15,ysize=15 
   device,/Color,Bits_per_pixel=8,filename='temp.eps',xsize=12,ysize=12,isolatin1=1,/encap



    plot,yarr,xarr,xtitle='!9D!3RA (mas)',ytitle='!9D!3DEC (mas)',ISOTROPIC=1, $
;    plot,yarr,xarr,xtitle='!9D!3RA (arcsec)',ytitle='!9D!3DEC (arcsec)',ISOTROPIC=1, $
;    plot,yarr,xarr,xtitle='!9D!3RA (arcsec)',ytitle='!9D!3DEC (arcsec)',ISOTROPIC=1, $
;   plot,yarr,xarr,xtitle='R.A. (!9''''!3)',ytitle='Dec. (!9''''!3)',ISOTROPIC=1, $
;   plot,yarr,xarr,xtitle='!4D!3RA (!9''''!3)',ytitle='!4D!3DEC
;   (!9''''!3)',ISOTROPIC=1, $
;   plot,yarr,xarr,xtitle='!4D!3RA (mas)',ytitle='!4D!3DEC (mas)',ISOTROPIC=1, $
;   plot,yarr,xarr,xtitle="!4D!3RA ('')",ytitle="!4D!3DEC ('')",ISOTROPIC=1, $
;   plot,yarr,xarr,xtitle='E-W Sep (mas)',ytitle='N-S Sep (mas)',ISOTROPIC=1, $
;        xrange=[ymax+dl,ymin-dl], yrange=[xmin-dl,xmax+dl], $
	 xrange=[sepmax,-sepmax], yrange=[-sepmax,sepmax], $
;	 xrange=[max(ymeas)+dl,min(ymeas)-dl], yrange=[min(xmeas)-dl,max(xmeas)+dl], $
;	 xrange=[0.12,-0.10], yrange=[-0.11,0.11], $
; 	 xrange=[-0.15,-0.30], yrange=[-0.04,0.14], $
;         xrange=[55,-55],yrange=[-10,100], $ ; HD 181655 - new
;        xrange=[70,-80],yrange=[-50,100], xtickinterval=50, ytickinterval=50,$ ; HD 181655
; 	 xrange=[400,-100], yrange=[-400,100], $
;	xrange=[60, -50], yrange=[-40,70], $  ; MML 53
; 	 xrange=[300,-200], yrange=[-300,200], $ ; ROX1 - SR2 - new-new
; 	 xrange=[300,-50], yrange=[-300,50], $ ; ROX1 - SR2 - new
; 	 xrange=[400,-200], yrange=[-400,200], $ ; ROX1 - SR2
;	xrange=[170,-170], yrange=[-120,220], $ ; ROXs 5 - alt 
;	xrange=[220,-120], yrange=[-120,220], $ ; ROXs 5 - e 
; 	 xrange=[300,-300], yrange=[-400,200], $ ; ROXs 42C
; 	 xrange=[80,-20], yrange=[-50,50], $ ; ROXs 47A
; 	 xrange=[250,-150], yrange=[-200,200], $ ; WSB41_SR24N
;	xrange=[1300,-100], yrange=[-300,1000], $ ; WSB 18 wide
;	xrange=[80, -80], yrange=[-50,110], $  ; WSB 18 close
;	xrange=[50,-150], yrange=[-100,100], $ ; WSB 59 alt
;	xrange=[150,-100], yrange=[-130,120], $ ; WSB 59
; 	 xrange=[200,-120], yrange=[-160,160], $ ; VSSG 14
; 	 xrange=[50,-200], yrange=[-200,50], $ ; VSSG 14 - alt
;          xrange=[-200,-450], yrange=[-100,150], $ V853 Oph
;    	  xrange=[0.2,-2.0], yrange=[-2.0,0.2], $ ; TWA 3 - zoom
;    	  xrange=[2.0,-2.0], yrange=[-2.0,2.0], $ ; TWA 3 - wide
;    	  xrange=[3.0,-3.0], yrange=[-3.0,3.0], $ ; TWA 3 - wide
;         xrange=[300,-50], yrange=[-300,50], $ ; XZ Tau
   ;      xrange=[100,-200], yrange=[-200,100], $
   ;      xrange=[50,-50], yrange=[-50,50], $
   ;      xrange=[45,-45], yrange=[-45,45], $
   ;      xrange=[3,-3], yrange=[-3,3], $
   ;      xrange=[5,-5], yrange=[-8,2], $
   ;      xrange=[0.1,-0.45], yrange=[-0.25,0.3], $
   ;      xrange=[0.1,-0.22], yrange=[-0.17,0.15], $
   ;      xrange=[0.7,-0.7], yrange=[-0.7,0.7], $
   ;      xrange=[-5850,-5950], yrange=[-50,50], $
   ;      xrange=[50,-200], yrange=[-100,150], $
   ;       xrange=[16,-12], yrange=[-16,12], $ ; HD 28350 - atlas
   ;      xrange=[2.0,-4.0],yrange=[-1.0,5.0], $  ; T Tau NS - wide arcsec
   ;      xrange=[0.100,-0.200],yrange=[-0.800,-0.500], $  ; T Tau NS - zoom - arcsec
   ;      xrange=[2000,-4000],yrange=[-1000,5000], $  ; T Tau NS
   ;      xrange=[1000,-1000],yrange=[-1000,1000], $  ; T Tau NS
   ;      xrange=[500,-500],yrange=[-900,100], $  ; T Tau NS - zoom
   ;      xrange=[100,-200],yrange=[-800,-500], $  ; T Tau NS - zoom
   ;      xrange=[0.1,-1], yrange=[-0.1,1], $ ; WR 133
   ;      xrange=[250,-250], yrange=[-450,50], $ ; L1551 IRS5
        xstyle=1,ystyle=1, /nodata, position=[0.19,0.12,0.95,0.98];,position=[0.17,0.12,0.97,0.98]

   if (ans_fit eq 'y') then begin
      oplot,y_ext,x_ext,linestyle=2
     ; connect data points to fit
      for i=0,num-1 do oplot,[yfit_lin(i),ymeas(i)],[xfit_lin(i),xmeas(i)]
   endif

; mark primary with an asterisk
oplot,[0],[0],psym=2
;sepmax = 200 ; WSB 59
;sepmax = 80 ; ROXs47A
;sepmax = 250 ; SR24N
;sepmax = 200 ; VSSG 14
oplot,[-sepmax,sepmax],[0,0],linestyle=1
oplot,[0,0],[-sepmax,sepmax],linestyle=1

endif

; Define user symbol to be filled rectangle
xrect = frac*[-1.0,-1.0,1.0,1.0,-1.0]
yrect = frac*[-1.0,1.0,1.0,-1.0,-1.0]
usersym,xrect,yrect,/FILL

; Define user symbol to be closed circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)

; Plot predicted positions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;    tpred = findgen(3)*2 + 2021.3
;    tpred = findgen(1) + 2023.0
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;    tpred = findgen(3)*2 + 2021.3
;    tpred = findgen(1) + 2021.3
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;    tpred = findgen(3)*2 + 2020.3
;    tpred = findgen(1) + 2020.3
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;    tpred = findgen(3)*2 + 2018.3
;    tpred = findgen(3)*2 + 2019.3
;    tpred = findgen(1) + 2019.3
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;     tpred = [59467.6271d,59467.7101d,59467.7941d,59471.6269d,59471.7099d,59471.7939d]
;     tpred = [59467.6691d,59468.6691d,59469.6690d,59470.6690d,59471.6689d]
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;    auto_ephemeris,orbEL,tpred,RApred,DECpred,/noprint
;    usersym, frac*cos(Z), frac*sin(Z), thick=2;1      ; black open circle outline
;    usersym,xrect,yrect,/FILL
;    ncol = 2
;    oplot,RApred,DECpred,psym=8,color=ncol             ;,symsize=0.6

; Plot Orbits

if (count eq 0) then ncol=3;1;1;3;1
if (count eq 1) then ncol=3;2;1;2
if (count eq 2) then ncol=3;1;3
if (count ge 3) then ncol=3;0

;oplot,yarr,xarr,linestyle=count
if (count eq 0) then oplot,yarr,xarr,color=ncol;ncol;3;1
if (count eq 1) then oplot,yarr,xarr,color=ncol;,linestyle=1;1;2
if (count eq 2) then oplot,yarr,xarr,color=ncol;,linestyle=1;1;3
if (count ge 3) then oplot,yarr,xarr,color=ncol

; compare measured and computed positions

; compute predicted position on date of observation
;
;xfit = dblarr(num)
;yfit = dblarr(num)
;
;for i=0, num-1 do begin
;
;    ti = time(i)
;
;    orbpos, ti, orbEL, xi, yi, sep, PA
;
;    xfit(i) = xi
;    yfit(i) = yi
;
;    sepfit
;
;endfor

calc_vbfit, orbEL, time, pafit, sepfit

xfit = sepfit * cos(pafit)	; x coordinate
yfit = sepfit * sin(pafit)	; y coordinate

; print O-C residuals (Observed - Calculated)

xOC = xmeas - xfit
yOC = ymeas - yfit

sepOC = sepmeas - sepfit
paOC = (PAmeas - PAfit)*180.0/!dpi
for i=0, num-1 do begin
   if (paOC(i) gt 180.0) then paOC(i) = paOC(i)-360.0
   if (paOC(i) lt -180.0) then paOC(i) = paOC(i)+360.0
endfor
pares_dpa = paOC/(dpa*180.0/!dpi)
sepres_dsep = sepOC/dsep

openw,lun,'xyresiduals'
printf,lun,"O-C residuals (data - fit)"
printf,lun,"date     dx       xO-C   xOC/dx dy      yO-C   yOC/dy"
;printf,lun,"date   dx   xdata - xfit   xOC/dx   dy   ydata - yfit   yOC/dy"
for i=0, num-1 do printf,lun,format='(f11.5,f7.3,f8.3,f8.3,f7.3,f8.3,f8.3)', $
    time(i), dx(i), xOC(i), xOC(i)/dx(i), dy(i), yOC(i), yOC(i)/dy(i)
;for i=0, num-1 do printf,lun,format='(f8.3,f7.3,f8.3,f8.3,f7.3,f8.3,f8.3)', $
;    time(i), dx(i), xOC(i), xOC(i)/dx(i), dy(i), yOC(i), yOC(i)/dy(i)
close,lun

openw,lun,'rtresiduals'
printf,lun,"O-C residuals (data - fit)"
printf,lun,"date     dpa       pa(O-C)   paOC/dpa  dsep    sep(O-C)   sepOC/dsep"
for i=0, num-1 do printf,lun,format='(f8.3,f9.3,f10.3,f10.3,f8.3,f9.3,f9.3)', $
    time(i), dPA(i)*180.0/!dpi, paOC(i), pares_dpa(i), dsep(i), sepOC(i), sepres_dsep(i)
close,lun

openw,lun,'seppa_radec'
printf,lun,'time sep PA RA DEC'
for i=0, num-1 do printf,lun,format='(f8.3,f8.3,f8.3,f8.3,f8.3)', $
    time(i),sepmeas(i),PAmeas(i)*180.0/!dpi,ymeas(i),xmeas(i)
close,lun


;2004.863  0.001  -0.051-96.010  0.000   0.063233.463

; mark secondary at computed time with a cross

ans='y'
print, 'Do you want to plot computed position of binary (y/n)?'
read, ans

if (ans eq 'y') then oplot,[yfit],[xfit],psym=7

; connect computed and measured positions with a line

print, 'Do you want to connect computed and measured positions with a line (y/n)?'
read, ans

if (ans eq 'y') then begin
    for i=0, num-1 do $
        oplot,[yfit(i),ymeas(i)],[xfit(i),xmeas(i)];,color=1;2;1
endif


;Define the symbol to be a closed circle
usersym, frac*cos(Z), frac*sin(Z), /fill

; Plot measured positions

; mark measured position of secondary with a square

plot_err,ymeas,abs(dy),xmeas,abs(dx),num


if (ans_nirc2 eq 'y') then begin
   ; Plot all data as grey circles
   usersym, frac*cos(Z), frac*sin(Z), /fill
   oplot, ymeas, xmeas ,psym=8,color=150
   usersym, frac*cos(Z), frac*sin(Z), thick=2 ; black open circle outline
   oplot, ymeas, xmeas, psym=8

   ; Plot nirc2 data as black circles
   usersym, frac*cos(Z), frac*sin(Z), /fill
   oplot, ymeas(ind_nirc2), xmeas(ind_nirc2), psym=8
endif else begin
   usersym, frac*cos(Z), frac*sin(Z), /fill
   oplot, ymeas, xmeas,psym=8
   ;usersym, frac*cos(Z), frac*sin(Z), /fill
   ;oplot, ymeas, xmeas, psym=8,color=150
   ;usersym, frac*cos(Z), frac*sin(Z), thick=2 ; black open circle outline
   ;oplot, ymeas, xmeas, psym=8
endelse



ans = 'y'

;print, 'Do you want to plot date labels (y/n)?'
;read, ans
;
;if (ans eq 'y') then begin
;
;    ;xyouts,ymeas-0.7,xmeas-0.06,string(format='(f6.1)',time),alignment=0.5
;    ;xyouts,ymeas,xmeas,string(format='(f6.1)',time),alignment=0.5
;
;   xyouts,ymeas(num-3),xmeas(num-3)-0.016,string(format='(f6.1)',time(num-3)),alignment=0.5
;   xyouts,ymeas(num-2)+0.005,xmeas(num-2)-0.004,string(format='(f6.1)',time(num-2)),alignment=1.0
;   xyouts,ymeas(num-1)+0.005,xmeas(num-1)-0.004,string(format='(f6.1)',time(num-1)),alignment=1.0
;
;endif

ans_peri='n'
; Plot periastron
if (ans_peri eq 'y') then begin
   orbpos, Tperi, orbEL, xperi, yperi, sepperi, PAperi
   oplot,[yperi],[xperi],psym=1
endif


ans_point = 'y'
print,'Do you want to plot an additional point (y/n)?'
read,ans_point
while (ans_point eq 'y') do begin
   ;usersym,xrect,yrect,/FILL
   usersym, 1.5*frac*cos(Z), 1.5*frac*sin(Z),thick=4.0,/fill
   time_add = 0.0d
   print,'Enter time for added point:'
   read,time_add
   orbpos, time_add, orbEL, xpos_add, ypos_add, sep_add, tPA_add
   oplot,[ypos_add],[xpos_add],psym=8,color=1
   print,'Do you want to plot an additional point (y/n)?'
   read,ans_point
endwhile

print,'Do you want to plot another orbit (y/n)?'
read,answer
count = count + 1
endwhile


; Plot time labels
xl =  0.07*sepmax
yl =  -0.03*sepmax
;xyouts,ymeas(0) + xl,xmeas(0) + yl,string(format='(f6.1)',time(0)),alignment=1.0
;xyouts,ymeas(num-1) + xl,xmeas(num-1) +yl,string(format='(f6.1)',time(num-1)),alignment=1.0

;xyouts,ymeas(0)-0.1,xmeas(0)-0.1,string(format='(f6.1)',time(0)),alignment=0.0
;xyouts,ymeas(num-1)-0.1,xmeas(num-1)-0.1,string(format='(f6.1)',time(num-1)),alignment=0.0

;xyouts,ymeas(0)-0.1,xmeas(0)-0.06,string(format='(f6.1)',time(0)),alignment=0.0  ; TWA 3
;xyouts,ymeas(num-1)-0.1,xmeas(num-1)-0.06,string(format='(f6.1)',time(num-1)),alignment=0.0  ; TWA 3

;xyouts,ymeas+yl,xmeas+xl,string(format='(f6.1)',time),alignment=0.5;,charsize=1.2

;xyouts,ymeas(0)+20,xmeas(0)-2,string(format='(f6.1)',time(0)),alignment=0.5;,charsize=1.2 ; HBC 243 SR 20
;xyouts,ymeas(1)+0,xmeas(1)+6,string(format='(f6.1)',time(1)),alignment=0.5;,charsize=1.2 ; HBC 243 SR 20
;xyouts,ymeas(2)+0,xmeas(2)+6,string(format='(f6.1)',time(2)),alignment=0.5;,charsize=1.2 ; HBC 243 SR 20
;xyouts,ymeas(3)+0,xmeas(3)+5,string(format='(f6.1)',time(3)),alignment=0.5;,charsize=1.2 ; HBC 243 SR 20
;xyouts,ymeas(4)+0,xmeas(4)-10,string(format='(f6.1)',time(4)),alignment=0.5;,charsize=1.2 ; HBC 243 SR 20

;xyouts,ymeas+5,xmeas-1,string(format='(f6.1)',time),alignment=1.0;,charsize=1.2  ; MML 53

; new
;xyouts,ymeas(0)-15,xmeas(0)-8,string(format='(f6.1)',time(0)),alignment=0.0 ;ROX1 - new
;xyouts,ymeas(1)+15,xmeas(1)-8,string(format='(f6.1)',time(1)),alignment=1.0 ;ROX1 - new
;xyouts,ymeas(6)+15,xmeas(6)-8,string(format='(f6.1)',time(6)),alignment=1.0 ;ROX1 - new
;xyouts,ymeas(num-1)+15,xmeas(num-1)-8,string(format='(f6.1)',time(num-1)),alignment=1.0 ;ROX1 - new

;xyouts,ymeas(0)+30,xmeas(0)-5,string(format='(f6.1)',time(0)),alignment=1.0 ;ROX1
;xyouts,ymeas(num-1),xmeas(num-1)+20,string(format='(f6.1)',time(num-1)),alignment=0.5 ;ROX1

;xyouts,ymeas(0),xmeas(0)-25,string(format='(f6.1)',time(0)),alignment=0.5;,charsize=1.2  ; ROXs 5 - e 
;xyouts,ymeas(1)+10,xmeas(1)-5,string(format='(f6.1)',time(1)),alignment=1.0;,charsize=1.2  ; ROXs 5 - e
;xyouts,ymeas(num-1)+10,xmeas(num-1)-5,string(format='(f6.1)',time(num-1)),alignment=1.0;,charsize=1.2  ; ROXs 5 - e

;xyouts,ymeas(0)-7,xmeas(0)-5,string(format='(f6.1)',time(0)),alignment=0.0;,charsize=1.2  ; ROXs 5 - alt
;xyouts,ymeas(1)-10,xmeas(1)-5,string(format='(f6.1)',time(1)),alignment=0.0;,charsize=1.2  ; ROXs 5 - alt
;xyouts,ymeas(num-1)+9,xmeas(num-1)-3,string(format='(f6.1)',time(num-1)),alignment=1.0;,charsize=1.2  ; ROXs 5 - alt
;;xyouts,ymeas(num-1)+9,xmeas(num-1)+2,string(format='(f6.1)',time(num-1)),alignment=1.0;,charsize=1.2  ; ROXs 5 - alt

;xyouts,ymeas(0) - 10,xmeas(0) + 50,string(format='(f6.1)',time(0)),alignment=1.0 ; ROXs 42C
;xyouts,ymeas(5) + xl,xmeas(5) + yl,string(format='(f6.1)',time(5)),alignment=1.0 ; ROXs 42C
;xyouts,ymeas(num-1) + xl,xmeas(num-1) +yl,string(format='(f6.1)',time(num-1)),alignment=1.0 ; ROXs 42C

;xyouts,ymeas(0)-10,xmeas(0)-30,string(format='(f6.1)',time(0)),alignment=0.5 ;V853Oph
;xyouts,ymeas(num-1),xmeas(num-1)-22,string(format='(f6.1)',time(num-1)),alignment=0.5 ;V853Oph

;xyouts,ymeas(0)+10,xmeas(0)-4,string(format='(f6.1)',time(0)),alignment=1.0 ; VSSG 14
;;xyouts,ymeas(0),xmeas(0)+10,string(format='(f6.1)',time(0)),alignment=1.0 ; VSSG 14
;xyouts,ymeas(2)-10,xmeas(2)-8,string(format='(f6.1)',time(2)),alignment=0.0 ; VSSG 14
;xyouts,ymeas(num-1),xmeas(num-1)-25,string(format='(f6.1)',time(num-1)),alignment=0.5 ; VSSG 14

;xyouts,ymeas(0),xmeas(0)+30,string(format='(f6.1)',time(0)),alignment=0.5 ; WSB41_SR24N
;xyouts,ymeas(3)-10,xmeas(3)+22,string(format='(f6.1)',time(3)),alignment=0.5 ; WSB41_SR24N
;xyouts,ymeas(num-1)+12,xmeas(num-1)-8,string(format='(f6.1)',time(num-1)),alignment=1.0 ; WSB41_SR24N

;xyouts,ymeas(0)-5,xmeas(0)-2,string(format='(f6.1)',time(0)),alignment=0.0;,charsize=1.2  ; WSB 18 close
;xyouts,ymeas(num-1)+5,xmeas(num-1)-2,string(format='(f6.1)',time(num-1)),alignment=1.0;,charsize=1.2  ; WSB 18 close

;xyouts,ymeas(0)-0,xmeas(0)+15,string(format='(f6.1)',time(0)),alignment=0.5 ; WSB 59 alt
;xyouts,ymeas(3)-0,xmeas(3)-20,string(format='(f6.1)',time(3)),alignment=0.5 ; WSB 59 alt
;xyouts,ymeas(0)-0,xmeas(0)-20,string(format='(f6.1)',time(0)),alignment=0.5 ; WSB 59
;xyouts,ymeas(3)-0,xmeas(3)+15,string(format='(f6.1)',time(3)),alignment=0.5 ; WSB 59

;xyouts,ymeas(0)-15,xmeas(0)-5,string(format='(f6.1)',time(0)),alignment=0.0;,charsize=1.2  ; XZ Tau
;xyouts,ymeas(num-1)+10,xmeas(num-1)-5,string(format='(f6.1)',time(num-1)),alignment=1.0;,charsize=1.2  ; XZ Tau

;; NRM position for HD 113449
;pa_nrm = 183.0 * !dpi/180.0
;sep_nrm = 21.0
;x_nrm = sep_nrm * cos(PA_nrm)	; x coordinate
;y_nrm = sep_nrm * sin(PA_nrm)	; y coordinate
;usersym, 1.5*frac*cos(Z), 1.5*frac*sin(Z),thick=4.0
;oplot,[y_nrm],[x_nrm],psym=8,color=3

; overplot first measurment for TWA 3
;pa_1989 = 230.0 * !dpi/180.0
;sep_1989 = 1.4
;x_1989 = sep_1989 * cos(PA_1989)	; x coordinate
;y_1989 = sep_1989 * sin(PA_1989)	; y coordinate
;usersym, frac*cos(Z), frac*sin(Z),thick=4.0
;oplot,[y_1989],[x_1989],psym=8


;xyouts,0.08,0.12,'HD 285931, vB 96'
;xyouts,0.05,0.25,'56 Persei'
;xyouts,-2.0,2.5,'GJ 86'
;xyouts,-1.3,1.7,'TWA 3'
;xyouts,-1.6,-1.9,'TWA 3'


ptitle = ' '
print,'Enter plot title:'
read,ptitle
xyouts,sepmax - 0.1*sepmax,-sepmax + 0.1*sepmax,ptitle,charsize=1.5
;xyouts,sepmax - 0.1*sepmax,sepmax - 0.1*sepmax,ptitle,charsize=1.5
;xyouts,-sepmax + 0.1*sepmax,-sepmax + 0.1*sepmax,ptitle,charsize=1.5,alignment=1.0
;xyouts,370, 35,ptitle,charsize=1.5
;xyouts,50,48,ptitle,charsize=1.5  ; HBC643_SR20
;xyouts,-18,-35,ptitle,charsize=1.5  ; MML 53
;xyouts,360, -360,ptitle,charsize=1.5 ; ROX1 - SR2
;xyouts,280, -280,ptitle,charsize=1.5 ; ROX1 - SR2 - new
;xyouts,280, 150,ptitle,charsize=1.5 ; ROX1 - SR2 - new-new
;xyouts,200,180,ptitle,charsize=1.5  ; ROXs 5 - e
;xyouts,-70,-100,ptitle,charsize=1.5  ; ROXs 5 - alt
;xyouts,280, 140,ptitle,charsize=1.5,alignment=0.0 ; ROXs 42C
;xyouts,70, -45,ptitle,charsize=1.5,alignment=0.0 ; ROXs 47A
;xyouts,180, 130,ptitle,charsize=1.5 ; VSSG 14
;xyouts,-215, 120,'V853 Oph B,A',charsize=1.5 ; V853 Oph
;xyouts,235, 160,ptitle,charsize=1.5 ; WSB41_SR24N
;xyouts,1200,850,ptitle,charsize=1.5  ; WSB 18 wide
;xyouts,70,90,ptitle,charsize=1.5  ; WSB 18 close
;xyouts,-90,80,ptitle,charsize=1.5,alignment=0.0 ;'WSB 59 alt
;xyouts,130,90,ptitle,charsize=1.5,alignment=0.0 ;'WSB 59
;xyouts,100,15,ptitle,charsize=1.5  ; XZ Tau
;xyouts, -0.1,-0.2,ptitle,charsize=1.5 ; TWA 3
;xyouts,50,90,'HD 181655',charsize=1.2 ; HD 181655

!P.Charsize=1.0

device, /close

;determine initial chi squared
chi2 = total(((xmeas-xfit)/dx)^2 + ((ymeas-yfit)/dy)^2)
print, "chi2", chi2

device,filename='temp_radec.ps',isolatin1=1

plot,time,ymeas,xtitle='!3Date',ytitle='RA',psym=4
oplot,time,yfit
plot,time,xmeas,xtitle='!3Date',ytitle='DEC',psym=4
oplot,time,xfit

device, /close
set_plot, 'x'

xOC = xmeas - xfit
yOC = ymeas - yfit

openw,lun,'xyresiduals'
printf,lun,"O-C residuals (data - fit)"
printf,lun,"date     dx       xO-C   xOC/dx dy      yO-C   yOC/dy"
;printf,lun,"date   dx   xdata - xfit   xOC/dx   dy   ydata - yfit   yOC/dy"
for i=0, num-1 do printf,lun,format='(f11.5,f7.3,f8.3,f8.3,f7.3,f8.3,f8.3)', $
    time(i), dx(i), xOC(i), xOC(i)/dx(i), dy(i), yOC(i), yOC(i)/dy(i)
;for i=0, num-1 do printf,lun,format='(f8.3,f7.3,f8.3,f8.3,f7.3,f8.3,f8.3)', $
;    time(i), dx(i), xOC(i), xOC(i)/dx(i), dy(i), yOC(i), yOC(i)/dy(i)
close,lun



set_plot, 'ps'
;device, /Color, Bits_per_pixel=8, filename='temp.ps',xsize=15,ysize=13.5
device, /Color, Bits_per_pixel=8, /encap, filename='temp_res.eps',xsize=18,ysize=15

!P.Multi=[0,1,2]

if time(0) gt 10000.0 then begin
   dt = 50000.0 
   datelabel = 'HJD - 50,000'
endif else begin
   dt = 0.0
   datelabel = 'Date'
endelse

tdiff = max(time-dt) - min(time-dt)
trange = [min(time-dt)-0.05*tdiff,max(time-dt)+0.05*tdiff]
seprange = max(abs([sepOC+dsep,sepOC-dsep]))*1.1
parange = max(abs([paOC+dpa*180.0/!dpi,paOC-dpa*180.0/!dpi]))*1.1

parange = 40.0 ; HD 181655
seprange = 16.0 ; HD 181655

Z = findgen(17) * (!pi*2/16.)
usersym, cos(Z), sin(Z), /fill

ploterror, time-dt, paOC,dpa*180.0/!dpi,psym=8,xtitle=datelabel,ytitle='!3 Residuals in PA (deg)',yrange=[-parange,parange],xrange=trange,xstyle=1,ystyle=1 
oplot,trange,[0,0],linestyle=1

ploterror, time-dt, sepOC,dsep,psym=8,xtitle=datelabel,ytitle='Residuals in sep (mas)',yrange=[-seprange,seprange],xrange=trange,xstyle=1,ystyle=1
oplot,trange,[0,0],linestyle=1

device, /close
set_plot, 'x'

!P.multi = 0

free_lun,lun

;print,'Difference between linear and orbital fits:'
;for i=0, num-1 do begin
;
;   diff = sqrt((xfit(i) - xfit_lin(i))^2 + (yfit(i) - yfit_lin(i))^2)
;   print,time(i),xfit(i) - xfit_lin(i),yfit(i) - yfit_lin(i),diff
;
;endfor

;print,tpred

;stop,'Type .cont to continue'

end

;
;
;

pro orbpos, time, orbEL, x, y, sep, PA

; Given set of orbital parameters, compute position at a given time
;
; INPUT:
; time
; orbEL = [period, Tperi, ecc, major, inc, W_cap, w_low]
;
; OUTPUT:
; (x,y)
; (sep, PA)

period = orbEL(0)
Tperi = orbEL(1)
ecc = orbEL(2)
major = orbEL(3)
inc = orbEL(4)
W_cap = orbEL(5)
w_low = orbEL(6)

; Determine the eccentric anomalies Ei: 
;     mu(ti - T) = Ei - esin(Ei)  ... Kepler's Equation
;     where mu = 360/P = 2*pi/P
; Solve this transcendental equation through an 
; iterative procedure.
; Use SOLVE_TRANS.PRO

; array to hold Ei's (the eccentric anomaly)

Ei = 0d

mu = 2*!dpi/period

Mi = mu*(time - Tperi) 

; reduce to same epoch

Mi = 2*!dpi*((time - Tperi)/period $	
	- fix((time - Tperi)/period))

solve_trans,ecc,Mi,Eit

Ei = Eit

; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)

Xi = cos(Ei) - ecc
Yi = sqrt(1 - ecc^2)*sin(Ei)

; Compute the four Thiele-Innes elements

A_cap = major*(cos(w_low)*cos(W_cap) - sin(w_low)*sin(W_cap)*cos(inc))
B_cap = major*(cos(w_low)*sin(W_cap) + sin(w_low)*cos(W_cap)*cos(inc))
F_cap = major*(-sin(w_low)*cos(W_cap) - cos(w_low)*sin(W_cap)*cos(inc))
G_cap = major*(-sin(w_low)*sin(W_cap) + cos(w_low)*cos(W_cap)*cos(inc))

; Compute coordinates of secondary

;	xi = AXi + FYi
;	yi = BXi + GYi

x = A_cap*Xi + F_cap*Yi
y = B_cap*Xi + G_cap*Yi

; convert (x,y) to (PAfit,sepfit)
; x = rho*sin(theta), y=rho*sin(theta)

sep = sqrt(x^2 + y^2)

; Get correct sign on PA by entering y,x separately into atan:
PA = atan(y,x)

if (PA lt 0.0) then PA = 2*!dpi + PA

end

;
;
;

pro plot_err, x, dx, y, dy, N

; plot error bars associated with each data point
; Input
;	x: array of x values
;	dx: array of error in x values
;	y: array of y values
;	dy: erray of error in y values
;	N: number of data points


for i=0, N-1 do begin

	oplot, [x(i)-dx(i), x(i)+dx(i)], [y(i), y(i)]
	oplot, [x(i), x(i)], [y(i)-dy(i), y(i)+dy(i)]

endfor

end
