pro plot_orbit_ellerr, data_file, num

; Plot orbit.  Use elliptical uncertainties.

; Given the 7 orbital elements (P,T,e,A,B,F,G) or (P,T,e,a,i,Omega,omega)
; plot orbit and compare with measurements
;
; Options include entering P,T,e and then doing a least squares fit for
; a,i,Omega,omega or entering (P,T,e,a,i,Omega,omega)
; [(P,T,e,A,B,F,G) entering to be added at a later date]
;
; Read times of observations and measurements in from data file
;	
; Equations:  
; Orbital elements (P,T,e,A,B,F,G) and a set of observations (ti,xi,yi) 
; are known
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)
; The fitted (xi,yi) positions are determined from the four Thiele-Innes 
; elements and the normalized rectangular coordinates:
;	xi = AXi + FYi
;	yi = BXi + GYi
; Calculate geometric elements a("),i,Omega,omega from (A,F,B,G)
;
; Essentially, the equations one needs to solve are:
;	E - esinE = 2Pi/P * (t - T)
;	x = A*(cos(Ei) - e) + F*(sqrt(1 - e^2)*sin(Ei))
;	y = B*(cos(Ei) - e) + G*(sqrt(1 - e^2)*sin(Ei))
;
; Paramers:
;	Period:	period (P)
;	Tperi: 	time of periastron passage (T)
;	ecc:	eccentricity (e)
;	major:	semi-major axis (in mas) (a)
;	inc:	inclination (i)
;	Omega:	position angle of node
;	omega:	angle between node and periastron
;	ti:	time of observation
;	xd:	RA - data
;	yd:	DEC - data
;	xf:	RA - fit (determined from orbital elements for time of obs.)
;	yf:	DEC - fit (determined from orbital elements for time of obs.)
;	A,B,F,G: Thiele-Innes elements
;
; Note: xi and yi are not the same convention for RA and DEC chosen to make 
; the orbital plots. Specifically, in the fitting the apparent ellipse 
; (ORBFITE.PRO):
;	x = RA = -rho*sin(theta),   y = DEC = rho*cos(theta)
;	(i.e. North is in the direction of the positive y axis, theta is
;	measured counterclockwise from positive y-axis)
; Whereas in this program, the coordinate system set up by Couteau,
;	x = rho*cos(theta),   y = rho*sin(theta)
;	(i.e. North id in the direction of the positive x-axis and the 
;	y-axis points east)
; Both coordinate systems have the same separation and position angle 
; orietations, they are just rotated by 90 deg with respect to each other.
;
; INPUT:
;	data_file: file containing data points in the format:
;		      time  PA  sep  dmajor  dminor  PAerr
;		   where theta is the position angle and rho is the separation.
;                  The error ellipse is defined by the size of the major and 
;                  minor axes, and the position angle of the major axis in the
;                  plane of the sky (measured East of North).
;		   (enter file name using quotes)
;	num: number of data points contained in data_file
;	
; Prompted inputs:
;	all of the orbital elements: P,T,e,A,B,F,G
;
; Began 22 May 2002
; Modified on 9 July 2002 to include least squares fit
;
; 1 Jan 2008: Modify to make consistent with othe plot_orbit programs
;  - data file includes errors in sep and PA
;  - plot orbit as solid line

!P.font=0
!P.multi=0
!P.charsize=1.0
!P.charthick=2.0
!P.thick=3.0
!X.thick=2.0
!Y.thick=2.0
;frac=0.7	;scale factor for adjusting size of plotting symbols
frac=1.4	;scale factor for adjusting size of plotting symbols

;!P.font=0
;!P.multi=0
;!P.charsize=1.0
;!P.charthick=3.0
;!P.thick=3.0
;!X.thick=3.0
;!Y.thick=3.0
;;frac=0.7	;scale factor for adjusting size of plotting symbols
;frac=1.4	;scale factor for adjusting size of plotting symbols

; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1

; Read in measurements from data file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d
temp6 = 0d

time = dblarr(num)
sepmeas = dblarr(num)
PAmeas = dblarr(num)
dmajor = dblarr(num)
dminor = dblarr(num)
PAerr = dblarr(num)

openr,lun,data_file,/get_lun

for i=0, num-1 do begin

	readf, lun, temp1, temp2, temp3, temp4, temp5, temp6

	time(i) = temp1
	PAmeas(i) = temp2
	sepmeas(i) = temp3	
	dmajor(i) = temp4
	dminor(i)= temp5
	PAerr(i) = temp6
	
endfor

close,lun

; convert theta and theta_err to radians

PAmeas = PAmeas*!dpi/180
PAerr = PAerr*!dpi/180

; convert data points to x and y coordinates

xmeas = sepmeas * cos(PAmeas)	; x coordinate
ymeas = sepmeas * sin(PAmeas)	; y coordinate

; project separation into components along the direction of the error ellipse

xpmeas = sepmeas*cos(PAmeas - PAerr)  ; x' (major)
ypmeas = sepmeas*sin(PAmeas - PAerr)  ; y' (minor) 


; Obtain values for P,T,e

period = 0.0d
Tperi = 0.0d
ecc = 0.0d
major = 0.0d
inc = 0.0d
W_cap = 0.0d
w_low = 0.0d

; Do you want to plot another orbit?
count = 0
answer = 'y'
while (answer eq 'y') do begin

   ; Obtain initial values for P,T,e,a,i,W,w

   print,"Enter P T e a i W w (period, time of periastron passage, eccentricity):"
   read, period, Tperi, ecc, major, inc, W_cap, w_low

   ;convert angles from degrees to radians

   inc = inc*!dpi/180.0
   W_cap = W_cap*!dpi/180.0
   w_low = w_low*!dpi/180.0

   orbEL = [period, Tperi, ecc, major, inc, W_cap, w_low]

; OK now plot data and orbital fit


; plot orbit with 1000 points

pnum = 1000.0

tstep = period/pnum

; x points N (DEC)
; y points E (increasing RA)
; therefore plot as (-y,x)

xarr=dblarr(pnum+1)
yarr=dblarr(pnum+1)
tarr=dblarr(pnum+1)

ti = Tperi

for i=0, pnum do begin

    ; compute orbital position at given time:

    orbpos, ti, orbEL, xpos, ypos, tsep, tPA

    xarr(i) = xpos
    yarr(i) = ypos
    tarr(i) = ti

    ti = ti + tstep

endfor

orbpos, Tperi, orbEL, xperi, yperi, tsep,tPA

;stop,'Type .cont to continue'

xmax = max(xarr)
xmin = min(xarr)

ymax = max(yarr)
ymin = min(yarr)

sepmax = max([xmax,ymax])
;sepmax = round(1.2*max(abs([yarr,xarr])))
sepmax = 1.2*max(abs([yarr,xarr]))
;dl = 0.10*sepmax
dl = 0.20*sepmax

;sepmax = 12.0

if (count eq 0) then begin

   set_plot, 'ps'
   device,/Color,Bits_per_pixel=8,filename='temp.eps',xsize=12,ysize=12,isolatin1=1,/encap

;   plot,yarr,xarr,xtitle='R.A. (!9''''!3)',ytitle='Dec. (!9''''!3)',ISOTROPIC=1, $
;   plot,yarr,xarr,xtitle='!4D!3RA (!9''''!3)',ytitle='!4D!3DEC (!9''''!3)',ISOTROPIC=1, $
;   plot,yarr,xarr,xtitle='E-W Sep (mas)',ytitle='N-S Sep (mas)',ISOTROPIC=1, $
    plot,yarr,xarr,xtitle='!9D!3RA (mas)',ytitle='!9D!3DEC (mas)',ISOTROPIC=1, $
;   plot,yarr,xarr,xtitle='!4D!3RA (mas)',ytitle='!4D!3DEC (mas)',ISOTROPIC=1, $
     	 xrange=[sepmax,-sepmax], yrange=[-sepmax,sepmax], $
;	 xrange=[3.0,-3.0], yrange=[-4,2], $
;	 xrange=[2.0,-2.0], yrange=[-1.8,2.2], $ ; HD 73731
;	 xrange=[12.0,-2.0], yrange=[-12.0,2.0], $ ; WR 140
;	 xrange=[7.8,4.6], yrange=[-12.2,-9.0], $  ; zoom WR 140
;	 xrange=[12.0,-7.0], yrange=[-9.5,9.5], $ ; WR 137
;         xrange=[100,-160],yrange=[-160,100], $ ; Polaris
;         xrange=[4,-4], yrange=[-4,4], $
         ;xrange=[-4,6], yrange=[-8,2], $
        ;xrange=[1.5,-1.5], yrange=[-1.5,1.5], $
        xstyle=1,ystyle=1, /nodata, position=[0.19,0.12,0.95,0.98];xstyle=1,ystyle=1, /nodata, position=[0.14,0.12,0.96,0.99]

endif

; mark primary with an asterisk
oplot,[0],[0],psym=2

; Mark x and y axes
;oplot,[4,-6],[0,0],linestyle=1
;oplot,[0,0],[-8,2],linestyle=1


col=[3,2,1]
lines = [0,2,4,1]
oplot,yarr,xarr,linestyle=count,color=3
;oplot,yarr,xarr,linestyle=lines(count),color=3;col(count)

; compare measured and computed positions

; compute predicted position on date of observation

xfit = dblarr(num)
yfit = dblarr(num)

xpfit = dblarr(num)
ypfit = dblarr(num)

for i=0, num-1 do begin

    ti = time(i)

    orbpos, ti, orbEL, xi, yi, sep, PA

    xfit(i) = xi
    yfit(i) = yi

    xpfit(i) = sep*cos(PA - PAerr(i))
    ypfit(i) = sep*sin(PA - PAerr(i))

endfor

; Compute chi2
chi2 = total((xpmeas - xpfit)^2/dmajor^2 + (ypmeas - ypfit)^2/dminor^2)
print,"chi2 (x',y'): ",chi2
; assume equal uncertainties in x and y:
chi2_eq = total((xmeas-xfit)^2/dmajor^2 + (ymeas-yfit)^2/dmajor^2)
print, "chi2(x,y) - equal uncertainties:", chi2_eq

; print O-C residuals (Observed - Calculated)

xOC = xpmeas - xpfit
yOC = ypmeas - ypfit

openw,lun,'xyresiduals'
printf,lun,"O-C residuals (data - fit)"
printf,lun,"date     dx       xO-C   xOC/dx dy      yO-C   yOC/dy"
;printf,lun,"date   dx   xdata - xfit   xOC/dx   dy   ydata - yfit   yOC/dy"
;chi2 = total((xpmeas - xpfit)^2/dmajor^2 + (ypmeas - ypfit)^2/dminor^2)
for i=0, num-1 do printf,lun,format='(f11.5,f7.3,f8.3,f8.3,f7.3,f8.3,f8.3)', $
    time(i), dmajor(i), xOC(i), xOC(i)/dmajor(i), dminor(i), yOC(i), yOC(i)/dminor(i)
;for i=0, num-1 do printf,lun,format='(f11.5,f7.3,f8.3,f8.3,f7.3,f8.3,f8.3)', $
;    time(i), dx(i), xOC(i), xOC(i)/dx(i), dy(i), yOC(i), yOC(i)/dy(i)
;;for i=0, num-1 do printf,lun,format='(f8.3,f7.3,f8.3,f8.3,f7.3,f8.3,f8.3)', $
;;    time(i), dx(i), xOC(i), xOC(i)/dx(i), dy(i), yOC(i), yOC(i)/dy(i)
close,lun

;openw,lun,'seppa_radec'
;printf,lun,'time sep PA RA DEC'
;for i=0, num-1 do printf,lun,format='(f8.3,f8.3,f8.3,f8.3,f8.3)', $
;    time(i),sepmeas(i),PAmeas(i)*180.0/!dpi,ymeas(i),xmeas(i)
;close,lun


;2004.863  0.001  -0.051-96.010  0.000   0.063233.463

; mark secondary at computed time with a cross

ans='y'
print, 'Do you want to plot computed position of binary (y/n)?'
read, ans

if (ans eq 'y') then oplot,[yfit],[xfit],psym=7

; Plot periastron
;oplot,[yperi],[xperi],psym=7

; connect computed and measured positions with a line

print, 'Do you want to connect computed and measured positions with a line (y/n)?'
read, ans

if (ans eq 'y') then begin
    for i=0, num-1 do $
        oplot,[yfit(i),ymeas(i)],[xfit(i),xmeas(i)],linestyle=1,thick=2,color=0;2
endif

; Define user symbol to be filled rectangle
xrect = frac*[-1.0,-1.0,1.0,1.0,-1.0]
yrect = frac*[-1.0,1.0,1.0,-1.0,-1.0]
usersym,xrect,yrect,/FILL

; Define user symbol to be closed circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)

;Define the symbol to be a closed circle
usersym, frac*cos(Z), frac*sin(Z), /fill

ans_point = 'y'
print,'Do you want to plot an additional point (y/n)?'
read,ans_point
pcount=0
; plot prediction track instead of point
ans_track = 'n'
while (ans_point eq 'y') do begin
   ; Define user symbol to be filled rectangle
   ;xrect = 1.5*frac*[-1.0,-1.0,1.0,1.0,-1.0]
   ;yrect = 1.5*frac*[-1.0,1.0,1.0,-1.0,-1.0]
   ;usersym,xrect,yrect,/FILL
   usersym, frac*cos(Z), frac*sin(Z),thick=4.0;,/fill
   time_add = 0.0d
   print,'Enter time for added point:'
   read,time_add
   ; If plotting line segments instead of points:
   if (ans_track eq 'y') then begin
      time_start = time_add
      time_stop = 0.0d
      print,'Enter stop time'
      read,time_stop
      ntrack=100
      time_diff = time_stop - time_start
      time_add = time_start + findgen(ntrack+1)*time_diff/ntrack
      usersym, frac*cos(Z), frac*sin(Z),thick=4.0,/fill
   endif
   ;orbpos, time_add, orbEL, xpos_add, ypos_add, sep_add, tPA_add
   calc_vbfit, orbEL, time_add, PA_add, sep_add
   ypos_add = sep_add*sin(PA_add) ; ra_pred
   xpos_add = sep_add*cos(PA_add) ; dec_pred
   oplot,[ypos_add],[xpos_add],psym=8,color=1
;   xyouts,ypos_add+0.2,xpos_add+0.2,pcount,alignment=0.5
;   if (pcount eq 1) then xyouts,ypos_add-0.35,xpos_add-0.15,'2013Oct22',alignment=0.0,charsize=0.9,color=1
;   if (pcount eq 6) then xyouts,ypos_add+0.0,xpos_add+0.4,'2013Nov01',alignment=0.5,charsize=0.9,color=1
;   if (pcount eq 10) then xyouts,ypos_add+0.35,xpos_add-0.15,'2013Nov09',alignment=1.0,charsize=0.9,color=1
;   if (pcount eq 1) then xyouts,ypos_add-0.35,xpos_add-0.15,'2013Oct20',alignment=0.0,charsize=0.9,color=1

   ;if (pcount eq 1) then xyouts,ypos_add+0.35,xpos_add-0.15,'2013Nov02',alignment=1.0,charsize=0.9,color=1
   ;if (pcount eq 2) then xyouts,ypos_add+0.35,xpos_add-0.15,'2013Nov10',alignment=1.0,charsize=0.9,color=1

   ;if (pcount eq 0) then xyouts,ypos_add-2,xpos_add-1,'2023Oct15',alignment=0.0,charsize=0.9,color=1
   ;if (pcount eq 1) then xyouts,ypos_add+2,xpos_add-1,'2023Nov15',alignment=1.0,charsize=0.9,color=1
   ;if (pcount eq 2) then xyouts,ypos_add-2,xpos_add-1,'2023Dec15',alignment=0.0,charsize=0.9,color=1
   pcount = pcount+1
   print,'Do you want to plot an additional point (y/n)?'
   read,ans_point
endwhile






;; Re-plot measured points
;; Define the symbol to be a closed circle
;usersym, frac*cos(Z), frac*sin(Z), /fill
;; Plot measured positions
;; mark measured position of secondary with a square
;oplot,[ymeas],[xmeas],psym=8
;plot_err,ymeas,abs(dy),xmeas,abs(dx),num


print,'Do you want to plot another orbit (y/n)?'
read,answer
count = count + 1
endwhile


; Define user symbol to be closed circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)

;Define the symbol to be a closed circle
usersym, frac*cos(Z), frac*sin(Z), /fill
;usersym, cos(Z), sin(Z), thick=1

; Plot measured positions of secondary
oplot,[ymeas],[xmeas],psym=8
;plot_err,ymeas,abs(dy),xmeas,abs(dx),num

; Plot error ellipses
for i=0, num-1 do begin

   create_ellipse,xe,ye,100,major=dmajor(i),minor=dminor(i),xcent=xmeas(i),ycent=ymeas(i),theta=PAerr(i)*180.0/!dpi

   oplot,ye,xe,thick=4,color=1

   ;stop,'Type .cont to continue'

endfor

; Compare Polaris measurements
;usersym,xrect,yrect,/FILL
;oplot,[ymeas(5:12)],[xmeas(5:12)],psym=8,color=3
;usersym, frac*cos(Z), frac*sin(Z), /fill
;oplot,[ymeas(13:20)],[xmeas(13:20)],psym=8,color=2

;xyouts,-2.5,1.0,'Sig Ori Aa,Ab'

ans = 'n'

;print, 'Do you want to plot date labels (y/n)?'
;read, ans

if (ans eq 'y') then begin

    ;xyouts,ymeas-0.7,xmeas-0.06,string(format='(f6.1)',time),alignment=0.5
    ;xyouts,ymeas,xmeas,string(format='(f6.1)',time),alignment=0.5

   ;;xsize=ysize=15
   ;xyouts,ymeas(0),xmeas(0)-0.6,'2010Nov05',alignment=0.5
   ;xyouts,ymeas(1)-1.5,xmeas(1)-0.13,'2011Sep29',alignment=0.5
   ;xyouts,ymeas(2)+1.5,xmeas(2)-0.13,'2011Dec09',alignment=0.5

endif

;xyouts,ymeas(5)-1.0,xmeas(5)-0.4,'2023Jul06 - mystic',charsize=0.9
;xyouts,ymeas(6)+1.0,xmeas(6)-0.4,'2023Jul06 - silmaril',charsize=0.9,alignment=1.0
;xyouts,ymeas(6)-1.0,xmeas(6)-0.4,'2023Jul06 - silmaril',charsize=0.9,alignment=0.0


; Experiment with rotating axes:
ans_rot = 'n'
if (ans_rot eq 'y') then begin

   iselect = 8

   ; RA axis:
   y_axis_y = 8.0*(findgen(201)/100.0 - 1.0)
   y_axis_x = dblarr(201)

   ; DEC axis:
   x_axis_y = dblarr(201)
   x_axis_x = 8.0*(findgen(201)/100.0 - 1.0)

   oplot,y_axis_y,y_axis_x,linestyle=1
   oplot,x_axis_y,x_axis_x,linestyle=1

   ; Rotate axes lines by dPAerr(num-1)
   PArot = PAerr(iselect)
   ;PArot = PAerr(0) 
   ;PArot = -30.0*!dpi/180.0

   xp_axis_x = x_axis_x*cos(PArot) - x_axis_y*sin(PArot)
   xp_axis_y = x_axis_x*sin(PArot) + x_axis_y*cos(PArot)

   yp_axis_x = y_axis_x*cos(PArot) - y_axis_y*sin(PArot)
   yp_axis_y = y_axis_x*sin(PArot) + y_axis_y*cos(PArot)

   oplot,yp_axis_y,yp_axis_x,linestyle=2
   oplot,xp_axis_y,xp_axis_x,linestyle=2

   print,'PAerr: ',PArot*180.0/!dpi

   ; Plot projections onto RA and DEC axes:

   xd = xmeas(iselect)
   yd = ymeas(iselect)
   oplot,[0,0],[0,xd],color=2
   oplot,[yd,yd],[0,xd],color=2
   oplot,[yd,0],[0,0],color=2
   oplot,[yd,0],[xd,xd],color=2

   xpd = xd*cos(PArot) - xd*sin(PArot)
   ypd = xd*sin(PArot) + xd*cos(PArot)

   print,'xd yd:',xd,yd
   ;print,'Rotated xpd ypd:',xpd,ypd

   xpd = xd*cos(PArot) + xd*sin(PArot)
   ypd = -xd*sin(PArot) + xd*cos(PArot)

   ;print,'Rotated xpd ypd:',xpd,ypd
  
   print,'Rotated:', sepmeas(iselect)*cos(Pameas(iselect) - PAerr(iselect)), sepmeas(iselect)*sin(Pameas(iselect) - PAerr(iselect))

endif

; Draw arrow to indicate direction of motion
;tarrow = [56270.813,56586.948]   ; [2012.9,2013.8]
;tarrow = [56570.0,56580.0]  
;calc_vbfit, orbEL, tarrow, PAarrow, separrow
;xarrow = separrow * cos(PAarrow)	; x coordinate
;yarrow = separrow * sin(PAarrow)	; y coordinate
;arrow_internal,yarrow(0)-0.9,xarrow(0),yarrow(1)-0.9,xarrow(1),/data,thick=3,hthick=3,hsize=-0.3;/solid
;arrow_ghs,yarrow(0)-0.8,xarrow(0),yarrow(1)-0.8,xarrow(1),/data,thick=3,hthick=3,hsize=-0.4,hangle=15.0,/solid

;ti = 56570.0
;tf = 56580.0
;tarrow = findgen(11.0)/10.0 * (tf-ti) + ti
;calc_vbfit, orbEL, tarrow, PAarrow, separrow
;xarrow = separrow * cos(PAarrow)	; x coordinate
;yarrow = separrow * sin(PAarrow)	; y coordinate
;oplot,yarrow-0.8,xarrow,thick=3

ptitle = ' '
print,'Enter plot title:'
read,ptitle
xyouts,sepmax - 0.1*sepmax,-sepmax + 0.1*sepmax,ptitle,charsize=1.2 ; bottom-right
;xyouts,sepmax - 0.1*sepmax,sepmax - 0.2*sepmax,ptitle,charsize=1.2 ; top right

!P.Charsize=1.0

device, /close

;;determine initial chi squared
;chi2 = total(((xmeas-xfit)/dx)^2 + ((ymeas-yfit)/dy)^2)
;print, "chi2", chi2

device,filename='temp_radec.ps',xsize=12,ysize=12

; NOTE: because times are in MJD, idl plots as scientific notation to 
; get \times sign to plot.  However, then the axes fonts stays in greek!

plot,time - min(time),ymeas,xtitle='MJD - MJD_0',psym=8
;oplot,trange,[0,0],linestyle=2

plot,time - min(time),xmeas,xtitle='MJD - MJD_0',psym=8

device, /close


; Plot residuals
device,filename='temp_res.eps',/encap,xsize=12,ysize=15

!P.Multi=[0,1,2]

xr = max(abs([ymeas-yfit,xmeas-xfit]))
xpr = 0.6; max(abs([ypmeas-ypfit,xpmeas-xpfit]))
;xpr = 0.05

tr = [5400,6800]

;plot,time - 50000.0d,ymeas-yfit,xtitle='MJD - 50,000',ytitle='!9D!3RA!dDATA!n - !9D!3RA!dFIT!n',psym=8,yrange=[-xr,xr]
;oplot,[time(0)-1000,time(num-1)+1000] - 50000.0d,[0.0,0.0],linestyle=1

;plot,time - 50000.0d,xmeas-xfit,xtitle='MJD - 50,000',ytitle='!9D!3DEC!dDATA!n - !9D!3DEC!dFIT!n',psym=8,yrange=[-xr,xr]
;oplot,[time(0)-1000,time(num-1)+1000] - 50000.0d,[0.0,0.0],linestyle=1


ploterror,time - 50000.0d,ypmeas-ypfit,dmajor,xtitle='HJD - 2,450,000',ytitle='Residuals - major (mas)',psym=8,yrange=[-xpr,xpr],xrange=tr
oplot,[time(0)-1000,time(num-1)+1000] - 50000.0d,[0.0,0.0],linestyle=1

ploterror,time - 50000.0d,xpmeas-xpfit,dminor,xtitle='HJD - 24,50,000',ytitle='Residuals - minor (mas)',psym=8,yrange=[-xpr,xpr],xrange=tr
oplot,[time(0)-1000,time(num-1)+1000] - 50000.0d,[0.0,0.0],linestyle=1

device, /close
set_plot, 'x'

free_lun,lun

stop,'Type .cont to continue'

end

;
;
;

pro orbpos, time, orbEL, x, y, sep, PA

; Given set of orbital parameters, compute position at a given time
;
; INPUT:
; time
; orbEL = [period, Tperi, ecc, major, inc, W_cap, w_low]
;
; OUTPUT:
; (x,y)
; (sep, PA)

period = orbEL(0)
Tperi = orbEL(1)
ecc = orbEL(2)
major = orbEL(3)
inc = orbEL(4)
W_cap = orbEL(5)
w_low = orbEL(6)

; Determine the eccentric anomalies Ei: 
;     mu(ti - T) = Ei - esin(Ei)  ... Kepler's Equation
;     where mu = 360/P = 2*pi/P
; Solve this transcendental equation through an 
; iterative procedure.
; Use SOLVE_TRANS.PRO

; array to hold Ei's (the eccentric anomaly)

Ei = 0d

mu = 2*!dpi/period

Mi = mu*(time - Tperi) 

; reduce to same epoch

Mi = 2*!dpi*((time - Tperi)/period $	
	- fix((time - Tperi)/period))

solve_trans,ecc,Mi,Eit

Ei = Eit

; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)

Xi = cos(Ei) - ecc
Yi = sqrt(1 - ecc^2)*sin(Ei)

; Compute the four Thiele-Innes elements

A_cap = major*(cos(w_low)*cos(W_cap) - sin(w_low)*sin(W_cap)*cos(inc))
B_cap = major*(cos(w_low)*sin(W_cap) + sin(w_low)*cos(W_cap)*cos(inc))
F_cap = major*(-sin(w_low)*cos(W_cap) - cos(w_low)*sin(W_cap)*cos(inc))
G_cap = major*(-sin(w_low)*sin(W_cap) + cos(w_low)*cos(W_cap)*cos(inc))

; Compute coordinates of secondary

;	xi = AXi + FYi
;	yi = BXi + GYi

x = A_cap*Xi + F_cap*Yi
y = B_cap*Xi + G_cap*Yi

; convert (x,y) to (PAfit,sepfit)
; x = rho*sin(theta), y=rho*sin(theta)

sep = sqrt(x^2 + y^2)

; Get correct sign on PA by entering y,x separately into atan:
PA = atan(y,x)

if (PA lt 0.0) then PA = 2*!dpi + PA

end

;
;
;

pro plot_err, x, dx, y, dy, N

; plot error bars associated with each data point
; Input
;	x: array of x values
;	dx: array of error in x values
;	y: array of y values
;	dy: erray of error in y values
;	N: number of data points


for i=0, N-1 do begin

	oplot, [x(i)-dx(i), x(i)+dx(i)], [y(i), y(i)]
	oplot, [x(i), x(i)], [y(i)-dy(i), y(i)+dy(i)]

endfor

end
