pro plot_orbit_nodata

; Given the 7 orbital elements (P,T,e,A,B,F,G) or (P,T,e,a,i,Omega,omega)
; plot orbit and compare with measurements
;
; Options include entering P,T,e and then doing a least squares fit for
; a,i,Omega,omega or entering (P,T,e,a,i,Omega,omega)
; [(P,T,e,A,B,F,G) entering to be added at a later date]
;
; Read times of observations and measurements in from data file
;	
; Equations:  
; Orbital elements (P,T,e,A,B,F,G) and a set of observations (ti,xi,yi) 
; are known
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)
; The fitted (xi,yi) positions are determined from the four Thiele-Innes 
; elements and the normalized rectangular coordinates:
;	xi = AXi + FYi
;	yi = BXi + GYi
; Calculate geometric elements a("),i,Omega,omega from (A,F,B,G)
;
; Essentially, the equations one needs to solve are:
;	E - esinE = 2Pi/P * (t - T)
;	x = A*(cos(Ei) - e) + F*(sqrt(1 - e^2)*sin(Ei))
;	y = B*(cos(Ei) - e) + G*(sqrt(1 - e^2)*sin(Ei))
;
; Paramers:
;	Period:	period (P)
;	Tperi: 	time of periastron passage (T)
;	ecc:	eccentricity (e)
;	major:	semi-major axis (in mas) (a)
;	inc:	inclination (i)
;	Omega:	position angle of node
;	omega:	angle between node and periastron
;	ti:	time of observation
;	xd:	RA - data
;	yd:	DEC - data
;	xf:	RA - fit (determined from orbital elements for time of obs.)
;	yf:	DEC - fit (determined from orbital elements for time of obs.)
;	A,B,F,G: Thiele-Innes elements
;
; Note: xi and yi are not the same convention for RA and DEC chosen to make 
; the orbital plots. Specifically, in the fitting the apparent ellipse 
; (ORBFITE.PRO):
;	x = RA = -rho*sin(theta),   y = DEC = rho*cos(theta)
;	(i.e. North is in the direction of the positive y axis, theta is
;	measured counterclockwise from positive y-axis)
; Whereas in this program, the coordinate system set up by Couteau,
;	x = rho*cos(theta),   y = rho*sin(theta)
;	(i.e. North id in the direction of the positive x-axis and the 
;	y-axis points east)
; Both coordinate systems have the same separation and position angle 
; orietations, they are just rotated by 90 deg with respect to each other.
;
; INPUT:
;	data_file: file containing data points in the format:
;		      time  theta  error(theta)  rho  error(rho)
;		      where theta is the position angle and rho the separation
;		      and their respective errors
;		      (enter file name using quotes)
;	num: number of data points contained in data_file
;	
; Prompted inputs:
;	all of the orbital elements: P,T,e,A,B,F,G
;
; Began 22 May 2002
; Modified on 9 July 2002 to include least squares fit
;
; 1 Jan 2008: Modify to make consistent with othe plot_orbit programs
;  - data file includes errors in sep and PA
;  - plot orbit as solid line

;!P.font=0
;!P.multi=0
;!P.charsize=1.2
;!P.charthick=2.0
;!P.thick=2.0
;!X.thick=2.0
;!Y.thick=2.0
;;frac=0.7	;scale factor for adjusting size of plotting symbols
;frac=1.2	;scale factor for adjusting size of plotting symbols

!P.font=0
!P.multi=0
!P.charsize=1.0
!P.charthick=3.0
!P.thick=3.0
!X.thick=3.0
!Y.thick=3.0
;frac=0.7	;scale factor for adjusting size of plotting symbols
frac=1.2	;scale factor for adjusting size of plotting symbols

; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1

; Read in measurements from data file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

; Do you want to plot another orbit?
count = 0
answer = 'y'
while (answer eq 'y') do begin

; Obtain values for P,T,e

period = 0d
Tperi = 0d
ecc = 0d
major = 0d
inc = 0d
W_cap = 0d
w_low = 0d

; Obtain initial values for P,T,e,a,i,W,w

print,"Enter P T e a i W w (period, time of periastron passage, eccentricity):"
read, period, Tperi, ecc, major, inc, W_cap, w_low

;convert angles from degrees to radians

inc = inc*!dpi/180.0
W_cap = W_cap*!dpi/180.0
w_low = w_low*!dpi/180.0

orbEL = [period, Tperi, ecc, major, inc, W_cap, w_low]

; OK now plot data and orbital fit


; plot orbit with 1000 points

pnum = 1000.0

tstep = period/pnum

; x points N (DEC)
; y points E (increasing RA)
; therefore plot as (-y,x)

xarr=dblarr(pnum+1)
yarr=dblarr(pnum+1)
tarr=dblarr(pnum+1)

ti = Tperi

for i=0, pnum do begin

    ; compute orbital position at given time:

    orbpos, ti, orbEL, xpos, ypos, tsep, tPA

    xarr(i) = xpos
    yarr(i) = ypos
    tarr(i) = ti

    ti = ti + tstep

endfor

; Calculate position during Tperi
orbpos, Tperi, orbEL, xperi, yperi, tsep, tPA

xmax = max(xarr)
xmin = min(xarr)

ymax = max(yarr)
ymin = min(yarr)

sepmax = max(abs([xarr,yarr]))
dl = 0.10*sepmax

if (count eq 0) then begin

   set_plot, 'ps'
   device,/Color,Bits_per_pixel=8,filename='temp.ps',xsize=10,ysize=10 

;   plot,yarr,xarr,xtitle='!9D!3RA (arcsec)',ytitle='!9D!3DEC (arcsec)',ISOTROPIC=1, $
   plot,yarr,xarr,xtitle='!9D!3RA (mas)',ytitle='!9D!3DEC (mas)',ISOTROPIC=1, $
;   plot,yarr,xarr,xtitle='!4D!3RA (mas)',ytitle='!4D!3DEC (mas)',ISOTROPIC=1, $
;   plot,yarr,xarr,xtitle='R.A. (!9''''!3)',ytitle='Dec. (!9''''!3)',ISOTROPIC=1, $
;   plot,yarr,xarr,xtitle='!4D!3RA (!9''''!3)',ytitle='!4D!3DEC (!9''''!3)',ISOTROPIC=1, $
;   plot,yarr,xarr,xtitle='E-W Sep (mas)',ytitle='N-S Sep (mas)',ISOTROPIC=1, $
;        xrange=[ymax+dl,ymin-dl], yrange=[xmin-dl,xmax+dl], $
   ;      xrange=[50,-50], yrange=[-50,50], $
   ;      xrange=[3,-3], yrange=[-3,3], $
;         xrange=[5.5,-5.5], yrange=[-5.5,5.5], $
;         xrange=[6.0,-6.0], yrange=[-6.0,6.0], $
;         xrange=[4.0,-4.0], yrange=[-4.0,4.0], $
;         xrange=[0.7,-0.7], yrange=[-0.7,0.7], $
         xrange=[sepmax,-sepmax], yrange=[-sepmax,sepmax], $
;          xrange=[1.5,-2.2], yrange=[-2.0,1.7], $
   ;      xrange=[5,-5], yrange=[-8,2], $
   ;      xrange=[0.1,-0.45], yrange=[-0.25,0.3], $
   ;      xrange=[0.1,-0.22], yrange=[-0.17,0.15], $
        xstyle=1,ystyle=1, /nodata,position=[0.17,0.12,0.97,0.98]
endif

; mark primary with an asterisk
oplot,[0],[0],psym=2

oplot,yarr,xarr,linestyle=count

; Define user symbol to be closed circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)
;Define the symbol to be a unit circle with 16 points
usersym, frac*cos(Z), frac*sin(Z)
;oplot,[yperi],[xperi],psym=8

ans_point = 'y'
print,'Do you want to plot an additional point (y/n)?'
read,ans_point
while (ans_point eq 'y') do begin
   ; Define user symbol to be filled rectangle
   ;xrect = 1.5*frac*[-1.0,-1.0,1.0,1.0,-1.0]
   ;yrect = 1.5*frac*[-1.0,1.0,1.0,-1.0,-1.0]
   ;usersym,xrect,yrect,/FILL
   usersym, frac*cos(Z), frac*sin(Z),thick=4.0,/fill
   time_add = 0.0d
   print,'Enter time for added point:'
   read,time_add
   orbpos, time_add, orbEL, xpos_add, ypos_add, sep_add, tPA_add
   oplot,[ypos_add],[xpos_add],psym=8,color=1
   print, 'Time   sep   PA:',time_add, sep_add, tPA_add*180/!dpi
   print,'Do you want to plot an additional point (y/n)?'
   read,ans_point
endwhile



print,'Do you want to plot another orbit (y/n)?'
read,answer
count = count + 1
endwhile

;xyouts,0.08,0.12,'HD 285931, vB 96'
;xyouts,0.05,0.25,'56 Persei'
;xyouts,-2.0,2.5,'GJ 86'

!P.Charsize=1.0

device, /close
set_plot, 'x'

sep = sqrt(xarr^2 + yarr^2)

print,'Max sep:',max(sep)
print,'Min sep:',min(sep)
print,'Mean sep:',mean(sep)

ave_sep = mean(sep)
ind = where(abs(sep-ave_sep) eq min(abs(sep-ave_sep)))
print,'Time at mean separation: ',tarr(ind(0)),sep(ind(0))


print,'Apastron: a(1+e) = ',major*(1 + ecc)
print,'Periastron: a(1-e) = ',major*(1 - ecc)


end

;
;
;

pro orbpos, time, orbEL, x, y, sep, PA

; Given set of orbital parameters, compute position at a given time
;
; INPUT:
; time
; orbEL = [period, Tperi, ecc, major, inc, W_cap, w_low]
;
; OUTPUT:
; (x,y)
; (sep, PA)

period = orbEL(0)
Tperi = orbEL(1)
ecc = orbEL(2)
major = orbEL(3)
inc = orbEL(4)
W_cap = orbEL(5)
w_low = orbEL(6)

; Determine the eccentric anomalies Ei: 
;     mu(ti - T) = Ei - esin(Ei)  ... Kepler's Equation
;     where mu = 360/P = 2*pi/P
; Solve this transcendental equation through an 
; iterative procedure.
; Use SOLVE_TRANS.PRO

; array to hold Ei's (the eccentric anomaly)

Ei = 0d

mu = 2*!dpi/period

Mi = mu*(time - Tperi) 

; reduce to same epoch

Mi = 2*!dpi*((time - Tperi)/period $	
	- fix((time - Tperi)/period))

solve_trans,ecc,Mi,Eit

Ei = Eit

; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)

Xi = cos(Ei) - ecc
Yi = sqrt(1 - ecc^2)*sin(Ei)

; Compute the four Thiele-Innes elements

A_cap = major*(cos(w_low)*cos(W_cap) - sin(w_low)*sin(W_cap)*cos(inc))
B_cap = major*(cos(w_low)*sin(W_cap) + sin(w_low)*cos(W_cap)*cos(inc))
F_cap = major*(-sin(w_low)*cos(W_cap) - cos(w_low)*sin(W_cap)*cos(inc))
G_cap = major*(-sin(w_low)*sin(W_cap) + cos(w_low)*cos(W_cap)*cos(inc))

; Compute coordinates of secondary

;	xi = AXi + FYi
;	yi = BXi + GYi

x = A_cap*Xi + F_cap*Yi
y = B_cap*Xi + G_cap*Yi

; convert (x,y) to (PAfit,sepfit)
; x = rho*sin(theta), y=rho*sin(theta)

sep = sqrt(x^2 + y^2)

; Get correct sign on PA by entering y,x separately into atan:
PA = atan(y,x)

if (PA lt 0.0) then PA = 2*!dpi + PA

end

;
;
;

pro plot_err, x, dx, y, dy, N

; plot error bars associated with each data point
; Input
;	x: array of x values
;	dx: array of error in x values
;	y: array of y values
;	dy: erray of error in y values
;	N: number of data points


for i=0, N-1 do begin

	oplot, [x(i)-dx(i), x(i)+dx(i)], [y(i), y(i)]
	oplot, [x(i), x(i)], [y(i)-dy(i), y(i)+dy(i)]

endfor

end
