pro plot_orbit_shade, data_file, datanum, pte_file, fnum, orb_deg=orb_deg

; Given a set of orbital elements {P,T,e,a,i,Omega,omega}
; compute position at a given set of times
;
; Compare with measured values of (sep, PA)
;
; Shade region defined by all possible solutions
;
; Read times of observations and measurements in from data file
; Read in orbital solutions from pte file
;
; INPUT:
;	data_file: file containing data points in the format:
;		      time  theta  error(theta)  rho  error(rho)
;		      where theta is the position angle and rho the separation
;		      and their respective errors
;		      (enter file name using quotes)
;	datanum: number of data points contained in data_file
;	pte_file: name of file containing P,T,e,a,i,W,w,chi2 
;                 for possible solutions(one solution per line)
;                 Set orb_deg keywod if i,Omega,omega are in degrees
;	fnum: number of lines in pte_file
;
; Equations:  
; Orbital elements (P,T,e) and a set of observations (ti,xi,yi) are known
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)
; The four Thiele-Innes elements are solved through linear least squares fit:
;	xi = AXi + FYi
;	yi = BXi + GYi
; Calculate geometric elements a("),i,Omega,omega from (A,F,B,G)
;
; Essentially, the equations one needs to solve are:
;	E - esinE = 2Pi/P * (t - T)
;	x = A*(cos(Ei) - e) + F*(sqrt(1 - e^2)*sin(Ei))
;	y = B*(cos(Ei) - e) + G*(sqrt(1 - e^2)*sin(Ei))
;
; Paramers:
;	Period:	period (P)
;	Tperi: 	time of periastron passage (T)
;	ecc:	eccentricity (e)
;	major:	semi-major axis (in mas) (a)
;	inc:	inclination (i)
;	Omega:	position angle of node
;	omega:	angle between node and periastron
;	ti:	time of observation
;	xd:	RA - data
;	yd:	DEC - data
;	xf:	RA - fit (determined from orbital elements for time of obs.)
;	yf:	DEC - fit (determined from orbital elements for time of obs.)
;	A,B,F,G: Thiele-Innes elements
;
; Note: xi and yi are not the same convention for RA and DEC chosen to make 
; the orbital plots. Specifically, in the fitting the apparent ellipse 
; (ORBFITE.PRO):
;	x = RA = -rho*sin(theta),   y = DEC = rho*cos(theta)
;	(i.e. North is in the direction of the positive y axis, theta is
;	measured counterclockwise from positive y-axis)
; Whereas in this program, the coordinate system set up by Couteau,
;	x = rho*cos(theta),   y = rho*sin(theta)
;	(i.e. North id in the direction of the positive x-axis and the 
;	y-axis points east)
; Both coordinate systems have the same separation and position angle 
; orietations, they are just rotated by 90 deg with respect to each other.
;
; INPUT:
;	data_file: file containing data points in the format:
;		      time  theta  error(theta)  rho  error(rho)
;		      where theta is the position angle and rho the separation
;		      and their respective errors
;		      (enter file name using quotes)
;	num: number of data points contained in data_file
;	
; Prompted inputs:
;	all of the orbital elements: P,T,e
;
; OUTPUT: 
;	plot of data points and orbital solutions
;
; Modifications to be added:
; 	- error bars on data points
;	- line connecting measurement to claculated point

close,1

;!P.charsize=1.2
;!P.charthick=3.0
;!P.thick=4.0
;!X.thick=4.0
;!Y.thick=4.0
;frac=1.5	;scale factor for adjusting size of plotting symbols

!P.font=0
!P.charsize=1.2
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.4	;scale factor for adjusting size of plotting symbols
nthick = 4

; Set color table
;  red=1, green=2, blue=3
tvlct, [255,0,0], [0,255,0], [0,0,255], 1

; Set rainbow color table
;loadct,40                       ; rainbow+black
; Colors:
; red = 250, orange=220, yellow=195, green=150, blue=60, purple=30, black=0
;ctab = [250,220,195,150,60,30,0]

; Set orb_deg keyword if i,Omega,omega in pte_file are already in degrees
; keyword_set returns True (1) if orb_deg is defined, False (0) if undefined

ans_deg = keyword_set(orb_deg)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Read in measurements from data file ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time = dblarr(datanum)
sepmeas = dblarr(datanum)
PAmeas = dblarr(datanum)
dsep = dblarr(datanum)
dPA = dblarr(datanum)

openr,lun,data_file,/get_lun

for i=0, datanum-1 do begin

	readf, lun, temp1, temp2, temp3, temp4, temp5

	time(i) = temp1
	PAmeas(i) = temp2
	dPA(i) = temp3
	sepmeas(i) = temp4*1000.0	
	dsep(i) = temp5*1000.0
	
endfor

close,lun

; convert theta and dtheta to radians

PAmeas = PAmeas*!dpi/180
dPA = dPA*!dpi/180

; convert data points to x and y coordinates

xmeas = sepmeas * cos(PAmeas)	; x coordinate
ymeas = sepmeas * sin(PAmeas)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(PAmeas)^2*dsep^2 + sepmeas^2*sin(PAmeas)^2*dPA^2)
dy = sqrt(sin(PAmeas)^2*dsep^2 + sepmeas^2*cos(PAmeas)^2*dPA^2)

period = 0d
Tperi = 0d
ecc = 0d
major = 0d
inc = 0d
W_cap = 0d
w_low = 0d

print,"Enter P T e a i W w for best fit solution:"
read, period, Tperi, ecc, major, inc, W_cap, w_low

;convert angles from degrees to radians

inc = inc*!dpi/180.0
W_cap = W_cap*!dpi/180.0
w_low = w_low*!dpi/180.0

orbEL = [period, Tperi, ecc, major, inc, W_cap, w_low]


; compute orbital position at times of observation:

calc_vbfit, orbEL, time, PAfit, sepfit
xfit = sepfit * cos(PAfit)	; x coordinate
yfit = sepfit * sin(PAfit)	; y coordinate

;;;;;;;;;;;;;;;;;;;;;;;;;;
; Read in pte file names ;
;;;;;;;;;;;;;;;;;;;;;;;;;;

openr,1,pte_file

period = 0d
Tperi = 0d
ecc = 0d
major = 0d
inc = 0d
W_cap = 0d
w_low = 0d
xchi = 0d

massi = dblarr(fnum)
a_arr = dblarr(fnum)
e_arr = dblarr(fnum)
i_arr = dblarr(fnum)
P_arr = dblarr(fnum)
T_arr = dblarr(fnum)
W_cap_arr = dblarr(fnum)
w_low_arr = dblarr(fnum)
chi_arr = dblarr(fnum)

freepar = 1.0		;number of free parameters (up to 7)
	
; degrees of freedom
dof = 2.0*datanum - freepar
	
parallax = 0.0d

print, 'Enter the parallax of the system (in same units as separation):'
print,'(distance of 140, parallax = 7.14 mas)'
read, parallax

ansyd=' '
print,'Is period in years or days (y/d)?'
read,ansyd

if (ansyd eq 'y') then yd = 1.0d
if (ansyd eq 'd') then yd = 1.0d/365.25

print, 'Computing masses...'

for j=0, fnum-1 do begin

   ; Read in P,T,e,a,i,W,w from pte_file

   readf, 1, period, Tperi, ecc, major, inc, W_cap, w_low, xchi

   major = major*1000.0

   EL = [period,Tperi,ecc,major,inc,W_cap,w_low]

   ; calculate mass of system using distance of 140 pc
   ; parallax = 7.14 mas
	
   mass = major^3/parallax^3/(yd*period)^2

   ; If i,Omega,omega are in degrees, then convert to radians:
   if (ans_deg) then begin
      inc = inc*!dpi/180.0
      W_cap = W_cap*!dpi/180.0
      w_low = w_low*!dpi/180.0
   endif

   massi(j) = mass
   P_arr(j) = period
   T_arr(j) = Tperi
   e_arr(j) = ecc
   a_arr(j) = major
   i_arr(j) = 180*inc/!dpi
   W_cap_arr(j) = 180*W_cap/!dpi
   w_low_arr(j) = 180*w_low/!dpi
   chi_arr(j) = xchi            ; read chi2 in from data file

   if (W_cap_arr(j) ge 360.0) then W_cap_arr(j) = W_cap_arr(j) - 360.0

endfor

close,1


print,"Error estimates: min max mean median stddev"
print,format='(%"P:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(P_arr),max(P_arr),mean(P_arr),median(P_arr),stddev(P_arr)
print,format='(%"a:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(a_arr),max(a_arr),mean(a_arr),median(a_arr),stddev(a_arr)
print,format='(%"e:",f10.6,f10.6,f10.6,f10.6,f10.6)', $
	min(e_arr),max(e_arr),mean(e_arr),median(e_arr),stddev(e_arr)
print,format='(%"i:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(i_arr),max(i_arr),mean(i_arr),median(i_arr),stddev(i_arr)
print,format='(%"T:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(T_arr),max(T_arr),mean(T_arr),median(T_arr),stddev(T_arr)
print,format='(%"W:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(W_cap_arr),max(W_cap_arr),mean(W_cap_arr), $
	median(W_cap_arr),stddev(W_cap_arr)
print,format='(%"w:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(w_low_arr),max(w_low_arr),mean(w_low_arr), $
	median(w_low_arr),stddev(w_low_arr)
print,format='(%"Mtot:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(massi),max(massi),mean(massi),median(massi),stddev(massi)

; OK... now we have to compute bounded regions of orbital solution ranges
; Take PAs going from 0 to 360 deg
; Use PA to compute separation for each orbit
; For every PA value, save maximum and minimum binary separation
; Equations:
; tan(PA - Omega) = tan(nu - omega)*cosi
; r = a*(1 - e^2)/(1 + e*cos(nu))
; sep = r*cos(nu + omega)/cos(PA - Omega)

nshade = 100.0 ; 1000.0
pa_shade = findgen(nshade)/(nshade-1)*2.0*!dpi

sep_shade_min = dblarr(nshade)
sep_shade_max = dblarr(nshade)

xshade_min = dblarr(nshade)
yshade_min = dblarr(nshade)

xshade_max = dblarr(nshade)
yshade_max = dblarr(nshade)

; At start, set min,max separation to best fit values
ecc = orbEL(2)
major = orbEL(3)
inc  = orbEL(4)
W_cap = orbEL(5)
w_low = orbEL(6)
for j=0, nshade-1 do begin
   PA = pa_shade(j)
   nu = atan(tan(PA - W_cap)/cos(inc)) - w_low
   rad = major*(1.0 - ecc^2)/(1 + ecc*cos(nu))
   sep = rad *cos(nu + w_low)/cos(PA - W_cap)
   if (sep lt 0.0) then begin
      nu = nu + !dpi
      rad = major*(1.0 - ecc^2)/(1 + ecc*cos(nu))
      sep = rad *cos(nu + w_low)/cos(PA - W_cap)
   endif
   sep_shade_min(j) = sep
   sep_shade_max(j) = sep
   xshade_min(j) = sep*cos(PA) ; x coordinate
   yshade_min(j) = sep*sin(PA) ; y coordinate
   xshade_max(j) = sep*cos(PA) ; x coordinate
   yshade_max(j) = sep*sin(PA) ; y coordinate
endfor

; Save values for best fit:
sep_best = sep_shade_min
xbest = xshade_min
ybest = yshade_min

print,'Computing shaded regions...'

for i=0, fnum-1 do begin

   ecc = e_arr(i)
   major = a_arr(i)
   inc = i_arr(i)*!dpi/180.
   W_cap = W_cap_arr(i)*!dpi/180.0
   w_low = w_low_arr(i)*!dpi/180.0

   for j=0, nshade-1 do begin
      PA = pa_shade(j)
      nu = atan(tan(PA - W_cap)/cos(inc)) - w_low
      rad = major*(1.0 - ecc^2)/(1 + ecc*cos(nu))
      sep = rad *cos(nu + w_low)/cos(PA - W_cap)
      if (sep lt 0.0) then begin
         nu = nu + !dpi
         rad = major*(1.0 - ecc^2)/(1 + ecc*cos(nu))
         sep = rad *cos(nu + w_low)/cos(PA - W_cap)
      endif

      if (sep lt sep_shade_min(j)) then begin
         sep_shade_min(j) = sep
         xshade_min(j) = sep*cos(PA) ; x coordinate
         yshade_min(j) = sep*sin(PA) ; y coordinate
      endif

      if (sep gt sep_shade_max(j)) then begin
         sep_shade_max(j) = sep
         xshade_max(j) = sep*cos(PA) ; x coordinate
         yshade_max(j) = sep*sin(PA) ; y coordinate
      endif

   endfor
endfor

; compute orbital position at times of observation:

answer = 'y'		; answer for plotting another orbital solution
count = 0

; Define user symbol to be closed circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)
;Define the symbol to be a closed circle
usersym, frac*cos(Z), frac*sin(Z), /fill


sepmax = round(1.2*max(abs([yfit,xfit,xmeas,ymeas])))
sepmax = 5.0

set_plot,'ps'
device, /Color, Bits_per_pixel=8, filename='temp.eps',xsize=15,ysize=15,/encap

plot,ybest,xbest, ISOTROPIC=1, /nodata, xstyle=1,ystyle=1, $
     xtitle='!9D!3RA (mas)',ytitle='!9D!3DEC (mas)', $
    ;xtitle='!9D!3RA (arcsec)',ytitle='!9D!3DEC (arcsec)', $
     xrange=[sepmax,-sepmax], yrange=[-sepmax,sepmax]
;     xrange=[50,-200], yrange=[-100,150]
;     xrange=[50,-250], yrange=[-50,250]
;     xrange=[0.4,-0.8], yrange=[-0.8,0.4]
;     xrange=[2.0,-2.0], yrange=[-2.0,2.0]

polyfill, [yshade_min, Reverse(yshade_max)], [xshade_min, Reverse(xshade_max)],$
    thick=nthick,color=170,noclip=0, $
    clip=[sepmax,-sepmax,-sepmax,sepmax]
    ;clip=[0.4,-0.8,-0.8,0.4]
    ;clip=[2.0,-2.0,-2.0,2.0]
    ;clip=[50,-50,-250,250];Color=90,clip=[50,-100,-200,150]
oplot,ybest,xbest,thick=nthick,color=3;,color=ctab(4)

; compare measured and computed positions

; compute predicted position on date of observation


; print O-C residuals (Observed - Calculated)

xOC = xmeas - xfit
yOC = ymeas - yfit

;for i=0, num-1 do print,format='(f8.3,f7.3,f8.3,f8.3,f7.3,f8.3,f8.3)', $
;	time(i), dx(i), xOC(i), xOC(i)/dx(i), dy(i), yOC(i), yOC(i)/dy(i)
    
if (count eq 0) then begin
   ans_comp='n'
   ans_line='y'
   print, 'Do you want to plot computed position of binary (y/n)?'
   read, ans_comp
   print, 'Do you want to connect computed and measured positions with a line (y/n)?'
   read, ans_line
endif

; mark secondary at computed time with a cross
if (ans_comp eq 'y') then oplot,[yfit],[xfit],psym=7,color=ncol

; connect computed and measured positions with a line
if (ans_line eq 'y') then begin
   for i=0, datanum-1 do begin
      if (count eq 0) then oplot,[yfit(i),ymeas(i)],[xfit(i),xmeas(i)],linestyle=0,thick=4.0,color=1
      ;oplot,[yfit(i),ymeas(i)],[xfit(i),xmeas(i)],linestyle=0,thick=4.0
   endfor
endif

; Plot another orbit?
ans_plot = 'y' 
print,'Do you want to plot another orbit (y/n)?'
read,ans_plot

while (ans_plot eq 'y') do begin

   print,"Enter P T e a i W w for best fit solution:"
   read, pnew, Tnew, enew, anew, inew, Wc_new, wl_new

   ;convert angles from degrees to radians

   inew = inew*!dpi/180.0
   Wc_new = Wc_new*!dpi/180.0
   wl_new = wl_new*!dpi/180.0

   orbnew = [pnew, Tnew, enew, anew, inew, Wc_new, wl_new]

   ; compute orbit fit:

   pnum = 1000
   tarr = findgen(pnum+2)/pnum*pnew + Tperi

   calc_vbfit, orbnew, tarr, PAnew, sepnew
   xnew = sepnew * cos(PAnew)	; x coordinate
   ynew = sepnew * sin(PAnew)	; y coordinate

   oplot,ynew,xnew,thick=nthick,color=3,linestyle=1
   oplot,ybest,xbest,thick=nthick,color=3

   print,'Do you want to plot another orbit (y/n)?'
   read,ans_plot

endwhile


; Plot measured positions

; mark measured position of secondary with a square

;Define the symbol to be a closed circle
usersym, frac*cos(Z), frac*sin(Z), /fill

oplot,[ymeas],[xmeas],psym=8


; Plot error bars in direction of sep, PA?
ans_ell = 'n'
if (ans_ell eq 'y') then begin

   ; Plot ACS error bars
   for i=0, 1 do begin
      ;oplot, [ymeas(i)-dy(i),ymeas(i)+dy(i)],[xmeas(i),xmeas(i)],color=1,thick=4
      ;oplot, [ymeas(i),ymeas(i)],[xmeas(i)-dx(i),xmeas(i)+dx(i)],color=1,thick=4
   endfor

   ; Plot non-ACS measurements
   for i=0, datanum-1 do begin

      print,'WE SHOULD NOT BE HERE!'

      errsep_ra = dsep(i)*sin(PAmeas(i))
      errsep_dec = dsep(i)*cos(PAmeas(i))

      ;;;oplot,[ymeas(i)-errsep_ra,ymeas(i)+errsep_ra],[xmeas(i)-errsep_dec,xmeas(i)+errsep_dec],thick=4,color=1;,color=ctab(0);,color=1

      errpa_ra_p = sepmeas(i)*sin(PAmeas(i) + dPA(i))
      errpa_ra_m = sepmeas(i)*sin(PAmeas(i) - dPA(i))
      errpa_dec_p = sepmeas(i)*cos(PAmeas(i) + dPA(i))
      errpa_dec_m = sepmeas(i)*cos(PAmeas(i) - dPA(i))

      ;;;;oplot,[errpa_ra_m,errpa_ra_p],[errpa_dec_m,errpa_dec_p],thick=4,color=1;,color=ctab(0);

	;dPA(i) = temp3
	;sepmeas(i) = temp4	
	;dsep(i) = temp5

   endfor
endif $
else plot_err,ymeas,abs(dy),xmeas,abs(dx),datanum


; mark primary with an asterisk

oplot,[0],[0],psym=2

ans = 'y'

print, 'Do you want to plot date labels (y/n)?'
read, ans

if (ans eq 'y') then begin

    ; determine offsets in x,y for date labels

;    offset = [0.10,0.002]
;    offset = [0.085,0.0030]    ; proret plot
    offset = [0.035,0.005]      ; bestfit

    ; Polaris values
    deltax = offset(0)*sin(PAmeas)
    deltay = offset(1)*cos(PAmeas)

    ; plot date labels

    ;xyouts,ymeas+deltax,xmeas+deltay,string(format='(f7.2)',time),alignment=0.5,charsize=0.7

;xyouts,ymeas(0)+deltax(0),xmeas(0)+deltay(0),string(format='(f7.2)',time(0)),alignment=0.5,charsize=0.7
;xyouts,ymeas(1)+deltax(1),xmeas(1)+deltay(1),string(format='(f7.2)',time(1)),alignment=0.5,charsize=0.7
;xyouts,ymeas(2),xmeas(2)+0.012,string(format='(f7.2)',time(2)),alignment=0.5,charsize=0.7

xyouts,ymeas(0)+deltax(0),xmeas(0)+deltay(0),string(format='(f7.1)',time(0)),alignment=0.5,charsize=1.0
xyouts,ymeas(1)+deltax(1),xmeas(1)+deltay(1),string(format='(f7.1)',time(1)),alignment=0.5,charsize=1.0
xyouts,ymeas(2)+0,xmeas(2)-0.030,string(format='(f7.1)',time(2)),alignment=0.5,charsize=1.0
xyouts,ymeas(3)+0,xmeas(3)-0.030,string(format='(f7.1)',time(3)),alignment=0.5,charsize=1.0
xyouts,ymeas(4)+0.041,xmeas(4)-0.0035,string(format='(f7.1)',time(4)),alignment=0.5,charsize=1.0

endif

; Plot predicted positions
ans_pred = 'n'
if (ans_pred eq 'y') then begin

   tpred = [2018.7,2019.7,2020.7]
   auto_ephemeris,orbEL,tpred,RApred,DECpred,/noprint
   usersym, frac*cos(Z), frac*sin(Z),thick=nthick
   oplot,RApred,DECpred,psym=8

endif

; Re-plot axes (if shaded area extends further)
plot,ybest,xbest, ISOTROPIC=1, /nodata, xstyle=1,ystyle=1, /noerase, $
     xtitle='!9D!3RA (mas)',ytitle='!9D!3DEC (mas)', $
     ;xtitle='!9D!3RA (arcsec)',ytitle='!9D!3DEC (arcsec)', $
     xrange=[sepmax,-sepmax], yrange=[-sepmax,sepmax]
     ;xrange=[50,-250], yrange=[-50,250]
     ;xrange=[0.4,-0.8], yrange=[-0.8,0.4]
     ;xrange=[2.0,-2.0], yrange=[-2.0,2.0]

!P.Charsize=1.0

device, /close

; Make residual plots
scale = 1.0

pares = (PAmeas - PAfit)*180.0/!dpi          ; in mas
sepres = (sepmeas - sepfit)*scale  ; in degrees

RAres = (ymeas - yfit)*scale            ; in mas
DECres = (xmeas - xfit)*scale           ; in mas

yr = max(abs([RAres,DECres]))
yr = 30

if time(0) gt 10000.0 then begin
   dt = 50000.0 
   datelabel = 'HJD - 50,000'
endif else begin
   dt = 0.0
   datelabel = 'Date'
endelse

tdiff = max(time-dt) - min(time-dt)
trange = [min(time-dt)-0.05*tdiff,max(time-dt)+0.05*tdiff]
seprange = max(abs([sepres+dsep,sepres-dsep]))*1.1
parange = max(abs([pares+dpa*180.0/!dpi,pares-dpa*180.0/!dpi]))*1.1

parange = 40.0 
seprange = 20.0

device, /Color, Bits_per_pixel=8, filename='temp_xy.eps',xsize=15,ysize=13.5,/encap
!P.Multi=[0,1,2]


ploterror, time, RAres, dy*scale,psym=4,xtitle='Date',ytitle='Residuals in RA (mas)',yrange=[-yr,yr],xrange=trange,xstyle=1,ystyle=1 
oplot,trange,[0,0],linestyle=2

ploterror, time, DECres, dx*scale,psym=4,xtitle='Date',ytitle='Residuals in Dec (mas)',yrange=[-yr,yr],xrange=trange,xstyle=1,ystyle=1
oplot,trange,[0,0],linestyle=2

device, /close

set_plot, 'ps'
;device, /Color, Bits_per_pixel=8, filename='temp.ps',xsize=15,ysize=13.5
device, /Color, Bits_per_pixel=8, /encap, filename='temp_res.eps',xsize=18,ysize=15

!P.Multi=[0,1,2]

Z = findgen(17) * (!pi*2/16.)
usersym, cos(Z), sin(Z), /fill

ploterror, time-dt, pares,dpa*180.0/!dpi,psym=8,xtitle=datelabel,ytitle='!3 Residuals in PA (deg)',yrange=[-parange,parange],xrange=trange,xstyle=1,ystyle=1 
oplot,trange,[0,0],linestyle=1

ploterror, time-dt, sepres,dsep,psym=8,xtitle=datelabel,ytitle='Residuals in sep (mas)',yrange=[-seprange,seprange],xrange=trange,xstyle=1,ystyle=1
oplot,trange,[0,0],linestyle=1

device, /close
set_plot, 'x'



for i=0,datanum-1 do print,time(i),pares(i),dPA(i)*180/!dpi,pares(i)/(dPA(i)*180/!dpi),sepres(i),dsep(i)*scale,sepres(i)/(dsep(i)*scale)

!P.Multi=[0,0,0]


nsig = 1.0
;nsig = 3.53 ; 3 free parameters
;nsig = 9.0
print,'Solutions within nsig: ',nsig

isig = where(chi_arr le min(chi_arr)+nsig)
chi_select = chi_arr(isig)
P_select = P_arr(isig)
T_select = T_arr(isig)
e_select = e_arr(isig)
a_select = a_arr(isig)
i_select = i_arr(isig)
W_cap_select = W_cap_arr(isig)
w_low_select = w_low_arr(isig)
M_select = massi(isig)

ichimin = where(chi_select eq min(chi_select))
iPmin = where(P_select eq min(P_select))
iPmax = where(P_select eq max(P_select))
iemin = where(e_select eq min(e_select))
iemax = where(e_select eq max(e_select))
iamin = where(a_select eq min(a_select))
iamax = where(a_select eq max(a_select))
iimin = where(i_select eq min(i_select))
iimax = where(i_select eq max(i_select))

iPmin = iPmin(0)
iPmax = iPmax(0)
iemin = iemin(0)
iemax = iemax(0)
iamin = iamin(0)
iamax = iamax(0)
iimin = iimin(0)
iimax = iimax(0)

print,'Minimum P:'
print,P_select(iPmin),T_select(iPmin),e_select(iPmin),a_select(iPmin),i_select(iPmin),W_cap_select(iPmin),w_low_select(iPmin),massi(iPmin),chi_select(iPmin)

print,'Maximum P:'
print,P_select(iPmax),T_select(iPmax),e_select(iPmax),a_select(iPmax),i_select(iPmax),W_cap_select(iPmax),w_low_select(iPmax),massi(iPmax),chi_select(iPmax)

print,'Minimum e:'
print,P_select(iemin),T_select(iemin),e_select(iemin),a_select(iemin),i_select(iemin),W_cap_select(iemin),w_low_select(iemin),massi(iemin),chi_select(iemin)

print,'Maximum e:'
print,P_select(iemax),T_select(iemax),e_select(iemax),a_select(iemax),i_select(iemax),W_cap_select(iemax),w_low_select(iemax),massi(iemax),chi_select(iemax)

print,'Minimum a:'
print,P_select(iamin),T_select(iamin),e_select(iamin),a_select(iamin),i_select(iamin),W_cap_select(iamin),w_low_select(iamin),massi(iamin),chi_select(iamin)

print,'Maximum a:'
print,P_select(iamax),T_select(iamax),e_select(iamax),a_select(iamax),i_select(iamax),W_cap_select(iamax),w_low_select(iamax),massi(iamax),chi_select(iamax)

print,'Minimum i:'
print,P_select(iimin),T_select(iimin),e_select(iimin),a_select(iimin),i_select(iimin),W_cap_select(iimin),w_low_select(iimin),massi(iimin),chi_select(iimin)

print,'Maximum i:'
print,P_select(iimax),T_select(iimax),e_select(iimax),a_select(iimax),i_select(iimax),W_cap_select(iimax),w_low_select(iimax),massi(iimax),chi_select(iimax)


set_plot,'ps'
device, /Color, Bits_per_pixel=8, filename='temp_select.eps',xsize=15,ysize=15,/encap
   

plot,ybest,xbest, ISOTROPIC=1, /nodata, xstyle=1,ystyle=1, $
     xtitle='!9D!3RA (mas)',ytitle='!9D!3DEC (mas)', $
    ;xtitle='!9D!3RA (arcsec)',ytitle='!9D!3DEC (arcsec)', $
     xrange=[sepmax,-sepmax], yrange=[-sepmax,sepmax]
;     xrange=[50,-200], yrange=[-100,150]
;     xrange=[50,-250], yrange=[-50,250]
;     xrange=[0.4,-0.8], yrange=[-0.8,0.4]
;     xrange=[2.0,-2.0], yrange=[-2.0,2.0]


nplot = 10.0
Pstep = (max(P_select) - min(P_select))/nplot

P = min(P_select)
   pnum = 1000

for i=0,nplot do begin
   
   iP = where(abs(P_select-P) eq min(abs(P_select-P)))
   iP = iP(0)

   tarr = findgen(pnum+2)/pnum*P_select(iP) + T_select(iP)

   EL = [P_select(iP),T_select(iP),e_select(iP),a_select(iP),i_select(iP)*!dpi/180.0,W_cap_select(iP)*!dpi/180.0,w_low_select(iP)*!dpi/180.0]

   calc_vbfit, EL, tarr, PA_fit, sep_fit
   xfit = sep_fit * cos(PA_fit)	; x coordinate
   yfit = sep_fit * sin(PA_fit)	; y coordinate

   oplot,yfit,xfit,thick=nthick,color=3 ;,color=ctab(4)


   ; connect computed and measured positions with a line
   if (ans_line eq 'y') then begin
      for i=0, datanum-1 do begin
         if (count eq 0) then oplot,[yfit(i),ymeas(i)],[xfit(i),xmeas(i)],linestyle=0,thick=4.0,color=1
      endfor
   endif


   P = P + Pstep

endfor


; Plot measured positions

; mark measured position of secondary with a square

;Define the symbol to be a closed circle
usersym, frac*cos(Z), frac*sin(Z), /fill

oplot,[ymeas],[xmeas],psym=8

oplot,ybest,xbest,thick=nthick,color=3;,color=ctab(4)

plot_err,ymeas,abs(dy),xmeas,abs(dx),datanum

; mark primary with an asterisk

oplot,[0],[0],psym=2

device,/close



; Plot all orbits within selected interval

; First select e:
;isig = where(e_arr le 0.7)
;isig = where(e_arr le 0.55)
;isig = where(e_arr ge 0.7)
;chi_select = chi_arr(isig)
;P_select = P_arr(isig)
;T_select = T_arr(isig)
;e_select = e_arr(isig)
;a_select = a_arr(isig)
;i_select = i_arr(isig)
;W_cap_select = W_cap_arr(isig)
;w_low_select = w_low_arr(isig)
;M_select = massi(isig)

; Then reselect confidence interval:
nsig = 1.0
isig = where(chi_select le min(chi_select)+nsig)
chi_select = chi_select(isig)
P_select = P_select(isig)
T_select = T_select(isig)
e_select = e_select(isig)
a_select = a_select(isig)
i_select = i_select(isig)
W_cap_select = W_cap_select(isig)
w_low_select = w_low_select(isig)
M_select = M_select(isig)

; Then select P:
;ind = where(P_select lt 0.992)
;ind = where(P_select gt 1.020)
;ind = where((i_select ge 80) and (i_select le 110))
ind = where(P_select ge min(P_select)) ; select all P!


nplot = n_elements(ind)

print,'Number orbits plotted in temp_all.eps',nplot
print,'P min max mean:',min(P_select(ind)),max(P_select(ind)),mean(P_select(ind))
print,'e min max mean:',min(e_select(ind)),max(e_select(ind)),mean(e_select(ind))
print,'a min max mean:',min(a_select(ind)),max(a_select(ind)),mean(a_select(ind))
print,'i min max mean:',min(i_select(ind)),max(i_select(ind)),mean(i_select(ind))
print,'W min max mean:',min(W_cap_select(ind)),max(W_cap_select(ind)),mean(W_cap_select(ind))


set_plot,'ps'
device, /Color, Bits_per_pixel=8, filename='temp_all.eps',xsize=15,ysize=15,/encap
 

plot,ybest,xbest, ISOTROPIC=1, /nodata, xstyle=1,ystyle=1, $
     xtitle='!9D!3RA (mas)',ytitle='!9D!3DEC (mas)', $
    ;xtitle='!9D!3RA (arcsec)',ytitle='!9D!3DEC (arcsec)', $
     xrange=[sepmax,-sepmax], yrange=[-sepmax,sepmax]
;     xrange=[50,-200], yrange=[-100,150]
;     xrange=[50,-250], yrange=[-50,250]
;     xrange=[0.4,-0.8], yrange=[-0.8,0.4]
;     xrange=[2.0,-2.0], yrange=[-2.0,2.0]

pnum = 1000

for i=0,nplot-1 do begin
   
   iP = ind(i)

   tarr = findgen(pnum+2)/pnum*P_select(iP) + T_select(iP)

   EL = [P_select(iP),T_select(iP),e_select(iP),a_select(iP),i_select(iP)*!dpi/180.0,W_cap_select(iP)*!dpi/180.0,w_low_select(iP)*!dpi/180.0]

   calc_vbfit, EL, tarr, PA_fit, sep_fit
   xfit = sep_fit * cos(PA_fit)	; x coordinate
   yfit = sep_fit * sin(PA_fit)	; y coordinate

   oplot,yfit,xfit,thick=nthick,color=3 ;,color=ctab(4)

   ; connect computed and measured positions with a line
   if (ans_line eq 'y') then begin
      for i=0, datanum-1 do begin
         if (count eq 0) then oplot,[yfit(i),ymeas(i)],[xfit(i),xmeas(i)],linestyle=0,thick=4.0,color=1
      endfor
   endif

endfor

; Plot measured positions

; mark measured position of secondary with a square

;Define the symbol to be a closed circle
usersym, frac*cos(Z), frac*sin(Z), /fill

oplot,[ymeas],[xmeas],psym=8

oplot,ybest,xbest,thick=nthick,color=3;,color=ctab(4)

plot_err,ymeas,abs(dy),xmeas,abs(dx),datanum

; mark primary with an asterisk

oplot,[0],[0],psym=2

device,/close







free_lun,lun

stop,'Type .cont to continue'

end

;
;
;

pro solve_trans, e, M, EE

; Solve transcendental equation of the form E - esinE = M.
; Use iterative procedure to determine E.
; Initial approximation: E_0 = M + esinM - e^2/2 sin(2M)
; Improve solution by iterating the following to formulae:
;	M_0 = E_0 - esin(E_0)
;	E_1 = E_0 + (M - M_0)/(1 - ecos(E_0))
;	(derivative of Kepler's equation)
;
; Method adapted from Heintz 1978 (p.34-35)
; Results compared with point-&-click graphical method.  Iterative approach 
; leads to exact solution that satisfies E - esinE = M.  Therefore, 
; point-&-click method is subsequently removed from orbit fitting.
;
; INPUT:
;	e: eccentricity
;	M: mean anomaly   M= 2*Pi/P
;
; OUTPUT:
;	EE: eccentric anomaly
;
; Created: 9 May 2002

; Initial approximation:

EE = M + e*sin(M) + e^2/2*sin(2*M)

;print, "Initial approximation for E:", EE

EEi = 0d	; parameter to hold initial value once enter while loop

count = 0

while (abs(EE - EEi) gt 0.000001) do begin

	EEi = EE

	Mi = EEi - e*sin(EEi)

	EE = EEi + (M - Mi)/(1 - e*cos(EEi))

	count=count+1

endwhile

;print, "Final iterated E:", EE

;print,"Number of iterations:",count

end

;
;
;

pro orbpos, time, orbEL, x, y, sep, PA

; Given set of orbital parameters, compute position at a given time
;
; INPUT:
; time
; orbEL = [period, Tperi, ecc, major, inc, W_cap, w_low]
;
; OUTPUT:
; (x,y)
; (sep, PA)

period = orbEL(0)
Tperi = orbEL(1)
ecc = orbEL(2)
major = orbEL(3)
inc = orbEL(4)
W_cap = orbEL(5)
w_low = orbEL(6)

; Determine the eccentric anomalies Ei: 
;     mu(ti - T) = Ei - esin(Ei)  ... Kepler's Equation
;     where mu = 360/P = 2*pi/P
; Solve this transcendental equation through an 
; iterative procedure.
; Use SOLVE_TRANS.PRO

; array to hold Ei's (the eccentric anomaly)

Ei = 0d

mu = 2*!dpi/period

Mi = mu*(time - Tperi) 

; reduce to same epoch

Mi = 2*!dpi*((time - Tperi)/period $	
	- fix((time - Tperi)/period))

solve_trans,ecc,Mi,Eit

Ei = Eit

; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)

Xi = cos(Ei) - ecc
Yi = sqrt(1 - ecc^2)*sin(Ei)

; Compute the four Thiele-Innes elements

A_cap = major*(cos(w_low)*cos(W_cap) - sin(w_low)*sin(W_cap)*cos(inc))
B_cap = major*(cos(w_low)*sin(W_cap) + sin(w_low)*cos(W_cap)*cos(inc))
F_cap = major*(-sin(w_low)*cos(W_cap) - cos(w_low)*sin(W_cap)*cos(inc))
G_cap = major*(-sin(w_low)*sin(W_cap) + cos(w_low)*cos(W_cap)*cos(inc))

; Compute coordinates of secondary

;	xi = AXi + FYi
;	yi = BXi + GYi

x = A_cap*Xi + F_cap*Yi
y = B_cap*Xi + G_cap*Yi

; convert (x,y) to (PAfit,sepfit)
; x = rho*sin(theta), y=rho*sin(theta)

sep = sqrt(x^2 + y^2)

; Get correct sign on PA by entering y,x separately into atan:
PA = atan(y,x)

if (PA lt 0.0) then PA = 2*!dpi + PA

end

;
;
;

pro plot_err, x, dx, y, dy, N

; plot error bars associated with each data point
; Input
;	x: array of x values
;	dx: array of error in x values
;	y: array of y values
;	dy: erray of error in y values
;	N: number of data points


for i=0, N-1 do begin

	oplot, [x(i)-dx(i), x(i)+dx(i)], [y(i), y(i)],thick=4;,color=1
	oplot, [x(i), x(i)], [y(i)-dy(i), y(i)+dy(i)],thick=4;,color=1

endfor

end
