pro plot_orbit_vblunar_ditau, vb_file, vb_num, lun_file, lun_num, orbit_file, orb_num,allpar=allpar
  
; Modified to include lunar occultation projection in orbit fit
;
; Compute the orbital elements through a Newton-Raphson technique.
;
; This program should work directly with the data points (rho,theta,t), not 
; the fitting points. 
;
; The program requires an initial guess that can be calculated from using
; the points from the fitting curves.
;
; Equations:  
; Initial estimates (P0,T0,e0,A0,B0,F0,G0) and a set of observations 
; (ti,rhoi,thetai) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Radius vector:
;	r = a(1-e^2)/(1 + ecos(nu))
; The fitted (rho_i,theta_i) positions are determined from the orbital elements
; 	tan(theta - Omega) = tan(nu + omega)*cosi
;	rho = r*cos(nu + omega)*sec(theta - Omega)
;
; Procedure:
; Minimize chi squared between data point positions and fitted positions:
;	chi^2 = sum[(rhodata - rhoi)^2/sigmarho^2 
;		    + (thetadata - thetai)^2/sigmatheta^2]
; Take partial derivatives of chi^2 with respect to the orbital elements,
; (P,T,e,a,i,Omega,omega) and set to zero. Solve for the value of the orbital 
; elements that minimize chi^2.
;
; Since the partial derivatives are non-linear, cannot solve for the orbital
; elements that minimize chi^2 analytically.
;
; In turn, use a Newton-Raphson technique to converge upon the solution.
; Replace (rhoi,thetai) in the chi^2 equation with the Taylor series 
; approximation,
; 	x = x| + dx/dP|(P-P0) + dx/dT|(T-T0) + dx/de|(e-e0) 
;	       + dx/da|(a-Aa) + dx/di|(i-i0) + dx/dW|(W-W0) + dx/dw|(w-w0)
; where x=(rho,theta) and the | represents the value evaluated at for the
; initial estimates for the orbital elements. Now, rhoi and thetai are linear 
; in (P,T,e,a,i,W,w), so the partial derivatives can easily be taken and solved
; by setting up a matrix and using Cramer's method.
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;	major:	 semi-major axis (in mas) (a)
;	inc:	 inclination (i)
;	Omega:	 position angle of node (W)
;	omega:	 angle between node and periastron (w)
;	ti:	 time of observation
;	rho_d:	 separation - data
;	theta_d: postion angle - data
;	rho_f:	 sep - fit (determined from orbital elements for time of obs.)
;	theta_f: PA - fit (determined from orbital elements for time of obs.)
;
; INPUT:
;	vb_file: file containing data points in the format:
;		      time  theta  etheta  rho  erho
;		      where theta is the position angle, rho is the separation
;                     and etheta, erho are the measurement uncertainties
;		      (enter file name using quotes)
;	num: number of data points contained in vb_file
;	
; Prompted inputs:
;	initial estimates for all the orbital elements: P,T,e,a,i,Omega,omega
;
; Calls the following routines:
;     calc_deriv_vb.pro
;     calc_deriv_lunar
;     calc_Ei.pro
;     solve_trans.pro
;     calc_vbfit.pro - for plotting model orbits!
;
; Began 14 May 2002
; Modified 11 Nov 2003: 
;       This is the current version of NEWT_RAPH.PRO
;	(v1.1 and v1.2 are older versions)
; 	- restructured program into separate subroutines
; 	- add option of holding elements fixed during iterations
; 13 January 2005:
;       Fix bugs
;       - if solving for only one element, don't use cramer.pro to solve eqn's
;       - fix 360 degree roll overs in (PAdata - PAfit)
;       Save version v4 as backup
; 11 July 2005:
;       - Include Marquadt Method of adjusting diagonal elements of
;         the alpha matrix to ensure convergence.
;       - Fixed non-convergence problems 
;           + do not remove P wrap around effects from dx/dP
;             leave (t-T)/P in full form; net effect is that adjustments
;             are smaller
;       Save v5 as backup 
; 13 July 2005:
;       - Change format of plotting routines to include a call to
;         calc_vbfit
;       Save v6 as backup
; 3 May 2006:
;       - Modified to include lunar occultation projection in orbit fit

; Read in data points from data_file

close,1

!P.font=0
!P.multi=0
!P.charsize=1.2
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.2	;scale factor for adjusting size of plotting symbols

; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1


; keyword_set returns True (1) if variable is defined, False (0) if undefined

ans_mcrit = keyword_set(mcrit)

; keyword_set returns True (1) if variable is defined, False (0) if undefined

ans_allpar = keyword_set(allpar)

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time_vb = dblarr(vb_num)
theta = dblarr(vb_num)
rho = dblarr(vb_num)
dtheta = dblarr(vb_num)
drho = dblarr(vb_num)

openr,lun,vb_file,/get_lun

for i=0, vb_num-1 do begin

	readf, lun, temp1, temp2, temp3, temp4, temp5

	time_vb(i) = temp1
	theta(i) = temp2
	dtheta(i) = temp3
	rho(i) = temp4	
	drho(i) = temp5
	
endfor

close,lun

; convert theta and dtheta to radians

theta = theta*!dpi/180
dtheta = dtheta*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(theta)^2*drho^2 + rho^2*sin(theta)^2*dtheta^2)
dy = sqrt(sin(theta)^2*drho^2 + rho^2*cos(theta)^2*dtheta^2)

; Read in lunar occultation measurements:

time_lun = dblarr(lun_num)
PA_lun = dblarr(lun_num)
sep_lun = dblarr(lun_num)
dsep_lun = dblarr(lun_num)

openr,lun,lun_file

for i=0, lun_num-1 do begin

   
   readf, lun, temp1, temp2, temp3, temp4

   time_lun(i) = temp1
   PA_lun(i) = temp2
   sep_lun(i) = temp3
   dsep_lun(i) = temp4

endfor

close,lun

; convert PA to radians

PA_lun = PA_lun*!dpi/180



; Obtain values for P,T,e,a,i,Omega,omega 

P_arr = dblarr(orb_num)
T_arr = dblarr(orb_num)
e_arr = dblarr(orb_num)
a_arr = dblarr(orb_num)
i_arr = dblarr(orb_num)
W_Cap_arr = dblarr(orb_num)
w_low_arr = dblarr(orb_num)
chi2_arr = dblarr(orb_num)

openr,1,orbit_file

tempstring = ''

temp1 = 0.0d
temp2 = 0.0d
temp3 = 0.0d
temp4 = 0.0d
temp5 = 0.0d
temp6 = 0.0d
temp7 = 0.0d
temp8 = 0.0d
temp9 = 0.0d
temp10 = 0.0d
temp11 = 0.0d

for i=0, orb_num-1 do begin

   ;readf, 1, tempstring
   ;sepvalues = strsplit(tempstring,/extract)

   if (ans_allpar) then readf, 1, temp1,temp2,temp3,temp4,temp5,temp6,temp7,temp8,temp9,temp10,temp11 $ ;result_chi_all   
   else readf, 1, temp1,temp2,temp3,temp4,temp5,temp6,temp7,temp8 ;result_chi

   P_arr(i) = temp1;double(sepvalues(0))
   T_arr(i) = temp2;double(sepvalues(1))
   e_arr(i) = temp3;double(sepvalues(2))
   a_arr(i) = temp4;double(sepvalues(3))
   i_arr(i) = temp5*!dpi/180.0;double(sepvalues(4))*!dpi/180.0
   W_cap_arr(i) = temp6*!dpi/180.0;double(sepvalues(5))*!dpi/180.0
   w_low_arr(i) = temp7*!dpi/180.0;double(sepvalues(6))*!dpi/180.0
   if (ans_allpar) then chi2_arr(i) = temp11 $;double(sepvalues(7))
   else chi2_arr(i) = temp8;double(sepvalues(7))
   
endfor


parallax = 0.0d

print, 'Enter the parallax of the system (in same units as separation):'
print,'(distance of 140, parallax = 7.14 mas)'
read, parallax

ansyd=' '
print,'Is period in years or days (y/d)?'
read,ansyd

if (ansyd eq 'y') then yd = 1.0d
if (ansyd eq 'd') then yd = 1.0d/365.25

; calculate mass of system
mass = a_arr^3/parallax^3/(yd*P_arr)^2


ind = where(chi2_arr eq min(chi2_arr))

period = P_arr(ind(0))
Tperi = T_arr(ind(0))
ecc = e_arr(ind(0))
major = a_arr(ind(0))
inc = i_arr(ind(0))
W_cap = W_cap_arr(ind(0))
w_low = w_low_arr(ind(0))
chi2best = chi2_arr(ind(0))

EL = [period,Tperi,ecc,major,inc,W_cap,w_low]
elLabel = ['P','T','e','a','i','Omega','omega','K1','Vsys1','Vsys2']

EL_vb = [period,Tperi,ecc,major,inc,W_cap,w_low]

; Compute initial model

calc_vbfit, EL_vb, time_vb, theta_f, rho_f, /flag_wa

calc_vbfit, EL_vb, time_lun, theta_lun_f, rho_lun_f, /flag_wa

sep_lun_f =  rho_lun_f*cos(theta_lun_f - PA_lun)

; convert data points to x and y coordinates

xfit = rho_f * cos(theta_f)	; x coordinate
yfit = rho_f * sin(theta_f)	; y coordinate
			
diff_theta = theta - theta_f

; account for any wrap-arounds from 360 deg to 0 deg

for k=0, vb_num-1 do begin
   if (diff_theta(k) ge !dpi) then $
      diff_theta(k) = diff_theta(k) - 2*!dpi
   if (diff_theta(k) le -!dpi) then $
      diff_theta(k) = 2*!dpi + diff_theta(k)	
endfor

chi2 = total((diff_theta)^2/dtheta^2 + (rho - rho_f)^2/drho^2) $
     + total((sep_lun - sep_lun_f)^2/dsep_lun^2) 

chi2_best = chi2

; degrees of freedom
dof = 2d*vb_num + lun_num - 3d

; reduced chi squared:
chi2red = chi2/dof

; plot best-fit orbit and data

tnum=1000.0
tstep = period/tnum
tmin = Tperi

tarr = findgen(tnum)*tstep + tmin

calc_vbfit, EL_vb, tarr, theta_mod, rho_mod, /flag_wa

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

print,'Lunar occultation separation (meas/calc):'
print,sep_lun,sep_lun_f

; Filled Squares for lunar occultation prediction
; Define user symbol to be open rectangle
xrect = frac*[-1.0,-1.0,1.0,1.0,-1.0]
yrect = frac*[-1.0,1.0,1.0,-1.0,-1.0]

; Define user symbol to be closed circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)

; Plot measured positions as filled circles

usersym, frac*cos(Z), frac*sin(Z),/fill

sepmax = 1.2*max(abs([yarr,xarr]))





set_plot, 'ps'
device,/Color,Bits_per_pixel=8,filename='temp_vb_multi.eps',xsize=13,ysize=13,isolatin1=1,/encap

    ;ploterror,yarr,xarr,dy,dx,xtitle='!9D!3RA (mas)',ytitle='!9D!3DEC (mas)',ISOTROPIC=1, $
     plot,yarr,xarr,xtitle='!9D!3RA (mas)',ytitle='!9D!3DEC (mas)',ISOTROPIC=1, $
    	 ;xrange=[sepmax,-sepmax], yrange=[-sepmax,sepmax], $
	 xrange=[30,-130], yrange=[-60,100], $
         xstyle=1,ystyle=1, /nodata,position=[0.18,0.12,0.98,0.98]


    for j=0, orb_num-1 do begin

       tnum=1000.0
       tstep = P_arr(j)/tnum
       tmin = T_arr(j)

       tarr = findgen(tnum)*tstep + tmin

       EL_temp = [P_arr(j),T_arr(j),e_arr(j),a_arr(j),i_arr(j),W_cap_arr(j),w_low_arr(j)]

       print,'Compare these:'
       print,EL_vb
       print,EL_temp
       
       calc_vbfit, EL_temp, tarr, theta_m, rho_m, /flag_wa

       xm = rho_m * cos(theta_m) ; x coordinate
       ym = rho_m * sin(theta_m) ; y coordinate

       ;if (j eq 0) then ncol=1 else ncol=3
       
       oplot,ym,xm,color=170;ncol
    
       ; Connected measured and fit values
       calc_vbfit, EL_temp, time_vb, tf, rf, /flag_wa
       xf = rf * cos(tf) ; x coordinate
       yf = rf * sin(tf) ; y coordinate
       for i=0,vb_num-1 do oplot,[yf(i),yarr(i)],[xf(i),xarr(i)],thick=2

       calc_vbfit, EL_temp, time_lun, theta_f_lun, rho_f_lun, /flag_wa

       xf_lun = rho_f_lun * cos(theta_f_lun) ; x coordinate
       yf_lun = rho_f_lun * sin(theta_f_lun) ; y coordinate

       usersym,xrect,yrect, thick=2      ;,/fill
       oplot,[yf_lun],[xf_lun],psym=8;,symsize=4
       tpred = findgen(2) + 2023.0
       EL_temp(6) = EL_temp(6) + !dpi
       auto_ephemeris,EL_temp,tpred,RApred,DECpred,/noprint
       usersym, frac*cos(Z), frac*sin(Z), thick=2 ; black open circle outline
       ;oplot,RApred,DECpred,psym=8,color=ncol

       
    endfor

    
    ; Connected measured and fit values
    for i=0,vb_num-1 do oplot,[yfit(i),yarr(i)],[xfit(i),xarr(i)],thick=2

    ; Mark computed location with x
    ;oplot,[yfit],[xfit],psym=7

   ; plot best-fit orbit
    oplot,ymod,xmod,color=3

                                ; mark primary with an asterisk
    oplot,[0],[0],psym=2
    oplot,[-sepmax,sepmax],[0,0],linestyle=1
    oplot,[0,0],[-sepmax,sepmax],linestyle=1

    ;Define the symbol to be a closed circle
    usersym, frac*cos(Z), frac*sin(Z), /fill
    oplot,yarr,xarr,psym=8;,color=2

    plot_err,yarr,abs(dy),xarr,abs(dx),vb_num

; plot lunar occultation projection and projected separation of binary

calc_vbfit, EL_vb, time_lun, theta_fit_lun, rho_fit_lun, /flag_wa

xfit_lun = rho_fit_lun * cos(theta_fit_lun)	; x coordinate
yfit_lun = rho_fit_lun * sin(theta_fit_lun)	; y coordinate

ra_lun = 150.0*sin(PA_lun)
dec_lun = 150.0*cos(PA_lun)

;oplot,[ra_lun,-ra_lun],[dec_lun,-dec_lun]
;print,ra_lun,dec_lun

ra_lun_meas = sep_lun*sin(PA_lun)
dec_lun_meas = sep_lun*cos(PA_lun)

if (yfit_lun le 0) then oplot,[0,ra_lun_meas],[0,dec_lun_meas],color=2,thick=5
if (yfit_lun ge 0) then oplot,[0,-ra_lun_meas],[0,-dec_lun_meas],color=2,thick=5


; find equation of line that passes through projected separation
; (perpendicular to lunar occultation PA)
; b = y - mx

m_slope = -sin(PA_lun)/cos(PA_lun)
b_int = sep_lun*cos(PA_lun) - m_slope*sep_lun*sin(PA_lun)

oplot,[150,-150],[150*m_slope+b_int,-150*m_slope+b_int],LineStyle=2

; and equivalent line at minus projected sep
b_int = -sep_lun*cos(PA_lun) + m_slope*sep_lun*sin(PA_lun)

;oplot,[100,-100],[100*m_slope+b_int,-100*m_slope+b_int],LineStyle=2

usersym,xrect,yrect,/fill
oplot,[yfit_lun],[xfit_lun],psym=8,color=127;,symsize=1.5
usersym,xrect,yrect, thick=2;,/fill
oplot,[yfit_lun],[xfit_lun],psym=8
print,yfit_lun,xfit_lun

tpred = findgen(2) + 2023.0
calc_vbfit, EL_vb, tpred, pa_pred, sep_pred, /flag_wa
RApred = sep_pred*sin(pa_pred)
DECpred = sep_pred*cos(pa_pred)
usersym, frac*cos(Z), frac*sin(Z), thick=3
;oplot,RApred,DECpred,psym=8,color=1

xyouts, -10, 85, 'DI Tau',charsize=1.5

xyouts,-100,60,'1991.8',charsize=0.9
xyouts,-60,18,'2022.1',charsize=0.9

device,/close

set_plot,'x'

;stop

free_lun,lun

print,'P min max',min(P_arr),max(P_arr)
print,'T min max',min(T_arr),max(T_arr)
print,'e min max',min(e_arr),max(e_arr)
print,'a min max',min(a_arr),max(a_arr)
print,'i min max',min(i_arr)*180.0/!dpi,max(i_arr)*180.0/!dpi
print,'W min max',min(W_cap_arr)*180.0/!dpi,max(W_cap_arr)*180.0/!dpi
print,'w min max',min(w_low_arr)*180.0/!dpi,max(w_low_arr)*180.0/!dpi

end

;
;
;

pro plot_err, x, dx, y, dy, N

; plot error bars associated with each data point
; Input
;	x: array of x values
;	dx: array of error in x values
;	y: array of y values
;	dy: erray of error in y values
;	N: number of data points


for i=0, N-1 do begin

	oplot, [x(i)-dx(i), x(i)+dx(i)], [y(i), y(i)]
	oplot, [x(i), x(i)], [y(i)-dy(i), y(i)+dy(i)]

endfor

end
