pro plot_range, data_file, num

; Plot a number of possible orbital solutions on one plot
;
; (plot orbits at yearly intervals)
;
; Enter P,T,e for each solution
; Program does a weighted least squares fit to determine Theile-Innes
; elements A,B,F,G (and hence a,i,Omega,omega)
; 
; Equations:  
; Orbital elements (P,T,e) and a set of observations (ti,xi,yi) are known
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)
; The four Thiele-Innes elements are solved through linear least squares fit:
;	xi = AXi + FYi
;	yi = BXi + GYi
; Calculate geometric elements a("),i,Omega,omega from (A,F,B,G)
;
; Essentially, the equations one needs to solve are:
;	E - esinE = 2Pi/P * (t - T)
;	x = A*(cos(Ei) - e) + F*(sqrt(1 - e^2)*sin(Ei))
;	y = B*(cos(Ei) - e) + G*(sqrt(1 - e^2)*sin(Ei))
;
; Paramers:
;	Period:	period (P)
;	Tperi: 	time of periastron passage (T)
;	ecc:	eccentricity (e)
;	major:	semi-major axis (in mas) (a)
;	inc:	inclination (i)
;	Omega:	position angle of node
;	omega:	angle between node and periastron
;	ti:	time of observation
;	xd:	RA - data
;	yd:	DEC - data
;	xf:	RA - fit (determined from orbital elements for time of obs.)
;	yf:	DEC - fit (determined from orbital elements for time of obs.)
;	A,B,F,G: Thiele-Innes elements
;
; Note: xi and yi are not the same convention for RA and DEC chosen to make 
; the orbital plots. Specifically, in the fitting the apparent ellipse 
; (ORBFITE.PRO):
;	x = RA = -rho*sin(theta),   y = DEC = rho*cos(theta)
;	(i.e. North is in the direction of the positive y axis, theta is
;	measured counterclockwise from positive y-axis)
; Whereas in this program, the coordinate system set up by Couteau,
;	x = rho*cos(theta),   y = rho*sin(theta)
;	(i.e. North id in the direction of the positive x-axis and the 
;	y-axis points east)
; Both coordinate systems have the same separation and position angle 
; orietations, they are just rotated by 90 deg with respect to each other.
;
; INPUT:
;	data_file: file containing data points in the format:
;		      time  theta  error(theta)  rho  error(rho)
;		      where theta is the position angle and rho the separation
;		      and their respective errors
;		      (enter file name using quotes)
;	num: number of data points contained in data_file
;	
; Prompted inputs:
;	all of the orbital elements: P,T,e
;
; OUTPUT: 
;	plot of data points and orbital solutions
;
; Modifications to be added:
; 	- error bars on data points
;	- line connecting measurement to claculated point

!P.charsize=1.5
!P.charthick=4.0
!P.thick=5.0
!X.thick=5.0
!Y.thick=5.0
frac=1.4	;scale factor for adjusting size of plotting symbols

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time = dblarr(num)
theta = dblarr(num)
rho = dblarr(num)
dtheta = dblarr(num)
drho = dblarr(num)

openr,lun,data_file,/get_lun

for i=0, num-1 do begin

	readf, lun, temp1, temp2, temp3, temp4, temp5

	time(i) = temp1
	theta(i) = temp2
	dtheta(i) = temp3
	rho(i) = temp4	
	drho(i) = temp5
	
endfor

close,lun

; convert theta and dtheta to radians

theta = theta*!dpi/180
dtheta = dtheta*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(theta)^2*drho^2 + rho^2*sin(theta)^2*dtheta^2)
dy = sqrt(sin(theta)^2*drho^2 + rho^2*cos(theta)^2*dtheta^2)

print, "Do you want to save plot as a postscript file?  (Type yes or no):"
pans = ""
read,pans
; Set plotting device to ps if answer is yes
if (pans eq "yes") then begin
	mydevice=!d.name
	set_plot, 'ps'
	device, /Color, Bits_per_pixel=8, filename='temp.ps' 
endif

answer = "yes"		; answer for plotting another orbital solution
count = 0
	
period = 0d
Tperi = 0d
ecc = 0d

; Define user symbol to be closed circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)

;Define the symbol to be a unit circle with 16 points
usersym, frac*cos(Z), frac*sin(Z), /fill

while (answer eq "yes") do begin

	; Obtain initial values for P,T,e

	print,"Enter P T e (period, time of periastron passage, eccentricity):"
	read, period, Tperi, ecc

	; degrees of freedom
	dof = 2*num - 7


	; Determine the eccentric anomalies Ei: 
	;     mu(ti - T) = Ei - esin(Ei)  ... Kepler's Equation
	;     where mu = 360/P = 2*pi/P
	; Solve this transcendental equation through an 
	; iterative procedure.
	; Use SOLVE_TRANS.PRO

	; array to hold Ei's (the eccentric anomaly)

	Ei = dblarr(num) 

	mu = 2*!dpi/period

	for i=0, num-1 do begin


		Mi = mu*(time(i) - Tperi) 

		; reduce to same epoch

		Mi = 2*!dpi*((time(i) - Tperi)/period $	
			- fix((time(i) - Tperi)/period))

		solve_trans,ecc,Mi,Eit

		Ei(i) = Eit

	endfor


	; Normalized rectangular coordinates Xi & Yi:
	;	Xi = cos(Ei) - e
	;	Yi = sqrt(1 - e^2)*sin(Ei)

	; Xi and Yi are both dblarr(num)

	Xi = cos(Ei) - ecc
	Yi = sqrt(1 - ecc^2)*sin(Ei)

	; print,"Xi:",Xi
	; print,"Yi:",Yi

	; Four Thiele-Innes elements determined from least 
	; squares solution of equations:
	;	xi = AXi + FYi
	;	yi = BXi + GYi
	; Calculate geometric elements a("),i,Omega,omega 
	; from (A,F,B,G)

	linfith, Xi,Yi,xarr,dx,A_cap,F_cap
	linfith, Xi,Yi,yarr,dy,B_cap,G_cap

	; Determine chi squared

	xfit = A_cap*Xi + F_cap*Yi	;fitted points
	yfit = B_cap*Xi + G_cap*Yi

	chi2 = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

	; reduced chi squared:
	chi2 = chi2/dof

	; determine a,i,Omega,omega (using AIWW.PRO)

	aiWw, A_cap,B_cap,F_cap,G_cap,major,inc,W_cap,w_low

	print, "a i Omega omega:", major, (180/!dpi)*inc, $
	(180/!dpi)*W_cap, (180/!dpi)*w_low

	EL = [period,Tperi,ecc,major,inc,W_cap,w_low]

	print, "chi2", chi2
	print, "A B F G:", A_cap,B_cap,F_cap,G_cap
	print, "P T e a i Omega omega"
	print,EL(0),El(1),EL(2),EL(3),180*EL(4)/!dpi,180*EL(5)/!dpi, $
		180*EL(6)/!dpi

	; determine the O-C residuals (Observed - Calculated) for each data pt

	xdiff = xarr - xfit
	ydiff = yarr - yfit

;	print, "O-C residuals (Observed - Calculated):"
;	print, "time xdata xfit e(x) xdif"
;	for i=0, num-1 do print, time(i), xarr(i), xfit(i), dx(i), xdiff(i)

;	print, "time ydata yfit e(y) ydiff"
;	for i=0, num-1 do print, time(i), yarr(i), yfit(i), dy(i), ydiff(i)


	; number of one-year intervals in P (not extended beyond one rotation)

	pnum = fix(period)

	pfac = 1.0

	if pnum lt 10 then begin

		pnum = fix(period*10)
		pfac = 0.1
		
		print,"plotting at 0.1 year intervals!"
	endif

	; determine x,y locations at yearly intervals along orbit

	xorb = dblarr(pnum+1)
	yorb = dblarr(pnum+1)
	torb = dblarr(pnum+1)

	torb(0) = Tperi

	for i=0, pnum do begin

		; Determine the eccentric anomalies Ei:
		; (reduce to same epoch)

		Morb = 2*!dpi*((torb(i) - Tperi)/period $ 
			- fix((torb(i) - Tperi)/period))

		solve_trans,ecc,Morb,Eorb

		; Normalized rectangular coordinates Xi & Yi:
		;	Xi = cos(Ei) - e
		;	Yi = sqrt(1 - e^2)*sin(Ei)
	
		Xiorb = cos(Eorb) - ecc
		Yiorb = sqrt(1 - ecc^2)*sin(Eorb)
		
		xorb(i) = A_cap*Xiorb + F_cap*Yiorb	;fitted points
		yorb(i) = B_cap*Xiorb + G_cap*Yiorb
	
		if (i lt pnum) then torb(i+1) = torb(i) + 1*pfac

	endfor

	;plot data and fitted orbit

	if (count eq 0) then begin

		; axes will be rotated by +90 deg (clockwise direction) inorder
		; to align N-S sep with +y and E-W sep with +x
		; transformation (new coordinates are x',y')
		; 	x' = -y
		;	y' = +x

		maxx = max(-yorb)
		minx = min(-yorb)
		maxy = max(xorb)
		miny = min(xorb)

		ex = 0d
		print, "x: max min ", maxx,minx
		print, "y: max min ", maxy,miny
		print, "By what value would you like to extend x and y axes?"
		read, ex

		plot_title=""
		print, "Enter plot title:"
		read, plot_title

		plot, -yarr, xarr, psym=8, XSTYLE=1, YSTYLE=1, ISOTROPIC=1, $
			xrange=[minx-ex,maxx+ex], yrange=[miny-ex,maxy+ex], $
;			xrange=[-300,150], yrange=[-250,200], $
			xtitle="E-W sep (mas)", ytitle="N-S sep (mas)", $
			title = plot_title

		plot_err,-yarr,abs(dy),xarr,abs(dx),num

		count = count + 1
	endif

	oplot, -yorb, xorb, psym=8

	print, "Would you like to plot another orbit (yes/no)?"
	read, answer

endwhile

; draw axes through origin

plotS, [minx-ex,maxx+ex],[0,0], LineStyle=1
plotS, [0,0],[miny-ex,maxy+ex], LineStyle=1
;plotS, [-300,150],[0,0], LineStyle=1
;plotS, [0,0],[-250,200], LineStyle=1

; plot primary as an asterisk

oplot, [0],[0],psym=2

; set plotting device back to terminal if previously saving to ps file
if (pans eq "yes") then begin 
	device, /close
	set_plot, mydevice
endif

free_lun,lun

end

;
;
;

pro linfith, X, Y, z, sigz, c, d

; Least squares linear fit to the equation: z = cX + dY
; where z = (x,y), c = (A,B), and d = (F,G)
; sigz = errors in z=(x,y) ... (for a weighted least squares fit)
; The user supplies the X,Y,z arrays and errors in z
; Program returns values for c,d
; The least squares fit is performed in the same manner as Harthopf's code
; Minimize chi squared with respect to A,B,F,G (set partial derivatives to 0):
; 	chi^2 = sum[(x - AX - FY)^2/sigx^2 + (y - BX - GY)^2/sigy^2]
; Because x is only a function of (A,F), and y is only a funtion of (B,G),
; minimizing chi squared can proceed independently for each of the x,y 
; variables.

num_data = n_elements(z)	;number of data points
ind_data = num_data -1


c = (total(X*Y/sigz^2)*total(z*Y/sigz^2) - $
	total(z*X/sigz^2)*total(Y*Y/sigz^2))/ $
	(total(X*Y/sigz^2)^2 - total(X*X/sigz^2)*total(Y*Y/sigz^2))

d = (total(X*Y/sigz^2)*total(z*X/sigz^2) - $
	total(z*Y/sigz^2)*total(X*X/sigz^2))/ $
	(total(X*Y/sigz^2)^2 - total(X*X/sigz^2)*total(Y*Y/sigz^2))

;print, "e^2:", e2new

end

;
;
;

pro solve_trans, e, M, EE

; Solve transcendental equation of the form E - esinE = M.
; Use iterative procedure to determine E.
; Initial approximation: E_0 = M + esinM - e^2/2 sin(2M)
; Improve solution by iterating the following to formulae:
;	M_0 = E_0 - esin(E_0)
;	E_1 = E_0 + (M - M_0)/(1 - ecos(E_0))
;	(derivative of Kepler's equation)
;
; Method adapted from Heintz 1978 (p.34-35)
; Results compared with point-&-click graphical method.  Iterative approach 
; leads to exact solution that satisfies E - esinE = M.  Therefore, 
; point-&-click method is subsequently removed from orbit fitting.
;
; INPUT:
;	e: eccentricity
;	M: mean anomaly   M= 2*Pi/P
;
; OUTPUT:
;	EE: eccentric anomaly
;
; Created: 9 May 2002

; Initial approximation:

EE = M + e*sin(M) + e^2/2*sin(2*M)

;print, "Initial approximation for E:", EE

EEi = 0d	; parameter to hold initial value once enter while loop

count = 0

while (abs(EE - EEi) gt 0.000001) do begin

	EEi = EE

	Mi = EEi - e*sin(EEi)

	EE = EEi + (M - Mi)/(1 - e*cos(EEi))

	count=count+1

endwhile

;print, "Final iterated E:", EE

;print,"Number of iterations:",count

end

;
;
;

pro aiWw, A_cap, B_cap, F_cap, G_cap, true_a, inc, Omega_cap, omega_low

; Determine a,i,Omega,omega from Thiele-Innes elements A,B,F,G
; INPUT: A,B,F,G
; OUTPUT: a,i,Omega,omega

BmF = (B_cap - F_cap)
BpF = (B_cap + F_cap)
AmG = (A_cap - G_cap)
ApG = (A_cap + G_cap)

; Determine sign of (Omega + omega) and (Omega - omega)
;print, "Check sign of (Omega + omega) and (Omega - omega)"

sinOpo = BmF		; sin(Omega + omega) is propotional to (B-F)
cosOpo = ApG		; cos(Omega + omega) is propotional to (A+G)

sinOmo = BpF		; sin(Omega - omega) is propotional to (B+F)
cosOmo = AmG		; cos(Omega - omega) is propotional to (A-G)

Opo = atan((BmF)/(ApG))
Omo = atan((BpF)/(AmG))

;print,"Check sign of Omega +/- omega"
;print,"O+o:", Opo
;print,"O-o:", Omo

if ((sinOpo/abs(sinOpo)) ne (sin(Opo)/abs(sin(Opo)))) then Opo = !dpi+Opo
if ((sinOmo/abs(sinOmo)) ne (sin(Omo)/abs(sin(Omo)))) then Omo = !dpi+Omo

;print,"Corrected O+o:", Opo
;print,"Corrected O-o:", Omo

Omega_cap = (Opo + Omo)/2d

;print,"Check if 0 < Omega < pi"
;print,"Omega:", Omega_cap 

if (Omega_cap gt !dpi) then omega_cap = omega_cap - !dpi
if (Omega_cap lt 0d) then omega_cap = omega_cap + !dpi

;print,"Corrected Omega:", Omega_cap 

omega_low = Opo - Omega_cap

inc = 2* atan(sqrt((BpF*sin(Opo))/(BmF*sin(Omo))))

true_a = sqrt((A_cap*G_cap - B_cap*F_cap)/cos(inc))

end

;
;
;

pro plot_err, x, dx, y, dy, N

; plot error bars associated with each data point
; Input
;	x: array of x values
;	dx: array of error in x values
;	y: array of y values
;	dy: erray of error in y values
;	N: number of data points


for i=0, N-1 do begin

	oplot, [x(i)-dx(i), x(i)+dx(i)], [y(i), y(i)]
	oplot, [x(i), x(i)], [y(i)-dy(i), y(i)+dy(i)]

endfor

end
