pro plot_velocity_sb1_multi_orb, sb1_file, sb1_num, bootstrap_file

; Plot SB1 orbital fits
;
; The program requires an initial guess that can be calculated from using
; the points from the fitting curves.
;
; Calls the following routines:
;     calc_deriv_SB1.pro
;     calc_deriv_SB2.pro
;     calc_Ei.pro
;     solve_trans.pro
;     calc_sb2fit.pro - for plotting model orbits
;
; Equations:  
; Initial estimates (P0,T0,e0,a1sini|0,a2sini|0,omega0,Vsys0) and  
; a set of observations (ti,V1i,V2i) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Projected semi-major axes (a1sini, a2sini):
;       a1sini = (P/2pi)*K1*(1 - e^2)^1/2
;       a2sini = (P/2pi)*K2*(1 - e^2)^1/2
; Velocity amplitudes (K1, K2):
;       K = (2pi/P)(asini/(1 - e^2)^1/2)
; The fitted velocities (V1_i, V2_i) are determined from the orbital elements:
;       V1 = K1*[e*cos(omega) + cos(nu + omega)] + Vsys
;       V2 = K2*[e*cos(omega) + cos(nu + omega)] + Vsys
;
; Procedure:
; Minimize chi squared between measured velocities and fitted velocities:
;	chi^2 = sum[(V1data - V1fit)^2/sigmaV1^2 
;		    + (V2data - V2fit)^2/sigmaV2^2]
; where the sums run over all data points
; Take partial derivatives of chi^2 with respect to the orbital elements,
; (P,T,e,a1sini,a2sini,omega,Vsys) and set to zero. Solve for the value of
; the orbital elements that minimize chi^2.
;
; Since the equations for V1 and V2 are non-linear, cannot solve for 
; the orbital elements that minimize chi^2 analytically.
;
; In turn, use a Newton-Raphson technique to converge upon the solution.
; Replace (V1fit,V2fit) in the chi^2 equation with the Taylor series 
; approximation,
; 	x = x| + dx/dP|(P-P0) + dx/dT|(T-T0) + dx/de|(e-e0) + dx/dw|(w-w0)
;	       + dx/d(a1sini)|(a1sini - a1sini|0) 
;              + dx/d(a2sini)|(a2sini - a2sini|0)
;              + dx/d(Vsys)|(Vsys - Vsys0)
; where x=(V1,V2) and the | represents the value evaluated for the
; initial estimates of the orbital elements. Now, V1fit and V2fit 
; are linear in (P,T,e,a1sini,a2sini,w,Vsys), so the partial derivatives
; can easily be taken and solved by setting up a matrix and using 
; Cramer's method.
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;       K1:      velocity amplitude of primary (in km/s) (K1)
;       K2:      velocity amplitude of secondary (in km/s) (K2)
;	a1sini:	 projected semi-major axis of primary orbit (in km) (a1sini)
;	a2sini:	 projected semi-major axis of secondary orbit (in km) (a2sini)
;	omega:	 angle between node and periastron (w)
;	ti:	 time of observation
;	V1_d:	 velocity of primary - measured data
;	V2_d:    velocity of secondary - measured data
;	V1_f:	 V1 - fit (determined from orbital elements for time of obs.)
;	V2_f:    V2 - fit (determined from orbital elements for time of obs.)
;       Vsys:    systemic velocity
;
; INPUT:
;	data_file: file containing data points in the format:
;		      time  V1  errV1  V2  errV2
;		      where V1 is the radial velocity of the primary,
;                     V2 is the radial velocity of the secondary,
;                     and errV1 and errV2 are the measurement errors
;		      (enter file name using quotes)
;	num: number of data points contained in data_file
;
; Prompted inputs:
;	initial estimates for all the orbital elements: 
;             P, T, e, a1sini, a2sini, omega, Vsys
;       vector identifying which orbital elements to solve for
;             enter 0 to hold element fixed, 1 to vary)
;             ex.   1 1 1 1 1 1 1 -> solves for all
;                   0 0 0 1 0 0 0 -> solves only for a1 of primary
; OUTPUT:
;	best fit values of orbital elements: P,T,e,a1sini,a2sini,omega,Vsys
;       and corresponding errors
;
; Notes on units:
;    P, T, and time should be in same units (days or years)
;    K1, K2, V1, V2 should be in same units (km/s)
; Notes on units:
;    Velocities should be in km/s
;    P, T, and time should be in same units (days or years)
;    asini can be in any units (conversion to meters is prompted for)
;        -> program converts to km when computed fit-velocities    
;
; Began 23 June 2005
;
; Version 1: Orignal copy
; Version 2: 
;    - Include Marquadt Method of adjusting diagonal elements of
;      the alpha matrix to ensure convergence. 5 July 2005
;    - Fixed non-convergence problems   7 July 2005
;        + do not remove P wrap around effects from dV/dP
;          leave (t-T)/P in full form; net effect is that adjustments
;          smaller
;        + fixed primary/secondary indices in setting dV1/dVsys and dV2/dVsys
;    - Designed for SB2's only!!!
; Version 3: 7 July 2005
;    - Add SB1 + SB2 joint solutions
; 
; 20 October 2005: Modification of fit_velocity_sb2sb1.pro
;
; 26 March 2007 Save unmodified program as version 1
; 26 March 2007 Fit SB2 parameters in terms of K1 and K2
;               (as opposed to a1sini and a2sini)
;

close,1

!P.font = 0
!P.charsize=1.0
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
;frac=1.0
frac=0.7
symthick=2.0
lthick = 2

; WR 137
!P.font = 0
!P.charsize=1.0
!P.charthick=1.0
!P.thick=1.0
!X.thick=2.0
!Y.thick=2.0
;frac=1.0
frac=0.7
symthick=1.0
lthick = 2

;!P.charsize=1.5
;!P.charthick=4.0
;!P.thick=5.0
;!X.thick=5.0
;!Y.thick=5.0
;frac=1.5        ; scale factor for adjusting size of plotting symbols
;symthick=5.0    ; thickness for symbol lines

;!P.charsize=1.25
;!P.charthick=4.0
;!P.thick=5.0
;!X.thick=5.0
;!Y.thick=5.0
;frac=1.2
;symthick=5.0


ans_new = 'n'
time_new = [57264.68414303d,57357.57184577d,57645.72620430d,57708.55729279d,58388.66245791d,58438.60805040d,58476.57539703d,58672.89993467d,58713.89288440d,58739.87343602d,58770.80688949d,58774.65030062d]


; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1

; Read in data points from data files

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time_sb1 = dblarr(sb1_num)
V1_sb1 = dblarr(sb1_num)
dV1_sb1 = dblarr(sb1_num)

; Read SB1 measurements

openr,lun,sb1_file,/get_lun

for i=0, sb1_num-1 do begin

	readf, lun, temp1, temp2, temp3

	time_sb1(i) = temp1
	V1_sb1(i) = temp2
	dV1_sb1(i) = temp3
	
endfor

close,lun

totnum = sb1_num
time = time_sb1


if (ans_new eq 'y') then begin
; Figure new RVs to highlight
num_new = n_elements(time_new)
ind_new = dblarr(num_new)
for i=0, num_new-1 do begin
   tdiff = abs(time_sb1 - time_new(i))
   ind = where(tdiff eq min(tdiff))
   ind_new(i) = ind(0)
endfor
endif



; Obtain values for P,T,e,a1sini,a2sini,omega,Vsys

period = 0d
Tperi = 0d
ecc = 0d
K1 = 0d
omega = 0d
Vsys = 0d

print,"Enter P,T,e,K1,omega,Vsys:"
read,period,Tperi,ecc,K1,omega,Vsys

print,"Initial P,T,e,K1,omega,Vsys:"
print,period,Tperi,ecc,K1,omega,Vsys

;convert omega to radians
omega = omega*!dpi/180

ansyd=' '
print,'Is period in years or days (y/d)?'
read,ansyd

if (ansyd eq 'y') then y_d = 365.25
if (ansyd eq 'd') then y_d = 1.0

; conversion factor for y/d to seconds
; (need to convert period to sections when computing velocities)
sec = 1.0*y_d*24.0*60.0*60.0

;meters = 0.0
;print,'How many meters are in semi-major axis units?'
;print,'(1 Gm = 1d9 m)'
;read,meters
;
;km = meters/1d3

EL = [period,Tperi,ecc,K1,omega,Vsys]
EL_sb1 = [period,Tperi,ecc,K1,omega,Vsys]

ELadj = EL
ELadj_sb1 = EL_sb1

elLabel = ['P','T','e','K1','omega','Vsys']


; Plot random sample of multiple solutions:
nplot = 0
print,'How many orbits do you want to plot:'
read,nplot

; Read in orbital parameters from bootstrap iterations

nsamp = 20000

; Intialize parameters to hold results from bootstrap iterations
P_boot = dblarr(nsamp)
T_boot = dblarr(nsamp)
e_boot = dblarr(nsamp)
a_boot = dblarr(nsamp)
i_boot = dblarr(nsamp)
Wc_boot = dblarr(nsamp)
wl_boot = dblarr(nsamp)
K1_boot = dblarr(nsamp)
Vsys_boot = dblarr(nsamp)
M1_boot = dblarr(nsamp)
M2_boot = dblarr(nsamp)
Mratio_boot = dblarr(nsamp)
Mtot_boot = dblarr(nsamp)
Mfunc_boot = dblarr(nsamp)

tempstring = ''

openr,1,bootstrap_file
; Read in header line and discard
readf, 1, tempstring
i = 0
while not EOF(1) do begin
    readf, 1, tempstring
    sepvalues = strsplit(tempstring,/extract)
    P_boot(i) = double(sepvalues(0))
    T_boot(i) = double(sepvalues(1))
    e_boot(i) = double(sepvalues(2))
    a_boot(i) = double(sepvalues(3))
    i_boot(i) = double(sepvalues(4))
    Wc_boot(i) = double(sepvalues(5))
    wl_boot(i) = double(sepvalues(6))
    K1_boot(i) = double(sepvalues(7))
    Vsys_boot(i) = double(sepvalues(8))
    i = i + 1
endwhile
close,1
nsamp = i
P_boot = P_boot(0:nsamp-1)
T_boot = T_boot(0:nsamp-1) 
e_boot = e_boot(0:nsamp-1) 
a_boot = a_boot(0:nsamp-1) 
i_boot = i_boot(0:nsamp-1) 
Wc_boot = Wc_boot(0:nsamp-1) 
wl_boot = wl_boot(0:nsamp-1) 
K1_boot = K1_boot(0:nsamp-1) 
Vsys_boot = Vsys_boot(0:nsamp-1)

; Convert i, Omega, omega from degrees to radians:
i_boot = i_boot*!dpi/180.0d
Wc_boot = Wc_boot*!dpi/180.0d
wl_boot = wl_boot*!dpi/180.0d

; Compute model orbit for best fit:

calc_sb1fit, EL_sb1, time_sb1, V1fit

; compute phase from time of periastron passage
phase_sb1 = (time_sb1 - Tperi)/period
; reduce phase to -1.0 to 1.0
phase_sb1 = phase_sb1 - fix(phase_sb1)
; keep phase between 0.0 to 1.0
for i=0, sb1_num-1 do if (phase_sb1(i) lt 0.0) $
  then phase_sb1(i) = phase_sb1(i) + 1.0

; plot model and data of initial guess

tnum = 1000.0
tmin = Tperi - period*0.25
tmax = Tperi + period + period*0.25
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

phase_mod = (tarr-Tperi)/period
;phase_mod = phase_mod - fix(phase_mod)
;for i=0, tnum-1 do if (phase_mod(i) lt 0.0) $
;  then phase_mod(i) = phase_mod(i) + 1.0

calc_sb1fit, EL_sb1, tarr, V1mod

vmax = max(V1_sb1  + dV1_sb1)
vmin = min(V1_sb1 - dV1_sb1)


ans_plot2 = 'n'
print,'Do you want to plot another orbit (y/n)?'
read, ans_plot2

if (ans_plot2 eq 'y') then begin
   print,"Enter P,T,e,K1,omega,Vsys:"
   read,period2,Tperi2,ecc2,K12,omega2,Vsys2

   omega2 = omega2*!dpi/180

   EL_sb1_2 = [period2,Tperi2,ecc2,K12,omega2,Vsys2]

   calc_sb1fit, EL_sb1_2, time_sb1, V1fit_2
   calc_sb1fit, EL_sb1_2, tarr, V1mod_2

endif



; Define user symbol to be rectangle
xrect = frac*[-1.0,-1.0,1.0,1.0,-1.0]
yrect = frac*[-1.0,1.0,1.0,-1.0,-1.0]
usersym,xrect,yrect

; Define user symbol to be plus sign
xplus = frac*[-1.0,1.0,0.0,0.0,0.0]
yplus = frac*[0.0,0.0,0.0,1.0,-1.0]
usersym,xplus,yplus

; Define user symbol to be open circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)
;Define the symbol to be a unit circle with 16 points
usersym, frac*cos(Z), frac*sin(Z)


calc_sb1fit, EL_sb1, time_sb1, V1calc

print,'Time  V1  dV1  O-C  (O-C)/dv1  Phase'
for i=0, sb1_num-1 do  print,time_sb1(i),V1_sb1(i),dV1_sb1(i),(V1_sb1(i) - V1calc(i)),(V1_sb1(i) - V1calc(i))/dV1_sb1(i),phase_sb1(i)

;; since model is plotted as solid line, it will suffice just to
;; connect data point directly to solid lne (without plotting fit positions)
;for l=0, sb2_num-1 do oplot,[phase_sb2(l),phase_sb2(l)],[V1_sb2(l),V1_f_sb2(l)]
;for l=0, sb2_num-1 do oplot,[phase_sb2(l),phase_sb2(l)],[V2_sb2(l),V2_f_sb2(l)]
;for l=0, sb1_num-1 do oplot,[phase_sb1(l),phase_sb1(l)],[V1_sb1(l),V1_f_sb1(l)]


; Determine plotting ranges based on RV curves and measurements
vmax = max([V1mod,V1_sb1+dV1_sb1])
vmin = min([V1mod,V1_sb1-dV1_sb1])

; Extend plotting ranges by 10% 
; (be careful - this will be different depending whether RVs or
; positve or negative)
vrange = abs(vmax - vmin)
;if (vmax ge 0) then vmax = vmax + 0.1*vrange else vmax = vmax - 0.1*vrange
;if (vmin lt 0) then vmin = vmin - 0.1*vrange else vmin = vmin + 0.1*vrange
vmax = vmax + 0.02*vrange
vmin = vmin - 0.02*vrange

; Define user symbol to be rectangle
xrect = frac*[-1.0,-1.0,1.0,1.0,-1.0]
yrect = frac*[-1.0,1.0,1.0,-1.0,-1.0]
usersym,xrect,yrect

; Define user symbol to be plus sign
xplus = frac*[-1.0,1.0,0.0,0.0,0.0]
yplus = frac*[0.0,0.0,0.0,1.0,-1.0]
usersym,xplus,yplus

; Define user symbol to be open circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)
;Define the symbol to be a unit circle with 16 points
usersym, frac*cos(Z), frac*sin(Z)



set_plot, 'ps'
device, /Color, Bits_per_pixel=8, filename='temp.eps',xsize=15,ysize=12,/encap

xr = [-0.1,1.1]
;vmin = -40.0
;vmax = 45.0

;ploterror,phase_sb1,V1_sb1,dV1_sb1,psym=4, errthick=1, $
plot,phase_sb1,V1_sb1,psym=4, $
  xtitle='Orbital Phase',ytitle = 'Radial Velocity (km/s)', $
  xstyle=1,ystyle=1,/NoData, yrange=[vmin,vmax],xrange = xr;, $
  ;position=[0.19,0.12,0.95,0.98];

for i=1, nplot-1 do begin

   P_i    = P_boot(i)
   T_i    = T_boot(i)
   e_i    = e_boot(i)
   a_i    = a_boot(i)
   i_i    = i_boot(i)
   Wc_i   = Wc_boot(i)
   wl_i   = wl_boot(i)
   K1_i   = K1_boot(i)
   Vsys_i = Vsys_boot(i)

   EL_i = [P_i,T_i,e_i,a_i,i_i,Wc_i,wl_i,K1_i,Vsys_i]
   EL_vb_i = [P_i,T_i,e_i,a_i,i_i,Wc_i,wl_i]
   EL_sb1_i = [P_i,T_i,e_i,K1_i,wl_i,Vsys_i]

   ti = findgen(tnum)*tstep + tmin
   phase_i = (tarr-T_i)/P_i

   calc_sb1fit, EL_sb1_i, ti, Vmodi

   ; plot best-fit orbit
   oplot,phase_i,Vmodi,color=160;130;

endfor

oplot,phase_mod,V1mod,LineStyle=0.0,thick=lthick,color=3;0;3;1

if (ans_plot2 eq 'y') then oplot,phase_mod,V1mod_2,LineStyle=2.0,color=1,thick=lthick


if (ans_new eq 'y') then begin

   usersym, frac*cos(Z), frac*sin(Z),thick=symthick
   oplot,phase_sb1,V1_sb1,psym=8,color=3

   usersym, frac*cos(Z), frac*sin(Z),thick=symthick,/fill
   oplot,phase_sb1(ind_new),V1_sb1(ind_new),psym=8;,color=3
 
endif else begin
   usersym, frac*cos(Z), frac*sin(Z),thick=symthick,/fill
   ;oploterror,phase_sb1,V1_sb1,dV1_sb1,psym=8, errthick=1;,color=3
   oplot,phase_sb1,V1_sb1,psym=8 ;,color=3
   ind = where(phase_sb1-1 lt 0.0)
   ;oploterror,phase_sb1(ind)-1,V1_sb1(ind),dV1_sb1(ind),psym=8, errthick=1;,color=3
   oplot,phase_sb1(ind)-1,V1_sb1(ind),psym=8;,color=3
   ind = where(phase_sb1+1 gt 0.0)
   ;oploterror,phase_sb1(ind)+1,V1_sb1(ind),dV1_sb1(ind),psym=8, errthick=1;,color=3
   oplot,phase_sb1(ind)+1,V1_sb1(ind),psym=8 ;,color=3endelse
endelse
   
plot,phase_sb1,V1_sb1,psym=4,yrange=[vmin,vmax],xrange = xr, $
  xstyle=1,ystyle=1,/NoData, /noerase;, $
          ;position=[0.19,0.12,0.95,0.98],XTICKFORMAT="(A1)"


;plot,phase_sb1,V1_sb1,psym=4,yrange=[vmin,vmax],xrange =[-0.1,1.1], $
;  xtitle = 'Phase', ytitle = 'Velocity (km/s)', xstyle=1,/NoData
;usersym,xplus,yplus,thick=symthick
;oplot,phase_sb1,V1_sb1,psym=8,color=3
;oplot,phase_mod,V1mod,LineStyle=0.0,color=3,thick=lthick

;plot_erry,phase_sb1,V1_sb1,dV1_sb1,sb1_num,3

;oplot,[-0.1,1.1],[Vsys,Vsys],Linestyle=2

device, /close
set_plot, 'x'


; Plot residuals and RV curve


; Determine plotting ranges based on RV curves and measurements
vmax = max([V1mod,V1_sb1+dV1_sb1])
vmin = min([V1mod,V1_sb1-dV1_sb1])

; Extend plotting ranges by 10% 
; (be careful - this will be different depending whether RVs or
; positve or negative)
vrange = abs(vmax - vmin)
;if (vmax ge 0) then vmax = vmax + 0.1*vrange else vmax = vmax - 0.1*vrange
;if (vmin lt 0) then vmin = vmin - 0.1*vrange else vmin = vmin + 0.1*vrange
vmax = vmax + 0.02*vrange
vmin = vmin - 0.02*vrange

;vmin=-60
;vmax=50

;vmin = -114
;vmax = -58

;vmin = -102.5;-102
;vmax = -94.5;-95

set_plot, 'ps'
device, /Color, Bits_per_pixel=8, filename='temp_res.eps',xsize=15,ysize=14,/encap

xr = [-0.1,1.1]
;xr = [-0.25,1.25]

ploterror,phase_sb1,V1_sb1,dV1_sb1,psym=4,yrange=[vmin,vmax],xrange = xr, $
  ytitle = 'Radial Velocity (km/s)', xstyle=1,ystyle=1,/NoData, $
  position=[0.15,0.35,0.98,0.98],XTICKFORMAT="(A1)", errthick=1


if (ans_new eq 'y') then begin

   usersym, frac*cos(Z), frac*sin(Z),thick=symthick
   oplot,phase_sb1,V1_sb1,psym=8;,color=3

   usersym, frac*cos(Z), frac*sin(Z),thick=symthick,/fill
   oplot,phase_sb1(ind_new),V1_sb1(ind_new),psym=8,color=3
 
endif else begin
   usersym, frac*cos(Z), frac*sin(Z),thick=symthick,/fill
   oploterror,phase_sb1,V1_sb1,dV1_sb1,psym=8, errthick=1,color=3
   ind = where(phase_sb1-1 lt 0.0)
   oploterror,phase_sb1(ind)-1,V1_sb1(ind),dV1_sb1(ind),psym=8, errthick=1,color=3
   ind = where(phase_sb1+1 gt 0.0)
   oploterror,phase_sb1(ind)+1,V1_sb1(ind),dV1_sb1(ind),psym=8, errthick=1,color=3
endelse

for i=1, nplot-1 do begin

   P_i    = P_boot(i)
   T_i    = T_boot(i)
   e_i    = e_boot(i)
   a_i    = a_boot(i)
   i_i    = i_boot(i)
   Wc_i   = Wc_boot(i)
   wl_i   = wl_boot(i)
   K1_i   = K1_boot(i)
   Vsys_i = Vsys_boot(i)

   EL_i = [P_i,T_i,e_i,a_i,i_i,Wc_i,wl_i,K1_i,Vsys_i]
   EL_vb_i = [P_i,T_i,e_i,a_i,i_i,Wc_i,wl_i]
   EL_sb1_i = [P_i,T_i,e_i,K1_i,wl_i,Vsys_i]

   ti = findgen(tnum)*tstep + tmin
   phase_i = (tarr-T_i)/P_i

   calc_sb1fit, EL_sb1_i, ti, Vmodi

   ; plot best-fit orbit
   oplot,phase_i,Vmodi,color=130;

endfor

oplot,phase_mod,V1mod,LineStyle=0.0,thick=lthick,color=0;3;1

if (ans_plot2 eq 'y') then oplot,phase_mod,V1mod_2,LineStyle=2.0,color=1,thick=lthick

plot,phase_sb1,V1_sb1,psym=4,yrange=[vmin,vmax],xrange = xr, $
  xstyle=1,ystyle=1,/NoData, /noerase, $
          position=[0.15,0.35,0.98,0.98],XTICKFORMAT="(A1)"


diffV1 = V1_sb1 - V1fit
if (ans_plot2 eq 'y') then diffV1_2 = V1_sb1 - V1fit_2

dmax = max(abs([diffV1+dV1_sb1,diffV1-dV1_sb1]))*1.05
dmin=-dmax
;print,dmax
;dmax = 19.99
;dmax=25.0
;dmin=-16
;dmax = 1.5;1.3
;dmin = -2.5;-2.3

ploterror,phase_sb1,diffV1,dV1_sb1,psym=4,yrange=[dmin,dmax],xrange = xr, $
  xtitle = 'Phase', ytitle = 'Residuals (km/s)', xstyle=1,/NoData,ystyle=1, $
  position=[0.15,0.08,0.98,0.35],/noerase, errthick=1

if (ans_new eq 'y') then begin

   usersym, frac*cos(Z), frac*sin(Z),thick=symthick
   oplot,phase_sb1,diffV1,psym=8;,color=3

   usersym, frac*cos(Z), frac*sin(Z),thick=symthick,/fill
   oplot,phase_sb1(ind_new),diffV1(ind_new),psym=8;,color=3

endif else begin
   usersym, frac*cos(Z), frac*sin(Z),thick=symthick,/fill
   oploterror,phase_sb1,diffV1,dV1_sb1,psym=8, errthick=1,color=3;1
   ind = where(phase_sb1-1 lt 0.0)
   oploterror,phase_sb1(ind)-1,diffV1(ind),dV1_sb1(ind),psym=8, errthick=1,color=3;1
   ind = where(phase_sb1+1 gt 1.0)
   oploterror,phase_sb1(ind)+1,diffV1(ind),dV1_sb1(ind),psym=8, errthick=1,color=3;1
   if (ans_plot2 eq 'y') then begin
      usersym,xrect,yrect,/fill
      oploterror,phase_sb1,diffV1_2,dV1_sb1,psym=8, errthick=1,color=3
      oploterror,phase_sb1-1,diffV1_2,dV1_sb1,psym=8, errthick=1,color=3
      oploterror,phase_sb1+1,diffV1_2,dV1_sb1,psym=8, errthick=1,color=3
   endif
endelse

oplot,xr,[0,0],Linestyle=1,thick=1;lthick

plot,phase_sb1,diffV1,psym=4,yrange=[dmin,dmax],xrange = xr, $
  xstyle=1,/NoData,ystyle=1, $
  position=[0.15,0.08,0.98,0.35],/noerase

device, /close
set_plot, 'x'

free_lun,lun

print,'chi2: ',total(diffV1^2/dV1_sb1^2),total(diffV1^2/dV1_sb1^2)/n_elements(V1_sb1)
if (ans_plot2 eq 'y') then print,'chi2: ',total(diffV1_2^2/dV1_sb1^2),total(diffV1_2^2/dV1_sb1^2)/n_elements(V1_sb1)

;stop,'Type .cont to continue'

end

;
;
;

pro plot_erry, x, y, dy, N, color_ind

; plot error bars associated with each data point
; Input
;	x: array of x values
;	dx: array of error in x values
;	y: array of y values
;	dy: erray of error in y values
;	N: number of data points
;       color_ind: color index

for i=0, N-1 do begin

;	oplot, [x(i)-dx(i), x(i)+dx(i)], [y(i), y(i)]
	oplot, [x(i), x(i)], [y(i)-dy(i), y(i)+dy(i)],color=color_ind

endfor

end
