pro plot_velocity_sb2, sb2_file, sb2_num,rainbow=rainbow

; Plots SB2 orbital data vs. phase!
;
; The program requires an initial guess that can be calculated from using
; the points from the fitting curves.
;
; Calls the following routines:
;     calc_deriv_SB1.pro
;     calc_deriv_SB2.pro
;     calc_Ei.pro
;     solve_trans.pro
;     calc_sb2fit.pro - for plotting model orbits
;
; Equations:  
; Initial estimates (P0,T0,e0,K1_0,K2_0,omega0,Vsys0) and  
; a set of observations (ti,V1i,V2i) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Projected semi-major axes (a1sini, a2sini):
;       a1sini = (P/2pi)*K1*(1 - e^2)^1/2
;       a2sini = (P/2pi)*K2*(1 - e^2)^1/2
; Velocity amplitudes (K1, K2):
;       K = (2pi/P)(asini/(1 - e^2)^1/2)
; The fitted velocities (V1_i, V2_i) are determined from the orbital elements:
;       V1 = K1*[e*cos(omega) + cos(nu + omega)] + Vsys
;       V2 = K2*[e*cos(omega) + cos(nu + omega)] + Vsys
;
; Procedure:
; Minimize chi squared between measured velocities and fitted velocities:
;	chi^2 = sum[(V1data - V1fit)^2/sigmaV1^2 
;		    + (V2data - V2fit)^2/sigmaV2^2]
; where the sums run over all data points
; Take partial derivatives of chi^2 with respect to the orbital elements,
; (P,T,e,a1sini,a2sini,omega,Vsys) and set to zero. Solve for the value of
; the orbital elements that minimize chi^2.
;
; Since the equations for V1 and V2 are non-linear, cannot solve for 
; the orbital elements that minimize chi^2 analytically.
;
; In turn, use a Newton-Raphson technique to converge upon the solution.
; Replace (V1fit,V2fit) in the chi^2 equation with the Taylor series 
; approximation,
; 	x = x| + dx/dP|(P-P0) + dx/dT|(T-T0) + dx/de|(e-e0) + dx/dw|(w-w0)
;	       + dx/d(a1sini)|(a1sini - a1sini|0) 
;              + dx/d(a2sini)|(a2sini - a2sini|0)
;              + dx/d(Vsys)|(Vsys - Vsys0)
; where x=(V1,V2) and the | represents the value evaluated for the
; initial estimates of the orbital elements. Now, V1fit and V2fit 
; are linear in (P,T,e,a1sini,a2sini,w,Vsys), so the partial derivatives
; can easily be taken and solved by setting up a matrix and using 
; Cramer's method.
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;       K1:      velocity amplitude of primary (in km/s) (K1)
;       K2:      velocity amplitude of secondary (in km/s) (K2)
;	a1sini:	 projected semi-major axis of primary orbit (in km) (a1sini)
;	a2sini:	 projected semi-major axis of secondary orbit (in km) (a2sini)
;	omega:	 angle between node and periastron (w)
;	ti:	 time of observation
;	V1_d:	 velocity of primary - measured data
;	V2_d:    velocity of secondary - measured data
;	V1_f:	 V1 - fit (determined from orbital elements for time of obs.)
;	V2_f:    V2 - fit (determined from orbital elements for time of obs.)
;       Vsys:    systemic velocity
;
; INPUT:
;	data_file: file containing data points in the format:
;		      time  V1  errV1  V2  errV2
;		      where V1 is the radial velocity of the primary,
;                     V2 is the radial velocity of the secondary,
;                     and errV1 and errV2 are the measurement errors
;		      (enter file name using quotes)
;	num: number of data points contained in data_file
;
; Prompted inputs:
;	initial estimates for all the orbital elements: 
;             P, T, e, a1sini, a2sini, omega, Vsys
;       vector identifying which orbital elements to solve for
;             enter 0 to hold element fixed, 1 to vary)
;             ex.   1 1 1 1 1 1 1 -> solves for all
;                   0 0 0 1 0 0 0 -> solves only for a1 of primary
; OUTPUT:
;	best fit values of orbital elements: P,T,e,a1sini,a2sini,omega,Vsys
;       and corresponding errors
;
; Notes on units:
;    P, T, and time should be in same units (days or years)
;    K1, K2, V1, V2 should be in same units (km/s)
; Notes on units:
;    Velocities should be in km/s
;    P, T, and time should be in same units (days or years)
;    asini can be in any units (conversion to meters is prompted for)
;        -> program converts to km when computed fit-velocities    
;
; Began 23 June 2005
;
; Version 1: Orignal copy
; Version 2: 
;    - Include Marquadt Method of adjusting diagonal elements of
;      the alpha matrix to ensure convergence. 5 July 2005
;    - Fixed non-convergence problems   7 July 2005
;        + do not remove P wrap around effects from dV/dP
;          leave (t-T)/P in full form; net effect is that adjustments
;          smaller
;        + fixed primary/secondary indices in setting dV1/dVsys and dV2/dVsys
;    - Designed for SB2's only!!!
; Version 3: 7 July 2005
;    - Add SB1 + SB2 joint solutions
; 
; 20 October 2005: Modification of fit_velocity_sb2sb1.pro
;
; 26 March 2007 Save unmodified program as version 1
; 26 March 2007 Fit SB2 parameters in terms of K1 and K2
;               (as opposed to a1sini and a2sini)
;
; 7 June 2007 Modified version of plot_velocity_sb2sb1.pro

;!P.font = 0
;!P.charsize=1.2
;!P.charthick=4.0
;!P.thick=4.0
;!X.thick=4.0
;!Y.thick=4.0
;frac=1.5        ; scale factor for adjusting size of plotting symbols
;symthick=4.0    ; thickness for symbol lines

!P.font = 0
!P.charsize=1.0
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.0        ; scale factor for adjusting size of plotting symbols
;frac = 0.6
symthick=2.0    ; thickness for symbol lines
lthick = 2

;!P.font = 0
;!P.multi=0
;!P.charsize=1.0
;!P.charthick=3.0
;!P.thick=3.0
;!X.thick=3.0
;!Y.thick=3.0
;frac=1.3        ; scale factor for adjusting size of plotting symbols
;;frac=1.0        ; scale factor for adjusting size of plotting symbols
;symthick=2.0    ; thickness for symbol lines
;lthick = 3

ans_new = 'n'
time_new = [58751.89608178d,58752.85947264d,58783.84372021d,58784.84049307d,58785.82335423d,58786.87812944d,58787.80539730d,58813.72459947d,58814.75423769d,58815.73936606d,58817.69200117d]   ; Iota Ori


; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1

ans_rainbow = keyword_set(rainbow)

if (ans_rainbow) then begin
    loadct,40  ; rainbow+black
    ; Colors:
    ; red = 250, orange=220, yellow=195, green=150, blue=60, purple=30
    ; dark red = 260, aqua = 100
    ctab = [250,220,195,150,60,30,40,100]
endif

; Read in data points from data files

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time_sb2 = dblarr(sb2_num)
V1_sb2 = dblarr(sb2_num)
V2_sb2 = dblarr(sb2_num)
dV1_sb2 = dblarr(sb2_num)
dV2_sb2 = dblarr(sb2_num)

; Read SB2 measurements

openr,lun,sb2_file,/get_lun

for i=0, sb2_num-1 do begin

	readf, lun, temp1, temp2, temp3, temp4, temp5

	time_sb2(i) = temp1
	V1_sb2(i) = temp2
	dV1_sb2(i) = temp3
	V2_sb2(i) = temp4	
	dV2_sb2(i) = temp5
	
endfor

close,lun

; Check to see if there are any 0 weight observations

izero = where((dV1_sb2 eq 0.0) or (dV1_sb2 eq 0.0), num_zero)

if (num_zero gt 0) then begin
   ; Save 0 weight data
   time_sb2_zero = time_sb2(izero)
   V1_sb2_zero = V1_sb2(izero)
   V2_sb2_zero = V2_sb2(izero)
   ; Extract good data
   igood = where((dV1_sb2 ne 0.0) or (dV1_sb2 ne 0.0),sb2_num)
   time_sb2 = time_sb2(igood)
   V1_sb2  = V1_sb2(igood) 
   dV1_sb2 = dV1_sb2(igood)
   V2_sb2  = V2_sb2(igood) 	
   dV2_sb2 = dV2_sb2(igood)
endif

totnum = sb2_num
time = dblarr(totnum)

if (ans_new eq 'y') then begin
; Figure new RVs to highlight
num_new = n_elements(time_new)
ind_new = dblarr(num_new)
for i=0, num_new-1 do begin
   tdiff = abs(time_sb2 - time_new(i))
   ind = where(tdiff eq min(tdiff))
   ind_new(i) = ind(0)
endfor
endif

for i=0, sb2_num-1 do time(i) = time_sb2(i)

; Obtain values for P,T,e,a1sini,a2sini,omega,Vsys

period = 0d
Tperi = 0d
ecc = 0d
K1 = 0d
K2 = 0d
omega = 0d
Vsys = 0d

print,"Enter P,T,e,K1,K2,omega,Vsys:"
read,period,Tperi,ecc,K1,K2,omega,Vsys

print,"Initial P,T,e,K1,K2,omega,Vsys:"
print,period,Tperi,ecc,K1,K2,omega,Vsys

;convert omega to radians
omega = omega*!dpi/180

EL = [period,Tperi,ecc,K1,K2,omega,Vsys]
EL_sb2 = [period,Tperi,ecc,K1,K2,omega,Vsys]

ELadj = EL
ELadj_sb2 = EL_sb2

elLabel = ['P','T','e','K1','K2','omega','Vsys']

calc_sb2fit, EL_sb2, time_sb2, V1fit, V2fit

; Calculate the chi2:

chi2 = total((V1_sb2 - V1fit)^2/dV1_sb2^2 + (V2_sb2 - V2fit)^2/dV2_sb2^2)

; degrees of freedom
dof = 2.0*sb2_num - 7

; reduced chi squared:
chi2red = chi2/dof

print,'chi2:        ',chi2
print,'Reduced chi2:',chi2red


;ansyd=' '
;print,'Is period in years or days (y/d)?'
;read,ansyd
;
;if (ansyd eq 'y') then y_d = 365.25
;if (ansyd eq 'd') then y_d = 1.0
;
;; conversion factor for y/d to seconds
;; (need to convert period to sections when computing velocities)
;sec = 1.0*y_d*24.0*60.0*60.0
;
;meters = 0.0
;print,'How many meters are in semi-major axis units?'
;print,'(1 Gm = 1d9 m)'
;read,meters
;
;km = meters/1d3
;
;; Determine the velocity amplitudes
;K1 = (2.0*!dpi/(period*sec))*(a1sini*km/sqrt(1 - ecc^2))
;K2 = (2.0*!dpi/(period*sec))*(a2sini*km/sqrt(1 - ecc^2))
;print,'K1',K1
;print,'K2',K2

; compute phase from time of periastron passage
phase_sb2 = (time_sb2 - Tperi)/period

; reduce phase to -1.0 to 1.0
phase_sb2 = phase_sb2 - fix(phase_sb2)

; keep phase between 0.0 to 1.0
for i=0, sb2_num-1 do if (phase_sb2(i) lt 0.0) $
  then phase_sb2(i) = phase_sb2(i) + 1.0

if (num_zero gt 0) then begin
   phase_sb2_zero = (time_sb2_zero - Tperi)/period
   phase_sb2_zero = phase_sb2_zero - fix(phase_sb2_zero)
   for i=0, num_zero-1 do if (phase_sb2_zero(i) lt 0.0) $
   then phase_sb2_zero(i) = phase_sb2_zero(i) + 1.0
endif

; plot model and data of initial guess

tnum = 1000.0
tmin = Tperi - period*0.1
tmax = Tperi + period + period*0.1
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

phase_mod = (tarr-Tperi)/period
;phase_mod = phase_mod - fix(phase_mod)
;for i=0, tnum-1 do if (phase_mod(i) lt 0.0) $
;  then phase_mod(i) = phase_mod(i) + 1.0

calc_sb2fit, EL_sb2, tarr, V1mod, V2mod

; Determine plotting ranges based on RV curves and measurements
vmax = max([max(V1mod),max(V2mod),V1_sb2+dV1_sb2,V1_sb2-dV1_sb2,V2_sb2+dV2_sb2,V2_sb2-dV2_sb2])
vmin = min([min(V1mod),min(V2mod),V1_sb2+dV1_sb2,V1_sb2-dV1_sb2,V2_sb2+dV2_sb2,V2_sb2-dV2_sb2])

; Extend plotting ranges by 10% 
; (be careful - this will be different depending whether RVs or
; positve or negative)
;if (vmax ge 0) then vmax = vmax*1.1 else vmax = vmax*1.1
;if (vmin lt 0) then vmin = vmin*1.1 else vmin = vmin*0.9

; Extend plotting ranges by 5%
vrange = vmax - vmin
vmin = vmin - 0.05*vrange
vmax = vmax + 0.05*vrange

; Define user symbol to be rectangle
xrect = frac*[-1.0,-1.0,1.0,1.0,-1.0]
yrect = frac*[-1.0,1.0,1.0,-1.0,-1.0]
usersym,xrect,yrect

; Define user symbol to be plus sign
xplus = frac*[-1.0,1.0,0.0,0.0,0.0]
yplus = frac*[0.0,0.0,0.0,1.0,-1.0]
usersym,xplus,yplus

; Define user symbol to be open circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)
;Define the symbol to be a unit circle with 16 points
usersym, frac*cos(Z), frac*sin(Z)


set_plot, 'ps'
;device, /Color, Bits_per_pixel=8, filename='temp.ps',xsize=17,ysize=15
;device, /Color, Bits_per_pixel=8, filename='temp.ps',xsize=14,ysize=12
device, /Color, Bits_per_pixel=8, filename='temp.eps',xsize=12,ysize=10,/encap

;vmax=220
;vmin = -120

;vmax=100
;vmin = -100

;vmax= 51
;vmin = 30

;vmax = 100
;vmin = -50

;vmax = 19
;vmin = 8

;vmax = 5
;vmin = -5

print,vmin,vmax

ploterror,phase_sb2,V1_sb2,dV1_sb2,psym=4,yrange=[vmin,vmax],xrange =[-0.1,1.1], $
  xtitle = '!3Phase', ytitle = 'Radial Velocity (km/s)', xstyle=1,ystyle=1,/NoData

oplot,[-0.1,1.1],[Vsys,Vsys],Linestyle=1

oplot,phase_mod,V1mod,LineStyle=0.0,color=3,thick=lthick
oplot,phase_mod,V2mod,Linestyle=0.0,color=1,thick=lthick*1.5

usersym, frac*cos(Z), frac*sin(Z),thick=symthick,/fill
oploterror,phase_sb2,V1_sb2,dV1_sb2,psym=8,color=3
usersym,xrect,yrect,thick=symthick,/fill
oploterror,phase_sb2,V2_sb2,dV2_sb2,psym=8,color=1

; Include time labels !!!!
vsign = abs(V1_sb2)/V1_sb2
;xyouts,phase_sb2,V1_sb2+vsign*0.5,string(format='(f6.1)',time_sb2),charsize=0.8,alignment=0.5
;xyouts,phase_sb2,V1_sb2+vsign*0.5,string(format='(i4)',time_sb2),charsize=0.8,alignment=0.5

;xyouts,0.0,4,'omega_A = 122.6'
;xyouts,0.0,4,'omega_A = 302.6'

;xyouts,0.0,4.3,'omega_A = 129.2'
;xyouts,0.0,4.3,'omega_A = 309.2'

device, /close
set_plot, 'x'

; Plot both residuals on same panel

set_plot, 'ps'
device, /Color, Bits_per_pixel=8, filename='temp_res.eps',xsize=12,ysize=14,/encap

ploterror,phase_sb2,V1_sb2,dV1_sb2,psym=4,yrange=[vmin,vmax],xrange =[-0.1,1.1], $
  ytitle = 'Radial Velocity (km/s)', xstyle=1,ystyle=1,/NoData, $
  position=[0.15,0.35,0.98,0.98],XTICKFORMAT="(A1)"

oploterror,phase_sb2,V2_sb2,dV2_sb2,psym=4,/NoData

oplot,[-0.1,1.1],[Vsys,Vsys],Linestyle=1

oplot,phase_mod,V1mod,LineStyle=0.0,color=3,thick=lthick
oplot,phase_mod,V2mod,Linestyle=0.0,color=1,thick=lthick*1.5

if (ans_new eq 'y') then begin

   usersym, frac*cos(Z), frac*sin(Z),thick=symthick
   oplot,phase_sb2,V1_sb2,psym=8,color=3
   usersym,xrect,yrect,thick=symthick
   oplot,phase_sb2,V2_sb2,psym=8,color=1

   usersym, frac*cos(Z), frac*sin(Z),thick=symthick,/fill
   oplot,phase_sb2(ind_new),V1_sb2(ind_new),psym=8,color=3
   usersym,xrect,yrect,thick=symthick,/fill
   oplot,phase_sb2(ind_new),V2_sb2(ind_new),psym=8,color=1

endif else begin
   usersym, frac*cos(Z), frac*sin(Z),thick=symthick,/fill
   oplot,phase_sb2,V1_sb2,psym=8,color=3
   usersym,xrect,yrect,thick=symthick,/fill
   oplot,phase_sb2,V2_sb2,psym=8,color=1
endelse

;usersym, frac*cos(Z), frac*sin(Z),thick=symthick,/fill
;oplot,phase_sb2(90:139),V1_sb2(90:139),psym=8
;usersym,xrect,yrect,thick=symthick,/fill
;oplot,phase_sb2(90:139),V2_sb2(90:139),psym=8

diffV1 = V1_sb2 - V1fit
diffV2 = V2_sb2 - V2fit

dmax = max(abs([diffV1,diffV2]))
;dmax = 12.0
;dmax = 20.0
;dmax = 100.0

ploterror,phase_sb2,diffV1,dV1_sb2,psym=4,yrange=[-dmax,dmax],xrange =[-0.1,1.1], $
  xtitle = 'Phase', ytitle = 'Residual (km/s)', xstyle=1,/NoData,ystyle=1, $
  position=[0.15,0.08,0.98,0.35],/noerase
oploterror,phase_sb2,diffV2,dV2_sb2,/nodata

if (ans_new eq 'y') then begin

   usersym, frac*cos(Z), frac*sin(Z),thick=symthick
   oplot,phase_sb2,diffV1,psym=8,color=3
   usersym,xrect,yrect,thick=symthick
   oplot,phase_sb2,diffV2,psym=8,color=1

   usersym, frac*cos(Z), frac*sin(Z),thick=symthick,/fill
   oplot,phase_sb2(ind_new),diffV1(ind_new),psym=8,color=3
   usersym,xrect,yrect,thick=symthick,/fill
   oplot,phase_sb2(ind_new),diffV2(ind_new),psym=8,color=1

endif else begin
   usersym, frac*cos(Z), frac*sin(Z),thick=symthick,/fill
   oplot,phase_sb2,diffV1,psym=8,color=3
   usersym,xrect,yrect,thick=symthick,/fill
   oplot,phase_sb2,diffV2,psym=8,color=1
endelse

;usersym, frac*cos(Z), frac*sin(Z),thick=symthick,/fill
;oplot,phase_sb2(90:139),diffV1(90:139),psym=8
;usersym,xrect,yrect,thick=symthick,/fill
;oplot,phase_sb2(90:139),diffV2(90:139),psym=8

oplot,[-0.1,1.1],[0,0],Linestyle=2,thick=lthick

device, /close
set_plot, 'x'

; Plot residuals below RV curves, but make separate residual plots
; for RV1 and RV2 measurements

set_plot, 'ps'
;device, /Color, Bits_per_pixel=8, filename='temp_res2.eps',xsize=12,ysize=16,/encap
device, /Color, Bits_per_pixel=8, filename='temp_res2.eps',xsize=15,ysize=16,/encap

;vmax = 220
;vmin = -110

;vmax = 5.0
;vmin = -5.0

;vmax = 19.0
;vmin = 8.0

;vmin=-50.0
;vmax=55.0

;vmin=-150
;vmax=350.0

plot,phase_sb2,V1_sb2,psym=4,yrange=[vmin,vmax],xrange =[-0.1,1.1], $
  ytitle = 'Radial Velocity (km/s)', xstyle=1,ystyle=1,/NoData, $
  position=[0.15,0.48,0.98,0.98],XTICKFORMAT="(A1)"

oplot,[-0.1,1.1],[Vsys,Vsys],Linestyle=1

if (num_zero gt 0) then begin
   usersym, frac*cos(Z), frac*sin(Z),thick=symthick,/fill
   oplot,phase_sb2_zero,V1_sb2_zero,psym=8
   usersym,xrect,yrect,thick=symthick,/fill
   oplot,phase_sb2_zero,V2_sb2_zero,psym=8
   calc_sb2fit, EL_sb2, time_sb2_zero, V1fit_zero, V2fit_zero
   diffV1_zero = V1_sb2_zero - V1fit_zero
   diffV2_zero = V2_sb2_zero - V2fit_zero
endif

oplot,phase_mod,V1mod,LineStyle=0.0,thick=lthick,color=3
oplot,phase_mod,V2mod,Linestyle=0.0,thick=lthick,color=1

if (ans_new eq 'y') then begin

   usersym, frac*cos(Z), frac*sin(Z),thick=symthick
   oploterror,phase_sb2,V1_sb2,dV1_sb2,psym=8, errthick=1
   oplot,phase_sb2,V1_sb2,psym=8;,color=3

   usersym,xrect,yrect,thick=symthick
   oploterror,phase_sb2,V2_sb2,dV2_sb2,psym=8, errthick=1
   oplot,phase_sb2,V2_sb2,psym=8;,color=1

   usersym, frac*cos(Z), frac*sin(Z),thick=symthick,/fill
   oploterror,phase_sb2(ind_new),V1_sb2(ind_new),dV1_sb2(ind_new),psym=8, errthick=1
   oplot,phase_sb2(ind_new),V1_sb2(ind_new),psym=8,color=3

   usersym,xrect,yrect,thick=symthick,/fill
   oploterror,phase_sb2(ind_new),V2_sb2(ind_new),dV2_sb2(ind_new),psym=8, errthick=1
   oplot,phase_sb2(ind_new),V2_sb2(ind_new),psym=8,color=1

endif else begin

   usersym, frac*cos(Z), frac*sin(Z),thick=symthick,/fill
   oploterror,phase_sb2,V1_sb2,dV1_sb2,psym=8, errthick=1
   oplot,phase_sb2,V1_sb2,psym=8,color=3
   ;oplot,phase_sb2(90:150),V1_sb2(90:150),psym=8,color=ctab(4)
   ;oplot,phase_sb2(0:89),V1_sb2(0:89),psym=8,color=ctab(7)

   usersym,xrect,yrect,thick=symthick,/fill
   oploterror,phase_sb2,V2_sb2,dV2_sb2,psym=8, errthick=1
   oplot,phase_sb2,V2_sb2,psym=8,color=1
   ;oplot,phase_sb2(90:150),V2_sb2(90:150),psym=8,color=ctab(6)
   ;oplot,phase_sb2(0:89),V2_sb2(0:89),psym=8,color=ctab(0)
endelse




;oplot,[-0.1,1.1],[0,0],Linestyle=2,thick=lthick

;usersym, frac*cos(Z), frac*sin(Z),thick=symthick,/fill
;oplot,phase_sb2(90:139),V1_sb2(90:139),psym=8
;usersym,xrect,yrect,thick=symthick,/fill
;oplot,phase_sb2(90:139),V2_sb2(90:139),psym=8

diffV1 = V1_sb2 - V1fit
diffV2 = V2_sb2 - V2fit

dmax1 = 1.1*max(abs([diffV1,dV1_sb2+diffV1,dV1_sb2-diffV1]))
dmax2 = 1.1*max(abs([diffV2,dV2_sb2+diffV2,dV2_sb2-diffV2]))
;dmax1 = 30.0
;dmax2 = 10.0
;dmax1 = 9.0
;dmax2 = 9.0
;dmax1 = 59.99
;dmax2 = 14.99
;dmax1 = 0.9
;dmax2 = 0.9
;dmax1 = 99.99
;dmax2 = 99.99

ploterror,phase_sb2,diffV1,dV1_sb2,psym=4,yrange=[-dmax1,dmax1],xrange =[-0.1,1.1], $
  ytitle = 'Residual (km/s)', xstyle=1, ystyle=1, errthick=1, /NoData, $
  position=[0.15,0.28,0.98,0.48],/noerase,XTICKFORMAT="(A1)"

if (ans_new eq 'y') then begin

   usersym, frac*cos(Z), frac*sin(Z),thick=symthick
   if (num_zero gt 0) then oplot,phase_sb2_zero,diffV1_zero,psym=8
   oplot,phase_sb2,diffV1,psym=8;,color=3

   usersym, frac*cos(Z), frac*sin(Z),thick=symthick,/fill
   if (num_zero gt 0) then oplot,phase_sb2_zero(ind_new),diffV1_zero(ind_new),psym=8
   oplot,phase_sb2(ind_new),diffV1(ind_new),psym=8,color=3

endif else begin

   usersym, frac*cos(Z), frac*sin(Z),thick=symthick,/fill
   if (num_zero gt 0) then oplot,phase_sb2_zero,diffV1_zero,psym=8
   oplot,phase_sb2,diffV1,psym=8,color=3
   ;oplot,phase_sb2(90:150),diffV1(90:150),psym=8,color=ctab(4)
   ;oplot,phase_sb2(0:89),diffV1(0:89),psym=8,color=ctab(7)

    ; red = 250, orange=220, yellow=195, green=150, blue=60, purple=30
    ; dark red = 260, aqua = 100
    ; ctab = [250,220,195,150,60,30,260,100]

endelse

oplot,[-0.1,1.1],[0,0],Linestyle=2,thick=lthick

ploterror,phase_sb2,diffV2,dV2_sb2,psym=4,yrange=[-dmax2,dmax2],xrange =[-0.1,1.1], $
  xtitle = 'Phase', ytitle = 'Residual (km/s)', xstyle=1, ystyle=1, errthick=1, /NoData,  $
  position=[0.15,0.08,0.98,0.28],/noerase

if (ans_new eq 'y') then begin

   usersym,xrect,yrect,thick=symthick
   if (num_zero gt 0) then oplot,phase_sb2_zero,diffV2_zero,psym=8
   oplot,phase_sb2,diffV2,psym=8;,color=1

   usersym,xrect,yrect,thick=symthick,/fill
   if (num_zero gt 0) then oplot,phase_sb2_zero(ind_new),diffV2_zero(ind_new),psym=8
   oplot,phase_sb2(ind_new),diffV2(ind_new),psym=8,color=1

endif else begin

   usersym,xrect,yrect,thick=symthick,/fill
   if (num_zero gt 0) then oplot,phase_sb2_zero,diffV2_zero,psym=8
   oplot,phase_sb2,diffV2,psym=8,color=1
   ;oplot,phase_sb2(90:150),diffV2(90:150),psym=8,color=ctab(6)
   ;oplot,phase_sb2(0:89),diffV2(0:89),psym=8,color=ctab(0)

endelse

oplot,[-0.1,1.1],[0,0],Linestyle=2,thick=lthick

device, /close
set_plot, 'x'





;; since model is plotted as solid line, it will suffice just to
;; connect data point directly to solid lne (without plotting fit positions)
;for l=0, sb2_num-1 do oplot,[phase_sb2(l),phase_sb2(l)],[V1_sb2(l),V1_f_sb2(l)]
;for l=0, sb2_num-1 do oplot,[phase_sb2(l),phase_sb2(l)],[V2_sb2(l),V2_f_sb2(l)]
;for l=0, sb1_num-1 do oplot,[phase_sb1(l),phase_sb1(l)],[V1_sb1(l),V1_f_sb1(l)]


diff = abs(V1mod - V2mod)
print,'Max RV Diff: ',max(diff)


; Make table of residuals

openw,lun,'temp_table'


;for i=0,sb2_num-1 do printf,lun,format='(f9.3,f7.3,f8.2,f6.2,f8.2,f6.2,f7.2,f7.2)',time_sb2(i),phase_sb2(i),V1_sb2(i),dV1_sb2(i),V2_sb2(i),dV2_sb2(i),diffV1(i),diffV2(i)

for i=0,sb2_num-1 do printf,lun,format='(f9.3,f7.3,f8.2,f6.1,f8.1,f6.1,f7.1,f7.1)',time_sb2(i),phase_sb2(i),V1_sb2(i),dV1_sb2(i),V2_sb2(i),dV2_sb2(i),diffV1(i),diffV2(i)

close,lun


free_lun,lun

;stop,'Type .cont to continue'

end

;
;
;

pro plot_erry, x, y, dy, N, color_ind

; plot error bars associated with each data point
; Input
;	x: array of x values
;	dx: array of error in x values
;	y: array of y values
;	dy: erray of error in y values
;	N: number of data points
;       color_ind: color index

for i=0, N-1 do begin

;	oplot, [x(i)-dx(i), x(i)+dx(i)], [y(i), y(i)]
	oplot, [x(i), x(i)], [y(i)-dy(i), y(i)+dy(i)],color=color_ind

endfor

end
