pro remove_orb_outliers, data_file, sigclip=sigclip, removal=removal


; Note: This file must use uncertainties from original WDS weights or modified WDS weights
; (not scaled uncertainties to force reduced chi2 to 1).
; The scaled uncertainties are only used to compute the number of
; sigma discrepancy for each residual.
;
; Method:
; - Scale uncertainties so to force redchi2 = 1
; - Next is to check measurements scaled uncertainties for outlier rejection
;
; Default rejection method (lower weights or removal):
; - If res > 3sig (res/sig > 3) use the follow approach
;   + If measurement weight > mean(all_weights) then lower weight by factor of 4
;     (lowering weights by a factor of 4 is the same as increasing 
;     uncertainties by a factor of 16!)
;   + If measurement weight < mean(all_weights) then remove the measurements 
;     from the data file
; - Approach modified so that no observations are removed - only the
;   weights are lowered by a factor of sqrt(10), corresponding to uncertainties
;   increased by a factor of 10.  [31 May 2017]
;
; Alternate rejection method (removal):
; - Set /removal flag
; - If res > 3sig (res/sig > 3), then remove the measurement from the data file
; 
; Alternate rejection method (sigma clipping):
; - Set /sigclip flag
; - Reject measurements if res(i) > 3*stdev(res)


close,1

!P.font=0
!P.multi=0
!P.charsize=1.0
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.2	;scale factor for adjusting size of plotting symbols

loadct,0

; Determine which optional arguments and flags are set

; Do you want to use sigma clipping?
; Reject measurements if res(i) > 3*stdev(res)
; keyword_set returns True (1) if sigclip is defined, False (0) if undefined

ans_sigclip = keyword_set(sigclip)

; Do you want to simply remove data that exceeds 3 sigma?
; Reject measurements if res(i) > 3*sigma(i)
; keyword_set returns True (1) if removal is defined, False (0) if undefined

ans_rem = keyword_set(removal)

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

num = 10000

time = dblarr(num)
sepmeas = dblarr(num)
PAmeas = dblarr(num)
dsep = dblarr(num)
dPA = dblarr(num)

openr,1,data_file

i = 0
while not EOF(1) do begin

   readf, 1, temp1, temp2, temp3, temp4, temp5

   time(i) = temp1
   PAmeas(i) = temp2
   dPA(i) = temp3
   sepmeas(i) = temp4	
   dsep(i) = temp5

    i = i + 1
endwhile
close,1
num = i
time = time(0:num-1)
pameas = pameas(0:num-1)
dpa = dpa(0:num-1)
sepmeas = sepmeas(0:num-1)
dsep = dsep(0:num-1)

; convert theta and dtheta to radians

PAmeas = PAmeas*!dpi/180
dPA = dPA*!dpi/180

; convert data points to x and y coordinates

xmeas = sepmeas * cos(PAmeas)	; x coordinate
ymeas = sepmeas * sin(PAmeas)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(PAmeas)^2*dsep^2 + sepmeas^2*sin(PAmeas)^2*dPA^2)
dy = sqrt(sin(PAmeas)^2*dsep^2 + sepmeas^2*cos(PAmeas)^2*dPA^2)


; Obtain values for P,T,e,a,i,Omega,omega 

period = 0d
Tperi = 0d
ecc = 0d
major = 0d
inc = 0d
W_cap = 0d
w_low = 0d

print,"Enter P,T,e,a,i,Omega,omega:"
read,period,Tperi,ecc,major,inc,W_cap,w_low

print, "Vary each orbital element?"
print, "For each element, enter 0 to hold element fixed, 1 to vary:"
print, "[P,T,e,a,i,Omega,omega]"
read, f0,f1,f2,f3,f4,f5,f6

elfix=[f0,f1,f2,f3,f4,f5,f6]
nEl = n_elements(elfix)

k=0
for i=0, 6 do if (elfix(i) eq 0) then k=k+1

mfit = nEl - k 		; number of elements to improve

print,"Initial P,T,e,a,i,Omega,omega:"
print,period,Tperi,ecc,major,inc,W_cap,w_low

EL_deg = [period,Tperi,ecc,major,inc,W_cap,w_low]

elLabel = ['P','T','e','a','i','Omega','omega']

; Arrays to hold rejected observations
time_remove = dblarr(num)
pameas_remove = dblarr(num)
dpa_remove = dblarr(num)
sepmeas_remove = dblarr(num)
dsep_remove = dblarr(num)
niter_remove = strarr(num)

; Arrays to hold observations with lowered weights
time_lower = dblarr(num)
pameas_lower = dblarr(num)
dpa_lower = dblarr(num)
sepmeas_lower = dblarr(num)
dsep_lower = dblarr(num)
niter_lower = strarr(num)

; Temporary arrays to hold saved observations during each iteration
time_save = dblarr(num)
pameas_save = dblarr(num)
dpa_save = dblarr(num)
sepmeas_save = dblarr(num)
dsep_save = dblarr(num)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; START big while loop here ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Continue rejection algorithm until no observations are removed
; or no uncertainties are adjusted.

; Total number of lowered or removed observations
total_remove = 0.0
total_lower = 0.0

; Number of observations rejected or adjusted during this iteration.
; Set to 1 to initiate while loop
count_remove = 1
count_lower = 1

niter = 0

num_orig = num

dsep_orig = dsep
dpa_orig = dpa

while (count_remove gt 0) or (count_lower gt 0) do begin

; Initialize arrays for scaled error bars:

dsep_scale = dsep
dpa_scale = dpa

; Compute initial orbit fit:

;auto_newt_raph, data_file, num, EL_deg, elfix

auto_newt_raph_data, time, pameas, dpa_scale, sepmeas, dsep_scale, EL_deg, elfix, ELerr=ELerr, /showplot

; Define EL_rad with i,W,w in radians

EL_rad = EL_deg
for i=4,6 do EL_rad(i)=EL_rad(i)*!dpi/180.0

; Compute residuals

calc_vbfit, EL_rad, time, pafit, sepfit

xfit = sepfit * cos(pafit)	; x coordinate
yfit = sepfit * sin(pafit)	; y coordinate

; Compute chi2

diffPA = (PAmeas - PAfit)
for i=0, num-1 do begin
    if (diffPA(i) ge !dpi) then diffPA(i) = diffPA(i) - 2.0*!dpi
    if (diffPA(i) le -!dpi) then diffPA(i) = diffPA(i) + 2.0*!dpi
endfor
 
chi2 = total((diffPA)^2/dPA^2 + (sepmeas - sepfit)^2/dsep^2)

; degrees of freedom
dof = 2d*num - mfit

; reduced chi squared:
chi2red = chi2/dof

scale = sqrt(chi2red)

; Scale uncertainties to force chi2red = 1
; The scale uncertainties are used to compute number of sigma
; discrepancy for the residuals
dsep_scale = dsep * scale

; Scale uncertaintiesin pa and sep independently.
; Don't automatically assume equal uncertainties in x and y.
; This will be the case if the measurements are already scaled that way.
dpa_scale = dpa * scale

; propagate errors in sep and PA to x and y:
; under the assumption that dx = dy = drho
; propagate errors to PA: 
;dpa_scale = cos(pameas)^2*sepmeas*dsep_scale/xmeas^2

print,'----------------------------------------------'
print,'Original chi2:', chi2
print,'Original reduced chi2:', chi2red

print,'Scale uncertainties by a factor of: ',scale

; Recompute chi2 using new uncertainties
chi2 = total((diffPA)^2/dPA_scale^2 + (sepmeas - sepfit)^2/dsep_scale^2)
; reduced chi squared:
chi2red = chi2/dof

 ; print O-C residuals (Observed - Calculated)

 xOC = xmeas - xfit
 yOC = ymeas - yfit

 openw,1,'xyresiduals'
 printf,1,"O-C residuals (data - fit)"
 printf,1,"date     dx       xO-C   xOC/dx dy      yO-C   yOC/dy"
 ;printf,1,"date   dx   xdata - xfit   xOC/dx   dy   ydata - yfit   yOC/dy"
 for i=0, num-1 do printf,1,format='(f10.4,f8.3,f9.3,f9.3,f8.3,f9.3,f9.3)', $
     time(i), dx(i), xOC(i), xOC(i)/dx(i), dy(i), yOC(i), yOC(i)/dy(i)
 close,1

; print O-C residuals in sep,PA
sepOC = sepmeas - sepfit
paOC = (PAmeas - PAfit)*180.0/!dpi
for i=0, num-1 do begin
   if (paOC(i) gt 180.0) then paOC(i) = paOC(i)-360.0
   if (paOC(i) lt -180.0) then paOC(i) = paOC(i)+360.0
endfor
pares_dpa = paOC/(dpa_scale*180.0/!dpi)
sepres_dsep = sepOC/dsep_scale

openw,1,'rtresiduals'
printf,1,"O-C residuals (data - fit)"
printf,1,"date     dpa       pa(O-C)   paOC/dpa  dsep    sep(O-C)   sepOC/dsep"
for i=0, num-1 do printf,1,format='(f10.4,f9.3,f10.3,f10.3,f8.3,f9.3,f9.3)', $
     time(i), dPA_scale(i)*180.0/!dpi, paOC(i), pares_dpa(i), dsep_scale(i), sepOC(i), sepres_dsep(i)
close,1



; Flag measurements where res/sigma > 3
; If original weight exceeds the average mean value then decrease
; weight by a factor of 4.  Otherwise remove from data file.

weight = 1.0/sqrt(dsep)
ave_weight = mean(weight)

print,'Mean weight: ',mean(weight)
print,'Median weight: ',median(weight)


stdev_pares = stdev(paOC)
stdev_sepres = stdev(sepOC)


print,'Residuals:'
print,'Mean and stdev in PA:  ',mean(paOC),stdev(paOC)
print,'Mean and stdev in sep: ',mean(sepOC),stdev(sepOC)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; sigma clipping rejection method
if (ans_sigclip) then begin

   print,'----------------------------------------------'
   print,'Measurements where res > 3*stdev_res:'
   print,"date        dpa     pa(O-C)  paOC/dpa    dsep  sep(O-C)   sepOC/dsep"
   for i=0, num-1 do begin

      if ( (abs(paOC(i)) gt 3.0*stdev_pares) or (abs(sepOC(i)) gt 3.0*stdev_sepres) ) then $
         print,format='(f10.4,f9.3,f10.3,f10.3,f8.3,f9.3,f9.3)', $
               time(i),dpa_scale(i)*180.0/!dpi,paOC(i),pares_dpa(i),dsep_scale(i),sepOC(i),sepres_dsep(i)
 
   endfor

   count_save = 0.0
   count_remove = 0.0
   count_lower = 0.0

   print,'----------------------------------------------'
   for i=0, num-1 do begin
      
      if ( (abs(paOC(i)) gt 3.0*stdev_pares) or (abs(sepOC(i)) gt 3.0*stdev_sepres) ) then begin

         print,'Removing: ',time(i),pameas(i)*180.0/!dpi,dpa_orig(i)*180.0/!dpi,sepmeas(i),dsep_orig(i)

         time_remove(total_remove) = time(i)
         pameas_remove(total_remove) = pameas(i)
         dpa_remove(total_remove) = dpa_orig(i)
         sepmeas_remove(total_remove) = sepmeas(i)
         dsep_remove(total_remove) = dsep_orig(i)
         niter_remove(total_remove) = strcompress(string(niter),/remove_all)

         count_remove = count_remove + 1
         total_remove = total_remove + 1

      endif else begin
         ;printf,1,format='(f10.4,f9.3,f7.3,f10.5,f8.5)',time(i),pameas(i)*180.0/!dpi,dpa(i)*180.0/!dpi,sepmeas(i),dsep(i)

         time_save(count_save) = time(i)
         pameas_save(count_save) = pameas(i)
         dpa_save(count_save) = dpa_scale(i)
         sepmeas_save(count_save) = sepmeas(i)
         dsep_save(count_save) = dsep_scale(i)

         count_save = count_save + 1
      endelse
   endfor
   close,1

   print,'----------------------------------------------'
   print,'Original number of observations: ', num
   print,'Number of observations saved: ',count_save
   print,'Number of observations removed: ',count_remove
   print,'----------------------------------------------'

   num = count_save
   time = time_save(0:num-1)
   pameas = pameas_save(0:num-1)
   dpa = dpa_save(0:num-1)
   sepmeas = sepmeas_save(0:num-1)
   dsep = dsep_save(0:num-1)

endif else begin

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Remove data if resisual > 3sigma 
if (ans_rem) then begin

   print,'----------------------------------------------'
   print,'Measurements where res/sigma > 3:'
   print,"date        dpa     pa(O-C)  paOC/dpa    dsep  sep(O-C)   sepOC/dsep"
   for i=0, num-1 do begin

      if ( (abs(pares_dpa(i)) gt 3.0) or (abs(sepres_dsep(i)) gt 3.0) ) then $
         print,format='(f10.4,f9.3,f10.3,f10.3,f8.3,f9.3,f9.3)', $
               time(i),dpa_scale(i)*180.0/!dpi,paOC(i),pares_dpa(i),dsep_scale(i),sepOC(i),sepres_dsep(i)
 
   endfor

   count_save = 0.0
   count_remove = 0.0
   count_lower = 0.0

   print,'----------------------------------------------'
   for i=0, num-1 do begin

      if ( (abs(pares_dpa(i)) gt 3.0) or (abs(sepres_dsep(i)) gt 3.0) ) then begin
         print,'Removing: ',time(i),pameas(i)*180.0/!dpi,dpa_orig(i)*180.0/!dpi,sepmeas(i),dsep_orig(i)
 
         time_remove(total_remove) = time(i)
         pameas_remove(total_remove) = pameas(i)
         dpa_remove(total_remove) = dpa_orig(i)
         sepmeas_remove(total_remove) = sepmeas(i)
         dsep_remove(total_remove) = dsep_orig(i)
         niter_remove(total_remove) = strcompress(string(niter),/remove_all)

         count_remove = count_remove + 1
         total_remove = total_remove + 1
      endif else begin
         ;printf,1,format='(f10.4,f9.3,f7.3,f10.5,f8.5)',time(i),pameas(i)*180.0/!dpi,dpa(i)*180.0/!dpi,sepmeas(i),dsep(i)

         time_save(count_save) = time(i)
         pameas_save(count_save) = pameas(i)
         dpa_save(count_save) = dpa_scale(i)
         sepmeas_save(count_save) = sepmeas(i)
         dsep_save(count_save) = dsep_scale(i)

         count_save = count_save + 1
      endelse
   endfor
   close,1

   print,'----------------------------------------------'
   print,'Original number of observations: ', num
   print,'Number of observations saved: ',count_save
   print,'Number of observations removed: ',count_remove
   print,'----------------------------------------------'

   num = count_save
   time = time_save(0:num-1)
   pameas = pameas_save(0:num-1)
   dpa = dpa_save(0:num-1)
   sepmeas = sepmeas_save(0:num-1)
   dsep = dsep_save(0:num-1)

endif else begin

   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
   ; default rejection method - lowering weights (no more removal)
   print,'----------------------------------------------'
   print,'Measurements where res/sigma > 3:'
   print,"date        dpa     pa(O-C)  paOC/dpa    dsep  sep(O-C)   sepOC/dsep"
   for i=0, num-1 do begin

      if ( (abs(pares_dpa(i)) gt 3.0) or (abs(sepres_dsep(i)) gt 3.0) ) then $
         print,format='(f10.4,f9.3,f10.3,f10.3,f8.3,f9.3,f9.3)', $
               time(i),dpa_scale(i)*180.0/!dpi,paOC(i),pares_dpa(i),dsep_scale(i),sepOC(i),sepres_dsep(i)
 
   endfor

   count_lower = 0.0
   count_save = 0.0
   count_remove = 0.0

   print,'----------------------------------------------'
   for i=0, num-1 do begin

      if ( (abs(pares_dpa(i)) gt 3.0) or (abs(sepres_dsep(i)) gt 3.0) ) then begin

         dsep_scale(i) = dsep_scale(i)*10.0
         dpa_scale(i) = dpa_scale(i)*10.0
         ;dpa_scale(i) = cos(pameas(i))^2*sepmeas(i)*dsep_scale(i)/xmeas(i)^2
         print,'Lowering weight: ',time(i),pameas(i)*180.0/!dpi,dpa_orig(i)*180.0/!dpi,sepmeas(i),dsep_orig(i)
         ;printf,1,format='(f10.4,f9.3,f7.3,f10.5,f8.5)',time(i),pameas(i)*180.0/!dpi,dpa(i)*180.0/!dpi,sepmeas(i),dsep(i)

         time_lower(total_lower) = time(i)
         pameas_lower(total_lower) = pameas(i)
         dpa_lower(total_lower) = dpa_orig(i)
         sepmeas_lower(total_lower) = sepmeas(i)
         dsep_lower(total_lower) = dsep_orig(i)
         niter_lower(total_lower) = strcompress(string(niter),/remove_all)

         count_save = count_save + 1
         count_lower = count_lower + 1
         total_lower = total_lower+1

         ;if (weight(i) gt ave_weight) then begin
         ;   dsep(i) = dsep(i)*16.0
         ;   dpa(i) = cos(pameas(i))^2*sepmeas(i)*dsep(i)/xmeas(i)^2
         ;   print,'Lowering weight: ',time(i),pameas(i)*180.0/!dpi,dpa(i)*180.0/!dpi,sepmeas(i),dsep(i)
         ;   printf,1,format='(f9.4,f9.3,f7.3,f10.5,f8.5)',time(i),pameas(i)*180.0/!dpi,dpa(i)*180.0/!dpi,sepmeas(i),dsep(i)
         ;   count_save = count_save + 1
         ;   count_lower = count_lower + 1
         ;endif else begin
         ;   print,'Removing: ',time(i),pameas(i)*180.0/!dpi,dpa(i)*180.0/!dpi,sepmeas(i),dsep(i)
         ;   count_remove = count_remove + 1
         ;endelse
      endif else begin
         ;printf,1,format='(f10.4,f9.3,f7.3,f10.5,f8.5)',time(i),pameas(i)*180.0/!dpi,dpa(i)*180.0/!dpi,sepmeas(i),dsep(i)
         count_save = count_save + 1  ; count all observations!
      endelse
   endfor
   close,1

   dpa = dpa_scale
   dsep = dsep_scale

   print,'----------------------------------------------'
   print,'Original number of observations: ', num
   print,'Number of observations saved: ',count_save
   print,'Number of weights lowered: ',count_lower
   print,'Number of observations removed: ',count_remove
   print,'----------------------------------------------'

endelse
endelse

ans_cont = ' '
print,'Iteration number:',niter
print,'Hit enter to continue'
read,ans_cont

; Increase iteration number
niter = niter + 1

endwhile
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

print,'----------------------------------------------'
print,'Summary of Final Results:'
print,format='(a45,i6)','Original number of measurements:             ',num_orig
print,format='(a45,i6)','Final number of measurements:                ',num
print,format='(a45,i6)','Number of measurements removed:              ',total_remove
print,format='(a45,i6)','Number of measurements with lowered weights: ',total_lower

print,'----------------------------------------------'
print,'Final Orbital Parameters:'
for i=0, 6 do print,elLabel(i),EL_deg(i),ELerr(i)
print,'chi2:', chi2
print,'reduced chi2:', chi2red
print,'----------------------------------------------'


; Print full set of removed observations

if (total_remove gt 0) then begin

   time_remove = time_remove(0:total_remove-1)
   pameas_remove = pameas_remove(0:total_remove-1)
   dpa_remove = dpa_remove(0:total_remove-1)
   sepmeas_remove = sepmeas_remove(0:total_remove-1)
   dsep_remove = dsep_remove(0:total_remove-1)
   niter_remove = niter_remove(0:total_remove-1)

   openw,1,'temp_remove_'+data_file
   for i=0, total_remove-1 do printf,1,format='(f10.4,f9.3,f7.3,f10.5,f9.5,a4)',time_remove(i),pameas_remove(i)*180.0/!dpi,dpa_remove(i)*180.0/!dpi,sepmeas_remove(i),dsep_remove(i),niter_remove(i)
   close,1

endif

; Print full set of observations with adjusted weights

if (total_lower gt 0) then begin

   time_lower = time_lower(0:total_lower-1)
   pameas_lower = pameas_lower(0:total_lower-1)
   dpa_lower = dpa_lower(0:total_lower-1)
   sepmeas_lower = sepmeas_lower(0:total_lower-1)
   dsep_lower = dsep_lower(0:total_lower-1)
   niter_lower = niter_lower(0:total_lower-1)

   openw,1,'temp_lower_'+data_file
   for i=0, total_lower-1 do printf,1,format='(f10.4,f9.3,f7.3,f10.5,f9.5,a4)',time_lower(i),pameas_lower(i)*180.0/!dpi,dpa_lower(i)*180.0/!dpi,sepmeas_lower(i),dsep_lower(i),niter_lower(i)
   close,1

endif

; Save non-rejected observations to a file
openw,1,'temp_save_'+data_file

for i=0, num-1 do printf,1,format='(f10.4,f9.3,f7.3,f10.5,f9.5)',time(i),pameas(i)*180.0/!dpi,dpa(i)*180.0/!dpi,sepmeas(i),dsep(i)
close,1



; Compute residuals for removed/adjusted measurements

if (ans_sigclip) or (ans_rem) then begin

   calc_vbfit, EL_rad, time_remove, pafit_remove, sepfit_remove

   diffPA_remove = (PAmeas_remove - PAfit_remove)
   for i=0, total_remove-1 do begin
      if (diffPA_remove(i) ge !dpi) then diffPA_remove(i) = diffPA_remove(i) - 2.0*!dpi
      if (diffPA_remove(i) le -!dpi) then diffPA_remove(i) = diffPA_remove(i) + 2.0*!dpi
   endfor

   ; Actually, these uncertainties won't be correct because
   ; we are scaling the scaled uncertainties in each iteration,
   ; we're not scaling the raw uncertainties.
   dsep_remove_scale = dsep_remove * scale
   dpa_remove_scale = dpa_remove * scale

   ; print O-C residuals in sep,PA
   sepOC_remove = sepmeas_remove - sepfit_remove
   paOC_remove = (PAmeas_remove - PAfit_remove)*180.0/!dpi
   for i=0, total_remove-1 do begin
      if (paOC_remove(i) gt 180.0) then paOC_remove(i) = paOC_remove(i)-360.0
      if (paOC_remove(i) lt -180.0) then paOC_remove(i) = paOC_remove(i)+360.0
   endfor

endif else begin

   calc_vbfit, EL_rad, time_lower, pafit_lower, sepfit_lower

   diffPA_lower = (PAmeas_lower - PAfit_lower)
   for i=0, total_lower-1 do begin
      if (diffPA_lower(i) ge !dpi) then diffPA_lower(i) = diffPA_lower(i) - 2.0*!dpi
      if (diffPA_lower(i) le -!dpi) then diffPA_lower(i) = diffPA_lower(i) + 2.0*!dpi
   endfor
 
   ; Actually, these uncertainties won't be correct because
   ; we are scaling the scaled uncertainties in each iteration,
   ; we're not scaling the raw uncertainties.
   dsep_lower_scale = dsep_lower * scale
   dpa_lower_scale = dpa_lower * scale

   ; print O-C residuals in sep,PA
   sepOC_lower = sepmeas_lower - sepfit_lower
   paOC_lower = (PAmeas_lower - PAfit_lower)*180.0/!dpi
   for i=0, total_lower-1 do begin
      if (paOC_lower(i) gt 180.0) then paOC_lower(i) = paOC_lower(i)-360.0
      if (paOC_lower(i) lt -180.0) then paOC_lower(i) = paOC_lower(i)+360.0
   endfor

endelse


; Make plot of residuals

seprange = 1.1*max(abs([sepOC+dsep,sepOC-dsep]))
;seprange=1.5

parange = 1.1*max(abs([paOC+dpa*180.0/!dpi,paOC-dpa*180.0/!dpi]))
;parange = 20.0

if time(0) gt 10000.0 then begin
   dt = 50000.0 
   datelabel = 'HJD - 50,000'
endif else begin
   dt = 0.0
   datelabel = 'Date'
endelse

trange = [min(time-dt),max(time-dt)]
tr = trange(1) - trange(0)
trange = [trange(0)-0.05*tr,trange(1)+0.05*tr]
;trange=[1995,2016]
;trange=[1995,2015]
;trange=[2000,2014]

;seplabel = 'Residuals in sep (arcsec)' 
seplabel = 'Residuals in sep (mas)'

Z = findgen(17) * (!pi*2/16.)
usersym, cos(Z), sin(Z), /fill

set_plot, 'ps'
;device, /Color, Bits_per_pixel=8, filename='temp.ps',xsize=15,ysize=13.5
device, /Color, Bits_per_pixel=8, /encap, filename='temp.eps',xsize=18,ysize=15

loadct,40  ; rainbow+black
; Colors:
; red = 250, orange=220, yellow=195, green=150, blue=60, purple=30, black=0
ctab = [250,220,195,150,60,30,0]

!P.Multi=[0,1,2]

ploterror, time-dt, paOC,dpa_scale*180.0/!dpi,psym=8,xtitle=datelabel,ytitle='Residuals in PA (deg)',yrange=[-parange,parange],xrange=trange,xstyle=1,ystyle=1 
oplot,trange,[0,0],linestyle=2

if (ans_sigclip) or (ans_rem) then oplot,time_remove-dt,paOC_remove,psym=8,color=ctab(0) $
else oplot,time_lower-dt,paOC_lower,psym=8,color=ctab(0)


ploterror, time-dt, sepOC,dsep_scale,psym=8,xtitle=datelabel,ytitle=seplabel,yrange=[-seprange,seprange],xrange=trange,xstyle=1,ystyle=1
oplot,trange,[0,0],linestyle=2

if (ans_sigclip) or (ans_rem) then oplot,time_remove-dt,sepOC_remove,psym=8,color=ctab(0) $
else oplot,time_lower-dt,sepOC_lower,psym=8,color=ctab(0)

device, /close
set_plot, 'x'

!P.Multi=0

loadct=0

;stop,'Type .cont to continue'

end

