pro remove_outliers_velocity_sb1, data_file, num, sigclip=sigclip, removal=removal, sigrej=sigrej

; Compute the spectroscopic orbital elements through a Newton-Raphson 
; technique.  Works for SB1!!!
;
; The program requires an initial guess that can be calculated from using
; the points from the fitting curves.
;
; Calls the following routines:
;     calc_deriv_SB1.pro
;     calc_Ei.pro
;     solve_trans.pro
;     calc_sb1fit.pro - for plotting model orbits
;
; Equations:  
; Initial estimates (P0,T0,e0,K1_0,omega0,Vsys0) and  
; a set of observations (ti,V1i) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Projected semi-major axis (a1sini):
;       a1sini = (P/2pi)*K1*(1 - e^2)^1/2
; The fitted velocities (V1_i) are determined from the orbital elements:
;       V1 = K1*[e*cos(omega) + cos(nu + omega)] 
;
; Procedure:
; Minimize chi squared between measured velocities and fitted velocities:
;	chi^2 = sum[(V1data - V1fit)^2/sigmaV1^2]
; where the sums run over all data points
; Take partial derivatives of chi^2 with respect to the orbital elements,
; (P,T,e,K1,omega,Vsys) and set to zero. Solve for the value of
; the orbital elements that minimize chi^2.
;
; Since the equations for V1 are non-linear, cannot solve for 
; the orbital elements that minimize chi^2 analytically.
;
; In turn, use a Newton-Raphson technique to converge upon the solution.
; Replace (V1fit,V2fit) in the chi^2 equation with the Taylor series 
; approximation,
; 	x = x| + dx/dP|(P-P0) + dx/dT|(T-T0) + dx/de|(e-e0) + dx/dw|(w-w0)
;	       + dx/d(K1)|(K1 - K1|0) + dx/d(Vsys)|(Vsys - Vsys0)
; where x=(V1) and the | represents the value evaluated for the
; initial estimates of the orbital elements. Now, V1fit and V2fit 
; are linear in (P,T,e,K1,w,Vsys), so the partial derivatives
; can easily be taken and solved by setting up a matrix and using 
; Cramer's method.
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;       K1:      velocity amplitude of primary (in km/s) (K1)
;	omega:	 angle between node and periastron (w)
;	ti:	 time of observation
;	V1_d:	 velocity of primary - measured data
;	V1_f:	 V1 - fit (determined from orbital elements for time of obs.)
;       Vsys:    systemic velocity
;
; INPUT:
;	data_file: file containing data points in the format:
;		      time  V1  errV1
;		      where V1 is the radial velocity of the primary
;                     and errV1 is the measurement error
;		      (enter file name using quotes)
;	num: number of data points contained in data_file
;
; Prompted inputs:
;	initial estimates for all the orbital elements: 
;             P, T, e, K1, omega, Vsys
;       vector identifying which orbital elements to solve for
;             enter 0 to hold element fixed, 1 to vary)
;             ex.   1 1 1 1 1 1 -> solves for all
;                   0 0 0 1 0 0 -> solves only for a1 of primary
; Notes: make sure P,T are given in same units
;        make sure K1,V1 are given in same units
;
; OUTPUT:
;	best fit values of orbital elements: P,T,e,a1sini,omega,Vsys
;       and corresponding errors
;
; Notes on units:
;    P, T, and time should be in same units (days or years)
;    K1 and V1 should be in same units (km/s)
;
;
; Method:
; - Scale uncertainties so to force redchi2 = 1
; - Next is to check measurements scaled uncertainties for outlier rejection
;
; Default rejection method (lower weights or removal):
; - If res > 3sig (res/sig > 3) use the follow approach
;   + If measurement weight > mean(all_weights) then lower weight by factor of 4
;     (lowering weights by a factor of 4 is the same as increasing 
;     uncertainties by a factor of 16!)
;   + If measurement weight < mean(all_weights) then remove the measurements 
;     from the data file
; - Approach modified so that no observations are removed - only the
;   weights are lowered by a factor of sqrt(10), corresponding to uncertainties
;   increased by a factor of 10.  [31 May 2017]
;
; Alternate rejection method (removal):
; - Set /removal flag
; - If res > 3sig (res/sig > 3), then remove the measurement from the data file
; 
; Alternate rejection method (sigma clipping):
; - Set /sigclip flag
; - Reject measurements if res(i) > 3*stdev(res)
;
; Began 23 June 2005
;
; Version 1: Orignal copy
; Version 2: 
;    - Include Marquadt Method of adjusting diagonal elements of
;      the alpha matrix to ensure convergence. 5 July 2005
;    - Fixed non-convergence problems   7 July 2005
;        + do not remove P wrap around effects from dV/dP
;          leave (t-T)/P in full form; net effect is that adjustments
;          smaller
;        + fixed primary/secondary indices in setting dV1/dVsys and dV2/dVsys
;    - Designed for SB1's only!!!
;
; 31 May 2006: Save unmodified program as version 2
; 31 May 2006: Fit SB2 parameters in terms of K1
;              (as opposed to a1sini)
; 03 Jun 2021: Add in outlier rejection.

close,1

!P.font=0
!P.multi=0
!P.charsize=1.0
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.2	;scale factor for adjusting size of plotting symbols

loadct,0

; Determine which optional arguments and flags are set

; Do you want to use sigma clipping?
; Reject measurements if res(i) > 3*stdev(res)
; keyword_set returns True (1) if sigclip is defined, False (0) if undefined

ans_sigclip = keyword_set(sigclip)

; Do you want to simply remove data that exceeds 3 sigma?
; Reject measurements if res(i) > 3*sigma(i)
; keyword_set returns True (1) if removal is defined, False (0) if undefined

ans_rem = keyword_set(removal)

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file

time = dblarr(num)
V1 = dblarr(num)
dV1 = dblarr(num)

openr,lun,data_file,/get_lun

for i=0, num-1 do begin

	readf, lun, temp1, temp2, temp3

	time(i) = temp1
	V1(i) = temp2
	dV1(i) = temp3
	
endfor

close,lun

; Obtain values for P,T,e,a1sini,omega,Vsys

period = 0d
Tperi = 0d
ecc = 0d
K1 = 0d
omega = 0d
Vsys = 0d

print,"Enter P,T,e,K1(km/s),omega,Vsys:"
read,period,Tperi,ecc,K1,omega,Vsys

print, "Vary each orbital element?"
print, "For each element, enter 0 to hold element fixed, 1 to vary:"
print, "[P,T,e,K1,omega,Vsys]"
read, f0,f1,f2,f3,f4,f5

elfix=[f0,f1,f2,f3,f4,f5]
nEl = n_elements(elfix)

k=0
for i=0, nEl-1 do begin

	if (elfix(i) eq 0) then k=k+1
endfor

mfit = nEl - k 		; number of elements to improve

print,"Initial P,T,e,K1,omega,Vsys:"
print,period,Tperi,ecc,K1,omega,Vsys

EL_deg = [period,Tperi,ecc,K1,omega,Vsys]

elLabel = ['P','T','e','K1','omega','Vsys']


; Arrays to hold rejected observations
time_remove = dblarr(num)
V1_remove = dblarr(num)
dV1_remove = dblarr(num)
niter_remove = strarr(num)
	
; Arrays to hold observations with lowered weights
time_lower = dblarr(num)
V1_lower = dblarr(num)
dV1_lower = dblarr(num)
niter_lower = strarr(num)

; Temporary arrays to hold saved observations during each iteration
time_save = dblarr(num)
V1_save = dblarr(num)
dV1_save = dblarr(num)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; START big while loop here ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Continue rejection algorithm until no observations are removed
; or no uncertainties are adjusted.

; Total number of lowered or removed observations
total_remove = 0.0
total_lower = 0.0

; Number of observations rejected or adjusted during this iteration.
; Set to 1 to initiate while loop
count_remove = 1
count_lower = 1

niter = 0

num_orig = num

dV1_orig = dV1

while (count_remove gt 0) or (count_lower gt 0) do begin

; Initialize arrays for scaled error bars:

dV1_scale = dV1

; Compute initial orbit fit:


auto_fit_velocity_sb1, time, V1, dV1_scale, EL_deg, elfix, ELerr=ELerr, /showplot


; Define EL_rad with i,W,w in radians

EL_rad = EL_deg
EL_rad(4)=EL_rad(4)*!dpi/180.0

; Compute residuals

calc_sb1fit, EL_rad, time, V1fit

; Compute chi2

chi2 = total((V1 - V1fit)^2/dV1^2)

; degrees of freedom
dof = num - mfit

; reduced chi squared:
chi2red = chi2/dof

scale = sqrt(chi2red)

; Scale uncertainties to force chi2red = 1
; The scale uncertainties are used to compute number of sigma
; discrepancy for the residuals
dV1_scale = dV1 * scale


print,'----------------------------------------------'
print,'Original chi2:', chi2
print,'Original reduced chi2:', chi2red

print,'Scale uncertainties by a factor of: ',scale

; Recompute chi2 using new uncertainties
chi2 = total((V1 - V1fit)^2/dV1_scale^2)
; reduced chi squared:
chi2red = chi2/dof


; O-C residuals (Observed - Calculated)

V1OC = V1 - V1fit
V1res_dV1 = V1OC/dV1_scale

; Flag measurements where res/sigma > 3
; If original weight exceeds the average mean value then decrease
; weight by a factor of 4.  Otherwise remove from data file.

weight = 1.0/sqrt(dV1)
ave_weight = mean(weight)

print,'Mean weight: ',mean(weight)
print,'Median weight: ',median(weight)

stdev_V1res = stdev(V1OC)

print,'Residuals:'
print,'Mean and stdev in V1:  ',mean(V1OC),stdev(V1OC)

; Set rejection level for sigma clipping:
; default sigrej = 3.0, otherwise use value defined in call

if (keyword_set(sigrej) eq 0) then sigrej = 3.0

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; sigma clipping rejection method
if (ans_sigclip) then begin

   print,'----------------------------------------------'
   print,'Measurements where res > 3*stdev_res:'
   print,"date        dV1     V1(O-C)  V1OC/dV1"
   for i=0, num-1 do begin

      if (abs(V1OC(i)) gt sigrej*stdev_V1res) then $
         print,format='(f10.4,f10.4,f10.4,f10.4)', $
               time(i),dV1_scale(i),V1OC(i),V1res_dV1(i)
 
   endfor

   count_save = 0.0
   count_remove = 0.0
   count_lower = 0.0

   print,'----------------------------------------------'
   for i=0, num-1 do begin
      
      if (abs(V1OC(i)) gt sigrej*stdev_V1res) then begin

         print,'Removing: ',time(i),V1(i),dV1_orig(i)

         time_remove(total_remove) = time(i)
         V1_remove(total_remove) = V1(i)
         dV1_remove(total_remove) = dV1_orig(i)
         niter_remove(total_remove) = strcompress(string(niter),/remove_all)

         count_remove = count_remove + 1
         total_remove = total_remove + 1

      endif else begin

         time_save(count_save) = time(i)
         V1_save(count_save) = V1(i)
         dV1_save(count_save) = dV1_scale(i)

         count_save = count_save + 1
      endelse
   endfor
   close,1

   print,'----------------------------------------------'
   print,'Original number of observations: ', num
   print,'Number of observations saved: ',count_save
   print,'Number of observations removed: ',count_remove
   print,'----------------------------------------------'

   num = count_save
   time = time_save(0:num-1)
   V1 = V1_save(0:num-1)
   dV1 = dV1_save(0:num-1)

endif else begin

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Remove data if resisual > 3sigma 
if (ans_rem) then begin

   print,'----------------------------------------------'
   print,'Measurements where res/sigma > 3:'
   print,"date         dV1      V1(O-C)  V1OC/dV1"
   for i=0, num-1 do begin

      if (abs(V1res_dV1(i)) gt sigrej) then $
         print,format='(f10.4,f10.4,f10.4,f10.4)', $
               time(i),dV1_scale(i),V1OC(i),V1res_dV1(i)
 
   endfor

   count_save = 0.0
   count_remove = 0.0
   count_lower = 0.0

   print,'----------------------------------------------'
   for i=0, num-1 do begin

      if (abs(V1res_dV1(i)) gt sigrej) then begin
         print,'Removing: ',time(i),V1(i),dV1_orig(i)
 
         time_remove(total_remove) = time(i)
         V1_remove(total_remove) = V1(i)
         dV1_remove(total_remove) = dV1_orig(i)
         niter_remove(total_remove) = strcompress(string(niter),/remove_all)

         count_remove = count_remove + 1
         total_remove = total_remove + 1
      endif else begin

         time_save(count_save) = time(i)
         V1_save(count_save) = V1(i)
         dV1_save(count_save) = dV1_scale(i)

         count_save = count_save + 1
      endelse
   endfor
   close,1

   print,'----------------------------------------------'
   print,'Original number of observations: ', num
   print,'Number of observations saved: ',count_save
   print,'Number of observations removed: ',count_remove
   print,'----------------------------------------------'

   num = count_save
   time = time_save(0:num-1)
   V1 = V1_save(0:num-1)
   dV1 = dV1_save(0:num-1)

endif else begin

   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
   ; default rejection method - lowering weights (no more removal)
   print,'----------------------------------------------'
   print,'Measurements where res/sigma > 3:'
   print,"date         dV1     v1(O-C)  V1OC/dV1"
   for i=0, num-1 do begin

      if (abs(V1res_dV1(i)) gt sigrej) then $
         print,format='(f10.4,f10.4,f10.4,f10.4)', $
               time(i),dV1_scale(i),V1OC(i),V1res_dV1(i)
 
   endfor

   count_lower = 0.0
   count_save = 0.0
   count_remove = 0.0

   print,'----------------------------------------------'
   for i=0, num-1 do begin

      if (abs(V1res_dV1(i)) gt sigrej) then begin

         dV1_scale(i) = dV1_scale(i)*10.0
         print,'Lowering weight: ',time(i),V1(i),dV1_orig(i)

         time_lower(total_lower) = time(i)
         V1_lower(total_lower) = V1(i)
         dV1_lower(total_lower) = dV1_orig(i)
         niter_lower(total_lower) = strcompress(string(niter),/remove_all)

         count_save = count_save + 1
         count_lower = count_lower + 1
         total_lower = total_lower+1

      endif else begin
         count_save = count_save + 1  ; count all observations!
      endelse
   endfor
   close,1

   dV1 = dV1_scale

   print,'----------------------------------------------'
   print,'Original number of observations: ', num
   print,'Number of observations saved: ',count_save
   print,'Number of weights lowered: ',count_lower
   print,'Number of observations removed: ',count_remove
   print,'----------------------------------------------'

endelse
endelse

ans_cont = ' '
print,'Iteration number:',niter
print,'Hit enter to continue'
read,ans_cont

; Increase iteration number
niter = niter + 1

endwhile
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

print,'----------------------------------------------'
print,'Summary of Final Results:'
print,format='(a45,i6)','Original number of measurements:             ',num_orig
print,format='(a45,i6)','Final number of measurements:                ',num
print,format='(a45,i6)','Number of measurements removed:              ',total_remove
print,format='(a45,i6)','Number of measurements with lowered weights: ',total_lower

print,'----------------------------------------------'
print,'Final Orbital Parameters:'
for i=0, 5 do print,elLabel(i),EL_deg(i),ELerr(i)
print,'chi2:', chi2
print,'reduced chi2:', chi2red
print,'----------------------------------------------'


; Print full set of removed observations

if (total_remove gt 0) then begin

   time_remove = time_remove(0:total_remove-1)
   V1_remove = V1_remove(0:total_remove-1)
   dV1_remove = dV1_remove(0:total_remove-1)
   niter_remove = niter_remove(0:total_remove-1)

   openw,1,'temp_remove_'+data_file
   for i=0, total_remove-1 do printf,1,format='(f10.4,f12.4,f12.4,a4)',time_remove(i),V1_remove(i),dV1_remove(i),niter_remove(i)
   close,1

endif

; Print full set of observations with adjusted weights

if (total_lower gt 0) then begin

   time_lower = time_lower(0:total_lower-1)
   V1_lower = V1_lower(0:total_lower-1)
   dV1_lower = dV1_lower(0:total_lower-1)
   niter_lower = niter_lower(0:total_lower-1)

   openw,1,'temp_lower_'+data_file
   for i=0, total_lower-1 do printf,1,format='(f10.4,f12.4,f12.4,a4)',time_lower(i),V1_lower(i),dV1_lower(i),niter_lower(i)
   close,1

endif

; Save non-rejected observations to a file
openw,1,'temp_save_'+data_file

for i=0, num-1 do printf,1,format='(f10.4,f9.3,f7.3,f10.5,f9.5)',time(i),V1(i),dV1(i)
close,1



; Compute residuals for removed/adjusted measurements

if (ans_sigclip) or (ans_rem) then begin

   calc_sb1fit, EL_rad, time_remove, V1fit_remove

   ; Actually, these uncertainties won't be correct because
   ; we are scaling the scaled uncertainties in each iteration,
   ; we're not scaling the raw uncertainties.
   dV1_remove_scale = dV1_remove * scale

   ; print O-C residuals in sep,PA
   V1OC_remove = V1_remove - V1fit_remove

endif else begin

   calc_sb1fit, EL_rad, time_lower, V1fit_lower

   ; Actually, these uncertainties won't be correct because
   ; we are scaling the scaled uncertainties in each iteration,
   ; we're not scaling the raw uncertainties.
   dV1_lower_scale = dV1_lower * scale

   ; print O-C residuals in sep,PA
   V1OC_lower = V1_lower - V1fit_lower

endelse

; Make plot of residuals

V1range = 1.1*max(abs([V1OC+dV1,V1OC-dV1]))

if time(0) gt 10000.0 then begin
   dt = 50000.0 
   datelabel = 'HJD - 50,000'
endif else begin
   dt = 0.0
   datelabel = 'Date'
endelse

trange = [min(time-dt),max(time-dt)]
tr = trange(1) - trange(0)
trange = [trange(0)-0.05*tr,trange(1)+0.05*tr]

Z = findgen(17) * (!pi*2/16.)
usersym, cos(Z), sin(Z), /fill

set_plot, 'ps'
;device, /Color, Bits_per_pixel=8, filename='temp.ps',xsize=15,ysize=13.5
device, /Color, Bits_per_pixel=8, /encap, filename='temp.eps',xsize=18,ysize=15

loadct,40  ; rainbow+black
; Colors:
; red = 250, orange=220, yellow=195, green=150, blue=60, purple=30, black=0
ctab = [250,220,195,150,60,30,0]

!P.Multi=0

ploterror, time-dt, V1OC,dV1_scale,psym=8,xtitle=datelabel,ytitle='Residuals in K1 (km/s)',yrange=[-V1range,V1range],xrange=trange,xstyle=1,ystyle=1 
oplot,trange,[0,0],linestyle=2

if (ans_sigclip) or (ans_rem) then oplot,time_remove-dt,V1OC_remove,psym=8,color=ctab(0) $
else oplot,time_lower-dt,V1OC_lower,psym=8,color=ctab(0)


device, /close
set_plot, 'x'

!P.Multi=0

loadct=0


end
