pro search_chi2_vblunar_aiW, vb_file, vb_num, lun_file, lun_num

; Modified to include lunar occultation projection in orbit fit
;
; Compute the orbital elements through a Newton-Raphson technique.
;
; This program should work directly with the data points (rho,theta,t), not 
; the fitting points. 
;
; The program requires an initial guess that can be calculated from using
; the points from the fitting curves.
;
; Equations:  
; Initial estimates (P0,T0,e0,A0,B0,F0,G0) and a set of observations 
; (ti,rhoi,thetai) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Radius vector:
;	r = a(1-e^2)/(1 + ecos(nu))
; The fitted (rho_i,theta_i) positions are determined from the orbital elements
; 	tan(theta - Omega) = tan(nu + omega)*cosi
;	rho = r*cos(nu + omega)*sec(theta - Omega)
;
; Procedure:
; Minimize chi squared between data point positions and fitted positions:
;	chi^2 = sum[(rhodata - rhoi)^2/sigmarho^2 
;		    + (thetadata - thetai)^2/sigmatheta^2]
; Take partial derivatives of chi^2 with respect to the orbital elements,
; (P,T,e,a,i,Omega,omega) and set to zero. Solve for the value of the orbital 
; elements that minimize chi^2.
;
; Since the partial derivatives are non-linear, cannot solve for the orbital
; elements that minimize chi^2 analytically.
;
; In turn, use a Newton-Raphson technique to converge upon the solution.
; Replace (rhoi,thetai) in the chi^2 equation with the Taylor series 
; approximation,
; 	x = x| + dx/dP|(P-P0) + dx/dT|(T-T0) + dx/de|(e-e0) 
;	       + dx/da|(a-Aa) + dx/di|(i-i0) + dx/dW|(W-W0) + dx/dw|(w-w0)
; where x=(rho,theta) and the | represents the value evaluated at for the
; initial estimates for the orbital elements. Now, rhoi and thetai are linear 
; in (P,T,e,a,i,W,w), so the partial derivatives can easily be taken and solved
; by setting up a matrix and using Cramer's method.
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;	major:	 semi-major axis (in mas) (a)
;	inc:	 inclination (i)
;	Omega:	 position angle of node (W)
;	omega:	 angle between node and periastron (w)
;	ti:	 time of observation
;	rho_d:	 separation - data
;	theta_d: postion angle - data
;	rho_f:	 sep - fit (determined from orbital elements for time of obs.)
;	theta_f: PA - fit (determined from orbital elements for time of obs.)
;
; INPUT:
;	vb_file: file containing data points in the format:
;		      time  theta  etheta  rho  erho
;		      where theta is the position angle, rho is the separation
;                     and etheta, erho are the measurement uncertainties
;		      (enter file name using quotes)
;	num: number of data points contained in vb_file
;	
; Prompted inputs:
;	initial estimates for all the orbital elements: P,T,e,a,i,Omega,omega
;
; Calls the following routines:
;     calc_deriv_vb.pro
;     calc_deriv_lunar
;     calc_Ei.pro
;     solve_trans.pro
;     calc_vbfit.pro - for plotting model orbits!
;
; Began 14 May 2002
; Modified 11 Nov 2003: 
;       This is the current version of NEWT_RAPH.PRO
;	(v1.1 and v1.2 are older versions)
; 	- restructured program into separate subroutines
; 	- add option of holding elements fixed during iterations
; 13 January 2005:
;       Fix bugs
;       - if solving for only one element, don't use cramer.pro to solve eqn's
;       - fix 360 degree roll overs in (PAdata - PAfit)
;       Save version v4 as backup
; 11 July 2005:
;       - Include Marquadt Method of adjusting diagonal elements of
;         the alpha matrix to ensure convergence.
;       - Fixed non-convergence problems 
;           + do not remove P wrap around effects from dx/dP
;             leave (t-T)/P in full form; net effect is that adjustments
;             are smaller
;       Save v5 as backup 
; 13 July 2005:
;       - Change format of plotting routines to include a call to
;         calc_vbfit
;       Save v6 as backup
; 3 May 2006:
;       - Modified to include lunar occultation projection in orbit fit

; Read in data points from data_file

; Elias 12 lunar occultation measurement:
;t_lun = [1992.1913]
;PA_lun = [(95.0)*!dpi/180]
;PA_lun = [(180.0+95.0)*!dpi/180]
;sep_lun = [23.0]
;dsep_lun = [2.3]
;lun_num = 1.0

;; ZZ Tau lunar occultation measurement:
;t_lun = [1991.5918]
;PA_lun = [(244.0)*!dpi/180]
;sep_lun = [29.0]
;dsep_lun = [2.9]
;lun_num = 1.0

!P.font=0
!P.multi=0
!P.charsize=1.2
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.2	;scale factor for adjusting size of plotting symbols

; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1


temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time_vb = dblarr(vb_num)
theta = dblarr(vb_num)
rho = dblarr(vb_num)
dtheta = dblarr(vb_num)
drho = dblarr(vb_num)

openr,lun,vb_file,/get_lun

for i=0, vb_num-1 do begin

	readf, lun, temp1, temp2, temp3, temp4, temp5

	time_vb(i) = temp1
	theta(i) = temp2
	dtheta(i) = temp3
	rho(i) = temp4	
	drho(i) = temp5
	
endfor

close,lun

; convert theta and dtheta to radians

theta = theta*!dpi/180
dtheta = dtheta*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(theta)^2*drho^2 + rho^2*sin(theta)^2*dtheta^2)
dy = sqrt(sin(theta)^2*drho^2 + rho^2*cos(theta)^2*dtheta^2)

; Read in lunar occultation measurements:

time_lun = dblarr(lun_num)
PA_lun = dblarr(lun_num)
sep_lun = dblarr(lun_num)
dsep_lun = dblarr(lun_num)

openr,lun,lun_file

for i=0, lun_num-1 do begin

   
   readf, lun, temp1, temp2, temp3, temp4

   time_lun(i) = temp1
   PA_lun(i) = temp2
   sep_lun(i) = temp3
   dsep_lun(i) = temp4

endfor

close,lun

; convert PA to radians

PA_lun = PA_lun*!dpi/180

; Obtain values for P,T,e,a,i,Omega,omega 

period = 0d
Tperi = 0d
ecc = 0d
major = 0d
inc = 0d
W_cap = 0d
w_low = 0d

print,"Enter P,T,e,a,i,Omega,omega:"
read,period,Tperi,ecc,major,inc,W_cap,w_low

print,"Initial P,T,e,a,i,Omega,omega:"
print,period,Tperi,ecc,major,inc,W_cap,w_low

amin = 0.0d
amax = 0.0d

imin = 0.0d
imax = 0.0d

Wmin = 0.0d
Wmax = 0.0d

print,'Enter min max search range for a:'
read,amin,amax

print,'Enter min max search range for i:'
read,imin,imax

print,'Enter min max search range for Omega:'
read,Wmin,Wmax

;convert i, Omega, omega to radians
inc = inc*!dpi/180.0
W_cap = W_cap*!dpi/180.0
w_low = w_low*!dpi/180.0

imin = imin*!dpi/180
imax = imax*!dpi/180
Wmin = Wmin*!dpi/180
Wmax = Wmax*!dpi/180

arange = amax - amin
irange = imax - imin
Wrange = Wmax - Wmin

EL = [period,Tperi,ecc,major,inc,W_cap,w_low]
elLabel = ['P','T','e','a','i','Omega','omega']

nEL = n_elements(EL)

; Compute initial model

calc_vbfit, EL, time_vb, theta_f, rho_f

calc_vbfit, EL, time_lun, theta_lun_f, rho_lun_f

sep_lun_f =  rho_lun_f*cos(theta_lun_f - PA_lun)

; Determine chi squared

; convert data points to x and y coordinates

xfit = rho_f * cos(theta_f)	; x coordinate
yfit = rho_f * sin(theta_f)	; y coordinate
			
diff_theta = theta - theta_f

; account for any wrap-arounds from 360 deg to 0 deg

for k=0, vb_num-1 do begin
   if (diff_theta(k) ge !dpi) then $
      diff_theta(k) = diff_theta(k) - 2*!dpi
   if (diff_theta(k) le -!dpi) then $
      diff_theta(k) = 2*!dpi + diff_theta(k)	
endfor

chi2 = total((diff_theta)^2/dtheta^2 + (rho - rho_f)^2/drho^2) $
     + total((sep_lun - sep_lun_f)^2/dsep_lun^2)

chi2_best = chi2


; degrees of freedom
dof = 2d*vb_num + lun_num - 3d

; reduced chi squared:
chi2red = chi2/dof

print,'Unscaled chi2:',chi2
ans_scale = 'y'
print,'Do you want to scale chi2 so that chi2red = 1.0 (y/n)?'
read,ans_scale
if (ans_scale eq 'y') then scale = 1.0/chi2red else scale = 1.0

chi2 = chi2*scale
print,'Scaled chi2:',chi2
print,'Scale factor for chi2:',scale

chi2_best = chi2
bestpar = EL

EL_original = EL


; begin modifying a,i,W

total_num = 0
chidiff = 0.0


print, 'Enter the variation of chi2 you wish to examine:'
read, chidiff
print, 'Enter number of solutions you wish to find within this variation of chi2:'
read,total_num

seed=1001L
print,'Enter seed for random generator (ex. 1001L):'
read,seed


count = 0

while (count lt total_num) do begin

   major = amin + arange*randomu(seed,/double,/uniform)
   inc = imin + irange*randomu(seed,/double,/uniform)
   W_cap = Wmin + Wrange*randomu(seed,/double,/uniform)

   EL = EL_original
   EL(3) = major
   EL(4) = inc
   EL(5) = W_cap

   ; Compute initial model

   calc_vbfit, EL, time_vb, theta_f, rho_f
   
   calc_vbfit, EL, time_lun, theta_lun_f, rho_lun_f

   sep_lun_f =  rho_lun_f*cos(theta_lun_f - PA_lun)

   ; Determine chi squared

   ; convert data points to x and y coordinates

   xfit = rho_f * cos(theta_f)	; x coordinate
   yfit = rho_f * sin(theta_f)	; y coordinate
			
   diff_theta = theta - theta_f

   ; account for any wrap-arounds from 360 deg to 0 deg

   for k=0, vb_num-1 do begin
      if (diff_theta(k) ge !dpi) then $
         diff_theta(k) = diff_theta(k) - 2*!dpi
      if (diff_theta(k) le -!dpi) then $
         diff_theta(k) = 2*!dpi + diff_theta(k)	
   endfor

   chi2 = total((diff_theta)^2/dtheta^2 + (rho - rho_f)^2/drho^2) $
        + total((sep_lun - sep_lun_f)^2/dsep_lun^2)

   chi2 = chi2*scale

   
   delta_chi  = abs(chi2 - chi2_best)

   if (delta_chi lt chidiff) then begin
      print,count,major,inc*180.0/!dpi,W_cap*180.0/!dpi,chi2
      count = count + 1
   endif
      
   if (chi2 lt chi2_best) then begin
      chi2_best = chi2
      bestpar = EL
   endif

endwhile

; Compute best fit

EL = bestpar

calc_vbfit, EL, time_vb, theta_f, rho_f

calc_vbfit, EL, time_lun, theta_lun_f, rho_lun_f

sep_lun_f =  rho_lun_f*cos(theta_lun_f - PA_lun)

; Determine chi squared

; convert data points to x and y coordinates

xfit = rho_f * cos(theta_f)	; x coordinate
yfit = rho_f * sin(theta_f)	; y coordinate
			
diff_theta = theta - theta_f

; account for any wrap-arounds from 360 deg to 0 deg

for k=0, vb_num-1 do begin
   if (diff_theta(k) ge !dpi) then $
      diff_theta(k) = diff_theta(k) - 2*!dpi
   if (diff_theta(k) le -!dpi) then $
      diff_theta(k) = 2*!dpi + diff_theta(k)	
endfor

chi2 = total((diff_theta)^2/dtheta^2 + (rho - rho_f)^2/drho^2) $
     + total((sep_lun - sep_lun_f)^2/dsep_lun^2)

chi2 = chi2*scale

chi2_best = chi2

; reduced chi squared:
chi2red = chi2/dof


;ELerr = dblarr(mfit)
;for i=0,mfit-1 do ELerr(i) = sqrt(chi2red)*sqrt(invmat(i,i))

elfix = fltarr(nEL)

k=0
print,"Orbital Element     Error"
for i=0, 3 do begin
    if (elfix(i) eq 0) then begin
        print, elLabel(i), ': ', El(i)
    endif else begin
        print, elLabel(i), ': ', El(i), ELerr(k)
        k=k+1
    endelse
endfor
for i=4, nEl-1 do begin
     if (elfix(i) eq 0) then begin
         print, elLabel(i), ': ', El(i)*180/!dpi
     endif else begin
         print, elLabel(i), ': ', El(i)*180/!dpi, ELerr(k)*180/!dpi
         k = k+1
     endelse
endfor
print, "chi2(PA,sep):", chi2_best
print, "chi2red:", chi2red

print,'Number of iterations:',count
;print,'Final chi2:',chi2

print,EL

; plot best-fit orbit and data

tnum=1000.0
tstep = period/tnum
tmin = Tperi

tarr = findgen(tnum)*tstep + tmin

calc_vbfit, EL, tarr, theta_mod, rho_mod

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

; plot data and best-fit positions
plot,-yarr,xarr,psym=6
oplot,-yfit,xfit,psym=7
for l=0, vb_num-1 do oplot,[-yarr(l),-yfit(l)],[xarr(l),xfit(l)]

; plot best-fit orbit
oplot,-ymod,xmod

free_lun,lun

print,'Lunar occultation separation (meas/calc):'
print,sep_lun,sep_lun_f


; Define user symbol to be closed circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)

; Plot measured positions as filled circles

usersym, frac*cos(Z), frac*sin(Z),/fill

sepmax = 1.2*max(abs([yarr,xarr]))

set_plot, 'ps'
device,/Color,Bits_per_pixel=8,filename='temp.eps',xsize=12,ysize=12,isolatin1=1,/encap

    ploterror,yarr,xarr,dy,dx,xtitle='!9D!3RA (mas)',ytitle='!9D!3DEC (mas)',ISOTROPIC=1, $
	 xrange=[sepmax,-sepmax], yrange=[-sepmax,sepmax], $
         xstyle=1,ystyle=1, /nodata, position=[0.19,0.12,0.95,0.98]

    ; Connected measured and fit values
    for i=0,vb_num-1 do oplot,[yfit(i),yarr(i)],[xfit(i),xarr(i)]

    ; Mark computed location with x
    oplot,[yfit],[xfit],psym=7

                                ; mark primary with an asterisk
    oplot,[0],[0],psym=2
    oplot,[-sepmax,sepmax],[0,0],linestyle=1
    oplot,[0,0],[-sepmax,sepmax],linestyle=1

    ;Define the symbol to be a closed circle
    usersym, frac*cos(Z), frac*sin(Z), /fill
    oplot,yarr,xarr,psym=8

; plot best-fit orbit
oplot,ymod,xmod,color=3



; plot lunar occultation projection and projected separation of binary

calc_vbfit, EL, time_lun, theta_fit_lun, rho_fit_lun

xfit_lun = rho_fit_lun * cos(theta_fit_lun)	; x coordinate
yfit_lun = rho_fit_lun * sin(theta_fit_lun)	; y coordinate

ra_lun = 150.0*sin(PA_lun)
dec_lun = 150.0*cos(PA_lun)

;oplot,[ra_lun,-ra_lun],[dec_lun,-dec_lun]
;print,ra_lun,dec_lun

ra_lun_meas = sep_lun*sin(PA_lun)
dec_lun_meas = sep_lun*cos(PA_lun)

if (yfit_lun le 0) then oplot,[0,ra_lun_meas],[0,dec_lun_meas],color=1,thick=5
if (yfit_lun ge 0) then oplot,[0,-ra_lun_meas],[0,-dec_lun_meas],color=1,thick=5


; find equation of line that passes through projected separation
; (perpendicular to lunar occultation PA)
; b = y - mx

m_slope = -sin(PA_lun)/cos(PA_lun)
b_int = sep_lun*cos(PA_lun) - m_slope*sep_lun*sin(PA_lun)

oplot,[150,-150],[150*m_slope+b_int,-150*m_slope+b_int],LineStyle=2

; and equivalent line at minus projected sep
b_int = -sep_lun*cos(PA_lun) + m_slope*sep_lun*sin(PA_lun)

;oplot,[100,-100],[100*m_slope+b_int,-100*m_slope+b_int],LineStyle=2

; Filled Squares for lunar occultation prediction
; Define user symbol to be open rectangle
xrect = frac*[-1.0,-1.0,1.0,1.0,-1.0]
yrect = frac*[-1.0,1.0,1.0,-1.0,-1.0]
usersym,xrect,yrect,/fill
oplot,[yfit_lun],[xfit_lun],psym=8,color=127;,symsize=1.5
usersym,xrect,yrect
oplot,[yfit_lun],[xfit_lun],psym=8

device,/close

end
