pro search_chi2_vbsb1, vb_file, vb_num, sb1_file, sb1_num, mcrit=mcrit

; Randomly search through orbital parameter space to find a range of
; best fit orbits,
; 
; Compute the orbital elements through a Newton-Raphson technique.
; Fits simultaneously for VB + SB1 orbits
;
; Calls the following routines:
;     calc_Ei.pro
;     solve_trans.pro
;     calc_sb1fit.pro - for plotting model orbits
;     calc_vbfit.pro - for plotting model orbits
;
; Equations:
; + Visual Binary +  
; Initial estimates (P0,T0,e0,a0,i0,Omega0,omega0) and a set of observations 
; (ti,rhoi,thetai) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Radius vector:
;	r = a(1-e^2)/(1 + ecos(nu))
; Fitted positions (rho_fit,theta_fit) are determined from the orbital elements
; 	tan(theta - Omega) = tan(nu + omega)*cosi
;	rho = r*cos(nu + omega)*sec(theta - Omega)
; + Spectroscopic Binary +  
; Initial estimates (P0,T0,e0,omega0,K1|0,K2|0,Vsys0) and  
; a set of observations (ti,V1i,V2i) are known.
; The eccentric anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; True anomaly:
;	tan(nu/2) = E - esin(E)
; Projected semi-major axes (a1sini, a2sini):
;       a1sini = (P/2pi)*K1*(1 - e^2)^1/2
;       a2sini = (P/2pi)*K2*(1 - e^2)^1/2
; The fitted velocities (V1_i, V2_i) are determined from the orbital elements:
;       V1 =  K1*[e*cos(omega) + cos(nu + omega)] + Vsys
;       V2 = -K2*[e*cos(omega) + cos(nu + omega)] + Vsys
;
; Procedure:
; Minimize chi squared between data point positions and fitted positions:
;	chi^2 = sum[(rhodata - rhofit)^2/sigmarho^2 
;		    + (thetadata - thetafit)^2/sigmatheta^2]
;             + sum[(V1data - V1fit)^2/sigmaV1^2]
;		    + (V2data - V2fit)^2/sigmaV2^2]
; Take partial derivatives of chi^2 with respect to the orbital elements,
; (P,T,e,a(mas),i,Omega,omega,K1,K2,Vsys) and set to zero. 
; Solve for the  value of the orbital elements that minimize chi^2.
;
; Since the partial derivatives are non-linear, cannot solve for the orbital
; elements that minimize chi^2 analytically.
;
; In turn, use a Newton-Raphson technique to converge upon the solution.
; Replace (rhoi,thetai) in the chi^2 equation with the Taylor series 
; approximation,
; 	x = x| + dx/dP|(P-P0) + dx/dT|(T-T0) + dx/de|(e-e0) 
;	       + dx/da|(a-Aa) + dx/di|(i-i0) + dx/dW|(W-W0) + dx/dw|(w-w0)
;	       + dx/d(K1)|(K1 - K1|0) + dx/d(K2)|(K2 - K2|0)
;              + dx/d(Vsys)|(Vsys - Vsys0)
; where x=(rho,theta,V1,V2) and the | represents the value evaluated at for the
; initial estimates for the orbital elements. Now, rhofit, thetafit, V1, and V2 
; are linear in (P,T,e,a,i,W,w,K1,K2,Vsys), so the partial derivatives can 
; easily be taken and solved by setting up a matrix and using Cramer's method.
;
; Parameters:
;	Period:	 period (P)
;	Tperi: 	 time of periastron passage (T)
;	ecc:	 eccentricity (e)
;	major:	 semi-major axis (in mas) (a)
;	inc:	 inclination (i)
;	Omega:	 position angle of node (W)
;	omega:	 angle between node and periastron (w)
;       K1:      velocity amplitude of primary (in km/s) (K1)
;       K2:      velocity amplitude of secondary (in km/s) (K2)
;       Vsys:    systemic velocity
;	t_vb:	 time of observation
;	rho_d:	 separation - data
;	theta_d: postion angle - data
;	rho_f:	 sep - fit (determined from orbital elements for time of obs.)
;	theta_f: PA - fit (determined from orbital elements for time of obs.)
;	t_sb1:	 time of observation
;	V1_d:	 velocity of primary - measured data
;	V1_f:	 V1 - fit (determined from orbital elements for time of obs.)
;
; INPUT:
;	vb_file: file containing data points in the format:
;		      time  theta  etheta  rho  erho
;		      where theta is the position angle, rho is the separation
;                     and etheta, erho are the measurement uncertainties
;		      (enter file name using quotes)
;	vb_num: number of data points contained in data_file
;	
;	sb1_file: file containing data points in the format:
;		      time  V1  errV1
;		      where V1 is the radial velocity of the primary
;                     and errV1 is the measurement error.
;		      (enter file name using quotes)
;	sb1_num: number of data points contained in data_file
;
; Prompted inputs:
;	initial estimates for all the orbital elements:
;             P, T, e, a, i, Omega, omega, K1, Vsys
;       vector identifying which orbital elements to solve for
;             enter 0 to hold element fixed, 1 to vary)
;             ex.   1 1 1 1 1 1 1 1 1 1 -> solves for all
;                   0 0 0 0 1 0 0 0 0 0 -> solves only for i
;
; OUTPUT:
;	best fit values of orbital elements: P,T,e,a,i,Omega,omega,K1,K2,Vsys
;       and corresponding errors
;
; Notes on units:
;    P, T, and time should be in same units (days or years)
;    K1, K2, V1, V2 should be in same units (km/s)
;
;
; Began 14 May 2002
; Modified 11 Nov 2003: 
;       This is the current version of NEWT_RAPH.PRO
;	(v1.1 and v1.2 are older versions)
; 	- restructured program into separate subroutines
; 	- add option of holding elements fixed during iterations
; 13 January 2005:
;       Fix bugs
;       - if solving for only one element, don't use cramer.pro to solve eqn's
;       - fix 360 degree roll overs in (PAdata - PAfit)
;       Save version v4 as backup
; 11 July 2005:
;       - Include Marquadt Method of adjusting diagonal elements of
;         the alpha matrix to ensure convergence.
;       - Fixed non-convergence problems 
;           + do not remove P wrap around effects from dx/dP
;             leave (t-T)/P in full form; net effect is that adjustments
;             are smaller
;       Save v5 as backup 
;
; 31 May 2006: Save unmodified program as version 2
; 31 May 2006: Fit SB2 parameters in terms of K1 and K2
;              (as opposed to a1sini and a2sini)
;
; 22 September 2006:
;       - Fixed the way subsets of parameters are handled
;       - Wasn't properly constructing matrices if not all parameters vary
;
; 06 November 2015:
;       - Add flag_wa to calc_deriv_vb.pro and calc_vbfit.pro.  This allows
;         the option of defining omega = omega_A in both the SB2 and VB orbits.

close,1
close,2
close,3

!P.font=0
!P.multi=0
!P.charsize=1.2
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.2	;scale factor for adjusting size of plotting symbols

; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1

; keyword_set returns True (1) if variable is defined, False (0) if undefined

ans_mcrit = keyword_set(mcrit)

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time_vb = dblarr(vb_num)
theta = dblarr(vb_num)
rho = dblarr(vb_num)
dtheta = dblarr(vb_num)
drho = dblarr(vb_num)

time_sb1 = dblarr(sb1_num)
V1_sb1 = dblarr(sb1_num)
dV1_sb1 = dblarr(sb1_num)

; Read VB measurements

openr,lun,vb_file,/get_lun

for i=0, vb_num-1 do begin

	readf, lun, temp1, temp2, temp3, temp4, temp5

	time_vb(i) = temp1
	theta(i) = temp2
	dtheta(i) = temp3
	rho(i) = temp4	
	drho(i) = temp5
	
endfor

close,lun

; convert theta and dtheta to radians

theta = theta*!dpi/180
dtheta = dtheta*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(theta)^2*drho^2 + rho^2*sin(theta)^2*dtheta^2)
dy = sqrt(sin(theta)^2*drho^2 + rho^2*cos(theta)^2*dtheta^2)

; Read SB1 measurements

openr,lun,sb1_file,/get_lun

for i=0, sb1_num-1 do begin

	readf, lun, temp1, temp2, temp3

	time_sb1(i) = temp1
	V1_sb1(i) = temp2
	dV1_sb1(i) = temp3
	
endfor

close,lun

; Obtain values for P,T,e,a(mas),i,Omega,omega,K1(km/s),K2(km/s),Vsys

period = 0d
Tperi = 0d
ecc = 0d
major = 0d
inc = 0d
W_cap = 0d
w_low = 0d
K1 = 0d
Vsys = 0d

print,"Enter P,T,e,a(mas),i,Omega,omega,K1(km/s),Vsys:"
read,period,Tperi,ecc,major,inc,W_cap,w_low,K1,Vsys

print,"Initial P,T,e,a,i,Omega,omega,K1,Vsys:"
print,period,Tperi,ecc,major,inc,W_cap,w_low,K1,Vsys

print,'Determine which parameters to search in grid search (fixed = 0)'
print,'and which elements to optimize (1) during each iteration.'
print, "[P,T,e,a,i,Omega,omega,K1,Vsys]"
read, f0,f1,f2,f3,f4,f5,f6,f7,f8

elfix=[f0,f1,f2,f3,f4,f5,f6,f7,f8]


Pmin = 0.0d
Pmax = 0.0d

Tpmin = 0.0d
Tpmax = 0.0d

emin = 0.0d
emax = 0.0d

amin = 0.0d
amax = 0.0d

imin = 0.0d
imax = 0.0d

Wcmin = 0.0d
Wcmax = 0.0d

wlmin = 0.0d
wlmax = 0.0d

K1min = 0.0d
K1max = 0.0d

Vmin = 0.0d
Vmax = 0.0d

if (f0 eq 0) then begin
   print,'Enter min max search range for P:'
   read,Pmin,Pmax
   Prange = Pmax - Pmin
endif
   
if (f1 eq 0) then begin
   print,'Enter min max search range for T:'
   read,Tpmin,Tpmax
   Tprange = Tpmax - Tpmin
endif
   
if (f2 eq 0) then begin
   print,'Enter min max search range for e:'
   read,emin,emax
   erange = emax - emin
endif
   
if (f3 eq 0) then begin
   print,'Enter min max search range for a:'
   read,amin,amax
   arange = amax - amin
endif
   
if (f4 eq 0) then begin
   print,'Enter min max search range for i:'
   read,imin,imax
   imin = imin*!dpi/180
   imax = imax*!dpi/180
   irange = imax - imin
   endif
   
if (f5 eq 0) then begin
   print,'Enter min max search range for Omega:'
   read,Wcmin,Wcmax
   Wcmin = Wcmin*!dpi/180
   Wcmax = Wcmax*!dpi/180
   Wcrange = Wcmax - Wcmin
   endif
   
if (f6 eq 0) then begin
   print,'Enter min max search range for omega:'
   read,wlmin,wlmax
   wlmin = wlmin*!dpi/180
   wlmax = wlmax*!dpi/180
   wlrange = wlmax - wlmin
endif
   
if (f7 eq 0) then begin
   print,'Enter min max search range for K1:'
   read,K1min,K1max
   K1range = K1max - K1min
endif
   
if (f8 eq 0) then begin
   print,'Enter min max search range for Vsys:'
   read,Vmin,Vmax
   Vrange = Vmax - Vmin
endif

; Convert i, Omega, omega to radians
inc = inc*!dpi/180.0
W_cap = W_cap*!dpi/180.0
w_low = w_low*!dpi/180.0

EL = [period,Tperi,ecc,major,inc,W_cap,w_low,K1,Vsys]
EL_vb = [period,Tperi,ecc,major,inc,W_cap,w_low]
EL_sb1 = [period,Tperi,ecc,K1,w_low,Vsys]

elLabel = ['P','T','e','a(mas)','i','Omega','omega_A','K1(km/s)','Vsys']

nEl = n_elements(EL)
nEl_vb = n_elements(EL_vb)
nEl_sb1 = n_elements(EL_sb1)

; Determine subarray of indices that are VB, SB1, and SB2 parameters

vb_par = dblarr(nEl_vb)
sb1_par = dblarr(nEl_sb1)

for i=0, nEl_vb-1 do begin
    ind = where(EL_vb(i) eq EL)
    vb_par(i) = ind
endfor

for i=0, nEl_sb1-1 do begin
    ind = where(EL_sb1(i) eq EL)
    sb1_par(i) = ind
endfor

; Create initial model


calc_vbfit,EL_vb,time_vb,theta_f, rho_f, /flag_wa

calc_sb1fit, EL_sb1, time_sb1, V1_f_sb1

diff_theta = theta - theta_f

; account for any wrap-arounds from 360 deg to 0 deg

for k=0, vb_num-1 do begin
   if (diff_theta(k) ge !dpi) then $
      diff_theta(k) = diff_theta(k) - 2*!dpi
   if (diff_theta(k) le -!dpi) then $
      diff_theta(k) = 2*!dpi + diff_theta(k)	
endfor

chi2 = total((diff_theta)^2/dtheta^2 + (rho - rho_f)^2/drho^2) $
       + total((V1_sb1 - V1_f_sb1)^2/dV1_sb1^2)

chi2_vb = total((diff_theta)^2/dtheta^2 + (rho - rho_f)^2/drho^2)

chi2_sb1 = total((V1_sb1 - V1_f_sb1)^2/dV1_sb1^2) 


; degrees of freedom
dof = 2d*vb_num + sb1_num - nEl

; reduced chi squared:
chi2red = chi2/dof

print, "chi2:", chi2
print, "Reduced chi2:", chi2red
print, "chi2 VB:",chi2_vb
print, "chi2 SB:",chi2_sb1

print,'Unscaled chi2:',chi2
ans_scale = 'y'
print,'Do you want to scale chi2 so that chi2red = 1.0 (y/n)?'
read,ans_scale
if (ans_scale eq 'y') then scale = 1.0/chi2red else scale = 1.0

chi2 = chi2*scale
print,'Scaled chi2:',chi2
print,'Scale factor for chi2:',scale

chi2_best = chi2
bestpar = EL

EL_orig = EL

; Save as best fit

EL_best = EL
EL_vb_best = EL_vb
EL_sb1_best = EL_sb1


chi2_min = chi2
chi2_redmin = chi2/dof

; begin modifying P,T,e

total_num = 0
chidiff = 0.0

print, 'Enter the variation of chi2 you wish to examine:'
read, chidiff
print, 'Enter number of solutions you wish to find within this variation of chi2:'
read,total_num

seed=1001L
print,'Enter seed for random generator (ex. 1001L):'
read,seed

; Set limits on total mass if the mcrit flag is set

if (ans_mcrit) then begin
   
   Mmin = 0.0d
   Mmax = 0.0d
   print,'Enter minimum and maximum total mass (in Msun) as search constraint:'
   read,Mmin,Mmax

   parallax = 0.0d

   print, 'Enter the parallax of the system (in same units as separation):'
   print,'(distance of 140, parallax = 7.14 mas)'
   read, parallax

   ansyd=' '
   print,'Is period in years or days (y/d)?'
   read,ansyd

   if (ansyd eq 'y') then yd = 1.0d
   if (ansyd eq 'd') then yd = 1.0d/365.25

   ; calculate mass of system
   mass = major^3/parallax^3/(yd*period)^2

   bestmass = mass

   M2 = 0.033573182d * major^2/ parallax^2*K1/(yd*period)/sin(inc)*sqrt(1.0 - ecc^2)

   M1 = major^3/parallax^3/(yd*period)^2 - M2

   
   if (mass ge Mmin) and (mass le Mmax) then check_mass = 1.0 else check_mass = 0.0
   
   if (M1 ge Mmin) and (M1 le Mmax) then check_mass = 1.0 else check_mass = 0.0
   if (M2 ge Mmin) and (M2 le Mmax) then check_mass = 1.0 else check_mass = 0.0

endif else check_mass = 1.0


; **** Plot model orbits ****

ans_cont = ' '

; plot model and data of initial guess for visual orbit

tnum=1000.0
tstep = period/tnum
tmin = Tperi

tarr = findgen(tnum)*tstep + tmin

calc_vbfit, EL_vb, tarr, theta_mod, rho_mod, /flag_wa

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

calc_vbfit, EL_vb, time_vb, theta_f, rho_f, /flag_wa

xfit = rho_f * cos(theta_f)	; x coordinate
yfit = rho_f * sin(theta_f)	; y coordinate

print,'Visual orbit'
plot,-yarr,xarr,psym=6
oplot,-ymod,xmod

; Connected measured and fit values
for i=0,vb_num-1 do oplot,[-yfit(i),-yarr(i)],[xfit(i),xarr(i)]

print,'Hit return to continue'
read,ans_cont

; plot model and data of initial guess for SB1 data

tnum = 1000.0
tmin = min(time_sb1)
tmax = max(time_sb1)
tstep = (tmax-tmin)/tnum

tarr = findgen(tnum)*tstep + tmin

calc_sb1fit, EL_sb1, tarr, V1mod

vmax = max(V1mod)
vmin = min(V1mod)

print,'SB1 velocity curve'
plot,time_sb1,V1_sb1,psym=2,yrange=[vmin,vmax]
oplot,tarr,V1mod

vmax = max(V1_sb1)
vmin = min(V1_sb1)

print,'Hit return to continue'
read,ans_cont

count = 0

; open file "result_chi" to save chi squared values
openw,1,'result_chi_vb'
printf,1,period,Tperi,ecc,major,inc*180.0/!dpi,W_cap*180.0/!dpi,w_low*180.0/!dpi,chi2
openw,2,"result_chi_sb1"
printf,2,period,Tperi,ecc,w_low*180.0/!dpi,K1,Vsys,chi2
openw,3,'result_chi_all'
printf,3,period,Tperi,ecc,major,inc*180.0/!dpi,W_cap*180.0/!dpi,w_low*180.0/!dpi,K1,Vsys,chi2

Parr = dblarr(total_num)
Tarr = dblarr(total_num)
earr = dblarr(total_num)
aarr = dblarr(total_num)
iarr = dblarr(total_num)
Wcaparr = dblarr(total_num)
wlowarr = dblarr(total_num)
K1arr = dblarr(total_num)
Vsysarr = dblarr(total_num)
if (ans_mcrit) then Marr = dblarr(total_num)

ind_optimize = where(elfix eq 1, count_optimize)
if (count_optimize gt 0) then ans_optimize = 1 else ans_optimize = 0

while (count lt total_num) do begin

   ; Set orbital parameters to original values:
   period = EL_orig(0)
   Tperi = EL_orig(1)
   ecc = EL_orig(2)
   major = EL_orig(3)
   inc = EL_orig(4)
   W_cap = EL_orig(5)
   w_low = EL_orig(6)
   K1 = EL_orig(7)
   Vsys = EL_orig(8)

   if (f0 eq 0) then period = Pmin + Prange*randomu(seed,/double,/uniform)
   if (f1 eq 0) then Tperi = Tpmin + Tprange*randomu(seed,/double,/uniform)
   if (f2 eq 0) then ecc = emin + erange*randomu(seed,/double,/uniform)
   if (f3 eq 0) then major = amin + arange*randomu(seed,/double,/uniform)
   if (f4 eq 0) then inc = imin + irange*randomu(seed,/double,/uniform)
   if (f5 eq 0) then W_cap = Wcmin + Wcrange*randomu(seed,/double,/uniform)
   if (f6 eq 0) then w_low = wlmin + wlrange*randomu(seed,/double,/uniform)
   if (f7 eq 0) then K1 = K1min + K1range*randomu(seed,/double,/uniform)
   if (f8 eq 0) then Vsys = Vmin + Vrange*randomu(seed,/double,/uniform)

   EL = [period,Tperi,ecc,major,inc,W_cap,w_low,K1,Vsys]

   ; Optimize parameters that need to be fit:
   if (ans_optimize) then begin

      auto_fit_orbit_vbsb1, time_vb, theta, dtheta, rho, drho, time_sb1, V1_sb1, dV1_sb1, EL, elfix

      ; Reset orbital parameters to optimized values:
      period = EL(0)
      Tperi = EL(1)
      ecc = EL(2)
      major = EL(3)
      inc = EL(4)
      W_cap = EL(5)
      w_low = EL(6)
      K1 = EL(7)
      Vsys = EL(8)

   endif
 
   EL_vb = [period,Tperi,ecc,major,inc,W_cap,w_low]
   EL_sb1 = [period,Tperi,ecc,K1,w_low,Vsys]

   ; Create model

   calc_vbfit,EL_vb,time_vb,theta_f, rho_f, /flag_wa

   diff_theta = theta - theta_f

   ; account for any wrap-arounds from 360 deg to 0 deg

   for k=0, vb_num-1 do begin
      if (diff_theta(k) ge !dpi) then $
         diff_theta(k) = diff_theta(k) - 2*!dpi
      if (diff_theta(k) le -!dpi) then $
         diff_theta(k) = 2*!dpi + diff_theta(k)	
   endfor

   calc_sb1fit, EL_sb1, time_sb1, V1_f_sb1

   chi2 = total((diff_theta)^2/dtheta^2 + (rho - rho_f)^2/drho^2) $
        + total((V1_sb1 - V1_f_sb1)^2/dV1_sb1^2)

   chi2 = chi2*scale

   delta_chi  = abs(chi2 - chi2_best)

   if (ans_mcrit) then begin

      ; calculate mass of system
      mass = major^3/parallax^3/(yd*period)^2

      M2 = 0.033573182d * major^2/ parallax^2*K1/(yd*period)/sin(inc)*sqrt(1.0 - ecc^2)

      M1 = major^3/parallax^3/(yd*period)^2 - M2

      if (mass lt Mmin) or (mass gt Mmax) then check_mass = 0.0 else begin
         if (M1 lt Mmin) or (M1 gt Mmax) then check_mass = 0.0 else begin
            if (M2 lt Mmin) and (M2 gt Mmax) then check_mass = 0.0 else check_mass = 1.0
         endelse
      endelse
   endif

  
   if (delta_chi lt chidiff) and (check_mass) then begin

      if (ans_mcrit) then print,count,period,Tperi,ecc,major,inc*180.0/!dpi,W_cap*180.0/!dpi,w_low*180.0/!dpi,K1,Vsys,mass,chi2 $
      else print,count,period,Tperi,ecc,major,inc*180.0/!dpi,W_cap*180.0/!dpi,w_low*180.0/!dpi,K1,Vsys,chi2

      printf,1,period,Tperi,ecc,major,inc*180.0/!dpi,W_cap*180.0/!dpi,w_low*180.0/!dpi,chi2
      printf,2,period,Tperi,ecc,w_low*180.0/!dpi,K1,Vsys,chi2
      printf,3,period,Tperi,ecc,major,inc*180.0/!dpi,W_cap*180.0/!dpi,w_low*180.0/!dpi,K1,Vsys,chi2

      Parr(count) = period
      Tarr(count) = Tperi
      earr(count) = ecc
      aarr(count) = major
      iarr(count) = inc*180.0/!dpi
      Wcaparr(count) = W_cap*180.0/!dpi
      wlowarr(count) = w_low*180.0/!dpi
      K1arr(count) = K1
      Vsysarr(count) = Vsys
      if (ans_mcrit) then Marr(count) = mass

      count = count+1

   endif
      
   if (chi2 lt chi2_best) and (check_mass) then begin
      chi2_min = chi2
      chi2_redmin = chi2/dof

      EL_best = EL
      EL_vb_best = EL_vb
      EL_sb1_best = EL_sb1

      chi2_best = chi2
      bestpar = EL
      if (ans_mcrit) then bestmass = mass
   endif


endwhile

close,1
close,2

period = EL_best(0)
Tperi = EL_best(1)
ecc = EL_best(2)
major = EL_best(3)
inc = EL_best(4)
Wcap = EL_best(5)
wlow = EL_best(6)
K1 = EL_best(7)
Vsys = EL_best(8)


EL = EL_best
EL_vb  = EL_vb_best
EL_sb1 = EL_sb1_best

bs = [1.0,1.0,1.0,1.0,180.0/!dpi,180.0/!dpi,180.0/!dpi,1.0,1.0]

print,"Best fit P T e a i W w K1 Vsys"
print, EL_best*bs

print,"chi2:", chi2_best
print,"reduced chi2:", chi2_best/dof

print,'Error estimates: min max mean median stddev'
print,'P:     ',min(Parr),max(Parr),mean(Parr),median(Parr),stdev(Parr)
print,'T:     ',min(Tarr),max(Tarr),mean(Tarr),median(Tarr),stdev(Tarr)
print,'e:     ',min(earr),max(earr),mean(earr),median(earr),stdev(earr)
print,'a:     ',min(aarr),max(aarr),mean(aarr),median(aarr),stdev(aarr)
print,'i:     ',min(iarr),max(iarr),mean(iarr),median(iarr),stdev(iarr)
print,'Omega: ',min(Wcaparr),max(Wcaparr),mean(Wcaparr),median(Wcaparr),stdev(Wcaparr)
print,'omega: ',min(wlowarr),max(wlowarr),mean(wlowarr),median(wlowarr),stdev(wlowarr)
print,'K1:    ',min(K1arr),max(K1arr),mean(K1arr),median(K1arr),stdev(K1arr)
print,'Vsys:  ',min(Vsysarr),max(Vsysarr),mean(Vsysarr),median(Vsysarr),stdev(Vsysarr)
if (ans_mcrit) then print,'M:     ',min(Marr),max(Marr),mean(Marr),median(Marr),stdev(Marr)


; **** Plot model orbits ****

ans_cont = ' '

; plot model and data of initial guess for visual orbit

tnum=1000.0
tstep = period/tnum
tstart = Tperi

tarr = findgen(tnum)*tstep + tstart

calc_vbfit, EL_vb_best, tarr, theta_mod, rho_mod, /flag_wa

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

calc_vbfit, EL_vb_best, time_vb, theta_f, rho_f, /flag_wa

xfit = rho_f * cos(theta_f)	; x coordinate
yfit = rho_f * sin(theta_f)	; y coordinate

print,'Visual orbit'
plot,-yarr,xarr,psym=6
oplot,-ymod,xmod

; Connected measured and fit values
for i=0,vb_num-1 do oplot,[-yfit(i),-yarr(i)],[xfit(i),xarr(i)]

print,'Hit return to continue'
read,ans_cont

; plot model and data of initial guess for SB1 data

tnum = 1000.0
tstart = min(time_sb1)
tstop = max(time_sb1)
tstep = (tstop-tstart)/tnum

tarr = findgen(tnum)*tstep + tstart

calc_sb1fit, EL_sb1, tarr, V1mod

vmax = max(V1mod)
vmin = min(V1mod)

print,'SB1 velocity curve'
plot,time_sb1,V1_sb1,psym=2,yrange=[vmin,vmax]
oplot,tarr,V1mod

print, "Final values:"

calc_vbfit,EL_vb,time_vb,theta_f, rho_f, /flag_wa

diff_theta = theta - theta_f

; account for any wrap-arounds from 360 deg to 0 deg

for k=0, vb_num-1 do begin
   if (diff_theta(k) ge !dpi) then $
      diff_theta(k) = diff_theta(k) - 2*!dpi
   if (diff_theta(k) le -!dpi) then $
      diff_theta(k) = 2*!dpi + diff_theta(k)	
endfor

calc_sb1fit, EL_sb1, time_sb1, V1_f_sb1


chi2 = total((diff_theta)^2/dtheta^2 + (rho - rho_f)^2/drho^2) $
            + total((V1_sb1 - V1_f_sb1)^2/dV1_sb1^2)

chi2_vb = total((diff_theta)^2/dtheta^2 + (rho - rho_f)^2/drho^2)

chi2_sb1 = total((V1_sb1 - V1_f_sb1)^2/dV1_sb1^2) 

chi2 = chi2*scale
chi2_vb = chi2_vb*scale
chi2_sb1 = chi2_sb1*scale

;; convert data points to x and y coordinates
;xfit = rho_f * cos(theta_f)	; x coordinate
;yfit = rho_f * sin(theta_f)	; y coordinate
;chi2xy = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

mfit = n_elements(EL)

; degrees of freedom
dof = 2.0*vb_num + 2.0*sb1_num - mfit

; reduced chi squared:
chi2red = chi2/dof

ELerr = dblarr(mfit)
;for i=0,mfit-1 do ELerr(i) = sqrt(chi2red/scale)*sqrt(invmat(i,i))

k=0
print,"Orbital Element     Error"
for i=0, 3 do begin
        print, elLabel(i), ': ', El(i)
endfor
for i=4, 6 do begin
         print, elLabel(i), ': ', El(i)*180/!dpi
endfor
for i=7,8 do begin
        print, elLabel(i), ': ', El(i)
endfor
;print,'K1:',K1
;print,'K2:',K2
;print,'Parallax',parallax
print, "Final chi2:", chi2
print, "Reduced chi2:", chi2red
print, "chi2 VB:",chi2_vb
print, "chi2 SB:",chi2_sb1
print, "Num VB:",vb_num
print, "Num SB:",sb1_num
print,'Number of iterations:',count
;print,'Final chi2:',chi2

!P.MULTI = [0,0,0]




;stop,'Type .cont to continue'

end
