pro select_mass_aiw, data_file, num, pte_file, fnum, datanum, orb_deg=orb_deg, scale_dof=scale_dof

; Uses result_chi file that contains all orbital parameters
; P T e a i W w chi2
; Create for use with search_chi2_aiW.pro

; Select orbits within a specified range of masses

; Modification of plot_PaeM.pro
; Make colored plots for thesis
;
; Visualize parameter space: 
;	e vs log(P) 
;	log(a) vs log(P) 
;	M vs e 
;	M vs log(P) 
;
; Color scheme: red   = 1-sigma (Deltachi2 = 1)
;		blue  = 2-sigma (Deltachi2 = 4)
;		green = 3-sigma (Deltachi2 = 9)
;
; Computes statistics for orbital elements
;
; Modification of LEAST_PTE.PRO
;
; INPUT:
; 	data_file: name of file containing the observations
;		format: time PA ePA sep esep
;	num: number of lines in data_file
;	pte_file: name of file containing P,T,e,chi2 for possible solutions
;		format: P T e chi2
;	fnum: number of lines in pte_file
;
; PROMPTED INPUTS:
;	system parallax (in mas)
;
; PROCEDURE:
; 	performs least squares fit
; 	computes a,i,Omega,omega,M for each P,T,e set
;	computes min, max, mean, median, and stdev for each orbital element
;
; OUTPUT: 
;	temp.ps: 4-panel colored plots of cross-cuts through M,P,a,e
;            parameter spaces
;
; Optional parameters:
; - Set orb_deg keyword if i,Omega,omega in pte_file are already in degrees.
; - Set scale_dof to the number of DOF if you want to scale chi2.


!P.font=0
!P.multi=0
!P.charsize=1.2
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
;frac=0.7	;scale factor for adjusting size of plotting symbols
frac=1.2	;scale factor for adjusting size of plotting symbols
nthick = 3

; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1

; Set orb_deg keyword if i,Omega,omega in pte_file are already in degrees
; keyword_set returns True (1) if orb_deg is defined, False (0) if undefined

ans_deg = keyword_set(orb_deg)

ans_scale = keyword_set(scale_dof)

close,1
close,2
close,3


; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time = dblarr(num)
theta = dblarr(num)
rho = dblarr(num)
dtheta = dblarr(num)
drho = dblarr(num)

openr,3,data_file

for i=0, num-1 do begin

	readf, 3, temp1, temp2, temp3, temp4, temp5

	time(i) = temp1
	theta(i) = temp2
	dtheta(i) = temp3
	rho(i) = temp4	
	drho(i) = temp5
	
endfor

close,3

; convert theta and dtheta to radians

theta = theta*!dpi/180
dtheta = dtheta*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(theta)^2*drho^2 + rho^2*sin(theta)^2*dtheta^2)
dy = sqrt(sin(theta)^2*drho^2 + rho^2*cos(theta)^2*dtheta^2)

; Read in pte file names

openr,1,pte_file
openw,2,"result_all"
printf,2, "P T e a i Omega omega mass chi2:"

period = 0d
Tperi = 0d
ecc = 0d
major = 0d
inc = 0d
W_cap = 0d
w_low = 0d
xchi = 0d

massi = dblarr(fnum)
a_arr = dblarr(fnum)
e_arr = dblarr(fnum)
i_arr = dblarr(fnum)
P_arr = dblarr(fnum)
T_arr = dblarr(fnum)
W_cap_arr = dblarr(fnum)
w_low_arr = dblarr(fnum)
chi_arr = dblarr(fnum)

mratio_arr  = dblarr(fnum)
M1_arr = dblarr(fnum)  ; compute on basis of K1 and a^3/P^2
M2_arr = dblarr(fnum)
M1_alt = dblarr(fnum)   ; compute on basis of K1 and K2
M2_alt = dblarr(fnum)
par_alt = dblarr(fnum)

freepar = 1.0		;number of free parameters (up to 7)
	
; degrees of freedom
dof = 2.0*datanum - freepar
	
parallax = 0.0d

print, 'Enter the parallax of the system (in same units as separation):'
print,'(distance of 140, parallax = 7.14 mas)'
read, parallax

K1 = 0.0d
K2 = 0.0d
print,'Enter K1 and K2 radial velocity amplitudes in km/s (if unknown enter 0):'
read, K1, K2

ansyd=' '
print,'Is period in years or days (y/d)?'
read,ansyd

if (ansyd eq 'y') then yd = 1.0d
if (ansyd eq 'd') then yd = 1.0d/365.25

print, 'Computing masses...'

for j=0, fnum-1 do begin

	; Read in P,T,e,a,i,W,w from pte_file

	readf, 1, period, Tperi, ecc, major, inc, W_cap, w_low, xchi

	EL = [period,Tperi,ecc,major,inc,W_cap,w_low]

	; calculate mass of system using distance of 140 pc
	; parallax = 7.14 mas
	
	mass = major^3/parallax^3/(yd*period)^2

        ; If i,Omega,omega are in degrees, then convert to radians:
        if (ans_deg) then begin
           inc = inc*!dpi/180.0
           W_cap = W_cap*!dpi/180.0
           w_low = w_low*!dpi/180.0
        endif

	massi(j) = mass
	a_arr(j) = major
	e_arr(j) = ecc
	i_arr(j) = 180*inc/!dpi
	P_arr(j) = period
	T_arr(j) = Tperi
	W_cap_arr(j) = 180*W_cap/!dpi
	w_low_arr(j) = 180*w_low/!dpi
	chi_arr(j) = xchi      ; read chi2 in from data file

        if (K1 ne 0) then begin
           mratio_arr(j) = 3.784229924d-5 * K1^3 * period * (1.0- ecc^2)^(1.5)
           M2_arr(j) = 0.033573182d * major^2/ parallax^2*K1/period/sin(inc)*sqrt(1.0 - ecc^2)
           M1_arr(j) = mass - M2_arr(j)
           if(K2 ne 0) then begin
              M1_alt(j) = 1.036d-7 * (K1 + K2)^2 * K2 * period * (1.0 - ecc^2)^(1.5) / (sin(inc))^3.0
              M2_alt(j) = 1.036d-7 * (K1 + K2)^2 * K1 * period * (1.0 - ecc^2)^(1.5) / (sin(inc))^3.0
              a1 = 0.01375d*period*K1*sqrt(1 - ecc^2)/sin(inc) *1d9 /1.4959787066d11
              a2 = 0.01375d*period*K2*sqrt(1 - ecc^2)/sin(inc) *1d9 /1.4959787066d11
              par_alt(j) = major/(a1 + a2)
           endif
        endif

	printf,2,EL(0),El(1),EL(2),EL(3),180*EL(4)/!dpi,180*EL(5)/!dpi, $
		180*EL(6)/!dpi, mass, xchi

endfor

close,1
close,2


print,"Error estimates: min max mean median stddev"
print,format='(%"M:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(massi),max(massi),mean(massi),median(massi),stddev(massi)
print,format='(%"P:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(P_arr),max(P_arr),mean(P_arr),median(P_arr),stddev(P_arr)
print,format='(%"a:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(a_arr),max(a_arr),mean(a_arr),median(a_arr),stddev(a_arr)
print,format='(%"e:",f10.6,f10.6,f10.6,f10.6,f10.6)', $
	min(e_arr),max(e_arr),mean(e_arr),median(e_arr),stddev(e_arr)
print,format='(%"i:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(i_arr),max(i_arr),mean(i_arr),median(i_arr),stddev(i_arr)
print,format='(%"T:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(T_arr),max(T_arr),mean(T_arr),median(T_arr),stddev(T_arr)
print,format='(%"W:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(W_cap_arr),max(W_cap_arr),mean(W_cap_arr), $
	median(W_cap_arr),stddev(W_cap_arr)
print,format='(%"w:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(w_low_arr),max(w_low_arr),mean(w_low_arr), $
	median(w_low_arr),stddev(w_low_arr)
print,format='(%"Mtot:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(massi),max(massi),mean(massi),median(massi),stddev(massi)

if (K1 ne 0) then begin
print,format='(%"M1:",f12.4,f10.4,f10.4,f10.4,f10.4)', $
	min(M1_arr),max(M1_arr),mean(M1_arr),median(M1_arr),stddev(M1_arr)
print,format='(%"M2:",f12.4,f10.4,f10.4,f10.4,f10.4)', $
	min(M2_arr),max(M2_arr),mean(M2_arr),median(M2_arr),stddev(M2_arr)
if(K2 ne 0) then begin
print,"M1_alt:   ",min(M1_alt),max(M1_alt),mean(M1_alt),median(M1_alt),stddev(M1_alt)
print,"M2_alt:   ",min(M2_alt),max(M2_alt),mean(M2_alt),median(M2_alt),stddev(M2_alt)
print,format='(%"par_alt:",f12.4,f10.4,f10.4,f10.4,f10.4)', $
	min(par_alt),max(par_alt),mean(par_alt),median(par_alt),stddev(par_alt)
endif
endif

ans_range = 'm'
print,'Choose orbits based on mass range or distance range (m/d)?'
read,ans_range

Mmin = 0.0
Mmax = 0.0

if (ans_range eq 'd') then print,'Enter min max of orbital parallax range to select:' $
else print,'Enter min max of Mass range to select:'
read,Mmin,Mmax

chimin = min(chi_arr)

print,'Minimum chi2:',chimin

chi2red = chimin/scale_dof

if (ans_scale) then scale = 1.0/chi2red else scale = 1.0
chi_arr = chi_arr*scale
chimin = min(chi_arr)
print,'Scaled chi2:',chimin
print,'Scale factor for chi2:',scale

; Choose from 1 sigma contours
;sig1 = where(chi_arr le (chimin+0.5), count1)
sig1 = where(chi_arr le (chimin+1.0), count1)
;sig1 = where(chi_arr le (chimin+9.0), count1)
;sig1 = where(chi_arr le (chimin+0.2825), count1)
;sig1 = where(chi_arr le (chimin+0.2822), count1)
;sig1 = where(chi_arr ge (chimin), count1)

mass_1sig = massi(sig1)
a_arr_1sig = a_arr(sig1)
e_arr_1sig = e_arr(sig1)
i_arr_1sig = i_arr(sig1)
P_arr_1sig = P_arr(sig1)
T_arr_1sig = T_arr(sig1)
W_cap_arr_1sig = W_cap_arr(sig1)
w_low_arr_1sig = w_low_arr(sig1)
chi_arr_1sig = chi_arr(sig1)
if (K1 ne 0) then begin
   M1_arr_1sig = M1_arr(sig1)
   M2_arr_1sig = M2_arr(sig1)
   if (K2 ne 0) then begin
      M1_alt_1sig = M1_alt(sig1)
      M2_alt_1sig = M2_alt(sig1)
      par_alt_1sig = par_alt(sig1)
   endif
endif

if (ans_range eq 'd') then indM = where(par_alt_1sig ge Mmin and par_alt_1sig le Mmax, countM) $
else indM = where(mass_1sig ge Mmin and mass_1sig le Mmax, countM)

M_select = mass_1sig(indM)
a_select = a_arr_1sig(indM) 
e_select = e_arr_1sig(indM) 
i_select = i_arr_1sig(indM) 
P_select = P_arr_1sig(indM) 
T_select = T_arr_1sig(indM) 
W_cap_select = W_cap_arr_1sig(indM)
w_low_select = w_low_arr_1sig(indM)
chi_select = chi_arr_1sig(indM)
if (K1 ne 0) then begin
   M1_select = M1_arr_1sig(indM)
   M2_select = M2_arr_1sig(indM)
   if (K2 ne 0) then begin
      M1_alt_select = M1_alt_1sig(indM)
      M2_alt_select = M2_alt_1sig(indM)
      par_alt_select = par_alt_1sig(indM)
   endif
endif

print, 'Number of solutions within specified mass range: ',countM

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Begin Orbit Plot

; Define user symbol to be closed circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)

; Plot orbits as small filled circles
usersym, frac*cos(Z), frac*sin(Z), /fill

;Save plots to file temp.ps
set_plot, 'ps'
;device, /ENCAPSULATED, /Color, filename='temp.eps',xsize=15,ysize=13.5
device, /ENCAPSULATED, /Color, filename='temp.eps',xsize=12,ysize=11.5
;device, /ENCAPSULATED, /Color, filename='temp.eps',xsize=10,ysize=9
;device, /Color, filename='temp.ps',xsize=15,ysize=13.5

;plot,yarr,xarr,xtitle='!4D!3RA (!9''''!3)',ytitle='!4D!3DEC (!9''''!3)',ISOTROPIC=1, $
plot,yarr,xarr,xtitle='!9D!3RA (!9''''!3)',ytitle='!9D!3DEC (!9''''!3)',ISOTROPIC=1, $
;plot,yarr,xarr,xtitle='!7D!6RA (!9''''!6)',ytitle='!7D!6DEC (!9''''!6)',ISOTROPIC=1, $
     xrange=[5,-4], yrange=[-3,6], $
     ;xrange=[3,-3], yrange=[-3,3], $
     xstyle=1,ystyle=1, /nodata, position=[0.13,0.11,0.99,0.99]

;oplot,[yarr],[xarr],psym=8,color=3
;plot_err,yarr,abs(dy),xarr,abs(dx),num

; mark primary with an asterisk
oplot,[0],[0],psym=2

tnum = 1000.0

ans_plotall = 'n'

if (ans_plotall eq 'y') then begin
for i=0, countM-1 do begin

   EL = [P_select(i),T_select(i),e_select(i),a_select(i),i_select(i)*!dpi/180.0,W_cap_select(i)*!dpi/180.0,w_low_select(i)*!dpi/180.0]

   tstep = P_select(i)/tnum
   tarr = findgen(tnum+5)*tstep + T_select(i)

   calc_vbfit, EL, tarr, PA_fit, sep_fit
   xfit = sep_fit * cos(PA_fit)	; x coordinate
   yfit = sep_fit * sin(PA_fit)	; y coordinate

   oplot,yfit,xfit

endfor

endif else print,'!!! NOT PLOTTING ALL ORBITS !!!'

plot_err,yarr,abs(dy),xarr,abs(dx),num
oplot,[yarr],[xarr],psym=8,color=3

xyouts,yarr(0)-0.18,xarr(0)-0.05,string(time(0),format='(f6.1)'),alignment=0.0
xyouts,yarr(num-1)-0.18,xarr(num-1)-0.08,string(time(num-1),format='(f6.1)'),alignment=0.0

device, /close
set_plot,'x'


print,"Error estimates: min max mean median stddev"
print,format='(%"Mtot:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(M_select),max(M_select),mean(M_select),median(M_select),stddev(M_select)
print,format='(%"P:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(P_select),max(P_select),mean(P_select),median(P_select),stddev(P_select)
print,format='(%"T:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(T_select),max(T_select),mean(T_select),median(T_select),stddev(T_select)
print,format='(%"e:",f10.6,f10.6,f10.6,f10.6,f10.6)', $
	min(e_select),max(e_select),mean(e_select),median(e_select),stddev(e_select)
print,format='(%"a:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(a_select),max(a_select),mean(a_select),median(a_select),stddev(a_select)
print,format='(%"i:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(i_select),max(i_select),mean(i_select),median(i_select),stddev(i_select)
print,format='(%"W:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(W_cap_select),max(W_cap_select),mean(W_cap_select),median(W_cap_select),stddev(W_cap_select)
print,format='(%"w:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(w_low_select),max(w_low_select),mean(w_low_select),median(w_low_select),stddev(w_low_select)
print,format='(%"chi2 (min max diff:",f10.4,f10.4,f10.4)', $
	min(chi_select),max(chi_select),max(chi_select)-min(chi_select)
if (K1 ne 0) then begin
print,format='(%"M1:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(M1_select),max(M1_select),mean(M1_select),median(M1_select),stddev(M1_select)
print,format='(%"M2:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(M2_select),max(M2_select),mean(M2_select),median(M2_select),stddev(M2_select)
if (K2 ne 0) then begin
print,format='(%"M1 alt:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(M1_alt_select),max(M1_alt_select),mean(M1_alt_select),median(M1_alt_select),stddev(M1_alt_select)
print,format='(%"M2 alt:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(M2_alt_select),max(M2_alt_select),mean(M2_alt_select),median(M2_alt_select),stddev(M2_alt_select)
print,format='(%"par alt:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(par_alt_select),max(par_alt_select),mean(par_alt_select),median(par_alt_select),stddev(par_alt_select)
endif
endif

; Select a few different sample orbits:

indchimin = where(chi_select eq min(chi_select))
indPmin = where(P_select eq min(P_select))
indPmax = where(P_select eq max(P_select))
indimin = where(i_select eq min(i_select))
indimax = where(i_select eq max(i_select))
indemin = where(e_select eq min(e_select))
indemax = where(e_select eq max(e_select))
indMmin = where(M_select eq min(M_select))
indMmax = where(M_select eq max(M_select))

Mbest = (Mmin + Mmax)/2.0
indMbest = where(abs(M_select-Mbest) eq min(abs(M_select-Mbest)))

indPmin = indPmin(0)
indPmax = indPmax(0)
indimin = indimin(0)
indimax = indimax(0)
indemin = indemin(0)
indemax = indemax(0)
indMmin = indMmin(0)
indMmax = indMmax(0)
indMbest = indMbest(0)


print,'P T e a i Omega omega M chi2'
print,'Best chi2:'
print,P_select(indchimin),T_select(indchimin),e_select(indchimin),a_select(indchimin),i_select(indchimin),W_cap_select(indchimin),w_low_select(indchimin),M_select(indchimin),chi_select(indchimin)

print,'Minimum P:'
print,P_select(indPmin),T_select(indPmin),e_select(indPmin),a_select(indPmin),i_select(indPmin),W_cap_select(indPmin),w_low_select(indPmin),M_select(indPmin),chi_select(indPmin),sig1(indM(indPmin))

print,'Maximum P:'
print,P_select(indPmax),T_select(indPmax),e_select(indPmax),a_select(indPmax),i_select(indPmax),W_cap_select(indPmax),w_low_select(indPmax),M_select(indPmax),chi_select(indPmax),sig1(indM(indPmax))

print,'Minimum i:'
print,P_select(indimin),T_select(indimin),e_select(indimin),a_select(indimin),i_select(indimin),W_cap_select(indimin),w_low_select(indimin),M_select(indimin),chi_select(indimin)

print,'Maximum i:'
print,P_select(indimax),T_select(indimax),e_select(indimax),a_select(indimax),i_select(indimax),W_cap_select(indimax),w_low_select(indimax),M_select(indimax),chi_select(indimax)

print,'Minimum e:'
print,P_select(indemin),T_select(indemin),e_select(indemin),a_select(indemin),i_select(indemin),W_cap_select(indemin),w_low_select(indemin),M_select(indemin),chi_select(indemin)

print,'Maximum e:'
print,P_select(indemax),T_select(indemax),e_select(indemax),a_select(indemax),i_select(indemax),W_cap_select(indemax),w_low_select(indemax),M_select(indemax),chi_select(indemax)

print,'Minimum Mass:'
print,P_select(indMmin),T_select(indMmin),e_select(indMmin),a_select(indMmin),i_select(indMmin),W_cap_select(indMmin),w_low_select(indMmin),M_select(indMmin),chi_select(indMmin)

print,'Maximum Mass:'
print,P_select(indMmax),T_select(indMmax),e_select(indMmax),a_select(indMmax),i_select(indMmax),W_cap_select(indMmax),w_low_select(indMmax),M_select(indMmax),chi_select(indMmax)

print,'Mbest: '
print,P_select(indMbest),T_select(indMbest),e_select(indMbest),a_select(indMbest),i_select(indMbest),W_cap_select(indMbest),w_low_select(indMbest),M_select(indMbest),chi_select(indMbest)


;;;;;;;;;;;;;;;;;;;;;;
; Plot sorted orbits ;
;;;;;;;;;;;;;;;;;;;;;;


!P.MULTI = [0,0,0]

set_plot, 'ps'
;device, /Color, Bits_per_pixel=8, filename='temp_sort.ps',xsize=12,ysize=12
device, /Color, Bits_per_pixel=8, filename='temp_sort.eps',xsize=12,ysize=12,isolatin1=1,/encapsulated

xmax = max(xarr)
xmin = min(xarr)

ymax = max(yarr)
ymin = min(yarr)

sepmax = round(1.2*max(abs([yarr,xarr])))

plot,yarr,xarr, ISOTROPIC=1, position=[0.16,0.12,0.95,0.98], $
  ;xtitle='E-W sep (!9''''!3)',ytitle='N-S sep (!9''''!3)', $
;  xtitle='!4D!3RA (!9''''!3)',ytitle='!4D!3DEC (!9''''!3)', $
  xtitle='!9D!3RA (mas)',ytitle='!9D!3DEC (mas)', $
;  xrange=[0.10,-0.20], yrange=[-0.20,0.10], /nodata
;  xrange=[5,-4.5], yrange=[-3,6],xstyle=1,ystyle=1, /nodata
   xrange=[10,-10], yrange=[-10,10],xstyle=1,ystyle=1, /nodata  ; TWA 3A
;  xrange=[0.10,-0.20], yrange=[-0.15,0.15], /nodata
;   xrange=[sepmax,-sepmax], yrange=[-sepmax,sepmax], $
;   xrange=[150,-100], yrange=[-150,100], $
;   xstyle=1,ystyle=1, /nodata
  
ind_sort = sort(i_select)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Undelete for generic orbit ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;Nplot = 5.0   ; number of orbits to plot
;
;nstep = n_elements(massi)/(Nplot -1.0)
;
;istep = (max(i_select) - min(i_select))/(Nplot - 1.0)
;
;; Plot sample of other orbits
;tnum = 1000.0
;norb = 0.0
;
;xpred_arr = dblarr(Nplot)
;ypred_arr = dblarr(Nplot)
;
;iplot = min(i_select)
;
;for i=0, Nplot-1 do begin
;
;   ind_i = where(abs(i_select - iplot) eq min(abs(i_select - iplot)))
;
;   period = P_select(ind_i(0))
;   Tperi = T_select(ind_i(0))
;   ecc = e_select(ind_i(0))
;   major = a_select(ind_i(0))
;   inc = i_select(ind_i(0))
;   W_cap = W_cap_select(ind_i(0))
;   w_low = w_low_select(ind_i(0))
;   Mtot = M_select(ind_i(0))
;   chi2 = chi_select(ind_i(0))
;
;   print,period,Tperi,ecc,major,inc,W_cap,w_low,Mtot,chi2
;
;   EL = [period,Tperi,ecc,major,inc*!dpi/180.0,W_cap*!dpi/180.0,w_low*!dpi/180.0]
;
;   tstep = period/tnum
;   tarr = findgen(tnum+5)*tstep + Tperi
;
;   calc_vbfit, EL, tarr, PA_fit, sep_fit
;   xfit = sep_fit * cos(PA_fit)	; x coordinate
;   yfit = sep_fit * sin(PA_fit)	; y coordinate
;
;   oplot,yfit,xfit
;
;;   tpred = [2013.5]
;   tpred = [2014.2]
;   calc_vbfit, EL, tpred, PA, sep
;   xpred_arr(i) = sep(0) * cos(PA(0))	; x coordinate
;   ypred_arr(i) = sep(0) * sin(PA(0))	; y coordinate
;
;   norb = norb + nstep
;   iplot = iplot + istep
;
;endfor

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; For TWA 3A, orbit fall into to inclination bins ;
; need to sort out these solutions                ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Nplot = 3.0   ; number of orbits to plot
;Nplot = 100.0   ; number of orbits to plot

nstep = n_elements(massi)/(Nplot -1.0)

; Plot sample of other orbits
tnum = 1000.0
norb = 0.0

ind90 = where(i_select gt 90.0, c90)

if (c90 ge 1) then begin

   i_90 = i_select(ind90)
   Omega_90 = W_cap_select(ind90)

   istep = (max(i_90) - min(i_90))/(Nplot - 1.0)
   iplot = min(i_90)

   chi_90 = chi_select(i_90)
   ic = where(chi_90 eq min(chi_90))
   ic = ic(0)

   print,'Restrict i > 90 deg: '
   print,'i: min max mean median stdev',min(i_90),max(i_90),mean(i_90),median(i_90),stdev(i_90)
   print,'Omega: min max mean median stdev',min(Omega_90),max(Omega_90),mean(Omega_90),median(Omega_90),stdev(Omega_90)
   print,'Best i +/- sigma: ',i_90(ic),' + ',max(i_90) - i_90(ic),' - ',i_90(ic) - min(i_90)
   print,'Best Omega +/- sigma',Omega_90(ic),' + ',max(Omega_90) - Omega_90(ic),' - ',Omega_90(ic) - min(Omega_90)

   for i=0, Nplot-1 do begin

      ind_i = where(abs(i_select - iplot) eq min(abs(i_90 - iplot)))

      period = P_select(ind_i(0))
      Tperi = T_select(ind_i(0))
      ecc = e_select(ind_i(0))
      major = a_select(ind_i(0))
      inc = i_select(ind_i(0))
      W_cap = W_cap_select(ind_i(0))
      w_low = w_low_select(ind_i(0))
      Mtot = M_select(ind_i(0))
      chi2 = chi_select(ind_i(0))

      print,period,Tperi,ecc,major,inc,W_cap,w_low,Mtot,chi2

      EL = [period,Tperi,ecc,major,inc*!dpi/180.0,W_cap*!dpi/180.0,w_low*!dpi/180.0]

      tstep = period/tnum
      tarr = findgen(tnum+5)*tstep + Tperi

      calc_vbfit, EL, tarr, PA_fit, sep_fit
      xfit = sep_fit * cos(PA_fit) ; x coordinate
      yfit = sep_fit * sin(PA_fit) ; y coordinate
      
      oplot,yfit,xfit,color=3,thick=nthick
   
      norb = norb + nstep
      iplot = iplot + istep

   ;imax = where(sep_fit eq max(sep_fit))
   ;print,'Max sep,PA,x,y: ',max(sep_fit),sep_fit(imax(0)),pa_fit(imax(0))*180/!dpi,yfit(imax(0)),xfit(imax(0))

   endfor


endif


   ind90 = where(i_select le 90.0,c90)

if (c90 ge 1) then begin

   i_90 = i_select(ind90)
   Omega_90 = W_cap_select(ind90)

   istep = (max(i_90) - min(i_90))/(Nplot - 1.0)
   iplot = min(i_90)

   chi_90 = chi_select(i_90)
   ic = where(chi_90 eq min(chi_90))
   ic = ic(0)

   print,'Restrict i < 90 deg: '
   print,'i min max mean median stdev',min(i_90),max(i_90),mean(i_90),median(i_90),stdev(i_90)
   print,'Omega: min max mean median stdev',min(Omega_90),max(Omega_90),mean(Omega_90),median(Omega_90),stdev(Omega_90)
   print,'Best i +/- sigma: ',i_90(ic),' + ',max(i_90) - i_90(ic),' - ',i_90(ic) - min(i_90)
   print,'Best Omega +/- sigma',Omega_90(ic),' + ',max(Omega_90) - Omega_90(ic),' - ',Omega_90(ic) - min(Omega_90)

   for i=0, Nplot-1 do begin

      ind_i = where(abs(i_select - iplot) eq min(abs(i_90 - iplot)))

      period = P_select(ind_i(0))
      Tperi = T_select(ind_i(0))
      ecc = e_select(ind_i(0))
      major = a_select(ind_i(0))
      inc = i_select(ind_i(0))
      W_cap = W_cap_select(ind_i(0))
      w_low = w_low_select(ind_i(0))
      Mtot = M_select(ind_i(0))
      chi2 = chi_select(ind_i(0))
      
      print,period,Tperi,ecc,major,inc,W_cap,w_low,Mtot,chi2

      EL = [period,Tperi,ecc,major,inc*!dpi/180.0,W_cap*!dpi/180.0,w_low*!dpi/180.0]

      tstep = period/tnum
      tarr = findgen(tnum+5)*tstep + Tperi

      calc_vbfit, EL, tarr, PA_fit, sep_fit
      xfit = sep_fit * cos(PA_fit) ; x coordinate
      yfit = sep_fit * sin(PA_fit) ; y coordinate

      oplot,yfit,xfit,color=1

      norb = norb + nstep
      iplot = iplot + istep

   ;imax = where(sep_fit eq max(sep_fit))
   ;print,'Max sep,PA,x,y: ',max(sep_fit),sep_fit(imax(0)),pa_fit(imax(0))*180/!dpi,yfit(imax(0)),xfit(imax(0))

   endfor

endif



; Plot measured positions

; mark measured position of secondary with a square

;Define the symbol to be a closed circle
usersym, frac*cos(Z), frac*sin(Z), /fill

oploterror,yarr,xarr,abs(dy),abs(dx),psym=8;,color=1

;plot_err,yarr,abs(dy),xarr,abs(dx),num
;oplot,[yarr],[xarr],psym=8,color=3

; mark primary with an asterisk

oplot,[0],[0],psym=2

device, /close
set_plot,'x'

;stop,'Type .cont to continue'

end


;
;
;

pro plot_err, x, dx, y, dy, N

; plot error bars associated with each data point
; Input
;	x: array of x values
;	dx: array of error in x values
;	y: array of y values
;	dy: erray of error in y values
;	N: number of data points


for i=0, N-1 do begin

	oplot, [x(i)-dx(i), x(i)+dx(i)], [y(i), y(i)]
	oplot, [x(i), x(i)], [y(i)-dy(i), y(i)+dy(i)]

endfor

end
