pro sig_interval, data_file, num, pte_file, fnum

; Print out 1-sigma upper and lower bounds 
;
; Modification of CHI_PLOTS.PRO
;
; INPUT:
; 	data_file: name of file containing the observations
;		format: time PA ePA sep esep
;	num: number of lines in data_file
;	pte_file: name of file containing P,T,e,chi2 for possible solutions
;		format: P T e chi2
;	fnum: number of lines in pte_file
;
; PROMPTED INPUTS:
;	system parallax (in mas)
;	step sizes and ranges for histograms
;
; PROCEDURE:
; 	performs least squares fit
; 	computes a,i,Omega,omega,M for each P,T,e set
;	computes min, max, mean, median, and stdev for each orbital element
;       conmputes 1-sigma error intervals for each parameter
;
; OUTPUT: 
;	temp.ps: multi-plot of P,T,e,a,i,Omega,omega,M chi2 plots
;
; Began: 25 November 2003

close,1
close,2

; Read in pte file names

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time = dblarr(num)
theta = dblarr(num)
rho = dblarr(num)
dtheta = dblarr(num)
drho = dblarr(num)

openr,3,data_file

for i=0, num-1 do begin

	readf, 3, temp1, temp2, temp3, temp4, temp5

	time(i) = temp1
	theta(i) = temp2
	dtheta(i) = temp3
	rho(i) = temp4	
	drho(i) = temp5
	
endfor

close,3

openr,1,pte_file
openw,2,"result_all"
printf,2, "P T e a i Omega omega mass chi2:"

; convert theta and dtheta to radians

theta = theta*!dpi/180
dtheta = dtheta*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(theta)^2*drho^2 + rho^2*sin(theta)^2*dtheta^2)
dy = sqrt(sin(theta)^2*drho^2 + rho^2*cos(theta)^2*dtheta^2)

period = 0d
Tperi = 0d
ecc = 0d
xchi = 0d

massi = dblarr(fnum)
a_arr = dblarr(fnum)
e_arr = dblarr(fnum)
i_arr = dblarr(fnum)
P_arr = dblarr(fnum)
T_arr = dblarr(fnum)
W_cap_arr = dblarr(fnum)
w_low_arr = dblarr(fnum)
chi_arr = dblarr(fnum)
	
parallax = 0.0d

print, 'Enter the parallax of the system (in same units as separation):'
print,'(distance of 140, parallax = 7.14 mas)'
read, parallax

ansyd=' '
print,'Is period in years or days (y/d)?'
read,ansyd

if (ansyd eq 'y') then yd = 1.0d
if (ansyd eq 'd') then yd = 1.0d/365.25

print,"Enter bestfit P T e (period, time of periastron passage, eccentricity):"
read, Pbest, Tbest, ebest

; Create initial model:
			
; Determine fitted points (xfit,yfit) by computing the Thiele Innes
; elements A,F,B,G

calc_AFBG, time, xarr, dx, yarr, dy, Pbest, Tbest, ebest, $
	   A_cap, F_cap, B_cap, G_cap, xfit, yfit

; determine a,i,Omega,omega (using AIWW.PRO)

aiWw, A_cap,B_cap,F_cap,G_cap,abest,ibest,Wcbest,wlbest

Mbest = abest^3/parallax^3/(yd*Pbest)^2

; Determine chi squared

chi2best = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

; degrees of freedom
dof = 2*num - 7
; reduced chi squared:
chi2red = chi2best/dof

print,'Unscaled chi2:',chi2best
ans_scale = 'y'
print,'Do you want to scale chi2 so that chi2red = 1.0 (y/n)?'
read,ans_scale
if (ans_scale eq 'y') then scale = 1.0/chi2red else scale = 1.0

chi2best = chi2best*scale

;variables to store 1-sigma intervals

Psigmin = pbest
Psigmax = pbest
Tsigmin = Tbest
Tsigmax = Tbest
esigmin = ebest
esigmax = ebest
asigmin = abest
asigmax = abest
isigmin = ibest
isigmax = ibest
Wcsigmin = Wcbest
Wcsigmax = Wcbest
wlsigmin = wlbest
wlsigmax = wlbest
Msigmin = Mbest
Msigmax = Mbest


print, 'Computing masses...'

for j=0, fnum-1 do begin

	; Obtain initial values for P,T,e

;	print,"Enter P T e (period, time of periastron passage, eccentricity):"
	readf, 1, period, Tperi, ecc, xchi

	; Determine fitted points (xfit,yfit) by computing the Thiele Innes
	; elements A,F,B,G

	calc_AFBG, time, xarr, dx, yarr, dy, period, Tperi, ecc, $
		   A_cap, F_cap, B_cap, G_cap, xfit, yfit

	; Determine chi squared

	chi2 = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

        chi2 = chi2*scale

	; reduced chi squared:
	chi2red = chi2/dof

	; determine a,i,Omega,omega (using AIWW.PRO)

	aiWw, A_cap,B_cap,F_cap,G_cap,major,inc,W_cap,w_low

;	print, "a i Omega omega:", major, (180/!dpi)*inc, $
;	(180/!dpi)*W_cap, (180/!dpi)*w_low

	EL = [period,Tperi,ecc,major,inc,W_cap,w_low]

	; calculate mass of system using distance of 140 pc
	; parallax = 7.14 mas
	
	mass = major^3/parallax^3/(yd*period)^2

	massi(j) = mass
	a_arr(j) = major
	e_arr(j) = ecc
	i_arr(j) = 180*inc/!dpi
	P_arr(j) = period
	T_arr(j) = Tperi
	W_cap_arr(j) = 180*W_cap/!dpi
	w_low_arr(j) = 180*w_low/!dpi
	chi_arr(j) = chi2
;	chi_arr(j) = xchi

;        chi2 = xchi

	printf,2,EL(0),El(1),EL(2),EL(3),180*EL(4)/!dpi,180*EL(5)/!dpi, $
		180*EL(6)/!dpi, mass, chi2

	if (chi2 le (chi2best + 1d)) then begin

		if (period lt Psigmin) then Psigmin = period
		if (period gt Psigmax) then Psigmax = period
		if (Tperi lt Tsigmin) then Tsigmin = Tperi
		if (Tperi gt Tsigmax) then Tsigmax = Tperi
		if (ecc lt esigmin) then esigmin = ecc
		if (ecc gt esigmax) then esigmax = ecc
		if (major lt asigmin) then asigmin = major
		if (major gt asigmax) then asigmax = major
		if (inc lt isigmin) then isigmin = inc
		if (inc gt isigmax) then isigmax = inc
		if (W_cap lt Wcsigmin) then Wcsigmin = W_cap
		if (W_cap gt Wcsigmax) then Wcsigmax = W_cap
		if (w_low lt wlsigmin) then wlsigmin = w_low
		if (w_low gt wlsigmax) then wlsigmax = w_low
		if (mass lt Msigmin) then Msigmin = mass
		if (mass gt Msigmax) then Msigmax = mass
	endif

        ; Compute minimum and maximum projected separations
        ans_sep = 'n'
        if (ans_sep eq 'y') then begin
           tnum = 1001.0
           tarr = findgen(tnum)/(tnum-1.0)* period + Tperi
           calc_vbfit, EL, tarr, pafit, sepfit
           if (j eq 0) then begin
              sepmin_min = min(sepfit)
              sepmin_max = min(sepfit)
              sepmax_min = max(sepfit)
              sepmax_max = max(sepfit)
           endif
           if (min(sepfit) lt sepmin_min) then sepmin_min = min(sepfit)
           if (min(sepfit) gt sepmin_max) then sepmin_max = min(sepfit)
           if (max(sepfit) lt sepmax_min) then sepmax_min = max(sepfit)
           if (max(sepfit) gt sepmax_max) then sepmax_max = max(sepfit)
        endif

endfor

close,1
close,2

ibest = ibest*180.0/!dpi
isigmin = isigmin*180.0/!dpi
isigmax = isigmax*180.0/!dpi
Wcbest = Wcbest*180.0/!dpi
Wcsigmin = Wcsigmin*180.0/!dpi
Wcsigmax = Wcsigmax*180.0/!dpi
wlbest = wlbest*180.0/!dpi
wlsigmin = wlsigmin*180.0/!dpi
wlsigmax = wlsigmax*180.0/!dpi 

print,"Error estimates: min max mean median stddev"
print,format='(%"M:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(massi),max(massi),mean(massi),median(massi),stddev(massi)
print,format='(%"P:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(P_arr),max(P_arr),mean(P_arr),median(P_arr),stddev(P_arr)
print,format='(%"a:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(a_arr),max(a_arr),mean(a_arr),median(a_arr),stddev(a_arr)
print,format='(%"e:",f10.6,f10.6,f10.6,f10.6,f10.6)', $
	min(e_arr),max(e_arr),mean(e_arr),median(e_arr),stddev(e_arr)
print,format='(%"i:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(i_arr),max(i_arr),mean(i_arr),median(i_arr),stddev(i_arr)
if (ansyd eq 'y') then begin
print,format='(%"T:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(T_arr),max(T_arr),mean(T_arr),median(T_arr),stddev(T_arr)
endif
if (ansyd eq 'd') then begin
print,format='(%"T:",f10.3,f10.3,f10.3,f10.3,f10.3)', $
	min(T_arr),max(T_arr),mean(T_arr),median(T_arr),stddev(T_arr)
endif
print,format='(%"W:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(W_cap_arr),max(W_cap_arr),mean(W_cap_arr), $
	median(W_cap_arr),stddev(W_cap_arr)
print,format='(%"w:",f10.4,f10.4,f10.4,f10.4,f10.4)', $
	min(w_low_arr),max(w_low_arr),mean(w_low_arr), $
	median(w_low_arr),stddev(w_low_arr)

; Determine mass at minchi2

min_ind = where(chi_arr eq min(chi_arr))

print,'Mass as chi2 minimum:', massi(min_ind)

print,'1-sigma uncertainty intervals:'
print,format='(%"M:",f10.4,%" + ",f10.4,%" - ",f10.4)', $
	Mbest,Msigmax-Mbest,Mbest-Msigmin
print,format='(%"P:",f10.4,%" + ",f10.4,%" - ",f10.4)', $
	Pbest,Psigmax-Pbest,Pbest-Psigmin
print,format='(%"a:",f10.4,%" + ",f10.4,%" - ",f10.4)', $
	abest,asigmax-abest,abest-asigmin
print,format='(%"e:",f10.6,%" + ",f10.6,%" - ",f10.6)', $
	ebest,esigmax-ebest,ebest-esigmin
print,format='(%"i:",f10.4,%" + ",f10.4,%" - ",f10.4)', $
	ibest,isigmax-ibest,ibest-isigmin
if (ansyd eq 'y') then begin
print,format='(%"T:",f10.4,%" + ",f10.4,%" - ",f10.4)', $
	Tbest,Tsigmax-Tbest,Tbest-Tsigmin
endif
if (ansyd eq 'd') then begin
print,format='(%"T:",f10.3,%" + ",f10.3,%" - ",f10.3)', $
	Tbest,Tsigmax-Tbest,Tbest-Tsigmin
endif
print,format='(%"W:",f10.4,%" + ",f10.4,%" - ",f10.4)', $
	Wcbest,Wcsigmax-Wcbest,Wcbest-Wcsigmin
print,format='(%"w:",f10.4,%" + ",f10.4,%" - ",f10.4)', $
	wlbest,wlsigmax-wlbest,wlbest-wlsigmin

print,'Distances of closest and farthest approaches:'
rmin_best = abest*(1.0 - ebest)
rmax_best = abest*(1.0 + ebest)
rmin = a_arr*(1.0 - e_arr)
rmax = a_arr*(1.0 + e_arr)
print,'rmin: ',rmin_best,' + ',max(rmin) - rmin_best,' - ',rmin_best - min(rmin)
print,'rmax: ',rmax_best,' + ',max(rmax) - rmax_best,' - ',rmax_best - min(rmax)

imin = where(rmax eq min(rmax))
imax = where(rmax eq max(rmax))

print,'Orbpar for smallest orbit:'
print,P_arr(imin),T_arr(imin),e_arr(imin),a_arr(imin),i_arr(imin),W_cap_arr(imin),w_low_arr(imin)
print,'Orbpar for biggest orbit:'
print,P_arr(imax),T_arr(imax),e_arr(imax),a_arr(imax),i_arr(imax),W_cap_arr(imax),w_low_arr(imax)

if (ans_sep eq 'y') then begin
   print,'Range of minimum separations: ',sepmin_min,sepmin_max
   print,'Range of maximum separations: ',sepmax_min,sepmax_max
endif

print,'Min chi2:',min(chi_arr)
print,'Max chi2:',max(chi_arr)

chimin = 0.0
chimax = 0.0
print,'Enter min max of chi range for plot:'
read,chimin,chimax

Mmin = 0.0
Mmax = 0.0

print,'Enter min max of M range for plot:'
read,Mmin,Mmax

!P.MULTI = [0,4,2]
;Save plots to file temp.ps
mydevice = !d.name
set_plot, 'ps'

device, filename='temp.ps',xsize=18,ysize=13

plot,P_arr,chi_arr,psym=3,xtitle='P (yr)',ytitle='Chi^2',yrange=[chimin,chimax]
plot,T_arr,chi_arr,psym=3,xtitle='T',ytitle='Chi^2',yrange=[chimin,chimax]
plot,e_arr,chi_arr,psym=3,xtitle='e',ytitle='Chi^2',yrange=[chimin,chimax]
plot,a_arr,chi_arr,psym=3,xtitle='a (mas)',ytitle='Chi^2', $
	yrange=[chimin,chimax]
plot,i_arr,chi_arr,psym=3,xtitle='i (deg)',ytitle='Chi^2', $
	yrange=[chimin,chimax]
plot,W_cap_arr,chi_arr,psym=3,xtitle='Omega (deg)',ytitle='Chi^2', $
	yrange=[chimin,chimax]
plot,w_low_arr,chi_arr,psym=3,xtitle='omega (deg)',ytitle='Chi^2', $
	yrange=[chimin,chimax]
plot,massi,chi_arr,psym=3,xtitle='Mass (Msun)',ytitle='Chi^2', $
	yrange=[chimin,chimax],xrange=[Mmin,Mmax]

device, /close
!P.MULTI = [0,0,0]

set_plot,mydevice

end

;
;
;

pro create_histo, xmin, xrange, xstep, data, xbins, h_xarr

; INPUT
; xmid: best fit solution for separation/flux fraction
; xrange: search range   (xmid-xrange:xmid+xrange)
; xstep: step size through search range
; data: array of solutions found in grid that are within 1 chi^2 of best fit
; OUTPUT
; xbins: array of all searched separations/flux fractions
; h_xarr: array to contain the number of values at each step

xnum = round(xrange/double(xstep)) + 2 ; number of separation steps searched

;xbins = dblarr(xnum)	; array of all possible separations searched
;h_xarr = intarr(xnum)	; array to contain the number of values at each step

x = xmin

for i=0, xnum-1 do begin

	xbins(i) = x
	x = x + xstep

endfor

datanum = n_elements(data)

for i=long(0), datanum-1 do begin

	for j=long(0), xnum-1 do begin

	if (round(data(i)/xstep)*xstep eq round(xbins(j)/xstep)*xstep) $ 
			then h_xarr(j) = h_xarr(j) + 1

	endfor
endfor

end

;
;
;

pro linfith, X, Y, z, sigz, c, d

; Least squares linear fit to the equation: z = cX + dY
; where z = (x,y), c = (A,B), and d = (F,G)
; sigz = errors in z=(x,y) ... (for a weighted least squares fit)
; The user supplies the X,Y,z arrays and errors in z
; Program returns values for c,d
; The least squares fit is performed in the same manner as Harthopf's code
; Minimize chi squared with respect to A,B,F,G (set partial derivatives to 0):
; 	chi^2 = sum[(x - AX - FY)^2/sigx^2 + (y - BX - GY)^2/sigy^2]
; Because x is only a function of (A,F), and y is only a funtion of (B,G),
; minimizing chi squared can proceed independently for each of the x,y 
; variables.

num_data = n_elements(z)	;number of data points
ind_data = num_data -1


c = (total(X*Y/sigz^2)*total(z*Y/sigz^2) - $
	total(z*X/sigz^2)*total(Y*Y/sigz^2))/ $
	(total(X*Y/sigz^2)^2 - total(X*X/sigz^2)*total(Y*Y/sigz^2))

d = (total(X*Y/sigz^2)*total(z*X/sigz^2) - $
	total(z*Y/sigz^2)*total(X*X/sigz^2))/ $
	(total(X*Y/sigz^2)^2 - total(X*X/sigz^2)*total(Y*Y/sigz^2))

;print, "e^2:", e2new

end

;
;
;

pro solve_trans, e, M, EE

; Solve transcendental equation of the form E - esinE = M.
; Use iterative procedure to determine E.
; Initial approximation: E_0 = M + esinM - e^2/2 sin(2M)
; Improve solution by iterating the following to formulae:
;	M_0 = E_0 - esin(E_0)
;	E_1 = E_0 + (M - M_0)/(1 - ecos(E_0))
;	(derivative of Kepler's equation)
;
; Method adapted from Heintz 1978 (p.34-35)
; Results compared with point-&-click graphical method.  Iterative approach 
; leads to exact solution that satisfies E - esinE = M.  Therefore, 
; point-&-click method is subsequently removed from orbit fitting.
;
; INPUT:
;	e: eccentricity
;	M: mean anomaly   M= 2*Pi/P
;
; OUTPUT:
;	EE: eccentric anomaly
;
; Created: 9 May 2002

; Initial approximation:

EE = M + e*sin(M) + e^2/2*sin(2*M)

;print, "Initial approximation for E:", EE

EEi = 0d	; parameter to hold initial value once enter while loop

count = 0

while (abs(EE - EEi) gt 0.000001) do begin

	EEi = EE

	Mi = EEi - e*sin(EEi)

	EE = EEi + (M - Mi)/(1 - e*cos(EEi))

	count=count+1

endwhile

;print, "Final iterated E:", EE

;print,"Number of iterations:",count

end

;
;
;

pro aiWw, A_cap, B_cap, F_cap, G_cap, true_a, inc, Omega_cap, omega_low

; Determine a,i,Omega,omega from Thiele-Innes elements A,B,F,G
; INPUT: A,B,F,G
; OUTPUT: a,i,Omega,omega

BmF = (B_cap - F_cap)
BpF = (B_cap + F_cap)
AmG = (A_cap - G_cap)
ApG = (A_cap + G_cap)

; Determine sign of (Omega + omega) and (Omega - omega)
;print, "Check sign of (Omega + omega) and (Omega - omega)"

sinOpo = BmF		; sin(Omega + omega) is propotional to (B-F)
cosOpo = ApG		; cos(Omega + omega) is propotional to (A+G)

sinOmo = BpF		; sin(Omega - omega) is propotional to (B+F)
cosOmo = AmG		; cos(Omega - omega) is propotional to (A-G)

Opo = atan((BmF)/(ApG))
Omo = atan((BpF)/(AmG))

;print,"Check sign of Omega +/- omega"
;print,"O+o:", Opo
;print,"O-o:", Omo

if ((sinOpo/abs(sinOpo)) ne (sin(Opo)/abs(sin(Opo)))) then Opo = !dpi+Opo
if ((sinOmo/abs(sinOmo)) ne (sin(Omo)/abs(sin(Omo)))) then Omo = !dpi+Omo

;print,"Corrected O+o:", Opo
;print,"Corrected O-o:", Omo

Omega_cap = (Opo + Omo)/2d

;print,"Check if 0 < Omega < pi"
;print,"Omega:", Omega_cap 

if (Omega_cap gt !dpi) then omega_cap = omega_cap - !dpi
if (Omega_cap lt 0d) then omega_cap = omega_cap + !dpi

;print,"Corrected Omega:", Omega_cap 

omega_low = Opo - Omega_cap

; keep omega_low between 0 and 2Pi

if (omega_low lt 0.0) then omega_low = omega_low + 2*!dpi

inc = 2* atan(sqrt((BpF*sin(Opo))/(BmF*sin(Omo))))

; keep inc between 0 and 2Pi

if (inc lt 0.0) then inc = inc + 2*!dpi

true_a = sqrt((A_cap*G_cap - B_cap*F_cap)/cos(inc))

end

;
;
;

pro calc_Ei, time, period, Tperi, ecc, Ei

; Determine the eccentric anomalies Ei: 
;     mu(ti - T) = Ei - esin(Ei)  ... Kepler's Equation
;     where mu = 360/P = 2*pi/P
; Solve this transcendental equation through an 
; iterative procedure.
; Use SOLVE_TRANS.PRO

; array to hold Ei's (the eccentric anomaly)

num = n_elements(time)

Ei = dblarr(num) 

mu = 2*!dpi/period

for i=0, num-1 do begin

	; Mi: mean anomoly - increases uniformly with time
	; zero at time Tperi, 2Pi each orbit

	Mi = mu*(time(i) - Tperi) 

	; reduce to same epoch

	Mi = 2*!dpi*((time(i) - Tperi)/period $	
		- fix((time(i) - Tperi)/period))

	; keep Mi between 0 and 2Pi

	if (Mi lt 0.0) then Mi = Mi + 2*!dpi

	solve_trans,ecc,Mi,Eit

	; keep Ei between 0 and 2Pi

	if (Eit lt 0.0) then Eit = Eit + 2*!dpi

	Ei(i) = Eit

endfor

end

;
;
;

pro calc_AFBG, time, xarr, dx, yarr, dy, period, Tperi, ecc, $
	       A_cap, F_cap, B_cap, G_cap, xfit, yfit

; Calculate the Thiele-Innes elements

; Determine the eccentric anomalies Ei: 
; (Ei: array to hold the eccentric anomaly)

calc_Ei, time, period, Tperi, ecc, Ei
	
; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)

; Xi and Yi are both dblarr(num)

Xi = cos(Ei) - ecc
Yi = sqrt(1 - ecc^2)*sin(Ei)

; print,"Xi:",Xi
; print,"Yi:",Yi

; Four Thiele-Innes elements determined from least 
; squares solution of equations:
;	xi = AXi + FYi
;	yi = BXi + GYi
; Calculate geometric elements a("),i,Omega,omega 
; from (A,F,B,G)

linfith, Xi,Yi,xarr,dx,A_cap,F_cap
linfith, Xi,Yi,yarr,dy,B_cap,G_cap

; Determine fit points

xfit = A_cap*Xi + F_cap*Yi	;fitted points
yfit = B_cap*Xi + G_cap*Yi

end
