pro test_chi, data_file, num

; This is a copy of HARTKOPF_E_V1.PRO that uses a variation of 1 in
; chi2, rather than in the reduced chi2, to compute errors.
;
; Compute the orbital elements of a binary system, using the least squares
; technique described by Hartkopf, McAlister, Franz 1989, AJ, 98, 1014
;
; Procedure: Given (P,T,e) and a set of observations (ti,xi,yi), the eccentric
; anomalies Ei can be found from: 
;	u(ti - T) = Ei - esin(Ei)
;	where u = 360/P
; Normalized rectangular coordinates Xi & Yi:
;	Xi = cos(Ei) - e
;	Yi = sqrt(1 - e^2)*sin(Ei)
; Four Thiele-Innes elements determined from least squares solution of 
; equations:
;	xi = AXi + FYi
;	yi = BXi + GYi
; Calculate geometric elements a("),i,Omega,omega from (A,F,B,G)
; 
; Paramers:
;	P:	period
;	T: 	time of periastron passage
;	e:	eccentricity
;	a:	semi-major axis (in mas)
;	i:	inclination
;	Omega:	position angle of node
;	omega:	angle between node and periastron
;	ti:	time of observation
;	xi:	RA
;	yi:	DEC
;
; Note: xi and yi are not the same convention for RA and DEC chosen to make 
; the orbital plots. Specifically, in fitting the apparent ellipse 
; (ORBFITE.PRO):
;	x = RA = -rho*sin(theta),   y = DEC = rho*cos(theta)
;	(i.e. North is in the direction of the positive y axis, theta is
;	measured counterclockwise from positive y-axis)
; Whereas in this program, the coordinate system set up by Couteau,
;	x = rho*cos(theta),   y = rho*sin(theta)
;	(i.e. North id in the direction of the positive x-axis and the 
;	y-axis points east)
; Both coordinate systems have the same separation and position angle 
; orientations, they are just rotated by 90 deg with respect to each other.
;
; INPUT: 
;	data_file: file containing data points in the format:
;		      time  theta  dtheta rho drho 
;		      where theta is the position angle and rho the separation
;		      dtheta and drho are the errors
;		      (enter file name using quotes)
;	num: number of data points contained in data_file
;
; Prompted inputs:
;	period (P)
;	time of periastron passage (T)
;	eccentricity (e)
;       epoch: search for T within P/2 of epoch
;
; OUTPUT: 
;       result_chi:  file containing all solutions within Delta(chi2)=1
;                    format: P T e chi2
;
;       xyresiduals: file containing (Observed-Calculated) residuals
;                    format:
;                    date     dx       xO-C   xOC/dx dy      yO-C   yOC/dy
;                    where dx and dy are the measurement uncertainties
;                          xO-C and yO-C are the (data - fit) values
;                          xOC/dx and yOC/dy are the ratio of the two
;
; Calls the following routines:
;     calc_AFBG.pro
;     calc_Ei.pro
;     solve_trans.pro
;     linfith.pro
;     aiWw.pro
;     calc_deriv_vb.pro
;     gaussj.pro
;
; Began: May 2002
;
; 31 August 2011: Move subroutines at end to separate files

!P.font=0
!P.multi=0
!P.charsize=1.2
!P.charthick=2.0
!P.thick=2.0
!X.thick=2.0
!Y.thick=2.0
frac=1.2	;scale factor for adjusting size of plotting symbols

; Set color table
;  red=1, green=2, blue=3

tvlct, [255,0,0], [0,255,0], [0,0,255], 1

; Read in data points from data_file

temp1 = 0d
temp2 = 0d
temp3 = 0d	; temporary variables to read data from file
temp4 = 0d
temp5 = 0d

time = dblarr(num)
theta = dblarr(num)
rho = dblarr(num)
dtheta = dblarr(num)
drho = dblarr(num)

openr,lun,data_file,/get_lun

for i=0, num-1 do begin

	readf, lun, temp1, temp2, temp3, temp4, temp5

	time(i) = temp1
	theta(i) = temp2
	dtheta(i) = temp3
	rho(i) = temp4	
	drho(i) = temp5
	
endfor

close,lun

; convert theta and dtheta to radians

theta = theta*!dpi/180
dtheta = dtheta*!dpi/180

; convert data points to x and y coordinates

xarr = rho * cos(theta)	; x coordinate
yarr = rho * sin(theta)	; y coordinate

; propagate errors in sep and PA to x and y:

dx = sqrt(cos(theta)^2*drho^2 + rho^2*sin(theta)^2*dtheta^2)
dy = sqrt(sin(theta)^2*drho^2 + rho^2*cos(theta)^2*dtheta^2)

; Obtain initial values for P,T,e

P0 = 0d
T0 = 0d
e0 = 0d

print,"Enter P T e (period, time of periastron passage, eccentricity):"
read, P0, T0, e0

; only search for orbital solutions where the time of periaston
; passage is within a cycle of the period from the current
; epoch (ex. 2000)

epoch=0d

print,' '
print,'This program will only search for solutions of the time of'
print,'periastron passage within +/- P/2 of the chosen epoch.'

print,'Enter epoch of periastron passage:'
read, epoch

; read in step sizes and searching ranges for P, T, e

Pstep = 0d
minP = 0d
maxP = 0d
Tstep = 0d
minT = 0d
maxT = 0d
estep = 0d
mine = 0d
maxe = 0d

print," "
print,"Determine searching ranges and step sizes for P,T,e."
print,"step is the incremental size for which to vary given parameter"
print,"min is the lower value for the searching range, ie: Pi = P-minP"
print,"max is the higher value for the searching range, ie: Pf = P+maxP"
print," "
print, "Enter Pstep size, minP, maxP for search in P:"
read, Pstep, minP, maxP
print, "Enter Tstep size, minT, maxT for search in T:"
read, Tstep, minT, maxT
print, "Enter estep size, mine, maxe for search in e:"
read, estep, mine, maxe

; open file "result_chi" to save chi squared values
openw,lun,"result_chi"

; degrees of freedom
dof = 2d*num - 7d

; Create initial model:
			
	period = P0
	Tperi = T0
	ecc =e0

	; Determine fitted points (xfit,yfit) by computing the Thiele Innes
	; elements A,F,B,G

	calc_AFBG, time, xarr, dx, yarr, dy, period, Tperi, ecc, $
		   A_cap, F_cap, B_cap, G_cap, xfit, yfit

	; Determine chi squared

	chi2 = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

	; reduced chi squared:
	chi2red = chi2/dof

	chimin = chi2
	chiredmin = chi2red
	bestpar = [period,Tperi,ecc,chi2]
	bestABFG = [A_cap,B_cap,F_cap,G_cap]
	xbest = xfit
	ybest = yfit

; begin modifying P,T,e

;(the 0.000001d added to the upper limit is to account for errors in precision)

minT0 = minT
maxT0 = maxT

for period = (P0 - minP) , (P0 + maxP + 0.000001d), Pstep do begin

  ; only search for orbital solutions where the time of periaston
  ; passage is within a cycle of the period from the current
  ; epoch (ex. 2000)

   minTP = period/2.0
   maxTP = period/2.0

   minT = min([minTP,minT0])
   maxT = min([maxTP,maxT0])

   for Tperi = (T0 - minT) , (T0 + maxT + 0.000001d), Tstep do begin
      
      for ecc = (e0 - mine), (e0 + maxe + 0.000001d), estep do begin

	 ; Calculate Thiele-Innes elements and fitted points
	 ; (A,F,B,G;xfit,yfit)	

         calc_AFBG, time, xarr, dx, yarr, dy, $
                    period, Tperi, ecc, $
                    A_cap, F_cap, B_cap, G_cap, xfit, yfit
	
	 ; Determine chi squared

; 	 chi2 = total((xarr-xfit)^2 + (yarr-yfit)^2)
         chi2 = total((xarr-xfit)^2/dx^2 + (yarr-yfit)^2/dy^2)

	 ; reduced chi squared:
         chi2red = chi2/dof

;	 print, "chi2", chi2

         if (chi2 le chimin) then begin
            chimin = chi2
            chiredmin = chi2red
            bestpar = [period,Tperi,ecc,chi2]
            print,"Best fit P T e chi^2"
            print, bestpar
            printf,lun, bestpar
            bestABFG = [A_cap,B_cap,F_cap,G_cap]
            xbest = xfit
            ybest = yfit

	    ;plot data and fitted points

            plot, xarr, yarr, psym=4, XSTYLE=1, YSTYLE=1, $
                  ISOTROPIC=1
;	          ISOTROPIC=1, xrange=[-50,+150], $
;		  yrange=[-150,+50]
            oplot, xfit, yfit, psym=5 

            wait,0.02

         endif else begin
	
            if (chi2 le (chimin + 1d)) then begin
               print,"P T e chi^2"
               print,period,Tperi,ecc,chi2
               printf,lun,period,Tperi,ecc,chi2
		
	       ;plot data and fitted points

;  	       plot, xarr, yarr, psym=4, XSTYLE=1, YSTYLE=1, $
;		   ISOTROPIC=1, xrange=[-50,+150], $
;		   yrange=[-150,+50]
;	       oplot, xfit, yfit, psym=5 

;	       wait,0.2
            endif
         endelse
      endfor
   endfor
endfor

close,lun

print,"Best fit P T e chi^2"
print, bestpar

print,"reduced chi2:", chiredmin

print, "A B F G:", bestABFG

period=bestpar(0)
Tperi = bestpar(1)
ecc = bestpar(2)
A_cap = bestABFG(0)
B_cap = bestABFG(1)
F_cap = bestABFG(2)
G_cap = bestABFG(3)

xfit = xbest
yfit = ybest

; determine a,i,Omega,omega (using AIWW.PRO)

aiWw, A_cap,B_cap,F_cap,G_cap,major,inc,W_cap,w_low

print, "a i Omega omega:", major, (180/!dpi)*inc, $
(180/!dpi)*W_cap, (180/!dpi)*w_low

EL = [period,Tperi,ecc,major,inc,W_cap,w_low]

; Determine errors in orbital elements
; set up matrices for error determination
; Invert matrix through Gauss Jordan elimination (Numerical Recipes in C)

; set up covariance matrix and column matrix
alpha = dblarr(7,7)
beta = dblarr(7)
;For each element, enter 0 to hold element fixed, 1 to vary
elfix = [1.0,1.0,1.0,1.0,1.0,1.0,1.0]
indfix = where(elfix eq 1.0, mfit)

;calc_err, EL, time, theta, rho, dtheta, drho, alpha, beta
calc_deriv_vb, EL, elfix, mfit, time, theta, rho, dtheta, drho, theta_f, rho_f, alpha, beta

;print,"covariance matrix and right-hand side vector:"
;print,"alpha"
;print,alpha
;print,"beta"
;print,beta	

invmat = invert(alpha, stat, /double)

;print,"stat (0 successful):",stat

;print,"invmat"
;print,invmat

;determine errors:

ELerr = dblarr(7)

for i=0,6 do ELerr(i) = sqrt(chiredmin)*sqrt(invmat(i,i))

print, "P T e a i Omega omega"
print,EL(0),El(1),EL(2),EL(3),180*EL(4)/!dpi,180*EL(5)/!dpi,180*EL(6)/!dpi
print, "Errors determined from covariance matrix:"
print,Elerr(0),Elerr(1),ELerr(2),ELerr(3),180*ELerr(4)/!dpi, $
	180*ELerr(5)/!dpi,180*ELerr(6)/!dpi

; perform gauss-jordan elimination to invert alpha and find solution vector B

gaussj, alpha, beta

;print,"after gauss-jordan elimination:"
;print,"alpha"
;print,alpha
;print,"beta"
;print,beta	

; print O-C residuals (Observed - Calculated)

xOC = xarr - xfit
yOC = yarr - yfit

openw,lun,'xyresiduals'
printf,lun,"O-C residuals (data - fit)"
printf,lun,"date     dx       xO-C   xOC/dx dy      yO-C   yOC/dy"
;printf,lun,"date   dx   xdata - xfit   xOC/dx   dy   ydata - yfit   yOC/dy"
for i=0, num-1 do printf,lun,format='(f10.4,f7.3,f8.3,f7.3,f7.3,f8.3,f7.3)', $
	time(i), dx(i), xOC(i), xOC(i)/dx(i), dy(i), yOC(i), yOC(i)/dy(i)
close,lun

;for i=0, num-1 do print, "x",xarr(i)," dx",dx(i)," y",yarr(i)," dy",dy(i)

;plot data and fitted points

;print, "Do you want to save plot as a postscript file?  (Type yes or no):"
;answer = ""
;read,answer
;; Set plotting device to ps if answer is yes
;if (answer eq "yes") then begin
;	mydevice=!d.name
;	set_plot, 'ps'
;	device, /Color, Bits_per_pixel=8, filename='temp.ps' 
;endif
;
;plot, xarr, yarr, psym=4, XSTYLE=1, YSTYLE=1, ISOTROPIC=1, $
;	xrange=[-50,+150], yrange=[-150,+50]
;oplot, xfit, yfit, psym=5 
;
;; set plotting device back to terminal if previously saving to ps file
;if (answer eq "yes") then begin 
;	device, /close
;	set_plot, mydevice
;endif
 

; plot best-fit orbit and data

tnum=1000.0
tstep = period/tnum
tmin = Tperi

tarr = findgen(tnum)*tstep + tmin

calc_vbfit, EL, tarr, theta_mod, rho_mod

xmod = rho_mod * cos(theta_mod)	; x coordinate
ymod = rho_mod * sin(theta_mod)	; y coordinate

; Define user symbol to be closed circle
; Make a vector of 16 points, Z[i] = 2pi/16:
Z = findgen(17) * (!pi*2/16.)

; Plot measured positions as filled circles

usersym, frac*cos(Z), frac*sin(Z),/fill

sepmax = 1.2*max(abs([yarr,xarr]))

set_plot, 'ps'
device,/Color,Bits_per_pixel=8,filename='temp_vb.eps',xsize=12,ysize=12,isolatin1=1,/encap

ploterror,yarr,xarr,dy,dx,xtitle='!9D!3RA (mas)',ytitle='!9D!3DEC (mas)',ISOTROPIC=1, $
     xrange=[sepmax,-sepmax], yrange=[-sepmax,sepmax], $
     xstyle=1,ystyle=1, /nodata, position=[0.19,0.12,0.95,0.98]

; Connected measured and fit values
for i=0,num-1 do oplot,[yfit(i),yarr(i)],[xfit(i),xarr(i)]

; Mark computed location with x
;oplot,[yfit],[xfit],psym=7

                            ; mark primary with an asterisk
oplot,[0],[0],psym=2
oplot,[-sepmax,sepmax],[0,0],linestyle=1
oplot,[0,0],[-sepmax,sepmax],linestyle=1

; plot best-fit orbit
oplot,ymod,xmod,color=3

; Plot measured positions
;Define the symbol to be a closed circle
usersym, frac*cos(Z), frac*sin(Z), /fill
oplot,yarr,xarr,psym=8

device,/close
set_plot, 'x'


free_lun,lun

end
