;; $Id: obj_FSU.pro,v 1.1 2007/10/09 09:44:05 rkoehler Exp $
;
; obj_FSU.pro
; Created:     Tue Sep 18 16:57:12 2007 by Rkoehler@lx40
; Last change: Fri Sep 24 15:35:23 2010
;
; PURPOSE:
;	treat FSUs as objects.
;
; WARNING:
;	This program may contain strong violence, adult content, and
;	language.  Parental guidance is advised.
;
; COMMON BLOCKS:
;
; PACSIM_DEFAULTS
;	kappa_center	double	1/m	wavenumber	1D-array
;	kappa_width	double	1/m	bandwidth of wavenumber channels
;
;;;;;;

PRO FSU__define
f= { FSU, $
     conf: ptr_new(), $
     seed: ptr_new(), $
     A_d_phi: ptr_new(),$
     A_wn_err: ptr_new(),$
     A_Intensity: ptr_new(),$
     A_Intensity_u: ptr_new(),$
     A_A0:	  ptr_new(),$
     A_A12:	  ptr_new(),$
     A_skyback:   ptr_new(),$
     A_ABCDphase: ptr_new(),$
     A_disp:      ptr_new(),$
     A_stepphase: ptr_new(),$
     no_comma: 0 $
   }
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function Default_FSU_config

  common PACSIM_DEFAULTS	;; for No. of kappas

  n_wave= 6	;;N_elements(kappa_center)

  return,{$
         time        :	0.1,	$ ; double s	raw integration time
         steptime    :	1.,   	$ ; double s	stepping time
         tel_diam    :	1.8,  	$ ; double m	telescope diameter
         star_angdiam:	0.05,  	$ ; double mas	star angular diameter
         star_temp   :	6000.,	$ ; double K	star temperature
         throughput  :	0.01, 	$ ; double -	instrumental throughput
         visi        :	0.7,  	$ ; double -	visibility
         pherrAVG    :	0.0,  	$ ; double deg	tracking phase error per stepping times [1 sigma]
         pherrjitter :	0.0,  	$ ; double deg	tracking jitter [1 sigma]
         d_g0        :	0.0, 	$ ; double -	delta g0  [1 sigma]
         d_g12       :	0.0, 	$ ; double -	delta g12 [1 sigma]
         d_phi       :	0.0, 	$ ; double deg	delta phi [1 sigma]
         n_phonoise  :	0, 	$ ; int    -	photon noise
         steppingnum:	1,	$
         steppingmin:	0.,	$
         steppingmax:	0.,	$
         refphase:	0 ,	$
         gain:		8.9d,	$
         n_seed:	0,	$ ;; set to 1 or 2 to use pre-defined seeds to reproduce results
         n_skyback:	0,	$
         skyback:	dblarr(n_wave),$	;; sky background in photons (not ADUs!)
         disp:		dblarr(n_wave),$
         relWE:		0.,	$		;; relative wave error
         readoutnoise:	0.	$
         }
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function FSU::init, config, SEED=seed

  if N_elements(config) gt 0 then self->configure,config

  if not Keyword_set(seed) then seed= systime(/seconds)
  self.seed= ptr_new(seed,/no_copy)

  return,1
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO FSU::cleanup
  ptr_free, self.conf
  ptr_free, self.seed
  ptr_free, self.A_d_phi
  ptr_free, self.A_wn_err
  ptr_free, self.A_Intensity
  ptr_free, self.A_Intensity_u
  ptr_free, self.A_A0
  ptr_free, self.A_A12
  ptr_free, self.A_skyback
  ptr_free, self.A_ABCDphase
  ptr_free, self.A_disp
  ptr_free, self.A_stepphase
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO FSU::configure, config

  common PACSIM_DEFAULTS	;; for No. of kappas

  ;;to turn off stepping, reference is set to 0
  ;;but one can add a reference phase to this ('refphase_in')
  if config.steppingNUM EQ 1 then begin
      config.steppingMin = 0.0
      config.steppingMax = 0.0
  endif

  ptr_free, self.conf
  self.conf= ptr_new(config)

  ;initilize seeds for random numbers (to reproduce results)
  ;1=FSUA, 2=FSUB
  IF config.n_seed EQ 1 THEN BEGIN
      seed_1 = 2.0              ;d_g0
      seed_2 = 1.0              ;d_g12
      seed_3 = 8.0              ;d_phi
      seed_4 = 4.0              ;wn_err
  ENDIF
  IF config.n_seed EQ 2 THEN BEGIN
      seed_1 = 5.0              ;d_g0
      seed_2 = 9.0              ;d_g12
      seed_3 = 3.0              ;d_phi
      seed_4 = 7.0              ;wn_err
  ENDIF

  n_wave = 6	;;N_ELEMENTS(kappa_center)	;; No. of wavenumber channels


  ;; normal distributed random g0 error (uncorrel.),
  ;; generated once for all ABCDs and wavenumbers,
  ;; scaled with one sigma input (n_wave * 4 ABCD)
  ;; [not scaled to the wavenumber]

  A_d_g0= config.d_g0 * double(transpose(RANDOMU(seed_1,4,n_wave,/NORMAL)))
  If min(A_d_g0) le -1. OR max(A_d_g0) gt 1. then print,"WARNING: produced |d_g0's| GT 1"
  ;;help, A_d_g0 & print, A_d_g0


  ;; normal distributed random g12 error (correl.),
  ;; generated once for all ABCDs and wavenumbers,
  ;; scaled with one sigma input (n_wave * 4 ABCD)
  ;; [not scaled to the wavenumber]

  A_d_g12= config.d_g12 * double(transpose(RANDOMU(seed_2,4,n_wave,/NORMAL)))
  If min(A_d_g12) le -1 OR max(A_d_g12) gt 1 then print,"WARNING: produced |d_g12's| GT 1"
  ;print, A_d_g12


  ;; normal distributed random phase error delta phi,
  ;; generated once for all ABCDs and wavenumbers,
  ;; scaled with one sigma input (n_wave * 4 ABCD)
  ;; [not scaled to the wavenumber]

  A_d_phi= config.d_phi * double(transpose(RANDOMU(seed_3,4,n_wave,/NORMAL)))
  ;print, A_d_phi

  ;save A_d_phi to calculate delta beta (only used in 'pro_sim_3.pro')
  IF (*self.conf).n_seed EQ 1 THEN save, A_d_phi, FILENAME='delta_phi_ABCD_FSUA.sav'
  IF (*self.conf).n_seed EQ 2 THEN save, A_d_phi, FILENAME='delta_phi_ABCD_FSUB.sav'

  ptr_free,self.A_d_phi
  self.A_d_phi= ptr_new(A_d_phi,/no_copy)


  ;* normal distributed random relative waven error,
  ;* generated at once for all ABCDs and wavenumbers,
  ;* scaled with one sigma input (n_wave * 4 ABCD)
  ;* [not scaled to the wavenumber]

  A_wn_err= config.relWE * double(transpose(RANDOMU(seed_4,4,n_wave,/NORMAL)))
  If min(A_wn_err) LE -1 OR max(A_wn_err) GT 1 then $
    print,"WARNING: produced |rel waven err's| greater than 1"
  ;print,"A_wn_err:" & print,A_wn_err
  ptr_free,self.A_wn_err
  self.A_wn_err= ptr_new(A_wn_err,/no_copy)


  ;; calculate number of ELECTRONS for all wavenumbers and all ABCD
  ;; modified with rel. wavenumber error (if choosen)
  ;; 'throughput' includes instrumental and atmospherical throughput as well as
  ;; the quantum efficiency of the detector (converts photons => electrons)
  ;; - skybackground is added later (because is uncorrelated)
  ;; - conversion from electrons to ADUs is done at the end

  A_Intensity = config.throughput * self->Photonnumber(*self.A_wn_err)
  ;;print,"A_Intensity:" & print, A_Intensity
  ptr_free,self.A_Intensity
  self.A_Intensity= ptr_new(A_Intensity,/no_copy)

  ;; DEACTIVATED: undisturbed number of ELECTRONS
  ;; IF config.n_output GT 0 THEN begin
  ;;     A_wn_err_u = MAKE_ARRAY(n_wave,4,VALUE=0.0,/DOUBLE)
  ;;     A_Intensity_u= config.throughput * self->Photonnumber(A_wn_err_u)
  ;; endif
  ptr_free, self.A_Intensity_u
  self.A_Intensity_u = ptr_new(A_Intensity_u,/no_copy)

  ;; calculate factor A0 (element by element), same for all steps and frames
  ptr_free, self.A_A0
  self.A_A0 = ptr_new( (1.+A_d_g0) * (*self.A_Intensity))

  ;; calculate factor A12 (element by element), same for all steps and frames
  ptr_free, self.A_A12
  self.A_A12= ptr_new( config.visi * (1.+A_d_g12) * (*self.A_Intensity))

  ;print,"Vibisillity: ",(*self.A_A12) / (*self.A_A0) ;;- debug visibility

  ;; Array of skybackgrounds: Function of wavelength, but the same for ABCD
  ptr_free, self.A_skyback
  self.A_skyback= ptr_new( config.skyback # [1.,1.,1.,1.])


  ;; define and expand hardware phase points to all wavenumbers,
  ;; NOT scaled, cos 0,90,180 and 270 are the same for all wavenumbers
  ptr_free, self.A_ABCDphase
  self.A_ABCDphase= ptr_new( replicate(1.0d,n_wave) # [0.,270.,180.,90.])
  ;; original: self.A_ABCDphase= ptr_new( replicate(1.0d,n_wave) # [0.,90.,180.,270.])


  ;; expand differential phase produced by dispersion to all ABCD,
  ;; input disp is already wavenumber dependent.  same for all ABCD
  ptr_free, self.A_disp
  self.A_disp= ptr_new( config.disp # [1.,1.,1.,1.])

end ;; of configure

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function FSU::simulate, AA_finalresult_u

  common PACSIM_DEFAULTS

  if not ptr_valid(self.conf) then begin
      print,"Please configure me first"
      return,0
  endif

  ;;save config values in variables
  time       = (*self.conf).time		;; double s	raw integration time
  steptime   = (*self.conf).steptime		;; double s	time for one step
  stepNUM    = (*self.conf).steppingNUM 	;; int    -    number of steps
  gain       = (*self.conf).gain 		;; double -    gain (electrons=>ADU)
  visi       = (*self.conf).visi 		;; double -    visibility

  ;; deactivated config variables:
  n_output   = 0	;; was: (*self.conf).n_output 		;; int    -    plot option


  datapoints = ROUND(steptime/time)		;; number of datapoints (for one step)
  n_wave     = 6	;;N_Elements(kappa_center)		;; number of wavenumber channels


  ;;create result array and undisturbed result array
  ;;(unit: electrons, at the end converted to ADUs):
  ;;(number of steps)*(datapoints per step)*(# of wavenumbers)*(4 ABCD)

  AA_finalresult   = MAKE_ARRAY(stepNUM,datapoints,n_wave,4,/DOUBLE)
  IF n_output GT 0 THEN $
    AA_finalresult_u= MAKE_ARRAY(stepNUM,datapoints,n_wave,4,/DOUBLE)

  if ptr_valid(self.seed) then seed = *self.seed

  ;######################### BEGIN LOOP (1) over all steps #########################
  FOR step=0, stepNUM-1 DO BEGIN

      self->init_step, step, A_refphase

      ;;undisturbed final result (in vacuum)
      IF n_output GT 0 THEN begin
          A_result_u = (*self.A_Intensity_u) * $
			(1. + visi * cos((!pi/180.)*(A_refphase + (*self.A_ABCDphase))))

          IF (*self.conf).n_skyback EQ 0 THEN $ ;; add sky background (in electrons)
            A_result_u += (*self.A_skyback)

          ;; multiply result by gain to convert from electrons to ADUs
          A_result_u *= gain
      endif

      ;;#################### BEGIN LOOP (2) over all datapoints (per step) ################

      FOR raw_time=0, datapoints-1 DO BEGIN

          A_result = self->simoneframe(seed)

          ;; save final result in a 4D array:
          ;; (number of steps)*(datapoints per step)*(# of wavenumbers)*(4 ABCD)
          ;; sky background was removed/not removed by simoneframe()

          AA_finalresult[step,raw_time,*,*]= A_result

          ;; save undisturbed final result in a 4D array:
          ;; (number of steps)*(datapoints per step)*(# of wavenumbers)*(4 ABCD)
          ;; sky background was added/not added once to A_result_u

          If n_output GT 0 then $
            AA_finalresult_u[step,raw_time,*,*]= A_result_u

          ;;print,"step",step,", frame",raw_time,": seed =",seed

      ENDFOR	;; loop over points in one step
  ENDFOR	;; loop over steps

  ptr_free,self.seed
  self.seed= ptr_new(seed)	;; store seed for next time

  ;If MIN(AA_finalresult)   LE 0 THEN PRINT, "WARNING: negative number of ADUs"
  ;If MIN(AA_finalresult_u) LE 0 THEN PRINT, "WARNING: negative number of ADUs"

  ;; RETURN a 4 dimensional array (unit: ADUs):
  ;; (number of steps)*(datapoints per step)*(# of wavenumbers)*(4 ABCD)
  RETURN, AA_finalresult
END

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO FSU::init_step, step, A_refphase

  Common PACSIM_DEFAULTS

  stepNUM   = (*self.conf).steppingNUM 	;; int    -    number of steps
  stepMin   = (*self.conf).steppingMin 	;; double deg  minimum stepping phase
  stepSize  =((*self.conf).steppingMax-stepMin) / double(stepNUM)  ;; double deg step size

  n_kc = size(kappa_center)
  kappa_c = total( total(kappa_center,3),2) / double(n_kc[2]*n_kc[3])	;; mean over FSU and ABCD

  v_rel_wave= kappa_c / kappa_c[0]	;; create relative wavenumber array

  ;;reference phase (for k(1)) for different steps:
  ;;define and SCALE the ref. phase (k(1)) to other steps and
  ;;wavenumbers, because the delayline is shifted for k(1) [assumption]
  ;;an additional phase can be added here ('refphase_in', e.g. different ones for FSUA/B)
  ;;[scaled to the wavenumber]

  stepPhase = stepMin + step*stepSize + (*self.conf).refphase
  A_refphase= v_rel_wave # replicate(stepPhase,4)

  ;; normal distributed stepping error for every step
  ;; (the same one value for ABCD, but new random for every step)
  ;; and scale stepping error for every step to all wavenumbers
  ;; [scaled to the wavenumber] (error in delayline)

  step_err_distr= (*self.conf).pherrAVG * double(RANDOMU(seed_5,/NORMAL))
  A_step_err = v_rel_wave # replicate(step_err_distr,4)

  ;; sum up 'constant' phases: ABCD, ref, step_err, and dispersion
  ptr_free, self.A_stepphase
  self.A_stepphase = ptr_new( (*self.A_ABCDphase) + A_refphase + A_step_err + (*self.A_disp))
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function FSU::simoneframe, seed

  Common PACSIM_DEFAULTS

  n_kc = size(kappa_center)
  kappa_c = total( total(kappa_center,3),2) / double(n_kc[2]*n_kc[3])	;; mean over FSU and ABCD

  n_wave     = N_ELEMENTS(kappa_c)		;; determine # of wavenumbers
  v_rel_wave = kappa_c / kappa_c[0]	;; create relative wavenumber array


  ;; normal distributed random jitter phase error
  ;; one number, since caused by delay line
  ;; scale phase jitter vector to all wavenumbers

  jitter_dist= (*self.conf).pherrjitter * double(RANDOMU(seed_6,/NORMAL))
  A_jitter = v_rel_wave # replicate(jitter_dist,4)

  ;;SUM UP all phase errors with the stepping reference
  ;;phase the hardware phase point (ABCD) and the dispersion phase
  ;;(without phase error delta phi)
  A_phase = (*self.A_stepphase) + A_jitter

  ;; modify all phase errors duo to relative wavenumber error
  ;; and add phase error delta phi)
  A_phase = (A_phase * (1. + (*self.A_wn_err)) + (*self.A_d_phi)) * (!dpi/180.)

  sflat= lf_flux/max(lf_flux)

  ;;final result without photon noise
  A_result = ((*self.A_A0) + (*self.A_A12)*cos(A_phase)) * sflat $
             + (*self.A_skyback)

  ;; add photon noise (poisson distributed)
  IF (*self.conf).n_phonoise EQ 1 THEN BEGIN
      IF MAX(A_result) gt 1e8 THEN begin
          print,"WARNING: electron number greater then 10^8"
          print," => photon noise maybe not correct"
      endif
      IF MIN(A_result) LE 0   THEN begin
          PRINT,"ERROR:"
          PRINT,"electron number is less/equal 0!"
          PRINT,"=> poisson distrubuted photon noise can not be added"
          PRINT,"=> reduce visibility or switch of the g0 and g12 errors"
          PRINT,"   or try again"
          return, A_result
      endif
      FOR i=0, n_wave-1 DO BEGIN
          FOR j=0, 3 DO BEGIN
              A_result[i,j]= RANDOMU(seed,/DOUBLE,POISSON=A_result[i,j])
          ENDFOR
      ENDFOR
  ENDIF

  ;;* add readout noise (normal distributed) generated at
  ;;* once for all ABCDs and wavenumbers for every datapoint,
  ;;* scaled with one sigma input (n_wave * 4 ABCD)
  ;;* [not scaled to the wavenumber]
  ;;* mean of readout noise is equal 0
  ;;* it is ONLY added for the bright star where (hardware) fringe
  ;;* tracking is performed (FSUB)

  IF (*self.conf).readoutnoise ne 0. THEN $
     A_result += double((*self.conf).readoutnoise * RANDOMU(seed_7,n_wave,4,/NORMAL))

  ;; remove/do not remove sky background
  ;; skyback was added before (optional) photonoise calculation

  IF (*self.conf).n_skyback EQ 1 THEN  A_result -= (*self.A_skyback)

  ;;multiply result by gain to convert from electrons to ADUs
  return, (*self.conf).gain * A_result
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; assimilated from:
;;
;# Name:        fct_photonnumber.pro
;# Version:     8.0
;# Created:     Tuesday, June   26 by Ronny Geisler
;# Last change: Wed Oct 24 15:23:08 2007
;#
;# PURPOSE: calculate number of photons for different
;#          wavenumbers depending on telescope diameter,
;#          star angular diameter and star temperature.
;#          also input relative wavenumber error
;#          (only bk is modified, filter widths are the same)
;#
;# Action items: ---
;#

Function FSU::photonnumber, relWE_array

  common PACSIM_DEFAULTS

  ;;INPUT:
  ;; relWE_array	  double  -     rel.wavenum err 2D-array (n_wave[# of wavenumbers]*4[ABCD])
  ;; config.tel_diam      double  m     telescope diameter,	scalar
  ;; config.star_angdiam  double  mas   star angular diam.,	scalar
  ;; config.star_temp     double  K     star temperature,	scalar
  ;; config.time          double  s	DIT,			scalar
  ;;
  ;; Common Block:
  ;; kappa_center      	  double  1/m	wavenumber,	1D-array (n_wave[# of wavenumbers])
  ;; kappa_width	  double  1/m	bandwidth of spectral channels

  ;;define constants:
  planck = 6.62606930d-34	;Js   Planck constant
  lspeed = 299792450.d0         ;m/s  Speed of Light
  boltz  = 1.38065050d-23       ;J/K  Boltzmann constant

  ;; Bk in [photons/(m*s*rad^2)]
  ;; this is Bk in [energy/(...)] / (h nu) = Bk_energy / (h c wave)

  ;bk    = dblarr( N_Elements(kappa_center), 4)
  bk    = dblarr( 6, 4)		;; 5.1 channels
  temper= (*self.conf).star_temp

  n_kc = size(kappa_center)
  kappa_c = total( total(kappa_center,3),2) / double(n_kc[2]*n_kc[3])	;; mean over FSU and ABCD

  FOR j=0, 3 DO BEGIN           ;ABCD (same wavenumber, but different error)
     kappa = kappa_c * (1. + relWE_array[*,j])	;; in 1/m

     bk[*,j]= 2.0 * lspeed * kappa^2 * kappa_width $
              / (exp((planck*lspeed*kappa) / (boltz*temper)) - 1.0)
  ENDFOR

  tel_area = !dpi * ((*self.conf).tel_diam/2.)^2
  star_area= !dpi * ((*self.conf).star_angdiam*!pi/(180.*60.*60.*1000.)/2.)^2	;; rad^2

  ;number of photons in [photons]
  photonM = (*self.conf).time * tel_area * star_area * bk

  return, photonM ;array: (# of wavenumbers)*(4 ABCD)
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; this is the end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
