;
; astrocorr.pro
; Created:     Thu Apr 16 17:27:43 2009 by Rkoehler@lx40
; Last change: Thu May  7 17:06:34 2009
;
; PURPOSE:
;	none
;
; WARNING:
;	This program may contain strong violence, adult content, and
;	language.  Parental guidance is advised.
;
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Modified Julian Date
;;
;; Note the order of arguments: largest unit first!
;;

Function MJulDay, Year, Month, Day, Hour, Minute, Second

  return, JulDay(Month, Day, Year, Hour, Minute, Second) - 2400000.5d
end

Function MJulDate, MJD, yr,mon,day,hr,min,sec
  caldat, MJD+2400000.5d, mon, day, yr, hr, min, sec
  return, string(Format="(C(CDI,'-',CMoA,'-',CYI,', ',CHI02,':',CMI02,':',CSF06.3))", MJD+2400000.5d)
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Compute LST from date and time
;;
;; MJD0 = Modified Julian Date at 0h UT
;; UT = Universal Time
;;
;; see R. Green: Spherical Astronomy, p.242 and p.31

function lst_from_date, MJD, geolong

  if size(geolong,/tname) eq 'UNDEFINED' then geolong= -70.40479659d

  ;; time in julian centuries since 1.1.2000, 12h UT
  Tu = (double(floor(MJD)) - MJulday(2000,1,1, 12,0,0)) / 36525.d
  UT = (MJD-floor(MJD))*24.d

  GMST= 6. + 41./60. + 50.54841/3600. $
	+ 8640184.812866d/3600. * Tu $
	+ 0.093104d/3600.  * Tu * Tu $
	- 6.2d-6/3600. * Tu * Tu * Tu $
	+ 1.002737909350795d * UT

  LST = GMST + geolong/15.	;; longitude of paranal

  return, LST mod 24.
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function RADec2uvec, RA_hour, Dec_deg	;; RA in hours, Dec in deg

  RA  = double(RA_hour) * !dpi/12.
  Dec = double(Dec_deg) * !dpi/180.

  X = cos(Dec) * cos(RA)
  Y = cos(Dec) * sin(RA)
  Z = sin(Dec)

  return, [ X, Y, Z ]
end

Function Velovec, RA_hour,Dec_deg,Parallax, pma,pmd,v_rad

  RA   = double(RA_hour) * !dpi/12.		;; input in hours
  Dec  = double(Dec_deg) * !dpi/180.		;; input in deg
  Parad= double(parallax)* !dpi/180./3600d3	;; input in mas
  pmRA = double(pma)/cos(Dec) * !dpi/180./3600.	;; input is mu*cos(dec) in arcsec/year
  pmDec= double(pmd) * !dpi/180./3600.   	;; input in arcsec/yr

  sRA  = sin(RA)  & cRA = cos(RA)
  sDec = sin(Dec) & cDec= cos(Dec)

  pmRad= double(v_rad) * 365.25*86400d/149.60d6 * Parad	;; v_radial/Distance in radian/year

  Vx= -sRA*pmRA - cRA*sDec*pmDec + cRA*cDec*pmRad	;; in radian/year
  Vy=  cRA*pmRA - sRA*sDec*pmDec + sRA*cDec*pmRad
  Vz=                 cDec*pmDec +     sDec*pmRad

  return, [ Vx,Vy,Vz ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;
;;;;;;		ABERRATION
;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function aberration1, MJD, uvec, DEBUG=debug

  X_geoc = uvec[0]
  Y_geoc = uvec[1]
  Z_geoc = uvec[2]

  ;;
  ;; get ephemeris for relevant time range
  ;;
  ;t1= min(MJD, max=t2)
  t1= MJD
  t2= MJD

  JPLEPHREAD, '/home/rkoehler/IDL/cmephem/JPLEPH.405', pinfo, pdata, [t1,t2]+2400000.5d
  JPLEPHINTERP, pinfo, pdata, MJD, Xe,Ye,Ze, Vxe,Vye,Vze, /EARTH,/VELOcity,$
                TBASE=2400000.5d, Posunits='AU', Velunits='KM/S'

  ha= (lst_from_date(MJD) + 6.) * !dpi/12.
  Vxe += 0.421d * cos(ha)	;; km/sec
  Vye += 0.421d * sin(ha)

  c_kms = 2.99792458d5 ;; lightspeed in km/sec

  Vxe /= c_kms	;; we need v/c all the time
  Vye /= c_kms
  Vze /= c_kms

  gaminv= sqrt(1.d - (Vxe*Vxe+Vye*Vye+Vze*Vze))	;; inverse of relativistic gamma
  pv    = X_geoc*Vxe + Y_geoc*Vye + Z_geoc*Vze  ;; pV/c
  nenner= 1.0d + pv                             ;; 1.+pV/c

  if keyword_set(debug) then print,format='("Gamma^-1:",4G15.12)',gaminv[0:3]
  if keyword_set(debug) then print,"Vx:",Vxe[0:3],"Vy:",Vye[0:3],"Vz:",Vze[0:3]

  X_geoc = (gaminv*X_geoc + Vxe + pv*Vxe/(1+gaminv)) / nenner
  Y_geoc = (gaminv*Y_geoc + Vye + pv*Vye/(1+gaminv)) / nenner
  Z_geoc = (gaminv*Z_geoc + Vze + pv*Vze/(1+gaminv)) / nenner

  len = sqrt( X_geoc*X_geoc + Y_geoc*Y_geoc + Z_geoc*Z_geoc)

  X_geoc /= len		;; now it's a unit vec again
  Y_geoc /= len
  Z_geoc /= len

  if keyword_set(debug) then begin
     print,"X:",X_geoc[0:3]
     print,"Y:",Y_geoc[0:3]
  endif

  return, [ X_geoc, Y_geoc, Z_geoc ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

pro aberration, MJD, RA1,Dec1, RA2,Dec2, DEBUG=debug

  vorher1 = RADec2uvec(RA1,Dec1)
  vorher2 = RADec2uvec(RA2,Dec2)

  sep_vorher= vorher2 - vorher1
  print,"separation vector before aberration:",sep_vorher

  nachher1 = aberration1( MJD[0],vorher1,DEBUG=debug)
  nachher2 = aberration1( MJD[0],vorher2,DEBUG=debug)

  sep_nachher= nachher2 - nachher1

  print,"separation vector after  aberration:",sep_nachher

  print,"separation correction:", sep_nachher - sep_vorher
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;
;;;;;		EPOCH TRANSFORMATION (incl. LIGHT TIME DELAY)
;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function epoch_trafo1, MJD, EpochMJD, uvec, Parallax, veloc

  ;; MJD is MJD of observation
  ;; EpochMJD is MJD of uvec etc.
  ;;
  ;; uvec = unit vector to star
  ;; Dec in deg
  ;; Parallax in mas
  ;; veloc in radian/year

  Parad = Parallax * !dpi/180./3600d3
  Distance= 1.d/Parad         ;; distance in AU

  c_kms = 2.99792458d5                          ;; lightspeed in km/sec
  c_AUyr= c_kms	* 365.25*86400d/149.60d6	;; lightspeed in AU/year
  ;;      km/sec  ( sec/year  )  km/AU

  time= (MJD - EpochMJD)/365.25d	;; julian years between Reference-Epoch and observation(s)
  ;;
  ;; compute position at time of observation (irrelevant, but useful)
  ;;
  tobs = Distance/c_AUyr + time	;; time from t0 to t_observation

  uv_obs= uvec + tobs * veloc

  X_tobs = uv_obs[0]	& Vx0 = veloc[0]
  Y_tobs = uv_obs[1]	& Vy0 = veloc[1]
  Z_tobs = uv_obs[2]	& Vz0 = veloc[2]

  lsu = c_AUyr * Parad	;; lightspeed/Distance (same unit as self.V_0)
  ;;
  ;; compute time between emission and observation of light
  ;;
  r_v = X_tobs*Vx0 + Y_tobs*Vy0 + Z_tobs*Vz0                    ;; (vec r) * (vec v)
  r2  = X_tobs*X_tobs + Y_tobs*Y_tobs + Z_tobs*Z_tobs           ;; r^2
  ls2_v2= lsu*lsu - Vx0*Vx0 + Vy0*Vy0 + Vz0*Vz0			;; c^2-v^2
  tau = (-r_v + sqrt( r_v*r_v + r2 * ls2_v2) ) / ls2_v2

  temit= tobs - tau	;; time when light was emitted

  if keyword_set(debug) then begin
     print,"tobs=",tobs
     print,"tau =",tau
     print,"temit =",temit
  endif

  uv_baryc  = uvec + temit * veloc
  Dist_baryc= sqrt(total(uv_baryc * uv_baryc))
  uv_baryc /= Dist_baryc

  return, uv_baryc
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Pro epoch_trafo, MJD, EpochMJD, RA1,Dec1,Par1,pma1,pmd1,vrad1, RA2,Dec2,Par2,pma2,pmd2,vrad2

  if N_params() lt 14 then begin
     print,"USAGE: epoch_trafo, MJD, EpochMJD, RA1,Dec1,Par1,pma1,pmd1,vrad1, RA2,Dec2,Par2,pma2,pmd2,vrad2"
     return
  end
  ;; MJD is MJD of observation
  ;; EpochMJD is MJD of uvec etc.
  ;;
  ;; RA in hours
  ;; Dec in deg
  ;; Parallax in mas
  ;; pma in arcsec/year, multiplied with cos(Dec)
  ;; pmd in arcsec/year
  ;; vrad in km/sec

  vorher1 = RADec2uvec(RA1,Dec1)  &  veloc1 = Velovec(RA1,Dec1,Par1,pma1,pmd1,vrad1)
  vorher2 = RADec2uvec(RA2,Dec2)  &  veloc2 = Velovec(RA2,Dec2,Par2,pma2,pmd2,vrad2)

  sep_vorher= vorher2 - vorher1
  print,"separation vector before epoch trafo:",sep_vorher

  nachher1= epoch_trafo1( MJD,EpochMJD, vorher1,Par1,veloc1)
  nachher2= epoch_trafo1( MJD,EpochMJD, vorher2,Par2,veloc2)

  sep_nachher= nachher2 - nachher1

  print,"separation vector after  epoch trafo:",sep_nachher
  print,"separation correction vector        :",sep_nachher - sep_vorher
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Pro epoch_trafo_single, MJD, EpochMJD, RA1,Dec1,Par1,pma1,pmd1,vrad1

  ;; MJD is MJD of observation
  ;; EpochMJD is MJD of uvec etc.
  ;;
  ;; RA in hours
  ;; Dec in deg
  ;; Parallax in mas
  ;; pma in arcsec/year, multiplied with cos(Dec)
  ;; pmd in arcsec/year
  ;; vrad in km/sec

  vorher1= RADec2uvec(RA1,Dec1)
  veloc1 = Velovec(RA1,Dec1,Par1,pma1,pmd1,vrad1)
  print,"unit vector before epoch trafo:",vorher1
  print,"velocity vector:               ",veloc1

  nachher1= epoch_trafo1( MJD,EpochMJD, vorher1,Par1,veloc1)
  print,"unit vector after  epoch trafo:",nachher1

  Dec= asin(nachher1[2])
  Ra = atan(nachher1[1],nachher1[0])

  print,"=> RA =",RA*180.*3600./!dpi," arcsec, Dec = ",Dec*180.*3600./!dpi," arcsec"

end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;
;;;;;	LIGHT DEFLECTION
;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function light_deflect1, MJD, uvec

  Masses= [ 0.,$	;; 0 - unused
            3.3022d23,$	;; 1 - Mercury, all in kg
            4.8685d24,$	;; 2 - Venus
            0.,$        ;; 3 - Earth (irrelevant)
            6.4185d23,$ ;; 4 - Mars
            1.8986d27,$ ;; 5 - Jupiter
            5.6846d26,$	;; 6 - Saturn
            8.6810d25,$	;; 7 - Uranus
            1.0243d26,$	;; 8 - Neptun
            1.305d22,$	;; 9 - Pluto
            7.3477d22,$	;;10 - Moon
            1.9891d30 $	;;11 - Sun
          ]
  plnames=['Oops','Mercury','Venus','Earth','Mars','Juputer','Saturn','Uranus','Neptune','Pluto','Moon','Sun']

  Grav = 6.67d-11	;; m^3/kg/s^2
  lspd = 2.99792458d8	;; lightspeed in m/s
  const= 2.d * Grav * Masses / lspd / lspd / 149.60d9	;; in AU (like XYZ_plan)
  ;;
  ;; get ephemeris for relevant time range
  ;;
  ;t1= min(MJD, max=t2)
  t1= MJD
  t2= MJD

  JPLEPHREAD, '/home/rkoehler/IDL/cmephem/JPLEPH.405', pinfo, pdata, [t1,t2]+2400000.5d

  ;; loop mercury...sun
  for pl=1,11 do begin
     if pl eq 3 then continue	;; skip Earth

     JPLEPHINTERP, pinfo,pdata, MJD, Xp,Yp,Zp, Object=pl,Center='EARTH', posunits='AU',TBASE=2400000.5d

     proj  = Xp*uvec[0] + Yp*uvec[1] + Zp*uvec[2]
     dist  = sqrt(Xp*Xp + Yp*Yp + Zp*Zp)
     cosang= proj/dist
     angle = acos(cosang)

     cosa= (Xp*uvec[0] + Yp*uvec[1] + Zp*uvec[2]) / dist
     ang = acos(cosa)
     phi = const[pl] / dist / tan(ang/2.)
     corr= const[pl] / dist / (1.-cosa)	;; Kovalevsky&Seidelmann, Eq. 6.53

     uvec[0] += corr * Xp	;; XYZp = -e in Kovalevsky&Seidelmann
     uvec[1] += corr * Yp
     uvec[2] += corr * Zp

     ;;print,"Planet",pl,uvec,sqrt(total(uvec*uvec))

     uvec /= sqrt(total(uvec*uvec))	;; normalize
  endfor

  return,uvec
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Pro light_deflect, MJD, RA1,Dec1, RA2,Dec2

  ;; MJD is MJD of observation
  ;; RA in hours
  ;; Dec in deg
  ;; Parallax in mas

  vorher1 = RADec2uvec(RA1,Dec1)  & print,"vorher1:",vorher1
  vorher2 = RADec2uvec(RA2,Dec2)  & print,"vorher2:",vorher2

  sep_vorher= vorher2 - vorher1
  print,"separation vector before light deflect:",sep_vorher

  nachher1= light_deflect1( MJD,vorher1)  & print,"nachher1:",nachher1
  nachher2= light_deflect1( MJD,vorher2)  & print,"nachher2:",nachher2

  sep_nachher= nachher2 - nachher1

  print,"separation vector after  light deflect:",sep_nachher
  print,"separation correction vector          :",sep_nachher - sep_vorher
end
