;
; astrometry.pro
; Created:     Tue Jul  3 14:41:49 2007 by Rkoehler@lx40
; Last change: Wed May 25 11:17:35 2011
;
; Copyright 2008 Rainer Koehler
;
; This file is part of Pacmart.
;
; Pacmart is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; Pacmart is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with Pacmart; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO Starpos__Define
  ;; Name: string; RA in hours; Dec in Degrees
  junk= { starpos,  Name: "",$
          Epoch: 2000.0, $
          RA:   0.d, Dec:   0.d, Parallax: 0.d,	$
          pmRA: 0.d, pmDec: 0.d, v_rad: 0.d	$
        }
end

PRO astrometry__Define
junk= { astrometry, $
        file_prefix: "Pacma", $
        MJD0: 54321.12345678d, $	;; MJD of start of obs
        stations: [ 'A2', 'Q7' ],$	;; telescope station names
        baseline: [ 100., 0., 0. ],$	;; 3-vector: West, South, Up in meters
        star_A: obj_new(),$		;; stars are nothing but objects
        star_B: obj_new(),$
        pos_A: { starpos }, $		;; position of star feeding FSU A
        pos_B: { starpos }, $		;; guess what
        deloff_A: 0.,$			;; delay offset star A (error in position)
        exptime: 150., $		;; time of one template (seconds)
        FSUA_rate: 8000., $
        FSUB_rate: 1000., $
        main_delay_rate: 1000., $	;; data rate of main delay (Hz)
        primet_rate:	 1000., $	;; data rate of primet (Hz)
        $
        FSUA_dit: 0.,$
        FSUB_dit: 0.,$
        detector_bias: [0,0,0,0,0,0],$
        skyback_flux:  [0,0,0,0,0,0],$
        primet_zero: [ 0d, 0d ], $	;; primet zero-point offset (constant and linear term)
        primet_noise:  0d,  $		;; gaussian noise (stddev in meters)
        FSU_config: ptr_new(), $
        $
        env_coeff: ptr_new() $
      }
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; compute OPD as function of local sidereal time
;;
function astrometry::compute_opd, lst, pos
  ;; lst = local sidereal time in seconds, double
  ;; pos = struct with RA,Dec of star 1,2, in hours,deg

  phi = -24.62794830d * !dpi/180.	;; latitude of paranal in rad

  lstrad = double(lst)/3600. * !dpi/12.
  HA = lstrad - pos.RA * !dpi/12.
  Dec= pos.Dec * !dpi/180.

  star_S = cos(Dec) * cos(HA) * sin(phi) - cos(phi) * sin(Dec)
  star_W = cos(Dec) * sin(HA)
  star_Z = cos(Dec) * cos(HA) * cos(phi) + sin(phi) * sin(Dec)

  ;;print, star_W, star_S, star_Z

  opd = self.baseline[0] * star_W $
      + self.baseline[1] * star_S $
      + self.baseline[2] * star_Z

  return, opd	;; in meters
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO astrometry::add_primet_trends, MJD0, time, dopd

  print,"ADDING DIFFERENTIAL METROLOGY TRENDS"

  etime= (MJD0-floor(MJD0)-0.5)*86400d + time
  ;print,"eTime0",etime[0],etime[10000]	; start and 10sec later

  ecoeff= *self.env_coeff
  evalue= environment_value( ecoeff.Sensor_ID[0], etime)
  ;print,evalue[0],evalue[10000]

  ;; hier muss der switch fuer verschiedene base-functions rein
  ;; bis jetzt nur linear:
  dvalue= (evalue - ecoeff.func_par[0]) * ecoeff.fit_coeff[0]
  ;print,dvalue[0],dvalue[10000]

  dopd += dvalue
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO astrometry::img_data_header, head, DIT, noise

  Common PACSIM_DEFAULTS

  head->addESOPar, 'ISS DID', 'JICD-1.0', Comment=' Data Dictionary name'
  head->addESOPar, 'ISS ID' , '1.0',	  Comment=' Data Dictionary version number'
  head->addESOPar, 'ISS DIP', '089-3200-6467', Comment=' Data Dictionary phone number'

  head->addPar, 'NREGION', 4,	  Comment=' Number of regions used (always 4)'
  head->addPar, 'MAXTEL', 0,	  Comment=' Maximum number of contributing telescopes'

  if DIT le 0 then return

  (*self.FSU_config).time= DIT
  (*self.FSU_config).skyback = (self.detector_bias + self.skyback_flux * DIT) $
                               / (*self.FSU_config).gain

  for i=0,N_elements(kappa_center)-1 do $
    head->addESOpar, 'SIM WAVENO'+string(i,format="(I1)"), kappa_center[i],$
                     Comment=' central wavenumber of channel '+strtrim(i+1,2)

  for i=0,N_elements(kappa_width)-1 do $
    head->addESOpar, 'SIM DWAVE'+string(i,format="(I1)"), kappa_width[i],$
                     Comment=' width of wavenumber channel '+strtrim(i+1,2)

  for i=0,N_elements(self.detector_bias)-1 do $
    head->addESOpar, 'SIM BIAS'+string(i,format="(I1)"),self.detector_bias[i],$
                     Comment=' bias of wavenumber channel '+strtrim(i+1,2)

  for i=0,N_elements(self.skyback_flux)-1 do $
    head->addESOpar, 'SIM FLUX'+string(i,format="(I1)"),self.skyback_flux[i],$
                      Comment=' sky + dark current in wavenumber ch.'+strtrim(i+1,2)

  if N_elements(noise) gt 1 then begin
     for i=0,N_elements(noise)-1 do $
        head->addESOpar, 'SIM NOISE'+string(i+1,format="(I1)"),noise[i],$
                         Comment=' noise in wavenumber channel '+strtrim(i+1,2)
  endif else if N_elements(noise) eq 1 then $
     head->addESOpar, 'SIM NOISE',noise, Comment=' noise in all wavenumber channels'

  sflat= lf_flux / max(lf_flux)	;; should be n_channels*n_ABCD
  if N_elements(sflat) gt 1 then  begin
     sz= size(sflat)
     for ch=0,sz[1]-1 do begin
        for i=0,sz[2]-1 do begin
           pixnm = string(byte('A')+byte(i)) + string(ch+1,format="(I1)")
           head->addESOpar, 'SIM FLAT'+pixnm, sflat[ch,i], Comment=' flatfield in pixel '+pixnm
        endfor
     endfor
  endif else $
     head->addESOpar, 'SIM FLAT',sflat, Comment=' flatfield in all channels'

  head->addESOpar, 'SIM DELOFF FSUA',self.deloff_A, Comment=' offset of FSUA [deg]'

  FSU_cfg = *self.fsu_config
  tags = tag_names(FSU_cfg)
  for i=0,N_elements(tags)-1 do begin
      val = fsu_cfg.(i)
      if N_elements(val) gt 1 then begin
          for j=0,N_elements(val)-1 do $
            head->addESOpar, 'SIM FSU '+tags[i]+strtrim(j,2), val[j]
      endif else head->addESOpar, 'SIM FSU '+tags[i], val
  endfor
end


PRO astrometry::img_data_data, file, rate, N_seconds

  Common PACSIM_DEFAULTS	;; for lf_flux

  N_pnts= long(rate * N_seconds)

  tabdat= { imaging_data_FSU_row }
  maxpix= 0d

  fsu= obj_new('FSU',*self.FSU_config)

  for i=0L, N_pnts-1 do begin
      sec = i/rate
      if fix(sec) eq sec then begin
         fsu->init_step,fix(sec)	;; each second is a new step
         ;;print, Format='(%" Second %d\r",$)',sec
      endif
      fsudata= fsu->simoneframe()	;; should be n_channels*n_ABCD

      maxpix = maxpix > max(fsudata)
      junk = where( finite(fsudata) eq 0,jcnt)
      if jcnt gt 0 then begin
          print,'FSU simulator returned garbage'
          print,'Config was:'
          print,*self.FSU_config
          print,'Result was:'
          print, fsudata
          stop
      endif
      junk = where(fsudata gt 32767,jcnt)
      if jcnt gt 0 then fsudata[junk]= 32767

      tabdat.time  = long( (double(i)+0.5) * 1d6 / rate ) ;; in mircoseconds
      tabdat.data1 = fsudata[*,0]
      tabdat.data2 = fsudata[*,1]
      tabdat.data3 = fsudata[*,2]
      tabdat.data4 = fsudata[*,3]
      file->writeRows, tabdat
  endfor
  obj_destroy,fsu

  if maxpix gt 32767d then begin
      print,"Max pixel value was",maxpix
      print,"Your data is garbage, and you toasted the detector!!!"
      print,""
      stop
  end
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO astrometry::writefile, MJD0, SWAP=swap

  if N_params() lt 1 then MJD0= self.MJD0

  filename = self.file_prefix + '.' + MJD_to_ESO_string(MJD0) + ".fits"
  print, "File: ",filename

  pfile= Pacman_binfile(MJD0)

  N_seconds = long(self.exptime)
  if keyword_set(swap) then N_seconds *= 2

  lst_start= lst_from_date(MJD0, geolong)
  lst_end  = lst_from_date(MJD0+self.exptime/86400d)	;; both in hours

  ;;
  ;; PRIMARY header
  ;;
  prihead= pfile.arrdesc->prihead()
  prihead->addPar,'DATE-OBS', MJD_to_ESO_string(MJD0), COMMENT=" Simulating date"
  prihead->addPar,'MJD-OBS', MJD0, FORMAT="F16.10"
  prihead->addPar,'UTC', (MJD0 mod 1.)*86400.,$
		   Comment= " " + hours_to_hms((MJD0 mod 1.)*24.) + " UTC at start (sec)"
  prihead->addPar,'LST', lst_start * 3600d, $
		   Comment= " " + hours_to_hms(lst_start) + " LST at start (sec)"

  if self.FSUB_dit le 0. then begin
     p  = self.star_A->current_position(MJD0)
     RA = p.RA
     Dec= p.Dec
  endif else $
     if self.FSUA_dit le 0. then begin
     p  = self.star_B->current_position(MJD0)
     RA = p.RA
     Dec= p.Dec
  endif else begin
     pos_A= self.star_A->current_position(MJD0)
     pos_B= self.star_B->current_position(MJD0)
     RA = (pos_A.RA + pos_B.RA )/2.
     Dec= (pos_A.Dec+ pos_B.Dec)/2.
     ;prihead->addPar,'DecA', pos_A.Dec,    Comment= " " + hours_to_hms(pos_A.Dec) + " DEC A pointing (deg)"
     ;prihead->addPar,'DecB', pos_B.Dec,    Comment= " " + hours_to_hms(pos_B.Dec) + " DEC B pointing (deg)"
  endelse
  prihead->addPar,'RA' , RA*15., Comment= " " + hours_to_hms(RA)  + " RA pointing (deg)"
  prihead->addPar,'Dec', Dec,    Comment= " " + hours_to_hms(Dec) + " DEC pointing (deg)"
  prihead->addPar,'EXPTIME',float(N_seconds),Comment= " (seconds)"

  standard_primary_header, prihead, self.FSUA_dit,self.FSUB_dit, SIM_Code='AM', OBS_Start=self.MJD0
  ;;CATG='SCIENCE ', TECH='INTERFEROMETRY', TYPE='OBJECT,ASTROMETRY'

  if self.FSUB_dit gt 0 then begin
     id = keyword_set(swap) ? '1' : '2'
     standard_targ_header, prihead, id, 'PS', self.pos_B, *self.FSU_config

     if not keyword_set(swap) and self.FSUB_dit lt 1. then $
         prihead->addESOPar,'ISS FSU2 STATE', 'TRACKING', Comment=' Current FSU state'
     ;; FSUB is always FSU1-no, FSU2!
  endif
  if self.FSUA_dit gt 0 then begin
     id = keyword_set(swap) ? '2' : '1'
     pss= self.FSUB_dit gt 0? 'SS': 'PS'
     standard_targ_header, prihead, id, pss, self.pos_A, *self.FSU_config

     ;; A may be tracking if Swapped or FSUB off
     if (keyword_set(swap) or self.FSUB_dit le 0) and self.FSUA_dit lt 1. then $
         prihead->addESOPar,'ISS FSU1 STATE', 'TRACKING', Comment=' Current FSU state'
     ;; FSUA is always 2- no, 1!!
  endif

  prihead->addESOPar,'ISS CONF NTEL', 2, Comment=' both telescopes'
  prihead->addESOPar,'ISS CONF STATION1', self.stations[0], Comment=' station of telescope 1'
  prihead->addESOPar,'ISS CONF T1X', 0., Comment=' position of telescope 1'
  prihead->addESOPar,'ISS CONF T1Y', 0., Comment=' position of telescope 1'
  prihead->addESOPar,'ISS CONF T1Z', 0., Comment=' position of telescope 1'
  prihead->addESOPar,'ISS CONF T1NAME', 'Pacman', Comment=' name of telescope 1'

  prihead->addESOPar,'ISS CONF STATION2', self.stations[1], Comment=' station of telescope 2'
  prihead->addESOPar,'ISS CONF T2X', self.baseline[0], Comment=' position of telescope 2'
  prihead->addESOPar,'ISS CONF T2Y', self.baseline[1], Comment=' position of telescope 2'
  prihead->addESOPar,'ISS CONF T2Z', self.baseline[2], Comment=' position of telescope 2'
  prihead->addESOPar,'ISS CONF T2NAME', 'Pacwoman', Comment=' name of telescope 2'

  prihead->addESOPar,'ISS GEOLAT',-24.62794830d,Comment=' geographic latitude'
  prihead->addESOPar,'ISS GEOLON', geolong, Comment=' geographic longitude'

  ha = lst_start - RA	;; hour angle in hours
  project_baseline, ha, Dec, self.baseline, base_len, base_pa, Debug=0
  prihead->addESOPar,'ISS PBL12 START', base_len,Format="F10.6",$
				Comment=" projected baseline length at start (m)"
  prihead->addESOPar,'ISS PBLA12 START', base_pa,Format="F10.5",$
				Comment=" projected baseline angle on sky at start (deg)"

  ha = lst_end - RA	;; hour angle in hours
  project_baseline, ha, Dec, self.baseline, base_len, base_pa, Debug=0
  prihead->addESOPar,'ISS PBL12 END', base_len,Format="F10.6",$
				Comment=" projected baseline length at end (m)"
  prihead->addESOPar,'ISS PBLA12 END', base_pa,Format="F10.5",$
				Comment=" projected baseline angle on sky at end (deg)"

  if self.FSUA_dit gt 0 and self.FSUB_dit gt 0 then begin
      insmode = (keyword_set(swap) ? "SWAPPED" : "NORMAL")
  endif else insmode= (self.FSUA_dit gt 0)? 'SINGLE_A' : 'SINGLE_B'

  prihead->addESOPar,'INS MODE', insmode, COMMENT=" instrument mode (normal or swapped)"
  prihead->addESOPar,'INS DOPDC MODE', 'FRINGE_TRACK', COMMENT=" blind or shortsighted tracking"

  ;;prihead->addESOPar,'TPL NAME', 'PACMAN_obs_Astrometry', Comment="Template name"

  prihead->addESOPar,'ISS DDL1 OPL START', 0.  &  prihead->addESOPar,'ISS DDL1 OPL END', 0.
  prihead->addESOPar,'ISS DDL2 OPL START', 0.  &  prihead->addESOPar,'ISS DDL2 OPL END', 0.
  prihead->addESOPar,'ISS DDL3 OPL START', 0.  &  prihead->addESOPar,'ISS DDL3 OPL END', 0.
  prihead->addESOPar,'ISS DDL4 OPL START', 0.  &  prihead->addESOPar,'ISS DDL4 OPL END', 0.

  prihead->addESOPar,'ISS DLT1 OPL START', 0.  &  prihead->addESOPar,'ISS DLT1 OPL END', 0.
  prihead->addESOPar,'DEL DLT1 OPL START', 0.  &  prihead->addESOPar,'DEL DLT1 OPL END', 0.

  prihead->addESOPar,'SIM AUTHOR', 'R. Koehler', COMMENT='Author of simulator'
  ;;
  ;; Tables in DICD-Order --------------------------------------------
  ;;
  Pacman_binfile_create_arrdesc, pfile, filename             ;; Array description
  pfile.arrdesc->writeRows, standard_array_description()

  Pacman_binfile_create_arrgeom, pfile                       ;; Array geometry
  pfile.arrgeom->writeRows, standard_array_geometry()

  Pacman_binfile_create_optrain, pfile                       ;; Optical train
  pfile.optrain->writeRows, standard_optical_train()
  ;;
  ;; Main Delay -------------------------------------------------------
  ;;
  ; standard_main_delay_header, pfile.MDL->head(), self.main_delay_rate
  ;
  ; N_pnts = N_seconds * self.main_delay_rate
  ; tabdat = replicate( { main_delay_row }, N_pnts )
  ;
  ; time= dindgen(N_seconds*self.main_delay_rate) / self.main_delay_rate	;; sec
  ; lst = lst_from_date(MJD0 + time/86400d) * 3600d	;; seconds
  ; ;;main_opd = self->compute_opd( lst, self.pos_A)
  ; main_opd = self->compute_opd( lst, self.star_A->current_position3(MJD0))
  ;
  ; tabdat.time = long(time * 1d6)  ;; microsecs
  ; tabdat.delay1= 0d
  ; tabdat.delay2= 0d + main_opd
  ;
  ;Pacman_binfile_create_MDL, pfile  ;; Main Delay
  ; pfile.MDL->writeRows, tabdat
  ;;
  ;; Img_detector_FSUA: not empty ----------------------------------------
  ;;
  standard_img_detector_header, pfile.det_A->head(), 'Romeo', (*self.FSU_config).gain
  Pacman_binfile_create_DetA, pfile
  ;if self.FSUA_dit gt 0 then
  standard_img_detector_table, pfile.det_A
  ;;
  ;; imaging data FSUA: not empty
  ;;
  self->img_data_header, pfile.dataA->head(), self.FSUA_dit
  Pacman_binfile_create_DataA, pfile

  if self.FSUA_dit gt 0 then begin
      print,"Simulating FSU A"
      if not keyword_set(swap) and self.deloff_A ne 0. then begin
         (*self.FSU_config).refphase= self.deloff_A
         ;;(*self.FSU_config).steppingnum= N_seconds
         ;;(*self.FSU_config).steppingmin= 0.
         ;;(*self.FSU_config).steppingmax= N_seconds*self.deloff_A
      endif else begin
         (*self.FSU_config).refphase= 0.
         ;;(*self.FSU_config).steppingnum= 1
         ;;(*self.FSU_config).steppingmin= 0.
         ;;(*self.FSU_config).steppingmax= 0.
      endelse
      self->img_data_data, pfile.dataA, self.FSUA_rate, N_seconds
  endif
  ;;;if dita gt 0 then skybg_write_imaging_data, pfile.dataA, self.FSUA_dit, noise
  ;;
  ;; Imaging detector FSUB: not empty
  ;;
  standard_img_detector_header, pfile.det_B->head(), 'Juliet', (*self.FSU_config).gain
  Pacman_binfile_create_DetB, pfile
  ;if self.FSUB_dit gt 0 then
  standard_img_detector_table, pfile.det_B
  ;;
  ;; imaging data FSUB: not empty
  ;;
  self->img_data_header, pfile.dataB->head(), self.FSUB_dit
  Pacman_binfile_create_DataB, pfile

  if self.FSUB_dit gt 0 then begin
      print,"Simulating FSU B"
      if keyword_set(swap) and self.deloff_A ne 0. then begin
         (*self.FSU_config).refphase= self.deloff_A
         ;;(*self.FSU_config).steppingnum= N_seconds
         ;;(*self.FSU_config).steppingmin= 0.
         ;;(*self.FSU_config).steppingmax= N_seconds*self.deloff_A
      endif else begin
         (*self.FSU_config).refphase= 0.
         ;;(*self.FSU_config).steppingnum= 1
         ;;(*self.FSU_config).steppingmin= 0.
         ;;(*self.FSU_config).steppingmax= 0.
      endelse
      self->img_data_data, pfile.dataB, self.FSUB_rate, N_seconds
  endif
  ;;;if ditb gt 0 then skybg_write_imaging_data, pfile.dataB, self.FSUB_dit, noise
  ;;
  ;; PRIMETrology ------------------------------------------------------------------
  ;;
  standard_metrology_header, pfile.metro->head(),$
                             self.primet_rate, self.primet_zero, self.primet_noise

  N_pnts = N_seconds * self.primet_rate
  tabdat = replicate( { metrology_data_row, $
                       time:   0L, $
                       deltal: 0.d, $
                       status: 319L }, $
                     N_pnts )

  time= dindgen(N_seconds * self.primet_rate) / self.primet_rate  ;; seconds
  lst = lst_from_date(MJD0 + time/86400d) * 3600.d	;; in seconds

  ;;opd_A= self->compute_opd( lst, self.pos_A)
  ;;opd_B= self->compute_opd( lst, self.pos_B)
  opd_A= self->compute_opd( lst, self.star_A->current_position(MJD0))
  opd_B= self->compute_opd( lst, self.star_B->current_position(MJD0))

  dopd = opd_A - opd_B	;; A-B agrees with scired2
  if keyword_set(swap) then dopd = -dopd

  dopd += self.primet_zero[0] + self.primet_zero[1] * time

  if self.primet_noise ne 0. then $
    dopd += self.primet_noise * randomn(seed, N_elements(dopd))

  if ptr_valid(self.env_coeff) then self->add_primet_trends, MJD0, time, dopd

  tabdat.time  = long(time * 1d6)  ;; microsecs
  tabdat.deltal= dopd

  Pacman_binfile_create_metro, pfile	;; PRIMET
  pfile.metro->writeRows, tabdat
  ;;
  ;; Metrology - the sequel
  ;;
  standard_metrology_header, pfile.metroB->head(), /FSUB, $
                             self.primet_rate, self.primet_zero, self.primet_noise

  N_pnts = N_seconds * self.primet_rate
  tabdat = replicate( { metrology_data_fsub_row,$
                        time:   0L, $
                        deltal: 0d, $
                        status: 319L },$
                      N_pnts )
  tabdat.time  = long(time * 1d6)  ;; microsecs
  tabdat.deltal= opd_B * disperse_metroB(MJD0,time)

  Pacman_binfile_create_metroB,pfile    ;; PRIMETB: empty
  pfile.metroB->writeRows, tabdat
  ;;
  ;; Difficult Delay Lines
  ;;
  Pacman_binfile_create_DDL, pfile    ;; DDL: empty
  ;;
  ;; diagnostic tables
  ;;
  Pacman_binfile_create_OPDC, pfile
  OPDC_rate= self.FSUB_rate * 2L
  N_pnts= OPDC_rate * N_seconds
  tabdat= { OPDC_row }
  tabdat.OPDCstate= 7
  tabdat.state = 7
  tabdat.status= 7
  tabdat= replicate(tabdat,N_pnts)
  tabdat.time= dindgen(N_pnts) / OPDC_rate * 1d6 + randomu(seed,N_pnts)*3	;; mircoseconds

  lst = lst_from_date(MJD0 + tabdat.time/86400d6) * 3600d	;; seconds
  tabdat.DLFDBaCK= self->compute_opd( lst, self.star_A->current_position(MJD0))

  pfile.OPDC->writeRows, tabdat
  ;;
  ;;
  Pacman_binfile_create_DOPDC, pfile
  DOPDC_rate= self.FSUB_rate * 2L
  N_pnts= DOPDC_rate * N_seconds
  tabdat= { DOPDC_row }
  tabdat.OPDCstate= 7
  tabdat.state = 7
  tabdat.status= 7
  tabdat= replicate(tabdat,N_pnts)
  tabdat.time= dindgen(N_pnts) / DOPDC_rate * 1d6 + randomu(seed,N_pnts)*3	;; mircoseconds
  pfile.DOPDC->writeRows, tabdat
  ;;
  ;; close the door and cleanup before you leave
  ;;
  Pacman_binfile_close,  pfile
  Pacman_binfile_cleanup,pfile
  ;;
  ;; tell our caller how long it took
  ;;
  MJD0 += self.exptime/86400d
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; simulate one OB with unswapped-swapped-swapped-unwapped templates
;;
PRO astrometry::simulate

  Common PACSIM_defaults

  print,"===== astrometry templates ====="
  print,"Adit",self.FSUA_dit, ", DitB",self.FSUB_dit

  MJD0  = self.MJD0
  ;;
  ;; we have to do everything ourselves...
  ;;
  FSUresponse, STARTTIME=MJD_to_ESO_string(MJD0),/FSUA,/FSUB,FSUnoise=0.1,DelayJit=0.1
  MJD0 += 0.25/24.d
  ;FSUresponse, STARTTIME=MJD_to_ESO_string(MJD0),/FSUB,FSUnoise=0.1,DelayJIT=0.1
  ;MJD0 += 0.25/24.d
  ;;
  ;; first, take a look at the sky and make it flat
  ;;
  skarflat, MJD=MJD0, DITa= self.FSUA_dit, DitB=self.FSUB_dit, $
            BIAS=self.detector_bias, TEMPER=(*self.FSU_config).star_temp, $
            NOISE=0.	;; do not apply blackbody

  ;;FLAT=lf_flux, NOISE=0.	;; do not apply blackbody

  MJD0 += 4./60./24.	;; skarflat adds one minute after its exposure
  ;;
  ;; first template: normal
  ;;
  self->writefile, MJD0

  MJD0 += (self.exptime + 60.d)/86400.d		;; exptime + 1 min later
  self.primet_zero[0] += self.primet_zero[1] * (self.exptime + 60.d)

  ;; no swapping with singles:
  if self.FSUA_dit eq 0. or self.FSUB_dit eq 0. then return

  ;;
  ;; second template: swapped and twice as long
  ;;
  save_DitA= self.FSUA_dit & save_rateA= self.FSUA_rate
  save_DitB= self.FSUB_dit & save_rateB= self.FSUB_rate

  self.FSUB_dit= save_DitA & self.FSUB_rate= save_rateA
  self.FSUA_dit= save_DitB & self.FSUA_rate= save_rateB

  skarflat, MJD=MJD0, DITa= self.FSUA_dit, DitB=self.FSUB_dit, $
            BIAS=self.detector_bias, TEMPER=(*self.FSU_config).star_temp, $
            NOISE=0.	;; do not apply blackbody

  self->writefile, MJD0, /swap

  self.FSUA_dit= save_DitA & self.FSUA_rate= save_rateA
  self.FSUB_dit= save_DitB & self.FSUB_rate= save_rateB

  MJD0 += (2.*self.exptime + 60.d)/86400.d	;; 2*exptime + 1min later
  self.primet_zero[0] += self.primet_zero[1] * (2.*self.exptime + 60.d)
  ;;
  ;; third template: normal again
  ;;
  skarflat, MJD=MJD0, DITa= self.FSUA_dit, DitB=self.FSUB_dit, $
            BIAS=self.detector_bias, TEMPER=(*self.FSU_config).star_temp, $
            NOISE=0.	;; do not apply blackbody

  self->writefile, MJD0
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Go ahead, make my object
;;
;; starttime = date and time of start of obs ('yyyy-mm-ddThh:mm:ss.sssssss')
;;
;; baseline = 3-vector: West, South, Up in meters
;;
;; pos_A = position of star going to FSU A, struct with RA, Dec in hours, degrees
;; pos_B = position of the other star
;;
;; primet0 = PRIMET-zero-point (constant added to PRIMET-result, meters)
;;

Function astrometry::init, FILE_PREFIX=file_prefix, STARTTIME=starttime, $
                   POS_A=pos_a, POS_B=pos_b, PRIMET0=primet0,$
                   FSU_RATE=fsu_rate, FSU_CONFIG= fsu_config

  common PACSIM_DEFAULTS

  ;; set defaults

  self.file_prefix = "PACMA"

  ;; when starttime is empty, the current date/time is used
  st  = eso_date(starttime)	;; year, month, day, hour, min, sec
  self.MJD0  = MJulday(st[0],st[1],st[2], 0,0,0)	;; year, mon, day, hms
  self.MJD0 += (st[3] + (st[4] + st[5]/60.d) / 60.d) / 24.d	;; add time later to preserve precision

  ;;self.stations = [ 'A2', 'Q7' ]
  ;;self.Baseline = [ 100., 0., 0. ] 		;; default baseline = 100m East-West

  self->set_stations, 'A2', 'H5'	;; 100m @ 45deg
  ;;self->set_stations, 'A2', 'Q7'	;; 100m E-W

  self.primet_zero= [ 0d, 0d ]
  self.primet_noise=  0d
  self.exptime = 150.
  self.main_delay_rate= 1000.
  self.primet_rate    = 1000.
  self.FSUA_rate= 10.
  self.FSUB_rate= 10.
  self.FSUA_dit = 1./self.FSUA_rate - 50.d-6	;; 50 usec for readout
  self.FSUB_dit = 1./self.FSUB_rate - 50.d-6

  self.detector_bias= ld_bias
  self.skyback_flux = sb_flux

  self.FSU_config= ptr_new( keyword_set(fsu_config)? fsu_config : Default_FSU_Config())

  if keyword_set(file_prefix)  then self.file_prefix= file_prefix
  if N_elements(primet0)  gt 0 then self.primet_zero= primet0
  if keyword_set(fsu_rate) then begin
      self.FSUA_rate= fsu_rate
      self.FSUB_rate= fsu_rate
      self.FSUA_dit = 1./fsu_rate - 50.d-6	;; 50 usec for readout
      self.FSUB_dit = 1./fsu_rate - 50.d-6	;; 50 usec for readout
  endif

  if N_elements(pos_A) le 0 then $
     pos_A={ Name: "Prima", Epoch: 2000., RA: 0.d,      Dec: 0d,	Parallax: 100., pmRA: 0.d, pmDec: 0.d, v_rad: 0.}

  if N_elements(pos_B) le 0 then $
     pos_B={ Name: "Secunda",Epoch: 2000.,RA: 0.d/3600.,Dec: 10.d/3600.,Parallax: 100., pmRA: 0.d, pmDec: 0.d, v_rad: 0.}

  self->set_pos, A=pos_A, B=pos_B

  return,1	;; Qapla'
end

function astrometry
  return, obj_new('astrometry')
end

PRO astrometry::cleanup
  print,"Astrometry cleanup"
  obj_destroy, self.star_A
  obj_destroy, self.star_B
  ptr_free, self.FSU_config
  ptr_free, self.env_coeff
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Set parameters
;;
PRO astrometry::set_prefix, file_prefix
  self.file_prefix= file_prefix
end

PRO astrometry::set_startMJD, MJD0
  self.MJD0= MJD0
end

PRO astrometry::set_starttime, starttime
  st  = eso_date(starttime)	;; year, month, day, hour, min, sec
  self.MJD0= MJulday(st[0],st[1],st[2], st[3],st[4],st[5]) ;; year, mon, day, hms
end

PRO astrometry::set_stations, T1, T2

  self.stations = [ T1, T2 ]
  self.Baseline = [ 0., 0., 0.]

  if T1 eq 'A2' then begin
     if T2 eq 'Q7' then      self.Baseline = [ 100., 0., 0.] $
     else if T2 eq 'A8' then self.Baseline = [ 0., 100., 0.] $
     else if T2 eq 'H5' then self.Baseline = [ 100., 100., 0.]/sqrt(2.) $
     else print,'Unknown station: ',T2
  endif else $
     print,'Unknown station: ',T1
end

PRO astrometry::set_baseline, baseline
  self.Baseline = baseline

  T2name= 'XY'
  if self.baseline[0] eq 100. then T2name= 'Q7' $
  else if self.baseline[1] eq 100. then T2name= 'A8'

  print,"T2name is ",T2name
  self.stations = [ 'A2', T2name ]
end

PRO astrometry::set_pos, A=pos_A, B=pos_B
  starpos= { starpos }
  if N_elements(pos_A) gt 0 then begin
     struct_assign, pos_A, starpos
     self.pos_A = starpos
     obj_destroy,self.star_A
     self.star_A= obj_new("star", pos_A.name, pos_A.Epoch,$
                          pos_A.RA, pos_A.Dec, pos_A.Parallax, pos_A.pmRA, pos_A.pmDec, 0d)
  endif
  if N_elements(pos_B) gt 0 then begin
     struct_assign, pos_B, starpos
     self.pos_B = starpos
     obj_destroy,self.star_B
     self.star_B= obj_new("star", pos_B.name, pos_B.Epoch,$
                          pos_B.RA, pos_B.Dec, pos_B.Parallax, pos_B.pmRA, pos_B.pmDec, 0d)
  endif
end

PRO astrometry::set_deloff, doff
  self.deloff_A= doff	;; in degrees
end

PRO astrometry::set_exptime, exptime
  self.exptime = exptime
end

PRO astrometry::set_rates, FSU=fsu, FSUA=fsua, FSUB=fsub, MAIN_DELAY=main, PRIMET=primet

  if keyword_set(fsu) then begin
      self.FSUA_rate= fsu  &  self.FSUA_dit = 1./fsu - 50.d-6	;; 50 usec for readout
      self.FSUB_rate= fsu  &  self.FSUB_dit = 1./fsu - 50.d-6	;; 50 usec for readout
  endif
  if keyword_set(fsua) then begin
      self.FSUA_rate= fsua  &  self.FSUA_dit = 1./fsua - 50.d-6	;; 50 usec for readout
  endif
  if keyword_set(fsub) then begin
      self.FSUB_rate= fsub  &  self.FSUB_dit = 1./fsub - 50.d-6	;; 50 usec for readout
  endif
  if keyword_set(main)   then self.main_delay_rate = main
  if keyword_set(primet) then self.primet_rate	   = primet
end

PRO astrometry::set_DITs, FSUA=fsua, FSUB=fsub
  ;; remember that a DIT may be 0
  if N_elements(fsua) gt 0 then self.FSUA_dit= fsua
  if N_elements(fsub) gt 0 then self.FSUB_dit= fsub

  print,"set_DITs:",self.FSUA_dit, self.fsuB_dit
end

PRO astrometry::set_detector_bias, bias
  self.detector_bias= bias
end

PRO astrometry::set_skyback_flux, flux
  self.skyback_flux = flux
end

PRO astrometry::set_FSU_config, FSU_config
  ptr_free, self.env_coeff
  self.FSU_config= ptr_new(FSU_config,/no_copy)
end

Function astrometry::get_FSU_config
  return, *self.FSU_config
end

PRO astrometry::set_n_phonoise, n_phonoise
  (*self.FSU_config).n_phonoise= n_phonoise
end

PRO astrometry::set_primet0, primet0
  self.primet_zero= primet0
end

PRO astrometry::set_primet_noise, stddev
  self.primet_noise= stddev
end

PRO astrometry::set_env_coeff, co
  ptr_free, self.env_coeff
  self.env_coeff= ptr_new(co,/no_copy)
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; set paramters by name
;;
PRO astrometry::set_FSU_param, name, value
  p= where(tag_names(*self.FSU_config) eq strupcase(name),cnt)
  if cnt ne 1 then print,"astrometry::set_FSU_param: No such parameter: ",name $
    else (*self.FSU_config).(p)= value
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO astrometry::info
  print,"Astrometry configuration:"
  print,'File prefix: "',self.file_prefix,'"'
  print,"Start time: ", self.MJD0, " = ", MJulDate(self.MJD0)
  print,"Baseline [m]:",self.baseline
  print,"Primary star:   ", self.pos_a
  print,"Secondary star: ", self.pos_b
  print,"Exptime:       ",self.exptime, " seconds"
  print,"Rate of FSUA:  ",self.fsua_rate, " Hz"
  print,"Rate of FSUB:  ",self.fsub_rate, " Hz"
  print,"Rate of MDL:   ",self.main_delay_rate," Hz"
  print,"Rate of PRIMET:",self.primet_rate, " Hz"
  print,""
  print,"FSUA DIT: ",self.FSUA_dit," sec"
  print,"FSUB DIT: ",self.FSUB_dit," sec"
  print,"Detector bias:", self.detector_bias
  print,"Skybackground:", self.skyback_flux
  print,"PRIMET zero:  ", self.primet_zero
  print,"PRIMET noise: ", self.primet_noise
  print,"FSU config:"
  help, *self.FSU_config, /str
end

PRO astrometry::configure
  print,"------------------------------------------------------------------------"
  print," Poor man's astrometry configurator"
  print,"------------------------------------------------------------------------"

  line= ""

  read, Prompt="File prefix ("+self.file_prefix+"): ",line
  if line ne "" then self.file_prefix = line

  read, Prompt="Start time ("+strtrim(self.MJD0,2)+"): ",line
  if line ne "" then begin
      st  = eso_date(line)	;; year, month, day, hour, min, sec
      self.MJD0= MJulday(st[0],st[1],st[2], st[3],st[4],st[5]) ;; year, mon, day, hms
  endif

  ;;read, Prompt="Baseline West  ("+strtrim(self.baseline[0],2)+"m): ",line & if line ne "" then self.baseline[0] = float(line)
  ;;read, Prompt="Baseline South ("+strtrim(self.baseline[1],2)+"m): ",line & if line ne "" then self.baseline[1] = float(line)
  ;;read, Prompt="Baseline Up    ("+strtrim(self.baseline[2],2)+"m): ",line & if line ne "" then self.baseline[2] = float(line)

  read, Prompt="Exptime ("+strtrim(self.exptime,2)+"s): ",line & if line ne "" then self.exptime = float(line)

  read, Prompt="Rate of FSUA   ("+strtrim(self.FSUA_rate,2)+"Hz): ",line
  if line ne "" then begin
      self.FSUA_rate= float(line) & self.FSUA_dit = 1./self.FSUA_rate - 50.d-6	;; 50 usec for readout
  end
  read, Prompt="Rate of FSUB   ("+strtrim(self.FSUB_rate,2)+"Hz): ",line
  if line ne "" then begin
      self.FSUB_rate= float(line) & self.FSUB_dit = 1./self.FSUB_rate - 50.d-6	;; 50 usec for readout
  end
  read, Prompt="Rate of MDL    ("+strtrim(self.main_delay_rate,2)+"Hz): ",line & if line ne "" then self.main_delay_rate= float(line)
  read, Prompt="Rate of PRIMET ("+strtrim(self.primet_rate,2)+"Hz): ",line & if line ne "" then self.primet_rate= float(line)

  ;; to be continued
end

PRO astrometry::help
  print,"I cannot help you."
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO astrometry_testfiles, NOBOTH=noboth, NOONLINE=noonline

  Common PACSIM_defaults

  ;ld_bias = [  0. ,  0. ,  0. ,  0. ,  0. ,  0.  ]
  ;sb_flux = ld_bias		;; NO skybackground
  ;sb_flux = replicate(1000,6)	;; constant skybackground
  ;lf_flux = replicate(1., 6,4)	;; flat flatfield
  ;lf_flux = [[ 1000., 1000., 1000., 1000., 1000., 1000.],$
  ;           [  900.,  900.,  900.,  900.,  900.,  900.],$
  ;           [ 1500., 1500., 1500., 1500., 1500., 1500.],$
  ;           [ 1100., 1100., 1100., 1100., 1100., 1100.] ]

  if not keyword_set(noboth) then begin
     ;; both trecking
     a = obj_new("astrometry",Starttime='2008-04-01T04:00:00')
     a->set_rates, FSUA=10., FSUB=9.
     a->set_n_phonoise, 0
     a->set_primet0, [1d-6, 0d]	;; 1 micrometer
     a->set_primet_noise, 1d-7	;; about 0.1 micrometer noise
     a->simulate
     a->info
     obj_destroy,a
  end
  ;environment, MJD=MJulDay(2008,3,31,12,0,0)

  ;; this obs takes until 04:54:30
  ;;return

  if not keyword_set(noonline) then begin
     ;; one trecking, one sitting
     a = obj_new("astrometry",Starttime='2008-04-01T05:00:00')
     a->set_rates, FSUA=4., FSUB=100.
     a->set_n_phonoise, 0
     a->set_primet_noise, 1d-6	;; about 1 micrometer noise
     a->simulate
     a->info
     obj_destroy,a
  endif

  ;; A only
  a = obj_new("astrometry",Starttime='2008-04-01T06:00:00')
  a->set_rates,FSUA=10.
  a->set_DITs, FSUB=0.
  a->set_n_phonoise, 0
  a->set_primet_noise, 1d-6	;; about 1 micrometer noise
  a->simulate
  a->info
  obj_destroy,a

  ;; B only
  a = obj_new("astrometry",Starttime='2008-04-01T07:00:00')
  a->set_rates,FSUB=10.
  a->set_DITs, FSUA=0.
  a->set_n_phonoise, 0
  a->set_primet_noise, 1d-6	;; about 1 micrometer noise
  a->simulate
  a->info
  obj_destroy,a
end

PRO sciave_kilohertz
     a = obj_new("astrometry",Starttime='2009-04-01T04:00:00')
     a->set_rates, FSUA=1000., FSUB=1000.
     a->set_n_phonoise, 0
     a->set_primet0, [1d-6, 0d]	;; 1 micrometer
     a->set_primet_noise, 1d-7	;; about 0.1 micrometer noise
     a->simulate
     a->info
     obj_destroy,a
end

PRO scired1_offsetest

  common PACSIM_DEFAULTS

  ;offstep = 80.	;; max = step*4 = 320 deg
  offstep = 160.	;; max = step*4 = 640 deg

  a = obj_new("astrometry",Starttime='2008-04-01T04:00:00')
  a->set_rates, FSUA=10., FSUB=10.
  ;;a->set_detector_bias, 0.
  ;;a->set_skyback_flux, 0.
  a->set_deloff, 0.
  a->set_n_phonoise, 0
  a->set_primet0, [1d-6, 0d]	;; 1 micrometer
  a->set_primet_noise, 1d-7	;; about 0.1 micrometer noise
  a->simulate

  a->set_starttime, '2008-04-01T05:00:00'
  a->simulate

  a->set_starttime, '2008-04-01T06:00:00'
  a->set_deloff, offstep
  a->simulate

  a->set_starttime, '2008-04-01T07:00:00'
  a->set_deloff, offstep*2.
  a->simulate

  a->set_starttime, '2008-04-01T08:00:00'
  a->set_deloff, offstep*3.
  a->simulate

  a->set_starttime, '2008-04-01T09:00:00'
  a->set_deloff, offstep*4.
  a->simulate

  obj_destroy,a
end

PRO scired1_testfiles

  env_MDL = { Func_ID: 0,$
              Sensor_ID: [ 1, 2, 3, 4],$
              Func_Par:  [1d,2d,3d,4d],$
              Fit_coeff: [1d,2d,3d,4d],$
              Covariances: [0] $
            }

  env_metro={ Func_ID: 0,$
              Sensor_ID: [  41, -1, -1, -1],$
              Func_Par:  [ 10d, 0d, 0d, 0d],$
              Fit_coeff: [1d-6, 0d, 0d, 0d],$
              Covariances: [1d-9] $
            }

  env_metB ={ Func_ID: 0,$
              Sensor_ID: [ 1, 2, 3, 4],$
              Func_Par:  [1d,2d,3d,4d],$
              Fit_coeff: [1d,2d,3d,4d],$
              Covariances: [0] $
            }

  MDL_cc= obj_new('FITSTABLE', env_MDL, extName='ENV_FIT_DELAY', iErr=iErr)
  if iErr then begin
      print,"creation of FITS-table 'ENV_FIT_MAIN_DELAY' failed, iErr =",iErr
      return
  endif
  MDL_cc->newfile, "Pacma_Parrot.fits"
  ;MDL_cc->writerows, env_MDL

  Metro_cc= obj_new('FITSTABLE', env_metro, extName='ENV_FIT_DIFF_DELAY', iErr=iErr)
  if iErr then begin
      print,"creation of FITS-table 'ENV_FIT_METROLOGY' failed, iErr =",iErr
      return
  endif
  Metro_cc->appendtofile, MDL_cc
  Metro_cc->writerows, env_metro

  MetroB_cc= obj_new('FITSTABLE', env_metB, extName='ENV_FIT_PRIMET_FSUB', iErr=iErr)
  if iErr then begin
      print,"creation of FITS-table 'ENV_FIT_METROLOGY_B' failed, iErr =",iErr
      return
  endif
  MetroB_cc->appendtofile, Metro_cc
  ;MetroB_cc->writerows, env_metB
  MetroB_cc->close

  obj_destroy,MDL_cc
  obj_destroy,Metro_cc
  obj_destroy,MetroB_cc

  a= obj_new("astrometry",File_prefix='WindTrend',Starttime='2007-08-14T22:00:00')
  a->set_env_coeff, env_metro
  a->simulate
  obj_destroy,a
end

PRO noisy_files
  a= obj_new("astrometry",File_prefix='Noise',Starttime='2007-08-16T16:45:00')
  a->set_n_phonoise, 1
  a->simulate
  obj_destroy,a
end

PRO stepped_files
  a= obj_new("astrometry",File_prefix='Stepped',Starttime='2007-08-21T00:00:00')
  a->set_n_phonoise, 1
  a->set_FSU_param, "steppingnum", 4
  a->set_FSU_param, "steppingmin", 0.
  a->set_FSU_param, "steppingmax", 270.
  a->info
  a->simulate
  obj_destroy,a
end

PRO fullsize_files

  a = obj_new("astrometry",File_prefix='Big',Starttime='2007-10-03T00:00:00')
  a->set_rates, FSUA=8000., FSUB=8000., MAIN_DELAY=8000., PRIMET=8000.
  a->set_n_phonoise, 0
  a->set_primet_noise, 1d-6	;; about 1 micrometer noise
  a->simulate
  a->info
  obj_destroy,a

  environment, MJD=MJulDay(2007,10,3-1,12,0,0)
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO scired3_testfiles

  Starpos__Define	;; make sure starpos is defined with the right data types

  RA0_A = 0.d		 & Dec0_A = 0.d
  RA0_B = 7.d/15.d/3600. & Dec0_B = 7.d/3600.

  pmRA_A= 0.d	& pmDec_A= 1.2d		;; arcsec/yr, 1.2 = 0.1 as/month
  pmRA_B= 0.1d	& pmDec_B= 0.d

  pos_A={ starpos, Name: "Prinda", Epoch: 2008.0, RA: RA0_A, Dec: Dec0_A, Parallax: 100., pmRA: pmRA_A, pmDec: pmDec_A, v_rad: 0.}
  pos_B={ starpos, Name: "Secuma", Epoch: 2008.0, RA: RA0_B, Dec: Dec0_B, Parallax: 100., pmRA: pmRA_B, pmDec: pmDec_B, v_rad: 0.}

  ;; lie about p.m.:
  ;pos_A= { starpos, Name: "Prinda", RA: RA0_A, Dec: Dec0_A, pmRA: 0., pmDec: 0., Epoch: 2008.0 }
  ;pos_B= { starpos, Name: "Secuma", RA: RA0_B, Dec: Dec0_B, pmRA: 0., pmDec: 0., Epoch: 2008.0 }

  MJD2008 = MJulday(2008,1,1,0,0,0)

  starspectrum_one, 'STARSPECTRUM_Prinda.fits', pos_A, MJD= MJD2008
  starspectrum_one, 'STARSPECTRUM_Secuma.fits', pos_B, MJD= MJD2008+1./24.

  a = obj_new("astrometry", POS_A=pos_A, Pos_B=pos_B)

  ;; 3 years = 36 months
  for Month=6,42 do begin
     ;; don't start before April 1. (when the RESPONSES are taken)
     yr = 2008 + fix(Month/12)
     mon= Month mod 12
     Day= 1

     MJD0= MJulday(yr,mon,Day, 0,0,0)

     print,Format='(I2,". ",I2," => ",F0)',Day,Month,MJD0

     environment, MJD=MJD0-0.5

     ;;now done by astrometry:
     ;;t = (MJD0-MJD2008)/365.25	;; time since epoch in years
     ;;pos_A.RA = RA0_A + pmRA_A*t	& pos_A.Dec = Dec0_A + pmDec_A*t
     ;;pos_B.RA = RA0_B + pmRA_B*t	& pos_B.Dec = Dec0_B + pmDec_B*t
     ;;pos_A.Epoch= 2008.+t
     ;;pos_B.Epoch= 2008.+t

     ;; a->set_pos, A=pos_A, B=pos_B
     a->set_rates, FSUA=4., FSUB=10.
     a->set_n_phonoise,0
     a->set_primet_noise, 0.	;; 1d-7
     a->set_primet0, [ 0., 0.]	;; 5.d-6/3600.]	;; 5um/hr

     ;; alternate between E-W and N-S baselines
     a->set_stations, 'A2', (Month mod 2 eq 0)? 'Q7' : 'A8'

     time=0.
     tstr= MJD_to_ESO_string(MJD0+time/24.d)
     print,">>>>>>>>>> ",tstr," <<<<<<<<<<"

     a->set_startMJD, MJD0+time/24.d
     a->simulate

  endfor
  obj_destroy,a
end


PRO daftestfiles, PRIMETNOISE=pmtnoise

  if not keyword_set(pmtnoise) then pmtnoise= 5e-3 * sqrt(1000.)	;; 5nm after 1sec avg.

  Nstars = 6

  primRA = dindgen(Nstars)*4.	;; 0, 4, 8, ... 20
  primDec= [ 0.d, -45.d, -90.d, -45.d, 0.d, -45.d ]
  primNm = strarr(Nstars)

  secRA = primRA + randomn(seed,Nstars)/360./15.  ;; sigma = 10 arcsec
  secDec= primDec+ randomn(seed,Nstars)/360.

  FSU_cfg= { star_temp: 0., visi: 0.7, star_angdiam: 0.05 }

  for iStar=0,Nstars-1 do begin
     RA = primRA[iStar]
     Dec= primDec[iStar]
     name = string(Format='("RK",A,I02,I02)', Dec ge 0.? '+':'-', fix(abs(Dec)), fix(RA))
     primNm[iStar]= name
     print,iStar,": Name ",name,", RA =",RA,", Dec =",Dec

     starspectrum_one, 'STARSPECTRUM_'+name+'.fits',$
                    { Name: name, RA: 0.d, Dec: 0d   },$
                    FSU_cfg,$
                    MJD= MJulday(2007,4,1, 0,0,0)
     starspectrum_one, 'STARSPECTRUM_'+name+'b.fits',$
                    { Name: name+'b', RA: 0.d/3600., Dec: 0.1d },$
                    FSU_cfg,$
                    MJD= MJulday(2007,4,1, 0,1,0)
  endfor

  for Month=1,13,3 do begin
     for Day=1,7 do begin
        MJD0= MJulday(2008,Month,Day, 0,0,0)

        print,Format='(I2,". ",I2," => ",F0)',Day,Month,MJD0

        environment, MJD=MJD0-0.5
        ;;environment, MJD=MJD0+0.5

        a = obj_new("astrometry")
        a->set_rates, FSUA=1., FSUB=10.
        a->set_n_phonoise,0
        a->set_primet_noise, pmtnoise
        a->set_primet0, [ 0., 5.d-3/3600.]	;; 5um/hr

        for time=0.,12. do begin
           tstr= MJD_to_ESO_string(MJD0+time/24.d)
           print,">>>>>>>>>> ",tstr," <<<<<<<<<<"
           iStar= fix(randomu(seed)*Nstars)
           print,"Star",iStar," = ",primNm[iStar],$
                 primRA[iStar], (secRA[iStar] -primRA[iStar])*3600.*15.,$
                 primDec[iStar],(secDec[iStar]-primDec[iStar])*3600.

           a->set_startMJD, MJD0+time/24.d
           a->set_pos, A={ Name: primNm[iStar],    RA: primRA[iStar], Dec: primDec[iStar] },$
                       B={ Name: primNm[iStar]+"b", RA: secRA[iStar],  Dec: secDec[iStar] }
           a->simulate
        endfor

        obj_destroy,a
     endfor
  endfor
end
