;
; environment.pro
; Created:     Wed Jun 13 14:35:46 2007 by Rkoehler@lx40
; Last change: Tue Feb 15 15:36:25 2011
;
; PURPOSE:
;	save the environment (to a FITS File)
;
; Copyright 2008 Rainer Koehler
;
; This file is part of Pacmart.
;
; Pacmart is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; Pacmart is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with Pacmart; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;;
;; Important Sensors:
;; 184 = VLTItempSens5
;; 205 = VLTI_HSS_RHUM2
;; 134 = SitePressure

Function environment_value, sensid, time, DEBUG=debug

  if keyword_set(debug) then print,"Sensor",sensid
  case sensid of
      1:  value=  16. + 5.*sin(time/86400.*2.*!PI)
      11: value=  15. + 6.*sin(time/86400.*2.*!PI)
      21: value=  15.5+ 6.*sin(time/86400.*2.*!PI)
      31: value= 773. + randomn(seed, N_elements(time))
      35: value=  42. + time/8640.
      41: value=  10. + time/86400.
      42: begin
          value= 300. + time/864.
          value[where(value ge 360.)] -= 360.
      end
      else: print,"Unkown Sensor ID:",sensid
  endcase
  return, float(value)
end


PRO environment, MJD=MJD

  ;; by default, we start at 12h UT
  if not keyword_set(MJD)   then MJD = MJulDay(2008, 3, 31, 12,0,0)

  filename = 'PACMA.' + MJD_to_ESO_string(MJD) + ".fits"
  print, "File: ",filename

  envdesc = fitstable_create('ENVIRONMENT_DESCRIPTION')
  envdata = fitstable_create('ENVIRONMENT')

  prihead= envdesc->prihead()
  prihead->addPar,'DATE-OBS', MJD_to_ESO_string(MJD), COMMENT=" Simulating date"
  prihead->addPar,'MJD-OBS', MJD, FORMAT="F16.10"
  prihead->addESOPar,'INS REC START', MJD, FORMAT="F16.10"
  prihead->addPar,'UTC', (MJD mod 1.)*86400., $
    Comment= " " + hours_to_hms((MJD mod 1.)*24.) + " UTC at start (sec)"

  prihead->addPar,'EXPTIME',86400.,Comment= " (seconds)"

  ;prihead->addESOpar,'DPR CATG', 'CALIB   ', Comment=' Environmental category'
  ;prihead->addESOpar,'DPR TECH', 'OTHER   ', Comment=' Environmental technique'
  ;prihead->addESOpar,'DPR TYPE', 'SENSORS ', Comment=' Environmental type'

  standard_primary_header, prihead, 0.,0., SIM_Code='EN'

  ;prihead->addESOPar,'TPL NAME', 'PACMAN_rec_Environment', Comment="Template name"

  (envdesc->head())->addPar, 'SENSORS', 7, Comment=' Estimated number of sensors'
  envdesc->newfile, filename

  envdesc->writeRows, $
    { environment_description_row,$
      id:	184,$
      name:	string(format='(A-32)',"VLTItempSens5"), $
      type:	string(format='(A-32)',"TC, whatever that means"), $
      location:	string(format='(A-32)',"out there in the cold"), $
      unit:	string(format='(A-32)',"C"), $
      comment:	string(format='(A-32)','16.0+5.\times\sin(t)'), $
      ccsenv:   string(format='(A-8)', "who?"), $
      keyword:  string(format='(A-32)',"what?"), $
      dict:     string(format='(A-20)',"where?"), $
      hw_id:    string(format='(A-16)','4763441874648-49') $
    }

  envdesc->writeRows, $
    { environment_description_row,$
      id:	11,$
      name:	string(format='(A-32)',"Temperature Tel1"), $
      type:	string(format='(A-32)',"TC, whatever that means"), $
      location:	string(format='(A-32)',"leftmost telescope"), $
      unit:	string(format='(A-32)',"Deg.C"), $
      comment:	string(format='(A-32)','15.0+6.\times\sin(t)'), $
      ccsenv:   string(format='(A-8)', "who?"), $
      keyword:  string(format='(A-32)',"what?"), $
      dict:     string(format='(A-20)',"where?"), $
      hw_id:    string(format='(A-16)','4763441874648-47') $
    }

  envdesc->writeRows, $
    { environment_description_row,$
      id:	21,$
      name:	string(format='(A-32)',"Temperature Tel2"), $
      type:	string(format='(A-32)',"TC, whatever that means"), $
      location:	string(format='(A-32)','2. telescope from the right'),$
      unit:	string(format='(A-32)',"Degrees Celcius"), $
      comment:	string(format='(A-32)','15.5+6.\times\sin(t)'), $
      ccsenv:   string(format='(A-8)', "who?"), $
      keyword:  string(format='(A-32)',"what?"), $
      dict:     string(format='(A-20)',"where?"), $
      hw_id:    string(format='(A-16)','4763441874648-48') $
    }

  envdesc->writeRows, $
    { environment_description_row,$
      id:	134,$
      name:	string(format='(A-32)',"SitePressure"), $
      type:	string(format='(A-32)',"Press"), $
      location:	string(format='(A-32)',"out there in the cold"), $
      unit:	string(format='(A-32)',"hPa"), $
      comment:	string(format='(A-32)','773.+{\rm rand}(\sigma=1)'), $
      ccsenv:   string(format='(A-8)', "who?"), $
      keyword:  string(format='(A-32)',"what?"), $
      dict:     string(format='(A-20)',"where?"), $
      hw_id:    string(format='(A-16)','4763441874648-42') $
    }

  envdesc->writeRows, $
    { environment_description_row,$
      id:	205,$
      name:	string(format='(A-32)',"VLTI_RSS_RHum2"), $
      type:	string(format='(A-32)',"Press"), $
      location:	string(format='(A-32)',"out there in the cold"), $
      unit:	string(format='(A-32)',"Percent"), $
      comment:	string(format='(A-32)','42. + t/8640.'), $
      ccsenv:   string(format='(A-8)', "who?"), $
      keyword:  string(format='(A-32)',"what?"), $
      dict:     string(format='(A-20)',"where?"), $
      hw_id:    string(format='(A-16)','4763441874648-43') $
    }

  envdesc->writeRows, $
    { environment_description_row,$
      id:	41,$
      name:	string(format='(A-32)',"Wind speed"), $
      type:	string(format='(A-32)',"Anemometer"), $
      location:	string(format='(A-32)',"out there in the cold"), $
      unit:	string(format='(A-32)',"furlong/fortnight"), $
      comment:	string(format='(A-32)','10. + t/86400.'), $
      ccsenv:   string(format='(A-8)', "who?"), $
      keyword:  string(format='(A-32)',"what?"), $
      dict:     string(format='(A-20)',"where?"), $
      hw_id:    string(format='(A-16)','4763441874648-45') $
    }

  envdesc->writeRows, $
    { environment_description_row,$
      id:	42,$
      name:	string(format='(A-32)',"Wind direction"), $
      type:	string(format='(A-32)',"Vane"), $
      location:	string(format='(A-32)',"out there in the cold"), $
      unit:	string(format='(A-32)',"Radian"), $
      comment:	string(format='(A-32)','300. + t/864.'), $
      ccsenv:   string(format='(A-8)', "who?"), $
      keyword:  string(format='(A-32)',"what?"), $
      dict:     string(format='(A-20)',"where?"), $
      hw_id:    string(format='(A-16)','4763441874648-46') $
    }
  ;;
  ;; and now for something completely different
  ;;
  envdata->appendtofile, envdesc

  ;; ambient temp.
  time = dindgen( fix(86400d/30d)) * 30.	;; 2 stamps per minute for 24h
  data = replicate( {environment_row}, N_elements(time))
  data.sensor= 184
  data.time = long( ulong64(time) mod 4294967296)
  Data.value= environment_value(1,time)		;; some kind of temperature
  envdata->writerows, data

  ;; tel 1 temp.
  time = dindgen( fix(86400d/59d)) * 59. + 2	;; one stamp per minute for 24h
  data = replicate( {environment_row}, N_elements(time))
  data.sensor= 11
  data.time = long(time)
  data.value= environment_value(11,time)
  envdata->writerows, data

  ;; tel 2 temp.
  time = dindgen( fix(86400d/61d)) * 61.	;; one stamp per minute for 24h
  data = replicate( {environment_row}, N_elements(time))
  data.sensor= 21
  data.time = long(time)
  data.value= environment_value(21,time)
  envdata->writerows, data

  ;; press.
  time = dindgen( fix(86400d/60d)) * 60. + 1	;; one stamp per minute for 24h
  data = replicate( {environment_row}, N_elements(time))
  data.sensor= 134
  data.time = long(time)
  data.value= environment_value(31,time)
  envdata->writerows, data

  ;; humid
  time = dindgen( fix(86400d/30d)) * 30. + 1	;; 2 stamps per minute for 24h
  data = replicate( {environment_row}, N_elements(time))
  data.sensor= 205
  data.time = long(time)
  data.value= environment_value(35,time)
  envdata->writerows, data

  ;; wind
  time = dindgen( fix(86400d/60d)) * 60. + 2	;; one stamp per minute for 24h
  data = replicate( {environment_row}, N_elements(time))
  data.sensor= 41
  data.time = long(time)
  data.value= environment_value(41,time)
  envdata->writerows, data

  time = dindgen( fix(86400d/60d)) * 60. + 2	;; one stamp per minute for 24h
  data = replicate( {environment_row}, N_elements(time))
  data.sensor= 42
  data.time = long(time)
  data.value= environment_value(42,time)
  envdata->writerows, data

  envdata->close

  obj_destroy, envdata
  obj_destroy, envdesc
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

pro convert_jeroen, infile

  edesc= fitstable_openr(infile,'ENVIRONMENT_DESCRIPTION')
  data = edesc->readrows()

  prihead= edesc->prihead()
  ;;headlines= prihead->sthead()
  ;;for i=0,N_elements(headlines)-1 do print,i,headlines[i]

  MJD= prihead->getpar('INS REC START') & print,"MJD =",MJD

  prihead->addPar,'DATE-OBS', MJD_to_ESO_string(MJD), COMMENT=" Simulating date"
  prihead->addPar,'MJD-OBS', MJD, FORMAT="F16.10"
  prihead->addESOPar,'INS REC START', MJD, FORMAT="F16.10"
  prihead->addPar,'UTC', (MJD mod 1.)*86400., $
    Comment= " " + hours_to_hms((MJD mod 1.)*24.) + " UTC at start (sec)"

  prihead->addPar,'EXPTIME',86400.,Comment= " (seconds)"
  prihead->addESOPar,'TPL NAME', 'PACMA_rec_Environment', Comment="Template name"

  outfile= 'PACMA.'+MJD_to_ESO_string(MJD)+'.fits'
  print,"writing to ",outfile
  edesc->newfile, outfile, prihead=prihead
  edesc->writerows,data

  edata= fitstable_openr(infile,'ENVIRONMENT')
  jdata= edata->readrows()
  data = replicate( { sensor: 1, time: 0L, value: 0.}, N_elements(jdata))
  data.sensor= jdata.sensor[0]
  data.time  = jdata.time[0]
  data.value = jdata.value[0]
  ;; forget 'STATUS' column

  envdata = fitstable_create('ENVIRONMENT')

  help,data

  envdata->appendtofile, edesc
  envdata->writerows,data

  envdata->close

  obj_destroy, edesc
  obj_destroy, edata
  obj_destroy, envdata
end
