;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;********************************************************************
;#class#
; FitsExtension
;#inheritance#
; none
;#description#
; Base class for FITS extension files
; initialize extensions/ read-write headers
; skip around files looking for other extensions etc.
;#end_class#
;********************************************************************

PRO FitsExtension__DEFINE
;**********************************************************************
;#structure#
; FitsExtension
;#inheritance#
; none
;#description#
; Internal variables for FitsExtension Object
;#structure_text#
a1={FitsExtension,$ ;#class members#
file:OBJ_NEW(),     $  ;ptr->FITSFILE & to which file does extension belong
extName:'',         $  ;string & type of extension
extNumber:0L,       $  ;long   & sequence number
headStart:0L,       $  ;long   & position of start of header (bytes)
dataStart:0L,       $  ;long   & position of start of data
dataEnd:0L,         $  ;long   & position of end of data
dataCurrent:0L,     $  ;long   & position currently being read/written
dataStatus:'',      $  ;string & READ/WRITE..
priHead:OBJ_NEW(),  $  ;ptr    & primary header of file
head:OBJ_NEW(),     $  ;ptr    & extension header object
headerStatus:''     $  ;string & 'DISK','LOCAL'...
}
;#end_structure#
;**********************************************************************

;**********************************************************************
;#structure#
; FitsExtensionDef
;#inheritance#
; none
;#description#
; structure to specify a particular extension
;#structure_text#
a6={EXTENSIONDEF, $       ;#non-class structure#
fileName:'',     $     ;string & file name
extName:'',      $     ;string & extension name
extNumber:0L     $     ;long   & extension number
}
;#end_structure
;**********************************************************************
END

FUNCTION extDef, fileName, extNum, extName

;********************************************************************
;#function#
; extDef
;#call#
; extDefStruct = extDef( fileName, extNum, extName)
;#description#
; A utility to construct an EXTENSIONDEF structure
; from  a filename/extNumber/extName set.
; Note that this is a global function rather than a member function.
;#inputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{fileName}  {string}  {name of FITS file }
;\anArg{extNum}{string} {name of extension}
;\anArg{extName}{long}  {number of extension (of type extName)}
;#return#
;\anArg{-}{ EXTENSIONDEF}{ structure with extension info.}
;#end_function#
;********************************************************************
;
   if (N_PARAMS() EQ 2) then RETURN, {EXTENSIONDEF,fileName,'',extNum} $
   else RETURN, {EXTENSIONDEF,fileName,extName,extNum}
END

FUNCTION fitsExtension::init, file, iErr=iErr, $
   extName=extName, extNumber=extNumber
;********************************************************************
;#function#
; init
;#call#
; fitsExtensionObj = OBJ_NEW(file, iErr=iErr, extName=extName, extNumber=extNumber)
;#description#
; FitsExtension constructor
; with polymorphic input.
;#inputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ extName }  { string }{ name of extension }
;\anArg{ extNumber }{ int    }{ number of extension (of type extName)}
; Polymorphic file can be of the following types:
; \begin{enumerate}
; \item absent \\
;     If file absent but extname is specified then
;     create a dummy extension header and store internally.
;     Also create a dummy primary ext for possible later use.
; \item string \\
;     File is a string and possibly extName and extNumber are given.
;     Open file, create new FITSFILE object containing  primary header,
;     then read through to find correct  extension, and store
;     extension header.  Stop with input file positions just after
;     ext. header.
;     extNum specifies sequence number of extension of
;     type extName to find.  extNum defaults to 1,
;     extName defaults to '' (any).
; \item  object.  \\
;     File is a FITSFILE object.  Proceed as in (2)
; \end{enumerate}
;#outputs#
;\anArg{ iErr }{ int    }{ optional:Return code, 0 => OK, else failed}
;#return#
; \anArg{-}{FitsExtension object}{ } .
;#end_function#
;********************************************************************
;
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN,0
   endif         ; handle actual error
;
;Default, no input but extName or extNumber set
   if (n_params() EQ 0) then begin
      if(KEYWORD_SET(extname)) then self.extName = extName
      self.head = OBJ_NEW('FITSHEADER', binTable=1, extName=extName)
      self.priHead = OBJ_NEW('FITSHEADER')
      self.headerStatus = 'INTERNAL'
      iErr = 0
      RETURN,1
   endif             ; default, no input
;
;There is input, check the type
   sInput = SIZE(file)
   isString = ((sinput(0) EQ 0) AND (sinput(1) EQ 7))
   isObj = 0
   if (NOT isString) then begin
      sInput = sInput(1+sInput[0])
      isObj =  sInput eq 11
      if (isObj) then isObj = OBJ_CLASS(file) eq 'FITSFILE'
   endif
   if (NOT (isString or IsObj)) then $
      midiSetError, 'File specified is not a string or FITSFILE'
;
;Check defaults
   if (NOT KEYWORD_SET(extName)) then extName = ''
   if (NOT KEYWORD_SET(extNumber)) then extNumber = 1
   self.extName   = extName
   self.extNumber = extNumber
;if string input Create FITSFILE object
;and open the file.  The primary header is then in (self.file).priHead
;and the input file is positioned just after the primary header
   if (isString) then begin
      fileName = file
      self.file = OBJ_NEW('FITSFILE', fileName, 'READ', /noSelf, iErr=iErr)
      if (NOT OBJ_VALID(self.file)) then $
         midiSetError, 'FITSFILE open error', /notInitial
;existing file object
   endif else self.file = file
   unit = (self.file)->unit()
;start reading just after primary header
   pointlun = ((self.file)->priHead())->headerLength()
   point_lun, unit, pointLun     ;Current position
;   header = ((self.file)->priHead())->sthead()
;Make sure that the file does contain extensions.
;
   if (0 EQ ((self.file)->priHead())->isPriHead()) then $
      midiSetError, 'FITS FILE does not contain any extensions'
;
;Initialize extension counter and current header object
;
   iExt = 0
   iExtType = 0
NEXT_EXT:
;How big is data area for current extension?
   if(iExt EQ 0) then nBytes = ((self.file)->priHead())->byteSize() else $
      nbytes = (self.head)->byteSize()
;
;Skip over unwanted data
;
   point_lun, -unit, pointLun    ;Current position
   mhead0 = pointLun + nbytes
   point_lun, unit, mhead0    ;Beginning of Next FITS extension
;store byte position of beginning of header
   self.headStart = mhead0
;Release heap space of stored header object if not wanted
   if (OBJ_VALID(self.head)) then OBJ_DESTROY,self.head
;Read in next extension header
   self.head = OBJ_NEW('FITSHEADER', unit, iErr=iErr)
;If EOF suppress most of the error messages
   if (iErr eq 4) then begin
       midiClearErrMsg
       midiSetError,'extension name '+self.extName+' ext# '+$
          STRTRIM(string(self.extNumber),2)+' not Found',/notInitial
   endif else if (iErr ne 0) then midiSetError, 'FITSHEADER create error', /notInitial
   self.headerStatus = 'DISK'
;Beginning of extension data after header
   point_lun, -unit, dataStart
;Update counters and pointers
   iExt    = iExt + 1
;
;Check to see if the current extension is the one desired.
;
   extName = self.head->extName()
   if ((self.extName EQ '') OR $
      (strTrim(self.extName,2) EQ strTrim(extName,2))) then begin
      iExtType = iExtType + 1
      if (iExtType EQ self.extNumber) then GOTO, DONE
   endif
   goto, NEXT_EXT
;
;Check to see if the extension type is BINTABLE IMAGE or A3DTABLE
;
DONE:
   xtension = self.head->type()
   if (xtension NE 'BINTABLE') AND (STRTRIM(xtension,2)  NE 'IMAGE') then $
      midiSetError, 'Requested Header is not Bintable or IMAGE'
;
;
   self.file->setState,1
   self.file->addExt, self
   self.extName     = extName
;  Store byte position of beginning of data
   self.dataStart   = dataStart
   self.dataCurrent = dataStart
   self.dataEnd     = dataStart + self.head->BYTESIZE()
   self.headerStatus = 'READ'
   self.dataStatus   = 'READ'
   iErr = 0
RETURN, 1
END

PRO fitsExtension::cleanup
;********************************************************************
;#procedure#
; cleanup
;#call#
; fitsExtensionObj->cleanup
;#description#
; Class destructor
;#end_procedure#
;********************************************************************
   OBJ_DESTROY, self.head
   OBJ_DESTROY, self.priHead
;check if we are the only object refering to this file.  If so destroy
;the object (which closes file)
   if (OBJ_VALID(self.file)) then $
      if (self.file->removeExt(self) LE 0) then OBJ_DESTROY, self.file
RETURN
END

FUNCTION fitsExtension::head, iErr=iErr
;********************************************************************
;#function#
; head
;#call#
; headerObj = fitsExtObj->head (iErr=iErr)
;#description#
; Get FitsHeader object for this extension.
;#outputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ iErr }{ int    }{optional: Return code, 0 => OK, else failed}
;#return#
;\anArg {-}{ headerObj}{ FitsHeader object for this extension.}
;#end_function#
;********************************************************************
   iErr = 0
   if (OBJ_VALID(self.head)) then RETURN,self.head
   iErr = 1
   RETURN,0
END

FUNCTION fitsExtension::stHead, iErr
;********************************************************************
;#function#
; stHead
;#call#
; headerText = fitsExtObj->stHead (iErr=iErr)
;#description#
; Get Header Text for this extension
;#outputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ iErr }{ int    }{optional:Return code, 0 => OK, else failed}
;#return#
;\anArg{-}{ string array}{lines of header}
;#end_function#
;********************************************************************
   iErr = 0
   if (OBJ_VALID(self.head)) then  RETURN, (self.head)->stHead()
   iErr = 1
RETURN, ''
END


FUNCTION fitsExtension::priHead, iErr=iErr
;********************************************************************
;#function#
; priHead
;#call#
; primaryHeader = fitsExtObj->priHead (iErr=iErr)
;#description#
; Get Primary header
; from FitsFile associated with this extension
; If not, returns local FITSHEADER object
;#outputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ iErr }{ int    }{ optional: Return code, 0 => OK, else failed}
;#return#
;\anArg{-}{ HeadObj}{ primary header object}
;#end_function#
;********************************************************************
   iErr = 0
   if (OBJ_VALID(self.file)) then RETURN,self.file->priHead() $
      else if (OBJ_VALID(self.priHead)) then RETURN, self.priHead
   iErr = 1
RETURN,PTR_NEW()
END

FUNCTION fitsExtension::extName
;********************************************************************
;#function#
; extName
;#call#
; extensionName = fitsExtObj->extName()
;#description#
; Return the (stored) name of this extension
;#return#
;\anArg{-}{ string}{extension name}
;#end_function#
;********************************************************************
RETURN,self.extName
END

FUNCTION fitsExtension::naxis, iErr=iErr
;********************************************************************
;#function#
; naxis
;#call#
; nAxis = fixtExtObj->naxis (iErr=iErr)
;#description#
;  Get naxis parameters for extension.
;  For Table extensions, nAxis(1) is bytes/row
;                        nAxis(2) is number of rows
;#outputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ iErr }{ int    }{ optional:Return code, 0 => OK, else failed}
;#return#
;\anArg{-}{ intarr[2]}{ naxis parameters for this extension.}
;#end_function#
;********************************************************************
   iErr = 0
   RETURN,[self.naxis1, self.naxis2]
END

FUNCTION fitsExtension::file, iErr=iErr
;********************************************************************
;#function#
; file
;#call#
; fitsFileObj = fitsExtObj->file (iErr=iErr)
;#description#
;  Get FITSFILE object
; for file containing this extension.
;#outputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ iErr }{ int    }{ optional:Return code, 0 => OK, else failed}
;#return#
;\anArg{- }{FITSFILEObj}{ FITSFILE contained in extension obj}
;#end_function#
;********************************************************************
   iErr = 0
   if (OBJ_VALID(self.file) ) then RETURN,self.file
   iErr = 1
   RETURN,OBJ_NEW()
END

FUNCTION fitsExtension::fileName, iErr=iErr
;********************************************************************
;#function#
; fileName
;#call#
; fileName = fitsExtObj->fileName (iErr=iErr)
;#description#
;  Returns file name
;  of file containing this extension
;#outputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ iErr }{ int    }{ optional:Return code, 0 => OK, else failed}
;#return#
; \anArg{-}{string}{fileName}
;#end_function#
;********************************************************************
   iErr = 0
   if (OBJ_VALID(self.file) ) then RETURN,self.file->fileName()
   iErr = 1
   RETURN,OBJ_NEW()
END

FUNCTION fitsExtension::dataStatus, iErr=iErr
;********************************************************************
;#function#
; dataStatus
;#call#
; status = fitsExtObj->dataStatus (iErr=iErr)
;#description#
; Get internal dataStatus value
;#outputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ iErr }{ int    }{ optional:Return code, 0 => OK, else failed}
;#return#
;\anArg{-}{int}{ dataStatus }
;#end_function#
;********************************************************************
   iErr = 0
   RETURN,self.dataStatus
END

FUNCTION fitsExtension::dataStart, iErr=iErr
;********************************************************************
;#function#
; dataStart
;#call#
; startPosition = fitsExtObj->dataStart (iErr=iErr)
;#description#
; Byte offset of extension data from BOF
;#outputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ iErr }{ int    }{ optional: Return code, 0 => OK, else failed}
;#return#
;\anArg{-}{ long}{offset of start of extension data from BOF}
;#end_function#
;********************************************************************
   iErr = 0
   RETURN,self.dataStart
END

FUNCTION fitsExtension::dataCurrent, iErr=iErr
;********************************************************************
;#function#
; dataCurrent
;#call#
; currentPosition = fitsExtObj->dataCurrent (iErr=iErr)
;#description#
; Byte offset of current data from BOF
;#outputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ iErr }{ int    }{ optional:Return code, 0 => OK, else failed}
;#return#
;\anArg{-}{ long}{offset of current data record from BOF in bytes.}
;#end_function#
;********************************************************************
   iErr = 0
   RETURN,self.dataCurrent
END

FUNCTION fitsExtension::headerStatus, iErr=iErr
;********************************************************************
;#function#
; headerStatus
;#call#
; status = fitsExtObj->headerStatus (iErr=iErr)
;#description#
; headerstatus ('DISK', 'INTERNAL'...)
;#outputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ iErr }{ int    }{ optional: Return code, 0 => OK, else failed}
;#return#
;\anArg{-}{ String}{headerstatus }
;#end_function#
;********************************************************************
   iErr = 0
   RETURN,self.headerStatus
END

FUNCTION fitsExtension::headStart, iErr=iErr
;********************************************************************
;#function#
; headStart
;#call#
; startPosition = fitsExtObj->headStart (iErr=iErr)
;#description#
; Byte offset of ext. header from BOF
;#outputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ iErr }{ int    }{ optional: Return code, 0 => OK, else failed}
;#return#
;\anArg{-}{ long}{offset of start of ext. header from BOF in bytes.}
;#end_function#
;********************************************************************
   iErr = 0
   RETURN,self.headStart
END

PRO fitsExtension::pad2880, iErr=iErr
;********************************************************************
;#procedure#
; pad2880
;#call#
; fitsExtObj->pad2880, iErr=iErr
;#description#
; Pad disk data to 2880 byte records
; Upon finishing writing extension data to disk, pad data
; out to 2880 byte records by calling FITSFILE padding routine
;#outputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ iErr }{ int    }{ optional:Return code, 0 => OK, else failed}
;#end_procedure#
;********************************************************************
; check here whether file is really open etc.
;
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
   iErr = 0
   if (OBJ_VALID(self.file)) then begin
      (self.file)->pad2880, rest, iErr=iErr
      if (iErr EQ 0) then begin
         self.dataEnd = self.dataEnd + rest
         self.dataCurrent = self.dataEnd
      endif
   endif else midiSetError, 'fitsfile not opened '
RETURN
END

PRO fitsExtension::detach
   self.file = obj_new()
RETURN
END

PRO fitsExtension::close, iErr=iErr
;********************************************************************
;#procedure#
; close
;#call#
; fitsExtObj->close, iErr=iErr
;#description#
; Close extension file
; i.e. the disk file containing this extension
;#outputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ iErr }{ int    }{ optional: Return code, 0 => OK, else failed}
;#end_procedure#
;********************************************************************
   iErr = 0
   if (OBJ_VALID(self.file)) then self.file->close, iErr=iErr
   if (iErr NE 0) then midiSetError, 'close error'
   self.file->setUnit,0
   RETURN
END

PRO fitsExtension::addPar, name, value, iErr=iErr, comment=comment, $
   BEFORE=BEFORE, AFTER=AFTER, FORMAT=FORMAT
;********************************************************************
;#procedure#
; addPar
;#call#
; fitsExtObj->addPar, name, value, iErr=iErr, comment=comment, $
;   BEFORE=BEFORE, AFTER=AFTER, FORMAT=FORMAT
;#description#
; Add/modify a parameter in extension header.
;  Pass through interface to add/modify keyword records to
;  the extension header which is a member of the current extension
; this version works only before the header has been committed to disk
;#inputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ name }{ string }
;   { String name of the parameter to set.
;     If NAME is already in the header the value and possibly comment
;     fields are modified.  Otherwise, a new record is added to the header.
;     If NAME is equal to either "COMMENT" or "HISTORY" then the value
;     will be added to the record  without replacement.
;     In this case the comment parameter is ignored}
;\anArg{ value}{ ?      }
;   { Value for parameter.  The value expression must be of the correct
;     type, e.g. integer, floating or string.
;     String values of 'T' or 'F' are considered logical values.
;     logical values.}
;\anArg{ comment }{     }
;   { String field.  The '/' is added by this routine.
;     Added starting in position 31. If not supplied, or set equal
;     to ''  (the null string), then any previous comment field in
;     the header for that keyword is retained (when found).}
;\anArg{ before }{ string }
;   { Keyword string name.  The parameter will be placed
;     before the location of this keyword.  For example,
;     if BEFORE='HISTORY' then the parameter will be placed
;     the first history  location.  This applies only when
;     adding a new keyword;  keywords already in the header
;     are kept in the same position.}
;\anArg{ after }{ string }
;   { Same as BEFORE, but the parameter will be placed after the
;     location of this keyword.  This keyword takes precedence over
;     BEFORE.}
;\anArg{ format }{ string }
;   { Specifies FORTRAN-like format for parameter, e.g.
;     "F7.3".  A scalar string should be used.
;      For complex numbers the format should be defined so
;      that it can be applied separately to the real and
;      imaginary parts.}
;#outputs#
;\anArg{ iErr=iErr }{ int    }{ optional:Return code, 0 => OK, else failed}
;#end_procedure#
;********************************************************************
  if (self.HeaderStatus ne 'INTERNAL') then $
     midiSetError, 'cant add new keywords if status ne INTERNAL'
  self.head->addPar,name, value, iErr=iErr, comment=COMMENT, $
     BEFORE=BEFORE, AFTER=AFTER, FORMAT=FORMAT
RETURN
END

FUNCTION fitsExtension::getPar, name, matches, comment=comments
;********************************************************************
;#function#
; getPar
;#call#
; parValue = fitsExtObj->getPar (name, matches, comment=comments)
;#description#
; Read header keyword value from header.
; Pass through interface to retrieve keyword records from
; the internally stored extension header
; If the parameter is complex, double precision, floating point, long or
; string, then the result is of that type.  Apostrophes are stripped from
; strings.  If the parameter is logical, 1 is returned for T, and 0 is
; returned for F.
; If NAME was of form 'keyword*' then a vector of values are returned.
;
; The system variable !err is set to -1 if parameter not found, 0 for a
; scalar value returned.  If a vector is returned it is set to the number
; of keyword matches found.
;
; If a keyword occurs more than once in a header, a warning is given,
; and the first occurence is used.  However, if the keyword is "HISTORY",
; "COMMENT", or "        " (blank), then multiple values are returned.
;#inputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ name }{ string } {
;    String name of the parameter to return.
;     If NAME is of the form 'keyword*' then an array
;     is returned containing values of keywordN where
;     N is an integer. The value of keywordN will be
;     placed in RESULT(N-1).  The data type of RESULT will
;     be the type of the first valid match of keywordN found.}
;#outputs#
;\anArg{ matches  }{int     }{ Number of matches found}
;\anArg{ comments }{ strarr }
;   { Array of comments associated with the returned values.}
;#return#
;\anArg{-}{ array}{the value(s) associated with the requested keyword in the
;   header array.}
;#end_function#
;********************************************************************
RETURN,self.head->getPar(name, matches, comment=comments)
END

PRO fitsExtension::replaceHeader, newHead, iErr=iErr
;**************************************************************
;#procedure#
; replaceHeader
;#call#
; fitsExtObj->replaceHeader, newHead, iErr=iErr
;#description#
;  Replace Extension Header
;  Replace the existing internal header object with
;  a new one.  This is very dangerous because things
;  may become inconsistent.  It is a cluge to make
;  "fitsTable::clone" work because IDL doesn't allow
;  copying of objects and should not be used otherwise
;#inputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ newHead }{ fitsHeader Obj }{new header to insert in extension object}
;#outputs#
;\anArg{ iErr=iErr }{ int    }{ optional:Return code, 0 => OK, else failed}
;#end_procedure#
;**************************************************************
   iErr = 0
   if (OBJ_VALID(newHead)) then begin
      if (OBJ_CLASS(newHead) EQ 'FITSHEADER') then begin
         obj_destroy, self.head
         self.head = newHead
      endif
   endif else iErr = 1
RETURN
END

