;
; fbt_read.pro
; Created:     Mon Jun 11 15:53:16 2007 by Rkoehler@lx40
; Last change: Fri Oct  9 17:22:17 2009
;
; Copyright 2008 Rainer Koehler
;
; This file is part of Pacmart.
;
; Pacmart is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; Pacmart is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with Pacmart; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;
; PURPOSE:
;	deal with fitsfiles that contain several binary tables
;
; ACRONYM LIST:
;	FBT	Fits Binary Table
;	FITS	Friedrich's Image Transport System
;
; SEE ALSO:
;	the non-existant documentation about all those FBT-objects
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; fbt_inspect:
; simple routine to print a list of extensions in a FitsFile
;
; Inputs:  filename
; Outputs: Extension names printed to screen
;	   NO data is returned
;
PRO fbt_inspect, filename, extpatt, PRIHEAD=prihead, HEADER=header,$
                 DATA=data, ROWS=rows, PLOT=plot, RETURNDATA=rdata, HELP=help

  if keyword_set(help) then begin
      print,"USAGE: fbt_inspect [,filename [,extpatt]] [,/PRIHEAD] [,/HEADER]"
      print,"			[,/DATA] [,ROWS=array] [,PLOT=array]"
      print,""
      print," filename is the name of the file."
      print,"          A file selection dialog pops up if it's omitted."
      print," extpatt is a regular expression specifying the extensions to display"
      print,"         in detail. If it's omitted, only the extension names are shown"
      print," /HEADER prints the header of the extensions matching extpatt"
      print," /DATA prints the data of extensions matching extpatt (this can be A LOT!)"
      print," ROWS gives an array of row numbers to print with /DATA"
      print," PLOT gives the name of the columns to plot, first x, then y"
      return
  endif

  if N_elements(filename) le 0 then begin
      filename = dialog_pickfile(Default_Ext=".fits",/Must_exist, /REad,$
                                 Filter=["Pac?a*.fits", "PRIMA*.fits", "P*.fits", "*.fits", "*"])
      if filename eq "" then return
      print,"Reading ",filename
  endif
  ffile= obj_new('FITSFILE',filename[0])
  unit = ffile->unit()

  if keyword_set(prihead) then fitsheader_print,(ffile->prihead())->sthead()

  while not eof(ffile->unit()) do begin
      extname = ffile->whichext()
      if N_params() gt 1 then begin
          if stregex(extname,extpatt,/boo) then begin
              print,"================================================================================"
              print, extname
              point_lun,-unit, here
              ext = fitstable_openr(ffile,extname)
              if keyword_set(header) then fitsheader_print,(ext->head())->sthead()
              if keyword_set(data) then begin
                  if keyword_set(rows) then rdata= ext->readrows(rows) $
                  else rdata= ext->readrows()
                  help,rdata,/str
                  if keyword_set(plot) then begin
                      tags = tag_names(rdata[0])
                      xt = where(tags eq strupcase(plot[0]))
                      if xt lt 0 then begin
                          print, 'No tag "',plot[0],'"'  &  xt=0  &  plot[0]= tags[0]
                      endif
                      yt = where(tags eq strupcase(plot[1]))
                      if yt lt 0 then begin
                          print, 'No tag "',plot[1],'"'  &  yt=1  &  plot[1]= tags[1]
                      endif
                      plot, rdata.(xt), rdata.(yt), xtitle=plot[0], ytitle=plot[1], title=filename
                  endif else $
                    print,rdata

              endif else begin
                  rdata= ext->readrows(1)
                  help,rdata,/str
              endelse
              obj_destroy, ext
              point_lun, unit, here
          endif
      endif else $
        print, extname
      ffile->skipext
  endwhile
  obj_destroy, ffile
end
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; fbt_open:
; opens a FitsFile and all the Extensions in it.
;
; Inputs:  filename
;
; Outputs:
; a struct with the following fields:
; - filename: name of the file
; - fileobj:  FITSFILE object created when the file was opened
; - The remaining fields are created dynamically from the extension
;   names found in the Fits file.  They contain references to
;   FITSTABLE objects.
;
; This struct should be deallocated with fbt_close (see below)
;
; Note: This cannot be an object because the contents of objects
;	cannot be created dynamically from the extension names found
;	in a Fits file.
;
FUNCTION fbt_open, filename, VERBOSE=verbose

  if N_elements(filename) le 0 then begin
      filename = dialog_pickfile(Default_Ext=".fits",/Must_exist, /REad,$
                                 Filter=["Pac?a*.fits", "PRIMA*.fits", "P*.fits", "*.fits", "*"])
      if filename eq "" then return,0
  endif
  filename= filename[0]
  ffile = obj_new('FITSFILE',filename,iErr=ierr)
  if ierr ne 0 then begin
      print,"fbt_open err: ",ierr
      return,0
  endif
  st = { filename: filename, fileobj: ffile }

  unit = ffile->unit()
  point_lun,-unit, here ;;& print,"fbt_open is here:",here,". Let's rewind!"
  point_lun, unit, 0
  ffile->skipext
  point_lun,-unit, here ;;& print,"skipped ext, fbt_open is here:",here

  extcnt = 1

  ;;print,"Looking for extensions..."
  while not eof(ffile->unit()) do begin
     ;;print,"fbt_open looking for ext"
     iErr = 0
     extname= ffile->whichext(iErr=iErr)

     if iErr eq 0 then begin
        if extname eq '0' then begin
           ;; Fitsfile::whichext() returns a string,
           ;; even if there is no EXTNAME
           extname = 'EXTENSION_'+strtrim(extcnt,2)
        endif
        if keyword_set(verbose) then print,"Found extension ",extname

        unit = ffile->unit()
        point_lun,-unit, here   ;& print,"fbt_open is here:",here,". opening...!"
        extobj = fitstable_openr(ffile,extname)
        point_lun, unit, here
        structname= IDL_ValidName(extname,/convert_all)

        st = create_struct(st, structname, extobj)
        point_lun,-unit, here   ;& print,"fbt_open is here:",here,". skipping...!"
     endif
     ffile->skipext
     extcnt++
     point_lun,-unit, here      ;& print,"skipped ext, fbt_open is here:",here
  endwhile
  return,st
end
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; fbt_close:
; cleans up a struct returned by fbt_open and destroys all the objects
; in it.  The object references are replaced by null objects.
;
; Inputs:  the struct to be deallocated
;
; Outputs: none
;
PRO fbt_close, st, VERBOSE=verbose
  tnames = tag_names(st)
  ;;print,Format='("Destroying",$)'
  for i=0,N_elements(tnames)-1 do begin
      type = size( st.(i), /type)
      if type eq 11 then begin
          if keyword_set(verbose) then print,"destroying ",tnames[i]
          obj_destroy, st.(i)
          st.(i) = obj_new()
      endif
  endfor
end
;
; This is the end
