;
; prima_tables.pro
; Created:     Wed Jan 17 11:25:50 2007 by Rkoehler@lx40
; Last change: Mon Jul  4 17:16:01 2011
;
; PURPOSE:
;	tabulate the world
;
; Copyright 2008 Rainer Koehler
;
; This file is part of Pacmart.
;
; Pacmart is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; Pacmart is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with Pacmart; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; open a fitstable for reading - ANY fitstable
;;
Function fitstable_openr, filename, extname, iErr=iErr

  if stregex(extname,'EXTENSION_[0-9]+',/Boolean) then begin
     extNum = strmid(extname,10)
     print,'Extension no.',extnum
     tabobj= obj_new('FITSTABLE',filename,extNum=extnum,iErr=iErr)
  endif else $
     tabobj= obj_new('FITSTABLE',filename,extName=extname,iErr=iErr)

  if iErr then print,"creation of FITS-table from disk failed: iErr =",iErr
  return,tabobj
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; fitsheader_print
;; print fitsheader without padding lines
;;
PRO fitsheader_print, head

  print,"--------------------------------------------------------------------------------"
  i = 0
  repeat begin
      print, head[i]
      i++
  endrep until (i ge N_elements(head) or strmid( head[i-1],0,8 ) eq 'END     ')
  print,"--------------------------------------------------------------------------------"
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; fitsdump - dump content of a fits-file to stdout
;;
;; Args:
;;  filename - file name
;;  extname  - optional extension name
;;
PRO fitsdump, filename, extName

  if N_elements(extName) le 0 then extName= ''

  fitsFile= obj_new('FITSFILE', filename, 'READ')

  head = fitsFile->stHead()
  print,"------------------------------------------------------------------"
  print," PRIMARY HEADER"
  fitsheader_print, head

  ;;useless: print, "File has ", fitsFile->numExt()," extensions"
  Extlist = fitsFile->listExtensions()
  ;print, "Extensions are ", Extlist

  for iExt = 1,N_Elements(Extlist)-1 do begin
      if extName eq '' or strmatch(Extlist[iExt],extName,/Fold_Case) then begin

          print," Extension ", Extlist[iExt]
          ext = fitstable_openr( filename, Extlist[iExt] )

          head= (ext->head())->sthead()
          fitsheader_print, head

          obj_destroy, ext
      endif
  endfor
  obj_destroy, fitsFile
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; routines to help reading specific tables
;;
Function read_imaging_data, filename, fsu

  if fsu ne 'A' and fsu ne 'B' then begin
      print, "USAGE: read_imaging_data, <filename>, A|B"
      return,0
  endif
  imgtab = fitstable_openr(filename, 'IMAGING_DATA_FSU'+fsu)
  tabdat = imgtab->readrows()
  obj_destroy, imgtab
  return, tabdat
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; create any fitstable from a template-row
;; this is not an object-method
;;
;; arguments:
;;  extName = name of the FITStable
;;  template= struct representing one table-row (optional)
;;	      if template is omitted, extName+"_row" will be used
;;  MJD     = Number to put as 'MJD-OBS' in table header (optional)
;;
Function fitstable_create, extName, template, iErr=iErr, MJD=MJD

  if N_elements(template) le 0 then $
    call_procedure, extName + '_row__define', template, units

  ;; construct the class by calling the Fitstable-constructor with the
  ;; correct template and extension-name
  tabobj= obj_new('FITSTABLE', template, extName=strupcase(extName), iErr=iErr)
  if iErr then begin
	print,"creation of FITS-table ",extName," failed, iErr =",iErr
	return,tabobj
  endif

  ;; units should contain column names, unit names, and comments
  if N_Elements(units) gt 0 then $
    tabobj->addUnits, units[0,*], units[1,*], units[2,*]

  if keyword_set(MJD) then (tabobj->head())->addPar,'MJD-OBS',MJD,FORMAT="F15.8"

  return,tabobj
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Define templates for the vaious PRIMA-tables
;;
;; based on Wallander-Memo from Dec 2005
;; and J.-DICD from FDR
;;
;; these are dual-use procedures:
;; - used for IDL's automatic struct definition
;; - called directly to get an instance of the struct
;;   (create_struct(name='...') is not available in IDL 6.0)

PRO array_description_row__define, r, u
r={array_description_row,$
   bintable:    '123456789.123456789.123456789.12',$	;; 32 chars
   opti_name:   '123456789.123456789.123456789.12',$	;; 32 chars
   value:	   1,$
   description: '000000000.111111111.222222222.333333333.444444444.555555555.666666666.777777777.888888888.999999999.aaaaaaaaa.bbbbbbbbb.cccccccc',$
   reference:   '123456789.123456789.123456789.123456789.123456789.123456789.123456789.123456789.123456789.123456789.123456789.123456789.12345678' $
  }
end

PRO ARRAY_GEOMETRY_row__define, r, u
  r={ ARRAY_GEOMETRY_row,$
      Tel_Name: '12345678',$
      Sta_Name: '12345678',$
      Sta_Index: 1,$
      Diameter:  1.0,$
      StaXYZ: [ 1.D, 2.D, 3.D ],$
      MNTSTA: 0 $
    }
  u= [["Diameter", "m", " meter"],$
      ["StaXYZ",   "m", " meter"] ]
end

PRO OPTICAL_TRAIN_row__define, r, u
r={ OPTICAL_TRAIN_row,$
    Index: 1,$
    Tel_Name: 'INGO',$
    OPTI_NAME1: '123456789.123456789.123456789.12',$	;; 32 chars
    Value1: 3  $
  }
end

;;;;;;;;;;;;;;;;;;;;;;

PRO main_delay_row__define, row, units
  row={ main_delay_row, $
        time:   0L,  $
        delay1: 0.D0,$
        delay2: 0.D0 $
      }
  units= [["time",  "usec"," itzy-bitzy microseconds"],$
          ["delay1","m",   " meter"		],$
          ["delay2","m",   " meter"		]]
end

;;;;;;;;;;;;;;;;;;;;;;

PRO imaging_detector_FSU_row__define, row, units
  row={ imaging_detector_FSU_row, $
        region: 4, $
        ports: [1,2], $
        correlation: 3, $
        regname: 'Y', $
        corner: [1,1], $
        naxis:  [1,6] $
      }
  units=[["corner","pix","pixels"],$
         ["naxis", "pix","pixels"] ]
end

PRO imaging_detector_FSUA_row__define, row, units
  imaging_detector_FSU_row__define, row, units
end

PRO imaging_detector_FSUB_row__define, row, units
  imaging_detector_FSU_row__define, row, units
end

PRO imaging_data_FSU_row__define, row, units
  row={ imaging_data_FSU_row, $
        time: 0L, $
        data1: fltarr(6), $
        data2: fltarr(6), $
        data3: fltarr(6), $
        data4: fltarr(6), $
        stepping_phase: 0, $
        opd: 0.d, $
        opdsnr: 0.d, $
        gd:  0.d  $
      }
  units=[["time", "usec","mircoseconds"],$
         ["data1","ADU","counts"],$
         ["data2","ADU","counts"],$
         ["data3","ADU","counts"],$
         ["data4","ADU","counts"],$
         ["opd",   "m", "m"],$
         ["opdsnr","m", "m"],$
         ["gd",    "m", "m"] $
        ]
end

PRO imaging_data_FSUA_row__define, row, units
  imaging_data_FSU_row__define, row, units
end

PRO imaging_data_FSUB_row__define, row, units
  imaging_data_FSU_row__define, row, units
end

;;;;;;;;;;;;;;;;;;;;;;

PRO metrology_data_row__define, row, units
  row={ metrology_data_row, $
        time:   0L, $
        deltal: 0.D0,$
        status: 319L $
      }
  units=[["time",  "usec"," microseconds"],$
         ["deltal","m",   " meter" ]]
end

;;;;;;;;;;;;;;;;;;;;;;

PRO metrology_data_fsub_row__define, row, units
  row={ metrology_data_fsub_row, $
        time:   0L, $
        deltal: 0.D0,$
        status: 319L $
      }
  units=[["time",  "usec"," microseconds"],$
         ["deltal","m",   " meter" ]]
end

;;;;;;;;;;;;;;;;;;;;;;

PRO DDL_row__define, row, units
  row={ DDL_row, $
        time:   0L, $
        delay1: 0.d,$
        delay2: 0.d,$
        delay3: 0.d,$
        delay4: 0.d $
      }
  units=[["time",  "usec","microseconds"],$
         ["delay1","m",   "meter"	],$
         ["delay2","m",   "meter"	],$
         ["delay3","m",   "meter"	],$
         ["delay4","m",   "meter"	]]
end

;;;;;;;;;;;;;;;;;;;;;;

PRO OPDC_row__define, row, units
  row={ OPDC_row, $
        time:     0L,$
        RTOffset: 0.,$
        FringeFlag: 0,$
        OffValid:   0,$
        OPDCState:  0L,$
        State:      0L,$
        Status:     0L,$
        UWPhase:    0.,$
        FullOffset: 0.,$
        DLFDBACK:   0.d,$ ;; Delay Line FeeDBACK
        stepping_phase: 0,$
        stepping: 0,$
        phase:    0 $
      }
  units=[["time",    "usec","microseconds"],$
         ["RTOffset",  "m", "metre"],$
         ["UWPhase",   "m", "meter"],$
         ["FullOffset","m", "meter"]]
end

PRO DOPDC_row__define, row, units
  OPDC_row__define, row, units
  row=create_struct(row,NAME='DOPDC_row')
end

;;;;;;;;;;;;;;;;;;;;;;

PRO environment_description_row__Define, row, units
  row={ environment_description_row, $
        id:	0,$
        name:     '123456789.123456789.123456789.12',$	;; 32 chars
        type:     '123456789.123456789.123456789.12',$
        location: '123456789.123456789.123456789.12',$
        unit:     '123456789.123456789.123456789.12',$
        comment:  '123456789.123456789.123456789.12',$
        ccsenv:   '12345678', $
        keyword:  '123456789.123456789.123456789.12',$
        dict:     '123456789.123456789.',$
        hw_id:    '123456789.123456' $
      }
end

PRO environment_row__Define, row, units
  row={ environment_row, $
        sensor: 0,  $
        time:   0L, $
        value:  0., $
        status: 0b  $
      }
  units=[["time","sec","full-grown seconds"]]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO spectrum_row__DEfine, row, units
  row={ spectrum_row,$
        wavenumber:   [1d,2d,3d,4d,5d,6d],$
        spectrum:     [1d,2d,3d,4d,5d,6d],$
        spectrum_err: [1d,2d,3d,4d,5d,6d] $
      }
  units=[["wavenumber",'1/m',"inverse meter"]]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
