;
; rkplot.pro
; Created:     Fri Feb 23 16:43:24 2007 by Koehler@rommie
; Last change: Fri Dec 10 14:12:53 2010
;
; Copyright 2008 Rainer Koehler
;
; This file is part of Pacmart.
;
; Pacmart is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; Pacmart is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with Pacmart; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;
; PURPOSE:
;	save the world (if it's not too inconvenient)
;
; CALLING SEQUENCE:
;	rkplot, x, y, PSYM=psym, LINESTYLE=lstyle, COLOR=color, X2=x2, Y2=y2,$
;		Title=title, Xtitle=xtitle, Ytitle=ytitle
;
; INPUT:
;
;  x:	array of x-values
;	if x is 1-dim., then all plots will use the same x-coordinates
;	if 2-dim, then x[i,*] will be used for plot i.
;	all plots must have the same number of points!
;
;  y:	array of y-values
;	if 2-dim, then multiple graphs y[i,*] will be plotted
;	all plots must have the same number of points!
;
; OUTPUT:
;	new window with the plot(s)
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; event handler
;;
PRO rkplot_event, ev
  WIDGET_CONTROL, ev.TOP, GET_UVALUE=me

  ;; show only press-events:
  ;if ev.type eq 0 then help,ev,/str

  if !D.name eq "X" then wset,me.drawID

  ;;geo = Widget_Info(me.draw, /Geometry)
  ;;print, "size: ", geo.xsize, geo.ysize

  xyclick = convert_coord( ev.x, ev.y, /device, /to_data)
  xclick = xyclick[0]
  yclick = xyclick[1]
  if ev.type eq 0 then print, "click at ",xclick,yclick, " old: ",me.oclick

  redraw = 0

  if ev.type eq 2 then begin
     ;; motion event - drag plot
     ;;print,"Motion x:",me.oclick[0]," ->",xclick," y:",me.oclick[1]," ->",yclick
     me.cur_xrange -= xclick - me.oclick[0]
     me.cur_yrange -= yclick - me.oclick[1]
     redraw = 1
  endif
  if ev.type eq 0 or ev.type eq 1 then begin
     print,"Button press event:",ev.press
     ;; button press/release
      if ev.press eq 4 then begin
         print,"Change plot style"
         print,"before: ",me.lstyle
         rkplot_style,me
         print," after: ",me.lstyle
         redraw = 1
      endif else if ev.press eq 2 then begin
          print, "center point under cursor"
          me.cur_xrange = me.cur_xrange + (xclick - (me.cur_xrange[0]+me.cur_xrange[1])/2.)
          me.cur_yrange = me.cur_yrange + (yclick - (me.cur_yrange[0]+me.cur_yrange[1])/2.)
          redraw = 1
      endif else if ev.press eq 1 then begin
          ;; begin dragging
          me.oclick = xyclick[0:1]
          widget_control, me.draw, DRAW_MOTION_EVENTS=1
          WIDGET_CONTROL, ev.top, SET_UVALUE=me
      endif else if ev.release eq 1 then begin
          ;; end dragging
          me.cur_xrange -= xclick - me.oclick[0]
          me.cur_yrange -= yclick - me.oclick[1]
          widget_control, me.draw, DRAW_MOTION_EVENTS=0
          redraw = 1
      endif else if ev.press eq 8 then begin
          ;; old wheel event: zoom out
          me.cur_xrange = 2.*me.cur_xrange - xclick
          me.cur_yrange = 2.*me.cur_yrange - yclick
          redraw = 1
      endif else if ev.press eq 16 then begin
          ;; old wheel event: zoom in
          me.cur_xrange = (xclick + me.cur_xrange)/2.
          me.cur_yrange = (yclick + me.cur_yrange)/2.
          redraw = 1
      endif
  endif else if (ev.type eq 5 or ev.type eq 6) and ev.press eq 1 then begin
      switch ev.ch of
          (byte('h'))[0]:
          (byte('?'))[0]: begin
              print,"rkplot by Rainer Koehler"
              print,'------------------------'
              print,"Mouse control:"
              print,"  Button-1:		drag the visible part of the plot"
              print,"  Button-2:		center point under mouse"
              print,"  Button-3:		open plot style dialog"
              print,"  Wheel:		zoom in/out"
              print,"    + Shift key:	zoom y-axis only"
              print,"    + Control key:	zoom x-axis only"
              print,''
              print,'Keys:'
              print,'  left/right/up/down:	shift visible part of the plot'
              print,'  ">" / "<":		zoom in/out'
              print,'  "l" / "k":		zoom x-axis in/out'
              print,'  "o" / "i":		zoom y-axis in/out'
              print,'  "1" / "r":		reset zoom'
              print,'  "h" / "?":		print this help'
              print,'  "Q" (with Shift):	Quit'
              print,''
              break
          end
          (byte('o'))[0]:
          (byte('l'))[0]:
          (byte('.'))[0]:
          (byte('>'))[0]: begin
              ;; make larger = zoom in
              if ev.ch ne (byte('o'))[0] then me.cur_xrange = (xclick + me.cur_xrange)/2.
              if ev.ch ne (byte('l'))[0] then me.cur_yrange = (yclick + me.cur_yrange)/2.
              redraw = 1
              break
          end
          (byte('i'))[0]:
          (byte('k'))[0]:
          (byte(','))[0]:
          (byte('<'))[0]: begin
              ;; make smaller = zoom out
              if ev.ch ne (byte('i'))[0] then me.cur_xrange = 2.*me.cur_xrange - xclick
              if ev.ch ne (byte('k'))[0] then me.cur_yrange = 2.*me.cur_yrange - yclick
              redraw = 1
              break
          end
          (byte('1'))[0]:
          (byte('r'))[0]: begin
              me.cur_xrange = me.max_xrange
              me.cur_yrange = me.max_yrange
              redraw = 1
              break
          end
          (byte('Q'))[0]: begin
              WIDGET_CONTROL, ev.TOP, /DESTROY
              rkplot_cleanup, ev.top
              break
          end
          (byte('p'))[0]: begin
              filename = Dialog_Pickfile(Dialog_parent=ev.top,Title="Select file for printing",$
                                         File="rkplot.ps",Default_Extension="ps", Filter=["*.ps"],$
                                         /Write,/Overwrite_prompt)
              if filename eq "" then return
              set_plot,"PS"
              ;; NOTE: printing in color doesn't work, because PS
              ;; does not understand 24-bit colors (I think)
              device,/landscape,file=filename,/color,bits_per_pix=8,decomposed=1
              rkplot_plot, me
              device,output="% Created by rkplot.pro"
              device,/close
              set_plot,"X"
              spawn,"gv "+filename+"&"
              break
          end
          ELSE: begin
              case ev.key of
                  5: begin          ;; left
                      me.cur_xrange -= (me.cur_xrange[1] - me.cur_xrange[0])/10.
                      redraw = 1
                  end
                  6: begin	;; right
                      me.cur_xrange += (me.cur_xrange[1] - me.cur_xrange[0])/10.
                      redraw = 1
                  end
                  7: begin	;; Up
                      me.cur_yrange += (me.cur_yrange[1] - me.cur_yrange[0])/10.
                      redraw = 1
                  end
                  8: begin	;; right
                      me.cur_yrange -= (me.cur_yrange[1] - me.cur_yrange[0])/10.
                      redraw = 1
                  end
                  else: help, ev, /str
              endcase
          end
      endswitch
  endif else if (ev.type eq 7) then begin
      help,ev,/str
      ;; new wheel events
      ;; modifiers: Shift  = 1 => zoom y only
      ;;	    Control= 2 => zoom x only
      if ev.clicks lt 0 then begin
          ;; zoom out
          if (ev.modifiers AND 1) eq 0 then me.cur_xrange = 2.*me.cur_xrange - xclick
          if (ev.modifiers AND 2) eq 0 then me.cur_yrange = 2.*me.cur_yrange - yclick
          redraw = 1
      endif else begin
          ;; zoom in
          if (ev.modifiers AND 1) eq 0 then me.cur_xrange = (xclick + me.cur_xrange)/2.
          if (ev.modifiers AND 2) eq 0 then me.cur_yrange = (yclick + me.cur_yrange)/2.
          redraw = 1
      endelse
  endif

  if redraw ne 0 then begin
      if me.cur_xrange[0] lt me.max_xrange[0] then begin
          ;; push so that cur_min == max_min
          me.cur_xrange += me.max_xrange[0]-me.cur_xrange[0]
          if me.cur_xrange[1] gt me.max_xrange[1] then me.cur_xrange= me.max_xrange
      endif
      if me.cur_xrange[1] gt me.max_xrange[1] then begin
          me.cur_xrange += me.max_xrange[1]-me.cur_xrange[1]
          if me.cur_xrange[0] lt me.max_xrange[0] then me.cur_xrange= me.max_xrange
      endif
      if me.cur_yrange[0] lt me.max_yrange[0] then begin
          ;; push so that cur_min == max_min
          me.cur_yrange += me.max_yrange[0]-me.cur_yrange[0]
          if me.cur_yrange[1] gt me.max_yrange[1] then me.cur_yrange= me.max_yrange
      endif
      if me.cur_yrange[1] gt me.max_yrange[1] then begin
          me.cur_yrange += me.max_yrange[1]-me.cur_yrange[1]
          if me.cur_yrange[0] lt me.max_yrange[0] then me.cur_yrange= me.max_yrange
      endif
      ;print, "xrange:", me.cur_xrange
      ;print, "yrange:", me.cur_yrange
      WIDGET_CONTROL, ev.top, SET_UVALUE=me

      rkplot_plot, me
  endif
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO rkplot_plot, me
  if !D.name eq "X" then wset, me.drawID

  ;;from tek_colors.pro:
  ;;bytscl([ 0,100,	100,	    0,		0,	  0,		100,	100
  ;;bytscl([ 0,100,	0,	    100,	0,	  100,		0,	100
  ;;bytscl([ 0,100,	0,	    0,		100,	  100,		83,	0
  ;;	black  white      red        green      blue       Cyan       Magenta    Orange
  coltab= [0L,'ffffff'x, '0000ff'x, '00ff00'x, 'ff0000'x, 'ffff00'x, 'd400ff'x, '00ffff'x]

  if !D.name eq 'PS' then coltab[1]= 0L

  ;; first an empty plot with white axes:
  plot, [0,1],[0,1],/noData, /xstyle,/ystyle, xrange=me.cur_xrange, yrange=me.cur_yrange,$
        XTitle=me.xtitle, YTitle=me.ytitle

  ;;help,me,/str
  ;;print,size(me.x,/N_dim)

  for i=0,me.Nplots-1 do begin
     if size(me.x,/N_dim) le 1 then x=me.x  else x=me.x[i,*]
     if         me.Nplots le 1 then y=me.y  else y=me.y[i,*]

     print,"Plot",i,", color=",coltab[me.color[i]]

     if me.lstyle[i] gt 0 then $
        oplot, x, y, LineStyle=me.lstyle[i]-1, COLOR=coltab[me.color[i]]

     if me.psym[i] gt 0 then $
        oplot, x, y, PSym=me.psym[i], COLOR=coltab[me.color[i]]

     if i lt N_elements(me.labels) then XYouts, x[0],y[0], me.labels[i]
  endfor

  if N_elements(me.x2) gt 1 then oplot, me.x2, me.y2, PSYM=me.psym[1], color='0000ff'x
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; dialog to change plot styles
;;
PRO rkplot_style, me

  desc= ['1, BASE,, Row']

  for i=0,me.Nplots-1 do begin
     gr= 'g'+strtrim(i,2)
     desc= [ desc,$
             '0, DROPLIST, None|Solid|Dotted|Dashed|Dash Dot|Dash Dot Dot|Long Dash, '+$
		'Set_Value='+string(me.lstyle[i]) + ', LABEL_Left='+me.labels[i], $
             '0, DROPLISt, None|Plus|Asterisk|Period|Diamond|Triangle|Square|X, '+$
             	'Set_value='+string(me.psym[i]), $
             '2, DROPLIst, White|Red|Green|Blue|Cyan|Magenta|Yellow, '+$
		'Set_Value='+string(me.color[i]-1),$
             '1, BASE,, Row' ]
  endfor

  desc[1] += ', Label_top=Linestyle'
  desc[2] += ', Label_top=Symbol'
  desc[3] += ', Label_top=Color'

  desc= [ desc,$
          '0, BUTTON, | Cancel |, QUIT, TAG=Cancel',$
          '0, BUTTON, |  Okay  |, QUIT, TAG=OK' ]

  res = cw_form(desc,/Column,Title='rkplot style '+me.title)
  ;;help,res,/str

  if res.ok then begin
     print,"You clicked OK"
     tag=0
     for i=0,me.Nplots-1 do begin
        me.lstyle[i]= res.(tag)  & tag++
        me.psym[i]  = res.(tag)  & tag++
        me.color[i] = res.(tag)+1 & tag++
     endfor
  endif
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; add a graph to existing plot.
;; please add one graph at a time!
;;
PRO rkplot_add_graph, base, x, y, PSYM=psym, LINESTYLE=lstyle, COLOR=color, LABEL=label

  WIDGET_CONTROL, base, GET_UVALUE=me

  help, me, /str

  if N_params() ge 3 then begin
     szx = size(me.x)
     if szx[0] eq 1 then begin
        ;; one x for everyone
        if max(abs(me.x-x)) gt 0. then begin
           print,"new x given"
           ;; x is different from the old x
           ;; make enough copies for all the old ys
           new_x= make_array(me.Nplots+1,szx[1], type=szx[2])
           for i=0,me.Nplots-1 do new_x[i,*]= me.x
           new_x[me.Nplots,*] = x
        endif else new_x= me.x	;; the new_x is the old x
     endif else begin
        ;; there were already multiple x vectors
        new_x= make_array(me.Nplots+1,szx[2], type=szx[3])
        new_x[0:me.Nplots-1,*]= me.x
        new_x[me.Nplots,*] = x
     endelse
  endif else begin
     y = x		;; there is no x, the x parameter is in fact y
     new_x= me.x	;; the new_x is the old x
  endelse

  szy = size(me.y)
  if szy[0] eq 1 then begin
     ;; we had one graph, add one more
     if szy[1] ne N_elements(y) then begin
        print,"Size of y must be",szy[1]
        return
     endif
     new_y= make_array(2,szy[1], type=szy[2])
     new_y[0,*]= me.y
     newidx= 1
  endif else begin
     ;; there is more than one graph already
     if szy[2] ne N_elements(y) then begin
        print,"Size of y must be",szy[1]
        return
     endif
     new_y= make_array(me.Nplots+1,szy[2], type=szy[3])
     new_y[0:me.Nplots-1,*]= me.y
     newidx= me.Nplots
  endelse
  new_y[newidx,*] = y
  ;;
  ;; copy me struct and sneak in the new x- and y-array
  ;;
  tags= tag_names(me)
  me2 = create_struct( tags[0], me.(0) )
  for i=1,N_elements(tags)-1 do begin
     if tags[i] eq 'X' then $
        me2= create_struct(me2, 'X', new_x) $
     else if tags[i] eq 'Y' then $
        me2= create_struct(me2, 'Y', new_y) $
     else me2= create_struct(me2, tags[i], me.(i))
  endfor
  me2.Nplots++

  if keyword_set(psym)   then me2.psym[newidx]  = psym
  if keyword_set(lstyle) then me2.lstyle[newidx]= lstyle
  if keyword_set(color)  then me2.color[newidx] = color
  if keyword_set(label)  then me2.Labels[newidx]= label
  ;;
  ;; set new max x/yrange

  xmin = min(me2.x, MAX=xmax)
  ymin = min(me2.y, MAX=ymax)

  xext = (xmax-xmin)/20.  &  xmin -= xext  &  xmax += xext
  yext = (ymax-ymin)/20.  &  ymin -= yext  &  ymax += yext

  me2.max_xrange= [xmin,xmax]
  me2.max_yrange= [ymin,ymax]

  help,me2,/str

  WIDGET_CONTROL, base, SET_UVALUE=me2
  rkplot_plot, me2
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; return list of current rkplot-instances
;;
;; optional: return window-names in argument
;;
Function rkplot_instances, names

  common rkplot_ids, rkplot_ids

  if N_elements(rkplot_ids) le 0 then return,-1

  validx= where(widget_info(rkplot_ids,/valid) ne 0)
  if validx[0] eq -1 then return,-1

  rkplot_ids= rkplot_ids[validx]	;; clean up common block while we're at it

  if rkplot_ids[0] gt -1 and N_params() ge 1 then begin
     names = strarr(N_elements(rkplot_ids))
     for i=0,N_elements(rkplot_ids)-1 do $
        names[i]= widget_info(rkplot_ids[i],/uname)
  endif
  return, rkplot_ids
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Main subroutine
;;
PRO rkplot, x, y, PSYM=psym, LINESTYLE=lstyle, COLOR=color, X2=x2, Y2=y2,$
            Title=title, Xtitle=xtitle, Ytitle=ytitle, Labels=labels

  COMMON rkplot_ids, rkplot_ids

  x= reform(x)	;; get rid of degenerate dimensions
  y= reform(y)

  sz = size(y)
  Nplots= sz[0] gt 1? sz[1] : 1

  labels8= 'Graph '+strtrim(indgen(8)+1,2)	;; Graph1 .. 8

  if N_elements(psym) lt 1   then psym= intarr(Nplots)
  if N_elements(lstyle) lt 1 then lstyle= intarr(Nplots)+1
  if N_elements(color)  lt 1 then color = indgen(Nplots)+1
  if N_elements(labels) gt 0 then labels8[0:N_elements(labels)-1]= labels

  if not keyword_set(x2) then x2=0
  if not keyword_set(y2) then y2=0
  if not keyword_set(title)  then title =""
  if not keyword_set(xtitle) then xtitle="x"
  if not keyword_set(ytitle) then ytitle="y"

  ;print,"psym",psym
  ;print,"lstyle",lstyle
  ;print,"colors",color

  xmin = min(x, MAX=xmax)
  ymin = min(y, MAX=ymax)

  xext = (xmax-xmin)/20.  &  xmin -= xext  &  xmax += xext
  yext = (ymax-ymin)/20.  &  ymin -= yext  &  ymax += yext

  print,"x range:", xmin, xmax

  set_plot,"X"
  screensz = get_screen_size()
  base= Widget_Base(Title="rkplot "+title, UName='rkplot '+title, Space=0, /COLUMN, /BASE_ALIGN_LEFT)
  draw= Widget_Draw(base, XSIZE=screensz[0]-16, YSIZE=400,/BUTTON_EVENT,/KEYBOARD_EVENT,/WHEEL_EVENT)

  WIDGET_CONTROL, base, /REALIZE
  WIDGET_CONTROL, draw, GET_VALUE=ID

  data = { draw: draw, drawid: ID, Nplots: Nplots, x: x, y: y, x2: x2, y2: y2,$
           title: title, psym: psym, lstyle: lstyle, color: color, $
           xtitle: xtitle, max_xrange: [xmin,xmax], cur_xrange: [xmin,xmax],$
           ytitle: ytitle, max_yrange: [ymin,ymax], cur_yrange: [ymin,ymax],$
           labels: labels8, oclick: [0.,0.], drag: 0 $
         }
  WIDGET_CONTROL, base, SET_UVALUE=data
  rkplot_plot, data
  XMANAGER, 'rkplot', base, /NO_BLOCK, cleanup=rkplot_cleanup

  if N_elements(rkplot_ids) gt 0 then begin
     if rkplot_ids[0] eq -1 then rkplot_ids= base $
     else rkplot_ids = [ rkplot_ids, base ]
  endif else rkplot_ids= base
end

PRO rkplot_cleanup, id
  COMMON rkplot_ids, rkplot_ids

  ;;print,"rkplot_cleanup",id

  ;; find the valid ids
  notme= where(rkplot_ids ne id and widget_info(rkplot_ids,/valid) ne 0)
  if notme[0] ne -1 then $
     rkplot_ids = rkplot_ids[notme] $
  else rkplot_ids= -1
end
;;
;; the end
