;
; spectral_resp.pro
; Created:     Thu Oct 18 14:56:19 2007 by Rkoehler@lx40
; Last change: Wed Jun  1 14:11:05 2011
;
; PURPOSE:
;	none
;
; TODO:
;	- phase shifts between ABCD not exactly 90deg
;	- finite DIT
;
; Copyright 2008 Rainer Koehler
;
; This file is part of Pacmart.
;
; Pacmart is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; Pacmart is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with Pacmart; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

function blackbody, kappa, dkappa, temper

  ;; kappa and dkappa in meter
  ;; temper in Kelvin

  ;; result is number of PHOTONS normalized to max = 1.0

  planck = 6.62606930d-34	;Js   Planck constant h
  lspeed = 299792450.d0		;m/s  Speed of Light c
  boltz  = 1.38065050d-23	;J/K  Boltzmann constant k

  black = kappa^2 / (exp((planck*lspeed*kappa) / (boltz*temper)) - 1.0) * dkappa

  null = where(kappa eq 0.)
  if null[0] ge 0 then black[null]= 0.

  return, black/max(black)
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO img_data_header, head, active, srctemp, noise

  head->addESOPar, 'ISS DID', 'JICD-1.0', Comment=' Data Dictionary name'
  head->addESOPar, 'ISS ID' , '1.0',	  Comment=' Data Dictionary version number'
  head->addESOPar, 'ISS DIP', '089-3200-6467', Comment=' Data Dictionary phone number'

  head->addPar, 'NREGION', 4,	  Comment=' Number of regions used (always 4)'
  head->addPar, 'MAXTEL', 0,	  Comment=' Maximum number of contributing telescopes'

  if not keyword_set(active) then return

  head->addESOpar, 'SIM SRC TEMP', srctemp, Comment=' Light source coolness [Kelvin]'

  ;;for i=0,N_elements(self.detector_bias)-1 do $
  ;;  head->addESOpar, 'SIM BIAS'+string(i,format="(I02)"),$
  ;;			self.detector_bias[i], Comment=' bias'

  ;;for i=0,N_elements(self.skyback_flux)-1 do $
  ;;  head->addESOpar, 'SIM FLUX'+string(i,format="(I02)"),$
  ;;			 self.skyback_flux[i], Comment=' sky background flux'

  for i=0,N_elements(noise)-1 do $
    head->addESOpar, 'SIM NOISE'+string(i,format="(I02)"),noise[i],Comment=' noise'
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; start_time in usec
;; scan_length in meter
;; delay_shift in meter
;; noise is in percent of the uncorrelated flux
;;
PRO ftscan,tabobj,srctemp,iScan,Npnts,scan_length,delay_shift,FSUrate,noise,tnoise,$
           FSU=fsu, PUTFILE=putfile, PHASE_OFFS=phase_offs, NOSTAND=nostand

  common PACSIM_DEFAULTS	;; contains kappa_center, kappa_width
  common fts_seeds, pix_seed,tseed

  ;;kappa_cs= 1d6 / [2.5, 2.4, 2.3, 2.2, 2.1]	;; [1/m]

  if N_elements(fsu) lt 1 then fsu=1	;; 0 for FSUA, 1 for FSUB
  if N_elements(phase_offs) lt 3 then phase_offs= [ 0.,0.,0. ]	;; offsets from 90deg phase shifts


  tabdat= replicate( {imaging_data_FSU_row}, Npnts)

  start_time= round(Npnts/FSUrate * (iScan-1) * 1d6)	;; usec, iScan starts at 1

  tabdat.time= start_time + round(dindgen(Npnts)*1d6/FSUrate + tnoise*randomn(tseed,Npnts))  ;; usec
  tabdat.stepping_phase= iScan

  if keyword_set(nostand) then begin
     ;; delay is not standing still at the start and end of a slew

     delta_slew= scan_length / double(Npnts-1)	;; meter
     tabdat.opd= delay_shift/1d6 - 0.5*scan_length + dindgen(Npnts)*delta_slew
  endif else begin
     Nslewp= Npnts/4		;; No. of points where we are really slewing
     Nstand= (Npnts-Nslewp)/2	;; No. of points where we are standing still in the beginning

     delta_slew= scan_length / double(Nslewp-1)	;; meter
     tabdat[0:Nstand-1].opd		= delay_shift/1d6 - 0.5*scan_length
     tabdat[Nstand:Nstand+Nslewp-1].opd = delay_shift/1d6 - 0.5*scan_length + dindgen(Nslewp)*delta_slew
     tabdat[Nstand+Nslewp:Npnts-1].opd  = delay_shift/1d6 + 0.5*scan_length
  endelse

  ;;DIT_slew = delta_slew * FSUrate * FSUdit	;; change in delay during one DIT

  kappa_step=  500.
  order = 4

  kappas= dblarr(6,321)	;; N FSU channels * NNN kappa points
  white = dblarr(6,4,321)

  ;; this could be simplified because all kappas are the same,
  ;; thus the blackbodies are the same, too,
  ;; but I'm too lazy to modify and test this now.

  ;;for ch=0,5 do kappas[ch,*]= kappa_cs[ch] + (dindgen(321)-160.)*kappa_step
  for ch=0,5 do kappas[ch,*]= 0.38d6 + dindgen(321)*kappa_step

  ;; do this once to get normalization across channels
  black = blackbody(kappas,kappa_step,srctemp)

  noise /= 100.	;; percent => factor


  for ch=0,5 do begin
     throughput = dblarr(4,321)		;; ABCD * kappa

     for iABCD=0,3 do begin
        kappa_c = kappa_center[ch,iABCD,FSU]	;; & print,"kappa",ch,":",kappa_c
        kappa_h = kappa_width[ch]/2.
        kappa = kappas[ch,*]
        white[  ch,iABCD,*] = 1.d / sqrt( 1.d + ((kappa-kappa_c)/kappa_h)^(2*order))
        throughput[iABCD,*] = 50.d * black[ch,*] * white[ch,iABCD,*]
     endfor

     ;;print,"Channel",ch,", throughput =",total(throughput)
     const = total(throughput) * 1.1 / 4.	;; per ABCD

     if const gt 32760./2. then $
        print,"Warning: Pixel overflow in channel",ch," (",const,")"

     tabdat.data1[ch] = const
     tabdat.data2[ch] = const
     tabdat.data3[ch] = const
     tabdat.data4[ch] = const

     for iKappa=0,320 do begin
        ;;kappa = (iKappa-84.)*kappa_step + kappa_c
        ;;throughput= 2.d2 / sqrt( 1.d + ((kappa-kappa_c)/kappa_h)^(2*order))
        phase = 2.0*!PI * kappa[iKappa] * tabdat.opd

        tabdat.data1[ch] += fix(throughput[0,iKappa] * cos(phase))
        tabdat.data2[ch] += fix(throughput[1,iKappa] * sin(phase+phase_offs[0]))
        tabdat.data3[ch] -= fix(throughput[2,iKappa] * cos(phase+phase_offs[1]))
        tabdat.data4[ch] -= fix(throughput[3,iKappa] * sin(phase+phase_offs[2]))
        ;;debug:
        ;;if ch eq 0 then $
        ;;  print,format='(G10.5,": ",F5.1,", ",F5.1," => ",F8.1)',$
        ;;  	kappa, throughput, kappa*tabdat[Npnts/2+340].opd, tabdat[Npnts/2+340].data1[0]
      endfor
      tabdat.data1[ch] += fix(const*noise*randomn(pix_seed,Npnts))
      tabdat.data2[ch] += fix(const*noise*randomn(pix_seed,Npnts))
      tabdat.data3[ch] += fix(const*noise*randomn(pix_seed,Npnts))
      tabdat.data4[ch] += fix(const*noise*randomn(pix_seed,Npnts))
  endfor

  tabobj->writeRows, tabdat

  if keyword_set(putfile) then save,kappas,white,file=putfile
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO FSUresponse, FILE_PREFIX=file_prefix, STARTTIME=starttime, FSUA=fsua, FSUB=fsub, VIEW=view,$
                 NSCANS=Nscans, NPNTS=Npnts, SCANLEN=scan_length, PHASE_OFFS=phase_offs, $
                 METNOISE=PRIMETnoise, FSUNOISE=FSUnoise, TIMENOISE= Tnoise, DELAYJIT=delay_jitter

  common PACSIM_DEFAULTS	;; contains kappa_center, kappa_width

  print,"===== FSUresponse template ====="

  ;; should work now - PRIMET_B is always negative
  ;;if keyword_set(fsua) and keyword_set(fsub) then begin
  ;;   print,"You can't set FSUA and FSUB simultaneously!"
  ;;   print,"The sign of the metrology is undefined!!!"
  ;;   return
  ;;endif

  if not keyword_set(file_prefix) then file_prefix= 'PACMA'

  MJD0= eso_MJD(starttime)                       ;; year, month, day, hour, min, sec

  MARCELtemp = 700. + 273.15

  if not keyword_set(Nscans) then Nscans= 10
  if not keyword_set(Npnts)  then Npnts = 2^12		;; 4096 points per scan
  ;if not keyword_set(scan_length)  then scan_length= 1.6d-4	;; 160 micron in [m] (\approx real lab test)
  if not keyword_set(scan_length)  then scan_length= 300d-6	;; 300 micron in [m] (better!? than lab)
  if N_elements(PRIMETnoise)  lt 1 then PRIMETnoise= 1d-8	;; [meter]
  if N_elements(FSUnoise)     lt 1 then FSUnoise = 0.5		;; percent of uncorrelated flux
  if N_elements(Tnoise)       lt 1 then Tnoise   = 0.1		;; usec
  if N_elements(delay_jitter) lt 1 then delay_jitter= 1.	;; mircons

  PRIMETrate= 4000.			;; [junk/second]
  FSUrate = 1000.			;; [junk/second]
  FSUdit  = 1./FSUrate - 50.d-6		;; 50 usec for readout
  FSUAdit = keyword_set(fsua) ? FSUdit : 0.
  FSUBdit = keyword_set(fsub) ? FSUdit : 0.
  ;;
  ;; labducks are no longer needed for FSUresponse
  ;;
  ;;if FSUnoise gt 0. then $
  ;;     labdark, MJD=MJD0, FSUA=fsua, FSUB=fsub, BIAS=0.,DARK=0. $
  ;;else labdark, MJD=MJD0, FSUA=fsua, FSUB=fsub, BIAS=0.,DARK=0.,NOISE=0.

  ;;MJD0 += 1./60./24.	;; labdark adds 4 minutes already

  filename = file_prefix + '.' + MJD_to_ESO_string(MJD0) + ".fits"
  print, "File: ",filename
  ;;print, "FSUdits:",FSUAdit,FSUBdit

  pfile= Pacman_binfile(MJD0)

  N_sec = Nscans*Npnts / FSUrate

  delay_shifts= delay_jitter * randomn(delayseed,Nscans,/normal)

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ;;
  ;; PRIMARY header
  ;;
  prihead= pfile.arrdesc->prihead()
  prihead->addPar,'EXPTIME',float(N_sec),Comment= " (seconds)"
  prihead->addESOPar,'ISS CONF NTEL', 0, Comment=' no telescope'

  insmode = 'ABSENT'
  if keyword_set(fsua) then begin
     insmode = keyword_set(fsub) ? 'NORMAL' : 'SINGLE_A'
  end else if keyword_set(fsub) then insmode= 'SINGLE_B'

  prihead->addESOPar,'INS MODE', insmode, COMMENT=" instrument mode"
  prihead->addESOPar,'INS OPDSCAN NSLEW', Nscans, COMMENT=" Number of scan slews in an exposure"
  prihead->addESOPar,'INS OPDSCAN SHAPE', 'saw-tooth', COMMENT=" hard coded"

  standard_primary_header, prihead, FSUAdit, FSUBdit, SIM_CODE='FR'
;;                           CATG='CALIB   ', TECH='INTERFEROMETRY,SPECTRUM', TYPE='FLAT,LAMP'
;;  prihead->addESOPar,'TPL NAME', 'PACMA(N)_cal_lab_FSUresponse', Comment="Template name"

  prihead->addESOpar,'SIM MARCEL TEMP', MARCELtemp, Comment=' MARCEL coolness [Kelvin]'
  prihead->addESOpar,'SIM NUM SCANSLEWS', Nscans, Comment=' Number of scan slews'
  prihead->addESOpar,'SIM SCAN POINTS', Npnts, Comment=' Data points per scan'
  prihead->addESOpar,'SIM MET NOISE', PRIMETnoise, Comment=' [meter]'
  prihead->addESOpar,'SIM FSU NOISE', FSUnoise,    Comment=' [% of uncorr. flux]'
  prihead->addESOpar,'SIM TIME NOISE', Tnoise,     Comment=' [usec]'
  prihead->addESOpar,'SIM DELAY JITTER',delay_jitter, Comment=' [mircons]'

  prihead->addESOpar,"SIM FSU ID", keyword_set(fsub), Comment=" 0=A, 1=B"

  for iCh=0,5 do begin
     for iABCD=0,3 do begin
        prihead->addESOpar,string(Format='("SIM EFFWNUM",I1,I1)',iCh,iABCD),$
                           kappa_center[iCh,iABCD,keyword_set(fsub)], Comment=' [1/m]'
     endfor
  endfor
  ;;
  ;; Tables in DICD-Order --------------------------------------------
  ;;
  Pacman_binfile_create_arrdesc, pfile, filename             ;; Array description
  pfile.arrdesc->writeRows, standard_array_description()

  Pacman_binfile_create_arrgeom, pfile                       ;; Array geometry
  pfile.arrgeom->writeRows, standard_array_geometry()

  Pacman_binfile_create_optrain, pfile                       ;; Optical train
  pfile.optrain->writeRows, standard_optical_train()
  ;;
  ;; Main Delay -------------------------------------------------------
  ;;
  standard_main_delay_header, pfile.MDL->head(), 0.
  Pacman_binfile_create_MDL, pfile                           ;; Main Delay: non existant
  ;;
  ;; Img_detector_FSUA: not empty ----------------------------------------
  ;;
  standard_img_detector_header, pfile.det_A->head(), 'Romeo', 8.9d

  Pacman_binfile_create_DetA, pfile
  standard_img_detector_table, pfile.det_A
  ;;
  ;; imaging data FSUA: not empty
  ;;
  img_data_header, pfile.dataA->head(), (FSUAdit gt 0), MARCELtemp, FSUnoise
  Pacman_binfile_create_DataA, pfile

  if FSUAdit gt 0 then begin
     print,format='("FSUA data...",$)'
     for iScan= 0,Nscans-1 do begin
        print,format='(I2,"...",$)',iScan
        ;;t0 = systime(/second)
        ftscan, pfile.dataA, MARCELtemp, iScan+1,Npnts, scan_length,delay_shifts[iScan],$
                FSUrate, FSUnoise, Tnoise, FSU=0, PHASE_OFFS=phase_offs, PUTfile='throughput_FSUA.sav'
        ;;print, systime(/second)-t0," seconds"
     endfor
     print,''
  endif
  ;;
  ;; Imaging detector FSUB: not empty
  ;;
  standard_img_detector_header, pfile.det_B->head(), 'Juliet', 8.9d

  Pacman_binfile_create_DetB, pfile
  standard_img_detector_table, pfile.det_B
  ;;
  ;; imaging data FSUB: not empty
  ;;
  img_data_header, pfile.dataB->head(), (FSUBdit gt 0), MARCELtemp, FSUnoise
  Pacman_binfile_create_DataB, pfile

  if FSUBdit gt 0 then begin
     print,format='("FSUB data...",$)'
     for iScan= 0,Nscans-1 do begin
        print,format='(I2,"...",$)',iScan
        ftscan, pfile.dataB, MARCELtemp, iScan+1,Npnts, scan_length,delay_shifts[iScan],$
                FSUrate, FSUnoise, Tnoise, FSU=1, PHASE_OFFS=phase_offs, PUTfile='throughput_FSUB.sav'
     endfor
     print,''
  endif

  ;;
  ;; PRIMETrology ------------------------------------------------------------------
  ;;
  head = pfile.metro->head()
  standard_metrology_header, head, PRIMETrate, [0.,0.], PRIMETnoise
  head->addESOPar,'SIM SCAN LENGTH', scan_length*1d6, COMMENT=" length of one scan [micron]"
  Pacman_binfile_create_metro, pfile	;; PRIMET

  N_pp = Npnts / FSUrate * PRIMETrate	;; PRIMET-points per scan

  metrology_data_row__define, tabdat
  tabdat = replicate( tabdat, N_pp )

  ;; status: '13F'xL  before COMMissioning #5

  print,"PRIMET data..."
  for iScan= 0,Nscans-1 do begin
      tabdat.time= round( (double(iScan*N_pp) + dindgen(N_pp)) / PRIMETrate * 1d6)  ;;[usec]

      if PRIMETnoise ne 0. then $
        tabdat.deltal = PRIMETnoise * randomn(primseed,N_pp,/normal)

      ;;print,format='(E)',mean(tabdat.deltal)
      pfile.metro->writeRows, tabdat
  endfor
  ;;
  ;; Metrology - the sequel
  ;;
  standard_metrology_header, pfile.metro->head(),/FSUB, PRIMETrate, [0.,0.], PRIMETnoise

  metrology_data_fsub_row__define, tabdat
  tabdat = replicate( tabdat, N_pp )

  Pacman_binfile_create_metroB,pfile    ;; PRIMETB

  print,"PRIMETB data..."
  for iScan= 0,Nscans-1 do begin
      tabdat.time= round( (double(iScan*N_pp) + dindgen(N_pp)) / PRIMETrate * 1d6)  ;;[usec]

      Nslewp= N_pp/4		;; No. of points where we are really slewing
      Nstand= (N_pp-Nslewp)/2	;; No. of points where we are standing still in the beginning

      delta_slew= scan_length / double(Nslewp-1)	;; meter
      dshift = delay_shifts[iScan] / 1d6		;; meter

      tabdat[0:Nstand-1].deltal		   = dshift - 0.5*scan_length
      tabdat[Nstand:Nstand+Nslewp-1].deltal= dshift - 0.5*scan_length + dindgen(Nslewp)*delta_slew
      tabdat[Nstand+Nslewp:N_pp-1].deltal  = dshift + 0.5*scan_length

      tabdat.deltal += PRIMETnoise*randomn(primseed,N_pp,/normal)

      ;;if keyword_set(fsub) then --- no, always
      tabdat.deltaL *= -1	;; weird ESO convention

      ;;print,format='(E)',tabdat[0].deltal
      pfile.metroB->writeRows, tabdat
  endfor
  ;;
  ;; Difficult Delay Lines
  ;;
  Pacman_binfile_create_DDL,   pfile    ;; DDL: empty
  Pacman_binfile_create_OPDC,  pfile
  Pacman_binfile_create_DOPDC, pfile
  ;;
  ;; close the door and cleanup before you leave
  ;;
  Pacman_binfile_close,  pfile
  Pacman_binfile_cleanup, pfile

  print,"===== FSUresponse finished ====="
  if keyword_set(view) then fbtv,filename
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO VLTIresponse, FILE_PREFIX=file_prefix, STARTTIME=starttime, VIEW=view,$
                  STARSPECTRUM=starspec, NSCANS=Nscans, NPNTS=Npnts, SCANLEN=scan_length,$
                  STARTEMP=StarTemp, PHASE_OFFS=phase_offs, $
                  METNOISE=PRIMETnoise, FSUNOISE=FSUnoise, TIMENOISE=Tnoise, DELAYJIT=delay_jitter

  if not keyword_set(file_prefix) then file_prefix= 'PACMA'

  MJD0= eso_MJD(starttime)

  if keyword_set(Starspec) then begin
     print,"===== Starspectrum template ====="
     sim_code= 'SR'
     dpr_catg= 'SCIENCE '
     tpl_name= 'PACMAN_obs_spectrum'
  endif else begin
     print,"===== VLTIresponse template ====="
     sim_code= 'VR'
     dpr_catg= 'CALIB   '
     tpl_name= 'PACMAN_cal_Sky_VLTIresponse'
  endelse

  if not keyword_set(Nscans) then Nscans= 10
  if not keyword_set(Npnts)  then Npnts = 2^12		;; 4096 points per scan
  ;;if not keyword_set(scan_length)  then scan_length= 1.6d-3	;; 1600 micron in [m]
  if not keyword_set(scan_length)  then scan_length= 300d-6	;; 300 micron in [m] (better!? than lab)
  if not keyword_set(StarTemp)	   then STARtemp = 4200.
  if N_elements(PRIMETnoise)  lt 1 then PRIMETnoise= 1d-7	;; [meter]
  if N_elements(FSUnoise)     lt 1 then FSUnoise = 0.5		;; percent of uncorrelated flux
  if N_elements(Tnoise)       lt 1 then Tnoise   = 0.2		;; usec
  if N_elements(delay_jitter) lt 1 then delay_jitter= 1.	;; mircons

  PRIMETrate= 4000.			;; [junk/second]
  FSUrate = 100.			;; [junk/second]
  FSUdit  = 1./FSUrate - 50.d-6		;; 50 usec for readout
  FSUAdit = FSUdit
  FSUBdit = FSUdit
  ;;
  ;; first, take a FSUrsponse and sky-backgronk
  ;;
  if not keyword_set(Starspec) then begin
     ;; don't do it again,
     ;; starspectrum should use the FSUrepo from the previous VLTIresp
     ;; have to response the FSU that is slewing in VLTIresponse
     ;;
     FSUresponse, STARTTIME=MJD_to_ESO_string(MJD0),/FSUA, PHASE_OFFS=phase_offs,$
                  METNOISE=PRIMETnoise, FSUNOISE=FSUnoise, TIMEnoise=Tnoise, DELAYJIT=delay_jitter
     MJD0 += 0.25/24.d
     starspectrum_one, undefined,$
                       { Starpos, Name:"Prima", RA:0d, Dec:0d, pmRA:0d, pmDec:0d, Epoch:2000., Parallax:0d, v_rad:0d},$
                       { star_temp: STARtemp, visi: 0.7, star_angdiam: 0.05 }
  endif

  ;;- no more SKYBACKs
  ;;if FSUnoise gt 0. then begin
  ;;   skybackground, MJD=MJD0, DITA=FSUAdit, DITB=FSUBdit, BIAS=0.,FLUX=0.
  ;;endif else begin
  ;;   skybackground, MJD=MJD0, DITA=FSUAdit, DITB=FSUBdit, BIAS=0.,FLUX=0.,NOISE=0.
  ;;endelse
  MJD0 += 0.1/24.d

  filename = file_prefix + '.' + MJD_to_ESO_string(MJD0) + ".fits"
  print, "File: ",filename
  ;;print, "FSUdits:",FSUAdit,FSUBdit

  pfile= Pacman_binfile(MJD0)

  N_sec = Nscans*Npnts / FSUrate

  delay_shifts= delay_jitter * randomn(delayseed,Nscans,/normal)

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ;;
  ;; PRIMARY header
  ;;
  prihead= pfile.arrdesc->prihead()
  prihead->addPar,'DATE-OBS', MJD_to_ESO_string(MJD0), COMMENT=" Simulating date"
  prihead->addPar,'MJD-OBS', MJD0, FORMAT="F16.10"
  prihead->addPar,'UTC', (MJD0 mod 1.)*86400.,$
		   Comment= " " + hours_to_hms((MJD0 mod 1.)*24.) + " UTC at start (sec)"
  ;prihead->addPar,'LST', lst_start * 3600d, $
  ;		   Comment= " " + hours_to_hms(lst_start) + " LST at start (sec)"
  ;prihead->addPar,'RA' , RA*15., Comment= " " + hours_to_hms(RA)  + " RA pointing (deg)"
  ;prihead->addPar,'Dec', Dec,    Comment= " " + hours_to_hms(Dec) + " DEC pointing (deg)"
  prihead->addPar,'EXPTIME',float(N_sec),Comment= " (seconds)"

  prihead->addESOpar,'INS OPDSCAN NSLEW', Nscans, Comment=' Number of slews'
  prihead->addESOPar,'ISS CONF NTEL', 0, Comment=' no telescope'

  insmode= 'NORMAL'
  prihead->addESOPar,'INS MODE', insmode, COMMENT=" instrument mode"

  standard_primary_header, prihead, FSUAdit, FSUBdit, SIM_Code= sim_code
  ;;CATG=dpr_catg, TYPE='OBJECT  ', TECH='INTERFEROMETRY,SPECTRUM'
  ;;prihead->addESOPar,'TPL NAME', tpl_name, Comment="Template name"

  ;; overwrite default set by standard_primate_header:
  prihead->addESOPar,'INS FSU1 STATE',    'SLEWING',Comment=' Current FSU state'
  prihead->addESOPar,'ISS PRI FSU1 STATE','SLEWING',Comment=' Another FSU state'

  prihead->addESOpar,'SIM STAR TEMP', STARtemp, Comment=' STAR coolness [Kelvin]'
  prihead->addESOpar,'SIM NUM SCANS',   Nscans, Comment=' Number of scans'
  prihead->addESOpar,'SIM NUM SLEWS', 2*Nscans, Comment=' Number of slews'
  prihead->addESOpar,'SIM SCAN POINTS', Npnts, Comment=' Data points per scan'
  prihead->addESOpar,'SIM MET NOISE', PRIMETnoise, Comment=' [meter]'
  prihead->addESOpar,'SIM FSU NOISE', FSUnoise,    Comment=' [% of uncorr. flux]'
  prihead->addESOpar,'SIM TIME NOISE', Tnoise,     Comment=' [usec]'
  prihead->addESOpar,'SIM DELAY JITTER',delay_jitter, Comment=' [mircons]'
  ;;
  ;; Tables in DICD-Order --------------------------------------------
  ;;
  Pacman_binfile_create_arrdesc, pfile, filename             ;; Array description
  pfile.arrdesc->writeRows, standard_array_description()

  Pacman_binfile_create_arrgeom, pfile                       ;; Array geometry
  pfile.arrgeom->writeRows, standard_array_geometry()

  Pacman_binfile_create_optrain, pfile                       ;; Optical train
  pfile.optrain->writeRows, standard_optical_train()
  ;;
  ;; Main Delay -------------------------------------------------------
  ;;
  standard_main_delay_header, pfile.MDL->head(), 0.
  Pacman_binfile_create_MDL, pfile                           ;; Main Delay
  ;;
  ;; Img_detector_FSUA: not empty ----------------------------------------
  ;;
  standard_img_detector_header, pfile.det_A->head(), 'Romeo', 8.9d

  Pacman_binfile_create_DetA, pfile
  standard_img_detector_table, pfile.det_A
  ;;
  ;; imaging data FSUA: not empty
  ;;
  img_data_header, pfile.dataA->head(), (FSUAdit gt 0), STARtemp, FSUnoise
  Pacman_binfile_create_DataA, pfile

  if FSUAdit gt 0 then begin
     print,Format='("FSUA data...",$)'
     for iScan= 0,Nscans-1 do begin
        print,format='(I2,"...",$)',iScan
        ;;t0 = systime(/second)
        ftscan, pfile.dataA, STARtemp, iScan+1,Npnts, scan_length, delay_shifts[iScan],$
                FSUrate, FSUnoise, Tnoise, FSU=0, PUTfile='throughput_FSUA.sav',$
                PHASE_OFFS=phase_offs, /NOSTAND
        ;;print, systime(/second)-t0," seconds"
     endfor
     print,''
  endif
  ;;
  ;; Imaging detector FSUB: not empty
  ;;
  standard_img_detector_header,pfile.det_B->head(), 'Juliet', 8.9d
  Pacman_binfile_create_DetB, pfile
  standard_img_detector_table,pfile.det_B
  ;;
  ;; imaging data FSUB: not empty
  ;;
  img_data_header, pfile.dataB->head(), (FSUBdit gt 0), STARtemp, FSUnoise
  Pacman_binfile_create_DataB, pfile

  if FSUBdit gt 0 then begin
     print,Format='("FSUB data...",$)'
     for iScan= 0,Nscans-1 do begin
        print,format='(I2,"...",$)',iScan
        ftscan, pfile.dataB, STARtemp, iScan+1, Npnts, 0., delay_shifts[iScan],$
                FSUrate, FSUnoise, Tnoise, FSU=1, PUTfile='throughput_FSUB.sav',$
                PHASE_OFFS=phase_offs, /NOSTAND
     endfor
     print,''
  endif

  ;;
  ;; PRIMETrology ------------------------------------------------------------------
  ;;
  head = pfile.metro->head()
  standard_metrology_header, head, PRIMETrate, [0.,0.], PRIMETnoise
  head->addESOPar,'SIM SCAN LENGTH', scan_length,COMMENT=" length of one scan [micron]"
  Pacman_binfile_create_metro, pfile	;; PRIMET

  N_pp = Npnts / FSUrate * PRIMETrate	;; PRIMET-points per scan

  metrology_data_row__define, tabdat
  tabdat = replicate( tabdat, N_pp )

  print,"PRIMET data..."
  for iScan= 0,Nscans-1 do begin
      tabdat.time= round( (double(iScan*N_pp) + dindgen(N_pp)) / PRIMETrate * 1d6)  ;;[usec]

      ;; old and wrong: OPD stands still at start and end of slew
      ;;Nslewp= N_pp/4		;; No. of points where we are really slewing
      ;;Nstand= (N_pp-Nslewp)/2	;; No. of points where we are standing still in the beginning
      ;;
      ;;delta_slew= scan_length / double(Nslewp-1)	;; meter
      delta_slew= scan_length / double(N_pp-1)	;; meter
      dshift = delay_shifts[iScan] / 1d6		;; meter

      ;;tabdat[0:Nstand-1].deltal		   = dshift - 0.5*scan_length
      ;;tabdat[Nstand:Nstand+Nslewp-1].deltal= dshift - 0.5*scan_length + dindgen(Nslewp)*delta_slew
      ;;tabdat[Nstand+Nslewp:N_pp-1].deltal  = dshift + 0.5*scan_length

      tabdat.deltal= dshift - 0.5*scan_length + dindgen(N_pp)*delta_slew

      tabdat.deltal += PRIMETnoise*randomn(primseed,N_pp,/normal)

      pfile.metro->writeRows, tabdat
  endfor
  ;;
  ;; Metrology - the sequel
  ;;
  standard_metrology_header, pfile.metro->head(),/FSUB, PRIMETrate, [0.,0.], PRIMETnoise

  metrology_data_fsub_row__define, tabdat
  tabdat = replicate( tabdat, N_pp )

  Pacman_binfile_create_metroB,pfile    ;; PRIMETB: empty

  print,"PRIMETB data..."
  for iScan= 0,Nscans-1 do begin
      tabdat.time= round( (double(iScan*N_pp) + dindgen(N_pp)) / PRIMETrate * 1d6)  ;;[usec]

      ;; METRO B is zero because FSU B is tracking perfectly
      ;; (except for noise, of course)
      if PRIMETnoise ne 0. then $
        tabdat.deltal = PRIMETnoise * randomn(primseed,N_pp,/normal)

      pfile.metroB->writeRows, tabdat
  endfor
  ;;
  ;; Difficult Delay Lines
  ;;
  Pacman_binfile_create_DDL,   pfile    ;; DDL: empty
  ;;
  ;; OPD Counsellor
  ;;
  ;; TODO: Define rate for (D)OPDC that is not the same as FSUrate
  ;;
  Pacman_binfile_create_OPDC,  pfile
  Ntot = long(Npnts)*long(Nscans)
  tabdat= replicate( {OPDC_row}, Ntot)
  tabdat.time= round(dindgen(Ntot)*1d6/FSUrate + tnoise*randomn(tseed,Ntot))  ;; usec
  tabdat.OPDCState= 7
  tabdat.State    = 7
  tabdat.Status   = 7
  pfile.OPDC->writeRows, tabdat
  ;;
  ;; Difficult OPD Counsellor
  ;;
  Pacman_binfile_create_DOPDC, pfile
  tabdat= replicate( {DOPDC_row}, Npnts)
  for iScan= 1L,Nscans do begin
     tabdat.time= round( (long(Npnts)*(iScan-1L) + dindgen(Ntot))*1d6/FSUrate + tnoise*randomn(tseed,Npnts))  ;; usec
     ;; first scan has stepping_phase=1, last scan has Nscans
     tabdat.stepping_Phase= iScan
     pfile.DOPDC->writeRows, tabdat
  endfor
  ;;
  ;; close the door and cleanup before you leave
  ;;
  Pacman_binfile_close,  pfile
  Pacman_binfile_cleanup,pfile

  if keyword_set(Starspec) then begin
     print,"===== Starspectrum finished ====="
  endif else begin
     print,"===== VLTIresponse finished ====="
  endelse
  if keyword_set(view) then fbtv,filename
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

pro starspectrum_one, fname, pos, FSU_cfg, MJD=MJD

  common PACSIM_DEFAULTS	;; contains kappa_center

  ppos={ starpos }
  struct_assign, { Name: "Zero", Epoch: 2000., RA: 0., Dec: 0.}, ppos

  if N_elements(fname) lt 1 then fname= 'STARSPECTRUM.fits'
  if N_elements(pos) lt 1 then pos= ppos
  if N_elements(FSU_cfg) lt 1 then $
     FSU_cfg= { star_temp: 4200., visi: 0.7, star_angdiam: 0.05 }

  ;; defaults like raw starspectrum (see above):
  FSUrate = 1000.			;; [junk/second]
  FSUdit  = 1./FSUrate - 50.d-6		;; 50 usec for readout
  FSUAdit = FSUdit
  FSUBdit = FSUdit


  ;;kappa_cs  = 1d6 / [2.5, 2.4, 2.3, 2.2, 2.1]	;; [1/m]
  kappa_start= 380000.d
  kappa_step =    500.d

  spectrum_row__Define, row, units
  tabdat= replicate(row,321)
  if FSU_cfg.star_temp eq 0 then tabdat.spectrum= 1.	;; very black body

  for ch=0,5 do begin
     tabdat.wavenumber[ch]= kappa_start + dindgen(321)*kappa_step
     if FSU_cfg.star_temp ne 0 then $
        tabdat.spectrum[ch] = blackbody(tabdat.wavenumber[ch],kappa_step,FSU_cfg.star_temp)
     tabdat.spectrum_err[ch]= 0.
  endfor

  if not keyword_set(MJD) then MJD= MJulday(2000,1,1, 0,0,0)

  stave = fitstable_create('spectrum')
  prihead= stave->prihead()
  prihead->addPar,'DATE-OBS', MJD_to_ESO_string(MJD), COMMENT=" Simulating date"
  prihead->addPar,'MJD-OBS', MJD, FORMAT="F16.10"
  prihead->addESOPar,'INS REC START', MJD, FORMAT="F16.10"
  prihead->addPar,'UTC', (MJD mod 1.)*86400., $
    Comment= " " + hours_to_hms((MJD mod 1.)*24.) + " UTC at start (sec)"

  insmode= 'NORMAL'
  prihead->addESOPar,'INS MODE', insmode, COMMENT=" instrument mode"

  standard_primary_header, prihead, FSUAdit, FSUBdit, SIM_CODE='SR'
  ;;CATG='SCIENCE ', TYPE='OBJECT  ', TECH='INTERFEROMETRY,SPECTRUM'
  prihead->addESOPar,'ISS FSU2 STATE', 'SLEWING', Comment=' Current FSU state'

  standard_targ_header, prihead, '1', 'PS',ppos, FSU_cfg
  standard_targ_header, prihead, '2', 'SS', pos, FSU_cfg

  prihead->addESOpar,'PRO CATG', 'STARSPEC', Comment=' star spectrum'
  prihead->addESOpar,'QC INSTRUMENT', 'FSUA,FSUB', Comment=' computed FSUs'
  prihead->addESOpar,'QC NUMCHNL', 6, Comment=' 5.1 channels, with subwoofer'

  stave->newfile,  fname, iErr=iErr  & if iErr then print,"cannot create file ",fname,", iErr =",iErr
  stave->writerows,tabdat,iErr=iErr  & if iErr then print,"cannot write data, iErr =",iErr
  stave->close
  obj_destroy, stave
end

pro starspectrum_both
  FSU_cfg= { star_temp: 0., visi: 0.7, star_angdiam: 0.05 }
  starspectrum_one, 'STARSPECTRUM1.fits',$
                    { Name: "Prima",   RA: 0.d, Dec: 0d, pmRA: 0.d, pmDec: 0.d, Epoch: 2000. },$
                    FSU_cfg,$
                    MJD= MJulday(2000,4,1, 0,0,0)
  starspectrum_one, 'STARSPECTRUM2.fits',$
                    { Name: "Secunda", RA: 0.d, Dec: 10d/3600., pmRA: 0.d, pmDec: 0.d, Epoch: 2000. },$
                    FSU_cfg,$
                    MJD= MJulday(2000,4,1, 0,1,0)
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

pro FSUresponse_testfiles
  print,"This is FSUresponse!"

  FSUresponse,STARTTIME='2008-04-01T01:00:00',/FSUA, METNOISE=0.,FSUNOISE=0.,TIMEnoise=0,DELAYJIT=0.
  FSUresponse,STARTTIME='2008-04-01T01:10:00',/FSUB, METNOISE=0.,FSUNOISE=0.,TIMEnoise=0,DELAYJIT=0.
  FSUresponse,STARTTIME='2008-04-01T01:20:00',/FSUA,/FSUB,METNOISE=0.,FSUNOISE=0.,TIMEnoise=0,DELAYJIT=0.

  FSUresponse,STARTTIME='2008-04-01T01:30:00', /FSUA
  FSUresponse,STARTTIME='2008-04-01T01:40:00', /FSUB
  FSUresponse,STARTTIME='2008-04-01T01:50:00', /FSUA,/FSUB
end


pro Shifted_testfiles
  print,"Phase-shifted testfiles"

  FSUresponse,STARTTIME='2011-04-01T00:00:00', /FSUA, PHASE_OFFS= [ 15,8,15 ] * !dpi/180.
  FSUresponse,STARTTIME='2011-04-01T00:30:00', /FSUB, PHASE_OFFS= [ 15,8,15 ] * !dpi/180.
end


pro VLTIresponse_testfiles
  ;; METNOISE=0. does not work with the slew-selection algortihm
  VLTIresponse,STARTTIME='2008-04-01T02:00:00',FSUNOISE=0.,TIMEnoise=0,DELAYJIT=0.
  VLTIresponse,STARTTIME='2008-04-01T02:30:00'
end


pro StarSpectrum_testfiles
  ;; METNOISE=0. does not work with the slew-selection algortihm
  VLTIresponse,StartTime='2008-04-01T03:00:00',/Starspectrum,Startemp=3333.,$
               FSUNOISE=0.,TIMEnoise=0,DELAYJIT=0.
  VLTIresponse,StartTime='2008-04-01T03:30:00',/Starspectrum,Startemp=3333.
end


pro VLTIresponse_dphase0_test
  ;; METNOISE=0. does not work with the slew-selection algortihm
  VLTIresponse,STARTTIME='2008-04-01T02:30:00', PHASE_OFFS=[90,180,270] * !dpi/180.
end
