;
; star.pro
; Create9 by Rkoehler@lx40
; Last change: Wed Apr  8 15:01:38 2009
;
; PURPOSE:
;	let the stars simulate themselves
;
; WARNING:
;	This program may contain strong violence, adult content, and
;	language.  Parental guidance is advised.
;
PRO Star__Define
  junk={ star, $
         Name: "Prisemacunda",$
         Epoch: 2000.0d,$
         RA:	   0.d,$	;; in radian or something
         Dec:	   0.d,$
         Parallax: 0.d,$	;; in mas
         Parad:	   0.d,$	;; in radian
         pmRA:	   0.d,$
         pmDec:	   0.d,$
         v_rad:	   0.d,$	;; km/sec
         X0:	0.d,$		;; 3D-unitvector of barycentric position observed at epoch
         Y0:	0.d,$
         Z0:	0.d,$
         Vx0:	0.d,$		;; 3D-vector of velocity/distance
         Vy0:	0.d,$
         Vz0:	0.d,$
         debug: 0L, $
         fbt:   obj_new() $
       }
end

PRO Star_positions_corrected_row__Define, row, units
  row= { MJD:	0.d,$
         Epoch: 0.d,$
         time:  0.d,$
         barycentric: [1d,2d,3d],$
         geocentric:  [1d,2d,3d],$
         deflected:   [1d,2d,3d],$
         aberrated:   [1d,2d,3d],$
         Earth_position: [1d,2d,3d],$
         Earth_velocity: [1d,2d,3d],$
         RA:  0d,$
         Dec: 0d $
       }
  units=[["MJD",  "days", "Modifried Julian's Date"],$
         ["Epoch","years","Years since J2000.0"],$
         ["time", "years","Years since star epoch"],$
         ["barycentric","unit vec","Unit vec from barycenter to star"],$
         ["geocentric", "unit vec","Unit vector from earth to star"],$
         ["deflected",  "unit vec","after relativistic light deflection"],$
         ["aberrated",  "unit vec","including aberration"],$
         ["RA", "radian", "Right Ascension"],$
         ["Dec","radian", "Declkination"] ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function star::init, name, epoch, ra, dec, parallax, pmRA, pmDec, v_rad

  self.Name = name		;; string
  self.Epoch= epoch		;; julian years
  self.RA   = ra * !dpi/12.	;; input in hours
  self.Dec  = dec* !dpi/180.	;; input in deg
  self.Parallax= parallax
  self.pmRA = pmRA/cos(self.Dec) * !dpi/180./3600.	;; input is mu*cos(dec) in arcsec/year
  self.pmDec= pmDec * !dpi/180./3600.			;; input in arcsec/yr
  self.v_rad= v_rad		;; km/sec


  sRA = sin(self.RA)	& cRA = cos(self.RA)
  sDec= sin(self.Dec)	& cDec= cos(self.Dec)
  self.Parad= parallax * !dpi/180./3600d3

  self.X0 = cRA*cDec	;; unit vector
  self.Y0 = sRA*cDec
  self.Z0 = sDec

  pmRad   = self.v_rad * 365.25*86400d/149.60d6 * self.Parad	;; v_radial/Distance in radian/year

  self.Vx0= -sRA*self.pmRA - cRA*sDec*self.pmDec + cRA*cDec*pmRad	;; in radian/year
  self.Vy0=  cRA*self.pmRA - sRA*sDec*self.pmDec + sRA*cDec*pmRad
  self.Vz0=                      cDec*self.pmDec +     sDec*pmRad

  openw,unit, name+"_positions",/get_lun
  self.debug= unit
  printf,unit,name,epoch,RA,Dec,Parallax,pmRA,pmDec,v_rad

  self.fbt= fitstable_create('Star_positions_corrected')
  standard_targ_header, self.fbt->prihead(), '1', 'PS',$
                        { Name:name, Epoch: epoch, RA:ra, Dec:dec, Parallax:parallax, pmRA:pmRA, pmDec:pmDec, v_rad:v_rad},$
                        { Star_temp: 6666., Visi: 1.1, star_angdiam: 42. }

  self.fbt->newFile, name+"_positions.fits"

  return,1	;; Qapla'
end

PRO star::cleanup
  free_lun, self.debug
  self.fbt->close
  obj_destroy, self.fbt
end


Function star::current_position, MJD

  epo = 2000.0d + (MJD - 51544.5d)/365.25
  time= epo - self.Epoch

  RA = self.RA  + time * self.pmRA
  Dec= self.Dec + time * self.pmDec

  JD = MJD+2400000.5d
  t1= min(JD, max=t2)

  JPLEPHREAD, getenv('HOME')+'/IDL/cmephem/JPLEPH.405', pinfo, pdata, [t1,t2]
  JPLEPHINTERP, pinfo, pdata, JD, Xe, Ye, Ze, /EARTH, posunits='AU'

  sRA = sin(RA)	 & cRA = cos(RA)
  sDec= sin(Dec) & cDec= cos(Dec)

  Parad= self.Parallax * !dpi/180./3600d3
  dRA = Parad * ( Xe * sRA - Ye * cRA) / cDec		;; in radian
  dDec= Parad * ((Xe * cRA + Ye * sRA)*sDec - Ze * cDec)

  printf,self.debug, MJD,epo,RA,Dec,Parad

  return, { Name: self.Name,$
            RA:  (RA +dRA )* 12./!dpi,$
            Dec: (Dec+dDec)*180./!dpi,$
            Epoch: epo $
          }
end


Function star::current_position3, MJD, DEBUG=debug, PLOT=plot

  epo = 2000.0d + (MJD - 51544.5d)/365.25d
  time= epo - self.Epoch	;; time between Reference-Epoch and observation(s)

  if keyword_set(debug) then print,"time =",time[0:3]

  Star_positions_corrected_row__Define, corrected_positions
  corrected_positions.MJD  = MJD[0]
  corrected_positions.Epoch= epo[0]
  corrected_positions.time = time[0]

  ;;X_baryc = self.X0 + time * self.Vx0	;; too simple!
  ;;Y_baryc = self.Y0 + time * self.Vy0
  ;;Z_baryc = self.Z0 + time * self.Vz0

  c_kms = 2.99792458d5                          ;; lightspeed in km/sec
  c_AUyr= c_kms	* 365.25*86400d/149.60d6	;; lightspeed in AU/year
  ;;      km/sec  ( sec/year  )  km/AU

  if self.parad gt 0. then begin

     Distance= 1.d/self.parad	;; distance in AU
     ;;
     ;; compute position at time of observation (irrelevant, but useful)
     ;;
     tobs = Distance/c_AUyr + time	;; time from t0 to t_observation
     X_tobs = self.X0 + tobs * self.Vx0
     Y_tobs = self.Y0 + tobs * self.Vy0
     Z_tobs = self.Z0 + tobs * self.Vz0

     lsu = c_AUyr * self.parad	;; lightspeed/Distance (same unit as self.V_0)
     ;;
     ;; compute time between emission and observation of light
     ;;
     r_v = X_tobs*self.Vx0 + Y_tobs*self.Vy0 + Z_tobs*self.Vz0                ;; (vec r) * (vec v)
     r2  = X_tobs*X_tobs + Y_tobs*Y_tobs + Z_tobs*Z_tobs                      ;; r^2
     ls2_v2= lsu*lsu - self.Vx0*self.Vx0 + self.Vy0*self.Vy0 + self.Vz0*self.Vz0	;; c^2-v^2
     tau = (-r_v + sqrt( r_v*r_v + r2 * ls2_v2) ) / ls2_v2

     ;;print," 2.term:",r2*ls2_v2
     if keyword_set(debug) then begin
        print,"tobs=",tobs[0:3]
        print,"tau =",tau[0:3]
     endif
     temit= tobs - tau	;; time when light was emitted
  endif else begin
     temit= time	;; if dist=infinite, then light delay is negligible
  endelse
  if keyword_set(debug) then print,"temit =",temit[0:3]

  X_baryc = self.X0 + temit * self.Vx0	;; pos of star at temit
  Y_baryc = self.Y0 + temit * self.Vy0
  Z_baryc = self.Z0 + temit * self.Vz0

  Dist_baryc= sqrt( X_baryc*X_baryc + Y_baryc*Y_baryc + Z_baryc*Z_baryc)

  X_baryc /= Dist_baryc		;; now it's a unit vec
  Y_baryc /= Dist_baryc
  Z_baryc /= Dist_baryc

  corrected_positions.barycentric = [ X_baryc[0], Y_baryc[0], Z_baryc[0] ]
  ;;
  ;; get ephemeris for relevant time range
  ;;
  t1= min(MJD, max=t2)

  JPLEPHREAD, getenv('HOME')+'/IDL/cmephem/JPLEPH.405', pinfo, pdata, [t1,t2]+2400000.5d
  JPLEPHINTERP, pinfo, pdata, MJD, Xe,Ye,Ze, Vxe,Vye,Vze, /EARTH,/VELOcity,$
                TBASE=2400000.5d, Posunits='AU', Velunits='KM/S'

  corrected_positions.Earth_position = [ Xe[0], Ye[0], Ze[0] ]
  corrected_positions.Earth_velocity = [ Vxe[0], Vye[0], Vze[0] ]

  X_geoc = X_baryc - Xe*self.parad	;; this is not a unit vector
  Y_geoc = Y_baryc - Ye*self.parad
  Z_geoc = Z_baryc - Ze*self.parad

  Dist_geoc= sqrt( X_geoc*X_geoc + Y_geoc*Y_geoc + Z_geoc*Z_geoc)

  X_geoc /= Dist_geoc		;; now it's a unit vec
  Y_geoc /= Dist_geoc
  Z_geoc /= Dist_geoc

  ;;help,X_geoc
  corrected_positions.geocentric = [ X_geoc[0], Y_geoc[0], Z_geoc[0] ]

  ;;
  ;; relativistic light defelction
  ;;
  if keyword_set(plot) then plot, [-20,30], [-33,12], /nodata, /yst,/iso

  N_obs = N_elements(MJD)
  X_plan= dblarr(11,N_obs)	;; 1.idx = planet, 2.idx = MJD
  Y_plan= dblarr(11,N_obs)
  Z_plan= dblarr(11,N_obs)

  for pl=1,11 do begin
     JPLEPHINTERP, pinfo,pdata, MJD, Xp,Yp,Zp, Object=pl,Center='EARTH', posunits='AU',TBASE=2400000.5d

     if keyword_set(debug) then print,"Planet",pl,Xp[0],Yp[0],Zp[0]

     X_plan[pl-1,*] = Xp
     Y_plan[pl-1,*] = Yp
     Z_plan[pl-1,*] = Zp

     if keyword_set(plot) then oplot,Xp,Yp,psym=3
  endfor
  if keyword_set(plot) then oplot, X_geoc*25,Y_geoc*25, color=200

  Masses= [ 3.3022d23,$	;; Mercury, all in kg
            4.8685d24,$	;; Venus
            0.,$        ;; Earth (irrelevant)
            6.4185d23,$ ;; Mars
            1.8986d27,$ ;; Jupiter
            5.6846d26,$	;; Saturn
            8.6810d25,$	;; Uranus
            1.0243d26,$	;; Neptun
            1.305d22,$	;; Pluto
            7.3477d22,$	;; Moon
            1.9891d30 $	;; Sun
          ]
  plnames=['Mercury','Venus','Earth','Mars','Juputer','Saturn','Uranus','Neptune','Pluto','Moon','Sun']

  Grav = 6.67d-11	;; m^3/kg/s^2
  lspd = 2.99792458d8	;; lightspeed in m/s
  const= 2.d * Grav * Masses / lspd / lspd / 149.60d9	;; in AU (like XYZ_plan)

  for t=0,N_elements(MJD)-1 do begin
     if keyword_set(debug) then print, MJulDate(MJD[t])

     Xp = X_plan[*,t]
     Yp = Y_plan[*,t]
     Zp = Z_plan[*,t]
     proj = Xp*X_geoc[t] + Yp*Y_geoc[t] + Zp*Z_geoc[t]
     if keyword_set(debug) then print,format='(I4,11F10.5," AU")',t, proj

     dist = sqrt(Xp*Xp + Yp*Yp + Zp*Zp)
     ;;print,format='("  d=",11F10.5)',dist

     cosang= proj/dist
     angle = acos(cosang)
     if keyword_set(debug) then print,format='("psi=",11F10.5," deg")',angle*180/!dpi

     phi = dblarr(11)	;; = const / dist / tan(angle/2.)

     pord = reverse(sort(proj))	;; idx of planet ordered by light path

     for i=0,10 do begin
        pl = pord[i]
        if pl eq 2 then continue	;; Earth does not deflect light falling onto it

        cosa= (Xp[pl]*X_geoc[t] + Yp[pl]*Y_geoc[t] + Zp[pl]*Z_geoc[t]) / dist[pl]
        ang = acos(cosa)
        phi[pl] = const[pl] / dist[pl] / tan(ang/2.)

        corr= const[pl] / dist[pl] / (1.-cosa)	;; Kovalevsky&Seidelmann, Eq. 6.53

        X_geoc[t] += corr * Xp[pl]	;; XYZp = -e in Kovalevsky&Seidelmann
        Y_geoc[t] += corr * Yp[pl]
        Z_geoc[t] += corr * Zp[pl]

        len = sqrt( X_geoc[t]*X_geoc[t] + Y_geoc[t]*Y_geoc[t] + Z_geoc[t]*Z_geoc[t])

        X_geoc[t] /= len		;; now it's a unit vec again
        Y_geoc[t] /= len
        Z_geoc[t] /= len

        ;;print,Format='(A8,F10.5," deg",F10.4," uas",3G20.12)',plnames[pl],$
        ;;      ang*180./!dpi, phi*180/!dpi*3600d6, X_geoc[t],Y_geoc[t],Z_geoc[t]
     endfor
     if keyword_set(debug) then print,format='("phi=",11F10.4," uas")',phi*180/!dpi*3600d6
  endfor
  if keyword_set(debug) then print,Format='("Tidx",11A10)', plnames

  ;;print,"corr for sun: ",const[10]*180./!dpi
  if keyword_set(debug) then begin
     print,"X:",X_geoc[0:3]
     print,"Y:",Y_geoc[0:3]
  endif

  corrected_positions.deflected = [ X_geoc[0], Y_geoc[0], Z_geoc[0] ]
  ;;
  ;; Aber-rat-ion (Kovalevsky&Seidelmann chapter 6.3.3)
  ;;
  ha= (lst_from_date(MJD) + 6.) * !dpi/12.
  Vxe += 0.421 * cos(ha)	;; km/sec
  Vye += 0.421 * sin(ha)

  Vxe /= c_kms	;; we need v/c all the time
  Vye /= c_kms
  Vze /= c_kms

  gaminv = sqrt(1. + (Vxe*Vxe+Vye*Vye+Vze*Vze))
  ;;help,gaminv
  if keyword_set(debug) then print,format='("Gamma^-1:",4G15.12)',gaminv[0:3]

  nenner= 1. + (X_geoc*Vxe + Y_geoc*Vye + Z_geoc*Vze)	;; 1.+pV/c

  if keyword_set(debug) then print,"Vx:",Vxe[0:3],"Vy:",Vye[0:3],"Vz:",Vze[0:3]

  X_geoc = gaminv * (X_geoc + Vxe) + Vxe
  Y_geoc = gaminv * (Y_geoc + Vye) + Vye
  Z_geoc = gaminv * (Z_geoc + Vze) + Vze

  len = sqrt( X_geoc*X_geoc + Y_geoc*Y_geoc + Z_geoc*Z_geoc)

  X_geoc /= len		;; now it's a unit vec again
  Y_geoc /= len
  Z_geoc /= len

  if keyword_set(debug) then begin
     print,"X:",X_geoc[0:3]
     print,"Y:",Y_geoc[0:3]
  endif

  corrected_positions.aberrated = [ X_geoc[0], Y_geoc[0], Z_geoc[0] ]
  ;;
  ;; convert to RA,Dec
  ;;
  RA  = atan( Y_geoc, X_geoc)
  Dec = asin( Z_geoc )

  corrected_positions.RA = RA[0]
  corrected_positions.Dec= Dec[0]

  printf,self.debug, MJD,epo,RA,Dec
  self.fbt->writeRows, corrected_positions

  return,{ Name: self.Name,$
           RA:   RA * 12./!dpi,$
           Dec:  Dec*180./!dpi,$
           Epoch: epo, $
           X: X_geoc, Y: Y_geoc, Z: Z_geoc $
         }
end


pro test_star
  ;;star = obj_new("star", "test", 2000.0d, 3.d0+24/60.+48/3600., 17.d0+47.5/60.,666.d, -1.d,1.d,-300d)
  star = obj_new("star", "test", 2000.0d, 0.d0, 0.d,666.d, -1.d,1.d,-300d)

  MJD = MJulday(2000,1,1,12,0,0) + dindgen(121)*365.25d/12.	;; 10 years every month
  ;;MJD = MJulday(2000,1,1,12,0,0) + dindgen(53)*365.25d/52.	;; 1 year every week
  ;;MJD = MJulday(2000,5,1,12,0,0) + dindgen(32)    ;; 1 month every day
  ;;MJD = MJulday(2000,5,31,0,0,0) + dindgen(13)/12	;; 1 day every 2 hours


  print,"test MJDs:",MJD
  cp = star->current_position(MJD)
  cp2= star->current_position3(MJD,/ploT,/deBug)
  help,cp,/str

  junk='' & read,junk

  plot, cp.RA *15.*3600., cp.Dec *3600.,xtitle='mas',ytitle='mas', iso=0
  oplot,cp2.RA*15.*3600., cp2.Dec*3600.,color=222

  print,"maxdiffra: ",max( abs(cp.RA-cp2.RA))
  print,"maxdiffdec:",max( abs(cp.Dec-cp2.Dec))

  obj_destroy,star
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

pro calc_sep_vec

  prinda_file= fitstable_openr('Prinda_positions.fits','STAR_POSITIONS_CORRECTED')
  secuma_file =fitstable_openr('Secuma_positions.fits','STAR_POSITIONS_CORRECTED')

  p_data= prinda_file->readrows()
  s_data= secuma_file->readrows()

  obj_destroy,prinda_file
  obj_destroy,secuma_file

  print,"Everything except MJD, Epoch, and Time is Secuma-Prinda"

  for row=0,N_elements(s_data)-1 do begin
     pd= p_data[row*2]
     sd= s_data[row]
     print,"MJD:	",pd.MJD[0],  sd.MJD[0]
     print,"Epoch:	",pd.Epoch[0],sd.Epoch[0]
     print,"Time:	",pd.Time[0], sd.Time[0]
     print,"Barycentric	",sd.barycentric- pd.barycentric
     print,"Geocentric	",sd.geocentric - pd.geocentric
     print,"Deflected	",sd.deflected  - pd.deflected
     print,"Aberrated	",sd.aberrated  - pd.aberrated
     print,"Earth_pos	",sd.Earth_position - pd.Earth_position
     print,"Earth_veloc	",sd.Earth_Velocity - pd.Earth_Velocity
     print,"d_RA  [rad]	",sd.RA - pd.RA
     print,"d_Dec [rad]	",sd.Dec- pd.Dec
     print,"------------------------------------------------------------------------"
  endfor
  print,"Everything except MJD, Epoch, and Time is Secuma-Prinda"
end
